(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[   1351502,      65632]*)
(*NotebookOutlinePosition[   1352983,      65676]*)
(*  CellTagsIndexPosition[   1352939,      65672]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Introduction to the Maths Lab - Part 2", "Title",
  CellFrame->False,
  TextAlignment->Center],

Cell["The METRIC Project", "Subsubtitle",
  TextAlignment->Center],

Cell[CellGroupData[{

Cell["Introduction", "Section",
  CellDingbat->None],

Cell[TextData[{
  "In this session we will concentrate on the graphical capabilities of ",
  StyleBox["Mathematica;",
    FontSlant->"Italic"],
  " we will also show you how data can be imported into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " from other applications and how it can be exported out of ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  ". "
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["1. Two Dimensional Plotting", "Section",
  CellDingbat->None],

Cell[TextData[{
  "A standard task in the physical laboratory is the plotting and analysis of \
two dimensional experimental data. In many cases this data will be held as a \
sequence of numbers in a file external to ",
  StyleBox["Mathematica ",
    FontSlant->"Italic"],
  "(in fact, all computer-driven lab instruments can give you results \
directly in this form). The numbers can be stored in a variety of formats; \
for this exercise the file is simply a text file holding pairs of values of \
time and intensity separated by a space, each pair appearing on a new line. \n\
\nTo see what is inside any file you simply type in two exclamation marks \
followed by the path to the file. The example from our computer was. "
}], "Text"],

Cell[BoxData[
    \(\(! \(! "\<Hard disk:Desktop Folder:expt1.dat\>"\)\)\)], "Input"],

Cell[TextData[{
  "The path will be different for you, and the way the path is expressed will \
look different on a PC. So to find ",
  StyleBox["expt1.dat",
    FontFamily->"Arial"],
  " on your computer, first start ",
  StyleBox["Mathematica,",
    FontSlant->"Italic"],
  " open up a new notebook, type in two exclamation marks and then go to  \
the",
  StyleBox[" Input",
    FontFamily->"Arial"],
  " menu and select ",
  StyleBox["Get File Path ...",
    FontFamily->"Arial"],
  "  to locate the data file ",
  StyleBox["expt1.dat ",
    FontFamily->"Arial"],
  "on your hard disk. When you click on ",
  StyleBox["Open",
    FontFamily->"Arial"],
  " the appropriate file path will appear on the input line. Press \
<shift-return> now and the contents of the file will appear.\n\nTo read the \
data into ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " we use the ",
  StyleBox["ReadList",
    FontFamily->"Courier"],
  " command. The command opens up the disk file and is told to read pairs of \
number into a list we have called data. Try this command, using your own file \
path, and compare it to the the contents you read above."
}], "Text"],

Cell[BoxData[
    \(data = 
      ReadList["\<Hard disk:Desktop Folder:expt1.dat\>", {Number, Number}]
        \)], "Input"],

Cell["We can plot this using the following command.", "Text"],

Cell[BoxData[
    \(ListPlot[data]\)], "Input"],

Cell[TextData[{
  "The data is from a kinetic experiment measuring current in \[Mu]A as a \
function of time in ",
  StyleBox["s",
    FontSlant->"Italic"],
  ", so we should label the plot appropriately. We also choose to join up the \
data points, colour the resulting line, put the plot into a frame, add a \
title and change the font."
}], "Text"],

Cell[BoxData[
    \(plot = 
      ListPlot[data, Frame -> True, 
        FrameLabel -> {"\<t/s\>", "\<I/\[Mu]A\>"}, \nRotateLabel -> False, 
        PlotLabel -> "\<Kinetic Data\>", DefaultFont -> {"\<Helvetica\>", 9}, 
        \nPlotStyle \[Rule] RGBColor[1, 0, 0]]\)], "Input"],

Cell["\<\
It is anticipated that the data should follow first order kinetics; we can \
check this by plotting the logarithm of the current against the time. To do \
this we first need to load up an add-on package which allows all sorts of \
graphical manipulations\
\>", "Text"],

Cell[BoxData[
    \(Needs[\ "\<Graphics`Graphics`\>"]\)], "Input"],

Cell["and then plot the intensity data on logarithmic scales.", "Text"],

Cell[BoxData[
    \(logplot = 
      LogListPlot[data, Frame -> True, \n
        FrameLabel -> {"\<t/s\>", "\<I/\[Mu]A\>"}, RotateLabel -> False, 
        PlotLabel -> "\<Kinetic Data\>", \n
        DefaultFont -> {"\<Helvetica\>", 9}, 
        PlotStyle \[Rule] {RGBColor[1, 0, 0]}]\)], "Input"],

Cell[" ", "Text",
  CellFrame->{{0, 0}, {0.5, 0}}],

Cell[CellGroupData[{

Cell["Exercises and Explorations 1", "Subsection"],

Cell[CellGroupData[{

Cell["1: Logging the data", "Subsubsection"],

Cell[TextData[{
  "You can estimate the gradient and intercept from this plot and use these \
values in the model of the current as a function of the time,",
  StyleBox[" t",
    FontSlant->"Italic"],
  ". "
}], "Text"],

Cell[BoxData[
    \(current[t_] := I\_0\ \[ExponentialE]\^\(\(-\[Lambda]\)\ t\)\)], "Input"],

Cell[TextData[{
  "Notice that you have to define the function via the \"colon-equals\" sign \
and each variable must be declared (on the left hand side) by being followed \
by an underscore. However, when you want to evaluate the function, for \
example at ",
  StyleBox["t",
    FontSlant->"Italic"],
  "=10, you simply type ",
  StyleBox["Current[10]",
    FontFamily->"Courier"],
  "; there is no underscore. Below we show how the function has been used to \
plot the model for rather poor guesses at the kinetic parameters. "
}], "Text"],

Cell[BoxData[
    \(logmodel = 
      LogPlot[current[t]\  /. {I\_0 -> \ 1.5, \[Lambda]\  -> \  .3}, {t, 0, 
          4}, Frame -> True, \nFrameLabel -> {"\<t/s\>", "\<I/\[Mu]A\>"}, 
        RotateLabel -> False, \nDefaultFont -> {"\<Helvetica\>", 9}, 
        PlotStyle \[Rule] {RGBColor[0, 0, 1]}]\)], "Input"],

Cell[TextData[{
  "Here we have introduced the idea of replacement rules when we asked for ",
  StyleBox["Current[t]",
    FontFamily->"Courier"],
  " to be evaluated for ",
  Cell[BoxData[
      StyleBox[\({I\_0 -> \ 1.5, \[Lambda]\  -> \  .3}\),
        FontSize->9,
        FontWeight->"Bold"]]],
  ". Using replacement rules means that the values for ",
  Cell[BoxData[
      FormBox[
        StyleBox[\(I\_0\),
          FontWeight->"Plain"], TraditionalForm]],
    FontFamily->"Courier",
    FontWeight->"Bold"],
  " and",
  StyleBox[" \[Lambda]",
    FontSize->9,
    FontWeight->"Bold"],
  " are only \"remembered\" during the calculation; we haven't asigned \
permanent values to them. Adjust the parameters until you find the best fit \
to the experimental data; checking your guesses by superimposing data and \
model. "
}], "Text"],

Cell[BoxData[
    \(Show[logplot, logmodel]\)], "Input"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "2: Saving a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " graphic"
}], "Subsubsection"],

Cell[TextData[{
  "With the best fit parameters plot the data and model on linear scales and \
call the result,",
  StyleBox[" ",
    FontWeight->"Bold"],
  StyleBox["fig1",
    FontFamily->"Courier"],
  ". You can transfer the resulting graphical output to a disk file using the \
following command, but with your own file path."
}], "Text"],

Cell[BoxData[
    \(fig1\  >> \ Hard\ \(disk : Desktop\ \(Folder : figure . dat\)\)\)], 
  "Input"],

Cell[TextData[{
  "Check that the file has been put in the place you expected to go. Open it \
up, by double-clicking on the icon, and you should see that instead of the \
plot the file contains the ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " code that produced it. Now close the file and return to your own \
notebook. To read the file back in and display it as a graphic is simple."
}], "Text"],

Cell[BoxData[
    \(fig1again = << \ figure.dat\)], "Input"],

Cell[BoxData[
    \(Show[fig1again]\)], "Input"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[{
  "3: Exporting a ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " plot"
}], "Subsubsection"],

Cell[TextData[{
  "What you have done above is all fine and well if you want to do is save a \
figure in a compact form. However, ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  "'s great power lies, not in its plotting abilities, but in its \
mathematical prowess. You may therefore want to export the complex functions \
you have generated with ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " to more convenient plotting programs like Excel, SigmaPlot or \
Kaleidagraph. There are a number of ways of doing this, but the most \
straightforward generates a simple data file which such programs can read. \n\
\nThe code to generate a file containing the time and the current generated \
by the theoretical model is shown below."
}], "Text"],

Cell[BoxData[
    \(OutputForm[
        TableForm[
          Table[{FortranForm[t], FortranForm[current[t]]}\  /. {
                I\_0 -> \ 1.5, \[Lambda]\  -> \  .3}, {t, 0, 4, 0.1}]]]\  >> 
      \ "\<Hard disk:Desktop Folder:mytheory.dat\>"\)], "Input"],

Cell[TextData[{
  "If you do not use the function ",
  StyleBox["OutputForm",
    FontFamily->"Courier"],
  " what is exported to ",
  StyleBox["mytheory.dat",
    FontFamily->"Arial"],
  "? What does ",
  StyleBox["FortranForm",
    FontFamily->"Courier"],
  " do and under what circumstances would it matter if you left it out of the \
code? (Hint: try to evaluate ",
  StyleBox["FortranForm[2.50 10^6]",
    FontFamily->"Courier"],
  ".)"
}], "Text"],

Cell[TextData[{
  "There are many other forms of input and output and you may wish to look \
some of these up under ",
  StyleBox["Help. ",
    FontFamily->"Arial"],
  "There are also very useful examples and ideas to be found at the Wolfram \
website; ",
  StyleBox["http://www.wolfram.co.uk/support/InputOutput/ExternalFiles/",
    FontFamily->"Courier"],
  ". However, for now we turn to the handling of three dimensional plots. \
Before doing this save your work from this part of the lab and quit the \
kernel, before opening a new Notebook to work on Section 2."
}], "Text"],

Cell[" ", "Text",
  CellFrame->{{0, 0}, {0.5, 0}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["2. Surface Plots", "Section",
  CellDingbat->None],

Cell[TextData[{
  "A surface plot arises when we have a function ",
  StyleBox["z",
    FontSlant->"Italic"],
  "=",
  StyleBox["f(x,y)",
    FontSlant->"Italic"],
  "; that is, the height",
  StyleBox[" z",
    FontSlant->"Italic"],
  " of the surface is a function of position in the ",
  StyleBox["x-y",
    FontSlant->"Italic"],
  " plane.  A good chemical example of this arises when we consider the \
quantum mechanical wave function of a particle confined to a rectangular \
planar surface of side ",
  Cell[BoxData[
      \(TraditionalForm\`l\_1\)]],
  "in the ",
  StyleBox["x",
    FontSlant->"Italic"],
  " direction and ",
  Cell[BoxData[
      \(TraditionalForm\`l\_2\)]],
  "in the ",
  StyleBox["y",
    FontSlant->"Italic"],
  " direction. (This is a reasonable model of the wave function for a \
molecule adsorbed on to a crystal surface.) The wave function, \[Psi]",
  StyleBox["(x,y)",
    FontSlant->"Italic"],
  ", for the particle is given by \n\n\[Psi](x,y)=",
  Cell[BoxData[
      \(TraditionalForm\`2\/\@\(\(l\_1\) l\_2\)\)]],
  "sin",
  Cell[BoxData[
      \(TraditionalForm\`\((\(\(n\_1\) \[Pi]x\)\/l\_1)\)\)]],
  "sin",
  Cell[BoxData[
      \(TraditionalForm\`\((\(\(n\_2\) \[Pi]y\)\/l\_2)\)\)]],
  "   ",
  Cell[BoxData[
      \(TraditionalForm\`n\_1\)]],
  "= 1, 2, ... and ",
  Cell[BoxData[
      \(TraditionalForm\`n\_2\)]],
  "= 1, 2, ...\n\nwhere ",
  Cell[BoxData[
      \(TraditionalForm\`n\_1\)]],
  "and ",
  Cell[BoxData[
      \(TraditionalForm\`n\_2\)]],
  " are the quantum numbers and the energy of each state is given by\n\nE = ",
  
  Cell[BoxData[
      \(TraditionalForm
      \`\(h\^2\/\(8  m\)\) \((n\_1\%2\/l\_1\%2 + n\_2\%2\/l\_2\%2)\)\)]],
  "\n\nIgnoring the scaling constants, you can plot the three dimensional \
surfaces using the following code. We start by looking at the wavefunction \
for a particle confined to a square surface."
}], "Text"],

Cell[BoxData[
    \(\[Psi][x_, y_] := 
      Sin[\(\(n\_1\) \[Pi]\ x\)\/l\_1] Sin[\(\(n\_2\) \[Pi]\ y\)\/l\_2]\)], 
  "Input"],

Cell[BoxData[
    \(Plot3D[
      \[Psi][x, y]\  /. {n\_1 -> 1, n\_2 -> 2, l\_1 -> 1, l\_2 -> 1}, {x, 0, 
        1}, {y, 0, 1}]\)], "Input"],

Cell[TextData[{
  "As with two dimensional plots we can change the appearance of the surface \
plot in a large number of ways. We increase the resolution of the plot, label \
the axes, change the viewpoint, the length of the edges of the box, within \
which we plot \[Psi], and colour code the height of the function. The latter \
makes high points red and low points orange. The view point was set using the \
",
  StyleBox["3D ViewPoint Selector ...",
    FontFamily->"Arial"],
  " under the ",
  StyleBox["Input",
    FontFamily->"Arial"],
  " pull down."
}], "Text"],

Cell[BoxData[
    \(Plot3D[
      \[Psi][x, y]\  /. {n\_1 -> 1, n\_2 -> 2, l\_1 -> 1, l\_2 -> 1}, {x, 0, 
        1}, {y, 0, 1}, \nPlotPoints -> 30, ColorFunction -> Hue, 
      AxesLabel -> {"\<x\>", "\<y\>", "\<\[Psi]\>"}, \n
      ViewPoint -> {5.375, \ 2.227, \ 2.965}, BoxRatios \[Rule] {1, 1, 1}]
      \)], "Input"],

Cell[" ", "Text",
  CellFrame->{{0, 0}, {0.5, 0}}],

Cell[CellGroupData[{

Cell["Exercises and Explorations 2", "Subsection"],

Cell[CellGroupData[{

Cell[TextData[{
  "1: Plotting ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\^2\)]]
}], "Subsubsection"],

Cell[TextData[{
  "The probability of finding the particle at a position ",
  StyleBox["(x, y)",
    FontSlant->"Italic"],
  " is ",
  StyleBox["equal",
    FontSlant->"Italic"],
  " to ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\^2\)]],
  ". Plot ",
  Cell[BoxData[
      \(TraditionalForm\`\[Psi]\^2\)]],
  " for the quantum numbers used above. (N.B. you should use the \
normalisation constant given above.)"
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["2: Contours", "Subsubsection"],

Cell["\<\
Plot the function from exercise 1 as a contour plot, making sure the axes are \
labelled.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["3: Degeneracy", "Subsubsection"],

Cell[TextData[{
  "Show that the wave functions for ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`n\_1\)]]],
  "= 1, ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`n\_2\)]]],
  "= 2 and for ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`n\_1\)]]],
  "= 2, ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`n\_2\)]]],
  "= 1 are degenerate when ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`l\_1\)]]],
  "= ",
  StyleBox[Cell[BoxData[
      \(TraditionalForm\`l\_2\)]]],
  ". That is, you should show that the energy for these two states is the \
same, but the wave functions are different. What is the relationship between \
these degenerate wavefunctions?"
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["4: Accidental degeneracy", "Subsubsection"],

Cell[TextData[{
  "Obvious symmetries do not always exist between degenerate states. Show \
that when ",
  Cell[BoxData[
      \(TraditionalForm\`l\_2 = 2  l\_1\)]],
  " there is degeneracy between states ",
  Cell[BoxData[
      \(TraditionalForm\`\(n\_1\ \)\)]],
  "= 1, ",
  Cell[BoxData[
      \(TraditionalForm\`n\_2\)]],
  " = 4 and  ",
  Cell[BoxData[
      \(TraditionalForm\`n\_1\)]],
  " = 2, ",
  Cell[BoxData[
      \(TraditionalForm\`n\_2\)]],
  " = 2, but that a simple symmetry operation does not connect the \
wavefunctions. If you have time, find another non-symmetrical pair of \
degenerate states for ",
  Cell[BoxData[
      \(TraditionalForm\`l\_2\)]],
  "= 2",
  Cell[BoxData[
      \(TraditionalForm\`l\_1\)]],
  ". N.B. since ",
  Cell[BoxData[
      \(TraditionalForm\`l\_1\)]],
  "\[NotEqual] ",
  Cell[BoxData[
      \(TraditionalForm\`l\_2\)]],
  "  now you will have to change the range over which you are plotting.\n\n\
Once again, save your work from this part of the lab and quit the kernel, \
before opening a new Notebook to work on Section 3."
}], "Text"],

Cell[" ", "Text",
  CellFrame->{{0, 0}, {0.5, 0}}]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["3. Three Dimensional  Surface Representations", "Section",
  CellDingbat->None],

Cell[TextData[{
  "The surface plots of section three are single valued functions of ",
  StyleBox["x",
    FontSlant->"Italic"],
  " and ",
  StyleBox["y",
    FontSlant->"Italic"],
  ", i.e. there is only one value of ",
  StyleBox["z",
    FontSlant->"Italic"],
  " for each position ",
  StyleBox["(x, y)",
    FontSlant->"Italic"],
  ". Frequently we want to make three dimensional surface representations \
where the surface is a function of ",
  StyleBox["x",
    FontSlant->"Italic"],
  ", ",
  StyleBox["y",
    FontSlant->"Italic"],
  " and ",
  StyleBox["z",
    FontSlant->"Italic"],
  " as well. A good chemical example of this are the molecular orbitals of \
hydrogen.\n\nIn this case the spherical symmetry of the problem indicates \
that we will simplify the mathematical description if we use the spherical \
coordinates ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", \[Theta] and \[Phi]. Here ",
  StyleBox["r",
    FontSlant->"Italic"],
  " is the radial distance from the origin, \[Theta] is the angle with \
respect to the ",
  StyleBox["z",
    FontSlant->"Italic"],
  "-axis and \[Phi] is the angle in the ",
  StyleBox["x-y",
    FontSlant->"Italic"],
  " plane with respect to the ",
  StyleBox["x",
    FontSlant->"Italic"],
  "-axis. (Hence ",
  StyleBox["x",
    FontSlant->"Italic"],
  " = ",
  StyleBox["r",
    FontSlant->"Italic"],
  " sin\[Theta] cos\[Phi], ",
  StyleBox["y",
    FontSlant->"Italic"],
  " = ",
  StyleBox["r",
    FontSlant->"Italic"],
  " sin\[Theta] sin\[Phi] and ",
  StyleBox["z",
    FontSlant->"Italic"],
  " = ",
  StyleBox["r",
    FontSlant->"Italic"],
  " cos\[Theta].) In these coordinates the hydrogenic wave function, \
\[CapitalPsi], can be expressed as\n\n\[CapitalPsi](",
  StyleBox["r",
    FontSlant->"Italic"],
  ",\[Theta],\[Phi]) = R(",
  StyleBox["r",
    FontSlant->"Italic"],
  ") \[CapitalTheta](\[Theta]) \[CapitalPhi](\[Phi])\n\nIn other words the \
wave function is the product of separate functions of ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", \[Theta] and \[Phi]. The angular parts, \[CapitalTheta](\[Theta]) \
\[CapitalPhi](\[Phi]), are functions called spherical harmonics and are what \
give us the hydrogenic molecular orbitals (henceforth MO). The MOs and their \
associated spherical harmonics are given below."
}], "Text",
  CellMargins->{{10, Inherited}, {Inherited, Inherited}},
  TextAlignment->Left,
  TextJustification->0],

Cell[TextData[{
  StyleBox[Cell[BoxData[
      FormBox[
        OverscriptBox[
          UnderscriptBox[GridBox[{
                {"MO", \(Spherical\ Harmonic\)}
                }], "_"], "_"], TraditionalForm]]]],
  "\n",
  StyleBox[Cell[BoxData[
      FormBox[
        StyleBox[
          UnderscriptBox[
            StyleBox[GridBox[{
                  {
                    AdjustmentBox["s",
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox["Constant",
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(p\_z\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox["cos\[Theta]",
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(p\_y\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    RowBox[{
                      AdjustmentBox["sin\[Theta]",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      AdjustmentBox["  ",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      StyleBox[
                        RowBox[{
                          AdjustmentBox["sin",
                            BoxMargins->{{-0.3, 0.3}, {0, 0}}], 
                          AdjustmentBox["\[Phi]",
                            BoxMargins->{{-0.1, 0.1}, {0, 0}}]}]]}]},
                  {
                    AdjustmentBox[\(p\_x\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox[
                      StyleBox["-",
                        SpanMaxSize->Infinity],
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(d\_\(z\^2\)\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox[\(3\ \(cos\^2\) \[Theta]\  - \ 1\),
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(d\_yz\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    RowBox[{
                      AdjustmentBox["cos\[Theta]",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      AdjustmentBox[" ",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      AdjustmentBox["sin\[Theta]",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      AdjustmentBox["  ",
                        BoxMargins->{{-0.1, 0.1}, {0, 0}}], 
                      StyleBox[
                        RowBox[{
                          AdjustmentBox["sin",
                            BoxMargins->{{-0.3, 0.3}, {0, 0}}], 
                          AdjustmentBox["\[Phi]",
                            BoxMargins->{{-0.1, 0.1}, {0, 0}}]}]]}]},
                  {
                    AdjustmentBox[\(d\_zx\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox[
                      StyleBox["-",
                        SpanMaxSize->Infinity],
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(d\_\(x\^2 - y\^2\)\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    AdjustmentBox[\(\ \(sin\^2\) \[Theta]\ cos2\[Phi]\),
                      BoxMargins->{{-0.1, 0.1}, {0, 0}}]},
                  {
                    AdjustmentBox[\(d\_xy\),
                      BoxMargins->{{0.1, -0.1}, {0, 0}}], 
                    StyleBox["-",
                      SpanMaxSize->Infinity]}
                  }],
              SpanMaxSize->Infinity], "_"],
          SpanMaxSize->Infinity], TraditionalForm]]]]
}], "Text",
  CellMargins->{{10, Inherited}, {Inherited, Inherited}},
  PageBreakAbove->Automatic,
  TextAlignment->Center,
  TextJustification->0],

Cell["\<\
We have created the three dimensional renditions of the spherical harmonics \
by loading  the following package\
\>", "Text"],

Cell[BoxData[
    \(Needs["\<Graphics`ParametricPlot3D`\>"]\)], "Input"],

Cell[TextData[{
  "which contains code to plot surfaces in three dimensions using spherical \
coordinates. Below is the code to plot the ",
  StyleBox[Cell[BoxData[
      FormBox[
        StyleBox[\(p\_z\),
          FontSlant->"Italic"], TraditionalForm]]]],
  "orbital. The radial distance from the origin, ",
  StyleBox["r",
    FontSlant->"Italic"],
  ", is being plotted as a function of \[Theta] and \[Phi]. In this case \
according to the MO table ",
  StyleBox["r",
    FontSlant->"Italic"],
  " \[Proportional] cos\[Theta], however, because of the way in which ",
  StyleBox["SphericalPlot3D",
    FontFamily->"Courier"],
  " does its computations, it is necessary to make ",
  StyleBox["r",
    FontSlant->"Italic"],
  " \[Proportional] \[VerticalSeparator]cos\[Theta]\[VerticalSeparator]. This \
is true for plotting all the spherical harmonics using this function. Opening \
up the ",
  StyleBox["Help",
    FontFamily->"Arial"],
  " file on ",
  StyleBox["ParametricPlot3D",
    FontFamily->"Courier"],
  " you may be able to work out why this is."
}], "Text"],

Cell[BoxData[
    \(SphericalPlot3D[\ Abs[Cos[\[Theta]]], {\[Theta], 0, \[Pi]}, 
      \ {\[Phi], \ 0, \ \ 2\ \[Pi]}]\)], "Input"],

Cell["\<\
The code which follows labels things neatly, and increases the number of \
points being used to plot the surface to give it a smoother appearance.\
\>", "Text"],

Cell[CellGroupData[{

Cell[BoxData[
    \(SphericalPlot3D[\ Abs[Cos[\[Theta]]], {\[Theta], 0, \[Pi]}, 
      \ {\[Phi], \ 0, \ \ 2\ \[Pi]}, PlotRange -> All, \n\t
      AxesLabel -> {"\<x\>", "\<y\>", "\<z\>"}, 
      PlotLabel\  -> \*"\"\<\!\(p\_z\)-orbital\>\"", PlotPoints -> 25, \n\t
      DefaultFont -> {"\<Helvetica\>", 8}, \n\t
      AxesEdge \[Rule] {{\(-1\), \(-1\)}, {1, \(-1\)}, {\(-1\), \(-1\)}}]\)], 
  "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .33922 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.31746 0.0080766 0.31746 [
[ 0 0 0 0 ]
[ 1 .33922 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .33922 L
0 .33922 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.0080766 0.321429 0.331141 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.0567234 1.16608 0 1.16608 [
[.5 1.098 -17.375 0 ]
[.5 1.098 17.375 10 ]
[.09072 .22988 -13.7418 -10 ]
[.09072 .22988 .19568 0 ]
[.21906 .17566 -19.4345 -10 ]
[.21906 .17566 1.19055 0 ]
[.3545 .11847 -7.58187 -10 ]
[.3545 .11847 .85563 0 ]
[.49765 .05807 -12.9305 -10 ]
[.49765 .05807 2.19452 0 ]
[.64919 -0.00583 -6.8447 -10 ]
[.64919 -0.00583 1.5928 0 ]
[.30907 .06148 -7.13257 -10 ]
[.30907 .06148 .80493 0 ]
[.6933 .00644 0 -7.08817 ]
[.6933 .00644 13.9375 2.91183 ]
[.76325 .12052 0 -6.95638 ]
[.76325 .12052 20.625 3.04362 ]
[.82642 .22355 0 -6.84024 ]
[.82642 .22355 8.4375 3.15976 ]
[.88375 .31706 0 -6.73712 ]
[.88375 .31706 15.125 3.26288 ]
[.93601 .40232 0 -6.64494 ]
[.93601 .40232 8.4375 3.35506 ]
[.89481 .19838 0 -6.84024 ]
[.89481 .19838 7.9375 3.15976 ]
[.07325 .26228 -13.9375 -2.90855 ]
[.07325 .26228 0 7.09145 ]
[.05474 .37813 -20.625 -3.02288 ]
[.05474 .37813 0 6.97712 ]
[.03462 .5041 -8.4375 -3.14821 ]
[.03462 .5041 0 6.85179 ]
[.01266 .64157 -15.125 -3.28621 ]
[.01266 .64157 0 6.71379 ]
[-0.01139 .79218 -8.4375 -3.43891 ]
[-0.01139 .79218 0 6.56109 ]
[-0.03373 .52941 -7.9375 -3.14821 ]
[-0.03373 .52941 0 6.85179 ]
[ 0 0 0 0 ]
[ 1 1.0855 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.098 -78.375 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(s) show
66.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
72.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
93.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.08846 .24555 m
.67245 0 L
s
.10087 .24033 m
.10595 .24556 L
s
gsave
.09072 .22988 -74.7418 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.22871 .18658 m
.23354 .19204 L
s
gsave
.21906 .17566 -80.4345 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.36358 .12987 m
.36813 .13557 L
s
gsave
.3545 .11847 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.50609 .06995 m
.51031 .07589 L
s
gsave
.49765 .05807 -73.9305 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.65689 .00654 m
.66074 .01273 L
s
gsave
.64919 -0.00583 -67.8447 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12591 .22981 m
.12892 .23297 L
s
.1512 .21917 m
.15419 .22236 L
s
.17676 .20842 m
.17972 .21164 L
s
.2026 .19756 m
.20553 .20081 L
s
.25511 .17548 m
.25797 .17879 L
s
.28179 .16426 m
.28462 .1676 L
s
.30875 .15292 m
.31155 .15628 L
s
.33602 .14146 m
.33878 .14485 L
s
.39145 .11815 m
.39414 .1216 L
s
.41963 .1063 m
.42229 .10978 L
s
.44813 .09432 m
.45074 .09783 L
s
.47695 .0822 m
.47952 .08574 L
s
.53556 .05756 m
.53805 .06115 L
s
.56537 .04502 m
.56782 .04865 L
s
.59553 .03235 m
.59793 .036 L
s
.62603 .01952 m
.62839 .0232 L
s
gsave
.30907 .06148 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.67245 0 m
.92713 .41497 L
s
.67985 .01205 m
.67313 .01486 L
s
gsave
.6933 .00644 -61 -11.0882 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.74968 .12583 m
.74289 .12848 L
s
gsave
.76325 .12052 -61 -10.9564 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.81274 .22858 m
.8059 .2311 L
s
gsave
.82642 .22355 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.86998 .32185 m
.86309 .32424 L
s
gsave
.88375 .31706 -61 -10.7371 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.92216 .40688 m
.91524 .40915 L
s
gsave
.93601 .40232 -61 -10.6449 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.69441 .03577 m
.69037 .03744 L
s
.70866 .059 m
.70461 .06064 L
s
.72262 .08174 m
.71856 .08337 L
s
.73629 .10401 m
.73222 .10562 L
s
.7628 .14721 m
.75872 .14878 L
s
.77566 .16816 m
.77157 .16972 L
s
.78826 .1887 m
.78417 .19024 L
s
.80062 .20884 m
.79652 .21036 L
s
.82463 .24795 m
.82052 .24945 L
s
.83629 .26695 m
.83217 .26843 L
s
.84773 .2856 m
.84361 .28706 L
s
.85896 .30389 m
.85483 .30534 L
s
.8808 .33948 m
.87666 .3409 L
s
.89142 .35678 m
.88728 .35819 L
s
.90185 .37378 m
.8977 .37518 L
s
.91209 .39048 m
.90795 .39186 L
s
gsave
.89481 .19838 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.08846 .24555 m
0 .80374 L
s
.0867 .25665 m
.09342 .25384 L
s
gsave
.07325 .26228 -74.9375 -6.90855 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.0683 .37277 m
.07507 .37009 L
s
gsave
.05474 .37813 -81.625 -7.02288 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.04829 .49904 m
.05512 .49651 L
s
gsave
.03462 .5041 -69.4375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.02645 .63684 m
.03334 .63448 L
s
gsave
.01266 .64157 -76.125 -7.28621 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00252 .78784 m
.00948 .78567 L
s
gsave
-0.01139 .79218 -69.4375 -7.43891 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08314 .27912 m
.08718 .27745 L
s
.07952 .30196 m
.08356 .30031 L
s
.07584 .32518 m
.07989 .32354 L
s
.0721 .34878 m
.07616 .34715 L
s
.06443 .39717 m
.0685 .39558 L
s
.0605 .42199 m
.06458 .42042 L
s
.0565 .44723 m
.06058 .44568 L
s
.05243 .47291 m
.05652 .47137 L
s
.04407 .52563 m
.04818 .52413 L
s
.03978 .55269 m
.0439 .55121 L
s
.03542 .58024 m
.03954 .57878 L
s
.03097 .60828 m
.0351 .60685 L
s
.02184 .66593 m
.02598 .66453 L
s
.01714 .69556 m
.0213 .69418 L
s
.01236 .72574 m
.01652 .72439 L
s
.00749 .7565 m
.01165 .75517 L
s
gsave
-0.03373 .52941 -68.9375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.40037 1.0855 L
s
.40037 1.0855 m
.41799 .58158 L
s
.41799 .58158 m
.08846 .24555 L
s
.67245 0 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.67245 0 m
.08846 .24555 L
s
.41799 .58158 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.40037 1.0855 L
s
.40037 1.0855 m
.41799 .58158 L
s
0 0 m
1 0 L
1 1.0855 L
0 1.0855 L
closepath
clip
newpath
.5 Mabswid
.46959 .53525 m .46274 .57135 L .52027 .57679 L p .657 g
F P
0 g
s
.52027 .57679 m .52092 .54021 L .46959 .53525 L p .657 g
F P
0 g
s
.51972 .61343 m .52027 .57679 L .46274 .57135 L p .645 g
F P
0 g
s
.52092 .54021 m .52027 .57679 L .57812 .57321 L p .629 g
F P
0 g
s
.52027 .57679 m .51972 .61343 L .58292 .60961 L p .613 g
F P
0 g
s
.58292 .60961 m .57812 .57321 L .52027 .57679 L p .613 g
F P
0 g
s
.46274 .57135 m .45685 .60762 L .51972 .61343 L p .645 g
F P
0 g
s
.51925 .64975 m .51972 .61343 L .45685 .60762 L p .632 g
F P
0 g
s
.51972 .61343 m .51925 .64975 L .58688 .64576 L p .597 g
F P
0 g
s
.58688 .64576 m .58292 .60961 L .51972 .61343 L p .597 g
F P
0 g
s
.47733 .49972 m .46959 .53525 L .52092 .54021 L p .669 g
F P
0 g
s
.52092 .54021 m .52166 .5041 L .47733 .49972 L p .669 g
F P
0 g
s
.52166 .5041 m .52092 .54021 L .57254 .53695 L p .644 g
F P
0 g
s
.57812 .57321 m .57254 .53695 L .52092 .54021 L p .629 g
F P
0 g
s
.57254 .53695 m .57812 .57321 L .63324 .56076 L p .602 g
F P
0 g
s
.57812 .57321 m .58292 .60961 L .64322 .59629 L p .583 g
F P
0 g
s
.64322 .59629 m .63324 .56076 L .57812 .57321 L p .583 g
F P
0 g
s
.58292 .60961 m .58688 .64576 L .65147 .63181 L p .563 g
F P
0 g
s
.65147 .63181 m .64322 .59629 L .58292 .60961 L p .563 g
F P
0 g
s
.42135 .52227 m .40857 .5571 L .46274 .57135 L p .688 g
F P
0 g
s
.46274 .57135 m .46959 .53525 L .42135 .52227 L p .688 g
F P
0 g
s
.45685 .60762 m .46274 .57135 L .40857 .5571 L p .679 g
F P
0 g
s
.57254 .53695 m .56624 .50122 L .52166 .5041 L p .644 g
F P
0 g
s
.56624 .50122 m .57254 .53695 L .62165 .52561 L p .62 g
F P
0 g
s
.63324 .56076 m .62165 .52561 L .57254 .53695 L p .602 g
F P
0 g
s
.45685 .60762 m .45198 .64367 L .51925 .64975 L p .632 g
F P
0 g
s
.5189 .68534 m .51925 .64975 L .45198 .64367 L p .617 g
F P
0 g
s
.51925 .64975 m .5189 .68534 L .58995 .68123 L p .58 g
F P
0 g
s
.58995 .68123 m .58688 .64576 L .51925 .64975 L p .58 g
F P
0 g
s
.40857 .5571 m .39755 .59237 L .45685 .60762 L p .679 g
F P
0 g
s
.45198 .64367 m .45685 .60762 L .39755 .59237 L p .668 g
F P
0 g
s
.43576 .4883 m .42135 .52227 L .46959 .53525 L p .697 g
F P
0 g
s
.46959 .53525 m .47733 .49972 L .43576 .4883 L p .697 g
F P
0 g
s
.48587 .46517 m .47733 .49972 L .52166 .5041 L p .681 g
F P
0 g
s
.52166 .5041 m .52247 .46885 L .48587 .46517 L p .681 g
F P
0 g
s
.52247 .46885 m .52166 .5041 L .56624 .50122 L p .658 g
F P
0 g
s
.58688 .64576 m .58995 .68123 L .65788 .6669 L p .542 g
F P
0 g
s
.65788 .6669 m .65147 .63181 L .58688 .64576 L p .542 g
F P
0 g
s
.56624 .50122 m .55928 .46643 L .52247 .46885 L p .658 g
F P
0 g
s
.55928 .46643 m .56624 .50122 L .60859 .49123 L p .638 g
F P
0 g
s
.62165 .52561 m .60859 .49123 L .56624 .50122 L p .62 g
F P
0 g
s
.39755 .59237 m .38844 .6277 L .45198 .64367 L p .668 g
F P
0 g
s
.4482 .67909 m .45198 .64367 L .38844 .6277 L p .656 g
F P
0 g
s
.45198 .64367 m .4482 .67909 L .5189 .68534 L p .617 g
F P
0 g
s
.45164 .45556 m .43576 .4883 L .47733 .49972 L p .705 g
F P
0 g
s
.47733 .49972 m .48587 .46517 L .45164 .45556 L p .705 g
F P
0 g
s
.51865 .71979 m .5189 .68534 L .4482 .67909 L p .601 g
F P
0 g
s
.5189 .68534 m .51865 .71979 L .59209 .71564 L p .561 g
F P
0 g
s
.59209 .71564 m .58995 .68123 L .5189 .68534 L p .561 g
F P
0 g
s
.62165 .52561 m .63324 .56076 L .6826 .5399 L p .575 g
F P
0 g
s
.63324 .56076 m .64322 .59629 L .69738 .57394 L p .552 g
F P
0 g
s
.69738 .57394 m .6826 .5399 L .63324 .56076 L p .552 g
F P
0 g
s
.60859 .49123 m .62165 .52561 L .66548 .50664 L p .598 g
F P
0 g
s
.6826 .5399 m .66548 .50664 L .62165 .52561 L p .575 g
F P
0 g
s
.49514 .43199 m .48587 .46517 L .52247 .46885 L p .692 g
F P
0 g
s
.52247 .46885 m .52335 .43488 L .49514 .43199 L p .692 g
F P
0 g
s
.52335 .43488 m .52247 .46885 L .55928 .46643 L p .673 g
F P
0 g
s
.58995 .68123 m .59209 .71564 L .66235 .70114 L p .519 g
F P
0 g
s
.66235 .70114 m .65788 .6669 L .58995 .68123 L p .519 g
F P
0 g
s
.64322 .59629 m .65147 .63181 L .70963 .60837 L p .528 g
F P
0 g
s
.70963 .60837 m .69738 .57394 L .64322 .59629 L p .528 g
F P
0 g
s
.37897 .50182 m .36079 .53459 L .40857 .5571 L p .723 g
F P
0 g
s
.40857 .5571 m .42135 .52227 L .37897 .50182 L p .723 g
F P
0 g
s
.39755 .59237 m .40857 .5571 L .36079 .53459 L p .716 g
F P
0 g
s
.55928 .46643 m .55173 .43298 L .52335 .43488 L p .673 g
F P
0 g
s
.55173 .43298 m .55928 .46643 L .59418 .45802 L p .657 g
F P
0 g
s
.60859 .49123 m .59418 .45802 L .55928 .46643 L p .638 g
F P
0 g
s
.39941 .47034 m .37897 .50182 L .42135 .52227 L p .728 g
F P
0 g
s
.42135 .52227 m .43576 .4883 L .39941 .47034 L p .728 g
F P
0 g
s
.59418 .45802 m .60859 .49123 L .64622 .47456 L p .62 g
F P
0 g
s
.66548 .50664 m .64622 .47456 L .60859 .49123 L p .598 g
F P
0 g
s
.38844 .6277 m .38136 .66267 L .4482 .67909 L p .656 g
F P
0 g
s
.44557 .71347 m .4482 .67909 L .38136 .66267 L p .641 g
F P
0 g
s
.4482 .67909 m .44557 .71347 L .51865 .71979 L p .601 g
F P
0 g
s
.46883 .42446 m .45164 .45556 L .48587 .46517 L p .712 g
F P
0 g
s
.48587 .46517 m .49514 .43199 L .46883 .42446 L p .712 g
F P
0 g
s
.36079 .53459 m .34507 .56824 L .39755 .59237 L p .716 g
F P
0 g
s
.38844 .6277 m .39755 .59237 L .34507 .56824 L p .708 g
F P
0 g
s
.42185 .4405 m .39941 .47034 L .43576 .4883 L p .732 g
F P
0 g
s
.43576 .4883 m .45164 .45556 L .42185 .4405 L p .732 g
F P
0 g
s
.65147 .63181 m .65788 .6669 L .71917 .64278 L p .501 g
F P
0 g
s
.71917 .64278 m .70963 .60837 L .65147 .63181 L p .501 g
F P
0 g
s
.51851 .75269 m .51865 .71979 L .44557 .71347 L p .581 g
F P
0 g
s
.51865 .71979 m .51851 .75269 L .59325 .74856 L p .539 g
F P
0 g
s
.59325 .74856 m .59209 .71564 L .51865 .71979 L p .539 g
F P
0 g
s
.59418 .45802 m .57858 .42639 L .55173 .43298 L p .657 g
F P
0 g
s
.57858 .42639 m .59418 .45802 L .62506 .44404 L p .643 g
F P
0 g
s
.64622 .47456 m .62506 .44404 L .59418 .45802 L p .62 g
F P
0 g
s
.34507 .56824 m .33205 .60239 L .38844 .6277 L p .708 g
F P
0 g
s
.38136 .66267 m .38844 .6277 L .33205 .60239 L p .699 g
F P
0 g
s
.44605 .41268 m .42185 .4405 L .45164 .45556 L p .734 g
F P
0 g
s
.45164 .45556 m .46883 .42446 L .44605 .41268 L p .734 g
F P
0 g
s
.59209 .71564 m .59325 .74856 L .66477 .73412 L p .493 g
F P
0 g
s
.66477 .73412 m .66235 .70114 L .59209 .71564 L p .493 g
F P
0 g
s
.50504 .40061 m .49514 .43199 L .52335 .43488 L p .703 g
F P
0 g
s
.52335 .43488 m .52429 .40262 L .50504 .40061 L p .703 g
F P
0 g
s
.52429 .40262 m .52335 .43488 L .55173 .43298 L p .689 g
F P
0 g
s
.55173 .43298 m .54366 .4013 L .52429 .40262 L p .689 g
F P
0 g
s
.54366 .4013 m .55173 .43298 L .57858 .42639 L p .676 g
F P
0 g
s
.38136 .66267 m .37643 .69687 L .44557 .71347 L p .641 g
F P
0 g
s
.44415 .7464 m .44557 .71347 L .37643 .69687 L p .624 g
F P
0 g
s
.44557 .71347 m .44415 .7464 L .51851 .75269 L p .581 g
F P
0 g
s
.48714 .39538 m .46883 .42446 L .49514 .43199 L p .719 g
F P
0 g
s
.49514 .43199 m .50504 .40061 L .48714 .39538 L p .719 g
F P
0 g
s
.65788 .6669 m .66235 .70114 L .72581 .67673 L p .472 g
F P
0 g
s
.72581 .67673 m .71917 .64278 L .65788 .6669 L p .472 g
F P
0 g
s
.64622 .47456 m .66548 .50664 L .70131 .48091 L p .577 g
F P
0 g
s
.66548 .50664 m .6826 .5399 L .7232 .5115 L p .549 g
F P
0 g
s
.7232 .5115 m .70131 .48091 L .66548 .50664 L p .549 g
F P
0 g
s
.62506 .44404 m .64622 .47456 L .67679 .45203 L p .604 g
F P
0 g
s
.70131 .48091 m .67679 .45203 L .64622 .47456 L p .577 g
F P
0 g
s
.6826 .5399 m .69738 .57394 L .74217 .54343 L p .519 g
F P
0 g
s
.74217 .54343 m .7232 .5115 L .6826 .5399 L p .519 g
F P
0 g
s
.57858 .42639 m .56194 .39672 L .54366 .4013 L p .676 g
F P
0 g
s
.56194 .39672 m .57858 .42639 L .60222 .41545 L p .666 g
F P
0 g
s
.62506 .44404 m .60222 .41545 L .57858 .42639 L p .643 g
F P
0 g
s
.33205 .60239 m .32191 .63661 L .38136 .66267 L p .699 g
F P
0 g
s
.37643 .69687 m .38136 .66267 L .32191 .63661 L p .686 g
F P
0 g
s
.37058 .44671 m .34512 .47482 L .37897 .50182 L p .763 g
F P
0 g
s
.37897 .50182 m .39941 .47034 L .37058 .44671 L p .763 g
F P
0 g
s
.36079 .53459 m .37897 .50182 L .34512 .47482 L p .762 g
F P
0 g
s
.47172 .38725 m .44605 .41268 L .46883 .42446 L p .736 g
F P
0 g
s
.46883 .42446 m .48714 .39538 L .47172 .38725 L p .736 g
F P
0 g
s
.39842 .42077 m .37058 .44671 L .39941 .47034 L p .761 g
F P
0 g
s
.39941 .47034 m .42185 .4405 L .39842 .42077 L p .761 g
F P
0 g
s
.5185 .7836 m .51851 .75269 L .44415 .7464 L p .559 g
F P
0 g
s
.51851 .75269 m .5185 .7836 L .59339 .77954 L p .513 g
F P
0 g
s
.59339 .77954 m .59325 .74856 L .51851 .75269 L p .513 g
F P
0 g
s
.60222 .41545 m .62506 .44404 L .64995 .4252 L p .631 g
F P
0 g
s
.67679 .45203 m .64995 .4252 L .62506 .44404 L p .604 g
F P
0 g
s
.69738 .57394 m .70963 .60837 L .75793 .5763 L p .487 g
F P
0 g
s
.75793 .5763 m .74217 .54343 L .69738 .57394 L p .487 g
F P
0 g
s
.34512 .47482 m .32236 .50477 L .36079 .53459 L p .762 g
F P
0 g
s
.34507 .56824 m .36079 .53459 L .32236 .50477 L p .76 g
F P
0 g
s
.4283 .39733 m .39842 .42077 L .42185 .4405 L p .758 g
F P
0 g
s
.42185 .4405 m .44605 .41268 L .4283 .39733 L p .758 g
F P
0 g
s
.59325 .74856 m .59339 .77954 L .66506 .76537 L p .462 g
F P
0 g
s
.66506 .76537 m .66477 .73412 L .59325 .74856 L p .462 g
F P
0 g
s
.66235 .70114 m .66477 .73412 L .72942 .70979 L p .438 g
F P
0 g
s
.72942 .70979 m .72581 .67673 L .66235 .70114 L p .438 g
F P
0 g
s
.51547 .37145 m .50504 .40061 L .52429 .40262 L p .714 g
F P
0 g
s
.52429 .40262 m .52528 .37249 L .51547 .37145 L p .714 g
F P
0 g
s
.52528 .37249 m .52429 .40262 L .54366 .4013 L p .705 g
F P
0 g
s
.32236 .50477 m .30263 .53618 L .34507 .56824 L p .76 g
F P
0 g
s
.33205 .60239 m .34507 .56824 L .30263 .53618 L p .755 g
F P
0 g
s
.54366 .4013 m .53516 .37181 L .52528 .37249 L p .705 g
F P
0 g
s
.53516 .37181 m .54366 .4013 L .56194 .39672 L p .696 g
F P
0 g
s
.37643 .69687 m .37375 .72986 L .44415 .7464 L p .624 g
F P
0 g
s
.44397 .77743 m .44415 .7464 L .37375 .72986 L p .603 g
F P
0 g
s
.44415 .7464 m .44397 .77743 L .5185 .7836 L p .559 g
F P
0 g
s
.50637 .36875 m .48714 .39538 L .50504 .40061 L p .724 g
F P
0 g
s
.50504 .40061 m .51547 .37145 L .50637 .36875 L p .724 g
F P
0 g
s
.60222 .41545 m .57796 .38916 L .56194 .39672 L p .666 g
F P
0 g
s
.57796 .38916 m .60222 .41545 L .62113 .40077 L p .658 g
F P
0 g
s
.64995 .4252 m .62113 .40077 L .60222 .41545 L p .631 g
F P
0 g
s
.70963 .60837 m .71917 .64278 L .77022 .60971 L p .452 g
F P
0 g
s
.77022 .60971 m .75793 .5763 L .70963 .60837 L p .452 g
F P
0 g
s
.32191 .63661 m .31483 .67049 L .37643 .69687 L p .686 g
F P
0 g
s
.37375 .72986 m .37643 .69687 L .31483 .67049 L p .671 g
F P
0 g
s
.45983 .37669 m .4283 .39733 L .44605 .41268 L p .754 g
F P
0 g
s
.44605 .41268 m .47172 .38725 L .45983 .37669 L p .754 g
F P
0 g
s
.56194 .39672 m .54446 .36944 L .53516 .37181 L p .696 g
F P
0 g
s
.54446 .36944 m .56194 .39672 L .57796 .38916 L p .69 g
F P
0 g
s
.49859 .36456 m .47172 .38725 L .48714 .39538 L p .735 g
F P
0 g
s
.48714 .39538 m .50637 .36875 L .49859 .36456 L p .735 g
F P
0 g
s
.30263 .53618 m .28623 .56867 L .33205 .60239 L p .755 g
F P
0 g
s
.32191 .63661 m .33205 .60239 L .28623 .56867 L p .749 g
F P
0 g
s
.64995 .4252 m .67679 .45203 L .69815 .42478 L p .59 g
F P
0 g
s
.67679 .45203 m .70131 .48091 L .72665 .44966 L p .556 g
F P
0 g
s
.72665 .44966 m .69815 .42478 L .67679 .45203 L p .556 g
F P
0 g
s
.62113 .40077 m .64995 .4252 L .66712 .40254 L p .622 g
F P
0 g
s
.69815 .42478 m .66712 .40254 L .64995 .4252 L p .59 g
F P
0 g
s
.57796 .38916 m .55256 .36554 L .54446 .36944 L p .69 g
F P
0 g
s
.55256 .36554 m .57796 .38916 L .59067 .37906 L p .686 g
F P
0 g
s
.62113 .40077 m .59067 .37906 L .57796 .38916 L p .658 g
F P
0 g
s
.5186 .81208 m .5185 .7836 L .44397 .77743 L p .531 g
F P
0 g
s
.5185 .7836 m .5186 .81208 L .59248 .80815 L p .482 g
F P
0 g
s
.59248 .80815 m .59339 .77954 L .5185 .7836 L p .482 g
F P
0 g
s
.71917 .64278 m .72581 .67673 L .77881 .64323 L p .412 g
F P
0 g
s
.77881 .64323 m .77022 .60971 L .71917 .64278 L p .412 g
F P
0 g
s
.70131 .48091 m .7232 .5115 L .75221 .47687 L p .52 g
F P
0 g
s
.75221 .47687 m .72665 .44966 L .70131 .48091 L p .52 g
F P
0 g
s
.49264 .35914 m .45983 .37669 L .47172 .38725 L p .746 g
F P
0 g
s
.47172 .38725 m .49859 .36456 L .49264 .35914 L p .746 g
F P
0 g
s
.66477 .73412 m .66506 .76537 L .72986 .7415 L p .398 g
F P
0 g
s
.72986 .7415 m .72942 .70979 L .66477 .73412 L p .398 g
F P
0 g
s
.59067 .37906 m .62113 .40077 L .63399 .38321 L p .653 g
F P
0 g
s
.66712 .40254 m .63399 .38321 L .62113 .40077 L p .622 g
F P
0 g
s
.41684 .3793 m .38305 .39748 L .39842 .42077 L p .783 g
F P
0 g
s
.39842 .42077 m .4283 .39733 L .41684 .3793 L p .783 g
F P
0 g
s
.37058 .44671 m .39842 .42077 L .38305 .39748 L p .793 g
F P
0 g
s
.38305 .39748 m .35137 .41867 L .37058 .44671 L p .793 g
F P
0 g
s
.34512 .47482 m .37058 .44671 L .35137 .41867 L p .801 g
F P
0 g
s
.59339 .77954 m .59248 .80815 L .66316 .79445 L p .424 g
F P
0 g
s
.66316 .79445 m .66506 .76537 L .59339 .77954 L p .424 g
F P
0 g
s
.7232 .5115 m .74217 .54343 L .77445 .50609 L p .481 g
F P
0 g
s
.77445 .50609 m .75221 .47687 L .7232 .5115 L p .481 g
F P
0 g
s
.45229 .36437 m .41684 .3793 L .4283 .39733 L p .771 g
F P
0 g
s
.4283 .39733 m .45983 .37669 L .45229 .36437 L p .771 g
F P
0 g
s
.52528 .37249 m .52632 .34495 L .51547 .37145 L closepath p .725 g
F P
0 g
s
.53516 .37181 m .52632 .34495 L .52528 .37249 L closepath p .721 g
F P
0 g
s
.35137 .41867 m .32225 .44262 L .34512 .47482 L p .801 g
F P
0 g
s
.32236 .50477 m .34512 .47482 L .32225 .44262 L p .807 g
F P
0 g
s
.37375 .72986 m .37343 .7612 L .44397 .77743 L p .603 g
F P
0 g
s
.4451 .80611 m .44397 .77743 L .37343 .7612 L p .577 g
F P
0 g
s
.44397 .77743 m .4451 .80611 L .5186 .81208 L p .531 g
F P
0 g
s
.28623 .56867 m .27342 .60184 L .32191 .63661 L p .749 g
F P
0 g
s
.31483 .67049 m .32191 .63661 L .27342 .60184 L p .739 g
F P
0 g
s
.54446 .36944 m .52632 .34495 L .53516 .37181 L closepath p .718 g
F P
0 g
s
.31483 .67049 m .31099 .70357 L .37375 .72986 L p .671 g
F P
0 g
s
.37343 .7612 m .37375 .72986 L .31099 .70357 L p .65 g
F P
0 g
s
.51547 .37145 m .52632 .34495 L .50637 .36875 L closepath p .729 g
F P
0 g
s
.59067 .37906 m .55893 .36035 L .55256 .36554 L p .686 g
F P
0 g
s
.55893 .36035 m .59067 .37906 L .59917 .36703 L p .683 g
F P
0 g
s
.63399 .38321 m .59917 .36703 L .59067 .37906 L p .653 g
F P
0 g
s
.55256 .36554 m .52632 .34495 L .54446 .36944 L closepath p .716 g
F P
0 g
s
.50637 .36875 m .52632 .34495 L .49859 .36456 L closepath p .733 g
F P
0 g
s
.48894 .35286 m .45229 .36437 L .45983 .37669 L p .757 g
F P
0 g
s
.45983 .37669 m .49264 .35914 L .48894 .35286 L p .757 g
F P
0 g
s
.32225 .44262 m .2961 .46905 L .32236 .50477 L p .807 g
F P
0 g
s
.30263 .53618 m .32236 .50477 L .2961 .46905 L p .811 g
F P
0 g
s
.55893 .36035 m .52632 .34495 L .55256 .36554 L closepath p .715 g
F P
0 g
s
.74217 .54343 m .75793 .5763 L .79299 .53694 L p .437 g
F P
0 g
s
.79299 .53694 m .77445 .50609 L .74217 .54343 L p .437 g
F P
0 g
s
.49859 .36456 m .52632 .34495 L .49264 .35914 L closepath p .736 g
F P
0 g
s
.72581 .67673 m .72942 .70979 L .78347 .67639 L p .366 g
F P
0 g
s
.78347 .67639 m .77881 .64323 L .72581 .67673 L p .366 g
F P
0 g
s
.56313 .35422 m .52632 .34495 L .55893 .36035 L closepath p .714 g
F P
0 g
s
.59917 .36703 m .56313 .35422 L .55893 .36035 L p .683 g
F P
0 g
s
.49264 .35914 m .52632 .34495 L .48894 .35286 L closepath p .74 g
F P
0 g
s
.59917 .36703 m .63399 .38321 L .63977 .36385 L p .651 g
F P
0 g
s
.63399 .38321 m .66712 .40254 L .67516 .3774 L p .616 g
F P
0 g
s
.67516 .3774 m .63977 .36385 L .63399 .38321 L p .616 g
F P
0 g
s
.56313 .35422 m .59917 .36703 L .60283 .35386 L p .684 g
F P
0 g
s
.63977 .36385 m .60283 .35386 L .59917 .36703 L p .651 g
F P
0 g
s
.2961 .46905 m .27332 .49764 L .30263 .53618 L p .811 g
F P
0 g
s
.28623 .56867 m .30263 .53618 L .27332 .49764 L p .812 g
F P
0 g
s
.27342 .60184 m .26448 .63525 L .31483 .67049 L p .739 g
F P
0 g
s
.31099 .70357 m .31483 .67049 L .26448 .63525 L p .724 g
F P
0 g
s
.66712 .40254 m .69815 .42478 L .70851 .39437 L p .578 g
F P
0 g
s
.70851 .39437 m .67516 .3774 L .66712 .40254 L p .578 g
F P
0 g
s
.66506 .76537 m .66316 .79445 L .72701 .77138 L p .35 g
F P
0 g
s
.72701 .77138 m .72986 .7415 L .66506 .76537 L p .35 g
F P
0 g
s
.51884 .83766 m .5186 .81208 L .4451 .80611 L p .494 g
F P
0 g
s
.5186 .81208 m .51884 .83766 L .59048 .83393 L p .442 g
F P
0 g
s
.59048 .83393 m .59248 .80815 L .5186 .81208 L p .442 g
F P
0 g
s
.48776 .34612 m .44969 .35106 L .45229 .36437 L p .766 g
F P
0 g
s
.45229 .36437 m .48894 .35286 L .48776 .34612 L p .766 g
F P
0 g
s
.41684 .3793 m .45229 .36437 L .44969 .35106 L p .788 g
F P
0 g
s
.44969 .35106 m .41261 .35971 L .41684 .3793 L p .788 g
F P
0 g
s
.38305 .39748 m .41684 .3793 L .41261 .35971 L p .808 g
F P
0 g
s
.75793 .5763 m .77022 .60971 L .80749 .56905 L p .387 g
F P
0 g
s
.80749 .56905 m .79299 .53694 L .75793 .5763 L p .387 g
F P
0 g
s
.56484 .34754 m .52632 .34495 L .56313 .35422 L closepath p .715 g
F P
0 g
s
.60283 .35386 m .56484 .34754 L .56313 .35422 L p .684 g
F P
0 g
s
.48894 .35286 m .52632 .34495 L .48776 .34612 L closepath p .743 g
F P
0 g
s
.69815 .42478 m .72665 .44966 L .73931 .41459 L p .535 g
F P
0 g
s
.73931 .41459 m .70851 .39437 L .69815 .42478 L p .535 g
F P
0 g
s
.41261 .35971 m .37704 .37201 L .38305 .39748 L p .808 g
F P
0 g
s
.35137 .41867 m .38305 .39748 L .37704 .37201 L p .826 g
F P
0 g
s
.59248 .80815 m .59048 .83393 L .65899 .82088 L p .376 g
F P
0 g
s
.65899 .82088 m .66316 .79445 L .59248 .80815 L p .376 g
F P
0 g
s
.31099 .70357 m .31053 .73539 L .37343 .7612 L p .65 g
F P
0 g
s
.37554 .79041 m .37343 .7612 L .31053 .73539 L p .623 g
F P
0 g
s
.37343 .7612 m .37554 .79041 L .4451 .80611 L p .577 g
F P
0 g
s
.44755 .83198 m .4451 .80611 L .37554 .79041 L p .541 g
F P
0 g
s
.4451 .80611 m .44755 .83198 L .51884 .83766 L p .494 g
F P
0 g
s
.56393 .34076 m .52632 .34495 L .56484 .34754 L closepath p .716 g
F P
0 g
s
.48776 .34612 m .52632 .34495 L .48922 .33939 L closepath p .745 g
F P
0 g
s
.56044 .33436 m .52632 .34495 L .56393 .34076 L closepath p .719 g
F P
0 g
s
.55457 .32877 m .52632 .34495 L .56044 .33436 L closepath p .722 g
F P
0 g
s
.48922 .33939 m .52632 .34495 L .49323 .33312 L closepath p .746 g
F P
0 g
s
.54673 .3244 m .52632 .34495 L .55457 .32877 L closepath p .726 g
F P
0 g
s
.49323 .33312 m .52632 .34495 L .49954 .32775 L closepath p .746 g
F P
0 g
s
.53745 .32156 m .52632 .34495 L .54673 .3244 L closepath p .73 g
F P
0 g
s
.49954 .32775 m .52632 .34495 L .50773 .32367 L closepath p .744 g
F P
0 g
s
.52738 .32046 m .52632 .34495 L .53745 .32156 L closepath p .735 g
F P
0 g
s
.51723 .32118 m .52632 .34495 L .52738 .32046 L closepath p .739 g
F P
0 g
s
.50773 .32367 m .52632 .34495 L .51723 .32118 L closepath p .742 g
F P
0 g
s
.56484 .34754 m .60283 .35386 L .60129 .3404 L p .687 g
F P
0 g
s
.60129 .3404 m .56393 .34076 L .56484 .34754 L p .687 g
F P
0 g
s
.72942 .70979 m .72986 .7415 L .78404 .70871 L p .31 g
F P
0 g
s
.78404 .70871 m .78347 .67639 L .72942 .70979 L p .31 g
F P
0 g
s
.44969 .35106 m .48776 .34612 L .48922 .33939 L p .774 g
F P
0 g
s
.37704 .37201 m .34348 .38783 L .35137 .41867 L p .826 g
F P
0 g
s
.32225 .44262 m .35137 .41867 L .34348 .38783 L p .843 g
F P
0 g
s
.27332 .49764 m .25432 .52803 L .28623 .56867 L p .812 g
F P
0 g
s
.27342 .60184 m .28623 .56867 L .25432 .52803 L p .809 g
F P
0 g
s
.72665 .44966 m .75221 .47687 L .76709 .43783 L p .488 g
F P
0 g
s
.76709 .43783 m .73931 .41459 L .72665 .44966 L p .488 g
F P
0 g
s
.60283 .35386 m .63977 .36385 L .63787 .34393 L p .654 g
F P
0 g
s
.63787 .34393 m .60129 .3404 L .60283 .35386 L p .654 g
F P
0 g
s
.48922 .33939 m .45232 .33766 L .44969 .35106 L p .774 g
F P
0 g
s
.41261 .35971 m .44969 .35106 L .45232 .33766 L p .802 g
F P
0 g
s
.56393 .34076 m .60129 .3404 L .59455 .32759 L p .693 g
F P
0 g
s
.59455 .32759 m .56044 .33436 L .56393 .34076 L p .693 g
F P
0 g
s
.77022 .60971 m .77881 .64323 L .81765 .60196 L p .328 g
F P
0 g
s
.81765 .60196 m .80749 .56905 L .77022 .60971 L p .328 g
F P
0 g
s
.45232 .33766 m .48922 .33939 L .49323 .33312 L p .778 g
F P
0 g
s
.26448 .63525 m .25961 .66843 L .31099 .70357 L p .724 g
F P
0 g
s
.31053 .73539 m .31099 .70357 L .25961 .66843 L p .702 g
F P
0 g
s
.63977 .36385 m .67516 .3774 L .67315 .35136 L p .615 g
F P
0 g
s
.67315 .35136 m .63787 .34393 L .63977 .36385 L p .615 g
F P
0 g
s
.34348 .38783 m .31244 .40702 L .32225 .44262 L p .843 g
F P
0 g
s
.2961 .46905 m .32225 .44262 L .31244 .40702 L p .857 g
F P
0 g
s
.75221 .47687 m .77445 .50609 L .79136 .46382 L p .432 g
F P
0 g
s
.79136 .46382 m .76709 .43783 L .75221 .47687 L p .432 g
F P
0 g
s
.45232 .33766 m .41614 .33985 L .41261 .35971 L p .802 g
F P
0 g
s
.37704 .37201 m .41261 .35971 L .41614 .33985 L p .83 g
F P
0 g
s
.56044 .33436 m .59455 .32759 L .58299 .31634 L p .702 g
F P
0 g
s
.58299 .31634 m .55457 .32877 L .56044 .33436 L p .702 g
F P
0 g
s
.66316 .79445 m .65899 .82088 L .72081 .79892 L p .288 g
F P
0 g
s
.72081 .79892 m .72701 .77138 L .66316 .79445 L p .288 g
F P
0 g
s
.46011 .3251 m .49323 .33312 L .49954 .32775 L p .778 g
F P
0 g
s
.49323 .33312 m .46011 .3251 L .45232 .33766 L p .778 g
F P
0 g
s
.25432 .52803 m .23944 .55982 L .27342 .60184 L p .809 g
F P
0 g
s
.26448 .63525 m .27342 .60184 L .23944 .55982 L p .8 g
F P
0 g
s
.67516 .3774 m .70851 .39437 L .70661 .3627 L p .569 g
F P
0 g
s
.70661 .3627 m .67315 .35136 L .67516 .3774 L p .569 g
F P
0 g
s
.60129 .3404 m .63787 .34393 L .62818 .32483 L p .661 g
F P
0 g
s
.62818 .32483 m .59455 .32759 L .60129 .3404 L p .661 g
F P
0 g
s
.5192 .85988 m .51884 .83766 L .44755 .83198 L p .445 g
F P
0 g
s
.51884 .83766 m .5192 .85988 L .58739 .85639 L p .388 g
F P
0 g
s
.58739 .85639 m .59048 .83393 L .51884 .83766 L p .388 g
F P
0 g
s
.55457 .32877 m .58299 .31634 L .56735 .3075 L p .714 g
F P
0 g
s
.56735 .3075 m .54673 .3244 L .55457 .32877 L p .714 g
F P
0 g
s
.41614 .33985 m .45232 .33766 L .46011 .3251 L p .811 g
F P
0 g
s
.31053 .73539 m .31355 .76547 L .37554 .79041 L p .623 g
F P
0 g
s
.38014 .81703 m .37554 .79041 L .31355 .76547 L p .584 g
F P
0 g
s
.37554 .79041 m .38014 .81703 L .44755 .83198 L p .541 g
F P
0 g
s
.47259 .31428 m .49954 .32775 L .50773 .32367 L p .775 g
F P
0 g
s
.49954 .32775 m .47259 .31428 L .46011 .3251 L p .778 g
F P
0 g
s
.31244 .40702 m .28442 .42938 L .2961 .46905 L p .857 g
F P
0 g
s
.27332 .49764 m .2961 .46905 L .28442 .42938 L p .869 g
F P
0 g
s
.72986 .7415 m .72701 .77138 L .78036 .7397 L p .24 g
F P
0 g
s
.78036 .7397 m .78404 .70871 L .72986 .7415 L p .24 g
F P
0 g
s
.41614 .33985 m .38119 .34601 L .37704 .37201 L p .83 g
F P
0 g
s
.34348 .38783 m .37704 .37201 L .38119 .34601 L p .858 g
F P
0 g
s
.59048 .83393 m .58739 .85639 L .65253 .84417 L p .313 g
F P
0 g
s
.65253 .84417 m .65899 .82088 L .59048 .83393 L p .313 g
F P
0 g
s
.54673 .3244 m .56735 .3075 L .54873 .30173 L p .728 g
F P
0 g
s
.54873 .30173 m .53745 .32156 L .54673 .3244 L p .728 g
F P
0 g
s
.77881 .64323 m .78347 .67639 L .82317 .6352 L p .256 g
F P
0 g
s
.82317 .6352 m .81765 .60196 L .77881 .64323 L p .256 g
F P
0 g
s
.77445 .50609 m .79299 .53694 L .81169 .49224 L p .366 g
F P
0 g
s
.81169 .49224 m .79136 .46382 L .77445 .50609 L p .366 g
F P
0 g
s
.48894 .30603 m .50773 .32367 L .51723 .32118 L p .767 g
F P
0 g
s
.50773 .32367 m .48894 .30603 L .47259 .31428 L p .775 g
F P
0 g
s
.45136 .85456 m .44755 .83198 L .38014 .81703 L p .491 g
F P
0 g
s
.44755 .83198 m .45136 .85456 L .5192 .85988 L p .445 g
F P
0 g
s
.53745 .32156 m .54873 .30173 L .52846 .2995 L p .742 g
F P
0 g
s
.52846 .2995 m .52738 .32046 L .53745 .32156 L p .742 g
F P
0 g
s
.50803 .30097 m .51723 .32118 L .52738 .32046 L p .755 g
F P
0 g
s
.51723 .32118 m .50803 .30097 L .48894 .30603 L p .767 g
F P
0 g
s
.70851 .39437 m .73931 .41459 L .73771 .37786 L p .515 g
F P
0 g
s
.73771 .37786 m .70661 .3627 L .70851 .39437 L p .515 g
F P
0 g
s
.52738 .32046 m .52846 .2995 L .50803 .30097 L p .755 g
F P
0 g
s
.63787 .34393 m .67315 .35136 L .66082 .32625 L p .62 g
F P
0 g
s
.66082 .32625 m .62818 .32483 L .63787 .34393 L p .62 g
F P
0 g
s
.25961 .66843 m .25902 .7009 L .31053 .73539 L p .702 g
F P
0 g
s
.31355 .76547 m .31053 .73539 L .25902 .7009 L p .669 g
F P
0 g
s
.59455 .32759 m .62818 .32483 L .61116 .30797 L p .674 g
F P
0 g
s
.61116 .30797 m .58299 .31634 L .59455 .32759 L p .674 g
F P
0 g
s
.42741 .32111 m .46011 .3251 L .47259 .31428 L p .813 g
F P
0 g
s
.46011 .3251 m .42741 .32111 L .41614 .33985 L p .811 g
F P
0 g
s
.38119 .34601 m .41614 .33985 L .42741 .32111 L p .846 g
F P
0 g
s
.23944 .55982 m .22903 .59258 L .26448 .63525 L p .8 g
F P
0 g
s
.25961 .66843 m .26448 .63525 L .22903 .59258 L p .784 g
F P
0 g
s
.28442 .42938 m .2599 .45464 L .27332 .49764 L p .869 g
F P
0 g
s
.25432 .52803 m .27332 .49764 L .2599 .45464 L p .876 g
F P
0 g
s
.38119 .34601 m .34801 .35616 L .34348 .38783 L p .858 g
F P
0 g
s
.31244 .40702 m .34348 .38783 L .34801 .35616 L p .884 g
F P
0 g
s
.79299 .53694 m .80749 .56905 L .82764 .52273 L p .286 g
F P
0 g
s
.82764 .52273 m .81169 .49224 L .79299 .53694 L p .286 g
F P
0 g
s
.73931 .41459 m .76709 .43783 L .7659 .39672 L p .447 g
F P
0 g
s
.7659 .39672 m .73771 .37786 L .73931 .41459 L p .447 g
F P
0 g
s
.65899 .82088 m .65253 .84417 L .7112 .82364 L p .205 g
F P
0 g
s
.7112 .82364 m .72081 .79892 L .65899 .82088 L p .205 g
F P
0 g
s
.58299 .31634 m .61116 .30797 L .58789 .29464 L p .692 g
F P
0 g
s
.58789 .29464 m .56735 .3075 L .58299 .31634 L p .692 g
F P
0 g
s
.78347 .67639 m .78404 .70871 L .82384 .66828 L p .168 g
F P
0 g
s
.82384 .66828 m .82317 .6352 L .78347 .67639 L p .168 g
F P
0 g
s
.44583 .30487 m .47259 .31428 L .48894 .30603 L p .807 g
F P
0 g
s
.47259 .31428 m .44583 .30487 L .42741 .32111 L p .813 g
F P
0 g
s
.67315 .35136 m .70661 .3627 L .69197 .33195 L p .565 g
F P
0 g
s
.69197 .33195 m .66082 .32625 L .67315 .35136 L p .565 g
F P
0 g
s
.72701 .77138 m .72081 .79892 L .77234 .76882 L p .15 g
F P
0 g
s
.77234 .76882 m .78036 .7397 L .72701 .77138 L p .15 g
F P
0 g
s
.31355 .76547 m .32016 .79331 L .38014 .81703 L p .584 g
F P
0 g
s
.38728 .84057 m .38014 .81703 L .32016 .79331 L p .527 g
F P
0 g
s
.38014 .81703 m .38728 .84057 L .45136 .85456 L p .491 g
F P
0 g
s
.51968 .87827 m .5192 .85988 L .45136 .85456 L p .375 g
F P
0 g
s
.5192 .85988 m .51968 .87827 L .58318 .87505 L p .313 g
F P
0 g
s
.58318 .87505 m .58739 .85639 L .5192 .85988 L p .313 g
F P
0 g
s
.34801 .35616 m .31714 .37025 L .31244 .40702 L p .884 g
F P
0 g
s
.28442 .42938 m .31244 .40702 L .31714 .37025 L p .907 g
F P
0 g
s
.2599 .45464 m .23936 .48252 L .25432 .52803 L p .876 g
F P
0 g
s
.23944 .55982 m .25432 .52803 L .23936 .48252 L p .875 g
F P
0 g
s
.62818 .32483 m .66082 .32625 L .63867 .30394 L p .632 g
F P
0 g
s
.63867 .30394 m .61116 .30797 L .62818 .32483 L p .632 g
F P
0 g
s
.42741 .32111 m .39563 .32133 L .38119 .34601 L p .846 g
F P
0 g
s
.34801 .35616 m .38119 .34601 L .39563 .32133 L p .88 g
F P
0 g
s
.56735 .3075 m .58789 .29464 L .56 .28592 L p .715 g
F P
0 g
s
.56 .28592 m .54873 .30173 L .56735 .3075 L p .715 g
F P
0 g
s
.47021 .29242 m .48894 .30603 L .50803 .30097 L p .791 g
F P
0 g
s
.48894 .30603 m .47021 .29242 L .44583 .30487 L p .807 g
F P
0 g
s
.39563 .32133 m .42741 .32111 L .44583 .30487 L p .849 g
F P
0 g
s
.22903 .59258 m .22336 .62584 L .25961 .66843 L p .784 g
F P
0 g
s
.25902 .7009 m .25961 .66843 L .22336 .62584 L p .755 g
F P
0 g
s
.58739 .85639 m .58318 .87505 L .64377 .86384 L p .225 g
F P
0 g
s
.64377 .86384 m .65253 .84417 L .58739 .85639 L p .225 g
F P
0 g
s
.25902 .7009 m .26286 .73215 L .31355 .76547 L p .669 g
F P
0 g
s
.32016 .79331 m .31355 .76547 L .26286 .73215 L p .62 g
F P
0 g
s
.76709 .43783 m .79136 .46382 L .79066 .41912 L p .362 g
F P
0 g
s
.79066 .41912 m .7659 .39672 L .76709 .43783 L p .362 g
F P
0 g
s
.80749 .56905 m .81765 .60196 L .83883 .55486 L p .188 g
F P
0 g
s
.83883 .55486 m .82764 .52273 L .80749 .56905 L p .188 g
F P
0 g
s
.45652 .87338 m .45136 .85456 L .38728 .84057 L p .417 g
F P
0 g
s
.45136 .85456 m .45652 .87338 L .51968 .87827 L p .375 g
F P
0 g
s
.54873 .30173 m .56 .28592 L .52953 .28253 L p .741 g
F P
0 g
s
.52953 .28253 m .52846 .2995 L .54873 .30173 L p .741 g
F P
0 g
s
.49882 .28476 m .50803 .30097 L .52846 .2995 L p .768 g
F P
0 g
s
.50803 .30097 m .49882 .28476 L .47021 .29242 L p .791 g
F P
0 g
s
.52846 .2995 m .52953 .28253 L .49882 .28476 L p .768 g
F P
0 g
s
.70661 .3627 m .73771 .37786 L .72109 .342 L p .492 g
F P
0 g
s
.72109 .342 m .69197 .33195 L .70661 .3627 L p .492 g
F P
0 g
s
.31714 .37025 m .28911 .38819 L .28442 .42938 L p .907 g
F P
0 g
s
.2599 .45464 m .28442 .42938 L .28911 .38819 L p .923 g
F P
0 g
s
.78404 .70871 m .78036 .7397 L .81948 .70066 L p .08 g
F P
0 g
s
.81948 .70066 m .82384 .66828 L .78404 .70871 L p .08 g
F P
0 g
s
.23936 .48252 m .22323 .51264 L .23944 .55982 L p .875 g
F P
0 g
s
.22903 .59258 m .23944 .55982 L .22323 .51264 L p .862 g
F P
0 g
s
.61116 .30797 m .63867 .30394 L .60804 .28621 L p .652 g
F P
0 g
s
.60804 .28621 m .58789 .29464 L .61116 .30797 L p .652 g
F P
0 g
s
.39563 .32133 m .36527 .3259 L .34801 .35616 L p .88 g
F P
0 g
s
.31714 .37025 m .34801 .35616 L .36527 .3259 L p .911 g
F P
0 g
s
.41968 .29983 m .44583 .30487 L .47021 .29242 L p .836 g
F P
0 g
s
.44583 .30487 m .41968 .29983 L .39563 .32133 L p .849 g
F P
0 g
s
.66082 .32625 m .69197 .33195 L .66507 .30448 L p .567 g
F P
0 g
s
.66507 .30448 m .63867 .30394 L .66082 .32625 L p .567 g
F P
0 g
s
.65253 .84417 m .64377 .86384 L .6982 .84502 L p .117 g
F P
0 g
s
.6982 .84502 m .7112 .82364 L .65253 .84417 L p .117 g
F P
0 g
s
.72081 .79892 m .7112 .82364 L .75995 .79554 L p .086 g
F P
0 g
s
.75995 .79554 m .77234 .76882 L .72081 .79892 L p .086 g
F P
0 g
s
.79136 .46382 m .81169 .49224 L .81148 .44479 L p .252 g
F P
0 g
s
.81148 .44479 m .79066 .41912 L .79136 .46382 L p .252 g
F P
0 g
s
.81765 .60196 m .82317 .6352 L .84493 .58816 L p .063 g
F P
0 g
s
.84493 .58816 m .83883 .55486 L .81765 .60196 L p .063 g
F P
0 g
s
.22336 .62584 m .22267 .65909 L .25902 .7009 L p .755 g
F P
0 g
s
.26286 .73215 m .25902 .7009 L .22267 .65909 L p .704 g
F P
0 g
s
.36527 .3259 m .39563 .32133 L .41968 .29983 L p .882 g
F P
0 g
s
.32016 .79331 m .33038 .8184 L .38728 .84057 L p .527 g
F P
0 g
s
.39695 .86054 m .38728 .84057 L .33038 .8184 L p .438 g
F P
0 g
s
.38728 .84057 m .39695 .86054 L .45652 .87338 L p .417 g
F P
0 g
s
.26286 .73215 m .27121 .76165 L .32016 .79331 L p .62 g
F P
0 g
s
.33038 .8184 m .32016 .79331 L .27121 .76165 L p .541 g
F P
0 g
s
.73771 .37786 m .7659 .39672 L .74763 .35639 L p .391 g
F P
0 g
s
.74763 .35639 m .72109 .342 L .73771 .37786 L p .391 g
F P
0 g
s
.58789 .29464 m .60804 .28621 L .57108 .27457 L p .682 g
F P
0 g
s
.57108 .27457 m .56 .28592 L .58789 .29464 L p .682 g
F P
0 g
s
.5203 .89234 m .51968 .87827 L .45652 .87338 L p .267 g
F P
0 g
s
.51968 .87827 m .5203 .89234 L .57788 .88946 L p .201 g
F P
0 g
s
.57788 .88946 m .58318 .87505 L .51968 .87827 L p .201 g
F P
0 g
s
.28911 .38819 m .26447 .40981 L .2599 .45464 L p .923 g
F P
0 g
s
.23936 .48252 m .2599 .45464 L .26447 .40981 L p .927 g
F P
0 g
s
.45183 .28325 m .47021 .29242 L .49882 .28476 L p .806 g
F P
0 g
s
.47021 .29242 m .45183 .28325 L .41968 .29983 L p .836 g
F P
0 g
s
.22323 .51264 m .2119 .54459 L .22903 .59258 L p .862 g
F P
0 g
s
.22336 .62584 m .22903 .59258 L .2119 .54459 L p .828 g
F P
0 g
s
.58318 .87505 m .57788 .88946 L .63274 .8794 L p .125 g
F P
0 g
s
.63274 .8794 m .64377 .86384 L .58318 .87505 L p .125 g
F P
0 g
s
.36527 .3259 m .33686 .33492 L .31714 .37025 L p .911 g
F P
0 g
s
.28911 .38819 m .31714 .37025 L .33686 .33492 L p .931 g
F P
0 g
s
.78036 .7397 m .77234 .76882 L .81 .73177 L p .028 g
F P
0 g
s
.81 .73177 m .81948 .70066 L .78036 .7397 L p .028 g
F P
0 g
s
.46304 .88795 m .45652 .87338 L .39695 .86054 L p .302 g
F P
0 g
s
.45652 .87338 m .46304 .88795 L .5203 .89234 L p .267 g
F P
0 g
s
.56 .28592 m .57108 .27457 L .53059 .27004 L p .721 g
F P
0 g
s
.53059 .27004 m .52953 .28253 L .56 .28592 L p .721 g
F P
0 g
s
.81169 .49224 m .82764 .52273 L .82787 .47342 L p .127 g
F P
0 g
s
.82787 .47342 m .81148 .44479 L .81169 .49224 L p .127 g
F P
0 g
s
.48977 .27302 m .49882 .28476 L .52953 .28253 L p .765 g
F P
0 g
s
.49882 .28476 m .48977 .27302 L .45183 .28325 L p .806 g
F P
0 g
s
.52953 .28253 m .53059 .27004 L .48977 .27302 L p .765 g
F P
0 g
s
.69197 .33195 m .72109 .342 L .68987 .30979 L p .467 g
F P
0 g
s
.68987 .30979 m .66507 .30448 L .69197 .33195 L p .467 g
F P
0 g
s
.63867 .30394 m .66507 .30448 L .62746 .28255 L p .578 g
F P
0 g
s
.62746 .28255 m .60804 .28621 L .63867 .30394 L p .578 g
F P
0 g
s
.82317 .6352 m .82384 .66828 L .84567 .6221 L p .04 g
F P
0 g
s
.84567 .6221 m .84493 .58816 L .82317 .6352 L p .04 g
F P
0 g
s
.39457 .2994 m .41968 .29983 L .45183 .28325 L p .852 g
F P
0 g
s
.41968 .29983 m .39457 .2994 L .36527 .3259 L p .882 g
F P
0 g
s
.22267 .65909 m .22714 .69179 L .26286 .73215 L p .704 g
F P
0 g
s
.27121 .76165 m .26286 .73215 L .22714 .69179 L p .618 g
F P
0 g
s
.33686 .33492 m .36527 .3259 L .39457 .2994 L p .899 g
F P
0 g
s
.7112 .82364 m .6982 .84502 L .74321 .81932 L p .028 g
F P
0 g
s
.74321 .81932 m .75995 .79554 L .7112 .82364 L p .028 g
F P
0 g
s
.7659 .39672 m .79066 .41912 L .77106 .37506 L p .253 g
F P
0 g
s
.77106 .37506 m .74763 .35639 L .7659 .39672 L p .253 g
F P
0 g
s
.26447 .40981 m .24373 .43488 L .23936 .48252 L p .927 g
F P
0 g
s
.22323 .51264 m .23936 .48252 L .24373 .43488 L p .904 g
F P
0 g
s
.64377 .86384 m .63274 .8794 L .68185 .86256 L p .063 g
F P
0 g
s
.68185 .86256 m .6982 .84502 L .64377 .86384 L p .063 g
F P
0 g
s
.2119 .54459 m .20573 .57789 L .22336 .62584 L p .828 g
F P
0 g
s
.22267 .65909 m .22336 .62584 L .20573 .57789 L p .758 g
F P
0 g
s
.33038 .8184 m .34421 .84022 L .39695 .86054 L p .438 g
F P
0 g
s
.40912 .87645 m .39695 .86054 L .34421 .84022 L p .306 g
F P
0 g
s
.39695 .86054 m .40912 .87645 L .46304 .88795 L p .302 g
F P
0 g
s
.27121 .76165 m .28412 .78885 L .33038 .8184 L p .541 g
F P
0 g
s
.34421 .84022 m .33038 .8184 L .28412 .78885 L p .417 g
F P
0 g
s
.33686 .33492 m .31093 .34841 L .28911 .38819 L p .931 g
F P
0 g
s
.26447 .40981 m .28911 .38819 L .31093 .34841 L p .923 g
F P
0 g
s
.82764 .52273 m .83883 .55486 L .8394 .50462 L p .056 g
F P
0 g
s
.8394 .50462 m .82787 .47342 L .82764 .52273 L p .056 g
F P
0 g
s
.60804 .28621 m .62746 .28255 L .58179 .26809 L p .603 g
F P
0 g
s
.58179 .26809 m .57108 .27457 L .60804 .28621 L p .603 g
F P
0 g
s
.77234 .76882 m .75995 .79554 L .79537 .76104 L p .002 g
F P
0 g
s
.79537 .76104 m .81 .73177 L .77234 .76882 L p .002 g
F P
0 g
s
.4341 .27888 m .45183 .28325 L .48977 .27302 L p .79 g
F P
0 g
s
.45183 .28325 m .4341 .27888 L .39457 .2994 L p .852 g
F P
0 g
s
.52104 .90165 m .5203 .89234 L .46304 .88795 L p .146 g
F P
0 g
s
.5203 .89234 m .52104 .90165 L .57151 .89914 L p .118 g
F P
0 g
s
.57151 .89914 m .57788 .88946 L .5203 .89234 L p .118 g
F P
0 g
s
.82384 .66828 m .81948 .70066 L .84086 .6561 L p .019 g
F P
0 g
s
.84086 .6561 m .84567 .6221 L .82384 .66828 L p .019 g
F P
0 g
s
.72109 .342 m .74763 .35639 L .71258 .31998 L p .318 g
F P
0 g
s
.71258 .31998 m .68987 .30979 L .72109 .342 L p .318 g
F P
0 g
s
.57788 .88946 m .57151 .89914 L .6195 .8904 L p .075 g
F P
0 g
s
.6195 .8904 m .63274 .8794 L .57788 .88946 L p .075 g
F P
0 g
s
.79066 .41912 m .81148 .44479 L .79083 .39784 L p .129 g
F P
0 g
s
.79083 .39784 m .77106 .37506 L .79066 .41912 L p .129 g
F P
0 g
s
.24373 .43488 m .22739 .46311 L .22323 .51264 L p .904 g
F P
0 g
s
.2119 .54459 m .22323 .51264 L .22739 .46311 L p .836 g
F P
0 g
s
.66507 .30448 m .68987 .30979 L .64578 .28395 L p .443 g
F P
0 g
s
.64578 .28395 m .62746 .28255 L .66507 .30448 L p .443 g
F P
0 g
s
.47087 .89783 m .46304 .88795 L .40912 .87645 L p .161 g
F P
0 g
s
.46304 .88795 m .47087 .89783 L .52104 .90165 L p .146 g
F P
0 g
s
.22714 .69179 m .23687 .72336 L .27121 .76165 L p .618 g
F P
0 g
s
.28412 .78885 m .27121 .76165 L .23687 .72336 L p .485 g
F P
0 g
s
.57108 .27457 m .58179 .26809 L .53162 .26245 L p .648 g
F P
0 g
s
.53162 .26245 m .53059 .27004 L .57108 .27457 L p .648 g
F P
0 g
s
.48101 .26617 m .48977 .27302 L .53059 .27004 L p .715 g
F P
0 g
s
.48977 .27302 m .48101 .26617 L .4341 .27888 L p .79 g
F P
0 g
s
.39457 .2994 m .37094 .30381 L .33686 .33492 L p .899 g
F P
0 g
s
.31093 .34841 m .33686 .33492 L .37094 .30381 L p .872 g
F P
0 g
s
.37094 .30381 m .39457 .2994 L .4341 .27888 L p .821 g
F P
0 g
s
.20573 .57789 m .20498 .61201 L .22267 .65909 L p .758 g
F P
0 g
s
.22714 .69179 m .22267 .65909 L .20498 .61201 L p .635 g
F P
0 g
s
.53059 .27004 m .53162 .26245 L .48101 .26617 L p .715 g
F P
0 g
s
.6982 .84502 m .68185 .86256 L .72225 .83964 L p F P
s
.72225 .83964 m .74321 .81932 L .6982 .84502 L p F P
s
.83883 .55486 m .84493 .58816 L .8457 .53792 L p .033 g
F P
0 g
s
.8457 .53792 m .8394 .50462 L .83883 .55486 L p .033 g
F P
0 g
s
.31093 .34841 m .28803 .36631 L .26447 .40981 L p .923 g
F P
0 g
s
.24373 .43488 m .26447 .40981 L .28803 .36631 L p .861 g
F P
0 g
s
.63274 .8794 m .6195 .8904 L .6623 .87579 L p .01 g
F P
0 g
s
.6623 .87579 m .68185 .86256 L .63274 .8794 L p .01 g
F P
0 g
s
.28412 .78885 m .30155 .81322 L .34421 .84022 L p .417 g
F P
0 g
s
.36158 .85828 m .34421 .84022 L .30155 .81322 L p .255 g
F P
0 g
s
.34421 .84022 m .36158 .85828 L .40912 .87645 L p .306 g
F P
0 g
s
.75995 .79554 m .74321 .81932 L .77567 .78787 L p F P
s
.77567 .78787 m .79537 .76104 L .75995 .79554 L p F P
s
.81948 .70066 m .81 .73177 L .8304 .68956 L p F P
s
.8304 .68956 m .84086 .6561 L .81948 .70066 L p F P
s
.42373 .88783 m .40912 .87645 L .36158 .85828 L p .158 g
F P
0 g
s
.40912 .87645 m .42373 .88783 L .47087 .89783 L p .161 g
F P
0 g
s
.22739 .46311 m .21589 .4941 L .2119 .54459 L p .836 g
F P
0 g
s
.20573 .57789 m .2119 .54459 L .21589 .4941 L p .704 g
F P
0 g
s
.74763 .35639 m .77106 .37506 L .73272 .33511 L p .176 g
F P
0 g
s
.73272 .33511 m .71258 .31998 L .74763 .35639 L p .176 g
F P
0 g
s
.81148 .44479 m .82787 .47342 L .80646 .42449 L p .051 g
F P
0 g
s
.80646 .42449 m .79083 .39784 L .81148 .44479 L p .051 g
F P
0 g
s
.62746 .28255 m .64578 .28395 L .59193 .26687 L p .435 g
F P
0 g
s
.59193 .26687 m .58179 .26809 L .62746 .28255 L p .435 g
F P
0 g
s
.23687 .72336 m .25188 .75322 L .28412 .78885 L p .485 g
F P
0 g
s
.30155 .81322 m .28412 .78885 L .25188 .75322 L p .314 g
F P
0 g
s
.41736 .27962 m .4341 .27888 L .48101 .26617 L p .69 g
F P
0 g
s
.4341 .27888 m .41736 .27962 L .37094 .30381 L p .821 g
F P
0 g
s
.20498 .61201 m .20985 .64638 L .22714 .69179 L p .635 g
F P
0 g
s
.23687 .72336 m .22714 .69179 L .20985 .64638 L p .468 g
F P
0 g
s
.68987 .30979 m .71258 .31998 L .66262 .29066 L p .28 g
F P
0 g
s
.66262 .29066 m .64578 .28395 L .68987 .30979 L p .28 g
F P
0 g
s
.37094 .30381 m .34929 .31321 L .31093 .34841 L p .872 g
F P
0 g
s
.28803 .36631 m .31093 .34841 L .34929 .31321 L p .76 g
F P
0 g
s
.5219 .90577 m .52104 .90165 L .47087 .89783 L p .083 g
F P
0 g
s
.52104 .90165 m .5219 .90577 L .56412 .90367 L p .055 g
F P
0 g
s
.56412 .90367 m .57151 .89914 L .52104 .90165 L p .055 g
F P
0 g
s
.84493 .58816 m .84567 .6221 L .84646 .57277 L p .006 g
F P
0 g
s
.84646 .57277 m .8457 .53792 L .84493 .58816 L p .006 g
F P
0 g
s
.28803 .36631 m .26868 .38849 L .24373 .43488 L p .861 g
F P
0 g
s
.22739 .46311 m .24373 .43488 L .26868 .38849 L p .719 g
F P
0 g
s
.57151 .89914 m .56412 .90367 L .60418 .89639 L p .008 g
F P
0 g
s
.60418 .89639 m .6195 .8904 L .57151 .89914 L p .008 g
F P
0 g
s
.34929 .31321 m .37094 .30381 L .41736 .27962 L p .69 g
F P
0 g
s
.47994 .90258 m .47087 .89783 L .42373 .88783 L p .091 g
F P
0 g
s
.47087 .89783 m .47994 .90258 L .5219 .90577 L p .083 g
F P
0 g
s
.58179 .26809 m .59193 .26687 L .53259 .26019 L p .469 g
F P
0 g
s
.53259 .26019 m .53162 .26245 L .58179 .26809 L p .469 g
F P
0 g
s
.68185 .86256 m .6623 .87579 L .69726 .85598 L p F P
s
.69726 .85598 m .72225 .83964 L .68185 .86256 L p F P
s
.47273 .26459 m .48101 .26617 L .53162 .26245 L p .558 g
F P
0 g
s
.48101 .26617 m .47273 .26459 L .41736 .27962 L p .69 g
F P
0 g
s
.81 .73177 m .79537 .76104 L .8143 .72183 L p .692 g
F P
0 g
s
.8143 .72183 m .8304 .68956 L .81 .73177 L p .692 g
F P
0 g
s
.21589 .4941 m .20961 .52739 L .20573 .57789 L p .704 g
F P
0 g
s
.20498 .61201 m .20573 .57789 L .20961 .52739 L p .004 g
F P
0 g
s
.53162 .26245 m .53259 .26019 L .47273 .26459 L p .558 g
F P
0 g
s
.74321 .81932 m .72225 .83964 L .75109 .81169 L p F P
s
.75109 .81169 m .77567 .78787 L .74321 .81932 L p F P
s
.30155 .81322 m .32335 .83421 L .36158 .85828 L p .255 g
F P
0 g
s
.38234 .87208 m .36158 .85828 L .32335 .83421 L p .122 g
F P
0 g
s
.36158 .85828 m .38234 .87208 L .42373 .88783 L p .158 g
F P
0 g
s
.82787 .47342 m .8394 .50462 L .81748 .45466 L p .44 g
F P
0 g
s
.81748 .45466 m .80646 .42449 L .82787 .47342 L p .44 g
F P
0 g
s
.6195 .8904 m .60418 .89639 L .63974 .88427 L p F P
s
.63974 .88427 m .6623 .87579 L .6195 .8904 L p F P
s
.77106 .37506 m .79083 .39784 L .74978 .35512 L p .226 g
F P
0 g
s
.74978 .35512 m .73272 .33511 L .77106 .37506 L p .226 g
F P
0 g
s
.42373 .88783 m .44063 .89426 L .47994 .90258 L p .091 g
F P
0 g
s
.44063 .89426 m .42373 .88783 L .38234 .87208 L p .272 g
F P
0 g
s
.20985 .64638 m .22044 .68036 L .23687 .72336 L p .468 g
F P
0 g
s
.25188 .75322 m .23687 .72336 L .22044 .68036 L p .046 g
F P
0 g
s
.25188 .75322 m .27207 .78076 L .30155 .81322 L p .314 g
F P
0 g
s
.32335 .83421 m .30155 .81322 L .27207 .78076 L p .094 g
F P
0 g
s
.84567 .6221 m .84086 .6561 L .8415 .60855 L p .7 g
F P
0 g
s
.8415 .60855 m .84646 .57277 L .84567 .6221 L p .7 g
F P
0 g
s
.64578 .28395 m .66262 .29066 L .60127 .27121 L p .086 g
F P
0 g
s
.60127 .27121 m .59193 .26687 L .64578 .28395 L p .086 g
F P
0 g
s
.26868 .38849 m .25338 .41473 L .22739 .46311 L p .719 g
F P
0 g
s
.21589 .4941 m .22739 .46311 L .25338 .41473 L p F P
s
.34929 .31321 m .33008 .32766 L .28803 .36631 L p .76 g
F P
0 g
s
.26868 .38849 m .28803 .36631 L .33008 .32766 L p F P
s
.71258 .31998 m .73272 .33511 L .6776 .30282 L p .151 g
F P
0 g
s
.6776 .30282 m .66262 .29066 L .71258 .31998 L p .151 g
F P
0 g
s
.41736 .27962 m .40196 .28573 L .34929 .31321 L p .69 g
F P
0 g
s
.40196 .28573 m .41736 .27962 L .47273 .26459 L p F P
s
.79537 .76104 m .77567 .78787 L .79271 .75226 L p .834 g
F P
0 g
s
.79271 .75226 m .8143 .72183 L .79537 .76104 L p .834 g
F P
0 g
s
.20961 .52739 m .20885 .56243 L .20498 .61201 L p .004 g
F P
0 g
s
.20985 .64638 m .20498 .61201 L .20885 .56243 L p .032 g
F P
0 g
s
.33008 .32766 m .34929 .31321 L .40196 .28573 L p F P
s
.52288 .90433 m .5219 .90577 L .47994 .90258 L p .45 g
F P
0 g
s
.5219 .90577 m .52288 .90433 L .55578 .90269 L p .511 g
F P
0 g
s
.55578 .90269 m .56412 .90367 L .5219 .90577 L p .511 g
F P
0 g
s
.8394 .50462 m .8457 .53792 L .82351 .48789 L p .615 g
F P
0 g
s
.82351 .48789 m .81748 .45466 L .8394 .50462 L p .615 g
F P
0 g
s
.6623 .87579 m .63974 .88427 L .66855 .86789 L p .745 g
F P
0 g
s
.66855 .86789 m .69726 .85598 L .6623 .87579 L p .745 g
F P
0 g
s
.72225 .83964 m .69726 .85598 L .72193 .83194 L p .822 g
F P
0 g
s
.72193 .83194 m .75109 .81169 L .72225 .83964 L p .822 g
F P
0 g
s
.56412 .90367 m .55578 .90269 L .58693 .89702 L p .609 g
F P
0 g
s
.58693 .89702 m .60418 .89639 L .56412 .90367 L p .609 g
F P
0 g
s
.49018 .90184 m .47994 .90258 L .44063 .89426 L p .434 g
F P
0 g
s
.47994 .90258 m .49018 .90184 L .52288 .90433 L p .45 g
F P
0 g
s
.79083 .39784 m .80646 .42449 L .7633 .37983 L p .428 g
F P
0 g
s
.7633 .37983 m .74978 .35512 L .79083 .39784 L p .428 g
F P
0 g
s
.59193 .26687 m .60127 .27121 L .53349 .2636 L p .077 g
F P
0 g
s
.53349 .2636 m .53259 .26019 L .59193 .26687 L p .077 g
F P
0 g
s
.32335 .83421 m .34932 .8513 L .38234 .87208 L p .122 g
F P
0 g
s
.40629 .88119 m .38234 .87208 L .34932 .8513 L p .325 g
F P
0 g
s
.38234 .87208 m .40629 .88119 L .44063 .89426 L p .272 g
F P
0 g
s
.84086 .6561 m .8304 .68956 L .83071 .6446 L p .829 g
F P
0 g
s
.83071 .6446 m .8415 .60855 L .84086 .6561 L p .829 g
F P
0 g
s
.22044 .68036 m .23672 .7133 L .25188 .75322 L p .046 g
F P
0 g
s
.27207 .78076 m .25188 .75322 L .23672 .7133 L p .09 g
F P
0 g
s
.46509 .26861 m .47273 .26459 L .53259 .26019 L p .039 g
F P
0 g
s
.47273 .26459 m .46509 .26861 L .40196 .28573 L p F P
s
.27207 .78076 m .29724 .80539 L .32335 .83421 L p .094 g
F P
0 g
s
.34932 .8513 m .32335 .83421 L .29724 .80539 L p .251 g
F P
0 g
s
.25338 .41473 m .24258 .44468 L .21589 .4941 L p F P
s
.20961 .52739 m .21589 .4941 L .24258 .44468 L p .016 g
F P
0 g
s
.53259 .26019 m .53349 .2636 L .46509 .26861 L p .039 g
F P
0 g
s
.60418 .89639 m .58693 .89702 L .61444 .8876 L p .727 g
F P
0 g
s
.61444 .8876 m .63974 .88427 L .60418 .89639 L p .727 g
F P
0 g
s
.33008 .32766 m .31379 .34714 L .26868 .38849 L p F P
s
.25338 .41473 m .26868 .38849 L .31379 .34714 L p F P
s
.45963 .89536 m .44063 .89426 L .40629 .88119 L p .452 g
F P
0 g
s
.44063 .89426 m .45963 .89536 L .49018 .90184 L p .434 g
F P
0 g
s
.20885 .56243 m .21381 .59859 L .20985 .64638 L p .032 g
F P
0 g
s
.22044 .68036 m .20985 .64638 L .21381 .59859 L p .054 g
F P
0 g
s
.77567 .78787 m .75109 .81169 L .76587 .78019 L p .908 g
F P
0 g
s
.76587 .78019 m .79271 .75226 L .77567 .78787 L p .908 g
F P
0 g
s
.73272 .33511 m .74978 .35512 L .69034 .32047 L p .323 g
F P
0 g
s
.69034 .32047 m .6776 .30282 L .73272 .33511 L p .323 g
F P
0 g
s
.66262 .29066 m .6776 .30282 L .6096 .28134 L p .164 g
F P
0 g
s
.6096 .28134 m .60127 .27121 L .66262 .29066 L p .164 g
F P
0 g
s
.8457 .53792 m .84646 .57277 L .82424 .52364 L p .743 g
F P
0 g
s
.82424 .52364 m .82351 .48789 L .8457 .53792 L p .743 g
F P
0 g
s
.38825 .29738 m .40196 .28573 L .46509 .26861 L p .048 g
F P
0 g
s
.40196 .28573 m .38825 .29738 L .33008 .32766 L p F P
s
.8304 .68956 m .8143 .72183 L .81416 .6802 L p .899 g
F P
0 g
s
.81416 .6802 m .83071 .6446 L .8304 .68956 L p .899 g
F P
0 g
s
.23672 .7133 m .25855 .74453 L .27207 .78076 L p .09 g
F P
0 g
s
.29724 .80539 m .27207 .78076 L .25855 .74453 L p .22 g
F P
0 g
s
.31379 .34714 m .33008 .32766 L .38825 .29738 L p .02 g
F P
0 g
s
.69726 .85598 m .66855 .86789 L .68861 .84811 L p .88 g
F P
0 g
s
.68861 .84811 m .72193 .83194 L .69726 .85598 L p .88 g
F P
0 g
s
.80646 .42449 m .81748 .45466 L .77287 .40897 L p .577 g
F P
0 g
s
.77287 .40897 m .7633 .37983 L .80646 .42449 L p .577 g
F P
0 g
s
.63974 .88427 m .61444 .8876 L .63654 .87494 L p .827 g
F P
0 g
s
.63654 .87494 m .66855 .86789 L .63974 .88427 L p .827 g
F P
0 g
s
.24258 .44468 m .23667 .4779 L .20961 .52739 L p .016 g
F P
0 g
s
.20885 .56243 m .20961 .52739 L .23667 .4779 L p .039 g
F P
0 g
s
.34932 .8513 m .37913 .86403 L .40629 .88119 L p .325 g
F P
0 g
s
.4331 .88522 m .40629 .88119 L .37913 .86403 L p .489 g
F P
0 g
s
.40629 .88119 m .4331 .88522 L .45963 .89536 L p .452 g
F P
0 g
s
.29724 .80539 m .32707 .82654 L .34932 .8513 L p .251 g
F P
0 g
s
.37913 .86403 m .34932 .8513 L .32707 .82654 L p .391 g
F P
0 g
s
.52395 .89704 m .52288 .90433 L .49018 .90184 L p .651 g
F P
0 g
s
.52288 .90433 m .52395 .89704 L .5466 .89591 L p .692 g
F P
0 g
s
.5466 .89591 m .55578 .90269 L .52288 .90433 L p .692 g
F P
0 g
s
.55578 .90269 m .5466 .89591 L .568 .89199 L p .744 g
F P
0 g
s
.568 .89199 m .58693 .89702 L .55578 .90269 L p .744 g
F P
0 g
s
.21381 .59859 m .22456 .6352 L .22044 .68036 L p .054 g
F P
0 g
s
.23672 .7133 m .22044 .68036 L .22456 .6352 L p .119 g
F P
0 g
s
.50144 .89532 m .49018 .90184 L .45963 .89536 L p .626 g
F P
0 g
s
.49018 .90184 m .50144 .89532 L .52395 .89704 L p .651 g
F P
0 g
s
.60127 .27121 m .6096 .28134 L .53429 .27292 L p .147 g
F P
0 g
s
.53429 .27292 m .53349 .2636 L .60127 .27121 L p .147 g
F P
0 g
s
.31379 .34714 m .30087 .37148 L .25338 .41473 L p F P
s
.24258 .44468 m .25338 .41473 L .30087 .37148 L p .031 g
F P
0 g
s
.75109 .81169 m .72193 .83194 L .7342 .80499 L p .929 g
F P
0 g
s
.7342 .80499 m .76587 .78019 L .75109 .81169 L p .929 g
F P
0 g
s
.45827 .27847 m .46509 .26861 L .53349 .2636 L p .109 g
F P
0 g
s
.46509 .26861 m .45827 .27847 L .38825 .29738 L p .048 g
F P
0 g
s
.84646 .57277 m .8415 .60855 L .81948 .56124 L p .816 g
F P
0 g
s
.81948 .56124 m .82424 .52364 L .84646 .57277 L p .816 g
F P
0 g
s
.58693 .89702 m .568 .89199 L .58678 .8855 L p .796 g
F P
0 g
s
.58678 .8855 m .61444 .8876 L .58693 .89702 L p .796 g
F P
0 g
s
.8143 .72183 m .79271 .75226 L .79203 .7146 L p .926 g
F P
0 g
s
.79203 .7146 m .81416 .6802 L .8143 .72183 L p .926 g
F P
0 g
s
.53349 .2636 m .53429 .27292 L .45827 .27847 L p .109 g
F P
0 g
s
.48047 .89084 m .45963 .89536 L .4331 .88522 L p .616 g
F P
0 g
s
.45963 .89536 m .48047 .89084 L .50144 .89532 L p .626 g
F P
0 g
s
.25855 .74453 m .28565 .77338 L .29724 .80539 L p .22 g
F P
0 g
s
.32707 .82654 m .29724 .80539 L .28565 .77338 L p .35 g
F P
0 g
s
.74978 .35512 m .7633 .37983 L .70046 .34353 L p .472 g
F P
0 g
s
.70046 .34353 m .69034 .32047 L .74978 .35512 L p .472 g
F P
0 g
s
.6776 .30282 m .69034 .32047 L .6167 .29738 L p .321 g
F P
0 g
s
.6167 .29738 m .6096 .28134 L .6776 .30282 L p .321 g
F P
0 g
s
.66855 .86789 m .63654 .87494 L .65167 .85975 L p .882 g
F P
0 g
s
.65167 .85975 m .68861 .84811 L .66855 .86789 L p .882 g
F P
0 g
s
.81748 .45466 m .82351 .48789 L .7781 .4421 L p .675 g
F P
0 g
s
.7781 .4421 m .77287 .40897 L .81748 .45466 L p .675 g
F P
0 g
s
.23667 .4779 m .23596 .51382 L .20885 .56243 L p .039 g
F P
0 g
s
.21381 .59859 m .20885 .56243 L .23596 .51382 L p .081 g
F P
0 g
s
.22456 .6352 m .24106 .67151 L .23672 .7133 L p .119 g
F P
0 g
s
.25855 .74453 m .23672 .7133 L .24106 .67151 L p .244 g
F P
0 g
s
.38825 .29738 m .37659 .31463 L .31379 .34714 L p .02 g
F P
0 g
s
.30087 .37148 m .31379 .34714 L .37659 .31463 L p .084 g
F P
0 g
s
.37659 .31463 m .38825 .29738 L .45827 .27847 L p .106 g
F P
0 g
s
.32707 .82654 m .36112 .84368 L .37913 .86403 L p .391 g
F P
0 g
s
.41233 .87197 m .37913 .86403 L .36112 .84368 L p .527 g
F P
0 g
s
.37913 .86403 m .41233 .87197 L .4331 .88522 L p .489 g
F P
0 g
s
.61444 .8876 m .58678 .8855 L .60173 .87683 L p .836 g
F P
0 g
s
.60173 .87683 m .63654 .87494 L .61444 .8876 L p .836 g
F P
0 g
s
.72193 .83194 m .68861 .84811 L .69823 .82608 L p .918 g
F P
0 g
s
.69823 .82608 m .7342 .80499 L .72193 .83194 L p .918 g
F P
0 g
s
.46239 .88387 m .4331 .88522 L .41233 .87197 L p .616 g
F P
0 g
s
.4331 .88522 m .46239 .88387 L .48047 .89084 L p .616 g
F P
0 g
s
.8415 .60855 m .83071 .6446 L .80917 .59996 L p .853 g
F P
0 g
s
.80917 .59996 m .81948 .56124 L .8415 .60855 L p .853 g
F P
0 g
s
.30087 .37148 m .29173 .40041 L .24258 .44468 L p .031 g
F P
0 g
s
.23667 .4779 m .24258 .44468 L .29173 .40041 L p .09 g
F P
0 g
s
.79271 .75226 m .76587 .78019 L .76466 .74704 L p .927 g
F P
0 g
s
.76466 .74704 m .79203 .7146 L .79271 .75226 L p .927 g
F P
0 g
s
.28565 .77338 m .31759 .79918 L .32707 .82654 L p .35 g
F P
0 g
s
.36112 .84368 m .32707 .82654 L .31759 .79918 L p .467 g
F P
0 g
s
.5251 .88372 m .52395 .89704 L .50144 .89532 L p .733 g
F P
0 g
s
.52395 .89704 m .5251 .88372 L .53673 .88314 L p .755 g
F P
0 g
s
.53673 .88314 m .5466 .89591 L .52395 .89704 L p .755 g
F P
0 g
s
.5466 .89591 m .53673 .88314 L .54767 .88111 L p .777 g
F P
0 g
s
.54767 .88111 m .568 .89199 L .5466 .89591 L p .777 g
F P
0 g
s
.51355 .88283 m .50144 .89532 L .48047 .89084 L p .715 g
F P
0 g
s
.50144 .89532 m .51355 .88283 L .5251 .88372 L p .733 g
F P
0 g
s
.6096 .28134 m .6167 .29738 L .53497 .28831 L p .276 g
F P
0 g
s
.53497 .28831 m .53429 .27292 L .6096 .28134 L p .276 g
F P
0 g
s
.23596 .51382 m .24062 .55179 L .21381 .59859 L p .081 g
F P
0 g
s
.22456 .6352 m .21381 .59859 L .24062 .55179 L p .205 g
F P
0 g
s
.24106 .67151 m .2631 .70675 L .25855 .74453 L p .244 g
F P
0 g
s
.28565 .77338 m .25855 .74453 L .2631 .70675 L p .355 g
F P
0 g
s
.7633 .37983 m .77287 .40897 L .70763 .37176 L p .572 g
F P
0 g
s
.70763 .37176 m .70046 .34353 L .7633 .37983 L p .572 g
F P
0 g
s
.45247 .29428 m .45827 .27847 L .53429 .27292 L p .189 g
F P
0 g
s
.45827 .27847 m .45247 .29428 L .37659 .31463 L p .106 g
F P
0 g
s
.82351 .48789 m .82424 .52364 L .77874 .47866 L p .732 g
F P
0 g
s
.77874 .47866 m .7781 .4421 L .82351 .48789 L p .732 g
F P
0 g
s
.63654 .87494 m .60173 .87683 L .61178 .86648 L p .855 g
F P
0 g
s
.61178 .86648 m .65167 .85975 L .63654 .87494 L p .855 g
F P
0 g
s
.568 .89199 m .54767 .88111 L .55723 .87777 L p .796 g
F P
0 g
s
.55723 .87777 m .58678 .8855 L .568 .89199 L p .796 g
F P
0 g
s
.50284 .88052 m .48047 .89084 L .46239 .88387 L p .702 g
F P
0 g
s
.48047 .89084 m .50284 .88052 L .51355 .88283 L p .715 g
F P
0 g
s
.68861 .84811 m .65167 .85975 L .65863 .84293 L p .89 g
F P
0 g
s
.65863 .84293 m .69823 .82608 L .68861 .84811 L p .89 g
F P
0 g
s
.36112 .84368 m .39881 .85636 L .41233 .87197 L p .527 g
F P
0 g
s
.44839 .8748 m .41233 .87197 L .39881 .85636 L p .621 g
F P
0 g
s
.41233 .87197 m .44839 .8748 L .46239 .88387 L p .616 g
F P
0 g
s
.83071 .6446 m .81416 .6802 L .79339 .63899 L p .869 g
F P
0 g
s
.79339 .63899 m .80917 .59996 L .83071 .6446 L p .869 g
F P
0 g
s
.76587 .78019 m .7342 .80499 L .73253 .77679 L p .915 g
F P
0 g
s
.73253 .77679 m .76466 .74704 L .76587 .78019 L p .915 g
F P
0 g
s
.53429 .27292 m .53497 .28831 L .45247 .29428 L p .189 g
F P
0 g
s
.69034 .32047 m .70046 .34353 L .62235 .31929 L p .448 g
F P
0 g
s
.62235 .31929 m .6167 .29738 L .69034 .32047 L p .448 g
F P
0 g
s
.37659 .31463 m .36732 .3374 L .30087 .37148 L p .084 g
F P
0 g
s
.29173 .40041 m .30087 .37148 L .36732 .3374 L p .158 g
F P
0 g
s
.31759 .79918 m .35383 .82135 L .36112 .84368 L p .467 g
F P
0 g
s
.39881 .85636 m .36112 .84368 L .35383 .82135 L p .557 g
F P
0 g
s
.58678 .8855 m .55723 .87777 L .56475 .87332 L p .81 g
F P
0 g
s
.56475 .87332 m .60173 .87683 L .58678 .8855 L p .81 g
F P
0 g
s
.29173 .40041 m .28672 .43351 L .23667 .4779 L p .09 g
F P
0 g
s
.23596 .51382 m .23667 .4779 L .28672 .43351 L p .191 g
F P
0 g
s
.49366 .87693 m .46239 .88387 L .44839 .8748 L p .692 g
F P
0 g
s
.46239 .88387 m .49366 .87693 L .50284 .88052 L p .702 g
F P
0 g
s
.36732 .3374 m .37659 .31463 L .45247 .29428 L p .199 g
F P
0 g
s
.2631 .70675 m .29034 .74015 L .28565 .77338 L p .355 g
F P
0 g
s
.31759 .79918 m .28565 .77338 L .29034 .74015 L p .444 g
F P
0 g
s
.24062 .55179 m .25072 .59106 L .22456 .6352 L p .205 g
F P
0 g
s
.24106 .67151 m .22456 .6352 L .25072 .59106 L p .302 g
F P
0 g
s
.7342 .80499 m .69823 .82608 L .69628 .80318 L p .895 g
F P
0 g
s
.69628 .80318 m .73253 .77679 L .7342 .80499 L p .895 g
F P
0 g
s
.81416 .6802 m .79203 .7146 L .77236 .67746 L p .874 g
F P
0 g
s
.77236 .67746 m .79339 .63899 L .81416 .6802 L p .874 g
F P
0 g
s
.82424 .52364 m .81948 .56124 L .7746 .51797 L p .765 g
F P
0 g
s
.7746 .51797 m .77874 .47866 L .82424 .52364 L p .765 g
F P
0 g
s
.65167 .85975 m .61178 .86648 L .61618 .85512 L p .855 g
F P
0 g
s
.61618 .85512 m .65863 .84293 L .65167 .85975 L p .855 g
F P
0 g
s
.60173 .87683 m .56475 .87332 L .56971 .86805 L p .817 g
F P
0 g
s
.56971 .86805 m .61178 .86648 L .60173 .87683 L p .817 g
F P
0 g
s
.39881 .85636 m .43945 .86418 L .44839 .8748 L p .621 g
F P
0 g
s
.48663 .87228 m .44839 .8748 L .43945 .86418 L p .687 g
F P
0 g
s
.44839 .8748 m .48663 .87228 L .49366 .87693 L p .692 g
F P
0 g
s
.77287 .40897 m .7781 .4421 L .71156 .40477 L p .633 g
F P
0 g
s
.71156 .40477 m .70763 .37176 L .77287 .40897 L p .633 g
F P
0 g
s
.35383 .82135 m .39366 .83933 L .39881 .85636 L p .557 g
F P
0 g
s
.43945 .86418 m .39881 .85636 L .39366 .83933 L p .628 g
F P
0 g
s
.52632 .86433 m .5251 .88372 L .51355 .88283 L closepath p .754 g
F P
0 g
s
.52632 .86433 m .53673 .88314 L .5251 .88372 L closepath p .76 g
F P
0 g
s
.29034 .74015 m .32227 .77096 L .31759 .79918 L p .444 g
F P
0 g
s
.35383 .82135 m .31759 .79918 L .32227 .77096 L p .518 g
F P
0 g
s
.52632 .86433 m .54767 .88111 L .53673 .88314 L closepath p .765 g
F P
0 g
s
.52632 .86433 m .51355 .88283 L .50284 .88052 L closepath p .748 g
F P
0 g
s
.6167 .29738 m .62235 .31929 L .53552 .30976 L p .393 g
F P
0 g
s
.53552 .30976 m .53497 .28831 L .6167 .29738 L p .393 g
F P
0 g
s
.52632 .86433 m .55723 .87777 L .54767 .88111 L closepath p .77 g
F P
0 g
s
.25072 .59106 m .26618 .63078 L .24106 .67151 L p .302 g
F P
0 g
s
.2631 .70675 m .24106 .67151 L .26618 .63078 L p .381 g
F P
0 g
s
.28672 .43351 m .28611 .47023 L .23596 .51382 L p .191 g
F P
0 g
s
.24062 .55179 m .23596 .51382 L .28611 .47023 L p .282 g
F P
0 g
s
.52632 .86433 m .50284 .88052 L .49366 .87693 L closepath p .743 g
F P
0 g
s
.44784 .31604 m .45247 .29428 L .53497 .28831 L p .315 g
F P
0 g
s
.45247 .29428 m .44784 .31604 L .36732 .3374 L p .199 g
F P
0 g
s
.69823 .82608 m .65863 .84293 L .65664 .82558 L p .87 g
F P
0 g
s
.65664 .82558 m .69628 .80318 L .69823 .82608 L p .87 g
F P
0 g
s
.36732 .3374 m .36076 .36547 L .29173 .40041 L p .158 g
F P
0 g
s
.28672 .43351 m .29173 .40041 L .36076 .36547 L p .261 g
F P
0 g
s
.79203 .7146 m .76466 .74704 L .74646 .71451 L p .871 g
F P
0 g
s
.74646 .71451 m .77236 .67746 L .79203 .7146 L p .871 g
F P
0 g
s
.52632 .86433 m .56475 .87332 L .55723 .87777 L closepath p .773 g
F P
0 g
s
.70046 .34353 m .70763 .37176 L .62635 .34688 L p .526 g
F P
0 g
s
.62635 .34688 m .62235 .31929 L .70046 .34353 L p .526 g
F P
0 g
s
.61178 .86648 m .56971 .86805 L .57176 .86231 L p .816 g
F P
0 g
s
.57176 .86231 m .61618 .85512 L .61178 .86648 L p .816 g
F P
0 g
s
.52632 .86433 m .49366 .87693 L .48663 .87228 L closepath p .738 g
F P
0 g
s
.53497 .28831 m .53552 .30976 L .44784 .31604 L p .315 g
F P
0 g
s
.39366 .83933 m .43629 .8527 L .43945 .86418 L p .628 g
F P
0 g
s
.48225 .86689 m .43945 .86418 L .43629 .8527 L p .685 g
F P
0 g
s
.43945 .86418 m .48225 .86689 L .48663 .87228 L p .687 g
F P
0 g
s
.32227 .77096 m .35826 .79846 L .35383 .82135 L p .518 g
F P
0 g
s
.39366 .83933 m .35383 .82135 L .35826 .79846 L p .581 g
F P
0 g
s
.81948 .56124 m .80917 .59996 L .76566 .5592 L p .785 g
F P
0 g
s
.76566 .5592 m .7746 .51797 L .81948 .56124 L p .785 g
F P
0 g
s
.52632 .86433 m .56971 .86805 L .56475 .87332 L closepath p .775 g
F P
0 g
s
.36076 .36547 m .36732 .3374 L .44784 .31604 L p .303 g
F P
0 g
s
.52632 .86433 m .48663 .87228 L .48225 .86689 L closepath p .735 g
F P
0 g
s
.65863 .84293 m .61618 .85512 L .61448 .84348 L p .841 g
F P
0 g
s
.61448 .84348 m .65664 .82558 L .65863 .84293 L p .841 g
F P
0 g
s
.26618 .63078 m .28676 .67008 L .2631 .70675 L p .381 g
F P
0 g
s
.29034 .74015 m .2631 .70675 L .28676 .67008 L p .447 g
F P
0 g
s
.76466 .74704 m .73253 .77679 L .71624 .74929 L p .863 g
F P
0 g
s
.71624 .74929 m .74646 .71451 L .76466 .74704 L p .863 g
F P
0 g
s
.35826 .79846 m .39755 .82202 L .39366 .83933 L p .581 g
F P
0 g
s
.43629 .8527 m .39366 .83933 L .39755 .82202 L p .636 g
F P
0 g
s
.7781 .4421 m .77874 .47866 L .71203 .442 L p .673 g
F P
0 g
s
.71203 .442 m .71156 .40477 L .7781 .4421 L p .673 g
F P
0 g
s
.52632 .86433 m .57176 .86231 L .56971 .86805 L closepath p .775 g
F P
0 g
s
.28611 .47023 m .29007 .50986 L .24062 .55179 L p .282 g
F P
0 g
s
.25072 .59106 m .24062 .55179 L .29007 .50986 L p .354 g
F P
0 g
s
.43629 .8527 m .48083 .8611 L .48225 .86689 L p .685 g
F P
0 g
s
.52632 .86433 m .48225 .86689 L .48083 .8611 L closepath p .733 g
F P
0 g
s
.61618 .85512 m .57176 .86231 L .57072 .85648 L p .809 g
F P
0 g
s
.57072 .85648 m .61448 .84348 L .61618 .85512 L p .809 g
F P
0 g
s
.39755 .82202 m .43928 .8411 L .43629 .8527 L p .636 g
F P
0 g
s
.48083 .8611 m .43629 .8527 L .43928 .8411 L p .686 g
F P
0 g
s
.28676 .67008 m .31207 .70807 L .29034 .74015 L p .447 g
F P
0 g
s
.32227 .77096 m .29034 .74015 L .31207 .70807 L p .504 g
F P
0 g
s
.80917 .59996 m .79339 .63899 L .75199 .60146 L p .798 g
F P
0 g
s
.75199 .60146 m .76566 .5592 L .80917 .59996 L p .798 g
F P
0 g
s
.73253 .77679 m .69628 .80318 L .68233 .78099 L p .852 g
F P
0 g
s
.68233 .78099 m .71624 .74929 L .73253 .77679 L p .852 g
F P
0 g
s
.52632 .86433 m .57072 .85648 L .57176 .86231 L closepath p .774 g
F P
0 g
s
.36076 .36547 m .35716 .39846 L .28672 .43351 L p .261 g
F P
0 g
s
.28611 .47023 m .28672 .43351 L .35716 .39846 L p .338 g
F P
0 g
s
.43928 .8411 m .48252 .8553 L .48083 .8611 L p .686 g
F P
0 g
s
.52632 .86433 m .48083 .8611 L .48252 .8553 L closepath p .732 g
F P
0 g
s
.52632 .86433 m .55971 .84614 L .56662 .85096 L closepath p .766 g
F P
0 g
s
.52632 .86433 m .55045 .84237 L .55971 .84614 L closepath p .761 g
F P
0 g
s
.52632 .86433 m .53948 .83991 L .55045 .84237 L closepath p .756 g
F P
0 g
s
.52632 .86433 m .52757 .83896 L .53948 .83991 L closepath p .751 g
F P
0 g
s
.52632 .86433 m .51557 .83959 L .52757 .83896 L closepath p .746 g
F P
0 g
s
.52632 .86433 m .50434 .84174 L .51557 .83959 L closepath p .741 g
F P
0 g
s
.52632 .86433 m .49467 .84526 L .50434 .84174 L closepath p .737 g
F P
0 g
s
.52632 .86433 m .48723 .84989 L .49467 .84526 L closepath p .734 g
F P
0 g
s
.52632 .86433 m .48252 .8553 L .48723 .84989 L closepath p .733 g
F P
0 g
s
.52632 .86433 m .56662 .85096 L .57072 .85648 L closepath p .771 g
F P
0 g
s
.62235 .31929 m .62635 .34688 L .5359 .3371 L p .468 g
F P
0 g
s
.5359 .3371 m .53552 .30976 L .62235 .31929 L p .468 g
F P
0 g
s
.70763 .37176 m .71156 .40477 L .62855 .37979 L p .578 g
F P
0 g
s
.62855 .37979 m .62635 .34688 L .70763 .37176 L p .578 g
F P
0 g
s
.44456 .34354 m .44784 .31604 L .53552 .30976 L p .398 g
F P
0 g
s
.44784 .31604 m .44456 .34354 L .36076 .36547 L p .303 g
F P
0 g
s
.69628 .80318 m .65664 .82558 L .64552 .80889 L p .838 g
F P
0 g
s
.64552 .80889 m .68233 .78099 L .69628 .80318 L p .838 g
F P
0 g
s
.31207 .70807 m .3416 .74386 L .32227 .77096 L p .504 g
F P
0 g
s
.35826 .79846 m .32227 .77096 L .3416 .74386 L p .555 g
F P
0 g
s
.61448 .84348 m .57072 .85648 L .56662 .85096 L p .798 g
F P
0 g
s
.44834 .83021 m .48723 .84989 L .48252 .8553 L p .69 g
F P
0 g
s
.48252 .8553 m .43928 .8411 L .44834 .83021 L p .69 g
F P
0 g
s
.29007 .50986 m .29862 .55157 L .25072 .59106 L p .354 g
F P
0 g
s
.26618 .63078 m .25072 .59106 L .29862 .55157 L p .413 g
F P
0 g
s
.65664 .82558 m .61448 .84348 L .60665 .83237 L p .82 g
F P
0 g
s
.60665 .83237 m .64552 .80889 L .65664 .82558 L p .82 g
F P
0 g
s
.56662 .85096 m .60665 .83237 L .61448 .84348 L p .798 g
F P
0 g
s
.53552 .30976 m .5359 .3371 L .44456 .34354 L p .398 g
F P
0 g
s
.35716 .39846 m .36076 .36547 L .44456 .34354 L p .376 g
F P
0 g
s
.3416 .74386 m .37467 .77663 L .35826 .79846 L p .555 g
F P
0 g
s
.39755 .82202 m .35826 .79846 L .37467 .77663 L p .602 g
F P
0 g
s
.77874 .47866 m .7746 .51797 L .70893 .48272 L p .7 g
F P
0 g
s
.70893 .48272 m .71203 .442 L .77874 .47866 L p .7 g
F P
0 g
s
.79339 .63899 m .77236 .67746 L .73384 .64378 L p .805 g
F P
0 g
s
.73384 .64378 m .75199 .60146 L .79339 .63899 L p .805 g
F P
0 g
s
.41053 .80562 m .44834 .83021 L .43928 .8411 L p .646 g
F P
0 g
s
.43928 .8411 m .39755 .82202 L .41053 .80562 L p .646 g
F P
0 g
s
.37467 .77663 m .41053 .80562 L .39755 .82202 L p .602 g
F P
0 g
s
.60665 .83237 m .56662 .85096 L .55971 .84614 L p .784 g
F P
0 g
s
.46298 .82081 m .49467 .84526 L .48723 .84989 L p .696 g
F P
0 g
s
.48723 .84989 m .44834 .83021 L .46298 .82081 L p .696 g
F P
0 g
s
.35716 .39846 m .35672 .43579 L .28611 .47023 L p .338 g
F P
0 g
s
.29007 .50986 m .28611 .47023 L .35672 .43579 L p .397 g
F P
0 g
s
.55971 .84614 m .59311 .8226 L .60665 .83237 L p .784 g
F P
0 g
s
.59311 .8226 m .55971 .84614 L .55045 .84237 L p .769 g
F P
0 g
s
.29862 .55157 m .31168 .59445 L .26618 .63078 L p .413 g
F P
0 g
s
.28676 .67008 m .26618 .63078 L .31168 .59445 L p .463 g
F P
0 g
s
.48222 .81362 m .50434 .84174 L .49467 .84526 L p .704 g
F P
0 g
s
.49467 .84526 m .46298 .82081 L .48222 .81362 L p .704 g
F P
0 g
s
.77236 .67746 m .74646 .71451 L .71158 .68516 L p .808 g
F P
0 g
s
.71158 .68516 m .73384 .64378 L .77236 .67746 L p .808 g
F P
0 g
s
.64552 .80889 m .60665 .83237 L .59311 .8226 L p .796 g
F P
0 g
s
.55045 .84237 m .57472 .8149 L .59311 .8226 L p .769 g
F P
0 g
s
.57472 .8149 m .55045 .84237 L .53948 .83991 L p .754 g
F P
0 g
s
.50473 .80921 m .51557 .83959 L .50434 .84174 L p .714 g
F P
0 g
s
.50434 .84174 m .48222 .81362 L .50473 .80921 L p .714 g
F P
0 g
s
.43196 .79138 m .46298 .82081 L .44834 .83021 L p .658 g
F P
0 g
s
.44834 .83021 m .41053 .80562 L .43196 .79138 L p .658 g
F P
0 g
s
.53948 .83991 m .55277 .80988 L .57472 .8149 L p .754 g
F P
0 g
s
.55277 .80988 m .53948 .83991 L .52757 .83896 L p .74 g
F P
0 g
s
.71156 .40477 m .71203 .442 L .62882 .41746 L p .614 g
F P
0 g
s
.62882 .41746 m .62855 .37979 L .71156 .40477 L p .614 g
F P
0 g
s
.52884 .80793 m .52757 .83896 L .51557 .83959 L p .726 g
F P
0 g
s
.51557 .83959 m .50473 .80921 L .52884 .80793 L p .726 g
F P
0 g
s
.7746 .51797 m .76566 .5592 L .70222 .52606 L p .719 g
F P
0 g
s
.70222 .52606 m .70893 .48272 L .7746 .51797 L p .719 g
F P
0 g
s
.62635 .34688 m .62855 .37979 L .53612 .36996 L p .52 g
F P
0 g
s
.53612 .36996 m .5359 .3371 L .62635 .34688 L p .52 g
F P
0 g
s
.74646 .71451 m .71624 .74929 L .68572 .7246 L p .809 g
F P
0 g
s
.68572 .7246 m .71158 .68516 L .74646 .71451 L p .809 g
F P
0 g
s
.52757 .83896 m .52884 .80793 L .55277 .80988 L p .74 g
F P
0 g
s
.68233 .78099 m .64552 .80889 L .62576 .7941 L p .803 g
F P
0 g
s
.59311 .8226 m .62576 .7941 L .64552 .80889 L p .796 g
F P
0 g
s
.31168 .59445 m .32901 .6375 L .28676 .67008 L p .463 g
F P
0 g
s
.31207 .70807 m .28676 .67008 L .32901 .6375 L p .507 g
F P
0 g
s
.44277 .37644 m .44456 .34354 L .5359 .3371 L p .456 g
F P
0 g
s
.44456 .34354 m .44277 .37644 L .35716 .39846 L p .376 g
F P
0 g
s
.71624 .74929 m .68233 .78099 L .65688 .7612 L p .807 g
F P
0 g
s
.65688 .7612 m .68572 .7246 L .71624 .74929 L p .807 g
F P
0 g
s
.62576 .7941 m .65688 .7612 L .68233 .78099 L p .803 g
F P
0 g
s
.40235 .75754 m .43196 .79138 L .41053 .80562 L p .622 g
F P
0 g
s
.41053 .80562 m .37467 .77663 L .40235 .75754 L p .622 g
F P
0 g
s
.35672 .43579 m .35716 .39846 L .44277 .37644 L p .43 g
F P
0 g
s
.62576 .7941 m .59311 .8226 L .57472 .8149 L p .771 g
F P
0 g
s
.32901 .6375 m .35024 .6797 L .31207 .70807 L p .507 g
F P
0 g
s
.3416 .74386 m .31207 .70807 L .35024 .6797 L p .547 g
F P
0 g
s
.35672 .43579 m .35956 .47675 L .29007 .50986 L p .397 g
F P
0 g
s
.29862 .55157 m .29007 .50986 L .35956 .47675 L p .444 g
F P
0 g
s
.5359 .3371 m .53612 .36996 L .44277 .37644 L p .456 g
F P
0 g
s
.46047 .78042 m .48222 .81362 L .46298 .82081 L p .672 g
F P
0 g
s
.46298 .82081 m .43196 .79138 L .46047 .78042 L p .672 g
F P
0 g
s
.37489 .72003 m .40235 .75754 L .37467 .77663 L p .585 g
F P
0 g
s
.37467 .77663 m .3416 .74386 L .37489 .72003 L p .585 g
F P
0 g
s
.35024 .6797 m .37489 .72003 L .3416 .74386 L p .547 g
F P
0 g
s
.76566 .5592 m .75199 .60146 L .69199 .57103 L p .733 g
F P
0 g
s
.69199 .57103 m .70222 .52606 L .76566 .5592 L p .733 g
F P
0 g
s
.57472 .8149 m .59857 .78238 L .62576 .7941 L p .771 g
F P
0 g
s
.59857 .78238 m .57472 .8149 L .55277 .80988 L p .748 g
F P
0 g
s
.49403 .77367 m .50473 .80921 L .48222 .81362 L p .688 g
F P
0 g
s
.48222 .81362 m .46047 .78042 L .49403 .77367 L p .688 g
F P
0 g
s
.65688 .7612 m .62576 .7941 L .59857 .78238 L p .77 g
F P
0 g
s
.71203 .442 m .70893 .48272 L .62708 .45914 L p .641 g
F P
0 g
s
.62708 .45914 m .62882 .41746 L .71203 .442 L p .641 g
F P
0 g
s
.35956 .47675 m .36571 .52046 L .29862 .55157 L p .444 g
F P
0 g
s
.31168 .59445 m .29862 .55157 L .36571 .52046 L p .484 g
F P
0 g
s
.4396 .74278 m .46047 .78042 L .43196 .79138 L p .642 g
F P
0 g
s
.43196 .79138 m .40235 .75754 L .4396 .74278 L p .642 g
F P
0 g
s
.55277 .80988 m .56587 .77469 L .59857 .78238 L p .748 g
F P
0 g
s
.56587 .77469 m .55277 .80988 L .52884 .80793 L p .726 g
F P
0 g
s
.62855 .37979 m .62882 .41746 L .53614 .4078 L p .558 g
F P
0 g
s
.53614 .4078 m .53612 .36996 L .62855 .37979 L p .558 g
F P
0 g
s
.53009 .7717 m .52884 .80793 L .50473 .80921 L p .706 g
F P
0 g
s
.50473 .80921 m .49403 .77367 L .53009 .7717 L p .706 g
F P
0 g
s
.75199 .60146 m .73384 .64378 L .67844 .61655 L p .745 g
F P
0 g
s
.67844 .61655 m .69199 .57103 L .75199 .60146 L p .745 g
F P
0 g
s
.52884 .80793 m .53009 .7717 L .56587 .77469 L p .726 g
F P
0 g
s
.44277 .37644 m .44255 .41416 L .35672 .43579 L p .43 g
F P
0 g
s
.35956 .47675 m .35672 .43579 L .44255 .41416 L p .473 g
F P
0 g
s
.44255 .41416 m .44277 .37644 L .53612 .36996 L p .499 g
F P
0 g
s
.68572 .7246 m .65688 .7612 L .62143 .74542 L p .766 g
F P
0 g
s
.59857 .78238 m .62143 .74542 L .65688 .7612 L p .77 g
F P
0 g
s
.73384 .64378 m .71158 .68516 L .66188 .66152 L p .753 g
F P
0 g
s
.66188 .66152 m .67844 .61655 L .73384 .64378 L p .753 g
F P
0 g
s
.36571 .52046 m .37508 .56589 L .31168 .59445 L p .484 g
F P
0 g
s
.32901 .6375 m .31168 .59445 L .37508 .56589 L p .52 g
F P
0 g
s
.42016 .7015 m .4396 .74278 L .40235 .75754 L p .612 g
F P
0 g
s
.40235 .75754 m .37489 .72003 L .42016 .7015 L p .612 g
F P
0 g
s
.53612 .36996 m .53614 .4078 L .44255 .41416 L p .499 g
F P
0 g
s
.71158 .68516 m .68572 .7246 L .64271 .70482 L p .761 g
F P
0 g
s
.62143 .74542 m .64271 .70482 L .68572 .7246 L p .766 g
F P
0 g
s
.62143 .74542 m .59857 .78238 L .56587 .77469 L p .739 g
F P
0 g
s
.64271 .70482 m .66188 .66152 L .71158 .68516 L p .761 g
F P
0 g
s
.70893 .48272 m .70222 .52606 L .62333 .50391 L p .662 g
F P
0 g
s
.62333 .50391 m .62708 .45914 L .70893 .48272 L p .662 g
F P
0 g
s
.48373 .73365 m .49403 .77367 L .46047 .78042 L p .663 g
F P
0 g
s
.46047 .78042 m .4396 .74278 L .48373 .73365 L p .663 g
F P
0 g
s
.37508 .56589 m .38748 .61197 L .32901 .6375 L p .52 g
F P
0 g
s
.35024 .6797 m .32901 .6375 L .38748 .61197 L p .552 g
F P
0 g
s
.40264 .65755 m .42016 .7015 L .37489 .72003 L p .582 g
F P
0 g
s
.37489 .72003 m .35024 .6797 L .40264 .65755 L p .582 g
F P
0 g
s
.38748 .61197 m .40264 .65755 L .35024 .6797 L p .552 g
F P
0 g
s
.44255 .41416 m .44397 .45597 L .35956 .47675 L p .473 g
F P
0 g
s
.36571 .52046 m .35956 .47675 L .44397 .45597 L p .509 g
F P
0 g
s
.56587 .77469 m .57846 .73503 L .62143 .74542 L p .739 g
F P
0 g
s
.57846 .73503 m .56587 .77469 L .53009 .7717 L p .711 g
F P
0 g
s
.62882 .41746 m .62708 .45914 L .53597 .44986 L p .587 g
F P
0 g
s
.53597 .44986 m .53614 .4078 L .62882 .41746 L p .587 g
F P
0 g
s
.5313 .73098 m .53009 .7717 L .49403 .77367 L p .686 g
F P
0 g
s
.49403 .77367 m .48373 .73365 L .5313 .73098 L p .686 g
F P
0 g
s
.53009 .7717 m .5313 .73098 L .57846 .73503 L p .711 g
F P
0 g
s
.44397 .45597 m .44255 .41416 L .53614 .4078 L p .533 g
F P
0 g
s
.64271 .70482 m .62143 .74542 L .57846 .73503 L p .729 g
F P
0 g
s
.70222 .52606 m .69199 .57103 L .61762 .55073 L p .679 g
F P
0 g
s
.61762 .55073 m .62333 .50391 L .70222 .52606 L p .679 g
F P
0 g
s
.47412 .69 m .48373 .73365 L .4396 .74278 L p .639 g
F P
0 g
s
.4396 .74278 m .42016 .7015 L .47412 .69 L p .639 g
F P
0 g
s
.53614 .4078 m .53597 .44986 L .44397 .45597 L p .533 g
F P
0 g
s
.44397 .45597 m .44704 .50094 L .36571 .52046 L p .509 g
F P
0 g
s
.37508 .56589 m .36571 .52046 L .44704 .50094 L p .539 g
F P
0 g
s
.66188 .66152 m .64271 .70482 L .59023 .69175 L p .718 g
F P
0 g
s
.57846 .73503 m .59023 .69175 L .64271 .70482 L p .729 g
F P
0 g
s
.69199 .57103 m .67844 .61655 L .61007 .59844 L p .694 g
F P
0 g
s
.61007 .59844 m .61762 .55073 L .69199 .57103 L p .694 g
F P
0 g
s
.59023 .69175 m .57846 .73503 L .5313 .73098 L p .696 g
F P
0 g
s
.62708 .45914 m .62333 .50391 L .53561 .49521 L p .611 g
F P
0 g
s
.53561 .49521 m .53597 .44986 L .62708 .45914 L p .611 g
F P
0 g
s
.46542 .64375 m .47412 .69 L .42016 .7015 L p .615 g
F P
0 g
s
.42016 .7015 m .40264 .65755 L .46542 .64375 L p .615 g
F P
0 g
s
.67844 .61655 m .66188 .66152 L .60086 .64584 L p .707 g
F P
0 g
s
.59023 .69175 m .60086 .64584 L .66188 .66152 L p .718 g
F P
0 g
s
.53243 .68664 m .5313 .73098 L .48373 .73365 L p .667 g
F P
0 g
s
.48373 .73365 m .47412 .69 L .53243 .68664 L p .667 g
F P
0 g
s
.60086 .64584 m .61007 .59844 L .67844 .61655 L p .707 g
F P
0 g
s
.44704 .50094 m .45171 .54801 L .37508 .56589 L p .539 g
F P
0 g
s
.38748 .61197 m .37508 .56589 L .45171 .54801 L p .566 g
F P
0 g
s
.44704 .50094 m .44397 .45597 L .53597 .44986 L p .562 g
F P
0 g
s
.5313 .73098 m .53243 .68664 L .59023 .69175 L p .696 g
F P
0 g
s
.45789 .59602 m .46542 .64375 L .40264 .65755 L p .591 g
F P
0 g
s
.40264 .65755 m .38748 .61197 L .45789 .59602 L p .591 g
F P
0 g
s
.45171 .54801 m .45789 .59602 L .38748 .61197 L p .566 g
F P
0 g
s
.53597 .44986 m .53561 .49521 L .44704 .50094 L p .562 g
F P
0 g
s
.60086 .64584 m .59023 .69175 L .53243 .68664 L p .681 g
F P
0 g
s
.62333 .50391 m .61762 .55073 L .53506 .54276 L p .631 g
F P
0 g
s
.53506 .54276 m .53561 .49521 L .62333 .50391 L p .631 g
F P
0 g
s
.53345 .6397 m .53243 .68664 L .47412 .69 L p .647 g
F P
0 g
s
.47412 .69 m .46542 .64375 L .53345 .6397 L p .647 g
F P
0 g
s
.45171 .54801 m .44704 .50094 L .53561 .49521 L p .586 g
F P
0 g
s
.53243 .68664 m .53345 .6397 L .60086 .64584 L p .681 g
F P
0 g
s
.61007 .59844 m .60086 .64584 L .53345 .6397 L p .666 g
F P
0 g
s
.61762 .55073 m .61007 .59844 L .53434 .59134 L p .649 g
F P
0 g
s
.53434 .59134 m .53506 .54276 L .61762 .55073 L p .649 g
F P
0 g
s
.53561 .49521 m .53506 .54276 L .45171 .54801 L p .586 g
F P
0 g
s
.53434 .59134 m .53345 .6397 L .46542 .64375 L p .628 g
F P
0 g
s
.46542 .64375 m .45789 .59602 L .53434 .59134 L p .628 g
F P
0 g
s
.53345 .6397 m .53434 .59134 L .61007 .59844 L p .666 g
F P
0 g
s
.45789 .59602 m .45171 .54801 L .53506 .54276 L p .608 g
F P
0 g
s
.53506 .54276 m .53434 .59134 L .45789 .59602 L p .608 g
F P
0 g
s
.25 Mabswid
.67245 0 m
.92713 .41497 L
s
.92713 .41497 m
1 .94814 L
s
1 .94814 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.08846 .24555 m
0 .80374 L
s
0 .80374 m
.70644 .58546 L
s
.70644 .58546 m
.67245 0 L
s
.67245 0 m
.08846 .24555 L
s
.08846 .24555 m
.67245 0 L
s
.10087 .24033 m
.10595 .24556 L
s
gsave
.09072 .22988 -74.7418 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.22871 .18658 m
.23354 .19204 L
s
gsave
.21906 .17566 -80.4345 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.36358 .12987 m
.36813 .13557 L
s
gsave
.3545 .11847 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.50609 .06995 m
.51031 .07589 L
s
gsave
.49765 .05807 -73.9305 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.65689 .00654 m
.66074 .01273 L
s
gsave
.64919 -0.00583 -67.8447 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12591 .22981 m
.12892 .23297 L
s
.1512 .21917 m
.15419 .22236 L
s
.17676 .20842 m
.17972 .21164 L
s
.2026 .19756 m
.20553 .20081 L
s
.25511 .17548 m
.25797 .17879 L
s
.28179 .16426 m
.28462 .1676 L
s
.30875 .15292 m
.31155 .15628 L
s
.33602 .14146 m
.33878 .14485 L
s
.39145 .11815 m
.39414 .1216 L
s
.41963 .1063 m
.42229 .10978 L
s
.44813 .09432 m
.45074 .09783 L
s
.47695 .0822 m
.47952 .08574 L
s
.53556 .05756 m
.53805 .06115 L
s
.56537 .04502 m
.56782 .04865 L
s
.59553 .03235 m
.59793 .036 L
s
.62603 .01952 m
.62839 .0232 L
s
gsave
.30907 .06148 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.35119 0.0080766 0.64881 0.331141 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.30443 1.63518 -0.0106411 1.63518 [
[.5 1.63703 -19.375 0 ]
[.5 1.63703 19.375 10 ]
[.08347 .2009 -19.4345 -10 ]
[.08347 .2009 1.19055 0 ]
[.21284 .14994 -23.0684 -10 ]
[.21284 .14994 1.99412 0 ]
[.3456 .09765 -7.58187 -10 ]
[.3456 .09765 .85563 0 ]
[.4819 .04399 -17.1514 -10 ]
[.4819 .04399 2.41108 0 ]
[.62186 -0.01111 -12.9305 -10 ]
[.62186 -0.01111 2.19452 0 ]
[.2819 .01774 -7.13257 -10 ]
[.2819 .01774 .80493 0 ]
[.67411 .0028 0 -6.95638 ]
[.67411 .0028 20.625 3.04362 ]
[.74455 .10274 0 -6.89654 ]
[.74455 .10274 25.0625 3.10346 ]
[.81167 .19797 0 -6.84024 ]
[.81167 .19797 8.4375 3.15976 ]
[.87569 .28881 0 -6.7872 ]
[.87569 .28881 19.5625 3.2128 ]
[.93683 .37557 0 -6.73712 ]
[.93683 .37557 15.125 3.26288 ]
[.90758 .16267 0 -6.84024 ]
[.90758 .16267 7.9375 3.15976 ]
[.06357 .25073 -13.9375 -3.04815 ]
[.06357 .25073 0 6.95185 ]
[.04583 .48129 -20.625 -3.16049 ]
[.04583 .48129 0 6.83951 ]
[.02659 .73148 -8.4375 -3.28285 ]
[.02659 .73148 0 6.71715 ]
[.00564 1.00394 -15.125 -3.41663 ]
[.00564 1.00394 0 6.58337 ]
[-0.01725 1.30176 -8.4375 -3.56352 ]
[-0.01725 1.30176 0 6.43648 ]
[-0.07007 .76468 -7.9375 -3.28285 ]
[-0.07007 .76468 0 6.71715 ]
[ 0 0 0 0 ]
[ 1 1.62453 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.63703 -80.375 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(p) show
67.437500 11.500000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
70.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
76.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
97.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
.64785 0 L
s
.09701 .21621 m
.10379 .22387 L
s
gsave
.08347 .2009 -80.4345 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.226 .16558 m
.23257 .17341 L
s
gsave
.21284 .14994 -84.0684 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.35834 .11363 m
.36471 .12163 L
s
gsave
.3456 .09765 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4942 .06031 m
.50035 .06847 L
s
gsave
.4819 .04399 -78.1514 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.63369 .00556 m
.63961 .01389 L
s
gsave
.62186 -0.01111 -73.9305 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12255 .20619 m
.12659 .2108 L
s
.14821 .19612 m
.15223 .20075 L
s
.17401 .18599 m
.178 .19064 L
s
.19993 .17581 m
.2039 .18049 L
s
.25219 .1553 m
.25611 .16001 L
s
.27852 .14497 m
.28242 .1497 L
s
.30499 .13458 m
.30886 .13933 L
s
.3316 .12413 m
.33545 .12891 L
s
.38523 .10308 m
.38903 .1079 L
s
.41226 .09247 m
.41603 .09731 L
s
.43943 .08181 m
.44317 .08666 L
s
.46674 .07109 m
.47046 .07596 L
s
.5218 .04948 m
.52546 .05439 L
s
.54955 .03858 m
.55318 .04352 L
s
.57745 .02763 m
.58105 .03259 L
s
.60549 .01662 m
.60907 .0216 L
s
gsave
.2819 .01774 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.64785 0 m
.92348 .39072 L
s
.65508 .01025 m
.64556 .01397 L
s
gsave
.67411 .0028 -61 -10.9564 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.72544 .10999 m
.71588 .11361 L
s
gsave
.74455 .10274 -61 -10.8965 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.79249 .20503 m
.78289 .20856 L
s
gsave
.81167 .19797 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.85644 .29569 m
.84682 .29913 L
s
gsave
.87569 .28881 -61 -10.7872 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.91753 .38228 m
.90787 .38563 L
s
gsave
.93683 .37557 -61 -10.7371 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.66943 .03059 m
.66371 .03281 L
s
.68363 .05073 m
.67791 .05294 L
s
.69771 .07067 m
.69198 .07287 L
s
.71164 .09043 m
.70591 .09261 L
s
.73911 .12936 m
.73337 .13152 L
s
.75264 .14855 m
.7469 .1507 L
s
.76605 .16756 m
.7603 .1697 L
s
.77933 .18638 m
.77358 .18851 L
s
.80552 .2235 m
.79976 .22561 L
s
.81843 .2418 m
.81266 .2439 L
s
.83122 .25993 m
.82545 .26202 L
s
.84389 .2779 m
.83812 .27997 L
s
.86888 .31333 m
.86311 .31538 L
s
.88121 .3308 m
.87543 .33284 L
s
.89343 .34812 m
.88764 .35015 L
s
.90553 .36527 m
.89974 .3673 L
s
gsave
.90758 .16267 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
0 1.32748 L
s
.08261 .2433 m
.09213 .23959 L
s
gsave
.06357 .25073 -74.9375 -7.04815 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.06502 .47423 m
.07461 .4707 L
s
gsave
.04583 .48129 -81.625 -7.16049 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.04592 .72484 m
.05559 .72152 L
s
gsave
.02659 .73148 -69.4375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.02512 .99776 m
.03487 .99468 L
s
gsave
.00564 1.00394 -76.125 -7.41663 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00239 1.29611 m
.01221 1.29329 L
s
gsave
-0.01725 1.30176 -69.4375 -7.56352 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.07921 .28803 m
.08493 .28582 L
s
.07574 .33346 m
.08147 .33128 L
s
.07222 .37963 m
.07796 .37747 L
s
.06865 .42655 m
.0744 .42441 L
s
.06132 .5227 m
.06709 .52061 L
s
.05757 .57198 m
.06334 .56991 L
s
.05375 .62208 m
.05953 .62004 L
s
.04987 .67303 m
.05566 .67101 L
s
.0419 .77755 m
.04771 .77559 L
s
.03782 .83117 m
.04364 .82924 L
s
.03366 .88573 m
.03949 .88382 L
s
.02943 .94125 m
.03527 .93937 L
s
.02074 1.05529 m
.0266 1.05347 L
s
.01628 1.11386 m
.02214 1.11207 L
s
.01173 1.1735 m
.01761 1.17174 L
s
.0071 1.23424 m
.01299 1.23251 L
s
gsave
-0.07007 .76468 -68.9375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.37669 1.62453 L
s
.37669 1.62453 m
.3971 .57309 L
s
.3971 .57309 m
.0843 .2212 L
s
.64785 0 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.64785 0 m
.0843 .2212 L
s
.3971 .57309 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.37669 1.62453 L
s
.37669 1.62453 m
.3971 .57309 L
s
0 0 m
1 0 L
1 1.62453 L
0 1.62453 L
closepath
clip
newpath
.5 Mabswid
.45497 .52499 m .44189 .59924 L .50638 .60568 L p .652 g
F P
0 g
s
.50638 .60568 m .50762 .53036 L .45497 .52499 L p .652 g
F P
0 g
s
.50556 .67826 m .50638 .60568 L .44189 .59924 L p .624 g
F P
0 g
s
.50762 .53036 m .50638 .60568 L .57126 .60145 L p .621 g
F P
0 g
s
.50638 .60568 m .50556 .67826 L .57823 .67362 L p .587 g
F P
0 g
s
.57823 .67362 m .57126 .60145 L .50638 .60568 L p .587 g
F P
0 g
s
.47184 .4526 m .45497 .52499 L .50762 .53036 L p .675 g
F P
0 g
s
.50762 .53036 m .50922 .4565 L .47184 .4526 L p .675 g
F P
0 g
s
.50922 .4565 m .50762 .53036 L .5606 .52683 L p .651 g
F P
0 g
s
.57126 .60145 m .5606 .52683 L .50762 .53036 L p .621 g
F P
0 g
s
.5606 .52683 m .57126 .60145 L .63268 .58675 L p .591 g
F P
0 g
s
.57126 .60145 m .57823 .67362 L .64708 .65749 L p .55 g
F P
0 g
s
.64708 .65749 m .63268 .58675 L .57126 .60145 L p .55 g
F P
0 g
s
.5606 .52683 m .54685 .45394 L .50922 .4565 L p .651 g
F P
0 g
s
.54685 .45394 m .5606 .52683 L .61069 .51459 L p .629 g
F P
0 g
s
.63268 .58675 m .61069 .51459 L .5606 .52683 L p .591 g
F P
0 g
s
.40588 .51101 m .38166 .58245 L .44189 .59924 L p .685 g
F P
0 g
s
.44189 .59924 m .45497 .52499 L .40588 .51101 L p .685 g
F P
0 g
s
.43334 .6712 m .44189 .59924 L .38166 .58245 L p .663 g
F P
0 g
s
.44189 .59924 m .43334 .6712 L .50556 .67826 L p .624 g
F P
0 g
s
.43707 .44249 m .40588 .51101 L .45497 .52499 L p .702 g
F P
0 g
s
.45497 .52499 m .47184 .4526 L .43707 .44249 L p .702 g
F P
0 g
s
.50523 .744 m .50556 .67826 L .43334 .6712 L p .588 g
F P
0 g
s
.50556 .67826 m .50523 .744 L .58107 .73924 L p .545 g
F P
0 g
s
.58107 .73924 m .57823 .67362 L .50556 .67826 L p .545 g
F P
0 g
s
.61069 .51459 m .58236 .44508 L .54685 .45394 L p .629 g
F P
0 g
s
.58236 .44508 m .61069 .51459 L .65476 .49431 L p .609 g
F P
0 g
s
.61069 .51459 m .63268 .58675 L .68691 .56236 L p .562 g
F P
0 g
s
.68691 .56236 m .65476 .49431 L .61069 .51459 L p .562 g
F P
0 g
s
.49159 .3862 m .47184 .4526 L .50922 .4565 L p .698 g
F P
0 g
s
.50922 .4565 m .5111 .38827 L .49159 .3862 L p .698 g
F P
0 g
s
.5111 .38827 m .50922 .4565 L .54685 .45394 L p .681 g
F P
0 g
s
.57823 .67362 m .58107 .73924 L .65296 .72273 L p .498 g
F P
0 g
s
.65296 .72273 m .64708 .65749 L .57823 .67362 L p .498 g
F P
0 g
s
.63268 .58675 m .64708 .65749 L .708 .63068 L p .509 g
F P
0 g
s
.708 .63068 m .68691 .56236 L .63268 .58675 L p .509 g
F P
0 g
s
.54685 .45394 m .53075 .38691 L .5111 .38827 L p .681 g
F P
0 g
s
.53075 .38691 m .54685 .45394 L .58236 .44508 L p .666 g
F P
0 g
s
.40715 .42674 m .36344 .48918 L .40588 .51101 L p .731 g
F P
0 g
s
.40588 .51101 m .43707 .44249 L .40715 .42674 L p .731 g
F P
0 g
s
.38166 .58245 m .40588 .51101 L .36344 .48918 L p .722 g
F P
0 g
s
.38166 .58245 m .3658 .65277 L .43334 .6712 L p .663 g
F P
0 g
s
.42986 .73676 m .43334 .6712 L .3658 .65277 L p .633 g
F P
0 g
s
.43334 .6712 m .42986 .73676 L .50523 .744 L p .588 g
F P
0 g
s
.4735 .38083 m .43707 .44249 L .47184 .4526 L p .716 g
F P
0 g
s
.47184 .4526 m .49159 .3862 L .4735 .38083 L p .716 g
F P
0 g
s
.36344 .48918 m .32938 .55619 L .38166 .58245 L p .722 g
F P
0 g
s
.3658 .65277 m .38166 .58245 L .32938 .55619 L p .707 g
F P
0 g
s
.58236 .44508 m .54924 .3822 L .53075 .38691 L p .666 g
F P
0 g
s
.54924 .3822 m .58236 .44508 L .61346 .43043 L p .654 g
F P
0 g
s
.65476 .49431 m .61346 .43043 L .58236 .44508 L p .609 g
F P
0 g
s
.45802 .3725 m .40715 .42674 L .43707 .44249 L p .736 g
F P
0 g
s
.43707 .44249 m .4735 .38083 L .45802 .3725 L p .736 g
F P
0 g
s
.64708 .65749 m .65296 .72273 L .71662 .69525 L p .442 g
F P
0 g
s
.71662 .69525 m .708 .63068 L .64708 .65749 L p .442 g
F P
0 g
s
.61346 .43043 m .65476 .49431 L .68994 .46712 L p .589 g
F P
0 g
s
.65476 .49431 m .68691 .56236 L .73044 .52957 L p .531 g
F P
0 g
s
.73044 .52957 m .68994 .46712 L .65476 .49431 L p .531 g
F P
0 g
s
.32938 .55619 m .30703 .62389 L .3658 .65277 L p .707 g
F P
0 g
s
.35932 .71789 m .3658 .65277 L .30703 .62389 L p .681 g
F P
0 g
s
.3658 .65277 m .35932 .71789 L .42986 .73676 L p .633 g
F P
0 g
s
.38407 .40631 m .33043 .46075 L .36344 .48918 L p .764 g
F P
0 g
s
.36344 .48918 m .40715 .42674 L .38407 .40631 L p .764 g
F P
0 g
s
.32938 .55619 m .36344 .48918 L .33043 .46075 L p .765 g
F P
0 g
s
.50541 .79893 m .50523 .744 L .42986 .73676 L p .535 g
F P
0 g
s
.50523 .744 m .50541 .79893 L .57952 .79436 L p .484 g
F P
0 g
s
.57952 .79436 m .58107 .73924 L .50523 .744 L p .484 g
F P
0 g
s
.61346 .43043 m .56535 .37445 L .54924 .3822 L p .654 g
F P
0 g
s
.56535 .37445 m .61346 .43043 L .63811 .41088 L p .644 g
F P
0 g
s
.68994 .46712 m .63811 .41088 L .61346 .43043 L p .589 g
F P
0 g
s
.68691 .56236 m .708 .63068 L .75709 .59457 L p .461 g
F P
0 g
s
.75709 .59457 m .73044 .52957 L .68691 .56236 L p .461 g
F P
0 g
s
.44619 .36175 m .38407 .40631 L .40715 .42674 L p .757 g
F P
0 g
s
.40715 .42674 m .45802 .3725 L .44619 .36175 L p .757 g
F P
0 g
s
.58107 .73924 m .57952 .79436 L .64976 .77848 L p .423 g
F P
0 g
s
.64976 .77848 m .65296 .72273 L .58107 .73924 L p .423 g
F P
0 g
s
.5111 .38827 m .51316 .32978 L .49159 .3862 L closepath p .719 g
F P
0 g
s
.53075 .38691 m .51316 .32978 L .5111 .38827 L closepath p .713 g
F P
0 g
s
.33043 .46075 m .28847 .52187 L .32938 .55619 L p .765 g
F P
0 g
s
.30703 .62389 m .32938 .55619 L .28847 .52187 L p .758 g
F P
0 g
s
.54924 .3822 m .51316 .32978 L .53075 .38691 L closepath p .707 g
F P
0 g
s
.43176 .79198 m .42986 .73676 L .35932 .71789 L p .584 g
F P
0 g
s
.42986 .73676 m .43176 .79198 L .50541 .79893 L p .535 g
F P
0 g
s
.49159 .3862 m .51316 .32978 L .4735 .38083 L closepath p .727 g
F P
0 g
s
.56535 .37445 m .51316 .32978 L .54924 .3822 L closepath p .703 g
F P
0 g
s
.4735 .38083 m .51316 .32978 L .45802 .3725 L closepath p .734 g
F P
0 g
s
.63811 .41088 m .57801 .36415 L .56535 .37445 L p .644 g
F P
0 g
s
.57801 .36415 m .63811 .41088 L .65459 .38762 L p .637 g
F P
0 g
s
.63811 .41088 m .68994 .46712 L .71377 .43463 L p .572 g
F P
0 g
s
.71377 .43463 m .65459 .38762 L .63811 .41088 L p .572 g
F P
0 g
s
.57801 .36415 m .51316 .32978 L .56535 .37445 L closepath p .7 g
F P
0 g
s
.708 .63068 m .71662 .69525 L .76799 .65822 L p .367 g
F P
0 g
s
.76799 .65822 m .75709 .59457 L .708 .63068 L p .367 g
F P
0 g
s
.68994 .46712 m .73044 .52957 L .76024 .49023 L p .495 g
F P
0 g
s
.76024 .49023 m .71377 .43463 L .68994 .46712 L p .495 g
F P
0 g
s
.45802 .3725 m .51316 .32978 L .44619 .36175 L closepath p .742 g
F P
0 g
s
.65296 .72273 m .64976 .77848 L .71192 .75208 L p .343 g
F P
0 g
s
.71192 .75208 m .71662 .69525 L .65296 .72273 L p .343 g
F P
0 g
s
.43884 .34927 m .36946 .38246 L .38407 .40631 L p .779 g
F P
0 g
s
.38407 .40631 m .44619 .36175 L .43884 .34927 L p .779 g
F P
0 g
s
.33043 .46075 m .38407 .40631 L .36946 .38246 L p .8 g
F P
0 g
s
.36946 .38246 m .3092 .4274 L .33043 .46075 L p .8 g
F P
0 g
s
.28847 .52187 m .33043 .46075 L .3092 .4274 L p .814 g
F P
0 g
s
.28847 .52187 m .26085 .58608 L .30703 .62389 L p .758 g
F P
0 g
s
.2979 .68829 m .30703 .62389 L .26085 .58608 L p .737 g
F P
0 g
s
.30703 .62389 m .2979 .68829 L .35932 .71789 L p .681 g
F P
0 g
s
.36286 .77383 m .35932 .71789 L .2979 .68829 L p .633 g
F P
0 g
s
.35932 .71789 m .36286 .77383 L .43176 .79198 L p .584 g
F P
0 g
s
.58635 .35196 m .51316 .32978 L .57801 .36415 L closepath p .699 g
F P
0 g
s
.65459 .38762 m .58635 .35196 L .57801 .36415 L p .637 g
F P
0 g
s
.44619 .36175 m .51316 .32978 L .43884 .34927 L closepath p .749 g
F P
0 g
s
.73044 .52957 m .75709 .59457 L .79091 .55114 L p .397 g
F P
0 g
s
.79091 .55114 m .76024 .49023 L .73044 .52957 L p .397 g
F P
0 g
s
.58975 .3387 m .51316 .32978 L .58635 .35196 L closepath p .7 g
F P
0 g
s
.58635 .35196 m .65459 .38762 L .66165 .36215 L p .634 g
F P
0 g
s
.66165 .36215 m .58975 .3387 L .58635 .35196 L p .634 g
F P
0 g
s
.3092 .4274 m .26185 .48145 L .28847 .52187 L p .814 g
F P
0 g
s
.26085 .58608 m .28847 .52187 L .26185 .48145 L p .818 g
F P
0 g
s
.65459 .38762 m .71377 .43463 L .72439 .39884 L p .555 g
F P
0 g
s
.72439 .39884 m .66165 .36215 L .65459 .38762 L p .555 g
F P
0 g
s
.50611 .8394 m .50541 .79893 L .43176 .79198 L p .441 g
F P
0 g
s
.50541 .79893 m .50611 .8394 L .57352 .83529 L p .378 g
F P
0 g
s
.57352 .83529 m .57952 .79436 L .50541 .79893 L p .378 g
F P
0 g
s
.43884 .34927 m .51316 .32978 L .4365 .33589 L closepath p .755 g
F P
0 g
s
.36946 .38246 m .43884 .34927 L .4365 .33589 L p .799 g
F P
0 g
s
.4365 .33589 m .36445 .35673 L .36946 .38246 L p .799 g
F P
0 g
s
.3092 .4274 m .36946 .38246 L .36445 .35673 L p .838 g
F P
0 g
s
.57952 .79436 m .57352 .83529 L .63736 .82103 L p .294 g
F P
0 g
s
.63736 .82103 m .64976 .77848 L .57952 .79436 L p .294 g
F P
0 g
s
.43912 .83315 m .43176 .79198 L .36286 .77383 L p .49 g
F P
0 g
s
.43176 .79198 m .43912 .83315 L .50611 .8394 L p .441 g
F P
0 g
s
.58795 .32524 m .51316 .32978 L .58975 .3387 L closepath p .702 g
F P
0 g
s
.4365 .33589 m .51316 .32978 L .4394 .32251 L closepath p .759 g
F P
0 g
s
.581 .31252 m .51316 .32978 L .58795 .32524 L closepath p .707 g
F P
0 g
s
.56934 .30142 m .51316 .32978 L .581 .31252 L closepath p .713 g
F P
0 g
s
.4394 .32251 m .51316 .32978 L .44737 .31005 L closepath p .762 g
F P
0 g
s
.55374 .29274 m .51316 .32978 L .56934 .30142 L closepath p .721 g
F P
0 g
s
.44737 .31005 m .51316 .32978 L .45992 .29939 L closepath p .762 g
F P
0 g
s
.53529 .2871 m .51316 .32978 L .55374 .29274 L closepath p .73 g
F P
0 g
s
.45992 .29939 m .51316 .32978 L .47621 .2913 L closepath p .759 g
F P
0 g
s
.51527 .28492 m .51316 .32978 L .53529 .2871 L closepath p .739 g
F P
0 g
s
.4951 .28635 m .51316 .32978 L .51527 .28492 L closepath p .747 g
F P
0 g
s
.47621 .2913 m .51316 .32978 L .4951 .28635 L closepath p .754 g
F P
0 g
s
.71662 .69525 m .71192 .75208 L .76205 .71651 L p .222 g
F P
0 g
s
.76205 .71651 m .76799 .65822 L .71662 .69525 L p .222 g
F P
0 g
s
.71377 .43463 m .76024 .49023 L .7739 .44671 L p .452 g
F P
0 g
s
.7739 .44671 m .72439 .39884 L .71377 .43463 L p .452 g
F P
0 g
s
.58975 .3387 m .66165 .36215 L .65863 .33613 L p .635 g
F P
0 g
s
.65863 .33613 m .58795 .32524 L .58975 .3387 L p .635 g
F P
0 g
s
.36445 .35673 m .4365 .33589 L .4394 .32251 L p .817 g
F P
0 g
s
.75709 .59457 m .76799 .65822 L .80348 .61365 L p .253 g
F P
0 g
s
.80348 .61365 m .79091 .55114 L .75709 .59457 L p .253 g
F P
0 g
s
.26085 .58608 m .24954 .64951 L .2979 .68829 L p .737 g
F P
0 g
s
.30288 .7454 m .2979 .68829 L .24954 .64951 L p .683 g
F P
0 g
s
.2979 .68829 m .30288 .7454 L .36286 .77383 L p .633 g
F P
0 g
s
.36445 .35673 m .30149 .3912 L .3092 .4274 L p .838 g
F P
0 g
s
.26185 .48145 m .3092 .4274 L .30149 .3912 L p .87 g
F P
0 g
s
.64976 .77848 m .63736 .82103 L .69375 .79734 L p .169 g
F P
0 g
s
.69375 .79734 m .71192 .75208 L .64976 .77848 L p .169 g
F P
0 g
s
.26185 .48145 m .23057 .54144 L .26085 .58608 L p .818 g
F P
0 g
s
.24954 .64951 m .26085 .58608 L .23057 .54144 L p .8 g
F P
0 g
s
.66165 .36215 m .72439 .39884 L .7207 .36208 L p .54 g
F P
0 g
s
.7207 .36208 m .65863 .33613 L .66165 .36215 L p .54 g
F P
0 g
s
.58795 .32524 m .65863 .33613 L .64553 .31138 L p .641 g
F P
0 g
s
.64553 .31138 m .581 .31252 L .58795 .32524 L p .641 g
F P
0 g
s
.36959 .33083 m .4394 .32251 L .44737 .31005 L p .83 g
F P
0 g
s
.4394 .32251 m .36959 .33083 L .36445 .35673 L p .817 g
F P
0 g
s
.37652 .81685 m .36286 .77383 L .30288 .7454 L p .527 g
F P
0 g
s
.36286 .77383 m .37652 .81685 L .43912 .83315 L p .49 g
F P
0 g
s
.30149 .3912 m .36445 .35673 L .36959 .33083 L p .875 g
F P
0 g
s
.76024 .49023 m .79091 .55114 L .8067 .50297 L p .299 g
F P
0 g
s
.8067 .50297 m .7739 .44671 L .76024 .49023 L p .299 g
F P
0 g
s
.581 .31252 m .64553 .31138 L .62308 .28965 L p .654 g
F P
0 g
s
.62308 .28965 m .56934 .30142 L .581 .31252 L p .654 g
F P
0 g
s
.38472 .30656 m .44737 .31005 L .45992 .29939 L p .832 g
F P
0 g
s
.44737 .31005 m .38472 .30656 L .36959 .33083 L p .83 g
F P
0 g
s
.30149 .3912 m .25176 .43741 L .26185 .48145 L p .87 g
F P
0 g
s
.23057 .54144 m .26185 .48145 L .25176 .43741 L p .885 g
F P
0 g
s
.72439 .39884 m .7739 .44671 L .76993 .40182 L p .391 g
F P
0 g
s
.76993 .40182 m .7207 .36208 L .72439 .39884 L p .391 g
F P
0 g
s
.56934 .30142 m .62308 .28965 L .59274 .27258 L p .674 g
F P
0 g
s
.59274 .27258 m .55374 .29274 L .56934 .30142 L p .674 g
F P
0 g
s
.76799 .65822 m .76205 .71651 L .79662 .67372 L p .061 g
F P
0 g
s
.79662 .67372 m .80348 .61365 L .76799 .65822 L p .061 g
F P
0 g
s
.40895 .28568 m .45992 .29939 L .47621 .2913 L p .823 g
F P
0 g
s
.45992 .29939 m .40895 .28568 L .38472 .30656 L p .832 g
F P
0 g
s
.65863 .33613 m .7207 .36208 L .70253 .32691 L p .526 g
F P
0 g
s
.70253 .32691 m .64553 .31138 L .65863 .33613 L p .526 g
F P
0 g
s
.71192 .75208 m .69375 .79734 L .73908 .76549 L p .058 g
F P
0 g
s
.73908 .76549 m .76205 .71651 L .71192 .75208 L p .058 g
F P
0 g
s
.50731 .86221 m .50611 .8394 L .43912 .83315 L p .225 g
F P
0 g
s
.50611 .8394 m .50731 .86221 L .56326 .85883 L p .148 g
F P
0 g
s
.56326 .85883 m .57352 .83529 L .50611 .8394 L p .148 g
F P
0 g
s
.36959 .33083 m .30821 .35457 L .30149 .3912 L p .875 g
F P
0 g
s
.25176 .43741 m .30149 .3912 L .30821 .35457 L p .922 g
F P
0 g
s
.30821 .35457 m .36959 .33083 L .38472 .30656 L p .899 g
F P
0 g
s
.55374 .29274 m .59274 .27258 L .55662 .26145 L p .702 g
F P
0 g
s
.55662 .26145 m .53529 .2871 L .55374 .29274 L p .702 g
F P
0 g
s
.44068 .26974 m .47621 .2913 L .4951 .28635 L p .801 g
F P
0 g
s
.47621 .2913 m .44068 .26974 L .40895 .28568 L p .823 g
F P
0 g
s
.23057 .54144 m .21774 .60368 L .24954 .64951 L p .8 g
F P
0 g
s
.25571 .70815 m .24954 .64951 L .21774 .60368 L p .713 g
F P
0 g
s
.24954 .64951 m .25571 .70815 L .30288 .7454 L p .683 g
F P
0 g
s
.57352 .83529 m .56326 .85883 L .61617 .84709 L p .099 g
F P
0 g
s
.61617 .84709 m .63736 .82103 L .57352 .83529 L p .099 g
F P
0 g
s
.32214 .79135 m .30288 .7454 L .25571 .70815 L p .533 g
F P
0 g
s
.30288 .7454 m .32214 .79135 L .37652 .81685 L p .527 g
F P
0 g
s
.79091 .55114 m .80348 .61365 L .82018 .56414 L p .057 g
F P
0 g
s
.82018 .56414 m .8067 .50297 L .79091 .55114 L p .057 g
F P
0 g
s
.45171 .85706 m .43912 .83315 L .37652 .81685 L p .263 g
F P
0 g
s
.43912 .83315 m .45171 .85706 L .50731 .86221 L p .225 g
F P
0 g
s
.53529 .2871 m .55662 .26145 L .51731 .25714 L p .734 g
F P
0 g
s
.51731 .25714 m .51527 .28492 L .53529 .2871 L p .734 g
F P
0 g
s
.47769 .25997 m .4951 .28635 L .51527 .28492 L p .769 g
F P
0 g
s
.4951 .28635 m .47769 .25997 L .44068 .26974 L p .801 g
F P
0 g
s
.51527 .28492 m .51731 .25714 L .47769 .25997 L p .769 g
F P
0 g
s
.25176 .43741 m .21878 .49265 L .23057 .54144 L p .885 g
F P
0 g
s
.21774 .60368 m .23057 .54144 L .21878 .49265 L p .839 g
F P
0 g
s
.64553 .31138 m .70253 .32691 L .6708 .29589 L p .514 g
F P
0 g
s
.6708 .29589 m .62308 .28965 L .64553 .31138 L p .514 g
F P
0 g
s
.63736 .82103 m .61617 .84709 L .66276 .82762 L p .031 g
F P
0 g
s
.66276 .82762 m .69375 .79734 L .63736 .82103 L p .031 g
F P
0 g
s
.7739 .44671 m .8067 .50297 L .80268 .45312 L p .144 g
F P
0 g
s
.80268 .45312 m .76993 .40182 L .7739 .44671 L p .144 g
F P
0 g
s
.32931 .32005 m .38472 .30656 L .40895 .28568 L p .893 g
F P
0 g
s
.38472 .30656 m .32931 .32005 L .30821 .35457 L p .899 g
F P
0 g
s
.39985 .84365 m .37652 .81685 L .32214 .79135 L p .262 g
F P
0 g
s
.37652 .81685 m .39985 .84365 L .45171 .85706 L p .263 g
F P
0 g
s
.7207 .36208 m .76993 .40182 L .74797 .35867 L p .296 g
F P
0 g
s
.74797 .35867 m .70253 .32691 L .7207 .36208 L p .296 g
F P
0 g
s
.30821 .35457 m .25948 .39261 L .25176 .43741 L p .922 g
F P
0 g
s
.21878 .49265 m .25176 .43741 L .25948 .39261 L p .911 g
F P
0 g
s
.25948 .39261 m .30821 .35457 L .32931 .32005 L p .926 g
F P
0 g
s
.76205 .71651 m .73908 .76549 L .77017 .72724 L p F P
s
.77017 .72724 m .79662 .67372 L .76205 .71651 L p F P
s
.62308 .28965 m .6708 .29589 L .62751 .27139 L p .507 g
F P
0 g
s
.62751 .27139 m .59274 .27258 L .62308 .28965 L p .507 g
F P
0 g
s
.80348 .61365 m .79662 .67372 L .81282 .62621 L p .013 g
F P
0 g
s
.81282 .62621 m .82018 .56414 L .80348 .61365 L p .013 g
F P
0 g
s
.36364 .29019 m .40895 .28568 L .44068 .26974 L p .839 g
F P
0 g
s
.40895 .28568 m .36364 .29019 L .32931 .32005 L p .893 g
F P
0 g
s
.21774 .60368 m .22474 .66415 L .25571 .70815 L p .713 g
F P
0 g
s
.27952 .758 m .25571 .70815 L .22474 .66415 L p .446 g
F P
0 g
s
.25571 .70815 m .27952 .758 L .32214 .79135 L p .533 g
F P
0 g
s
.69375 .79734 m .66276 .82762 L .7 .80151 L p F P
s
.7 .80151 m .73908 .76549 L .69375 .79734 L p F P
s
.21878 .49265 m .20523 .55352 L .21774 .60368 L p .839 g
F P
0 g
s
.22474 .66415 m .21774 .60368 L .20523 .55352 L p .6 g
F P
0 g
s
.35497 .8227 m .32214 .79135 L .27952 .758 L p .179 g
F P
0 g
s
.32214 .79135 m .35497 .8227 L .39985 .84365 L p .262 g
F P
0 g
s
.8067 .50297 m .82018 .56414 L .81616 .51285 L p .032 g
F P
0 g
s
.81616 .51285 m .80268 .45312 L .8067 .50297 L p .032 g
F P
0 g
s
.59274 .27258 m .62751 .27139 L .57568 .25537 L p .515 g
F P
0 g
s
.57568 .25537 m .55662 .26145 L .59274 .27258 L p .515 g
F P
0 g
s
.40898 .26731 m .44068 .26974 L .47769 .25997 L p .741 g
F P
0 g
s
.44068 .26974 m .40898 .26731 L .36364 .29019 L p .839 g
F P
0 g
s
.50895 .865 m .50731 .86221 L .45171 .85706 L p .044 g
F P
0 g
s
.50731 .86221 m .50895 .865 L .5492 .86257 L p .013 g
F P
0 g
s
.5492 .86257 m .56326 .85883 L .50731 .86221 L p .013 g
F P
0 g
s
.70253 .32691 m .74797 .35867 L .70901 .32044 L p .224 g
F P
0 g
s
.70901 .32044 m .6708 .29589 L .70253 .32691 L p .224 g
F P
0 g
s
.56326 .85883 m .5492 .86257 L .58719 .85414 L p F P
s
.58719 .85414 m .61617 .84709 L .56326 .85883 L p F P
s
.76993 .40182 m .80268 .45312 L .77831 .40506 L p .054 g
F P
0 g
s
.77831 .40506 m .74797 .35867 L .76993 .40182 L p .054 g
F P
0 g
s
.25948 .39261 m .22703 .44288 L .21878 .49265 L p .911 g
F P
0 g
s
.20523 .55352 m .21878 .49265 L .22703 .44288 L p .681 g
F P
0 g
s
.45171 .85706 m .46896 .8613 L .50895 .865 L p .044 g
F P
0 g
s
.46896 .8613 m .45171 .85706 L .39985 .84365 L p .326 g
F P
0 g
s
.55662 .26145 m .57568 .25537 L .51913 .24915 L p .555 g
F P
0 g
s
.51913 .24915 m .51731 .25714 L .55662 .26145 L p .555 g
F P
0 g
s
.46212 .25325 m .47769 .25997 L .51731 .25714 L p .634 g
F P
0 g
s
.47769 .25997 m .46212 .25325 L .40898 .26731 L p .741 g
F P
0 g
s
.28509 .35022 m .32931 .32005 L .36364 .29019 L p .791 g
F P
0 g
s
.32931 .32005 m .28509 .35022 L .25948 .39261 L p .926 g
F P
0 g
s
.51731 .25714 m .51913 .24915 L .46212 .25325 L p .634 g
F P
0 g
s
.22703 .44288 m .25948 .39261 L .28509 .35022 L p .723 g
F P
0 g
s
.61617 .84709 m .58719 .85414 L .6205 .8402 L p .693 g
F P
0 g
s
.6205 .8402 m .66276 .82762 L .61617 .84709 L p .693 g
F P
0 g
s
.73908 .76549 m .7 .80151 L .7253 .77028 L p .815 g
F P
0 g
s
.7253 .77028 m .77017 .72724 L .73908 .76549 L p .815 g
F P
0 g
s
.79662 .67372 m .77017 .72724 L .78451 .68489 L p .791 g
F P
0 g
s
.78451 .68489 m .81282 .62621 L .79662 .67372 L p .791 g
F P
0 g
s
.43174 .85167 m .39985 .84365 L .35497 .8227 L p .346 g
F P
0 g
s
.39985 .84365 m .43174 .85167 L .46896 .8613 L p .326 g
F P
0 g
s
.22474 .66415 m .25173 .7187 L .27952 .758 L p .446 g
F P
0 g
s
.27952 .758 m .32 .79539 L .35497 .8227 L p .179 g
F P
0 g
s
.32 .79539 m .27952 .758 L .25173 .7187 L p .16 g
F P
0 g
s
.20523 .55352 m .21262 .61603 L .22474 .66415 L p .6 g
F P
0 g
s
.25173 .7187 m .22474 .66415 L .21262 .61603 L p .055 g
F P
0 g
s
.6708 .29589 m .70901 .32044 L .65545 .29015 L p .127 g
F P
0 g
s
.65545 .29015 m .62751 .27139 L .6708 .29589 L p .127 g
F P
0 g
s
.82018 .56414 m .81282 .62621 L .8088 .57704 L p .741 g
F P
0 g
s
.8088 .57704 m .81616 .51285 L .82018 .56414 L p .741 g
F P
0 g
s
.36364 .29019 m .32733 .31341 L .28509 .35022 L p .791 g
F P
0 g
s
.32733 .31341 m .36364 .29019 L .40898 .26731 L p F P
s
.66276 .82762 m .6205 .8402 L .64693 .82158 L p .845 g
F P
0 g
s
.64693 .82158 m .7 .80151 L .66276 .82762 L p .845 g
F P
0 g
s
.22703 .44288 m .21367 .5023 L .20523 .55352 L p .681 g
F P
0 g
s
.21262 .61603 m .20523 .55352 L .21367 .5023 L p .036 g
F P
0 g
s
.3997 .83668 m .35497 .8227 L .32 .79539 L p .384 g
F P
0 g
s
.35497 .8227 m .3997 .83668 L .43174 .85167 L p .346 g
F P
0 g
s
.80268 .45312 m .81616 .51285 L .79082 .46333 L p .613 g
F P
0 g
s
.79082 .46333 m .77831 .40506 L .80268 .45312 L p .613 g
F P
0 g
s
.74797 .35867 m .77831 .40506 L .7346 .36237 L p .397 g
F P
0 g
s
.7346 .36237 m .70901 .32044 L .74797 .35867 L p .397 g
F P
0 g
s
.62751 .27139 m .65545 .29015 L .59104 .2703 L p .145 g
F P
0 g
s
.59104 .2703 m .57568 .25537 L .62751 .27139 L p .145 g
F P
0 g
s
.28509 .35022 m .25555 .39564 L .22703 .44288 L p .723 g
F P
0 g
s
.21367 .5023 m .22703 .44288 L .25555 .39564 L p .017 g
F P
0 g
s
.25555 .39564 m .28509 .35022 L .32733 .31341 L p F P
s
.38349 .2851 m .40898 .26731 L .46212 .25325 L p .031 g
F P
0 g
s
.40898 .26731 m .38349 .2851 L .32733 .31341 L p F P
s
.77017 .72724 m .7253 .77028 L .73665 .73584 L p .934 g
F P
0 g
s
.73665 .73584 m .78451 .68489 L .77017 .72724 L p .934 g
F P
0 g
s
.51094 .8465 m .50895 .865 L .46896 .8613 L p .696 g
F P
0 g
s
.50895 .865 m .51094 .8465 L .53214 .84521 L p .74 g
F P
0 g
s
.53214 .84521 m .5492 .86257 L .50895 .865 L p .74 g
F P
0 g
s
.7 .80151 m .64693 .82158 L .66464 .79941 L p .913 g
F P
0 g
s
.66464 .79941 m .7253 .77028 L .7 .80151 L p .913 g
F P
0 g
s
.5492 .86257 m .53214 .84521 L .55209 .84075 L p .787 g
F P
0 g
s
.55209 .84075 m .58719 .85414 L .5492 .86257 L p .787 g
F P
0 g
s
.48989 .84454 m .46896 .8613 L .43174 .85167 L p .663 g
F P
0 g
s
.46896 .8613 m .48989 .84454 L .51094 .8465 L p .696 g
F P
0 g
s
.57568 .25537 m .59104 .2703 L .5206 .26258 L p .13 g
F P
0 g
s
.5206 .26258 m .51913 .24915 L .57568 .25537 L p .13 g
F P
0 g
s
.81282 .62621 m .78451 .68489 L .78051 .64116 L p .899 g
F P
0 g
s
.78051 .64116 m .8088 .57704 L .81282 .62621 L p .899 g
F P
0 g
s
.21262 .61603 m .2411 .67583 L .25173 .7187 L p .055 g
F P
0 g
s
.29746 .76332 m .25173 .7187 L .2411 .67583 L p .271 g
F P
0 g
s
.25173 .7187 m .29746 .76332 L .32 .79539 L p .16 g
F P
0 g
s
.44958 .26766 m .46212 .25325 L .51913 .24915 L p .092 g
F P
0 g
s
.46212 .25325 m .44958 .26766 L .38349 .2851 L p .031 g
F P
0 g
s
.37494 .81722 m .32 .79539 L .29746 .76332 L p .447 g
F P
0 g
s
.32 .79539 m .37494 .81722 L .3997 .83668 L p .384 g
F P
0 g
s
.51913 .24915 m .5206 .26258 L .44958 .26766 L p .092 g
F P
0 g
s
.58719 .85414 m .55209 .84075 L .5695 .83341 L p .827 g
F P
0 g
s
.5695 .83341 m .6205 .8402 L .58719 .85414 L p .827 g
F P
0 g
s
.47036 .83945 m .43174 .85167 L .3997 .83668 L p .641 g
F P
0 g
s
.43174 .85167 m .47036 .83945 L .48989 .84454 L p .663 g
F P
0 g
s
.21367 .5023 m .22097 .56693 L .21262 .61603 L p .036 g
F P
0 g
s
.2411 .67583 m .21262 .61603 L .22097 .56693 L p .166 g
F P
0 g
s
.70901 .32044 m .7346 .36237 L .67421 .32847 L p .442 g
F P
0 g
s
.67421 .32847 m .65545 .29015 L .70901 .32044 L p .442 g
F P
0 g
s
.81616 .51285 m .8088 .57704 L .78399 .52959 L p .803 g
F P
0 g
s
.78399 .52959 m .79082 .46333 L .81616 .51285 L p .803 g
F P
0 g
s
.45025 1.00576 m .43612 1.08601 L .50583 1.09201 L p .653 g
F P
0 g
s
.50583 1.09201 m .50717 1.01079 L .45025 1.00576 L p .653 g
F P
0 g
s
.50495 1.17257 m .50583 1.09201 L .43612 1.08601 L p .624 g
F P
0 g
s
.50717 1.01079 m .50583 1.09201 L .57596 1.08807 L p .62 g
F P
0 g
s
.50583 1.09201 m .50495 1.17257 L .58352 1.16826 L p .585 g
F P
0 g
s
.58352 1.16826 m .57596 1.08807 L .50583 1.09201 L p .585 g
F P
0 g
s
.46847 .92967 m .45025 1.00576 L .50717 1.01079 L p .68 g
F P
0 g
s
.50717 1.01079 m .5089 .93333 L .46847 .92967 L p .68 g
F P
0 g
s
.5089 .93333 m .50717 1.01079 L .56445 1.00749 L p .653 g
F P
0 g
s
.57596 1.08807 m .56445 1.00749 L .50717 1.01079 L p .62 g
F P
0 g
s
.56445 1.00749 m .57596 1.08807 L .64241 1.07436 L p .588 g
F P
0 g
s
.57596 1.08807 m .58352 1.16826 L .65803 1.1533 L p .545 g
F P
0 g
s
.65803 1.1533 m .64241 1.07436 L .57596 1.08807 L p .545 g
F P
0 g
s
.56445 1.00749 m .5496 .93093 L .5089 .93333 L p .653 g
F P
0 g
s
.5496 .93093 m .56445 1.00749 L .61862 .99601 L p .628 g
F P
0 g
s
.64241 1.07436 m .61862 .99601 L .56445 1.00749 L p .588 g
F P
0 g
s
.303 .3545 m .32733 .31341 L .38349 .2851 L p .079 g
F P
0 g
s
.32733 .31341 m .303 .3545 L .25555 .39564 L p F P
s
.39715 .99265 m .37095 1.07035 L .43612 1.08601 L p .689 g
F P
0 g
s
.43612 1.08601 m .45025 1.00576 L .39715 .99265 L p .689 g
F P
0 g
s
.42684 1.16602 m .43612 1.08601 L .37095 1.07035 L p .665 g
F P
0 g
s
.43612 1.08601 m .42684 1.16602 L .50495 1.17257 L p .624 g
F P
0 g
s
.43085 .92014 m .39715 .99265 L .45025 1.00576 L p .71 g
F P
0 g
s
.45025 1.00576 m .46847 .92967 L .43085 .92014 L p .71 g
F P
0 g
s
.77831 .40506 m .79082 .46333 L .74518 .4193 L p .658 g
F P
0 g
s
.74518 .4193 m .7346 .36237 L .77831 .40506 L p .658 g
F P
0 g
s
.6205 .8402 m .5695 .83341 L .58318 .82364 L p .851 g
F P
0 g
s
.58318 .82364 m .64693 .82158 L .6205 .8402 L p .851 g
F P
0 g
s
.25555 .39564 m .24336 .45361 L .21367 .5023 L p .017 g
F P
0 g
s
.22097 .56693 m .21367 .5023 L .24336 .45361 L p .119 g
F P
0 g
s
.45364 .83156 m .3997 .83668 L .37494 .81722 L p .629 g
F P
0 g
s
.3997 .83668 m .45364 .83156 L .47036 .83945 L p .641 g
F P
0 g
s
.50458 1.24799 m .50495 1.17257 L .42684 1.16602 L p .587 g
F P
0 g
s
.50495 1.17257 m .50458 1.24799 L .58664 1.24361 L p .543 g
F P
0 g
s
.58664 1.24361 m .58352 1.16826 L .50495 1.17257 L p .543 g
F P
0 g
s
.24336 .45361 m .25555 .39564 L .303 .3545 L p .097 g
F P
0 g
s
.7253 .77028 m .66464 .79941 L .67228 .7751 L p .909 g
F P
0 g
s
.67228 .7751 m .73665 .73584 L .7253 .77028 L p .909 g
F P
0 g
s
.61862 .99601 m .588 .92258 L .5496 .93093 L p .628 g
F P
0 g
s
.588 .92258 m .61862 .99601 L .66634 .97697 L p .605 g
F P
0 g
s
.61862 .99601 m .64241 1.07436 L .70115 1.05159 L p .555 g
F P
0 g
s
.70115 1.05159 m .66634 .97697 L .61862 .99601 L p .555 g
F P
0 g
s
.48982 .86203 m .46847 .92967 L .5089 .93333 L p .706 g
F P
0 g
s
.5089 .93333 m .51093 .86399 L .48982 .86203 L p .706 g
F P
0 g
s
.51093 .86399 m .5089 .93333 L .5496 .93093 L p .687 g
F P
0 g
s
.58352 1.16826 m .58664 1.24361 L .66449 1.22838 L p .495 g
F P
0 g
s
.66449 1.22838 m .65803 1.1533 L .58352 1.16826 L p .495 g
F P
0 g
s
.64241 1.07436 m .65803 1.1533 L .72406 1.12841 L p .501 g
F P
0 g
s
.72406 1.12841 m .70115 1.05159 L .64241 1.07436 L p .501 g
F P
0 g
s
.78451 .68489 m .73665 .73584 L .73287 .70043 L p .906 g
F P
0 g
s
.73287 .70043 m .78051 .64116 L .78451 .68489 L p .906 g
F P
0 g
s
.5496 .93093 m .53219 .8627 L .51093 .86399 L p .687 g
F P
0 g
s
.53219 .8627 m .5496 .93093 L .588 .92258 L p .67 g
F P
0 g
s
.39845 .9053 m .35117 .97217 L .39715 .99265 L p .743 g
F P
0 g
s
.39715 .99265 m .43085 .92014 L .39845 .9053 L p .743 g
F P
0 g
s
.37095 1.07035 m .39715 .99265 L .35117 .97217 L p .73 g
F P
0 g
s
.37095 1.07035 m .35373 1.14892 L .42684 1.16602 L p .665 g
F P
0 g
s
.42301 1.24132 m .42684 1.16602 L .35373 1.14892 L p .632 g
F P
0 g
s
.42684 1.16602 m .42301 1.24132 L .50458 1.24799 L p .587 g
F P
0 g
s
.47024 .85696 m .43085 .92014 L .46847 .92967 L p .727 g
F P
0 g
s
.46847 .92967 m .48982 .86203 L .47024 .85696 L p .727 g
F P
0 g
s
.2411 .67583 m .28918 .72849 L .29746 .76332 L p .271 g
F P
0 g
s
.35923 .79449 m .29746 .76332 L .28918 .72849 L p .493 g
F P
0 g
s
.29746 .76332 m .35923 .79449 L .37494 .81722 L p .447 g
F P
0 g
s
.35117 .97217 m .31429 1.04582 L .37095 1.07035 L p .73 g
F P
0 g
s
.35373 1.14892 m .37095 1.07035 L .31429 1.04582 L p .71 g
F P
0 g
s
.65545 .29015 m .67421 .32847 L .60136 .30621 L p .424 g
F P
0 g
s
.60136 .30621 m .59104 .2703 L .65545 .29015 L p .424 g
F P
0 g
s
.64693 .82158 m .58318 .82364 L .5922 .81208 L p .856 g
F P
0 g
s
.5922 .81208 m .66464 .79941 L .64693 .82158 L p .856 g
F P
0 g
s
.22097 .56693 m .249 .63218 L .2411 .67583 L p .166 g
F P
0 g
s
.28918 .72849 m .2411 .67583 L .249 .63218 L p .388 g
F P
0 g
s
.588 .92258 m .5522 .85826 L .53219 .8627 L p .67 g
F P
0 g
s
.5522 .85826 m .588 .92258 L .62168 .90878 L p .656 g
F P
0 g
s
.66634 .97697 m .62168 .90878 L .588 .92258 L p .605 g
F P
0 g
s
.36637 .32281 m .38349 .2851 L .44958 .26766 L p .18 g
F P
0 g
s
.38349 .2851 m .36637 .32281 L .303 .3545 L p .079 g
F P
0 g
s
.44085 .82136 m .37494 .81722 L .35923 .79449 L p .623 g
F P
0 g
s
.37494 .81722 m .44085 .82136 L .45364 .83156 L p .629 g
F P
0 g
s
.45347 .84909 m .39845 .9053 L .43085 .92014 L p .75 g
F P
0 g
s
.43085 .92014 m .47024 .85696 L .45347 .84909 L p .75 g
F P
0 g
s
.8088 .57704 m .78051 .64116 L .75776 .59908 L p .845 g
F P
0 g
s
.75776 .59908 m .78399 .52959 L .8088 .57704 L p .845 g
F P
0 g
s
.65803 1.1533 m .66449 1.22838 L .73355 1.20302 L p .437 g
F P
0 g
s
.73355 1.20302 m .72406 1.12841 L .65803 1.1533 L p .437 g
F P
0 g
s
.62168 .90878 m .66634 .97697 L .70451 .95144 L p .582 g
F P
0 g
s
.66634 .97697 m .70115 1.05159 L .74843 1.02093 L p .519 g
F P
0 g
s
.74843 1.02093 m .70451 .95144 L .66634 .97697 L p .519 g
F P
0 g
s
.51316 .80695 m .51094 .8465 L .48989 .84454 L closepath p .748 g
F P
0 g
s
.51316 .80695 m .53214 .84521 L .51094 .8465 L closepath p .759 g
F P
0 g
s
.31429 1.04582 m .29 1.1221 L .35373 1.14892 L p .71 g
F P
0 g
s
.34662 1.22391 m .35373 1.14892 L .29 1.1221 L p .682 g
F P
0 g
s
.35373 1.14892 m .34662 1.22391 L .42301 1.24132 L p .632 g
F P
0 g
s
.51316 .80695 m .55209 .84075 L .53214 .84521 L closepath p .768 g
F P
0 g
s
.51316 .80695 m .48989 .84454 L .47036 .83945 L closepath p .738 g
F P
0 g
s
.37341 .88604 m .31533 .94545 L .35117 .97217 L p .78 g
F P
0 g
s
.35117 .97217 m .39845 .9053 L .37341 .88604 L p .78 g
F P
0 g
s
.31429 1.04582 m .35117 .97217 L .31533 .94545 L p .777 g
F P
0 g
s
.59104 .2703 m .60136 .30621 L .52159 .29754 L p .372 g
F P
0 g
s
.52159 .29754 m .5206 .26258 L .59104 .2703 L p .372 g
F P
0 g
s
.24336 .45361 m .25002 .52029 L .22097 .56693 L p .119 g
F P
0 g
s
.249 .63218 m .22097 .56693 L .25002 .52029 L p .329 g
F P
0 g
s
.50477 1.31386 m .50458 1.24799 L .42301 1.24132 L p .538 g
F P
0 g
s
.50458 1.24799 m .50477 1.31386 L .58503 1.30967 L p .487 g
F P
0 g
s
.58503 1.30967 m .58664 1.24361 L .50458 1.24799 L p .487 g
F P
0 g
s
.62168 .90878 m .56965 .85093 L .5522 .85826 L p .656 g
F P
0 g
s
.56965 .85093 m .62168 .90878 L .6484 .89035 L p .643 g
F P
0 g
s
.70451 .95144 m .6484 .89035 L .62168 .90878 L p .582 g
F P
0 g
s
.70115 1.05159 m .72406 1.12841 L .77742 1.09483 L p .448 g
F P
0 g
s
.77742 1.09483 m .74843 1.02093 L .70115 1.05159 L p .448 g
F P
0 g
s
.7346 .36237 m .74518 .4193 L .68197 .38429 L p .618 g
F P
0 g
s
.68197 .38429 m .67421 .32847 L .7346 .36237 L p .618 g
F P
0 g
s
.44114 .30324 m .44958 .26766 L .5206 .26258 L p .295 g
F P
0 g
s
.44958 .26766 m .44114 .30324 L .36637 .32281 L p .18 g
F P
0 g
s
.79082 .46333 m .78399 .52959 L .7394 .48743 L p .747 g
F P
0 g
s
.7394 .48743 m .74518 .4193 L .79082 .46333 L p .747 g
F P
0 g
s
.51316 .80695 m .5695 .83341 L .55209 .84075 L closepath p .774 g
F P
0 g
s
.44065 .83892 m .37341 .88604 L .39845 .9053 L p .774 g
F P
0 g
s
.39845 .9053 m .45347 .84909 L .44065 .83892 L p .774 g
F P
0 g
s
.51316 .80695 m .47036 .83945 L .45364 .83156 L closepath p .728 g
F P
0 g
s
.66464 .79941 m .5922 .81208 L .5959 .79949 L p .845 g
F P
0 g
s
.5959 .79949 m .67228 .7751 L .66464 .79941 L p .845 g
F P
0 g
s
.58664 1.24361 m .58503 1.30967 L .66116 1.29512 L p .428 g
F P
0 g
s
.66116 1.29512 m .66449 1.22838 L .58664 1.24361 L p .428 g
F P
0 g
s
.73665 .73584 m .67228 .7751 L .66912 .75025 L p .871 g
F P
0 g
s
.66912 .75025 m .73287 .70043 L .73665 .73584 L p .871 g
F P
0 g
s
.51093 .86399 m .51316 .80695 L .48982 .86203 L closepath p .731 g
F P
0 g
s
.53219 .8627 m .51316 .80695 L .51093 .86399 L closepath p .724 g
F P
0 g
s
.5206 .26258 m .52159 .29754 L .44114 .30324 L p .295 g
F P
0 g
s
.31533 .94545 m .26983 1.01373 L .31429 1.04582 L p .777 g
F P
0 g
s
.29 1.1221 m .31429 1.04582 L .26983 1.01373 L p .764 g
F P
0 g
s
.5522 .85826 m .51316 .80695 L .53219 .8627 L closepath p .717 g
F P
0 g
s
.42499 1.30749 m .42301 1.24132 L .34662 1.22391 L p .586 g
F P
0 g
s
.42301 1.24132 m .42499 1.30749 L .50477 1.31386 L p .538 g
F P
0 g
s
.48982 .86203 m .51316 .80695 L .47024 .85696 L closepath p .74 g
F P
0 g
s
.29294 .41118 m .303 .3545 L .36637 .32281 L p .263 g
F P
0 g
s
.303 .3545 m .29294 .41118 L .24336 .45361 L p .097 g
F P
0 g
s
.28918 .72849 m .35379 .76993 L .35923 .79449 L p .493 g
F P
0 g
s
.4329 .80952 m .35923 .79449 L .35379 .76993 L p .622 g
F P
0 g
s
.35923 .79449 m .4329 .80952 L .44085 .82136 L p .623 g
F P
0 g
s
.51316 .80695 m .58318 .82364 L .5695 .83341 L closepath p .778 g
F P
0 g
s
.25002 .52029 m .24336 .45361 L .29294 .41118 L p .297 g
F P
0 g
s
.51316 .80695 m .45364 .83156 L .44085 .82136 L closepath p .719 g
F P
0 g
s
.56965 .85093 m .51316 .80695 L .5522 .85826 L closepath p .711 g
F P
0 g
s
.47024 .85696 m .51316 .80695 L .45347 .84909 L closepath p .749 g
F P
0 g
s
.249 .63218 m .29617 .69318 L .28918 .72849 L p .388 g
F P
0 g
s
.35379 .76993 m .28918 .72849 L .29617 .69318 L p .527 g
F P
0 g
s
.78051 .64116 m .73287 .70043 L .71375 .66651 L p .844 g
F P
0 g
s
.71375 .66651 m .75776 .59908 L .78051 .64116 L p .844 g
F P
0 g
s
.6484 .89035 m .58338 .84119 L .56965 .85093 L p .643 g
F P
0 g
s
.58338 .84119 m .6484 .89035 L .66632 .8684 L p .633 g
F P
0 g
s
.6484 .89035 m .70451 .95144 L .73047 .92088 L p .559 g
F P
0 g
s
.73047 .92088 m .66632 .8684 L .6484 .89035 L p .559 g
F P
0 g
s
.58338 .84119 m .51316 .80695 L .56965 .85093 L closepath p .708 g
F P
0 g
s
.51316 .80695 m .5922 .81208 L .58318 .82364 L closepath p .779 g
F P
0 g
s
.72406 1.12841 m .73355 1.20302 L .78944 1.16878 L p .36 g
F P
0 g
s
.78944 1.16878 m .77742 1.09483 L .72406 1.12841 L p .36 g
F P
0 g
s
.70451 .95144 m .74843 1.02093 L .78093 .9841 L p .475 g
F P
0 g
s
.78093 .9841 m .73047 .92088 L .70451 .95144 L p .475 g
F P
0 g
s
.45347 .84909 m .51316 .80695 L .44065 .83892 L closepath p .758 g
F P
0 g
s
.51316 .80695 m .44085 .82136 L .4329 .80952 L closepath p .713 g
F P
0 g
s
.66449 1.22838 m .66116 1.29512 L .72866 1.27089 L p .351 g
F P
0 g
s
.72866 1.27089 m .73355 1.20302 L .66449 1.22838 L p .351 g
F P
0 g
s
.43267 .82711 m .35752 .86352 L .37341 .88604 L p .8 g
F P
0 g
s
.37341 .88604 m .44065 .83892 L .43267 .82711 L p .8 g
F P
0 g
s
.31533 .94545 m .37341 .88604 L .35752 .86352 L p .821 g
F P
0 g
s
.35752 .86352 m .29219 .91407 L .31533 .94545 L p .821 g
F P
0 g
s
.26983 1.01373 m .31533 .94545 L .29219 .91407 L p .832 g
F P
0 g
s
.26983 1.01373 m .23976 1.08693 L .29 1.1221 L p .764 g
F P
0 g
s
.27994 1.19659 m .29 1.1221 L .23976 1.08693 L p .738 g
F P
0 g
s
.29 1.1221 m .27994 1.19659 L .34662 1.22391 L p .682 g
F P
0 g
s
.25002 .52029 m .27556 .59084 L .249 .63218 L p .329 g
F P
0 g
s
.29617 .69318 m .249 .63218 L .27556 .59084 L p .46 g
F P
0 g
s
.35028 1.29085 m .34662 1.22391 L .27994 1.19659 L p .634 g
F P
0 g
s
.34662 1.22391 m .35028 1.29085 L .42499 1.30749 L p .586 g
F P
0 g
s
.67228 .7751 m .5959 .79949 L .59397 .78671 L p .826 g
F P
0 g
s
.59397 .78671 m .66912 .75025 L .67228 .7751 L p .826 g
F P
0 g
s
.59242 .82966 m .51316 .80695 L .58338 .84119 L closepath p .706 g
F P
0 g
s
.66632 .8684 m .59242 .82966 L .58338 .84119 L p .633 g
F P
0 g
s
.67421 .32847 m .68197 .38429 L .60564 .36129 L p .566 g
F P
0 g
s
.60564 .36129 m .60136 .30621 L .67421 .32847 L p .566 g
F P
0 g
s
.51316 .80695 m .5959 .79949 L .5922 .81208 L closepath p .777 g
F P
0 g
s
.44065 .83892 m .51316 .80695 L .43267 .82711 L closepath p .766 g
F P
0 g
s
.74843 1.02093 m .77742 1.09483 L .81437 1.05438 L p .374 g
F P
0 g
s
.81437 1.05438 m .78093 .9841 L .74843 1.02093 L p .374 g
F P
0 g
s
.35379 .76993 m .43035 .79682 L .4329 .80952 L p .622 g
F P
0 g
s
.51316 .80695 m .4329 .80952 L .43035 .79682 L closepath p .708 g
F P
0 g
s
.29617 .69318 m .35922 .74519 L .35379 .76993 L p .527 g
F P
0 g
s
.43035 .79682 m .35379 .76993 L .35922 .74519 L p .626 g
F P
0 g
s
.78399 .52959 m .75776 .59908 L .71726 .56178 L p .776 g
F P
0 g
s
.71726 .56178 m .7394 .48743 L .78399 .52959 L p .776 g
F P
0 g
s
.35929 .37845 m .36637 .32281 L .44114 .30324 L p .372 g
F P
0 g
s
.36637 .32281 m .35929 .37845 L .29294 .41118 L p .263 g
F P
0 g
s
.73287 .70043 m .66912 .75025 L .65514 .72659 L p .827 g
F P
0 g
s
.65514 .72659 m .71375 .66651 L .73287 .70043 L p .827 g
F P
0 g
s
.59613 .81709 m .51316 .80695 L .59242 .82966 L closepath p .706 g
F P
0 g
s
.59242 .82966 m .66632 .8684 L .67406 .84432 L p .627 g
F P
0 g
s
.67406 .84432 m .59613 .81709 L .59242 .82966 L p .627 g
F P
0 g
s
.29219 .91407 m .24075 .97587 L .26983 1.01373 L p .832 g
F P
0 g
s
.23976 1.08693 m .26983 1.01373 L .24075 .97587 L p .829 g
F P
0 g
s
.74518 .4193 m .7394 .48743 L .67773 .45393 L p .685 g
F P
0 g
s
.67773 .45393 m .68197 .38429 L .74518 .4193 L p .685 g
F P
0 g
s
.66632 .8684 m .73047 .92088 L .74216 .88717 L p .534 g
F P
0 g
s
.74216 .88717 m .67406 .84432 L .66632 .8684 L p .534 g
F P
0 g
s
.50552 1.36593 m .50477 1.31386 L .42499 1.30749 L p .459 g
F P
0 g
s
.50477 1.31386 m .50552 1.36593 L .57861 1.36218 L p .401 g
F P
0 g
s
.57861 1.36218 m .58503 1.30967 L .50477 1.31386 L p .401 g
F P
0 g
s
.29294 .41118 m .29843 .47966 L .25002 .52029 L p .297 g
F P
0 g
s
.27556 .59084 m .25002 .52029 L .29843 .47966 L p .423 g
F P
0 g
s
.43267 .82711 m .51316 .80695 L .43012 .81443 L closepath p .772 g
F P
0 g
s
.35752 .86352 m .43267 .82711 L .43012 .81443 L p .824 g
F P
0 g
s
.51316 .80695 m .59397 .78671 L .5959 .79949 L closepath p .772 g
F P
0 g
s
.43012 .81443 m .352 .8392 L .35752 .86352 L p .824 g
F P
0 g
s
.29219 .91407 m .35752 .86352 L .352 .8392 L p .864 g
F P
0 g
s
.35922 .74519 m .43345 .78411 L .43035 .79682 L p .626 g
F P
0 g
s
.51316 .80695 m .43035 .79682 L .43345 .78411 L closepath p .706 g
F P
0 g
s
.58503 1.30967 m .57861 1.36218 L .64789 1.34918 L p .326 g
F P
0 g
s
.64789 1.34918 m .66116 1.29512 L .58503 1.30967 L p .326 g
F P
0 g
s
.27556 .59084 m .3184 .65988 L .29617 .69318 L p .46 g
F P
0 g
s
.35922 .74519 m .29617 .69318 L .3184 .65988 L p .555 g
F P
0 g
s
.29843 .47966 m .29294 .41118 L .35929 .37845 L p .401 g
F P
0 g
s
.43286 1.36023 m .42499 1.30749 L .35028 1.29085 L p .507 g
F P
0 g
s
.42499 1.30749 m .43286 1.36023 L .50552 1.36593 L p .459 g
F P
0 g
s
.5942 .80435 m .51316 .80695 L .59613 .81709 L closepath p .708 g
F P
0 g
s
.58669 .79229 m .51316 .80695 L .5942 .80435 L closepath p .713 g
F P
0 g
s
.51316 .80695 m .43345 .78411 L .44205 .77227 L closepath p .706 g
F P
0 g
s
.43012 .81443 m .51316 .80695 L .43323 .80176 L closepath p .777 g
F P
0 g
s
.51316 .80695 m .58648 .77462 L .59397 .78671 L closepath p .766 g
F P
0 g
s
.57406 .78176 m .51316 .80695 L .58669 .79229 L closepath p .719 g
F P
0 g
s
.51316 .80695 m .44205 .77227 L .45561 .76214 L closepath p .708 g
F P
0 g
s
.43323 .80176 m .51316 .80695 L .44185 .78995 L closepath p .779 g
F P
0 g
s
.51316 .80695 m .57389 .76407 L .58648 .77462 L closepath p .758 g
F P
0 g
s
.51316 .80695 m .55704 .75581 L .57389 .76407 L closepath p .749 g
F P
0 g
s
.55717 .77352 m .51316 .80695 L .57406 .78176 L closepath p .728 g
F P
0 g
s
.51316 .80695 m .45561 .76214 L .4732 .75444 L closepath p .711 g
F P
0 g
s
.44185 .78995 m .51316 .80695 L .45544 .77984 L closepath p .778 g
F P
0 g
s
.51316 .80695 m .53709 .75045 L .55704 .75581 L closepath p .74 g
F P
0 g
s
.53716 .76817 m .51316 .80695 L .55717 .77352 L closepath p .738 g
F P
0 g
s
.51316 .80695 m .4732 .75444 L .49363 .74974 L closepath p .717 g
F P
0 g
s
.45544 .77984 m .51316 .80695 L .47309 .77216 L closepath p .774 g
F P
0 g
s
.51316 .80695 m .51544 .74837 L .53709 .75045 L closepath p .731 g
F P
0 g
s
.51545 .7661 m .51316 .80695 L .53716 .76817 L closepath p .748 g
F P
0 g
s
.51316 .80695 m .49363 .74974 L .51544 .74837 L closepath p .724 g
F P
0 g
s
.47309 .77216 m .51316 .80695 L .49357 .76746 L closepath p .768 g
F P
0 g
s
.49357 .76746 m .51316 .80695 L .51545 .7661 L closepath p .759 g
F P
0 g
s
.60136 .30621 m .60564 .36129 L .522 .35234 L p .511 g
F P
0 g
s
.522 .35234 m .52159 .29754 L .60136 .30621 L p .511 g
F P
0 g
s
.73355 1.20302 m .72866 1.27089 L .78325 1.23819 L p .236 g
F P
0 g
s
.78325 1.23819 m .78944 1.16878 L .73355 1.20302 L p .236 g
F P
0 g
s
.73047 .92088 m .78093 .9841 L .79602 .94327 L p .417 g
F P
0 g
s
.79602 .94327 m .74216 .88717 L .73047 .92088 L p .417 g
F P
0 g
s
.43765 .35823 m .44114 .30324 L .52159 .29754 L p .449 g
F P
0 g
s
.44114 .30324 m .43765 .35823 L .35929 .37845 L p .372 g
F P
0 g
s
.59613 .81709 m .67406 .84432 L .67088 .81971 L p .624 g
F P
0 g
s
.67088 .81971 m .5942 .80435 L .59613 .81709 L p .624 g
F P
0 g
s
.66912 .75025 m .59397 .78671 L .58648 .77462 L p .801 g
F P
0 g
s
.58648 .77462 m .65514 .72659 L .66912 .75025 L p .801 g
F P
0 g
s
.352 .8392 m .43012 .81443 L .43323 .80176 L p .843 g
F P
0 g
s
.77742 1.09483 m .78944 1.16878 L .82826 1.12749 L p .239 g
F P
0 g
s
.82826 1.12749 m .81437 1.05438 L .77742 1.09483 L p .239 g
F P
0 g
s
.52159 .29754 m .522 .35234 L .43765 .35823 L p .449 g
F P
0 g
s
.3184 .65988 m .37538 .72198 L .35922 .74519 L p .555 g
F P
0 g
s
.43345 .78411 m .35922 .74519 L .37538 .72198 L p .633 g
F P
0 g
s
.23976 1.08693 m .22729 1.16072 L .27994 1.19659 L p .738 g
F P
0 g
s
.28513 1.26475 m .27994 1.19659 L .22729 1.16072 L p .684 g
F P
0 g
s
.27994 1.19659 m .28513 1.26475 L .35028 1.29085 L p .634 g
F P
0 g
s
.75776 .59908 m .71375 .66651 L .68022 .63655 L p .785 g
F P
0 g
s
.68022 .63655 m .71726 .56178 L .75776 .59908 L p .785 g
F P
0 g
s
.37538 .72198 m .44205 .77227 L .43345 .78411 L p .633 g
F P
0 g
s
.352 .8392 m .28365 .87997 L .29219 .91407 L p .864 g
F P
0 g
s
.24075 .97587 m .29219 .91407 L .28365 .87997 L p .892 g
F P
0 g
s
.66116 1.29512 m .64789 1.34918 L .70918 1.32756 L p .217 g
F P
0 g
s
.70918 1.32756 m .72866 1.27089 L .66116 1.29512 L p .217 g
F P
0 g
s
.24075 .97587 m .20663 1.04532 L .23976 1.08693 L p .829 g
F P
0 g
s
.22729 1.16072 m .23976 1.08693 L .20663 1.04532 L p .801 g
F P
0 g
s
.67406 .84432 m .74216 .88717 L .73834 .85249 L p .505 g
F P
0 g
s
.73834 .85249 m .67088 .81971 L .67406 .84432 L p .505 g
F P
0 g
s
.5942 .80435 m .67088 .81971 L .65675 .79627 L p .625 g
F P
0 g
s
.65675 .79627 m .58669 .79229 L .5942 .80435 L p .625 g
F P
0 g
s
.29843 .47966 m .31948 .55491 L .27556 .59084 L p .423 g
F P
0 g
s
.3184 .65988 m .27556 .59084 L .31948 .55491 L p .51 g
F P
0 g
s
.65514 .72659 m .58648 .77462 L .57389 .76407 L p .776 g
F P
0 g
s
.35748 .81469 m .43323 .80176 L .44185 .78995 L p .854 g
F P
0 g
s
.43323 .80176 m .35748 .81469 L .352 .8392 L p .843 g
F P
0 g
s
.36491 1.34537 m .35028 1.29085 L .28513 1.26475 L p .546 g
F P
0 g
s
.35028 1.29085 m .36491 1.34537 L .43286 1.36023 L p .507 g
F P
0 g
s
.40132 .702 m .45561 .76214 L .44205 .77227 L p .643 g
F P
0 g
s
.44205 .77227 m .37538 .72198 L .40132 .702 L p .643 g
F P
0 g
s
.28365 .87997 m .352 .8392 L .35748 .81469 L p .902 g
F P
0 g
s
.68197 .38429 m .67773 .45393 L .6033 .43193 L p .63 g
F P
0 g
s
.6033 .43193 m .60564 .36129 L .68197 .38429 L p .63 g
F P
0 g
s
.78093 .9841 m .81437 1.05438 L .83186 1.0094 L p .256 g
F P
0 g
s
.83186 1.0094 m .79602 .94327 L .78093 .9841 L p .256 g
F P
0 g
s
.7394 .48743 m .71726 .56178 L .66149 .5322 L p .717 g
F P
0 g
s
.66149 .5322 m .67773 .45393 L .7394 .48743 L p .717 g
F P
0 g
s
.71375 .66651 m .65514 .72659 L .63111 .70581 L p .784 g
F P
0 g
s
.63111 .70581 m .68022 .63655 L .71375 .66651 L p .784 g
F P
0 g
s
.57389 .76407 m .63111 .70581 L .65514 .72659 L p .776 g
F P
0 g
s
.36316 .44834 m .35929 .37845 L .43765 .35823 L p .472 g
F P
0 g
s
.35929 .37845 m .36316 .44834 L .29843 .47966 L p .401 g
F P
0 g
s
.58669 .79229 m .65675 .79627 L .63246 .77568 L p .631 g
F P
0 g
s
.63246 .77568 m .57406 .78176 L .58669 .79229 L p .631 g
F P
0 g
s
.63111 .70581 m .57389 .76407 L .55704 .75581 L p .751 g
F P
0 g
s
.31948 .55491 m .35469 .63105 L .3184 .65988 L p .51 g
F P
0 g
s
.37538 .72198 m .3184 .65988 L .35469 .63105 L p .58 g
F P
0 g
s
.31948 .55491 m .29843 .47966 L .36316 .44834 L p .486 g
F P
0 g
s
.37381 .79171 m .44185 .78995 L .45544 .77984 L p .85 g
F P
0 g
s
.44185 .78995 m .37381 .79171 L .35748 .81469 L p .854 g
F P
0 g
s
.43535 .68674 m .4732 .75444 L .45561 .76214 L p .656 g
F P
0 g
s
.45561 .76214 m .40132 .702 L .43535 .68674 L p .656 g
F P
0 g
s
.35469 .63105 m .40132 .702 L .37538 .72198 L p .58 g
F P
0 g
s
.28365 .87997 m .22953 .93453 L .24075 .97587 L p .892 g
F P
0 g
s
.20663 1.04532 m .24075 .97587 L .22953 .93453 L p .894 g
F P
0 g
s
.74216 .88717 m .79602 .94327 L .792 .90107 L p .326 g
F P
0 g
s
.792 .90107 m .73834 .85249 L .74216 .88717 L p .326 g
F P
0 g
s
.55704 .75581 m .59858 .68946 L .63111 .70581 L p .751 g
F P
0 g
s
.59858 .68946 m .55704 .75581 L .53709 .75045 L p .728 g
F P
0 g
s
.47508 .67738 m .49363 .74974 L .4732 .75444 L p .671 g
F P
0 g
s
.4732 .75444 m .43535 .68674 L .47508 .67738 L p .671 g
F P
0 g
s
.57406 .78176 m .63246 .77568 L .59956 .75948 L p .645 g
F P
0 g
s
.59956 .75948 m .55717 .77352 L .57406 .78176 L p .645 g
F P
0 g
s
.78944 1.16878 m .78325 1.23819 L .8211 1.19878 L p .071 g
F P
0 g
s
.8211 1.19878 m .82826 1.12749 L .78944 1.16878 L p .071 g
F P
0 g
s
.40004 .77191 m .45544 .77984 L .47309 .77216 L p .828 g
F P
0 g
s
.45544 .77984 m .40004 .77191 L .37381 .79171 L p .85 g
F P
0 g
s
.67088 .81971 m .73834 .85249 L .71879 .81925 L p .468 g
F P
0 g
s
.71879 .81925 m .65675 .79627 L .67088 .81971 L p .468 g
F P
0 g
s
.72866 1.27089 m .70918 1.32756 L .75858 1.29843 L p .088 g
F P
0 g
s
.75858 1.29843 m .78325 1.23819 L .72866 1.27089 L p .088 g
F P
0 g
s
.53709 .75045 m .55982 .6788 L .59858 .68946 L p .728 g
F P
0 g
s
.55982 .6788 m .53709 .75045 L .51544 .74837 L p .707 g
F P
0 g
s
.60564 .36129 m .6033 .43193 L .52177 .42336 L p .58 g
F P
0 g
s
.52177 .42336 m .522 .35234 L .60564 .36129 L p .58 g
F P
0 g
s
.51761 .67466 m .51544 .74837 L .49363 .74974 L p .688 g
F P
0 g
s
.49363 .74974 m .47508 .67738 L .51761 .67466 L p .688 g
F P
0 g
s
.71726 .56178 m .68022 .63655 L .63439 .61288 L p .735 g
F P
0 g
s
.63439 .61288 m .66149 .5322 L .71726 .56178 L p .735 g
F P
0 g
s
.50681 1.40028 m .50552 1.36593 L .43286 1.36023 L p .306 g
F P
0 g
s
.50552 1.36593 m .50681 1.40028 L .56756 1.39721 L p .239 g
F P
0 g
s
.56756 1.39721 m .57861 1.36218 L .50552 1.36593 L p .239 g
F P
0 g
s
.51544 .74837 m .51761 .67466 L .55982 .6788 L p .707 g
F P
0 g
s
.35748 .81469 m .29075 .84539 L .28365 .87997 L p .902 g
F P
0 g
s
.22953 .93453 m .28365 .87997 L .29075 .84539 L p .935 g
F P
0 g
s
.43956 .429 m .43765 .35823 L .522 .35234 L p .529 g
F P
0 g
s
.43765 .35823 m .43956 .429 L .36316 .44834 L p .472 g
F P
0 g
s
.29075 .84539 m .35748 .81469 L .37381 .79171 L p .914 g
F P
0 g
s
.55717 .77352 m .59956 .75948 L .56035 .74891 L p .668 g
F P
0 g
s
.56035 .74891 m .53716 .76817 L .55717 .77352 L p .668 g
F P
0 g
s
.68022 .63655 m .63111 .70581 L .59858 .68946 L p .746 g
F P
0 g
s
.43446 .75679 m .47309 .77216 L .49357 .76746 L p .79 g
F P
0 g
s
.47309 .77216 m .43446 .75679 L .40004 .77191 L p .828 g
F P
0 g
s
.20663 1.04532 m .19245 1.11824 L .22729 1.16072 L p .801 g
F P
0 g
s
.23372 1.23049 m .22729 1.16072 L .19245 1.11824 L p .719 g
F P
0 g
s
.22729 1.16072 m .23372 1.23049 L .28513 1.26475 L p .684 g
F P
0 g
s
.57861 1.36218 m .56756 1.39721 L .62505 1.38655 L p .139 g
F P
0 g
s
.62505 1.38655 m .64789 1.34918 L .57861 1.36218 L p .139 g
F P
0 g
s
.30578 1.32208 m .28513 1.26475 L .23372 1.23049 L p .565 g
F P
0 g
s
.28513 1.26475 m .30578 1.32208 L .36491 1.34537 L p .546 g
F P
0 g
s
.36316 .44834 m .37798 .52726 L .31948 .55491 L p .486 g
F P
0 g
s
.35469 .63105 m .31948 .55491 L .37798 .52726 L p .55 g
F P
0 g
s
.40271 .60894 m .43535 .68674 L .40132 .702 L p .604 g
F P
0 g
s
.40132 .702 m .35469 .63105 L .40271 .60894 L p .604 g
F P
0 g
s
.522 .35234 m .52177 .42336 L .43956 .429 L p .529 g
F P
0 g
s
.81437 1.05438 m .82826 1.12749 L .84678 1.08152 L p .055 g
F P
0 g
s
.84678 1.08152 m .83186 1.0094 L .81437 1.05438 L p .055 g
F P
0 g
s
.44643 1.39561 m .43286 1.36023 L .36491 1.34537 L p .345 g
F P
0 g
s
.43286 1.36023 m .44643 1.39561 L .50681 1.40028 L p .306 g
F P
0 g
s
.59858 .68946 m .63439 .61288 L .68022 .63655 L p .746 g
F P
0 g
s
.53716 .76817 m .56035 .74891 L .51766 .74481 L p .701 g
F P
0 g
s
.51766 .74481 m .51545 .7661 L .53716 .76817 L p .701 g
F P
0 g
s
.47464 .74751 m .49357 .76746 L .51545 .7661 L p .744 g
F P
0 g
s
.49357 .76746 m .47464 .74751 L .43446 .75679 L p .79 g
F P
0 g
s
.67773 .45393 m .66149 .5322 L .59436 .5128 L p .667 g
F P
0 g
s
.59436 .5128 m .6033 .43193 L .67773 .45393 L p .667 g
F P
0 g
s
.51545 .7661 m .51766 .74481 L .47464 .74751 L p .744 g
F P
0 g
s
.37798 .52726 m .40271 .60894 L .35469 .63105 L p .55 g
F P
0 g
s
.37798 .52726 m .36316 .44834 L .43956 .429 L p .536 g
F P
0 g
s
.22953 .93453 m .19348 .99975 L .20663 1.04532 L p .894 g
F P
0 g
s
.19245 1.11824 m .20663 1.04532 L .19348 .99975 L p .833 g
F P
0 g
s
.63439 .61288 m .59858 .68946 L .55982 .6788 L p .712 g
F P
0 g
s
.65675 .79627 m .71879 .81925 L .68446 .78989 L p .418 g
F P
0 g
s
.68446 .78989 m .63246 .77568 L .65675 .79627 L p .418 g
F P
0 g
s
.64789 1.34918 m .62505 1.38655 L .67574 1.36885 L p .075 g
F P
0 g
s
.67574 1.36885 m .70918 1.32756 L .64789 1.34918 L p .075 g
F P
0 g
s
.45904 .59534 m .47508 .67738 L .43535 .68674 L p .628 g
F P
0 g
s
.43535 .68674 m .40271 .60894 L .45904 .59534 L p .628 g
F P
0 g
s
.79602 .94327 m .83186 1.0094 L .82786 .96276 L p .093 g
F P
0 g
s
.82786 .96276 m .792 .90107 L .79602 .94327 L p .093 g
F P
0 g
s
.31349 .81276 m .37381 .79171 L .40004 .77191 L p .864 g
F P
0 g
s
.37381 .79171 m .31349 .81276 L .29075 .84539 L p .914 g
F P
0 g
s
.39007 1.38343 m .36491 1.34537 L .30578 1.32208 L p .35 g
F P
0 g
s
.36491 1.34537 m .39007 1.38343 L .44643 1.39561 L p .345 g
F P
0 g
s
.66149 .5322 m .63439 .61288 L .57946 .5974 L p .693 g
F P
0 g
s
.57946 .5974 m .59436 .5128 L .66149 .5322 L p .693 g
F P
0 g
s
.73834 .85249 m .792 .90107 L .76841 .86042 L p .217 g
F P
0 g
s
.76841 .86042 m .71879 .81925 L .73834 .85249 L p .217 g
F P
0 g
s
.55982 .6788 m .57946 .5974 L .63439 .61288 L p .712 g
F P
0 g
s
.57946 .5974 m .55982 .6788 L .51761 .67466 L p .682 g
F P
0 g
s
.6033 .43193 m .59436 .5128 L .52092 .50525 L p .623 g
F P
0 g
s
.52092 .50525 m .52177 .42336 L .6033 .43193 L p .623 g
F P
0 g
s
.51949 .59138 m .51761 .67466 L .47508 .67738 L p .654 g
F P
0 g
s
.47508 .67738 m .45904 .59534 L .51949 .59138 L p .654 g
F P
0 g
s
.29075 .84539 m .23761 .8924 L .22953 .93453 L p .935 g
F P
0 g
s
.19348 .99975 m .22953 .93453 L .23761 .8924 L p .884 g
F P
0 g
s
.43956 .429 m .44686 .51022 L .37798 .52726 L p .536 g
F P
0 g
s
.40271 .60894 m .37798 .52726 L .44686 .51022 L p .585 g
F P
0 g
s
.44686 .51022 m .43956 .429 L .52177 .42336 L p .58 g
F P
0 g
s
.51761 .67466 m .51949 .59138 L .57946 .5974 L p .682 g
F P
0 g
s
.44686 .51022 m .45904 .59534 L .40271 .60894 L p .585 g
F P
0 g
s
.23761 .8924 m .29075 .84539 L .31349 .81276 L p .879 g
F P
0 g
s
.78325 1.23819 m .75858 1.29843 L .79263 1.26341 L p .005 g
F P
0 g
s
.79263 1.26341 m .8211 1.19878 L .78325 1.23819 L p .005 g
F P
0 g
s
.52177 .42336 m .52092 .50525 L .44686 .51022 L p .58 g
F P
0 g
s
.63246 .77568 m .68446 .78989 L .63749 .76667 L p .375 g
F P
0 g
s
.63749 .76667 m .59956 .75948 L .63246 .77568 L p .375 g
F P
0 g
s
.82826 1.12749 m .8211 1.19878 L .83909 1.15493 L p .017 g
F P
0 g
s
.83909 1.15493 m .84678 1.08152 L .82826 1.12749 L p .017 g
F P
0 g
s
.35066 .78449 m .40004 .77191 L .43446 .75679 L p .736 g
F P
0 g
s
.40004 .77191 m .35066 .78449 L .31349 .81276 L p .864 g
F P
0 g
s
.19245 1.11824 m .19976 1.18996 L .23372 1.23049 L p .719 g
F P
0 g
s
.2593 1.29159 m .23372 1.23049 L .19976 1.18996 L p .51 g
F P
0 g
s
.23372 1.23049 m .2593 1.29159 L .30578 1.32208 L p .565 g
F P
0 g
s
.59436 .5128 m .57946 .5974 L .51949 .59138 L p .655 g
F P
0 g
s
.51949 .59138 m .52092 .50525 L .59436 .5128 L p .655 g
F P
0 g
s
.70918 1.32756 m .67574 1.36885 L .71636 1.34509 L p F P
s
.71636 1.34509 m .75858 1.29843 L .70918 1.32756 L p F P
s
.52092 .50525 m .51949 .59138 L .45904 .59534 L p .62 g
F P
0 g
s
.45904 .59534 m .44686 .51022 L .52092 .50525 L p .62 g
F P
0 g
s
.19348 .99975 m .17846 1.07165 L .19245 1.11824 L p .833 g
F P
0 g
s
.19976 1.18996 m .19245 1.11824 L .17846 1.07165 L p .629 g
F P
0 g
s
.34121 1.36438 m .30578 1.32208 L .2593 1.29159 L p .31 g
F P
0 g
s
.30578 1.32208 m .34121 1.36438 L .39007 1.38343 L p .35 g
F P
0 g
s
.83186 1.0094 m .84678 1.08152 L .84282 1.03378 L p .028 g
F P
0 g
s
.84282 1.03378 m .82786 .96276 L .83186 1.0094 L p .028 g
F P
0 g
s
.59956 .75948 m .63749 .76667 L .58116 .75148 L p .363 g
F P
0 g
s
.58116 .75148 m .56035 .74891 L .59956 .75948 L p .363 g
F P
0 g
s
.39988 .76281 m .43446 .75679 L .47464 .74751 L p .572 g
F P
0 g
s
.43446 .75679 m .39988 .76281 L .35066 .78449 L p .736 g
F P
0 g
s
.50858 1.4137 m .50681 1.40028 L .44643 1.39561 L p .112 g
F P
0 g
s
.50681 1.40028 m .50858 1.4137 L .55236 1.4115 L p .083 g
F P
0 g
s
.55236 1.4115 m .56756 1.39721 L .50681 1.40028 L p .083 g
F P
0 g
s
.71879 .81925 m .76841 .86042 L .72624 .82434 L p .113 g
F P
0 g
s
.72624 .82434 m .68446 .78989 L .71879 .81925 L p .113 g
F P
0 g
s
.56756 1.39721 m .55236 1.4115 L .59369 1.40386 L p .036 g
F P
0 g
s
.59369 1.40386 m .62505 1.38655 L .56756 1.39721 L p .036 g
F P
0 g
s
.792 .90107 m .82786 .96276 L .8017 .91768 L p .041 g
F P
0 g
s
.8017 .91768 m .76841 .86042 L .792 .90107 L p .041 g
F P
0 g
s
.23761 .8924 m .20208 .95316 L .19348 .99975 L p .884 g
F P
0 g
s
.17846 1.07165 m .19348 .99975 L .20208 .95316 L p .645 g
F P
0 g
s
.46509 1.41035 m .44643 1.39561 L .39007 1.38343 L p .122 g
F P
0 g
s
.44643 1.39561 m .46509 1.41035 L .50858 1.4137 L p .112 g
F P
0 g
s
.56035 .74891 m .58116 .75148 L .51965 .74558 L p .025 g
F P
0 g
s
.51965 .74558 m .51766 .74481 L .56035 .74891 L p .025 g
F P
0 g
s
.47464 .74751 m .45765 .74946 L .39988 .76281 L p .572 g
F P
0 g
s
.45765 .74946 m .47464 .74751 L .51766 .74481 L p F P
s
.31349 .81276 m .2652 .85245 L .23761 .8924 L p .879 g
F P
0 g
s
.2652 .85245 m .31349 .81276 L .35066 .78449 L p F P
s
.51766 .74481 m .51965 .74558 L .45765 .74946 L p F P
s
.20208 .95316 m .23761 .8924 L .2652 .85245 L p F P
s
.62505 1.38655 m .59369 1.40386 L .62997 1.39123 L p F P
s
.62997 1.39123 m .67574 1.36885 L .62505 1.38655 L p F P
s
.75858 1.29843 m .71636 1.34509 L .74406 1.31663 L p F P
s
.74406 1.31663 m .79263 1.26341 L .75858 1.29843 L p F P
s
.8211 1.19878 m .79263 1.26341 L .80855 1.22454 L p .695 g
F P
0 g
s
.80855 1.22454 m .83909 1.15493 L .8211 1.19878 L p .695 g
F P
0 g
s
.42459 1.40163 m .39007 1.38343 L .34121 1.36438 L p .11 g
F P
0 g
s
.39007 1.38343 m .42459 1.40163 L .46509 1.41035 L p .122 g
F P
0 g
s
.19976 1.18996 m .22881 1.25558 L .2593 1.29159 L p .51 g
F P
0 g
s
.2593 1.29159 m .30306 1.33952 L .34121 1.36438 L p .31 g
F P
0 g
s
.30306 1.33952 m .2593 1.29159 L .22881 1.25558 L p .058 g
F P
0 g
s
.17846 1.07165 m .1862 1.14552 L .19976 1.18996 L p .629 g
F P
0 g
s
.22881 1.25558 m .19976 1.18996 L .1862 1.14552 L p .043 g
F P
0 g
s
.68446 .78989 m .72624 .82434 L .66808 .79572 L p .187 g
F P
0 g
s
.66808 .79572 m .63749 .76667 L .68446 .78989 L p .187 g
F P
0 g
s
.84678 1.08152 m .83909 1.15493 L .8351 1.10943 L p .716 g
F P
0 g
s
.8351 1.10943 m .84282 1.03378 L .84678 1.08152 L p .716 g
F P
0 g
s
.31095 .8177 m .35066 .78449 L .39988 .76281 L p F P
s
.35066 .78449 m .31095 .8177 L .2652 .85245 L p F P
s
.67574 1.36885 m .62997 1.39123 L .6588 1.37434 L p .696 g
F P
0 g
s
.6588 1.37434 m .71636 1.34509 L .67574 1.36885 L p .696 g
F P
0 g
s
.20208 .95316 m .18723 1.02395 L .17846 1.07165 L p .645 g
F P
0 g
s
.1862 1.14552 m .17846 1.07165 L .18723 1.02395 L p .033 g
F P
0 g
s
.34121 1.36438 m .38968 1.38805 L .42459 1.40163 L p .11 g
F P
0 g
s
.38968 1.38805 m .34121 1.36438 L .30306 1.33952 L p .263 g
F P
0 g
s
.82786 .96276 m .84282 1.03378 L .81562 .98758 L p .633 g
F P
0 g
s
.81562 .98758 m .8017 .91768 L .82786 .96276 L p .633 g
F P
0 g
s
.76841 .86042 m .8017 .91768 L .75438 .87755 L p .456 g
F P
0 g
s
.75438 .87755 m .72624 .82434 L .76841 .86042 L p .456 g
F P
0 g
s
.63749 .76667 m .66808 .79572 L .59799 .77693 L p .204 g
F P
0 g
s
.59799 .77693 m .58116 .75148 L .63749 .76667 L p .204 g
F P
0 g
s
.2652 .85245 m .23277 .90883 L .20208 .95316 L p F P
s
.18723 1.02395 m .20208 .95316 L .23277 .90883 L p .021 g
F P
0 g
s
.23277 .90883 m .2652 .85245 L .31095 .8177 L p .002 g
F P
0 g
s
.37197 .79094 m .39988 .76281 L .45765 .74946 L p .077 g
F P
0 g
s
.39988 .76281 m .37197 .79094 L .31095 .8177 L p F P
s
.79263 1.26341 m .74406 1.31663 L .75664 1.28518 L p .902 g
F P
0 g
s
.75664 1.28518 m .80855 1.22454 L .79263 1.26341 L p .902 g
F P
0 g
s
.51074 1.40404 m .50858 1.4137 L .46509 1.41035 L p .576 g
F P
0 g
s
.50858 1.4137 m .51074 1.40404 L .53383 1.40288 L p .627 g
F P
0 g
s
.53383 1.40288 m .55236 1.4115 L .50858 1.4137 L p .627 g
F P
0 g
s
.71636 1.34509 m .6588 1.37434 L .67818 1.35421 L p .872 g
F P
0 g
s
.67818 1.35421 m .74406 1.31663 L .71636 1.34509 L p .872 g
F P
0 g
s
.55236 1.4115 m .53383 1.40288 L .55557 1.39884 L p .7 g
F P
0 g
s
.55557 1.39884 m .59369 1.40386 L .55236 1.4115 L p .7 g
F P
0 g
s
.48781 1.40227 m .46509 1.41035 L .42459 1.40163 L p .553 g
F P
0 g
s
.46509 1.41035 m .48781 1.40227 L .51074 1.40404 L p .576 g
F P
0 g
s
.58116 .75148 m .59799 .77693 L .52126 .76962 L p .164 g
F P
0 g
s
.52126 .76962 m .51965 .74558 L .58116 .75148 L p .164 g
F P
0 g
s
.83909 1.15493 m .80855 1.22454 L .80452 1.18432 L p .89 g
F P
0 g
s
.80452 1.18432 m .8351 1.10943 L .83909 1.15493 L p .89 g
F P
0 g
s
.1862 1.14552 m .21693 1.21621 L .22881 1.25558 L p .043 g
F P
0 g
s
.27834 1.31028 m .22881 1.25558 L .21693 1.21621 L p .187 g
F P
0 g
s
.22881 1.25558 m .27834 1.31028 L .30306 1.33952 L p .058 g
F P
0 g
s
.4439 .77443 m .45765 .74946 L .51965 .74558 L p .127 g
F P
0 g
s
.45765 .74946 m .4439 .77443 L .37197 .79094 L p .077 g
F P
0 g
s
.36266 1.37039 m .30306 1.33952 L .27834 1.31028 L p .328 g
F P
0 g
s
.30306 1.33952 m .36266 1.37039 L .38968 1.38805 L p .263 g
F P
0 g
s
.51965 .74558 m .52126 .76962 L .4439 .77443 L p .127 g
F P
0 g
s
.59369 1.40386 m .55557 1.39884 L .57454 1.39219 L p .778 g
F P
0 g
s
.57454 1.39219 m .62997 1.39123 L .59369 1.40386 L p .778 g
F P
0 g
s
.46653 1.39767 m .42459 1.40163 L .38968 1.38805 L p .553 g
F P
0 g
s
.42459 1.40163 m .46653 1.39767 L .48781 1.40227 L p .553 g
F P
0 g
s
.18723 1.02395 m .19489 1.10006 L .1862 1.14552 L p .033 g
F P
0 g
s
.21693 1.21621 m .1862 1.14552 L .19489 1.10006 L p .105 g
F P
0 g
s
.72624 .82434 m .75438 .87755 L .68873 .84563 L p .483 g
F P
0 g
s
.68873 .84563 m .66808 .79572 L .72624 .82434 L p .483 g
F P
0 g
s
.84282 1.03378 m .8351 1.10943 L .80844 1.06543 L p .801 g
F P
0 g
s
.80844 1.06543 m .81562 .98758 L .84282 1.03378 L p .801 g
F P
0 g
s
.28419 .87015 m .31095 .8177 L .37197 .79094 L p .108 g
F P
0 g
s
.31095 .8177 m .28419 .87015 L .23277 .90883 L p .002 g
F P
0 g
s
.8017 .91768 m .81562 .98758 L .76617 .94641 L p .663 g
F P
0 g
s
.76617 .94641 m .75438 .87755 L .8017 .91768 L p .663 g
F P
0 g
s
.62997 1.39123 m .57454 1.39219 L .58947 1.38335 L p .84 g
F P
0 g
s
.58947 1.38335 m .6588 1.37434 L .62997 1.39123 L p .84 g
F P
0 g
s
.23277 .90883 m .2192 .9785 L .18723 1.02395 L p .021 g
F P
0 g
s
.19489 1.10006 m .18723 1.02395 L .2192 .9785 L p .096 g
F P
0 g
s
.44831 1.39052 m .38968 1.38805 L .36266 1.37039 L p .565 g
F P
0 g
s
.38968 1.38805 m .44831 1.39052 L .46653 1.39767 L p .553 g
F P
0 g
s
.2192 .9785 m .23277 .90883 L .28419 .87015 L p .105 g
F P
0 g
s
.74406 1.31663 m .67818 1.35421 L .68662 1.33211 L p .926 g
F P
0 g
s
.68662 1.33211 m .75664 1.28518 L .74406 1.31663 L p .926 g
F P
0 g
s
.80855 1.22454 m .75664 1.28518 L .75274 1.2528 L p .925 g
F P
0 g
s
.75274 1.2528 m .80452 1.18432 L .80855 1.22454 L p .925 g
F P
0 g
s
.21693 1.21621 m .2691 1.27847 L .27834 1.31028 L p .187 g
F P
0 g
s
.34545 1.34974 m .27834 1.31028 L .2691 1.27847 L p .425 g
F P
0 g
s
.27834 1.31028 m .34545 1.34974 L .36266 1.37039 L p .328 g
F P
0 g
s
.66808 .79572 m .68873 .84563 L .60937 .82464 L p .454 g
F P
0 g
s
.60937 .82464 m .59799 .77693 L .66808 .79572 L p .454 g
F P
0 g
s
.6588 1.37434 m .58947 1.38335 L .59932 1.37287 L p .871 g
F P
0 g
s
.59932 1.37287 m .67818 1.35421 L .6588 1.37434 L p .871 g
F P
0 g
s
.19489 1.10006 m .22521 1.17605 L .21693 1.21621 L p .105 g
F P
0 g
s
.2691 1.27847 m .21693 1.21621 L .22521 1.17605 L p .327 g
F P
0 g
s
.35312 .84029 m .37197 .79094 L .4439 .77443 L p .231 g
F P
0 g
s
.37197 .79094 m .35312 .84029 L .28419 .87015 L p .108 g
F P
0 g
s
.43435 1.38128 m .36266 1.37039 L .34545 1.34974 L p .581 g
F P
0 g
s
.36266 1.37039 m .43435 1.38128 L .44831 1.39052 L p .565 g
F P
0 g
s
.8351 1.10943 m .80452 1.18432 L .78002 1.14553 L p .856 g
F P
0 g
s
.78002 1.14553 m .80844 1.06543 L .8351 1.10943 L p .856 g
F P
0 g
s
.51316 1.3707 m .51074 1.40404 L .48781 1.40227 L closepath p .748 g
F P
0 g
s
.51316 1.3707 m .53383 1.40288 L .51074 1.40404 L closepath p .761 g
F P
0 g
s
.51316 1.3707 m .55557 1.39884 L .53383 1.40288 L closepath p .774 g
F P
0 g
s
.51316 1.3707 m .48781 1.40227 L .46653 1.39767 L closepath p .736 g
F P
0 g
s
.59799 .77693 m .60937 .82464 L .52235 .81646 L p .403 g
F P
0 g
s
.52235 .81646 m .52126 .76962 L .59799 .77693 L p .403 g
F P
0 g
s
.2192 .9785 m .2262 1.05679 L .19489 1.10006 L p .096 g
F P
0 g
s
.22521 1.17605 m .19489 1.10006 L .2262 1.05679 L p .292 g
F P
0 g
s
.75438 .87755 m .76617 .94641 L .6974 .91361 L p .621 g
F P
0 g
s
.6974 .91361 m .68873 .84563 L .75438 .87755 L p .621 g
F P
0 g
s
.4346 .82184 m .4439 .77443 L .52126 .76962 L p .332 g
F P
0 g
s
.4439 .77443 m .4346 .82184 L .35312 .84029 L p .231 g
F P
0 g
s
.81562 .98758 m .80844 1.06543 L .76008 1.02624 L p .748 g
F P
0 g
s
.76008 1.02624 m .76617 .94641 L .81562 .98758 L p .748 g
F P
0 g
s
.51316 1.3707 m .57454 1.39219 L .55557 1.39884 L closepath p .785 g
F P
0 g
s
.51316 1.3707 m .46653 1.39767 L .44831 1.39052 L closepath p .726 g
F P
0 g
s
.67818 1.35421 m .59932 1.37287 L .60339 1.36145 L p .872 g
F P
0 g
s
.60339 1.36145 m .68662 1.33211 L .67818 1.35421 L p .872 g
F P
0 g
s
.75664 1.28518 m .68662 1.33211 L .68328 1.30949 L p .902 g
F P
0 g
s
.68328 1.30949 m .75274 1.2528 L .75664 1.28518 L p .902 g
F P
0 g
s
.52126 .76962 m .52235 .81646 L .4346 .82184 L p .332 g
F P
0 g
s
.27297 .93881 m .28419 .87015 L .35312 .84029 L p .276 g
F P
0 g
s
.28419 .87015 m .27297 .93881 L .2192 .9785 L p .105 g
F P
0 g
s
.2691 1.27847 m .33941 1.3274 L .34545 1.34974 L p .425 g
F P
0 g
s
.42565 1.37055 m .34545 1.34974 L .33941 1.3274 L p .596 g
F P
0 g
s
.34545 1.34974 m .42565 1.37055 L .43435 1.38128 L p .581 g
F P
0 g
s
.51316 1.3707 m .58947 1.38335 L .57454 1.39219 L closepath p .792 g
F P
0 g
s
.2262 1.05679 m .2192 .9785 L .27297 .93881 L p .285 g
F P
0 g
s
.51316 1.3707 m .44831 1.39052 L .43435 1.38128 L closepath p .718 g
F P
0 g
s
.22521 1.17605 m .2765 1.24616 L .2691 1.27847 L p .327 g
F P
0 g
s
.33941 1.3274 m .2691 1.27847 L .2765 1.24616 L p .49 g
F P
0 g
s
.80452 1.18432 m .75274 1.2528 L .73208 1.22172 L p .865 g
F P
0 g
s
.73208 1.22172 m .78002 1.14553 L .80452 1.18432 L p .865 g
F P
0 g
s
.51316 1.3707 m .59932 1.37287 L .58947 1.38335 L closepath p .796 g
F P
0 g
s
.51316 1.3707 m .43435 1.38128 L .42565 1.37055 L closepath p .713 g
F P
0 g
s
.2262 1.05679 m .25389 1.13793 L .22521 1.17605 L p .292 g
F P
0 g
s
.2765 1.24616 m .22521 1.17605 L .25389 1.13793 L p .428 g
F P
0 g
s
.68662 1.33211 m .60339 1.36145 L .60132 1.34985 L p .855 g
F P
0 g
s
.60132 1.34985 m .68328 1.30949 L .68662 1.33211 L p .855 g
F P
0 g
s
.68873 .84563 m .6974 .91361 L .61415 .89203 L p .569 g
F P
0 g
s
.61415 .89203 m .60937 .82464 L .68873 .84563 L p .569 g
F P
0 g
s
.51316 1.3707 m .60339 1.36145 L .59932 1.37287 L closepath p .796 g
F P
0 g
s
.33941 1.3274 m .42285 1.35903 L .42565 1.37055 L p .596 g
F P
0 g
s
.51316 1.3707 m .42565 1.37055 L .42285 1.35903 L closepath p .71 g
F P
0 g
s
.2765 1.24616 m .34522 1.30487 L .33941 1.3274 L p .49 g
F P
0 g
s
.42285 1.35903 m .33941 1.3274 L .34522 1.30487 L p .611 g
F P
0 g
s
.80844 1.06543 m .78002 1.14553 L .73605 1.11108 L p .785 g
F P
0 g
s
.73605 1.11108 m .76008 1.02624 L .80844 1.06543 L p .785 g
F P
0 g
s
.3452 .90813 m .35312 .84029 L .4346 .82184 L p .38 g
F P
0 g
s
.35312 .84029 m .3452 .90813 L .27297 .93881 L p .276 g
F P
0 g
s
.75274 1.2528 m .68328 1.30949 L .66813 1.28791 L p .854 g
F P
0 g
s
.66813 1.28791 m .73208 1.22172 L .75274 1.2528 L p .854 g
F P
0 g
s
.76617 .94641 m .76008 1.02624 L .69292 .99504 L p .686 g
F P
0 g
s
.69292 .99504 m .6974 .91361 L .76617 .94641 L p .686 g
F P
0 g
s
.27297 .93881 m .27876 1.019 L .2262 1.05679 L p .285 g
F P
0 g
s
.25389 1.13793 m .2262 1.05679 L .27876 1.019 L p .403 g
F P
0 g
s
.51316 1.3707 m .60132 1.34985 L .60339 1.36145 L closepath p .792 g
F P
0 g
s
.34522 1.30487 m .4262 1.34749 L .42285 1.35903 L p .611 g
F P
0 g
s
.51316 1.3707 m .42285 1.35903 L .4262 1.34749 L closepath p .709 g
F P
0 g
s
.25389 1.13793 m .30056 1.21564 L .2765 1.24616 L p .428 g
F P
0 g
s
.34522 1.30487 m .2765 1.24616 L .30056 1.21564 L p .534 g
F P
0 g
s
.27876 1.019 m .27297 .93881 L .3452 .90813 L p .394 g
F P
0 g
s
.51316 1.3707 m .57945 1.32928 L .59317 1.33887 L closepath p .776 g
F P
0 g
s
.51316 1.3707 m .56107 1.32177 L .57945 1.32928 L closepath p .766 g
F P
0 g
s
.51316 1.3707 m .53929 1.3169 L .56107 1.32177 L closepath p .756 g
F P
0 g
s
.51316 1.3707 m .51565 1.31501 L .53929 1.3169 L closepath p .746 g
F P
0 g
s
.51316 1.3707 m .49183 1.31625 L .51565 1.31501 L closepath p .736 g
F P
0 g
s
.51316 1.3707 m .46953 1.32053 L .49183 1.31625 L closepath p .728 g
F P
0 g
s
.51316 1.3707 m .45033 1.32753 L .46953 1.32053 L closepath p .72 g
F P
0 g
s
.51316 1.3707 m .43556 1.33674 L .45033 1.32753 L closepath p .715 g
F P
0 g
s
.51316 1.3707 m .4262 1.34749 L .43556 1.33674 L closepath p .711 g
F P
0 g
s
.51316 1.3707 m .59317 1.33887 L .60132 1.34985 L closepath p .785 g
F P
0 g
s
.60937 .82464 m .61415 .89203 L .52281 .88362 L p .515 g
F P
0 g
s
.52281 .88362 m .52235 .81646 L .60937 .82464 L p .515 g
F P
0 g
s
.4307 .88916 m .4346 .82184 L .52235 .81646 L p .455 g
F P
0 g
s
.4346 .82184 m .4307 .88916 L .3452 .90813 L p .38 g
F P
0 g
s
.68328 1.30949 m .60132 1.34985 L .59317 1.33887 L p .828 g
F P
0 g
s
.59317 1.33887 m .66813 1.28791 L .68328 1.30949 L p .828 g
F P
0 g
s
.52235 .81646 m .52281 .88362 L .4307 .88916 L p .455 g
F P
0 g
s
.30056 1.21564 m .36275 1.28371 L .34522 1.30487 L p .534 g
F P
0 g
s
.4262 1.34749 m .34522 1.30487 L .36275 1.28371 L p .625 g
F P
0 g
s
.78002 1.14553 m .73208 1.22172 L .69563 1.19422 L p .801 g
F P
0 g
s
.69563 1.19422 m .73605 1.11108 L .78002 1.14553 L p .801 g
F P
0 g
s
.36275 1.28371 m .43556 1.33674 L .4262 1.34749 L p .625 g
F P
0 g
s
.27876 1.019 m .30162 1.10473 L .25389 1.13793 L p .403 g
F P
0 g
s
.30056 1.21564 m .25389 1.13793 L .30162 1.10473 L p .492 g
F P
0 g
s
.66813 1.28791 m .59317 1.33887 L .57945 1.32928 L p .799 g
F P
0 g
s
.39101 1.26547 m .45033 1.32753 L .43556 1.33674 L p .64 g
F P
0 g
s
.43556 1.33674 m .36275 1.28371 L .39101 1.26547 L p .64 g
F P
0 g
s
.6974 .91361 m .69292 .99504 L .61168 .97452 L p .63 g
F P
0 g
s
.61168 .97452 m .61415 .89203 L .6974 .91361 L p .63 g
F P
0 g
s
.76008 1.02624 m .73605 1.11108 L .67527 1.08371 L p .722 g
F P
0 g
s
.67527 1.08371 m .69292 .99504 L .76008 1.02624 L p .722 g
F P
0 g
s
.73208 1.22172 m .66813 1.28791 L .64198 1.26895 L p .805 g
F P
0 g
s
.64198 1.26895 m .69563 1.19422 L .73208 1.22172 L p .805 g
F P
0 g
s
.57945 1.32928 m .64198 1.26895 L .66813 1.28791 L p .799 g
F P
0 g
s
.34929 .98982 m .3452 .90813 L .4307 .88916 L p .468 g
F P
0 g
s
.3452 .90813 m .34929 .98982 L .27876 1.019 L p .394 g
F P
0 g
s
.64198 1.26895 m .57945 1.32928 L .56107 1.32177 L p .77 g
F P
0 g
s
.30162 1.10473 m .34004 1.18917 L .30056 1.21564 L p .492 g
F P
0 g
s
.36275 1.28371 m .30056 1.21564 L .34004 1.18917 L p .568 g
F P
0 g
s
.30162 1.10473 m .27876 1.019 L .34929 .98982 L p .475 g
F P
0 g
s
.42814 1.25153 m .46953 1.32053 L .45033 1.32753 L p .657 g
F P
0 g
s
.45033 1.32753 m .39101 1.26547 L .42814 1.25153 L p .657 g
F P
0 g
s
.34004 1.18917 m .39101 1.26547 L .36275 1.28371 L p .568 g
F P
0 g
s
.56107 1.32177 m .60649 1.25401 L .64198 1.26895 L p .77 g
F P
0 g
s
.60649 1.25401 m .56107 1.32177 L .53929 1.3169 L p .743 g
F P
0 g
s
.47154 1.24297 m .49183 1.31625 L .46953 1.32053 L p .675 g
F P
0 g
s
.46953 1.32053 m .42814 1.25153 L .47154 1.24297 L p .675 g
F P
0 g
s
.53929 1.3169 m .56415 1.24426 L .60649 1.25401 L p .743 g
F P
0 g
s
.56415 1.24426 m .53929 1.3169 L .51565 1.31501 L p .718 g
F P
0 g
s
.61415 .89203 m .61168 .97452 L .52258 .96652 L p .578 g
F P
0 g
s
.52258 .96652 m .52281 .88362 L .61415 .89203 L p .578 g
F P
0 g
s
.51803 1.24048 m .51565 1.31501 L .49183 1.31625 L p .695 g
F P
0 g
s
.49183 1.31625 m .47154 1.24297 L .51803 1.24048 L p .695 g
F P
0 g
s
.73605 1.11108 m .69563 1.19422 L .64566 1.17247 L p .746 g
F P
0 g
s
.64566 1.17247 m .67527 1.08371 L .73605 1.11108 L p .746 g
F P
0 g
s
.51565 1.31501 m .51803 1.24048 L .56415 1.24426 L p .718 g
F P
0 g
s
.43271 .97178 m .4307 .88916 L .52281 .88362 L p .526 g
F P
0 g
s
.4307 .88916 m .43271 .97178 L .34929 .98982 L p .468 g
F P
0 g
s
.69563 1.19422 m .64198 1.26895 L .60649 1.25401 L p .761 g
F P
0 g
s
.34929 .98982 m .36541 1.07914 L .30162 1.10473 L p .475 g
F P
0 g
s
.34004 1.18917 m .30162 1.10473 L .36541 1.07914 L p .54 g
F P
0 g
s
.39243 1.16884 m .42814 1.25153 L .39101 1.26547 L p .598 g
F P
0 g
s
.39101 1.26547 m .34004 1.18917 L .39243 1.16884 L p .598 g
F P
0 g
s
.52281 .88362 m .52258 .96652 L .43271 .97178 L p .526 g
F P
0 g
s
.60649 1.25401 m .64566 1.17247 L .69563 1.19422 L p .761 g
F P
0 g
s
.69292 .99504 m .67527 1.08371 L .60195 1.06574 L p .669 g
F P
0 g
s
.60195 1.06574 m .61168 .97452 L .69292 .99504 L p .669 g
F P
0 g
s
.36541 1.07914 m .39243 1.16884 L .34004 1.18917 L p .54 g
F P
0 g
s
.36541 1.07914 m .34929 .98982 L .43271 .97178 L p .53 g
F P
0 g
s
.64566 1.17247 m .60649 1.25401 L .56415 1.24426 L p .722 g
F P
0 g
s
.45398 1.15633 m .47154 1.24297 L .42814 1.25153 L p .627 g
F P
0 g
s
.42814 1.25153 m .39243 1.16884 L .45398 1.15633 L p .627 g
F P
0 g
s
.67527 1.08371 m .64566 1.17247 L .58565 1.15822 L p .698 g
F P
0 g
s
.58565 1.15822 m .60195 1.06574 L .67527 1.08371 L p .698 g
F P
0 g
s
.56415 1.24426 m .58565 1.15822 L .64566 1.17247 L p .722 g
F P
0 g
s
.58565 1.15822 m .56415 1.24426 L .51803 1.24048 L p .688 g
F P
0 g
s
.61168 .97452 m .60195 1.06574 L .52164 1.05874 L p .622 g
F P
0 g
s
.52164 1.05874 m .52258 .96652 L .61168 .97452 L p .622 g
F P
0 g
s
.52008 1.15268 m .51803 1.24048 L .47154 1.24297 L p .657 g
F P
0 g
s
.47154 1.24297 m .45398 1.15633 L .52008 1.15268 L p .657 g
F P
0 g
s
.43271 .97178 m .44067 1.06334 L .36541 1.07914 L p .53 g
F P
0 g
s
.39243 1.16884 m .36541 1.07914 L .44067 1.06334 L p .58 g
F P
0 g
s
.44067 1.06334 m .43271 .97178 L .52258 .96652 L p .577 g
F P
0 g
s
.51803 1.24048 m .52008 1.15268 L .58565 1.15822 L p .688 g
F P
0 g
s
.44067 1.06334 m .45398 1.15633 L .39243 1.16884 L p .58 g
F P
0 g
s
.52258 .96652 m .52164 1.05874 L .44067 1.06334 L p .577 g
F P
0 g
s
.60195 1.06574 m .58565 1.15822 L .52008 1.15268 L p .657 g
F P
0 g
s
.52008 1.15268 m .52164 1.05874 L .60195 1.06574 L p .657 g
F P
0 g
s
.52164 1.05874 m .52008 1.15268 L .45398 1.15633 L p .618 g
F P
0 g
s
.45398 1.15633 m .44067 1.06334 L .52164 1.05874 L p .618 g
F P
0 g
s
.25 Mabswid
.64785 0 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.64785 0 m
.0843 .2212 L
s
.0843 .2212 m
.64785 0 L
s
.09701 .21621 m
.10379 .22387 L
s
gsave
.08347 .2009 -80.4345 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.226 .16558 m
.23257 .17341 L
s
gsave
.21284 .14994 -84.0684 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.35834 .11363 m
.36471 .12163 L
s
gsave
.3456 .09765 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4942 .06031 m
.50035 .06847 L
s
gsave
.4819 .04399 -78.1514 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.63369 .00556 m
.63961 .01389 L
s
gsave
.62186 -0.01111 -73.9305 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12255 .20619 m
.12659 .2108 L
s
.14821 .19612 m
.15223 .20075 L
s
.17401 .18599 m
.178 .19064 L
s
.19993 .17581 m
.2039 .18049 L
s
.25219 .1553 m
.25611 .16001 L
s
.27852 .14497 m
.28242 .1497 L
s
.30499 .13458 m
.30886 .13933 L
s
.3316 .12413 m
.33545 .12891 L
s
.38523 .10308 m
.38903 .1079 L
s
.41226 .09247 m
.41603 .09731 L
s
.43943 .08181 m
.44317 .08666 L
s
.46674 .07109 m
.47046 .07596 L
s
.5218 .04948 m
.52546 .05439 L
s
.54955 .03858 m
.55318 .04352 L
s
.57745 .02763 m
.58105 .03259 L
s
.60549 .01662 m
.60907 .0216 L
s
gsave
.2819 .01774 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.678571 0.0080766 0.97619 0.331141 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.00456621 1.00457 -0.0697153 1.00457 [
[.5 .82445 -19.375 0 ]
[.5 .82445 19.375 10 ]
[.04837 .26216 -13.9375 -9.78772 ]
[.04837 .26216 0 .21228 ]
[.21306 .20111 -20.0812 -10 ]
[.21306 .20111 .54385 0 ]
[.38678 .13673 -7.80548 -10 ]
[.38678 .13673 .63202 0 ]
[.5703 .06875 -13.2579 -10 ]
[.5703 .06875 1.86709 0 ]
[.76445 -0.00313 -6.9864 -10 ]
[.76445 -0.00313 1.4511 0 ]
[.34612 .08889 -7.34293 -10 ]
[.34612 .08889 .59457 0 ]
[.80748 .00247 0 -6.83927 ]
[.80748 .00247 20.625 3.16073 ]
[.85024 .06868 0 -6.78058 ]
[.85024 .06868 25.0625 3.21942 ]
[.89084 .13154 0 -6.72552 ]
[.89084 .13154 8.4375 3.27448 ]
[.92943 .1913 0 -6.67376 ]
[.92943 .1913 19.5625 3.32624 ]
[.96616 .24819 0 -6.62501 ]
[.96616 .24819 15.125 3.37499 ]
[.95019 .11106 0 -6.72552 ]
[.95019 .11106 7.9375 3.27448 ]
[.02878 .28862 -20.625 -3.16049 ]
[.02878 .28862 0 6.83951 ]
[.01947 .36565 -25.0625 -3.22033 ]
[.01947 .36565 0 6.77967 ]
[.00976 .44587 -8.4375 -3.28285 ]
[.00976 .44587 0 6.71715 ]
[-0.00035 .52948 -19.5625 -3.34821 ]
[-0.00035 .52948 0 6.65179 ]
[-0.0109 .6167 -15.125 -3.41663 ]
[-0.0109 .6167 0 6.58337 ]
[-0.04962 .46627 -7.9375 -3.28285 ]
[-0.04962 .46627 0 6.71715 ]
[ 0 0 0 0 ]
[ 1 .81195 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 .82445 -80.375 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(p) show
67.437500 11.500000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
70.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
76.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
97.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.04148 .27674 m
.7913 0 L
s
.05744 .27085 m
.06197 .27519 L
s
gsave
.04837 .26216 -74.9375 -13.7877 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.2217 .21022 m
.22601 .21478 L
s
gsave
.21306 .20111 -81.0812 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.39492 .14629 m
.39898 .15108 L
s
gsave
.38678 .13673 -68.8055 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.57785 .07878 m
.58163 .08379 L
s
gsave
.5703 .06875 -74.2579 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.77133 .00737 m
.77478 .01262 L
s
gsave
.76445 -0.00313 -67.9864 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08961 .25897 m
.0923 .26161 L
s
.12211 .24698 m
.12478 .24963 L
s
.15496 .23486 m
.1576 .23754 L
s
.18815 .2226 m
.19077 .22531 L
s
.2556 .19771 m
.25816 .20047 L
s
.28987 .18506 m
.2924 .18785 L
s
.32451 .17228 m
.32701 .17509 L
s
.35952 .15936 m
.36199 .1622 L
s
.4307 .13309 m
.43311 .13598 L
s
.46688 .11973 m
.46925 .12266 L
s
.50346 .10623 m
.5058 .10919 L
s
.54045 .09258 m
.54275 .09556 L
s
.61567 .06482 m
.6179 .06786 L
s
.65393 .0507 m
.65612 .05377 L
s
.69262 .03642 m
.69477 .03951 L
s
.73175 .02198 m
.73386 .0251 L
s
gsave
.34612 .08889 -68.3429 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.7913 0 m
.95779 .2576 L
s
.7957 .0068 m
.7898 .00897 L
s
gsave
.80748 .00247 -61 -10.8393 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.83841 .07289 m
.8325 .075 L
s
gsave
.85024 .06868 -61 -10.7806 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.87897 .13564 m
.87303 .13769 L
s
gsave
.89084 .13154 -61 -10.7255 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.91752 .19529 m
.91157 .19728 L
s
gsave
.92943 .1913 -61 -10.6738 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.95422 .25207 m
.94825 .25401 L
s
gsave
.96616 .24819 -61 -10.625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.80442 .0203 m
.80088 .02159 L
s
.81305 .03366 m
.80951 .03494 L
s
.82159 .04687 m
.81805 .04815 L
s
.83005 .05995 m
.8265 .06122 L
s
.84669 .0857 m
.84314 .08696 L
s
.85488 .09838 m
.85133 .09963 L
s
.86299 .11092 m
.85944 .11217 L
s
.87102 .12334 m
.86746 .12458 L
s
.88683 .14781 m
.88327 .14903 L
s
.89462 .15986 m
.89105 .16107 L
s
.90233 .17178 m
.89876 .17299 L
s
.90996 .1836 m
.90639 .1848 L
s
.925 .20687 m
.92143 .20806 L
s
.93241 .21833 m
.92884 .21952 L
s
.93975 .22969 m
.93617 .23086 L
s
.94702 .24093 m
.94344 .2421 L
s
gsave
.95019 .11106 -61 -10.7255 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.04148 .27674 m
0 .62185 L
s
.04057 .28429 m
.04646 .28212 L
s
gsave
.02878 .28862 -81.625 -7.16049 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.0313 .36144 m
.03721 .35934 L
s
gsave
.01947 .36565 -86.0625 -7.22033 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.02164 .44179 m
.02758 .43975 L
s
gsave
.00976 .44587 -69.4375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.01157 .52554 m
.01754 .52357 L
s
gsave
-0.00035 .52948 -80.5625 -7.34821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00107 .61291 m
.00706 .61101 L
s
gsave
-0.0109 .6167 -76.125 -7.41663 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.03874 .29947 m
.04228 .29818 L
s
.0369 .31478 m
.04045 .31349 L
s
.03505 .33021 m
.03859 .32893 L
s
.03318 .34576 m
.03673 .34449 L
s
.0294 .37725 m
.03295 .37599 L
s
.02748 .39319 m
.03104 .39194 L
s
.02555 .40926 m
.02911 .40802 L
s
.0236 .42546 m
.02716 .42423 L
s
.01966 .45826 m
.02323 .45705 L
s
.01766 .47487 m
.02123 .47367 L
s
.01565 .49162 m
.01922 .49042 L
s
.01362 .50851 m
.0172 .50732 L
s
.00951 .54272 m
.01309 .54155 L
s
.00743 .56004 m
.01101 .55888 L
s
.00533 .57751 m
.00891 .57636 L
s
.00321 .59514 m
.0068 .59399 L
s
gsave
-0.04962 .46627 -68.9375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.04148 .27674 m
0 .62185 L
s
0 .62185 m
.24023 .81195 L
s
.24023 .81195 m
.25974 .48381 L
s
.25974 .48381 m
.04148 .27674 L
s
.7913 0 m
.95779 .2576 L
s
.95779 .2576 m
1 .60412 L
s
1 .60412 m
.82157 .36284 L
s
.82157 .36284 m
.7913 0 L
s
.04148 .27674 m
0 .62185 L
s
0 .62185 m
.82157 .36284 L
s
.82157 .36284 m
.7913 0 L
s
.7913 0 m
.04148 .27674 L
s
.25974 .48381 m
.95779 .2576 L
s
.95779 .2576 m
1 .60412 L
s
1 .60412 m
.24023 .81195 L
s
.24023 .81195 m
.25974 .48381 L
s
0 0 m
1 0 L
1 .81195 L
0 .81195 L
closepath
clip
newpath
.5 Mabswid
.3013 .41653 m .27447 .4548 L .354 .47912 L p .658 g
F P
0 g
s
.354 .47912 m .37109 .44175 L .3013 .41653 L p .658 g
F P
0 g
s
.3405 .51894 m .354 .47912 L .27447 .4548 L p .651 g
F P
0 g
s
.37109 .44175 m .354 .47912 L .4332 .48519 L p .58 g
F P
0 g
s
.354 .47912 m .3405 .51894 L .4272 .51928 L p .57 g
F P
0 g
s
.4272 .51928 m .4332 .48519 L .354 .47912 L p .57 g
F P
0 g
s
.27447 .4548 m .25317 .49666 L .3405 .51894 L p .651 g
F P
0 g
s
.3314 .55887 m .3405 .51894 L .25317 .49666 L p .636 g
F P
0 g
s
.3405 .51894 m .3314 .55887 L .4232 .55295 L p .55 g
F P
0 g
s
.4232 .55295 m .4272 .51928 L .3405 .51894 L p .55 g
F P
0 g
s
.33211 .38409 m .3013 .41653 L .37109 .44175 L p .657 g
F P
0 g
s
.37109 .44175 m .39082 .40902 L .33211 .38409 L p .657 g
F P
0 g
s
.39082 .40902 m .37109 .44175 L .44086 .45269 L p .579 g
F P
0 g
s
.4332 .48519 m .44086 .45269 L .37109 .44175 L p .58 g
F P
0 g
s
.24808 .38027 m .21334 .41555 L .27447 .4548 L p .737 g
F P
0 g
s
.27447 .4548 m .3013 .41653 L .24808 .38027 L p .737 g
F P
0 g
s
.25317 .49666 m .27447 .4548 L .21334 .41555 L p .735 g
F P
0 g
s
.25317 .49666 m .23867 .53969 L .3314 .55887 L p .636 g
F P
0 g
s
.32729 .59658 m .3314 .55887 L .23867 .53969 L p .61 g
F P
0 g
s
.3314 .55887 m .32729 .59658 L .42146 .58416 L p .518 g
F P
0 g
s
.42146 .58416 m .4232 .55295 L .3314 .55887 L p .518 g
F P
0 g
s
.28776 .35199 m .24808 .38027 L .3013 .41653 L p .731 g
F P
0 g
s
.3013 .41653 m .33211 .38409 L .28776 .35199 L p .731 g
F P
0 g
s
.21334 .41555 m .18562 .45577 L .25317 .49666 L p .735 g
F P
0 g
s
.23867 .53969 m .25317 .49666 L .18562 .45577 L p .725 g
F P
0 g
s
.36514 .35949 m .33211 .38409 L .39082 .40902 L p .644 g
F P
0 g
s
.39082 .40902 m .41204 .38297 L .36514 .35949 L p .644 g
F P
0 g
s
.41204 .38297 m .39082 .40902 L .44972 .42374 L p .562 g
F P
0 g
s
.44086 .45269 m .44972 .42374 L .39082 .40902 L p .579 g
F P
0 g
s
.4332 .48519 m .4272 .51928 L .49286 .49956 L p .455 g
F P
0 g
s
.4272 .51928 m .4232 .55295 L .49259 .52404 L p .428 g
F P
0 g
s
.49259 .52404 m .49286 .49956 L .4272 .51928 L p .428 g
F P
0 g
s
.44086 .45269 m .4332 .48519 L .49327 .47457 L p .465 g
F P
0 g
s
.49286 .49956 m .49327 .47457 L .4332 .48519 L p .455 g
F P
0 g
s
.33005 .33244 m .28776 .35199 L .33211 .38409 L p .715 g
F P
0 g
s
.33211 .38409 m .36514 .35949 L .33005 .33244 L p .715 g
F P
0 g
s
.18562 .45577 m .16664 .49858 L .23867 .53969 L p .725 g
F P
0 g
s
.23195 .58134 m .23867 .53969 L .16664 .49858 L p .704 g
F P
0 g
s
.23867 .53969 m .23195 .58134 L .32729 .59658 L p .61 g
F P
0 g
s
.32851 .62974 m .32729 .59658 L .23195 .58134 L p .567 g
F P
0 g
s
.32729 .59658 m .32851 .62974 L .42214 .61099 L p .466 g
F P
0 g
s
.42214 .61099 m .42146 .58416 L .32729 .59658 L p .466 g
F P
0 g
s
.4232 .55295 m .42146 .58416 L .49248 .5465 L p .381 g
F P
0 g
s
.49248 .5465 m .49259 .52404 L .4232 .55295 L p .381 g
F P
0 g
s
.44972 .42374 m .44086 .45269 L .4938 .4506 L p .456 g
F P
0 g
s
.49327 .47457 m .4938 .4506 L .44086 .45269 L p .465 g
F P
0 g
s
.39843 .34439 m .36514 .35949 L .41204 .38297 L p .611 g
F P
0 g
s
.41204 .38297 m .43349 .36537 L .39843 .34439 L p .611 g
F P
0 g
s
.43349 .36537 m .41204 .38297 L .45927 .40017 L p .515 g
F P
0 g
s
.44972 .42374 m .45927 .40017 L .41204 .38297 L p .562 g
F P
0 g
s
.37246 .32292 m .33005 .33244 L .36514 .35949 L p .682 g
F P
0 g
s
.36514 .35949 m .39843 .34439 L .37246 .32292 L p .682 g
F P
0 g
s
.42146 .58416 m .42214 .61099 L .49254 .56548 L p .303 g
F P
0 g
s
.49254 .56548 m .49248 .5465 L .42146 .58416 L p .303 g
F P
0 g
s
.45927 .40017 m .44972 .42374 L .49442 .42915 L p .417 g
F P
0 g
s
.4938 .4506 m .49442 .42915 L .44972 .42374 L p .456 g
F P
0 g
s
.26927 .31946 m .22526 .3406 L .24808 .38027 L p .82 g
F P
0 g
s
.24808 .38027 m .28776 .35199 L .26927 .31946 L p .82 g
F P
0 g
s
.21334 .41555 m .24808 .38027 L .22526 .3406 L p .839 g
F P
0 g
s
.22526 .3406 m .18654 .36963 L .21334 .41555 L p .839 g
F P
0 g
s
.18562 .45577 m .21334 .41555 L .18654 .36963 L p .849 g
F P
0 g
s
.16664 .49858 m .15776 .54141 L .23195 .58134 L p .704 g
F P
0 g
s
.23361 .61904 m .23195 .58134 L .15776 .54141 L p .662 g
F P
0 g
s
.23195 .58134 m .23361 .61904 L .32851 .62974 L p .567 g
F P
0 g
s
.31591 .30747 m .26927 .31946 L .28776 .35199 L p .791 g
F P
0 g
s
.28776 .35199 m .33005 .33244 L .31591 .30747 L p .791 g
F P
0 g
s
.33516 .65614 m .32851 .62974 L .23361 .61904 L p .489 g
F P
0 g
s
.32851 .62974 m .33516 .65614 L .42524 .63164 L p .376 g
F P
0 g
s
.42524 .63164 m .42214 .61099 L .32851 .62974 L p .376 g
F P
0 g
s
.18654 .36963 m .15552 .40487 L .18562 .45577 L p .849 g
F P
0 g
s
.16664 .49858 m .18562 .45577 L .15552 .40487 L p .849 g
F P
0 g
s
.36236 .3054 m .31591 .30747 L .33005 .33244 L p .741 g
F P
0 g
s
.33005 .33244 m .37246 .32292 L .36236 .3054 L p .741 g
F P
0 g
s
.42214 .61099 m .42524 .63164 L .49276 .57973 L p .167 g
F P
0 g
s
.49276 .57973 m .49254 .56548 L .42214 .61099 L p .167 g
F P
0 g
s
.45927 .40017 m .46892 .3836 L .43349 .36537 L p .515 g
F P
0 g
s
.46892 .3836 m .45927 .40017 L .49508 .41161 L p .321 g
F P
0 g
s
.49442 .42915 m .49508 .41161 L .45927 .40017 L p .417 g
F P
0 g
s
.41236 .32427 m .37246 .32292 L .39843 .34439 L p .611 g
F P
0 g
s
.39843 .34439 m .4299 .34004 L .41236 .32427 L p .611 g
F P
0 g
s
.4299 .34004 m .39843 .34439 L .43349 .36537 L p .535 g
F P
0 g
s
.43349 .36537 m .45383 .35764 L .4299 .34004 L p .535 g
F P
0 g
s
.45383 .35764 m .43349 .36537 L .46892 .3836 L p .404 g
F P
0 g
s
.15552 .40487 m .13427 .44423 L .16664 .49858 L p .849 g
F P
0 g
s
.15776 .54141 m .16664 .49858 L .13427 .44423 L p .825 g
F P
0 g
s
.15776 .54141 m .15981 .58156 L .23361 .61904 L p .662 g
F P
0 g
s
.24382 .65028 m .23361 .61904 L .15981 .58156 L p .569 g
F P
0 g
s
.23361 .61904 m .24382 .65028 L .33516 .65614 L p .489 g
F P
0 g
s
.49327 .47457 m .49286 .49956 L .51988 .46677 L p .153 g
F P
0 g
s
.49286 .49956 m .49259 .52404 L .5211 .4797 L p .109 g
F P
0 g
s
.5211 .4797 m .51988 .46677 L .49286 .49956 L p .109 g
F P
0 g
s
.4938 .4506 m .49327 .47457 L .518 .45357 L p .153 g
F P
0 g
s
.51988 .46677 m .518 .45357 L .49327 .47457 L p .153 g
F P
0 g
s
.49259 .52404 m .49248 .5465 L .52156 .49152 L p .05 g
F P
0 g
s
.52156 .49152 m .5211 .4797 L .49259 .52404 L p .05 g
F P
0 g
s
.49442 .42915 m .4938 .4506 L .51556 .44094 L p .131 g
F P
0 g
s
.518 .45357 m .51556 .44094 L .4938 .4506 L p .153 g
F P
0 g
s
.40579 .3135 m .36236 .3054 L .37246 .32292 L p .644 g
F P
0 g
s
.37246 .32292 m .41236 .32427 L .40579 .3135 L p .644 g
F P
0 g
s
.347 .67383 m .33516 .65614 L .24382 .65028 L p .332 g
F P
0 g
s
.33516 .65614 m .347 .67383 L .43063 .64459 L p .206 g
F P
0 g
s
.43063 .64459 m .42524 .63164 L .33516 .65614 L p .206 g
F P
0 g
s
.42524 .63164 m .43063 .64459 L .49315 .58819 L p .043 g
F P
0 g
s
.49315 .58819 m .49276 .57973 L .42524 .63164 L p .043 g
F P
0 g
s
.46892 .3836 m .47807 .37529 L .45383 .35764 L p .404 g
F P
0 g
s
.47807 .37529 m .46892 .3836 L .49575 .39918 L p .163 g
F P
0 g
s
.49508 .41161 m .49575 .39918 L .46892 .3836 L p .321 g
F P
0 g
s
.49248 .5465 m .49254 .56548 L .52123 .50144 L p .032 g
F P
0 g
s
.52123 .50144 m .52156 .49152 L .49248 .5465 L p .032 g
F P
0 g
s
.49508 .41161 m .49442 .42915 L .51272 .42972 L p .073 g
F P
0 g
s
.51556 .44094 m .51272 .42972 L .49442 .42915 L p .131 g
F P
0 g
s
.32723 .29226 m .28278 .29603 L .26927 .31946 L p .815 g
F P
0 g
s
.26927 .31946 m .31591 .30747 L .32723 .29226 L p .815 g
F P
0 g
s
.22526 .3406 m .26927 .31946 L .28278 .29603 L p .883 g
F P
0 g
s
.13427 .44423 m .12438 .48531 L .15776 .54141 L p .825 g
F P
0 g
s
.15981 .58156 m .15776 .54141 L .12438 .48531 L p .738 g
F P
0 g
s
.28278 .29603 m .24062 .30866 L .22526 .3406 L p .883 g
F P
0 g
s
.18654 .36963 m .22526 .3406 L .24062 .30866 L p .918 g
F P
0 g
s
.44713 .33672 m .41236 .32427 L .4299 .34004 L p .446 g
F P
0 g
s
.4299 .34004 m .45745 .34711 L .44713 .33672 L p .446 g
F P
0 g
s
.45745 .34711 m .4299 .34004 L .45383 .35764 L p .355 g
F P
0 g
s
.37125 .29754 m .32723 .29226 L .31591 .30747 L p .7 g
F P
0 g
s
.36236 .3054 m .37125 .29754 L p .35118 .30114 L .7 g
F P
0 g
s
.41213 .31157 m .37125 .29754 L .36236 .3054 L p .492 g
F P
0 g
s
.40579 .3135 m .41213 .31157 L p .38732 .30849 L .492 g
F P
0 g
s
.44339 .33148 m .40579 .3135 L .41236 .32427 L p .425 g
F P
0 g
s
.44713 .33672 m .44339 .33148 L p .41908 .32583 L .425 g
F P
0 g
s
.47262 .35843 m .44339 .33148 L .44713 .33672 L p .1 g
F P
0 g
s
.47428 .35985 m .44713 .33672 L .45745 .34711 L p .16 g
F P
0 g
s
.45745 .34711 m .47904 .36559 L .47428 .35985 L p .16 g
F P
0 g
s
.47904 .36559 m .45745 .34711 L .47165 .36066 L p .088 g
F P
0 g
s
.44713 .33672 m .47428 .35985 L .47262 .35843 L p .1 g
F P
0 g
s
.47262 .35843 m .47304 .35928 L p .44995 .33796 L .327 g
F P
0 g
s
.44995 .33796 m .47304 .35928 L .46534 .35321 L .327 g
F
.45205 .33945 m .47262 .35843 L p .44995 .33796 L F P
0 g
s
.44731 .33357 m .41999 .31648 L p .41997 .31667 L .098 g
F P
0 g
s
.40579 .3135 m .41963 .32012 L p .41963 .32019 L .098 g
F P
0 g
s
.41963 .32012 m .44339 .33148 L p .41963 .32019 L .098 g
F P
0 g
s
.44339 .33148 m .45205 .33945 L p .44995 .33796 L .327 g
F P
0 g
s
.47304 .35928 m .4745 .36227 L p .46534 .35321 L .327 g
F P
0 g
s
.44731 .33357 m .44379 .33186 L p .4745 .36227 L .327 g
F P
0 g
s
.44339 .33148 m .44731 .33357 L p .41963 .32019 L .098 g
F P
0 g
s
.41963 .32019 m .4362 .3282 L .41997 .31667 L .098 g
F
.4362 .3282 m .44731 .33357 L .41997 .31667 L F
.41999 .31648 m .41213 .31157 L p .41963 .32019 L F P
0 g
s
.419 .32665 m .44713 .33672 L p .41908 .32583 L .425 g
F P
0 g
s
.45383 .35764 m .47165 .36066 L .45745 .34711 L p .355 g
F P
0 g
s
.47165 .36066 m .45383 .35764 L .47807 .37529 L p .189 g
F P
0 g
s
.47807 .37529 m .48606 .37611 L .47165 .36066 L p .189 g
F P
0 g
s
.47165 .36066 m .48563 .37467 L .47904 .36559 L p .088 g
F P
0 g
s
.47165 .36066 m .48606 .37611 L p .48299 .37211 L .043 g
F P
0 g
s
.48493 .37397 m .48234 .37137 L p .48568 .37482 L .043 g
F P
0 g
s
.48234 .37137 m .47165 .36066 L p .48274 .37186 L .043 g
F P
0 g
s
.48568 .37482 m .48234 .37137 L .48274 .37186 L .043 g
F
.47262 .35843 m .47428 .35985 L p .47675 .36463 L .5 g
F P
0 g
s
.48606 .37611 m .47807 .37529 L .49638 .39278 L p .046 g
F P
0 g
s
.49575 .39918 m .49638 .39278 L .47807 .37529 L p .163 g
F P
0 g
s
.49126 .39283 m .47453 .36196 L p .47675 .36463 L .5 g
F P
0 g
s
.47453 .36196 m .47262 .35843 L p .47675 .36463 L .5 g
F P
0 g
s
.4929 .39465 m .49168 .39245 L p .48073 .36917 L .409 g
F P
0 g
s
.47904 .36559 m .48079 .36925 L p .48073 .36917 L .409 g
F P
0 g
s
.48079 .36925 m .4929 .39465 L p .48073 .36917 L .409 g
F P
0 g
s
.48606 .37611 m .48568 .37482 L .48299 .37211 L .043 g
F
.48563 .37467 m .48493 .37397 L p .48568 .37482 L F P
0 g
s
.47428 .35985 m .47904 .36559 L p .48073 .36917 L .409 g
F P
0 g
s
.49168 .39245 m .47697 .36488 L p .48073 .36917 L .409 g
F P
0 g
s
.47697 .36488 m .47428 .35985 L p .48073 .36917 L .409 g
F P
0 g
s
.4745 .36227 m .48064 .37425 L p .47934 .37262 L .579 g
F P
0 g
s
.47945 .372 m .4745 .36227 L p .47934 .37262 L .579 g
F P
0 g
s
.49126 .39283 m .49168 .39245 L .49328 .4031 L p .683 g
F P
0 g
s
.49269 .4016 m .49126 .39283 L p .49328 .4031 L .683 g
F P
0 g
s
.49176 .39597 m .49126 .39283 L .49269 .4016 L p .498 g
F P
0 g
s
.49269 .4016 m .49263 .40131 L .49176 .39597 L .498 g
F
.49383 .40869 m .49176 .39597 L p .49395 .40934 L F P
0 g
s
.49269 .4016 m .49395 .40934 L p .49263 .40131 L .498 g
F P
0 g
s
.49395 .40934 m .49545 .41863 L p .49383 .40869 L .498 g
F P
0 g
s
.49545 .41863 m .49772 .43257 L .49536 .41806 L p .498 g
F P
0 g
s
.49536 .41806 m .49383 .40869 L p .49545 .41863 L .498 g
F P
0 g
s
.49328 .4031 m .49446 .4109 L p .49269 .4016 L .683 g
F P
0 g
s
.49395 .40934 m .49269 .4016 L p .49446 .4109 L .683 g
F P
0 g
s
.49304 .40188 m .49255 .40083 L .49383 .40869 L p .047 g
F P
0 g
s
.49446 .4109 m .49772 .43257 L .49395 .40934 L p .683 g
F P
0 g
s
.49383 .40869 m .49536 .41806 L p .49406 .40898 L .047 g
F P
0 g
s
.49383 .40869 m .49406 .40898 L .49304 .40188 L .047 g
F
.49536 .41806 m .49556 .41838 L .49304 .40188 L F
.49176 .39597 m .49304 .40188 L p .47934 .37262 L .579 g
F P
0 g
s
.4929 .39465 m .49772 .43257 L .49168 .39245 L closepath p .676 g
F P
0 g
s
.49477 .41033 m .49304 .40188 L .49556 .41838 L p .186 g
F P
0 g
s
.49556 .41838 m .49772 .43257 L .49598 .41942 L p .186 g
F P
0 g
s
.49598 .41942 m .49477 .41033 L p .49556 .41838 L .186 g
F P
0 g
s
.49575 .39918 m .49508 .41161 L .50968 .42064 L p .049 g
F P
0 g
s
.51272 .42972 m .50968 .42064 L .49508 .41161 L p .073 g
F P
0 g
s
.4946 .39911 m .49772 .43257 L .4929 .39465 L closepath p .628 g
F P
0 g
s
.49304 .40188 m .49477 .41033 L p .48563 .38745 L .612 g
F P
0 g
s
.49143 .39844 m .49254 .4008 L .49111 .39764 L .579 g
F
.49304 .40188 m .49253 .40074 L p .49254 .4008 L F P
0 g
s
.49253 .40074 m .49111 .39764 L p .49254 .4008 L .579 g
F P
0 g
s
.48064 .37425 m .49176 .39597 L p .47934 .37262 L .579 g
F P
0 g
s
.47934 .37262 m .49143 .39844 L .47945 .372 L .579 g
F
.49111 .39764 m .47945 .372 L p .49143 .39844 L F P
0 g
s
.48555 .38722 m .49304 .40188 L .48563 .38745 L .612 g
F
.48335 .38056 m .48839 .39165 L p .48555 .38722 L F P
0 g
s
.48839 .39165 m .49304 .40188 L p .48555 .38722 L .612 g
F P
0 g
s
.47945 .372 m .48335 .38056 L p .48563 .38745 L .612 g
F P
0 g
s
.48618 .38763 m .47945 .372 L p .49135 .40176 L .612 g
F P
0 g
s
.4745 .36227 m .47945 .372 L p .45833 .34674 L .487 g
F P
0 g
s
.44731 .33357 m .46475 .35197 L p .45833 .34674 L .487 g
F P
0 g
s
.46475 .35197 m .4745 .36227 L p .45833 .34674 L .487 g
F P
0 g
s
.45404 .34068 m .44731 .33357 L p .46813 .35846 L .487 g
F P
0 g
s
.47945 .372 m .45796 .34481 L .45404 .34068 L p .487 g
F P
0 g
s
.46813 .35846 m .47945 .372 L .45404 .34068 L .487 g
F
.45796 .34481 m .47945 .372 L .48101 .37563 L p .575 g
F P
0 g
s
.48101 .37563 m .48618 .38763 L p .47316 .36787 L .575 g
F P
0 g
s
.47252 .36658 m .48101 .37563 L .47316 .36787 L .575 g
F
.45796 .34481 m .48101 .37563 L .47181 .36582 L F
.47181 .36582 m .47252 .36658 L .47316 .36787 L F
.48618 .38763 m .47255 .36592 L .46373 .35316 L p F P
0 g
s
.47308 .36775 m .48618 .38763 L .46373 .35316 L .575 g
F
.49361 .40742 m .49441 .40942 L .49356 .40712 L .612 g
F
.49477 .41033 m .49445 .40947 L p .49441 .40942 L F P
0 g
s
.49445 .40947 m .49356 .40712 L p .49441 .40942 L .612 g
F P
0 g
s
.49135 .40176 m .49194 .40325 L .49171 .40262 L .612 g
F
.49135 .40176 m .49171 .40262 L .48618 .38763 L F
.49194 .40325 m .49361 .40742 L .49095 .40022 L F
.49356 .40712 m .49095 .40022 L p .49361 .40742 L F P
0 g
s
.49095 .40022 m .48618 .38763 L p .49194 .40325 L .612 g
F P
0 g
s
.49477 .41033 m .49772 .43257 L .49648 .42081 L closepath p .331 g
F P
0 g
s
.48618 .38763 m .49477 .41033 L .49648 .42081 L p .648 g
F P
0 g
s
.49288 .40836 m .48618 .38763 L p .49362 .41157 L .648 g
F P
0 g
s
.49059 .40127 m .49288 .40836 L p .48647 .39498 L .636 g
F P
0 g
s
.48094 .38343 m .49059 .40127 L .48647 .39498 L .636 g
F
.49288 .40836 m .4877 .39723 L p .48298 .38769 L F P
0 g
s
.4877 .39723 m .48714 .39602 L p .48298 .38769 L .636 g
F P
0 g
s
.48714 .39602 m .47966 .3846 L .46046 .35526 L p .616 g
F P
0 g
s
.46373 .35316 m .45796 .34481 L p .47308 .36775 L .575 g
F P
0 g
s
.44731 .33357 m .45796 .34481 L p .43367 .32719 L .311 g
F P
0 g
s
.43019 .32467 m .43197 .3256 L .43367 .32719 L .311 g
F
.43197 .3256 m .44731 .33357 L .43367 .32719 L F
.41213 .31157 m .44731 .33357 L p .43019 .32467 L F P
0 g
s
.45796 .34481 m .43 .32197 L .42289 .31783 L p .311 g
F P
0 g
s
.42289 .31783 m .41405 .31269 L p .43863 .33067 L .311 g
F P
0 g
s
.43863 .33067 m .45796 .34481 L .42289 .31783 L .311 g
F
.41831 .31605 m .45796 .34481 L .41405 .31269 L F
.43 .32197 m .45796 .34481 L .46461 .35443 L p .511 g
F P
0 g
s
.46461 .35443 m .47255 .36592 L p .45324 .34597 L .511 g
F P
0 g
s
.43 .32197 m .46461 .35443 L .45324 .34597 L .511 g
F
.45476 .34655 m .43933 .33123 L p .45357 .34609 L F P
0 g
s
.47255 .36592 m .45476 .34655 L p .45357 .34609 L .511 g
F P
0 g
s
.45098 .34364 m .47255 .36592 L .43933 .33123 L .511 g
F
.41213 .31157 m .40579 .3135 L p .41963 .32019 L .098 g
F P
0 g
s
.43933 .33123 m .43 .32197 L p .44938 .34199 L .511 g
F P
0 g
s
.44938 .34199 m .45098 .34364 L .4498 .34238 L .511 g
F
.44938 .34199 m .4498 .34238 L .43933 .33123 L F
.41405 .31269 m .41213 .31157 L p .41831 .31605 L .311 g
F P
0 g
s
.41213 .31157 m .43 .32197 L p .40565 .31184 L .154 g
F P
0 g
s
.37125 .29754 m .39743 .30653 L p .40222 .30962 L .154 g
F P
0 g
s
.37125 .29754 m .40222 .30962 L .39606 .30786 L .154 g
F
.39349 .3056 m .39646 .30693 L .39591 .30446 L F
.39743 .30653 m .41213 .31157 L p .40565 .31184 L F P
0 g
s
.39646 .30693 m .43 .32197 L .39723 .30514 L .154 g
F
.39646 .30693 m .39723 .30514 L .39591 .30446 L F
.43 .32197 m .39735 .30486 L .39591 .30446 L p F P
0 g
s
.39591 .30446 m .37125 .29754 L p .39215 .30623 L .154 g
F P
0 g
s
.35604 .29571 m .37125 .29754 L .39735 .30486 L p .098 g
F P
0 g
s
.32723 .29226 m .35604 .29571 L p .36908 .2986 L .098 g
F P
0 g
s
.32723 .29226 m .36908 .2986 L .35743 .29769 L .098 g
F
.39735 .30486 m .36205 .29456 L .34604 .2935 L p F P
0 g
s
.34604 .2935 m .34355 .29334 L p .35113 .29496 L .098 g
F P
0 g
s
.35743 .29769 m .36908 .2986 L .39735 .30486 L .098 g
F
.35113 .29496 m .39735 .30486 L .34604 .2935 L F
.34355 .29334 m .32723 .29226 L p .34575 .29559 L F P
0 g
s
.34575 .29559 m .39735 .30486 L .34355 .29334 L .098 g
F
.28278 .29603 m .32723 .29226 L .33514 .29278 L p .057 g
F P
0 g
s
.33514 .29278 m .36205 .29456 L p .31607 .29541 L .057 g
F P
0 g
s
.28278 .29603 m .33514 .29278 L .31607 .29541 L .057 g
F
.36205 .29456 m .32626 .29173 L .30652 .29368 L p F P
0 g
s
.30652 .29368 m .28278 .29603 L p .32613 .29522 L .057 g
F P
0 g
s
.32613 .29522 m .36205 .29456 L .30652 .29368 L .057 g
F
.24062 .30866 m .28278 .29603 L .28909 .2954 L p .037 g
F P
0 g
s
.28909 .2954 m .32626 .29173 L p .27392 .30208 L .037 g
F P
0 g
s
.24062 .30866 m .28909 .2954 L .27392 .30208 L .037 g
F
.32626 .29173 m .29226 .29664 L .27061 .30168 L p F P
0 g
s
.27061 .30168 m .24062 .30866 L p .28819 .29926 L .037 g
F P
0 g
s
.28819 .29926 m .32626 .29173 L .27061 .30168 L .037 g
F
.29226 .29664 m .26225 .30911 L .24202 .31611 L p .038 g
F P
0 g
s
.24062 .30866 m .20341 .32947 L .18654 .36963 L p .918 g
F P
0 g
s
.24202 .31611 m .20341 .32947 L p .2549 .31045 L .038 g
F P
0 g
s
.2549 .31045 m .29226 .29664 L .24202 .31611 L .038 g
F
.39606 .30786 m .40222 .30962 L .40565 .31184 L .154 g
F
.40492 .31154 m .43 .32197 L .39349 .3056 L F
.44358 .33544 m .45476 .34655 L p .43037 .32884 L .44 g
F P
0 g
s
.39735 .30486 m .44358 .33544 L .43037 .32884 L .44 g
F
.45476 .34655 m .43395 .33075 L .41019 .31394 L p F P
0 g
s
.42364 .32395 m .45476 .34655 L .41019 .31394 L .44 g
F
.38495 .30961 m .40579 .3135 L p .38732 .30849 L .492 g
F P
0 g
s
.43395 .33075 m .41141 .31954 L .37875 .30301 L p .38 g
F P
0 g
s
.37875 .30301 m .36205 .29456 L p .39353 .3104 L .38 g
F P
0 g
s
.39353 .3104 m .43395 .33075 L .37875 .30301 L .38 g
F
.39735 .30486 m .43 .32197 L .44358 .33544 L p .44 g
F P
0 g
s
.41019 .31394 m .39735 .30486 L p .41771 .31965 L .44 g
F P
0 g
s
.41771 .31965 m .42364 .32395 L .4188 .32035 L .44 g
F
.41771 .31965 m .4188 .32035 L .41019 .31394 L F
.46046 .35526 m .45476 .34655 L p .46681 .36496 L .616 g
F P
0 g
s
.46681 .36496 m .48714 .39602 L .46046 .35526 L .616 g
F
.48618 .38763 m .49059 .40127 L p .48094 .38343 L .636 g
F P
0 g
s
.47255 .36592 m .48618 .38763 L p .48094 .38343 L .636 g
F P
0 g
s
.48714 .39602 m .47255 .36592 L p .48298 .38769 L .636 g
F P
0 g
s
.4959 .41893 m .4963 .42024 L .49597 .41905 L .648 g
F
.49648 .42081 m .49635 .42034 L p .4963 .42024 L F P
0 g
s
.49635 .42034 m .49597 .41905 L p .4963 .42024 L .648 g
F P
0 g
s
.49362 .41157 m .49502 .4161 L .49454 .41438 L .648 g
F
.49362 .41157 m .49454 .41438 L .49288 .40836 L F
.49502 .4161 m .4959 .41893 L .49488 .41527 L F
.49597 .41905 m .49488 .41527 L p .4959 .41893 L F P
0 g
s
.49536 .41806 m .49772 .43257 L p .49556 .41838 L .047 g
F P
0 g
s
.49488 .41527 m .49288 .40836 L p .49502 .4161 L .648 g
F P
0 g
s
.39215 .30623 m .40492 .31154 L .39349 .3056 L .154 g
F
.41236 .32427 m .419 .32665 L p .41908 .32583 L .425 g
F P
0 g
s
.36236 .3054 m .38495 .30961 L p .38732 .30849 L .492 g
F P
0 g
s
.31591 .30747 m .35563 .3057 L p .35118 .30114 L .7 g
F P
0 g
s
.35563 .3057 m .36236 .3054 L p .35118 .30114 L .7 g
F P
0 g
s
.49254 .56548 m .49276 .57973 L .52012 .50879 L p .005 g
F P
0 g
s
.49315 .58819 m .49366 .59016 L .49738 .5774 L p .879 g
F P
0 g
s
.52012 .50879 m .52123 .50144 L .50932 .52802 L p .005 g
F P
0 g
s
.52012 .50879 m .50932 .52802 L .50789 .53392 L .005 g
F
.50567 .54625 m .50101 .55834 L p .50546 .54667 L .687 g
F P
0 g
s
.50101 .55834 m .49276 .57973 L p .50252 .55425 L .687 g
F P
0 g
s
.5183 .51305 m .50101 .55834 L .50252 .55425 L .687 g
F
.49276 .57973 m .49315 .58819 L .49939 .56953 L p F P
0 g
s
.5183 .51305 m .52012 .50879 L .50567 .54625 L p .687 g
F P
0 g
s
.5183 .51305 m .50567 .54625 L .50546 .54667 L .687 g
F
.49939 .56953 m .5183 .51305 L p .50528 .54705 L F P
0 g
s
.50528 .54705 m .49276 .57973 L .49939 .56953 L .687 g
F
.51589 .5139 m .5183 .51305 L .49828 .57285 L p .879 g
F P
0 g
s
.49828 .57285 m .49315 .58819 L p .49741 .57428 L .879 g
F P
0 g
s
.51589 .5139 m .49828 .57285 L .49741 .57428 L .879 g
F
.43063 .64459 m .43805 .6487 L .49366 .59016 L p 0 g
F P
s
.49388 .5858 m .49315 .58819 L .49425 .58538 L .879 g
F
.49388 .5858 m .49425 .58538 L .49738 .5774 L F
.50089 .5629 m .49388 .5858 L .49738 .5774 L F
.49366 .59016 m .49315 .58819 L .45636 .62137 L p 0 g
F P
s
.45636 .62137 m .43063 .64459 L p .45976 .61944 L F P
s
.49366 .59016 m .45636 .62137 L .45976 .61944 L F
.45836 .62732 m .44912 .63704 L p .46097 .62348 L .873 g
F P
0 g
s
.45328 .63154 m .44912 .63704 L .44688 .63875 L .873 g
F
.49429 .58534 m .45477 .63058 L .45328 .63154 L F
.45477 .63058 m .44912 .63704 L .45328 .63154 L F
.36344 .68129 m .38351 .67755 L .42575 .65483 L p .68 g
F P
0 g
s
.347 .67383 m .36344 .68129 L .43805 .6487 L p .053 g
F P
0 g
s
.43805 .6487 m .43063 .64459 L .347 .67383 L p .053 g
F P
0 g
s
.44912 .63704 m .43805 .6487 L p .44688 .63875 L .873 g
F P
0 g
s
.49429 .58534 m .49366 .59016 L .45836 .62732 L p .873 g
F P
0 g
s
.49429 .58534 m .45836 .62732 L .46097 .62348 L .873 g
F
.46838 .61452 m .43805 .6487 L .47056 .61328 L F
.43805 .6487 m .44704 .64337 L .47922 .60385 L p F P
0 g
s
.44704 .64337 m .43805 .6487 L .37639 .67563 L p .68 g
F P
0 g
s
.37639 .67563 m .36344 .68129 L p .37498 .67605 L .68 g
F P
0 g
s
.44704 .64337 m .37639 .67563 L .37498 .67605 L .68 g
F
.36344 .68129 m .347 .67383 L .26223 .67275 L p .115 g
F P
0 g
s
.26223 .67275 m .28793 .68457 L .36344 .68129 L p .115 g
F P
0 g
s
.4316 .65037 m .36344 .68129 L .42575 .65483 L .68 g
F
.38351 .67755 m .36344 .68129 L .28793 .68457 L p .578 g
F P
0 g
s
.28793 .68457 m .31937 .68444 L .3624 .67981 L p .578 g
F P
0 g
s
.3624 .67981 m .38351 .67755 L p .36558 .67886 L .578 g
F P
0 g
s
.36558 .67886 m .28793 .68457 L .3624 .67981 L .578 g
F
.38351 .67755 m .4059 .66244 L .42459 .65009 L p .894 g
F P
0 g
s
.42459 .65009 m .45703 .62864 L p .43173 .64547 L .894 g
F P
0 g
s
.43173 .64547 m .38351 .67755 L .42459 .65009 L .894 g
F
.44704 .64337 m .45703 .62864 L .47911 .59677 L p .932 g
F P
0 g
s
.47911 .59677 m .49497 .57387 L p .47447 .6036 L .932 g
F P
0 g
s
.47447 .6036 m .44704 .64337 L .47911 .59677 L .932 g
F
.42575 .65483 m .44704 .64337 L p .4316 .65037 L .68 g
F P
0 g
s
.48229 .59886 m .46838 .61452 L .47922 .60385 L .873 g
F
.46838 .61452 m .47056 .61328 L .47922 .60385 L F
.47922 .60385 m .49429 .58534 L p .48229 .59886 L F P
0 g
s
.49366 .59016 m .49429 .58534 L .50258 .55254 L p .922 g
F P
0 g
s
.50932 .52802 m .49254 .56548 L p .50789 .53392 L .005 g
F P
0 g
s
.50258 .55254 m .51303 .51123 L p .50321 .55126 L .922 g
F P
0 g
s
.50321 .55126 m .49366 .59016 L .50258 .55254 L .922 g
F
.49738 .5774 m .51013 .53365 L p .50089 .5629 L .879 g
F P
0 g
s
.51013 .53365 m .51589 .5139 L p .50844 .53824 L .879 g
F P
0 g
s
.50844 .53824 m .50089 .5629 L .51013 .53365 L .879 g
F
.15981 .58156 m .17301 .61634 L .24382 .65028 L p .569 g
F P
0 g
s
.26223 .67275 m .24382 .65028 L .17301 .61634 L p .348 g
F P
0 g
s
.24382 .65028 m .26223 .67275 L .347 .67383 L p .332 g
F P
0 g
s
.15552 .40487 m .18654 .36963 L .20341 .32947 L p .908 g
F P
0 g
s
.20341 .32947 m .17358 .35728 L .15552 .40487 L p .908 g
F P
0 g
s
.15552 .40487 m .17358 .35728 L p .13637 .43958 L .8 g
F P
0 g
s
.13557 .44181 m .15552 .40487 L p .13637 .43958 L .8 g
F P
0 g
s
.17358 .35728 m .15323 .39044 L .13427 .44423 L p .8 g
F P
0 g
s
.14397 .42688 m .12438 .48531 L p .14926 .40352 L F P
s
.15323 .39044 m .15152 .39717 L p .14926 .40352 L F P
s
.15152 .39717 m .14397 .42688 L p .14926 .40352 L F P
s
.13427 .44423 m .13557 .44181 L p .13637 .43958 L .8 g
F P
0 g
s
.58154 .38365 m .60292 .39434 L .66444 .35421 L p .645 g
F P
0 g
s
.66444 .35421 m .63041 .34422 L .58154 .38365 L p .645 g
F P
0 g
s
.69641 .37095 m .66444 .35421 L .60292 .39434 L p .621 g
F P
0 g
s
.63041 .34422 m .66444 .35421 L .71196 .30336 L p .637 g
F P
0 g
s
.66444 .35421 m .69641 .37095 L .75382 .31737 L p .602 g
F P
0 g
s
.75382 .31737 m .71196 .30336 L .66444 .35421 L p .602 g
F P
0 g
s
.60292 .39434 m .6229 .40959 L .69641 .37095 L p .621 g
F P
0 g
s
.72435 .39339 m .69641 .37095 L .6229 .40959 L p .595 g
F P
0 g
s
.69641 .37095 m .72435 .39339 L .79072 .33934 L p .562 g
F P
0 g
s
.79072 .33934 m .75382 .31737 L .69641 .37095 L p .562 g
F P
0 g
s
.12438 .48531 m .12682 .52546 L .15981 .58156 L p .738 g
F P
0 g
s
.17301 .61634 m .15981 .58156 L .12682 .52546 L p .487 g
F P
0 g
s
.49638 .39278 m .49575 .39918 L .50662 .41433 L p .42 g
F P
0 g
s
.50968 .42064 m .50662 .41433 L .50173 .40752 L p .049 g
F P
0 g
s
.50968 .42064 m .50173 .40752 L .50198 .40879 L .049 g
F
.50662 .41433 m .50375 .41124 L .50132 .40517 L p .42 g
F P
0 g
s
.50132 .40517 m .49638 .39278 L p .50257 .40581 L .42 g
F P
0 g
s
.50662 .41433 m .50132 .40517 L .50257 .40581 L .42 g
F
.49692 .393 m .49638 .39278 L .50121 .40488 L p .553 g
F P
0 g
s
.49735 .4 m .49692 .393 L .49955 .40425 L p .646 g
F P
0 g
s
.50173 .40752 m .49575 .39918 L p .50198 .40879 L .049 g
F P
0 g
s
.49955 .40425 m .50127 .41161 L p .49862 .40378 L .646 g
F P
0 g
s
.49862 .40378 m .49735 .4 L .49955 .40425 L .646 g
F
.50121 .40488 m .50375 .41124 L p .50103 .40396 L .553 g
F P
0 g
s
.50103 .40396 m .49692 .393 L .50121 .40488 L .553 g
F
.56012 .37823 m .58154 .38365 L .63041 .34422 L p .666 g
F P
0 g
s
.63041 .34422 m .59644 .34165 L .56012 .37823 L p .666 g
F P
0 g
s
.59644 .34165 m .63041 .34422 L .66777 .29808 L p .666 g
F P
0 g
s
.71196 .30336 m .66777 .29808 L .63041 .34422 L p .637 g
F P
0 g
s
.6229 .40959 m .64021 .42842 L .72435 .39339 L p .595 g
F P
0 g
s
.74647 .42019 m .72435 .39339 L .64021 .42842 L p .567 g
F P
0 g
s
.72435 .39339 m .74647 .42019 L .82027 .36801 L p .516 g
F P
0 g
s
.82027 .36801 m .79072 .33934 L .72435 .39339 L p .516 g
F P
0 g
s
.5423 .42026 m .55077 .42975 L .6229 .40959 L p .652 g
F P
0 g
s
.6229 .40959 m .60292 .39434 L .5423 .42026 L p .652 g
F P
0 g
s
.64021 .42842 m .6229 .40959 L .55077 .42975 L p .638 g
F P
0 g
s
.55077 .42975 m .55807 .44091 L .64021 .42842 L p .638 g
F P
0 g
s
.65377 .44963 m .64021 .42842 L .55807 .44091 L p .622 g
F P
0 g
s
.64021 .42842 m .65377 .44963 L .74647 .42019 L p .567 g
F P
0 g
s
.53322 .41308 m .5423 .42026 L .60292 .39434 L p .666 g
F P
0 g
s
.60292 .39434 m .58154 .38365 L .53322 .41308 L p .666 g
F P
0 g
s
.55807 .44091 m .56372 .45301 L .65377 .44963 L p .622 g
F P
0 g
s
.66269 .47189 m .65377 .44963 L .56372 .45301 L p .607 g
F P
0 g
s
.65377 .44963 m .66269 .47189 L .7613 .44968 L p .537 g
F P
0 g
s
.7613 .44968 m .74647 .42019 L .65377 .44963 L p .537 g
F P
0 g
s
.66777 .29808 m .62394 .30185 L .59644 .34165 L p .666 g
F P
0 g
s
.62394 .30185 m .66777 .29808 L .68404 .25172 L p .684 g
F P
0 g
s
.66777 .29808 m .71196 .30336 L .73333 .24934 L p .646 g
F P
0 g
s
.73333 .24934 m .68404 .25172 L .66777 .29808 L p .646 g
F P
0 g
s
.54005 .37853 m .56012 .37823 L .59644 .34165 L p .683 g
F P
0 g
s
.59644 .34165 m .56468 .34673 L .54005 .37853 L p .683 g
F P
0 g
s
.56468 .34673 m .59644 .34165 L .62394 .30185 L p .69 g
F P
0 g
s
.71196 .30336 m .75382 .31737 L .78043 .25726 L p .598 g
F P
0 g
s
.78043 .25726 m .73333 .24934 L .71196 .30336 L p .598 g
F P
0 g
s
.74647 .42019 m .7613 .44968 L .84041 .40167 L p .463 g
F P
0 g
s
.84041 .40167 m .82027 .36801 L .74647 .42019 L p .463 g
F P
0 g
s
.17301 .61634 m .19689 .64319 L .26223 .67275 L p .348 g
F P
0 g
s
.28793 .68457 m .26223 .67275 L .19689 .64319 L p .09 g
F P
0 g
s
.62394 .30185 m .58319 .31447 L .56468 .34673 L p .69 g
F P
0 g
s
.58319 .31447 m .62394 .30185 L .63555 .26412 L p .713 g
F P
0 g
s
.68404 .25172 m .63555 .26412 L .62394 .30185 L p .684 g
F P
0 g
s
.52411 .40869 m .53322 .41308 L .58154 .38365 L p .677 g
F P
0 g
s
.58154 .38365 m .56012 .37823 L .52411 .40869 L p .677 g
F P
0 g
s
.12438 .48531 m .13427 .44423 L .15323 .39044 L p F P
s
.56372 .45301 m .56736 .46527 L .66269 .47189 L p .607 g
F P
0 g
s
.66634 .49378 m .66269 .47189 L .56736 .46527 L p .592 g
F P
0 g
s
.66269 .47189 m .66634 .49378 L .76776 .48002 L p .505 g
F P
0 g
s
.76776 .48002 m .7613 .44968 L .66269 .47189 L p .505 g
F P
0 g
s
.75382 .31737 m .79072 .33934 L .82235 .27518 L p .537 g
F P
0 g
s
.82235 .27518 m .78043 .25726 L .75382 .31737 L p .537 g
F P
0 g
s
.48716 .37781 m .48606 .37611 L p .48733 .37815 L .046 g
F P
0 g
s
.49638 .39278 m .49692 .393 L .48716 .37781 L p .046 g
F P
0 g
s
.49638 .39278 m .48716 .37781 L .48733 .37815 L .046 g
F
.49232 .38631 m .48712 .37782 L p .49692 .393 L .51 g
F P
0 g
s
.48712 .37782 m .48606 .37611 L .48716 .37781 L p .51 g
F P
0 g
s
.48716 .37781 m .49692 .393 L p .48712 .37782 L .51 g
F P
0 g
s
.4946 .39911 m .4929 .39465 L p .48682 .37793 L .415 g
F P
0 g
s
.48606 .37611 m .48712 .37782 L p .48563 .37467 L .043 g
F P
0 g
s
.48712 .37782 m .49232 .38631 L .48563 .37467 L p .043 g
F P
0 g
s
.48573 .37494 m .48683 .37793 L .48682 .37793 L .415 g
F
.48683 .37793 m .4946 .39911 L .48682 .37793 L F
.48563 .37467 m .48573 .37494 L p .48573 .37494 L F P
0 g
s
.48573 .37494 m .4946 .39911 L p .48573 .37494 L .415 g
F P
0 g
s
.49772 .43257 m .518 .45357 L .51988 .46677 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .51988 .46677 L .5211 .4797 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .51556 .44094 L .518 .45357 L closepath p .751 g
F P
0 g
s
.53721 .35926 m .56468 .34673 L .58319 .31447 L p .704 g
F P
0 g
s
.58319 .31447 m .54808 .33526 L .53721 .35926 L p .704 g
F P
0 g
s
.54808 .33526 m .58319 .31447 L .59079 .28573 L p .732 g
F P
0 g
s
.63555 .26412 m .59079 .28573 L .58319 .31447 L p .713 g
F P
0 g
s
.52267 .3846 m .54005 .37853 L .56468 .34673 L p .692 g
F P
0 g
s
.56468 .34673 m .53721 .35926 L .52267 .3846 L p .692 g
F P
0 g
s
.51559 .40741 m .52411 .40869 L .56012 .37823 L p .685 g
F P
0 g
s
.56012 .37823 m .54005 .37853 L .51559 .40741 L p .685 g
F P
0 g
s
.49772 .43257 m .5211 .4797 L .52156 .49152 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .51272 .42972 L .51556 .44094 L closepath p .751 g
F P
0 g
s
.7613 .44968 m .76776 .48002 L .84958 .43822 L p .399 g
F P
0 g
s
.84958 .43822 m .84041 .40167 L .7613 .44968 L p .399 g
F P
0 g
s
.56736 .46527 m .56873 .47689 L .66634 .49378 L p .592 g
F P
0 g
s
.66442 .51386 m .66634 .49378 L .56873 .47689 L p .577 g
F P
0 g
s
.66634 .49378 m .66442 .51386 L .76525 .50922 L p .471 g
F P
0 g
s
.76525 .50922 m .76776 .48002 L .66634 .49378 L p .471 g
F P
0 g
s
.12682 .52546 m .14182 .56194 L .17301 .61634 L p .487 g
F P
0 g
s
.19689 .64319 m .17301 .61634 L .14182 .56194 L p .105 g
F P
0 g
s
.59079 .28573 m .63555 .26412 L .62789 .23526 L p .754 g
F P
0 g
s
.63555 .26412 m .68404 .25172 L .67431 .21402 L p .725 g
F P
0 g
s
.67431 .21402 m .62789 .23526 L .63555 .26412 L p .725 g
F P
0 g
s
.20341 .32947 m .24062 .30866 L .29226 .29664 L p .038 g
F P
0 g
s
.49772 .43257 m .52156 .49152 L .52123 .50144 L closepath p .751 g
F P
0 g
s
.59079 .28573 m .55244 .31528 L .54808 .33526 L p .732 g
F P
0 g
s
.55244 .31528 m .59079 .28573 L .58533 .26531 L p .774 g
F P
0 g
s
.62789 .23526 m .58533 .26531 L .59079 .28573 L p .754 g
F P
0 g
s
.49772 .43257 m .50968 .42064 L .51272 .42972 L closepath p .751 g
F P
0 g
s
.79072 .33934 m .82027 .36801 L .85625 .30224 L p .457 g
F P
0 g
s
.85625 .30224 m .82235 .27518 L .79072 .33934 L p .457 g
F P
0 g
s
.68404 .25172 m .73333 .24934 L .72188 .20301 L p .682 g
F P
0 g
s
.72188 .20301 m .67431 .21402 L .68404 .25172 L p .682 g
F P
0 g
s
.51592 .37858 m .53721 .35926 L .54808 .33526 L p .703 g
F P
0 g
s
.54808 .33526 m .52093 .36304 L .51592 .37858 L p .703 g
F P
0 g
s
.52093 .36304 m .54808 .33526 L .55244 .31528 L p .734 g
F P
0 g
s
.55807 .44091 m .55077 .42975 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.56372 .45301 m .55807 .44091 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.12682 .52546 m .12438 .48531 L .14397 .42688 L p .038 g
F P
0 g
s
.50823 .40937 m .51559 .40741 L .54005 .37853 L p .689 g
F P
0 g
s
.54005 .37853 m .52267 .3846 L .50823 .40937 L p .689 g
F P
0 g
s
.55077 .42975 m .5423 .42026 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .52123 .50144 L .52012 .50879 L closepath p .751 g
F P
0 g
s
.56736 .46527 m .56372 .45301 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.4929 .39465 m .47904 .36559 L .48563 .37467 L p .415 g
F P
0 g
s
.49772 .43257 m .50662 .41433 L .50968 .42064 L closepath p .751 g
F P
0 g
s
.55244 .31528 m .5229 .35106 L .52093 .36304 L p .734 g
F P
0 g
s
.5229 .35106 m .55244 .31528 L .54909 .30234 L p .78 g
F P
0 g
s
.58533 .26531 m .54909 .30234 L .55244 .31528 L p .774 g
F P
0 g
s
.51303 .51123 m .51589 .5139 L .49366 .59016 L p .922 g
F P
0 g
s
.47428 .35985 m .49168 .39245 L .49126 .39283 L p .5 g
F P
0 g
s
.50921 .39615 m .52267 .3846 L .53721 .35926 L p .688 g
F P
0 g
s
.53721 .35926 m .51592 .37858 L .50921 .39615 L p .688 g
F P
0 g
s
.5423 .42026 m .53322 .41308 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.56873 .47689 m .56774 .48708 L .66442 .51386 L p .577 g
F P
0 g
s
.65696 .53077 m .66442 .51386 L .56774 .48708 L p .564 g
F P
0 g
s
.66442 .51386 m .65696 .53077 L .75373 .53528 L p .435 g
F P
0 g
s
.75373 .53528 m .76525 .50922 L .66442 .51386 L p .435 g
F P
0 g
s
.56873 .47689 m .56736 .46527 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.4946 .39911 m .48563 .37467 L .49232 .38631 L p .536 g
F P
0 g
s
.17358 .35728 m .20341 .32947 L .26225 .30911 L p .06 g
F P
0 g
s
.49692 .393 m .49735 .4 L .49232 .38631 L p .51 g
F P
0 g
s
.50375 .41124 m .50127 .41161 L .49692 .393 L p .553 g
F P
0 g
s
.76776 .48002 m .76525 .50922 L .84684 .47527 L p .322 g
F P
0 g
s
.84684 .47527 m .84958 .43822 L .76776 .48002 L p .322 g
F P
0 g
s
.73333 .24934 m .78043 .25726 L .76769 .20314 L p .619 g
F P
0 g
s
.76769 .20314 m .72188 .20301 L .73333 .24934 L p .619 g
F P
0 g
s
.54909 .30234 m .58533 .26531 L .56805 .25786 L p .821 g
F P
0 g
s
.58533 .26531 m .62789 .23526 L .6025 .22221 L p .809 g
F P
0 g
s
.6025 .22221 m .56805 .25786 L .58533 .26531 L p .809 g
F P
0 g
s
.49176 .39597 m .4745 .36227 L .47262 .35843 L p .55 g
F P
0 g
s
.47262 .35843 m .49126 .39283 L .49176 .39597 L p .55 g
F P
0 g
s
.19689 .64319 m .23018 .65992 L .28793 .68457 L p .09 g
F P
0 g
s
.31937 .68444 m .28793 .68457 L .23018 .65992 L p .588 g
F P
0 g
s
.54909 .30234 m .52132 .34422 L .5229 .35106 L p .78 g
F P
0 g
s
.52132 .34422 m .54909 .30234 L .53886 .29919 L p .828 g
F P
0 g
s
.56805 .25786 m .53886 .29919 L .54909 .30234 L p .821 g
F P
0 g
s
.53322 .41308 m .52411 .40869 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.56774 .48708 m .56873 .47689 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .52012 .50879 L .5183 .51305 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .50375 .41124 L .50662 .41433 L closepath p .751 g
F P
0 g
s
.82027 .36801 m .84041 .40167 L .87965 .33691 L p .349 g
F P
0 g
s
.87965 .33691 m .85625 .30224 L .82027 .36801 L p .349 g
F P
0 g
s
.62789 .23526 m .67431 .21402 L .64027 .1944 L p .789 g
F P
0 g
s
.64027 .1944 m .6025 .22221 L .62789 .23526 L p .789 g
F P
0 g
s
.45703 .62864 m .44704 .64337 L .38351 .67755 L p .894 g
F P
0 g
s
.49497 .57387 m .49429 .58534 L .44704 .64337 L p .932 g
F P
0 g
s
.36205 .29456 m .39735 .30486 L .43395 .33075 L p .38 g
F P
0 g
s
.49232 .38631 m .49633 .40551 L .4946 .39911 L p .536 g
F P
0 g
s
.49633 .40551 m .49232 .38631 L .49735 .4 L p .66 g
F P
0 g
s
.49772 .43257 m .50127 .41161 L .50375 .41124 L closepath p .751 g
F P
0 g
s
.50254 .41445 m .50823 .40937 L .52267 .3846 L p .685 g
F P
0 g
s
.52267 .3846 m .50921 .39615 L .50254 .41445 L p .685 g
F P
0 g
s
.50367 .39616 m .52093 .36304 L .5229 .35106 L p .699 g
F P
0 g
s
.5229 .35106 m .50418 .39097 L .50367 .39616 L p .699 g
F P
0 g
s
.50418 .39097 m .5229 .35106 L .52132 .34422 L p .763 g
F P
0 g
s
.52411 .40869 m .51559 .40741 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.50367 .39616 m .50418 .39097 L .50275 .40016 L p .801 g
F P
0 g
s
.50239 .40355 m .5029 .40259 L p .50289 .40244 L .667 g
F P
0 g
s
.52093 .36304 m .50367 .39616 L .50239 .40355 L p .667 g
F P
0 g
s
.50921 .39615 m .51592 .37858 L p .50182 .40991 L .662 g
F P
0 g
s
.49935 .41548 m .49735 .4 L p .50057 .40956 L .646 g
F P
0 g
s
.50127 .41161 m .50014 .41388 L p .50057 .40956 L .646 g
F P
0 g
s
.50014 .41388 m .49935 .41548 L p .50057 .40956 L .646 g
F P
0 g
s
.50182 .40991 m .50102 .41168 L .50187 .41017 L .662 g
F
.50066 .41249 m .50098 .41187 L p .50102 .41168 L F P
0 g
s
.50098 .41187 m .50187 .41017 L p .50102 .41168 L .662 g
F P
0 g
s
.51592 .37858 m .50239 .40355 L .50066 .41249 L p .662 g
F P
0 g
s
.50239 .40355 m .50367 .39616 L .5016 .40882 L p F P
s
.49772 .43257 m .49935 .41548 L .50127 .41161 L closepath p .751 g
F P
0 g
s
.50041 .41863 m .49953 .42087 L .50051 .41878 L .671 g
F
.49946 .42124 m .50254 .41445 L p .49953 .42087 L F P
0 g
s
.50921 .39615 m .50066 .41249 L .49895 .42236 L p .671 g
F P
0 g
s
.50254 .41445 m .49895 .42236 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49921 .42331 m .50239 .40355 L p .49917 .4237 L F P
s
.50066 .41249 m .50239 .40355 L .49988 .41914 L p .031 g
F P
0 g
s
.49953 .42087 m .49913 .4219 L .49946 .42124 L .671 g
F
.49988 .41914 m .49772 .43257 L .49946 .42069 L p .031 g
F P
0 g
s
.49946 .42069 m .50066 .41249 L p .49988 .41914 L .031 g
F P
0 g
s
.49895 .42236 m .49909 .42206 L p .49913 .4219 L .671 g
F P
0 g
s
.49909 .42206 m .49946 .42124 L p .49913 .4219 L .671 g
F P
0 g
s
.50149 .40826 m .49998 .41799 L p .5016 .40882 L .801 g
F P
0 g
s
.50008 .41813 m .49917 .4237 L p .49997 .41868 L F P
s
.49917 .4237 m .49772 .43257 L .49921 .42331 L p F P
s
.49998 .41799 m .49772 .43257 L .50008 .41813 L p .801 g
F P
0 g
s
.5016 .40882 m .50008 .41813 L p .50133 .41016 L F P
s
.50008 .41813 m .49997 .41868 L .50133 .41016 L F
.50008 .41813 m .5016 .40882 L p .49998 .41799 L .801 g
F P
0 g
s
.50254 .41445 m .50921 .39615 L p .50041 .41863 L .671 g
F P
0 g
s
.50041 .41863 m .50051 .41878 L .50254 .41445 L .671 g
F
.5016 .40882 m .50133 .41016 L .50239 .40355 L 0 g
F
.50187 .41017 m .50921 .39615 L p .50182 .40991 L .662 g
F P
0 g
s
.50275 .40016 m .50149 .40826 L p .50282 .40135 L .801 g
F P
0 g
s
.5016 .40882 m .50282 .40135 L p .50149 .40826 L .801 g
F P
0 g
s
.50282 .40135 m .50367 .39616 L p .50275 .40016 L .801 g
F P
0 g
s
.51592 .37858 m .52093 .36304 L p .50289 .40244 L .667 g
F P
0 g
s
.5029 .40259 m .51592 .37858 L p .50289 .40244 L .667 g
F P
0 g
s
.56442 .49516 m .56774 .48708 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.56774 .48708 m .56442 .49516 L .65696 .53077 L p .564 g
F P
0 g
s
.32626 .29173 m .36205 .29456 L .41141 .31954 L p .344 g
F P
0 g
s
.4059 .66244 m .38351 .67755 L .31937 .68444 L p .82 g
F P
0 g
s
.53886 .29919 m .51658 .3438 L .52132 .34422 L p .828 g
F P
0 g
s
.51658 .3438 m .53886 .29919 L .52441 .30754 L p .854 g
F P
0 g
s
.53886 .29919 m .56805 .25786 L .54342 .26656 L p .855 g
F P
0 g
s
.54342 .26656 m .52441 .30754 L .53886 .29919 L p .855 g
F P
0 g
s
.64436 .5433 m .65696 .53077 L .56442 .49516 L p .552 g
F P
0 g
s
.65696 .53077 m .64436 .5433 L .73378 .55634 L p .397 g
F P
0 g
s
.73378 .55634 m .75373 .53528 L .65696 .53077 L p .397 g
F P
0 g
s
.52132 .34422 m .50376 .38851 L .50418 .39097 L p .763 g
F P
0 g
s
.50376 .38851 m .52132 .34422 L .51658 .3438 L p .835 g
F P
0 g
s
.49772 .43257 m .5183 .51305 L .51589 .5139 L closepath p .751 g
F P
0 g
s
.14182 .56194 m .16874 .59206 L .19689 .64319 L p .105 g
F P
0 g
s
.23018 .65992 m .19689 .64319 L .16874 .59206 L p .397 g
F P
0 g
s
.56805 .25786 m .6025 .22221 L .56591 .22996 L p .859 g
F P
0 g
s
.56591 .22996 m .54342 .26656 L .56805 .25786 L p .859 g
F P
0 g
s
.78043 .25726 m .82235 .27518 L .80878 .21478 L p .521 g
F P
0 g
s
.80878 .21478 m .76769 .20314 L .78043 .25726 L p .521 g
F P
0 g
s
.49429 .58534 m .49497 .57387 L .50993 .50517 L p .895 g
F P
0 g
s
.50993 .50517 m .51303 .51123 L .49429 .58534 L p .895 g
F P
0 g
s
.26225 .30911 m .23828 .32848 L .17358 .35728 L p .06 g
F P
0 g
s
.15323 .39044 m .17358 .35728 L .23828 .32848 L p .106 g
F P
0 g
s
.51658 .3438 m .50254 .38915 L .50376 .38851 L p .835 g
F P
0 g
s
.50254 .38915 m .51658 .3438 L .50994 .35051 L p .86 g
F P
0 g
s
.52441 .30754 m .50994 .35051 L .51658 .3438 L p .854 g
F P
0 g
s
.76525 .50922 m .75373 .53528 L .83207 .51023 L p .228 g
F P
0 g
s
.83207 .51023 m .84684 .47527 L .76525 .50922 L p .228 g
F P
0 g
s
.14397 .42688 m .14673 .46422 L .12682 .52546 L p .038 g
F P
0 g
s
.14182 .56194 m .12682 .52546 L .14673 .46422 L p .156 g
F P
0 g
s
.51559 .40741 m .50823 .40937 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.67431 .21402 m .72188 .20301 L .67918 .1763 L p .752 g
F P
0 g
s
.67918 .1763 m .64027 .1944 L .67431 .21402 L p .752 g
F P
0 g
s
.55898 .50056 m .56442 .49516 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.56442 .49516 m .55898 .50056 L .64436 .5433 L p .552 g
F P
0 g
s
.49735 .4 m .49762 .41347 L .49633 .40551 L p .66 g
F P
0 g
s
.49762 .41347 m .49735 .4 L .49935 .41548 L p .709 g
F P
0 g
s
.41141 .31954 m .38858 .31368 L .32626 .29173 L p .344 g
F P
0 g
s
.29226 .29664 m .32626 .29173 L .38858 .31368 L p .335 g
F P
0 g
s
.50994 .35051 m .52441 .30754 L .50963 .32748 L p .85 g
F P
0 g
s
.52441 .30754 m .54342 .26656 L .51812 .29198 L p .863 g
F P
0 g
s
.51812 .29198 m .50963 .32748 L .52441 .30754 L p .863 g
F P
0 g
s
.50994 .35051 m .50084 .39301 L .50254 .38915 L p .86 g
F P
0 g
s
.50084 .39301 m .50994 .35051 L .50317 .36417 L p .844 g
F P
0 g
s
.50963 .32748 m .50317 .36417 L .50994 .35051 L p .85 g
F P
0 g
s
.49772 .43257 m .51589 .5139 L .51303 .51123 L closepath p .751 g
F P
0 g
s
.84041 .40167 m .84958 .43822 L .89059 .37706 L p .195 g
F P
0 g
s
.89059 .37706 m .87965 .33691 L .84041 .40167 L p .195 g
F P
0 g
s
.6025 .22221 m .64027 .1944 L .59061 .19998 L p .86 g
F P
0 g
s
.59061 .19998 m .56591 .22996 L .6025 .22221 L p .86 g
F P
0 g
s
.49633 .40551 m .49772 .43257 L .4946 .39911 L closepath p .577 g
F P
0 g
s
.50823 .40937 m .50254 .41445 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.62739 .55051 m .64436 .5433 L .55898 .50056 L p .544 g
F P
0 g
s
.64436 .5433 m .62739 .55051 L .70662 .57077 L p .358 g
F P
0 g
s
.70662 .57077 m .73378 .55634 L .64436 .5433 L p .358 g
F P
0 g
s
.55175 .50289 m .55898 .50056 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.55898 .50056 m .55175 .50289 L .62739 .55051 L p .544 g
F P
0 g
s
.45476 .34655 m .47255 .36592 L .48714 .39602 L p .616 g
F P
0 g
s
.43395 .33075 m .45476 .34655 L .47966 .3846 L p .593 g
F P
0 g
s
.54342 .26656 m .56591 .22996 L .52815 .25978 L p .882 g
F P
0 g
s
.52815 .25978 m .51812 .29198 L .54342 .26656 L p .882 g
F P
0 g
s
.50066 .41249 m .49772 .43257 L .49895 .42236 L closepath p .17 g
F P
0 g
s
.50376 .38851 m .49772 .43257 L .50418 .39097 L closepath p .805 g
F P
0 g
s
.50254 .38915 m .49772 .43257 L .50376 .38851 L closepath p .743 g
F P
0 g
s
.50317 .36417 m .49911 .39991 L .50084 .39301 L p .844 g
F P
0 g
s
.49911 .39991 m .50317 .36417 L .49809 .38373 L p .813 g
F P
0 g
s
.50317 .36417 m .50963 .32748 L .49852 .35731 L p .824 g
F P
0 g
s
.49852 .35731 m .49809 .38373 L .50317 .36417 L p .824 g
F P
0 g
s
.50963 .32748 m .51812 .29198 L .4991 .33168 L p .843 g
F P
0 g
s
.4991 .33168 m .49852 .35731 L .50963 .32748 L p .843 g
F P
0 g
s
.49935 .41548 m .49813 .42262 L .49762 .41347 L p .709 g
F P
0 g
s
.49772 .43257 m .49813 .42262 L .49935 .41548 L closepath p .751 g
F P
0 g
s
.50084 .39301 m .49772 .43257 L .50254 .38915 L closepath p .683 g
F P
0 g
s
.47966 .3846 m .47091 .37485 L .43395 .33075 L p .593 g
F P
0 g
s
.41141 .31954 m .43395 .33075 L .47091 .37485 L p .573 g
F P
0 g
s
.46729 .60531 m .45703 .62864 L .4059 .66244 L p .888 g
F P
0 g
s
.45703 .62864 m .46729 .60531 L .49567 .55645 L p .891 g
F P
0 g
s
.49567 .55645 m .49497 .57387 L .45703 .62864 L p .891 g
F P
0 g
s
.49762 .41347 m .49772 .43257 L .49633 .40551 L closepath p .526 g
F P
0 g
s
.23018 .65992 m .27078 .66497 L .31937 .68444 L p .588 g
F P
0 g
s
.35446 .67192 m .31937 .68444 L .27078 .66497 L p .748 g
F P
0 g
s
.31937 .68444 m .35446 .67192 L .4059 .66244 L p .82 g
F P
0 g
s
.38858 .31368 m .36693 .31362 L .29226 .29664 L p .335 g
F P
0 g
s
.26225 .30911 m .29226 .29664 L .36693 .31362 L p .349 g
F P
0 g
s
.49911 .39991 m .49772 .43257 L .50084 .39301 L closepath p .632 g
F P
0 g
s
.49772 .43257 m .51303 .51123 L .50993 .50517 L closepath p .751 g
F P
0 g
s
.49497 .57387 m .49567 .55645 L .5068 .49612 L p .854 g
F P
0 g
s
.5068 .49612 m .50993 .50517 L .49497 .57387 L p .854 g
F P
0 g
s
.23828 .32848 m .22211 .3536 L .15323 .39044 L p .106 g
F P
0 g
s
.14397 .42688 m .15323 .39044 L .22211 .3536 L p .213 g
F P
0 g
s
.54323 .50194 m .55175 .50289 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.55175 .50289 m .54323 .50194 L .60715 .55179 L p .538 g
F P
0 g
s
.60715 .55179 m .62739 .55051 L .55175 .50289 L p .538 g
F P
0 g
s
.82235 .27518 m .85625 .30224 L .84227 .23757 L p .355 g
F P
0 g
s
.84227 .23757 m .80878 .21478 L .82235 .27518 L p .355 g
F P
0 g
s
.75373 .53528 m .73378 .55634 L .806 .54056 L p .122 g
F P
0 g
s
.806 .54056 m .83207 .51023 L .75373 .53528 L p .122 g
F P
0 g
s
.49809 .38373 m .49781 .40935 L .49911 .39991 L p .813 g
F P
0 g
s
.49781 .40935 m .49809 .38373 L .49608 .40732 L p .774 g
F P
0 g
s
.49809 .38373 m .49852 .35731 L .49414 .39374 L p .784 g
F P
0 g
s
.49414 .39374 m .49608 .40732 L .49809 .38373 L p .784 g
F P
0 g
s
.49852 .35731 m .4991 .33168 L .4916 .38068 L p .802 g
F P
0 g
s
.4916 .38068 m .49414 .39374 L .49852 .35731 L p .802 g
F P
0 g
s
.49813 .42262 m .49772 .43257 L .49762 .41347 L closepath p .47 g
F P
0 g
s
.49781 .40935 m .49772 .43257 L .49911 .39991 L closepath p .584 g
F P
0 g
s
.51812 .29198 m .52815 .25978 L .49977 .30845 L p .87 g
F P
0 g
s
.49977 .30845 m .4991 .33168 L .51812 .29198 L p .87 g
F P
0 g
s
.72188 .20301 m .76769 .20314 L .71684 .16936 L p .676 g
F P
0 g
s
.71684 .16936 m .67918 .1763 L .72188 .20301 L p .676 g
F P
0 g
s
.47091 .37485 m .46146 .3674 L .41141 .31954 L p .573 g
F P
0 g
s
.38858 .31368 m .41141 .31954 L .46146 .3674 L p .559 g
F P
0 g
s
.42901 .6367 m .4059 .66244 L .35446 .67192 L p .85 g
F P
0 g
s
.4059 .66244 m .42901 .6367 L .46729 .60531 L p .888 g
F P
0 g
s
.16874 .59206 m .206 .61346 L .23018 .65992 L p .397 g
F P
0 g
s
.27078 .66497 m .23018 .65992 L .206 .61346 L p .593 g
F P
0 g
s
.48714 .39602 m .49288 .40836 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.47966 .3846 m .48714 .39602 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.4973 .42055 m .49772 .43257 L .49781 .40935 L closepath p .536 g
F P
0 g
s
.49608 .40732 m .4973 .42055 L .49781 .40935 L p .774 g
F P
0 g
s
.62739 .55051 m .60715 .55179 L .67404 .57741 L p .322 g
F P
0 g
s
.67404 .57741 m .70662 .57077 L .62739 .55051 L p .322 g
F P
0 g
s
.47091 .37485 m .47966 .3846 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49288 .40836 m .49648 .42081 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.4991 .33168 m .49977 .30845 L .48863 .36899 L p .827 g
F P
0 g
s
.48863 .36899 m .4916 .38068 L .4991 .33168 L p .827 g
F P
0 g
s
.56591 .22996 m .59061 .19998 L .53915 .23291 L p .905 g
F P
0 g
s
.53915 .23291 m .52815 .25978 L .56591 .22996 L p .905 g
F P
0 g
s
.14673 .46422 m .16166 .49983 L .14182 .56194 L p .156 g
F P
0 g
s
.16874 .59206 m .14182 .56194 L .16166 .49983 L p .364 g
F P
0 g
s
.49772 .43257 m .50993 .50517 L .5068 .49612 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .4973 .42055 L .49608 .40732 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .49608 .40732 L .49414 .39374 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .49414 .39374 L .4916 .38068 L closepath p .751 g
F P
0 g
s
.53398 .49775 m .54323 .50194 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.54323 .50194 m .53398 .49775 L .58505 .54694 L p .539 g
F P
0 g
s
.58505 .54694 m .60715 .55179 L .54323 .50194 L p .539 g
F P
0 g
s
.84958 .43822 m .84684 .47527 L .88788 .42005 L p .04 g
F P
0 g
s
.88788 .42005 m .89059 .37706 L .84958 .43822 L p .04 g
F P
0 g
s
.46146 .3674 m .47091 .37485 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.64027 .1944 m .67918 .1763 L .6161 .17864 L p .842 g
F P
0 g
s
.6161 .17864 m .59061 .19998 L .64027 .1944 L p .842 g
F P
0 g
s
.49772 .43257 m .4916 .38068 L .48863 .36899 L closepath p .751 g
F P
0 g
s
.46146 .3674 m .45193 .36278 L .38858 .31368 L p .559 g
F P
0 g
s
.36693 .31362 m .38858 .31368 L .45193 .36278 L p .551 g
F P
0 g
s
.36693 .31362 m .34792 .31945 L .26225 .30911 L p .349 g
F P
0 g
s
.23828 .32848 m .26225 .30911 L .34792 .31945 L p .376 g
F P
0 g
s
.49633 .53423 m .49567 .55645 L .46729 .60531 L p .843 g
F P
0 g
s
.49567 .55645 m .49633 .53423 L .50386 .48468 L p .814 g
F P
0 g
s
.50386 .48468 m .5068 .49612 L .49567 .55645 L p .814 g
F P
0 g
s
.49772 .43257 m .5068 .49612 L .50386 .48468 L closepath p .751 g
F P
0 g
s
.47706 .57492 m .46729 .60531 L .42901 .6367 L p .85 g
F P
0 g
s
.46729 .60531 m .47706 .57492 L .49633 .53423 L p .843 g
F P
0 g
s
.45193 .36278 m .46146 .3674 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.52466 .49058 m .53398 .49775 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .50386 .48468 L .50132 .47166 L closepath p .751 g
F P
0 g
s
.49896 .44438 m .50259 .45697 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.50259 .45697 m .49896 .44438 L .50069 .45502 L p .652 g
F P
0 g
s
.49772 .43257 m .50069 .45502 L .49896 .44438 L closepath p .331 g
F P
0 g
s
.50431 .47318 m .50378 .46985 L .50323 .46642 L p .498 g
F P
0 g
s
.50088 .45521 m .50185 .45783 L .50154 .45589 L .652 g
F
.50102 .4559 m .50457 .46533 L p .50088 .45521 L F P
0 g
s
.50457 .46533 m .50945 .47826 L p .50458 .46519 L .652 g
F P
0 g
s
.50457 .46533 m .50458 .46519 L .50088 .45521 L .652 g
F
.50945 .47826 m .50069 .45502 L .50246 .4637 L p .47 g
F P
0 g
s
.51647 .49475 m .50945 .47826 L p .50504 .46908 L .47 g
F P
0 g
s
.50246 .4637 m .50298 .46487 L p .50297 .46477 L .47 g
F P
0 g
s
.51647 .49475 m .50246 .4637 L .50378 .46985 L p .356 g
F P
0 g
s
.50323 .46642 m .50378 .46985 L p .50193 .46024 L .047 g
F P
0 g
s
.50323 .46642 m .50306 .46539 L p .50413 .47204 L .498 g
F P
0 g
s
.50193 .46024 m .49772 .43257 L .50323 .46642 L p .047 g
F P
0 g
s
.50306 .46539 m .49772 .43257 L .50313 .46592 L p .498 g
F P
0 g
s
.50313 .46592 m .50413 .47204 L p .50306 .46539 L .498 g
F P
0 g
s
.50297 .46477 m .50443 .46806 L .50504 .46908 L .47 g
F
.50298 .46487 m .50443 .46806 L p .50297 .46477 L F P
0 g
s
.50443 .46806 m .51647 .49475 L p .50504 .46908 L .47 g
F P
0 g
s
.50389 .47363 m .50403 .47348 L p .50327 .46857 L .683 g
F P
0 g
s
.50322 .46867 m .50389 .47363 L .50327 .46857 L .683 g
F
.5026 .46504 m .50389 .47363 L p .50277 .46532 L F P
0 g
s
.50069 .45502 m .50102 .4559 L p .50088 .45521 L .652 g
F P
0 g
s
.50224 .46263 m .50069 .45502 L .49772 .43257 L p .186 g
F P
0 g
s
.49772 .43257 m .50246 .4637 L .50224 .46263 L p .186 g
F P
0 g
s
.50264 .47139 m .50389 .47363 L .50223 .46255 L p .676 g
F P
0 g
s
.50154 .46267 m .50264 .47139 L p .50223 .46255 L .676 g
F P
0 g
s
.49772 .43257 m .5009 .46675 L .50264 .47139 L closepath p .628 g
F P
0 g
s
.52393 .51001 m .50431 .47318 L .50389 .47363 L p .435 g
F P
0 g
s
.50389 .47363 m .50264 .47139 L p .50992 .48527 L .488 g
F P
0 g
s
.52232 .50893 m .51742 .49953 L p .51741 .49956 L .488 g
F P
0 g
s
.50378 .46985 m .52178 .50542 L .51647 .49475 L p .356 g
F P
0 g
s
.52178 .50542 m .50378 .46985 L .50431 .47318 L p .37 g
F P
0 g
s
.54159 .52043 m .52341 .50061 L .53889 .52344 L p .361 g
F P
0 g
s
.51647 .49475 m .53889 .52344 L .52341 .50061 L p .431 g
F P
0 g
s
.53889 .52344 m .56816 .54763 L .54159 .52043 L p .361 g
F P
0 g
s
.50431 .47318 m .52393 .51001 L .52178 .50542 L p .37 g
F P
0 g
s
.51736 .503 m .52232 .50893 L p .51741 .49956 L .488 g
F P
0 g
s
.50992 .48527 m .51463 .49714 L .51741 .49956 L .488 g
F
.50884 .48471 m .51463 .49714 L p .50992 .48527 L F P
0 g
s
.51463 .49714 m .51736 .503 L p .51741 .49956 L .488 g
F P
0 g
s
.50264 .47139 m .50884 .48471 L p .50992 .48527 L .488 g
F P
0 g
s
.50968 .48472 m .50389 .47363 L p .50992 .48527 L .488 g
F P
0 g
s
.52232 .50893 m .52393 .51001 L p .51014 .48497 L .435 g
F P
0 g
s
.50389 .47363 m .50951 .48438 L p .51014 .48497 L .435 g
F P
0 g
s
.50951 .48438 m .52232 .50893 L p .51014 .48497 L .435 g
F P
0 g
s
.50264 .47139 m .5009 .46675 L p .51477 .49731 L .525 g
F P
0 g
s
.51463 .49714 m .50264 .47139 L p .51477 .49731 L .525 g
F P
0 g
s
.50992 .48527 m .51741 .49956 L .50968 .48472 L .488 g
F
.51742 .49953 m .50968 .48472 L p .51741 .49956 L F P
0 g
s
.51736 .503 m .51463 .49714 L p .51477 .49731 L .525 g
F P
0 g
s
.51039 .49327 m .51736 .503 L p .50956 .48582 L .525 g
F P
0 g
s
.50883 .48523 m .51039 .49327 L .50956 .48582 L .525 g
F
.50628 .48133 m .51039 .49327 L .50849 .48347 L F
.53889 .52344 m .51647 .49475 L .52178 .50542 L p .207 g
F P
0 g
s
.55071 .53666 m .53889 .52344 L p .53576 .52015 L .207 g
F P
0 g
s
.52178 .50542 m .53717 .52203 L p .53576 .52015 L .207 g
F P
0 g
s
.53717 .52203 m .55071 .53666 L p .53576 .52015 L .207 g
F P
0 g
s
.55071 .53666 m .52178 .50542 L .52393 .51001 L p .111 g
F P
0 g
s
.56816 .54763 m .53889 .52344 L .55071 .53666 L p .122 g
F P
0 g
s
.58869 .56118 m .56816 .54763 L p .55838 .54148 L .122 g
F P
0 g
s
.55071 .53666 m .55708 .54077 L p .55838 .54148 L .122 g
F P
0 g
s
.55708 .54077 m .58869 .56118 L p .55838 .54148 L .122 g
F P
0 g
s
.58869 .56118 m .55071 .53666 L .55565 .54037 L p .328 g
F P
0 g
s
.52393 .51001 m .55565 .54037 L .55071 .53666 L p .111 g
F P
0 g
s
.53945 .52509 m .55469 .53947 L .53886 .5243 L .128 g
F
.55565 .54037 m .55472 .53949 L p .55469 .53947 L F P
0 g
s
.55472 .53949 m .53886 .5243 L p .55469 .53947 L .128 g
F P
0 g
s
.52393 .51001 m .52232 .50893 L p .53945 .52509 L .128 g
F P
0 g
s
.53886 .5243 m .52393 .51001 L p .53945 .52509 L .128 g
F P
0 g
s
.53398 .49775 m .52466 .49058 L .56265 .53625 L p .546 g
F P
0 g
s
.56265 .53625 m .58505 .54694 L .53398 .49775 L p .546 g
F P
0 g
s
.52466 .49058 m .51592 .48092 L .54159 .52043 L p .561 g
F P
0 g
s
.54159 .52043 m .56265 .53625 L .52466 .49058 L p .561 g
F P
0 g
s
.52341 .50061 m .54159 .52043 L .51592 .48092 L p .584 g
F P
0 g
s
.54643 .51574 m .57322 .53559 L p .53821 .51677 L .323 g
F P
0 g
s
.51736 .503 m .51039 .49327 L p .53462 .51819 L .2 g
F P
0 g
s
.53462 .51819 m .53913 .52282 L .53422 .51835 L .2 g
F
.53462 .51819 m .53422 .51835 L .51736 .503 L F
.54129 .52505 m .54021 .52406 L p .53913 .52282 L F P
0 g
s
.54021 .52406 m .51736 .503 L p .53913 .52282 L .2 g
F P
0 g
s
.52583 .51011 m .54129 .52505 L p .51728 .50035 L .2 g
F P
0 g
s
.52583 .51011 m .54311 .51483 L p .53821 .51677 L .323 g
F P
0 g
s
.54311 .51483 m .54643 .51574 L p .53821 .51677 L .323 g
F P
0 g
s
.51153 .49444 m .51722 .50072 L .51728 .50035 L .2 g
F
.51039 .49327 m .51121 .49416 L p .51153 .49444 L F P
0 g
s
.51121 .49416 m .51722 .50072 L p .51153 .49444 L .2 g
F P
0 g
s
.51722 .50072 m .52583 .51011 L p .51728 .50035 L .2 g
F P
0 g
s
.50242 .4701 m .50628 .48133 L .50849 .48347 L .525 g
F
.5009 .46675 m .50222 .47045 L p .50242 .4701 L F P
0 g
s
.50222 .47045 m .51039 .49327 L p .50242 .4701 L .525 g
F P
0 g
s
.49772 .43257 m .49937 .45799 L .49814 .44464 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .50132 .47166 L .49937 .45799 L closepath p .751 g
F P
0 g
s
.49814 .44464 m .49937 .45799 L .49734 .48205 L p .689 g
F P
0 g
s
.49937 .45799 m .50132 .47166 L .4969 .5088 L p .739 g
F P
0 g
s
.4969 .5088 m .49734 .48205 L .49876 .46518 L p .739 g
F P
0 g
s
.49876 .46518 m .49937 .45799 L p .49891 .46741 L .739 g
F P
0 g
s
.49891 .46741 m .4969 .5088 L .49876 .46518 L .739 g
F
.49913 .46011 m .5009 .46675 L .49978 .45471 L p .577 g
F P
0 g
s
.49879 .45357 m .49913 .46011 L p .49978 .45471 L .577 g
F P
0 g
s
.49772 .43257 m .49814 .44464 L .49762 .45599 L closepath p .536 g
F P
0 g
s
.49734 .48205 m .49762 .45599 L .49796 .44851 L p .689 g
F P
0 g
s
.49796 .44851 m .49814 .44464 L p .49804 .44897 L .689 g
F P
0 g
s
.49804 .44897 m .49734 .48205 L .49796 .44851 L .689 g
F
.49772 .43257 m .49762 .45599 L .49631 .46564 L closepath p .584 g
F P
0 g
s
.49809 .46607 m .49781 .45191 L .49747 .44575 L p .568 g
F P
0 g
s
.49747 .44575 m .4973 .44258 L p .49739 .44528 L .568 g
F P
0 g
s
.49739 .44528 m .49809 .46607 L .49747 .44575 L .568 g
F
.49772 .43257 m .49781 .45191 L .49913 .46011 L closepath p .526 g
F P
0 g
s
.49913 .46011 m .49794 .4527 L p .49805 .45329 L .55 g
F P
0 g
s
.50739 .48444 m .50933 .49013 L .50708 .4839 L .54 g
F
.51039 .49327 m .50985 .49177 L p .50933 .49013 L F P
0 g
s
.50985 .49177 m .50697 .48371 L p .50933 .49013 L .54 g
F P
0 g
s
.50697 .48371 m .5009 .46675 L p .50708 .4839 L .54 g
F P
0 g
s
.50933 .49013 m .50697 .48371 L .50708 .4839 L .54 g
F
.5009 .46675 m .49913 .46011 L p .50739 .48444 L F P
0 g
s
.50739 .48444 m .50708 .4839 L .5009 .46675 L .54 g
F
.50333 .4807 m .50835 .48726 L p .50008 .46293 L F P
0 g
s
.51039 .49327 m .50333 .4807 L p .51836 .50036 L .315 g
F P
0 g
s
.52208 .50603 m .51039 .49327 L p .51836 .50036 L .315 g
F P
0 g
s
.50453 .48246 m .51574 .49889 L .50478 .48259 L .315 g
F
.51574 .49889 m .51572 .49886 L .50478 .48259 L F
.50478 .48259 m .51572 .49886 L .51111 .49087 L F
.49971 .46184 m .49993 .46296 L .50008 .46293 L .54 g
F
.49913 .46011 m .49993 .46296 L .49971 .46184 L F
.49913 .46011 m .49972 .46299 L p .49993 .46296 L F P
0 g
s
.49972 .46299 m .50333 .4807 L p .50008 .46293 L .54 g
F P
0 g
s
.50333 .4807 m .49913 .46011 L p .49894 .45893 L .55 g
F P
0 g
s
.49805 .45329 m .50333 .4807 L .49894 .45893 L .55 g
F
.49793 .4525 m .50333 .4807 L .49794 .4527 L F
.50132 .47166 m .50386 .48468 L .49633 .53423 L p .777 g
F P
0 g
s
.49633 .53423 m .4969 .5088 L .50132 .47166 L p .777 g
F P
0 g
s
.49809 .46607 m .50333 .4807 L p .49902 .45824 L .55 g
F P
0 g
s
.50338 .48077 m .50569 .48486 L .50784 .48731 L .315 g
F
.50569 .48486 m .51014 .49277 L .50784 .48731 L F
.50333 .4807 m .50336 .48076 L p .50338 .48077 L F P
0 g
s
.50336 .48076 m .51014 .49277 L p .50338 .48077 L .315 g
F P
0 g
s
.50333 .4807 m .50453 .48246 L .50478 .48259 L .315 g
F
.51014 .49277 m .50333 .4807 L .50211 .47728 L p .422 g
F P
0 g
s
.50211 .47728 m .49809 .46607 L p .50415 .47949 L .422 g
F P
0 g
s
.50415 .47949 m .51014 .49277 L .50211 .47728 L .422 g
F
.49855 .474 m .51014 .49277 L p .50297 .47688 L F P
0 g
s
.50849 .48347 m .50883 .48523 L .50956 .48582 L .525 g
F
.51111 .49087 m .51595 .49919 L .51396 .4946 L .315 g
F
.51572 .49886 m .51595 .49919 L .51111 .49087 L F
.50784 .48731 m .51325 .49621 L .51547 .49849 L F
.51014 .49277 m .51325 .49621 L p .50784 .48731 L F P
0 g
s
.52341 .50061 m .50945 .47826 L .51647 .49475 L p .431 g
F P
0 g
s
.51014 .49277 m .49855 .474 L p .50602 .48167 L .162 g
F P
0 g
s
.51921 .4952 m .5131 .49357 L p .50602 .48167 L .162 g
F P
0 g
s
.5131 .49357 m .51014 .49277 L p .50602 .48167 L .162 g
F P
0 g
s
.49113 .475 m .49165 .46966 L .49221 .46625 L p .801 g
F P
0 g
s
.47891 .48833 m .48594 .4699 L p .49246 .46263 L .838 g
F P
0 g
s
.49238 .46522 m .49772 .43257 L .4923 .46743 L p .801 g
F P
0 g
s
.49221 .46625 m .49238 .46522 L p .49131 .47382 L .801 g
F P
0 g
s
.4923 .46743 m .49131 .47382 L p .49238 .46522 L .801 g
F P
0 g
s
.49131 .47382 m .49113 .475 L p .49221 .46625 L .801 g
F P
0 g
s
.49165 .46966 m .49297 .46206 L .4935 .45878 L p F P
s
.49772 .43257 m .4928 .47679 L .49155 .4775 L closepath p .743 g
F P
0 g
s
.49772 .43257 m .49455 .47275 L .4928 .47679 L closepath p .683 g
F P
0 g
s
.49772 .43257 m .49631 .46564 L .49455 .47275 L closepath p .632 g
F P
0 g
s
.49216 .50228 m .49631 .46564 L .4975 .45692 L p .705 g
F P
0 g
s
.4975 .45692 m .49762 .45599 L p .49751 .45695 L .705 g
F P
0 g
s
.49751 .45695 m .49216 .50228 L .4975 .45692 L .705 g
F
.49762 .45599 m .49734 .48205 L .49216 .50228 L p F P
0 g
s
.49479 .45717 m .49855 .474 L p .49524 .4516 L .53 g
F P
0 g
s
.48519 .51669 m .49455 .47275 L .49631 .46564 L p .728 g
F P
0 g
s
.49631 .46564 m .49216 .50228 L .48519 .51669 L p .728 g
F P
0 g
s
.49855 .474 m .49809 .46607 L .4975 .46131 L p .53 g
F P
0 g
s
.4975 .46131 m .49675 .45539 L p .49733 .46139 L .53 g
F P
0 g
s
.49733 .46139 m .49855 .474 L .4975 .46131 L .53 g
F
.4935 .45878 m .49586 .44411 L p .49221 .46625 L 0 g
F P
s
.49589 .44373 m .49221 .46625 L p .49586 .44411 L F P
s
.49221 .46625 m .49165 .46966 L p .4935 .45878 L F P
s
.48594 .4699 m .49283 .45093 L p .49389 .45202 L .779 g
F P
0 g
s
.4935 .45527 m .48594 .4699 L p .49389 .45202 L .779 g
F P
0 g
s
.49772 .43257 m .49155 .4775 L .49113 .475 L closepath p .805 g
F P
0 g
s
.49474 .45288 m .49772 .43257 L .49394 .45606 L p .031 g
F P
0 g
s
.49394 .45606 m .49297 .46206 L .49409 .45628 L p .031 g
F P
0 g
s
.49409 .45628 m .49474 .45288 L p .49394 .45606 L .031 g
F P
0 g
s
.49455 .45269 m .4935 .45527 L .49389 .45202 L .779 g
F
.49441 .45352 m .4935 .45527 L p .49455 .45269 L F P
0 g
s
.48594 .4699 m .49474 .45288 L .49297 .46206 L p .838 g
F P
0 g
s
.48698 .45627 m .49283 .45093 L .48594 .4699 L p .742 g
F P
0 g
s
.49408 .45402 m .49148 .45461 L .49418 .46178 L p .428 g
F P
0 g
s
.49677 .45573 m .49855 .474 L .49675 .45539 L .53 g
F
.49652 .45433 m .49855 .474 L .49667 .45472 L F
.49606 .4499 m .49524 .4516 L p .49855 .474 L F P
0 g
s
.4983 .46976 m .49855 .474 L p .50297 .47688 L .422 g
F P
0 g
s
.49855 .474 m .49887 .47454 L p .50093 .47645 L .162 g
F P
0 g
s
.49887 .47454 m .49915 .47501 L p .50093 .47645 L .162 g
F P
0 g
s
.49915 .47501 m .49855 .474 L .49515 .4588 L p .428 g
F P
0 g
s
.49542 .45956 m .49915 .47501 L .49515 .4588 L .428 g
F
.49474 .45288 m .49441 .45352 L p .49455 .45269 L .779 g
F P
0 g
s
.49772 .43257 m .49474 .45288 L .49648 .44284 L closepath p .17 g
F P
0 g
s
.49283 .45093 m .49648 .44284 L .49474 .45288 L p .779 g
F P
0 g
s
.49524 .4516 m .49408 .45402 L .49479 .45717 L p .53 g
F P
0 g
s
.49515 .4588 m .49408 .45402 L p .49542 .45956 L .428 g
F P
0 g
s
.49408 .45402 m .49418 .46178 L .49717 .4668 L .428 g
F
.49418 .46178 m .49736 .47026 L p .49717 .4668 L F P
0 g
s
.49736 .47026 m .49915 .47501 L p .49723 .46707 L .428 g
F P
0 g
s
.49717 .4668 m .49736 .47026 L .49723 .46707 L .428 g
F
.49809 .46607 m .4983 .46976 L p .50297 .47688 L .422 g
F P
0 g
s
.49797 .45496 m .49809 .46607 L .49902 .45824 L .55 g
F
.49782 .45244 m .49809 .46607 L p .49794 .45258 L F P
0 g
s
.4973 .44258 m .49606 .4499 L .49809 .46607 L p .568 g
F P
0 g
s
.49986 .46882 m .49915 .47501 L .49148 .45461 L p .272 g
F P
0 g
s
.49552 .46228 m .49737 .46507 L p .49589 .4621 L .272 g
F P
0 g
s
.49737 .46507 m .49986 .46882 L p .49695 .4639 L .272 g
F P
0 g
s
.49589 .4621 m .49737 .46507 L .49695 .4639 L .272 g
F
.49283 .45093 m .48698 .45627 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.48698 .45627 m .47932 .45847 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49148 .45461 m .48843 .45159 L .49552 .46228 L p .272 g
F P
0 g
s
.49148 .45461 m .49552 .46228 L .49589 .4621 L .272 g
F
.49675 .45539 m .49667 .45472 L p .49677 .45573 L .53 g
F P
0 g
s
.49667 .45472 m .49606 .4499 L p .49652 .45433 L .53 g
F P
0 g
s
.49586 .44411 m .49772 .43257 L .49589 .44373 L p F P
s
.49297 .46206 m .49244 .46304 L p .49246 .46263 L .838 g
F P
0 g
s
.49244 .46304 m .47891 .48833 L p .49246 .46263 L .838 g
F P
0 g
s
.49915 .47501 m .51154 .48749 L p .50093 .47645 L .162 g
F P
0 g
s
.51154 .48749 m .51921 .4952 L p .50886 .48458 L .162 g
F P
0 g
s
.50093 .47645 m .51154 .48749 L .50886 .48458 L .162 g
F
.52583 .51011 m .52208 .50603 L p .51836 .50036 L .315 g
F P
0 g
s
.51574 .49889 m .51595 .49919 L .51572 .49886 L .315 g
F
.49794 .45258 m .49797 .45496 L .49902 .45824 L .55 g
F
.49794 .4527 m .49781 .45191 L p .49793 .4525 L F P
0 g
s
.49781 .45191 m .49782 .45244 L p .49794 .45258 L .55 g
F P
0 g
s
.49978 .45471 m .49772 .43257 L .49879 .45357 L p .577 g
F P
0 g
s
.51325 .49621 m .51595 .49919 L p .51547 .49849 L .315 g
F P
0 g
s
.51595 .49919 m .52583 .51011 L p .51396 .4946 L .315 g
F P
0 g
s
.50223 .46255 m .49772 .43257 L .50154 .46267 L p .676 g
F P
0 g
s
.50277 .46532 m .50322 .46867 L .50327 .46857 L .683 g
F
.50378 .46985 m .50268 .46474 L p .50222 .46172 L .047 g
F P
0 g
s
.50268 .46474 m .50246 .4637 L p .50222 .46172 L .047 g
F P
0 g
s
.50246 .4637 m .50193 .46024 L p .50222 .46172 L .047 g
F P
0 g
s
.49772 .43257 m .5026 .46504 L p .50277 .46532 L .683 g
F P
0 g
s
.50403 .47348 m .50431 .47318 L .50413 .47204 L p .683 g
F P
0 g
s
.50413 .47204 m .49772 .43257 L p .50403 .47348 L .683 g
F P
0 g
s
.50413 .47204 m .50431 .47318 L p .50323 .46642 L .498 g
F P
0 g
s
.50945 .47826 m .50259 .45697 L p .50154 .45589 L .652 g
F P
0 g
s
.50185 .45783 m .50945 .47826 L .50154 .45589 L .652 g
F
.49772 .43257 m .4973 .44258 L .49781 .45191 L closepath p .47 g
F P
0 g
s
.51592 .48092 m .52466 .49058 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .48863 .36899 L .48541 .35945 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .49606 .4499 L .4973 .44258 L closepath p .751 g
F P
0 g
s
.50839 .46944 m .51592 .48092 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .48541 .35945 L .48217 .35271 L closepath p .751 g
F P
0 g
s
.44295 .3613 m .45193 .36278 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.50259 .45697 m .50839 .46944 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49648 .44284 m .49283 .45093 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .49408 .45402 L .49606 .4499 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .48217 .35271 L .4791 .34924 L closepath p .751 g
F P
0 g
s
.43515 .36312 m .44295 .3613 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .49148 .45461 L .49408 .45402 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .4791 .34924 L .47643 .34934 L closepath p .751 g
F P
0 g
s
.42905 .36813 m .43515 .36312 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .48843 .45159 L .49148 .45461 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47643 .34934 L .47434 .35303 L closepath p .751 g
F P
0 g
s
.4251 .37604 m .42905 .36813 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.47932 .45847 m .47034 .45734 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .48514 .44511 L .48843 .45159 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47434 .35303 L .47299 .36012 L closepath p .751 g
F P
0 g
s
.42361 .38632 m .4251 .37604 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.47034 .45734 m .46066 .45291 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.46066 .45291 m .45095 .44548 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.45095 .44548 m .44191 .43554 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .48183 .43558 L .48514 .44511 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47299 .36012 L .47247 .37015 L closepath p .751 g
F P
0 g
s
.42469 .39827 m .42361 .38632 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.42829 .41106 m .42469 .39827 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.43417 .42379 m .42829 .41106 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.44191 .43554 m .43417 .42379 L .49772 .43257 L closepath p .688 g
F P
0 g
s
.49772 .43257 m .47874 .42366 L .48183 .43558 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47247 .37015 L .47285 .38243 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .4761 .41018 L .47874 .42366 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47285 .38243 L .47408 .3961 L closepath p .751 g
F P
0 g
s
.49772 .43257 m .47408 .3961 L .4761 .41018 L closepath p .751 g
F P
0 g
s
.52815 .25978 m .53915 .23291 L .50051 .2891 L p .903 g
F P
0 g
s
.50051 .2891 m .49977 .30845 L .52815 .25978 L p .903 g
F P
0 g
s
.48541 .35945 m .48863 .36899 L .49977 .30845 L p .856 g
F P
0 g
s
.49977 .30845 m .50051 .2891 L .48541 .35945 L p .856 g
F P
0 g
s
.73378 .55634 m .70662 .57077 L .77027 .56393 L p .078 g
F P
0 g
s
.77027 .56393 m .806 .54056 L .73378 .55634 L p .078 g
F P
0 g
s
.60715 .55179 m .58505 .54694 L .63835 .57563 L p .294 g
F P
0 g
s
.63835 .57563 m .67404 .57741 L .60715 .55179 L p .294 g
F P
0 g
s
.27078 .66497 m .3159 .65757 L .35446 .67192 L p .748 g
F P
0 g
s
.39064 .64755 m .35446 .67192 L .3159 .65757 L p .796 g
F P
0 g
s
.35446 .67192 m .39064 .64755 L .42901 .6367 L p .85 g
F P
0 g
s
.22211 .3536 m .21503 .38289 L .14397 .42688 L p .213 g
F P
0 g
s
.14673 .46422 m .14397 .42688 L .21503 .38289 L p .324 g
F P
0 g
s
.45193 .36278 m .44295 .3613 L .36693 .31362 L p .551 g
F P
0 g
s
.48217 .35271 m .48541 .35945 L .50051 .2891 L p .887 g
F P
0 g
s
.85625 .30224 m .87965 .33691 L .86554 .27039 L p .112 g
F P
0 g
s
.86554 .27039 m .84227 .23757 L .85625 .30224 L p .112 g
F P
0 g
s
.45106 .60201 m .42901 .6367 L .39064 .64755 L p .838 g
F P
0 g
s
.42901 .6367 m .45106 .60201 L .47706 .57492 L p .85 g
F P
0 g
s
.4969 .5088 m .49633 .53423 L .47706 .57492 L p .799 g
F P
0 g
s
.50839 .46944 m .50259 .45697 L .50945 .47826 L p .616 g
F P
0 g
s
.51592 .48092 m .50839 .46944 L .52341 .50061 L p .584 g
F P
0 g
s
.34792 .31945 m .36693 .31362 L .44295 .3613 L p .55 g
F P
0 g
s
.50945 .47826 m .52341 .50061 L .50839 .46944 L p .616 g
F P
0 g
s
.76769 .20314 m .80878 .21478 L .75077 .17447 L p .494 g
F P
0 g
s
.75077 .17447 m .71684 .16936 L .76769 .20314 L p .494 g
F P
0 g
s
.59061 .19998 m .6161 .17864 L .55046 .21316 L p .913 g
F P
0 g
s
.55046 .21316 m .53915 .23291 L .59061 .19998 L p .913 g
F P
0 g
s
.58505 .54694 m .56265 .53625 L .60216 .56547 L p .285 g
F P
0 g
s
.60216 .56547 m .63835 .57563 L .58505 .54694 L p .285 g
F P
0 g
s
.206 .61346 m .25113 .62433 L .27078 .66497 L p .593 g
F P
0 g
s
.3159 .65757 m .27078 .66497 L .25113 .62433 L p .691 g
F P
0 g
s
.48558 .53968 m .47706 .57492 L .45106 .60201 L p .813 g
F P
0 g
s
.47706 .57492 m .48558 .53968 L .4969 .5088 L p .799 g
F P
0 g
s
.49734 .48205 m .4969 .5088 L .48558 .53968 L p .757 g
F P
0 g
s
.48558 .53968 m .49216 .50228 L .49734 .48205 L p .757 g
F P
0 g
s
.34792 .31945 m .33286 .33091 L .23828 .32848 L p .376 g
F P
0 g
s
.22211 .3536 m .23828 .32848 L .33286 .33091 L p .411 g
F P
0 g
s
.16166 .49983 m .18799 .53112 L .16874 .59206 L p .364 g
F P
0 g
s
.206 .61346 m .16874 .59206 L .18799 .53112 L p .499 g
F P
0 g
s
.44295 .3613 m .43515 .36312 L .34792 .31945 L p .55 g
F P
0 g
s
.84684 .47527 m .83207 .51023 L .87132 .46281 L p .004 g
F P
0 g
s
.87132 .46281 m .88788 .42005 L .84684 .47527 L p .004 g
F P
0 g
s
.50126 .27495 m .50051 .2891 L .53915 .23291 L p .933 g
F P
0 g
s
.50051 .2891 m .50126 .27495 L .48217 .35271 L p .887 g
F P
0 g
s
.4791 .34924 m .48217 .35271 L .50126 .27495 L p .91 g
F P
0 g
s
.47828 .52404 m .4928 .47679 L .49455 .47275 L p .756 g
F P
0 g
s
.49455 .47275 m .48519 .51669 L .47828 .52404 L p .756 g
F P
0 g
s
.47891 .48833 m .49297 .46206 L .49165 .46966 L p .854 g
F P
0 g
s
.70662 .57077 m .67404 .57741 L .72738 .57846 L p .058 g
F P
0 g
s
.72738 .57846 m .77027 .56393 L .70662 .57077 L p .058 g
F P
0 g
s
.67918 .1763 m .71684 .16936 L .64077 .1676 L p .748 g
F P
0 g
s
.64077 .1676 m .6161 .17864 L .67918 .1763 L p .748 g
F P
0 g
s
.53915 .23291 m .55046 .21316 L .50126 .27495 L p .933 g
F P
0 g
s
.56265 .53625 m .54159 .52043 L .56816 .54763 L p .308 g
F P
0 g
s
.56816 .54763 m .60216 .56547 L .56265 .53625 L p .308 g
F P
0 g
s
.47329 .524 m .49155 .4775 L .4928 .47679 L p .788 g
F P
0 g
s
.4928 .47679 m .47828 .52404 L .47329 .524 L p .788 g
F P
0 g
s
.49165 .46966 m .47365 .50467 L .47891 .48833 L p .854 g
F P
0 g
s
.47365 .50467 m .49165 .46966 L .49113 .475 L p .843 g
F P
0 g
s
.45106 .60201 m .47031 .56095 L .48558 .53968 L p .813 g
F P
0 g
s
.49216 .50228 m .48558 .53968 L .47031 .56095 L p .775 g
F P
0 g
s
.47031 .56095 m .48519 .51669 L .49216 .50228 L p .775 g
F P
0 g
s
.47159 .51713 m .49113 .475 L .49155 .4775 L p .818 g
F P
0 g
s
.49155 .4775 m .47329 .524 L .47159 .51713 L p .818 g
F P
0 g
s
.49113 .475 m .47159 .51713 L .47365 .50467 L p .843 g
F P
0 g
s
.39064 .64755 m .42511 .61293 L .45106 .60201 L p .838 g
F P
0 g
s
.47031 .56095 m .45106 .60201 L .42511 .61293 L p .817 g
F P
0 g
s
.43515 .36312 m .42905 .36813 L .33286 .33091 L p .555 g
F P
0 g
s
.33286 .33091 m .34792 .31945 L .43515 .36312 L p .555 g
F P
0 g
s
.47932 .45847 m .48698 .45627 L .4718 .4824 L p .723 g
F P
0 g
s
.48594 .4699 m .4718 .4824 L .48698 .45627 L p .742 g
F P
0 g
s
.3159 .65757 m .36219 .63798 L .39064 .64755 L p .796 g
F P
0 g
s
.42511 .61293 m .39064 .64755 L .36219 .63798 L p .811 g
F P
0 g
s
.21503 .38289 m .21778 .41439 L .14673 .46422 L p .324 g
F P
0 g
s
.16166 .49983 m .14673 .46422 L .21778 .41439 L p .414 g
F P
0 g
s
.47643 .34934 m .4791 .34924 L .50197 .26704 L p .891 g
F P
0 g
s
.50126 .27495 m .50197 .26704 L .49016 .30949 L p .91 g
F P
0 g
s
.50197 .26704 m .50126 .27495 L .55046 .21316 L p .913 g
F P
0 g
s
.56136 .20204 m .55046 .21316 L .6161 .17864 L p .825 g
F P
0 g
s
.55046 .21316 m .56136 .20204 L .50502 .26371 L p .913 g
F P
0 g
s
.50502 .26371 m .50197 .26704 L p .50525 .26339 L .913 g
F P
0 g
s
.50525 .26339 m .55046 .21316 L .50502 .26371 L .913 g
F
.48824 .31176 m .47643 .34934 L p .488 .31205 L .891 g
F P
0 g
s
.488 .31205 m .50197 .26704 L .48824 .31176 L .891 g
F
.50261 .26608 m .50197 .26704 L .56136 .20204 L p .703 g
F P
0 g
s
.50197 .26704 m .50261 .26608 L .48824 .31176 L p .891 g
F P
0 g
s
.48824 .31176 m .50216 .26751 L p .48745 .3127 L .763 g
F P
0 g
s
.50216 .26751 m .50261 .26608 L p .50213 .26755 L .763 g
F P
0 g
s
.48745 .3127 m .50216 .26751 L .50213 .26755 L .763 g
F
.47434 .35303 m .47643 .34934 L .48824 .31176 L p F P
0 g
s
.47434 .35303 m .48824 .31176 L .48745 .3127 L .763 g
F
.49016 .30949 m .4791 .34924 L p .49139 .30804 L .91 g
F P
0 g
s
.49139 .30804 m .50126 .27495 L .49016 .30949 L .91 g
F
.4718 .4824 m .48594 .4699 L .47891 .48833 L p .781 g
F P
0 g
s
.45512 .57066 m .47828 .52404 L .48519 .51669 L p .792 g
F P
0 g
s
.48519 .51669 m .47031 .56095 L .45512 .57066 L p .792 g
F P
0 g
s
.42511 .61293 m .45512 .57066 L .47031 .56095 L p .817 g
F P
0 g
s
.67404 .57741 m .63835 .57563 L .68045 .58296 L p .049 g
F P
0 g
s
.68045 .58296 m .72738 .57846 L .67404 .57741 L p .049 g
F P
0 g
s
.42905 .36813 m .4251 .37604 L .32288 .34731 L p .563 g
F P
0 g
s
.32288 .34731 m .33286 .33091 L .42905 .36813 L p .563 g
F P
0 g
s
.25113 .62433 m .30089 .62361 L .3159 .65757 L p .691 g
F P
0 g
s
.36219 .63798 m .3159 .65757 L .30089 .62361 L p .745 g
F P
0 g
s
.47034 .45734 m .47932 .45847 L .45318 .48943 L p .706 g
F P
0 g
s
.4718 .4824 m .45318 .48943 L .47932 .45847 L p .723 g
F P
0 g
s
.87965 .33691 m .89059 .37706 L .87653 .31129 L p .026 g
F P
0 g
s
.87653 .31129 m .86554 .27039 L .87965 .33691 L p .026 g
F P
0 g
s
.33286 .33091 m .32288 .34731 L .22211 .3536 L p .411 g
F P
0 g
s
.21503 .38289 m .22211 .3536 L .32288 .34731 L p .449 g
F P
0 g
s
.44409 .56851 m .47329 .524 L .47828 .52404 L p .806 g
F P
0 g
s
.47828 .52404 m .45512 .57066 L .44409 .56851 L p .806 g
F P
0 g
s
.83207 .51023 m .806 .54056 L .84176 .50215 L p F P
s
.84176 .50215 m .87132 .46281 L .83207 .51023 L p F P
s
.18799 .53112 m .22403 .55568 L .206 .61346 L p .499 g
F P
0 g
s
.25113 .62433 m .206 .61346 L .22403 .55568 L p .591 g
F P
0 g
s
.36219 .63798 m .40607 .60753 L .42511 .61293 L p .811 g
F P
0 g
s
.45512 .57066 m .42511 .61293 L .40607 .60753 L p .813 g
F P
0 g
s
.47891 .48833 m .45626 .5095 L .4718 .4824 L p .781 g
F P
0 g
s
.45626 .5095 m .47891 .48833 L .47365 .50467 L p .805 g
F P
0 g
s
.80878 .21478 m .84227 .23757 L .77849 .19182 L p .191 g
F P
0 g
s
.77849 .19182 m .75077 .17447 L .80878 .21478 L p .191 g
F P
0 g
s
.44023 .55579 m .47159 .51713 L .47329 .524 L p .814 g
F P
0 g
s
.47329 .524 m .44409 .56851 L .44023 .55579 L p .814 g
F P
0 g
s
.40607 .60753 m .44409 .56851 L .45512 .57066 L p .813 g
F P
0 g
s
.6161 .17864 m .64077 .1676 L .56136 .20204 L p .825 g
F P
0 g
s
.63835 .57563 m .60216 .56547 L .63303 .57704 L p .055 g
F P
0 g
s
.63303 .57704 m .68045 .58296 L .63835 .57563 L p .055 g
F P
0 g
s
.4251 .37604 m .42361 .38632 L .31877 .36761 L p .575 g
F P
0 g
s
.31877 .36761 m .32288 .34731 L .4251 .37604 L p .575 g
F P
0 g
s
.47365 .50467 m .44468 .53504 L .45626 .5095 L p .805 g
F P
0 g
s
.44468 .53504 m .47365 .50467 L .47159 .51713 L p .815 g
F P
0 g
s
.46066 .45291 m .47034 .45734 L .43131 .49039 L p .689 g
F P
0 g
s
.45318 .48943 m .43131 .49039 L .47034 .45734 L p .706 g
F P
0 g
s
.47159 .51713 m .44023 .55579 L .44468 .53504 L p .815 g
F P
0 g
s
.45318 .48943 m .4718 .4824 L .45626 .5095 L p .746 g
F P
0 g
s
.60216 .56547 m .56816 .54763 L .58869 .56118 L p .081 g
F P
0 g
s
.58869 .56118 m .63303 .57704 L .60216 .56547 L p .081 g
F P
0 g
s
.30089 .62361 m .35154 .61121 L .36219 .63798 L p .745 g
F P
0 g
s
.40607 .60753 m .36219 .63798 L .35154 .61121 L p .779 g
F P
0 g
s
.21778 .41439 m .23037 .44587 L .16166 .49983 L p .414 g
F P
0 g
s
.18799 .53112 m .16166 .49983 L .23037 .44587 L p .488 g
F P
0 g
s
.50063 .45569 m .49986 .46882 L .48843 .45159 L p .085 g
F P
0 g
s
.48843 .45159 m .48514 .44511 L .50063 .45569 L p .085 g
F P
0 g
s
.50261 .26608 m .50311 .27233 L .47434 .35303 L p .763 g
F P
0 g
s
.47299 .36012 m .47434 .35303 L .50311 .27233 L p F P
s
.42361 .38632 m .42469 .39827 L .32093 .39042 L p .589 g
F P
0 g
s
.32093 .39042 m .31877 .36761 L .42361 .38632 L p .589 g
F P
0 g
s
.45095 .44548 m .46066 .45291 L .40769 .48511 L p .672 g
F P
0 g
s
.43131 .49039 m .40769 .48511 L .46066 .45291 L p .689 g
F P
0 g
s
.3992 .588 m .44023 .55579 L .44409 .56851 L p .801 g
F P
0 g
s
.44409 .56851 m .40607 .60753 L .3992 .588 L p .801 g
F P
0 g
s
.66298 .16802 m .64077 .1676 L .71684 .16936 L p .414 g
F P
0 g
s
.71684 .16936 m .75077 .17447 L .66298 .16802 L p .414 g
F P
0 g
s
.35154 .61121 m .3992 .588 L .40607 .60753 L p .779 g
F P
0 g
s
.22403 .55568 m .26726 .57156 L .25113 .62433 L p .591 g
F P
0 g
s
.30089 .62361 m .25113 .62433 L .26726 .57156 L p .658 g
F P
0 g
s
.32288 .34731 m .31877 .36761 L .21503 .38289 L p .449 g
F P
0 g
s
.21778 .41439 m .21503 .38289 L .31877 .36761 L p .487 g
F P
0 g
s
.806 .54056 m .77027 .56393 L .80122 .53502 L p .811 g
F P
0 g
s
.80122 .53502 m .84176 .50215 L .806 .54056 L p .811 g
F P
0 g
s
.42469 .39827 m .42829 .41106 L .3293 .41415 L p .604 g
F P
0 g
s
.3293 .41415 m .32093 .39042 L .42469 .39827 L p .604 g
F P
0 g
s
.44191 .43554 m .45095 .44548 L .38402 .47388 L p .655 g
F P
0 g
s
.40769 .48511 m .38402 .47388 L .45095 .44548 L p .672 g
F P
0 g
s
.45626 .5095 m .42644 .52393 L .45318 .48943 L p .746 g
F P
0 g
s
.42644 .52393 m .45626 .5095 L .44468 .53504 L p .766 g
F P
0 g
s
.52232 .50893 m .55223 .53584 L .55565 .54037 L p .128 g
F P
0 g
s
.55223 .53584 m .52232 .50893 L .51736 .503 L p .092 g
F P
0 g
s
.40661 .55843 m .44468 .53504 L .44023 .55579 L p .785 g
F P
0 g
s
.44023 .55579 m .3992 .588 L .40661 .55843 L p .785 g
F P
0 g
s
.50141 .43639 m .50063 .45569 L .48514 .44511 L p .159 g
F P
0 g
s
.48514 .44511 m .48183 .43558 L .50141 .43639 L p .159 g
F P
0 g
s
.51736 .503 m .54129 .52505 L .55223 .53584 L p .092 g
F P
0 g
s
.42829 .41106 m .43417 .42379 L .34333 .43706 L p .621 g
F P
0 g
s
.34333 .43706 m .3293 .41415 L .42829 .41106 L p .621 g
F P
0 g
s
.50311 .27233 m .50345 .28555 L .47299 .36012 L p F P
s
.47247 .37015 m .47299 .36012 L .50345 .28555 L p .025 g
F P
0 g
s
.43417 .42379 m .44191 .43554 L .36203 .45747 L p .637 g
F P
0 g
s
.38402 .47388 m .36203 .45747 L .44191 .43554 L p .655 g
F P
0 g
s
.43131 .49039 m .45318 .48943 L .42644 .52393 L p .715 g
F P
0 g
s
.89059 .37706 m .88788 .42005 L .874 .35754 L p .744 g
F P
0 g
s
.874 .35754 m .87653 .31129 L .89059 .37706 L p .744 g
F P
0 g
s
.36203 .45747 m .34333 .43706 L .43417 .42379 L p .637 g
F P
0 g
s
.44468 .53504 m .40661 .55843 L .42644 .52393 L p .766 g
F P
0 g
s
.26726 .57156 m .31447 .57749 L .30089 .62361 L p .658 g
F P
0 g
s
.35154 .61121 m .30089 .62361 L .31447 .57749 L p .709 g
F P
0 g
s
.23037 .44587 m .25204 .47503 L .18799 .53112 L p .488 g
F P
0 g
s
.22403 .55568 m .18799 .53112 L .25204 .47503 L p .55 g
F P
0 g
s
.50214 .41221 m .50141 .43639 L .48183 .43558 L p .185 g
F P
0 g
s
.48183 .43558 m .47874 .42366 L .50214 .41221 L p .185 g
F P
0 g
s
.36212 .57298 m .40661 .55843 L .3992 .588 L p .75 g
F P
0 g
s
.3992 .588 m .35154 .61121 L .36212 .57298 L p .75 g
F P
0 g
s
.50345 .28555 m .50359 .30493 L .47247 .37015 L p .025 g
F P
0 g
s
.47285 .38243 m .47247 .37015 L .50359 .30493 L p .046 g
F P
0 g
s
.77027 .56393 m .72738 .57846 L .75267 .5588 L p .843 g
F P
0 g
s
.75267 .5588 m .80122 .53502 L .77027 .56393 L p .843 g
F P
0 g
s
.31447 .57749 m .36212 .57298 L .35154 .61121 L p .709 g
F P
0 g
s
.84227 .23757 m .86554 .27039 L .79777 .22072 L p .394 g
F P
0 g
s
.79777 .22072 m .77849 .19182 L .84227 .23757 L p .394 g
F P
0 g
s
.31877 .36761 m .32093 .39042 L .21778 .41439 L p .487 g
F P
0 g
s
.23037 .44587 m .21778 .41439 L .32093 .39042 L p .524 g
F P
0 g
s
.63303 .57704 m .58869 .56118 L .59754 .56186 L p .535 g
F P
0 g
s
.55565 .54037 m .59754 .56186 L .58869 .56118 L p .328 g
F P
0 g
s
.50276 .38491 m .50214 .41221 L .47874 .42366 L p .196 g
F P
0 g
s
.47874 .42366 m .4761 .41018 L .50276 .38491 L p .196 g
F P
0 g
s
.50359 .30493 m .50351 .32917 L .47285 .38243 L p .046 g
F P
0 g
s
.47408 .3961 m .47285 .38243 L .50351 .32917 L p .093 g
F P
0 g
s
.40769 .48511 m .43131 .49039 L .39143 .53045 L p .685 g
F P
0 g
s
.42644 .52393 m .39143 .53045 L .43131 .49039 L p .715 g
F P
0 g
s
.39143 .53045 m .42644 .52393 L .40661 .55843 L p .728 g
F P
0 g
s
.64077 .1676 m .66298 .16802 L .5711 .20063 L p F P
s
.5711 .20063 m .56136 .20204 L .64077 .1676 L p F P
s
.40661 .55843 m .36212 .57298 L .39143 .53045 L p .728 g
F P
0 g
s
.50323 .35652 m .50276 .38491 L .4761 .41018 L p .165 g
F P
0 g
s
.4761 .41018 m .47408 .3961 L .50323 .35652 L p .165 g
F P
0 g
s
.50351 .32917 m .50323 .35652 L .47408 .3961 L p .093 g
F P
0 g
s
.54643 .51574 m .52583 .51011 L .51014 .49277 L p .227 g
F P
0 g
s
.51014 .49277 m .51921 .4952 L .54643 .51574 L p .227 g
F P
0 g
s
.59754 .56186 m .55565 .54037 L .55223 .53584 L p .397 g
F P
0 g
s
.52999 .48745 m .51921 .4952 L .49915 .47501 L p .253 g
F P
0 g
s
.49915 .47501 m .49986 .46882 L .52999 .48745 L p .253 g
F P
0 g
s
.56136 .20204 m .5711 .20063 L .50261 .26608 L p .703 g
F P
0 g
s
.72738 .57846 m .68045 .58296 L .69984 .57165 L p .791 g
F P
0 g
s
.69984 .57165 m .75267 .5588 L .72738 .57846 L p .791 g
F P
0 g
s
.25204 .47503 m .28133 .49968 L .22403 .55568 L p .55 g
F P
0 g
s
.26726 .57156 m .22403 .55568 L .28133 .49968 L p .602 g
F P
0 g
s
.50311 .27233 m .50261 .26608 L .5711 .20063 L p F P
s
.68045 .58296 m .63303 .57704 L .64679 .57263 L p .689 g
F P
0 g
s
.59754 .56186 m .64679 .57263 L .63303 .57704 L p .535 g
F P
0 g
s
.32093 .39042 m .3293 .41415 L .23037 .44587 L p .524 g
F P
0 g
s
.64679 .57263 m .69984 .57165 L .68045 .58296 L p .689 g
F P
0 g
s
.88788 .42005 m .87132 .46281 L .85769 .4058 L p .884 g
F P
0 g
s
.85769 .4058 m .874 .35754 L .88788 .42005 L p .884 g
F P
0 g
s
.38402 .47388 m .40769 .48511 L .35373 .52844 L p .655 g
F P
0 g
s
.39143 .53045 m .35373 .52844 L .40769 .48511 L p .685 g
F P
0 g
s
.35373 .52844 m .39143 .53045 L .36212 .57298 L p .69 g
F P
0 g
s
.36212 .57298 m .31447 .57749 L .35373 .52844 L p .69 g
F P
0 g
s
.75077 .17447 m .77849 .19182 L .68108 .18038 L p .148 g
F P
0 g
s
.68108 .18038 m .66298 .16802 L .75077 .17447 L p .148 g
F P
0 g
s
.25204 .47503 m .23037 .44587 L .3293 .41415 L p .559 g
F P
0 g
s
.28133 .49968 m .31609 .51793 L .26726 .57156 L p .602 g
F P
0 g
s
.31447 .57749 m .26726 .57156 L .31609 .51793 L p .648 g
F P
0 g
s
.55223 .53584 m .59197 .55215 L .59754 .56186 L p .397 g
F P
0 g
s
.59197 .55215 m .55223 .53584 L .54129 .52505 L p .393 g
F P
0 g
s
.31609 .51793 m .35373 .52844 L .31447 .57749 L p .648 g
F P
0 g
s
.57322 .53559 m .54129 .52505 L .52583 .51011 L p .323 g
F P
0 g
s
.3293 .41415 m .34333 .43706 L .25204 .47503 L p .559 g
F P
0 g
s
.36203 .45747 m .38402 .47388 L .31609 .51793 L p .624 g
F P
0 g
s
.35373 .52844 m .31609 .51793 L .38402 .47388 L p .655 g
F P
0 g
s
.54129 .52505 m .57322 .53559 L .59197 .55215 L p .393 g
F P
0 g
s
.34333 .43706 m .36203 .45747 L .28133 .49968 L p .592 g
F P
0 g
s
.28133 .49968 m .25204 .47503 L .34333 .43706 L p .592 g
F P
0 g
s
.31609 .51793 m .28133 .49968 L .36203 .45747 L p .624 g
F P
0 g
s
.54177 .46968 m .52999 .48745 L .49986 .46882 L p .397 g
F P
0 g
s
.49986 .46882 m .50063 .45569 L .54177 .46968 L p .397 g
F P
0 g
s
.64679 .57263 m .59754 .56186 L .59197 .55215 L p .615 g
F P
0 g
s
.86554 .27039 m .87653 .31129 L .80683 .25953 L p .626 g
F P
0 g
s
.80683 .25953 m .79777 .22072 L .86554 .27039 L p .626 g
F P
0 g
s
.5711 .20063 m .57897 .20939 L .50311 .27233 L p F P
s
.50345 .28555 m .50311 .27233 L .57897 .20939 L p .088 g
F P
0 g
s
.87132 .46281 m .84176 .50215 L .82857 .45236 L p .907 g
F P
0 g
s
.82857 .45236 m .85769 .4058 L .87132 .46281 L p .907 g
F P
0 g
s
.57099 .50923 m .54643 .51574 L .51921 .4952 L p .448 g
F P
0 g
s
.51921 .4952 m .52999 .48745 L .57099 .50923 L p .448 g
F P
0 g
s
.57897 .20939 m .5711 .20063 L .66298 .16802 L p .092 g
F P
0 g
s
.66298 .16802 m .68108 .18038 L .57897 .20939 L p .092 g
F P
0 g
s
.69984 .57165 m .64679 .57263 L .63902 .55626 L p .734 g
F P
0 g
s
.59197 .55215 m .63902 .55626 L .64679 .57263 L p .615 g
F P
0 g
s
.5537 .44286 m .54177 .46968 L .50063 .45569 L p .461 g
F P
0 g
s
.50063 .45569 m .50141 .43639 L .5537 .44286 L p .461 g
F P
0 g
s
.63902 .55626 m .59197 .55215 L .57322 .53559 L p .601 g
F P
0 g
s
.84176 .50215 m .80122 .53502 L .78875 .49354 L p .89 g
F P
0 g
s
.78875 .49354 m .82857 .45236 L .84176 .50215 L p .89 g
F P
0 g
s
.77849 .19182 m .79777 .22072 L .69358 .20434 L p .392 g
F P
0 g
s
.69358 .20434 m .68108 .18038 L .77849 .19182 L p .392 g
F P
0 g
s
.57897 .20939 m .58433 .2281 L .50345 .28555 L p .088 g
F P
0 g
s
.50359 .30493 m .50345 .28555 L .58433 .2281 L p .233 g
F P
0 g
s
.61121 .53336 m .57322 .53559 L .54643 .51574 L p .548 g
F P
0 g
s
.54643 .51574 m .57099 .50923 L .61121 .53336 L p .548 g
F P
0 g
s
.75267 .5588 m .69984 .57165 L .69009 .54745 L p .807 g
F P
0 g
s
.63902 .55626 m .69009 .54745 L .69984 .57165 L p .734 g
F P
0 g
s
.57322 .53559 m .61121 .53336 L .63902 .55626 L p .601 g
F P
0 g
s
.80122 .53502 m .75267 .5588 L .74133 .52606 L p .856 g
F P
0 g
s
.74133 .52606 m .78875 .49354 L .80122 .53502 L p .856 g
F P
0 g
s
.87653 .31129 m .874 .35754 L .80459 .30567 L p .719 g
F P
0 g
s
.80459 .30567 m .80683 .25953 L .87653 .31129 L p .719 g
F P
0 g
s
.69009 .54745 m .74133 .52606 L .75267 .5588 L p .807 g
F P
0 g
s
.56489 .4088 m .5537 .44286 L .50141 .43639 L p .479 g
F P
0 g
s
.50141 .43639 m .50214 .41221 L .56489 .4088 L p .479 g
F P
0 g
s
.58433 .2281 m .58668 .2557 L .50359 .30493 L p .233 g
F P
0 g
s
.50351 .32917 m .50359 .30493 L .58668 .2557 L p .343 g
F P
0 g
s
.59787 .49055 m .57099 .50923 L .52999 .48745 L p .55 g
F P
0 g
s
.52999 .48745 m .54177 .46968 L .59787 .49055 L p .55 g
F P
0 g
s
.58433 .2281 m .57897 .20939 L .68108 .18038 L p .277 g
F P
0 g
s
.68108 .18038 m .69358 .20434 L .58433 .2281 L p .277 g
F P
0 g
s
.69009 .54745 m .63902 .55626 L .61121 .53336 L p .692 g
F P
0 g
s
.57444 .36999 m .56489 .4088 L .50214 .41221 L p .473 g
F P
0 g
s
.50214 .41221 m .50276 .38491 L .57444 .36999 L p .473 g
F P
0 g
s
.58668 .2557 m .58577 .29033 L .50351 .32917 L p .343 g
F P
0 g
s
.50323 .35652 m .50351 .32917 L .58577 .29033 L p .409 g
F P
0 g
s
.58159 .32944 m .57444 .36999 L .50276 .38491 L p .45 g
F P
0 g
s
.50276 .38491 m .50323 .35652 L .58159 .32944 L p .45 g
F P
0 g
s
.58577 .29033 m .58159 .32944 L .50323 .35652 L p .409 g
F P
0 g
s
.79777 .22072 m .80683 .25953 L .69931 .23856 L p .531 g
F P
0 g
s
.69931 .23856 m .69358 .20434 L .79777 .22072 L p .531 g
F P
0 g
s
.874 .35754 m .85769 .4058 L .79089 .35573 L p .755 g
F P
0 g
s
.79089 .35573 m .80459 .30567 L .874 .35754 L p .755 g
F P
0 g
s
.65269 .51799 m .61121 .53336 L .57099 .50923 L p .632 g
F P
0 g
s
.57099 .50923 m .59787 .49055 L .65269 .51799 L p .632 g
F P
0 g
s
.61121 .53336 m .65269 .51799 L .69009 .54745 L p .692 g
F P
0 g
s
.62512 .46064 m .59787 .49055 L .54177 .46968 L p .586 g
F P
0 g
s
.54177 .46968 m .5537 .44286 L .62512 .46064 L p .586 g
F P
0 g
s
.58668 .2557 m .58433 .2281 L .69358 .20434 L p .415 g
F P
0 g
s
.74133 .52606 m .69009 .54745 L .65269 .51799 L p .736 g
F P
0 g
s
.85769 .4058 m .82857 .45236 L .76661 .40579 L p .768 g
F P
0 g
s
.76661 .40579 m .79089 .35573 L .85769 .4058 L p .768 g
F P
0 g
s
.69358 .20434 m .69931 .23856 L .58668 .2557 L p .415 g
F P
0 g
s
.78875 .49354 m .74133 .52606 L .69459 .49019 L p .758 g
F P
0 g
s
.65269 .51799 m .69459 .49019 L .74133 .52606 L p .736 g
F P
0 g
s
.80683 .25953 m .80459 .30567 L .6976 .28077 L p .601 g
F P
0 g
s
.6976 .28077 m .69931 .23856 L .80683 .25953 L p .601 g
F P
0 g
s
.82857 .45236 m .78875 .49354 L .73362 .45183 L p .768 g
F P
0 g
s
.73362 .45183 m .76661 .40579 L .82857 .45236 L p .768 g
F P
0 g
s
.69459 .49019 m .73362 .45183 L .78875 .49354 L p .758 g
F P
0 g
s
.65062 .42148 m .62512 .46064 L .5537 .44286 L p .594 g
F P
0 g
s
.5537 .44286 m .56489 .4088 L .65062 .42148 L p .594 g
F P
0 g
s
.58577 .29033 m .58668 .2557 L .69931 .23856 L p .493 g
F P
0 g
s
.69459 .49019 m .65269 .51799 L .59787 .49055 L p .665 g
F P
0 g
s
.59787 .49055 m .62512 .46064 L .69459 .49019 L p .665 g
F P
0 g
s
.6723 .37601 m .65062 .42148 L .56489 .4088 L p .587 g
F P
0 g
s
.56489 .4088 m .57444 .36999 L .6723 .37601 L p .587 g
F P
0 g
s
.69931 .23856 m .6976 .28077 L .58577 .29033 L p .493 g
F P
0 g
s
.58159 .32944 m .58577 .29033 L .6976 .28077 L p .54 g
F P
0 g
s
.80459 .30567 m .79089 .35573 L .68839 .32782 L p .64 g
F P
0 g
s
.68839 .32782 m .6976 .28077 L .80459 .30567 L p .64 g
F P
0 g
s
.73362 .45183 m .69459 .49019 L .62512 .46064 L p .675 g
F P
0 g
s
.68839 .32782 m .6723 .37601 L .57444 .36999 L p .57 g
F P
0 g
s
.57444 .36999 m .58159 .32944 L .68839 .32782 L p .57 g
F P
0 g
s
.6976 .28077 m .68839 .32782 L .58159 .32944 L p .54 g
F P
0 g
s
.62512 .46064 m .65062 .42148 L .73362 .45183 L p .675 g
F P
0 g
s
.79089 .35573 m .76661 .40579 L .6723 .37601 L p .662 g
F P
0 g
s
.6723 .37601 m .68839 .32782 L .79089 .35573 L p .662 g
F P
0 g
s
.76661 .40579 m .73362 .45183 L .65062 .42148 L p .674 g
F P
0 g
s
.65062 .42148 m .6723 .37601 L .76661 .40579 L p .674 g
F P
0 g
s
.25 Mabswid
.7913 0 m
.95779 .2576 L
s
.95779 .2576 m
1 .60412 L
s
1 .60412 m
.82157 .36284 L
s
.82157 .36284 m
.7913 0 L
s
.04148 .27674 m
0 .62185 L
s
0 .62185 m
.82157 .36284 L
s
.82157 .36284 m
.7913 0 L
s
.7913 0 m
.04148 .27674 L
s
.04148 .27674 m
.7913 0 L
s
.05744 .27085 m
.06197 .27519 L
s
gsave
.04837 .26216 -74.9375 -13.7877 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.2217 .21022 m
.22601 .21478 L
s
gsave
.21306 .20111 -81.0812 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.39492 .14629 m
.39898 .15108 L
s
gsave
.38678 .13673 -68.8055 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.57785 .07878 m
.58163 .08379 L
s
gsave
.5703 .06875 -74.2579 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.77133 .00737 m
.77478 .01262 L
s
gsave
.76445 -0.00313 -67.9864 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08961 .25897 m
.0923 .26161 L
s
.12211 .24698 m
.12478 .24963 L
s
.15496 .23486 m
.1576 .23754 L
s
.18815 .2226 m
.19077 .22531 L
s
.2556 .19771 m
.25816 .20047 L
s
.28987 .18506 m
.2924 .18785 L
s
.32451 .17228 m
.32701 .17509 L
s
.35952 .15936 m
.36199 .1622 L
s
.4307 .13309 m
.43311 .13598 L
s
.46688 .11973 m
.46925 .12266 L
s
.50346 .10623 m
.5058 .10919 L
s
.54045 .09258 m
.54275 .09556 L
s
.61567 .06482 m
.6179 .06786 L
s
.65393 .0507 m
.65612 .05377 L
s
.69262 .03642 m
.69477 .03951 L
s
.73175 .02198 m
.73386 .0251 L
s
gsave
.34612 .08889 -68.3429 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  PageBreakAbove->True,
  ImageSize->{545.75, 184.75},
  ImageMargins->{{44.75, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40008Q0000^1000`40O003h00Oogooogoo8goo003o
OoooOolSOol00?mooomoob=oo`00ogooogoo8goo003oOoooOolSOol00?mooomoob=oo`00ogooogoo
8goo003oOoooOolSOol007Moo`03001oogoo09Yoo`@00003Ool007oo00800?moo`Eoo`00Mgoo00<0
07ooOol0VWoo00@007ooOol0009oo`04001oogoo003oOol4Ool007Moo`03001oogoo09Yoo`04001o
ogoo0005Ool00`00Oomoo`3oOol2Ool007Moo`03001oogoo081oo`03001oo`0001Moo`04001oogoo
0002Ool3003oOol5Ool007Ioo`8008=oo`03001oogoo01Ioo`04001oogoo0002Ool00`00Oomoo`3o
Ool5Ool007Moo`03001oogoo011oo`03001oogoo06ioo`03001oogoo01Ioo`@000=oo`<000eoo`@0
0003Ool007oo00800>eoo`00Pgoo10000goo00<007ooOol0KWoo00<007ooOol01Goo100000=oo`00
00000`0000=oo`0000005goo1@000Woo00@007ooOol0009oo`03001oogoo0>Yoo`00O7oo0P0037oo
00<007ooOol0KGoo00<007oo00001Woo00@007ooOol000=oo`04001oogoo0002Ool00`00Oomoo`0=
Ool20009Ool01000Oomoo`001Goo00<007ooOol0jWoo001jOol200000goo0000000;Ool00`00Oomo
o`1fOol01000Oomoo`0017oo00<007ooOol00Woo00<007ooOol02goo1@0027oo00@007ooOol0009o
o`<00>eoo`00FWoo100000=oo`00Ool00P0057oo0`000Woo0`002Woo0P00N7oo00@007ooOol000Eo
o`@000]oo`@0009oo`8000Qoo`04001oogoo0002Ool00`00Oomoo`3]Ool004Eoo`03001oo`00019o
o`04001oogoo0002Ool01000Oomoo`004Goo0`001Goo00<007oo00002Woo00<007ooOol0MWoo00@0
07ooOol0009oo`03001oogoo008000]oo`<000Eoo`03001oo`0000Moo`@000=oo`<00>aoo`00AWoo
00<007ooOol04Goo00@007ooOol000Eoo`03001oogoo00eoo`8000Qoo`03001oo`0007moo`P000=o
o`<00004Ool000000005Ool30008Ool01000Oomoo`00o7oo0016Ool00`00Oomoo`0AOol01000Oomo
o`000Woo0`003Goo10002Goo00@007ooOol007ioo`04001oogoo000AOol3000:Ool00`00Oomoo`02
003kOol004Ioo`03001oogoo015oo`04001oogoo0002Ool00`00Oomoo`0;Ool2000=Ool01@00Oomo
ogoo0000OGoo00@007ooOol000ioo`<000eoo`05001oogooOol0003kOol004Eoo`03001oo`00019o
o`@000=oo`<000Moo`<00003Ool007oo00aoo`05001oogooOol0001mOol01000Oomoo`0037oo0`00
00=oo`00Ool037oo00@007ooOomoo`8000]oo`@00003Ool0000000<00003Ool000000>9oo`00JGoo
0P004Woo00<007ooOol00Woo00<007ooOol0NWoo00@007ooOol000Uoo`<0019oo`03001oogoo00=o
o`03001oogoo00=oo`@00005Ool007ooOol00003Ool01000Oomoo`000Woo00<007ooOol0ggoo001W
Ool20002Ool00`00Oomoo`0@Ool00`00Oomoo`02Ool00`00Oomoo`1SOol400000goo000000030000
0goo00000009Ool40007Ool3000DOol00`00Oomoo`03Ool00`00Oomoo`08Ool01000Oomoo`0017oo
00<007ooOol00Woo00<007ooOol0ggoo001TOol3000GOol00`00Oomoo`02Ool00`00Oomoo`1NOol4
00001Goo001oogoo00000goo00@007ooOol0009oo`03001oogoo00ioo`<001Moo`03001oogoo00Ao
o`03001oogoo00Moo`04001oogoo0005Ool4003ROol004]oo`@001=oo`8000=oo`03001oogoo01Ao
o`03001oogoo00=oo`03001oogoo069oo`04001oogoo0004Ool00`00Oomoo`02Ool00`00Oomoo`0;
Ool3000JOol00`00Oomoo`04Ool20008Ool01000Oomoo`000Woo00<007ooOol00`00hgoo001;Ool0
1000Oomoo`0047oo10006goo00<007ooOol00goo00<007ooOol0HWoo00@007ooOol000Eoo`@000ao
o`<0009oo`03001oogoo01Moo`03001oogoo00Eoo`8000Moo`@000=oo`<00004Ool00000003QOol0
04]oo`04001oogoo000>Ool2000OOol00`00Oomoo`04Ool00`00Oomoo`1HOol400000goo001oo`03
0002Ool01000Oomoo`000Woo0P002goo0`007goo00<007ooOol01Goo0P004Woo00<007ooOol0hGoo
001;Ool01000Oomoo`0037oo0P0000=oo`00Ool07Woo00<007ooOol01Goo00<007ooOol02Goo1000
00=oo`00Ool00P00@Goo1@000Woo00@007ooOol0009oo`@000=oo`<00004Ool000000006Ool3000Q
Ool00`00Oomoo`07Ool00`00Oomoo`0>Ool00`00Ool0003ROol004]oo`04001oogoo0009Ool3000S
Ool00`00Oomoo`05Ool00`00Oomoo`04Ool400001Goo001oogoo00000Woo00@007ooOol004Aoo`04
001oogoo0005Ool00`00Oomoo`0>Ool3000TOol00`00Oomoo`08Ool00`00Oomoo`0=Ool00`00Ool0
003ROol004]oo`@000Moo`8002Ioo`03001oogoo00Ioo`03001oogoo00Qoo`04001oogoo0005Ool0
0`00Oomoo`12Ool01000Oomoo`000Woo0`003goo0`009Woo00<007ooOol027oo00<007ooOol017oo
10001Goo00<007oo0000hWoo001COol4000WOol00`00Oomoo`07Ool00`00Oomoo`07Ool01000Oomo
o`000Woo0`00AGoo00@007ooOol0009oo`03001oogoo00aoo`@002Qoo`03001oogoo00Uoo`03001o
ogoo00=oo`04001oogoo0004Ool01@00Oomoogoo0000hGoo001AOol2000[Ool00`00Oomoo`06Ool2
0009Ool01000Oomoo`000Woo00<007ooOol0AGoo10000goo0`002Goo0`00;7oo00<007ooOol027oo
0P0017oo00@007ooOol00:moo`03001oogoo03Qoo`00<Woo100000=oo`00Ool00P004goo0`000Woo
00<007ooOol0:7oo00<007ooOol027oo00<007ooOol01Woo10000goo0`00E7oo1000;Woo00<007oo
Ool02Goo0P000goo00@007ooOol00:moo`03001oogoo03Qoo`00;Goo100000Eoo`00Oomoo`00009o
o`04001oogoo000@Ool4000^Ool00`00Oomoo`08Ool00`00Oomoo`1ROol3000aOol00`00Oomoo`09
Ool20003Ool01000Oomoo`00[goo00<007ooOol0>7oo000bOol01000Oomoo`001Goo00<007ooOol0
37oo0P00<Woo00<007ooOol02Goo00<007ooOol0Ggoo0`00<goo00<007ooOol02goo00<007ooOol0
1000[goo00<007ooOol0>7oo000bOol01000Oomoo`000Woo0`0037oo0`0000=oo`00Ool0<Goo00<0
07ooOol02Woo00<007ooOol0GWoo0P00=7oo00<007ooOol02goo00<007ooOol0/Woo0P003goo1000
00=oo`00Ool00P008Woo000bOol01000Oomoo`000Woo00<007ooOol02Woo0`00=Woo00<007ooOol0
2Woo00<007ooOol04Woo00<007ooOol0BGoo0`00<goo00<007ooOol037oo00<007ooOol0/Woo00<0
07ooOol027oo100000Eoo`00Oomoo`00009oo`04001oogoo000QOol0039oo`@000=oo`<000Ioo`<0
03Uoo`03001oogoo00Yoo`8001=oo`03001oogoo04Uoo`04001oogoo000NOol>0006Ool00`00Oomo
o`0=Ool00`00Oomoo`2fOol20009Ool01000Oomoo`001Goo00<007ooOol07goo0010Ool200000goo
001oo`0jOol00`00Oomoo`0;Ool00`00Oomoo`0@Ool00`00Oomoo`13Ool00`00Oomoo`03Ool01@00
Oomoogoo00006Woo3P000Q240`0017oo00<007ooOol03Goo0P001Goo100000=oo`0000000`0000=o
o`000000XWoo0`0000=oo`00000027oo00@007ooOol0009oo`<0029oo`00:7oo00<007ooOol04goo
0P00?goo00<007ooOol02goo0P0047oo00<007oo0000?Goo10000goo00<007ooOol00goo00<007oo
Ool00Woo00<007ooOol05Goo1@0000<DY@009BT012DY00@002e[;F/]J`P00003Ool007oo00moo`80
00Aoo`04001oogoo0003Ool01000Oomoo`000Woo00<007ooOol0WGoo0P000Woo00@007oo000000Qo
o`04001oogoo0002Ool00`00Oomoo`0ROol0025oo`@000=oo`03001oogoo011oo`<0009oo`03001o
ogoo03aoo`03001oogoo00]oo`80011oo`03001oo`0004Aoo`03001oogoo00=oo`03001oogoo009o
o`03001oogoo01=oo`<000@DY@03000U:BDY00@U:@03000]Jbe[00@]J`03000e[CF]00@0015oo`03
001oogoo009oo`04001oogoo0004Ool00`00Oomoo`02Ool00`00Oomoo`1lOol400000goo001oo`02
000EOol30006Ool00`00Ool00007Ool40003Ool3000QOol002Qoo`03001oogoo00ioo`@0049oo`03
001oogoo00aoo`03001oogoo00ioo`03001oo`0004Aoo`03001oogoo009oo`03001oogoo00Aoo`03
001oogoo015oo`030008@P0000DDY@04000U:BDY9BT70004;F/00`00=Jde[@02=Jd3000?Ool20003
Ool01000Oomoo`001Goo1000Ogoo00@007ooOol0009oo`04001oogoo000AOol30009Ool01000Oomo
o`00<Goo000XOol00`00Oomoo`0;Ool30016Ool00`00Oomoo`0=Ool00`00Oomoo`03Ool40005Ool0
1@00Oomoogoo0000@Woo0P0017oo00<007ooOol01Goo00<007ooOol03Woo0P0000<8@P005:D011BU
1@000cW>00@0048@@Q1240H000<e[@800003>Lh0000000ioo`05001oogooOol00002Ool01000Oomo
o`000Woo0P00PGoo00@007ooOol000Eoo`03001oogoo00aoo`@000]oo`04001oogoo000aOol002Mo
o`8000]oo`80009oo`03001oogoo04Aoo`03001oogoo00ioo`03001oogoo009oo`04001oogoo001=
Ool00`00Oomoo`02Ool00`00Oomoo`06Ool00`00Oomoo`0<Ool01P002488@PQ2000DY@@00003;F/0
03W>00DicP030012448@00E24003001:DP0000@000<icP03001oogoo00aoo`03001oogoo00@000=o
o`<00004Ool00000001WOol00`00Ool0000DOol01000Oomoo`000Woo0`003Goo0P0047oo00@007oo
Ool000Yoo`@00003Ool0000000<00003Ool0000001Uoo`00:7oo00<007ooOol01goo1000BGoo00<0
07ooOol03Woo00<007ooOol00Woo00@007ooOol0059oo`8000Qoo`03001oogoo00Yoo`030008@PQ2
0088@P8000D]J`03000icSW>00@icP030012448@00E24003001:DTYB009:DP<0008icP8000ioo`03
001oogoo07Eoo`03001oogoo01=oo`04001oogoo0002Ool00`00Oomoo`0:Ool4000AOol01@00Oomo
ogoo00001Goo1@000Woo00D007ooOomoo`00009oo`04001oogoo000HOol002moo`<004eoo`03001o
ogoo00ioo`8000=oo`04001oogoo001BOol20009Ool00`00Oomoo`07Ool3000224830006;F/00`00
>LhicP04>Lh00`00@Q124005@Q000`00BU9:DP02BU800`00BU80000200000cW>0000000=Ool00`00
Oomoo`1eOol00`00Oomoo`0COol40003Ool30006Ool3000EOol01@00Oomoogoo00002Goo00@007oo
Ool000Aoo`03001oogoo009oo`03001oogoo01Ioo`00;Goo0P000Woo00<007ooOol0Bgoo00<007oo
Ool03goo00D007ooOomoo`00009oo`03001oogoo051oo`03001oogoo00Uoo`03001oogoo00Eoo`<0
000324800000008Q2003000]Jbe[00@]J`04000icSW>>Lh70004@Q000`00BU9:DP02BU800`00BU80
0002CW<200000c6<001oo`0<Ool00`00Oomoo`03Ool400000goo001oo`02001XOol00`00Oomoo`0P
Ool300000goo001oo`0EOol00`00Oomoo`02Ool00`00Oomoo`06Ool01000Oomoo`001Goo10006Goo
000[Ool4001@Ool00`00Oomoo`0@Ool00`00Oomoo`04001BOol00`00Oomoo`0:Ool00`00Oomoo`03
Ool500038@P00`00;F/]J`04;F/50003@Q001000BU9:DTYB1P000dYB00@004YBBU8000=>L`03000a
S00000eoo`03001oogoo009oo`04001oogoo0002Ool01000Oomoo`00IWoo00<007oo00007goo0P00
6goo00<007ooOol00Woo0P001goo00@007ooOol0009oo`03001oogoo008001]oo`00:goo0P00DWoo
00<007ooOol047oo0P00Egoo00<007ooOol02goo00@007ooOomoo`D000@Q2003000]Jbe[00@00003
?Nl0048@00E24003001:DTYB00E:DP03001BU00000<0009:DP04001>LdicCW<200000c6<001oo`0;
Ool20003Ool01000Oomoo`001Goo00<007ooOol0Pgoo0`000Woo00<007ooOol067oo00<007ooOol0
0goo00<007ooOol01Goo10000goo0`0000Aoo`00000001Qoo`00:goo0`00>7oo20004Goo00<007oo
Ool04Goo00<007ooOol0EGoo0P003Goo00<007ooOol00P000PaS00<002488@P00R480`001Cg_00<0
048@@Q00148@00<004YBBU801DYB00<005:DDY@00U:D10000dic00@005Je000000eoo`80009oo`04
001oogoo0002Ool3001/Ool4000COol4000OOol00`00Oomoo`04Ool00`00Oomoo`0=Ool00`00Oomo
o`0JOol002]oo`04001oogoo000aOol500000`@Q0000000<000;Ool00`00Oomoo`0BOol00`00Oomo
o`1DOol00`00Oomoo`0=Ool2000336<010008@PQ22480`001cg_00<0048@@Q00148@00<004YBBU80
1DYB00<005:DDY@00U:D00<005:DDY@00P000Tic00D005JeE[EF]@0000aoo`80009oo`04001oogoo
0002Ool00`00Oomoo`1/Ool01000Oomoo`0047oo0`008goo00<007ooOol017oo0P003Goo00<007oo
00006goo000UOol00`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`0ZOolJ0009Ool00`00Oomo
o`0BOol2001EOol00`00Oomoo`0=Ool2000336<010008@P000000S6<00<003g_?Nl01Cg_00<0048@
@Q00148@00<004YBBU801DYB00<005:DDY@00U:D00D005:DDY@005Je00800003CW<005Je009F]@03
001oogoo00Ioo`<00004Ool007ooOol40003Ool3001[Ool01000Oomoo`003Woo0P0000=oo`00Ool0
8goo00<007ooOol01Goo00<007ooOol00goo100017oo00<007oo00006goo000NOol40003Ool00`00
Oomoo`02Ool20004Ool00`00Oomoo`0VOol<00056<H00`008@PQ20038@P90006Ool00`00Oomoo`0C
Ool00`00Oomoo`1COol00`00Oomoo`0=Ool2000336<20004<H`00`00?Nlmk`05?Nl00`00@Q124004
@Q000`00BU9:DP05BU800`00DYABU002DY@01000DYABU0000eJe0P000eJe00<007ooOol00goo0`00
17oo00<007ooOol0MGoo00@007ooOol000]oo`<002Qoo`03001oogoo00Ioo`03001oogoo009oo`04
001oogoo0004Ool00`00Ool0000KOol002Eoo`03001oogoo009oo`03001oogoo00Aoo`03001oogoo
02=oo`<0000312400000008<H`03000HaQS600<HaPP000PQ2003000YBP0000<000Aoo`03001oogoo
01=oo`8005Aoo`03001oogoo00eoo`04000<H`aS36<20006<H`00`00?Nlmk`04?Nl01000@Q12448@
1`0014YB00<005:DDY@00U:D00<005:D00001EJe00@005JeE[EF]@800004Ool000000007Ool00`00
Oomoo`1eOol01000Oomoo`0027oo0`0000=oo`00Ool0:7oo00<007ooOol01Woo0P000goo00@007oo
Ool000=oo`05001oogooOol0000JOol002Eoo`03001oogoo009oo`03001oogoo00Eoo`03001oogoo
021oo`H000<<H`L000DYBP03000aS36<00H000<Q2004000YBRU::DX30002Ool00`00Oomoo`0DOol0
0`00Oomoo`1AOol00`00Oomoo`0=Ool200001@aS000002DY00001S6<00@003g_?Nlmk`L000=:DP04
001>LdicCW<60003DY@00`00DY@00005E[D20002E[D3000:Ool00`00Oomoo`1eOol40006Ool3000]
Ool00`00Oomoo`07Ool20002Ool01000Oomoo`008Woo000TOol20004Ool20007Ool00`00Oomoo`0M
Ool4000336<300026<H00`00:DXYBP08:DX00`00<H`aS006<H`50005:DX200000goo001oo`0EOol0
0`00Oomoo`03Ool400000goo001oo`020014Ool00`00Oomoo`0=Ool300039BT00`00<H`aS003<H`5
0003AS400`00BU9:DP04BU800`00CW=>L`05CW<00`00E[D0000300000e:D001F]@04E[D01@00G_L0
0000E[D00P002Woo00<007ooOol0O7oo0`00<Goo00<007ooOol01Woo0P000Woo00@007ooOol0029o
o`009Goo00<007ooOol00Woo00<007ooOol01goo00<007ooOol06Woo100000<<H`0000000P0011S6
00<002U::DX012U:20002C6<00<003W>00000P0012U:0P005Woo0P0017oo00@007ooOol0009oo`04
001oogoo0013Ool00`00Oomoo`0=Ool200049BT01000<H`aS36<0P0000<mk`00AS401DHa00<004YB
BU8014YB00<004icCW<01Dic00<005JeE[D00UJe0`001EJe00@005kgG_MNm`<000Yoo`03001oogoo
07Uoo`@003=oo`03001oogoo00Moo`800003Ool000000080029oo`00:Goo00<007ooOol02Goo00<0
07ooOol067oo1P001aS620001C6<00<003W>>Lh01P000c6<00<003W>>Lh00SW>20005Goo00<007oo
Ool00Woo00@007ooOol000Eoo`03001oogoo045oo`8000eoo`<000@U:@04000aS0000003?Nl00`00
AS56<@04AS400`00BU9:DP04BU800`00CW=>L`05CW<00`00E[EF]@02E[D00`00E[D000020003E[D0
0`00G_MNm`02G_L2000:Ool00`00Oomoo`1HOol400000goo001oo`02000EOol3000gOol00`00Oomo
o`08Ool00`00Oomoo`0UOol002Uoo`8000]oo`03001oogoo01Eoo`D00003248001S600<HaP@00003
9BT0036<00XaS003000icSW>00HicPD000LicP800003:DX0000001Aoo`8000=oo`04001oogoo0002
Ool30014Ool2000=Ool00`0036<000049BT20005?Nl00`00AS56<@04AS400`00BU9:DP04BU800`00
CW=>L`04CW<00`00E[EF]@03E[D00`00E[D00002F]H200000eJe001Nm`03G_L2000;Ool00`00Oomo
o`1BOol400001Goo001oogoo00000Woo00@007ooOol0019oo`800003Ool007oo03Moo`03001oogoo
00Qoo`8002Ioo`00:Goo0P0037oo00<007ooOol04goo1@0000<8@P006<H0100012DY00<0036<<H`0
2S6<00<003W>>Lh02CW>00<0048@00000`0013W>0`0000<YBP00000057oo00@007ooOol0009oo`04
001oogoo0016Ool00`00Oomoo`0;Ool200000`aS000U:@029BT20006?Nl00`00AS56<@04AS400`00
BU9:DP04BU800`00CW=>L`04CW<00`00E[EF]@03E[D00`00E[D00004F]H20004G_L3000:Ool00`00
Oomoo`1GOol01000Oomoo`001Goo00<007ooOol03Goo0`009Woo40001Woo00<007ooOol02Goo0P00
17oo100000=oo`0000000`0000=oo`00000057oo000YOol00`00Oomoo`0<Ool00`00Oomoo`0AOol4
0003248200089BT00`00<H`aS004<H`:0008>Lh00`00@Q124003@Q0600001Cg_000YBRU:00004goo
00<007ooOol010000goo0`00=Goo100000=oo`00Ool00P001Goo00<007ooOol02goo00L000aS36<<
H`009BT00002=Jd00`00?Nlmk`04?Nl00`00AS56<@03AS400`00BU9:DP04BU800`00CW=>L`04CW<0
0`00E[EF]@03E[D20006F]H20002G_L01000IcT000002Woo00<007ooOol0Egoo00@007ooOol0009o
o`<000eoo`<002Moo`@000<8@PD000H@Q003001oogoo00=oo`03001oogoo00Yoo`03001oogoo009o
o`04001oogoo0003Ool01000Oomoo`000Woo00<007ooOol04Goo000YOol00`00Oomoo`0=Ool00`00
Oomoo`0?Ool3000324800`005:D000089BT80006>Lh01000@Q12448@1P000cW>00<0048@@Q001T8@
0P000Sg_0P0000<YBP00Ool04Goo0P00>goo100000Eoo`00Oomoo`00009oo`04001oogoo0004Ool0
0`00Oomoo`0;Ool0100036<<H`aS0P000cF]00<003g_?Nl013g_00<004HaAS400THa4@0015Je0P00
1E[F00H006<HHaP005kg0002IcT00`00Oomoo`08Ool00`00Oomoo`1GOol01000Oomoo`000Woo00<0
07ooOol02Woo1@009Goo1@000PQ20P000b4800<0036<<H`01P0000<@Q00000000goo00<007ooOol0
2Woo0P000goo00@007ooOol000Aoo`03001oogoo009oo`03001oogoo015oo`00:Goo0P003goo00<0
07ooOol037oo0`000`Q20P0000<DY@009BT012DY100000<aS000>Lh02cW>00<0048@@Q001d8@1@00
1d8@10000Sg_0`004Woo00<007ooOol0?Woo00@007ooOol000Eoo`03001oogoo009oo`8000aoo`05
000<H`aS36<00004=Jd00`00?Nlmk`04?Nl00`00AS40000200000dYB001>L`05CW<00`00DYABU004
DY@00`00F]H000050005F]H01000HaQS66<H0P000VLi00<007ooOol027oo00<007ooOol0Egoo1000
0goo0`001goo0P00:7oo100000<8@P0000000P001B4800<0036<<H`01S6<10000Woo00<007ooOol0
2goo0P000Woo00@007ooOol000Eoo`@001Aoo`00:Goo00<007ooOol03goo00<007ooOol02Woo0`00
0PQ20P000aBU00<002DY00000`001C6<00<003W>>Lh02SW>00<0048@@Q002T8@00<004YB00000`00
0d8@0P000TYB0P000Sg_00<007ooOol04Goo00<007ooOol0?Goo00@007ooOol0009oo`<000Eoo`03
001oogoo00]oo`03000<H`aS008000@e[@03000mkcg_00<mk`<000A:DP03001>Ldic00A>L`03001B
U5:D00ABU003001JeU[F00=JeP8000EJeP03001S66<H009S6004001W>FLi000:Ool00`00Oomoo`1U
Ool300000goo001oo`0UOol3000010Q20000000022481P00236<0`003Woo00@007ooOol0009oo`04
001oogoo0002Ool2000FOol002Qoo`03001oogoo015oo`03001oogoo00Qoo`H000DDY@8000TaS003
000icSW>00XicP030012448@00Y24003001:DTYB00=:DP@00003@Q0004YB009:DP800003?Nl007oo
019oo`03001oogoo03aoo`04001oogoo0002Ool00`00Oomoo`05Ool00`00Oomoo`0;Ool00`@Q0000
0002:DX00`00=Jde[@02=Jd00`00?Nl0000200000dHa001:DP05BU800`00CW=>L`03CW<00`00DYAB
U004DY@00`00F]IJeP03F]H40003F]H00`00HaQS6002HaP01@00IcT005kg00002Goo00<007ooOol0
HWoo0`00:Woo00=JeP0000000P001b4810000SF]00@0048@@Q1240H0008aS003000aS00000ioo`80
0003Ool00000008000=oo`<00004Ool00000000COol002Qoo`<0019oo`03001oogoo00Ioo`D000DD
Y@03000Q200000XaS003000icSW>00@icPd000I24003001:DTYB00I:DP<000E:DP<0019oo`8003eo
o`@000=oo`<000Aoo`03001oogoo00]oo`0312400000008YBP03000e[CF]008e[@8000E6<@03001:
DTYB00=:DP03001>Ldic00=>L`03001BU5:D00=BU003001JeU[F00AJeP04001NmekgG_L30004HaP0
1P00Kg/00000G_L000Uoo`03001oogoo04Yoo`03001oogoo019oo`<000=oo`03001oogoo02Ioo`03
F]H00000008000<Q20D000He[@030012448@00M240L000eoo`80025oo`00:7oo00<007ooOol04goo
00<007ooOol01Goo100011BU0P000R4800<0036<<H`01c6<20001Sg_00<004HaAS4014Ha20001dYB
00<004YB00000P001DYB0P004Goo0P00Bgoo00<007ooOol02goo00<48@00:DX00RU:00@003F]=Jde
[@8000I6<@03001:DTYB00=:DP03001>Ldic00=>L`03001BU5:D00=BU003001JeU[F00AJeP03001N
mekg00=Nm`03001S66<H009S6006001_N`00001Nm`002Goo00<007ooOol0@goo10000goo00<007oo
Ool047oo0P0000=oo`000000:goo00AJeP0000000088@P<000`e[@030012448@00U24080008aS080
00aoo`03001oogoo00=oo`@00003Ool007oo008001=oo`006Woo100000=oo`00Ool00P001Goo00<0
07ooOol057oo00<007ooOol00goo10000aBU0P000b4800<0036<<H`0136<100000<e[@00?Nl02cg_
00<004HaAS402THa00<004ic00000P0014YB00<004YB00000U:D0P000dYB0`004Woo00<007ooOol0
17oo00<007ooOol0@7oo0P003Goo00<48@00:DX00RU:00@003F]=Jd0009240030016<DHa00=6<@03
001:DTYB00=:DP03001>Ldic00=>L`03001BU5:D00=BU003001JeU[F00=JeP03001Nm`0000=Nm`03
001W>@0000=S6006001_Nfmk001Nm`002Goo00<007ooOol0BWoo00<007ooOol03Goo0`00;goo0`00
0`Q200<Li`00=Jd03CF]00<0048@@Q002D8@00@005:D0000008aS003001oogoo00Uoo`8000Aoo`04
001oogoo0002Ool01000Oomoo`004Woo000EOol400001Goo001oogoo00000Woo00@007ooOol000Ao
o`8001Ioo`04001oogooOol500000aBU000000058@P00`00<H`aS0040005=Jd00`00?Nlmk`0:?Nl0
0`00AS56<@0:AS400`00CW=>L`02CW<300001DYB001:DTYB00000e:D0P0000=:DP00BU800P0047oo
0P001Woo00<007ooOol0@7oo00<007ooOol037oo00<48@0000000bU:0P000d8@00<004HaAS400dHa
00<004YBBU8014YB00<004icCW<00Tic00<005:DDY@00e:D00<005[FF]H00e[F00<005kg00000ekg
00@006LiIcT0009S6006001_Nfmk001Nm`002Goo00<007ooOol0BWoo00<007ooOol02Woo0`00<Goo
0`000PQ200@001cW7>L000/e[@L000U240800005DY@0036<<H`0000;Ool20003Ool01000Oomoo`00
1Goo00<007ooOol047oo000JOol01000Oomoo`001Goo00<007ooOol00Woo00<007ooOol05Woo00<0
07ooOol01P001b480`002CF]00<003g_?Nl02Sg_00<004HaAS402THa00<004icCW<01Dic0`0000=:
DP00DY@015:D0`0000=:DP0000004Goo00<007ooOol00goo00<007ooOol0@7oo00<007ooOol037oo
00<001BU00000bU:00<0048@@Q000d8@00<004HaAS400dHa00@004YBBU9:DPl000AJeP03001Nm`00
009Nm`03001W>FLi009W>@80009_N`<000Yoo`03001oogoo04Uoo`8000Uoo`<00003Ool007oo031o
o`<00088@P05000LiacW7>L00003=Jd80003?Nl01000CW=>Ldic1P000d8@0P000U:D00<0036<0000
37oo00D007ooOomoo`00009oo`03001oogoo00<001=oo`006Woo00@007ooOol0009oo`<000Eoo`03
001oogoo01Moo`L000HQ2003000YBP0000/e[@03000mkcg_00@mk`X000U6<@03001>Ldic00I>L`04
001>L`000007DY@01000BU80000047oo0P001Goo00<007ooOol0@7oo00<007ooOol037oo00P001BU
5:D002U:000icP00148@00<004HaAS400dHa10000U:D00<005:DDY@00U:D00<005JeE[D00eJe00H0
05[F000005[F0002G_L01000G_MNm`001FLi00H006mkKg/007>L000:Ool00`00Oomoo`1:Ool00`00
Oomoo`05Ool3000dOol3000224800`007>LLi`027>L4000;?Nl00`00CW=>L`07CW<50003DY@3000:
Ool30002Ool01000Oomoo`000Woo00<007ooOol04goo000JOol01000Oomoo`000Woo00<007ooOol0
17oo00<007ooOol067oo1@0000<8@P008@P00b480P000RU:00<003F]=Jd023F]20001THa00@004YB
BU9:DPH000=6<@03001>Ldic00M>L`03001>L`00008000EBU0800003BU800000011oo`8000=oo`80
049oo`8000eoo`08000DYABU000YBP00>Lh000E240040016<DHaAS420005DY@00`00DYABU002DY@0
0`00E[EF]@02E[D00`00F]IJeP02F]H20002G_L00`00G_L00005IcT01`00Kg/006mk001cW00000Yo
o`03001oogoo04moo`<0009oo`03001oogoo035oo`8000<8@P07000LiacW7>L002U:00003Cg_00<0
04icCW<02dic0P001E:D00<007ooOol01Woo0`0000Aoo`00Oomoo`@000=oo`<0019oo`006Woo1000
0goo0`000goo0P006Goo100000<8@P008@P00R480P0012U:00<003F]=Jd013F]100000<mk`00AS40
2dHa00<004YBBU801dYB1@0024ic00<004ic00000UJe0P000e:D00D005Je001:DP0000eoo`<00003
Ool007oo00=oo`03001oogoo041oo`03001oogoo00aoo`05000DYABU5:D00003>Lh00`00@Q124002
@Q001000AS4000000Tic00<005:DDY@00U:D00<005:DDY@00U:D00<005JeE[D00UJe00<005[FF]H0
0e[F100000=Nm`00IcT016Li00@006mk0000009cW003001oogoo00Uoo`03001oogoo04aoo`@003Mo
o`030008@PQ20080008Li`80008YBP03000mkcg_00/mk`03001>Ldic00]>L`@000ABU003001oogoo
009oo`<000Aoo`03001oogoo01aoo`009goo00<007ooOol067oo0P000`Q200@00248000000HYBP03
000e[CF]00@000Dmk`030016<DHa00Y6<@03001:DTYB00Y:DP03001BU00000<000A>L`03001>L`00
00AF]@80009BU005001F]@00BU800009Ool30004Ool00`00Oomoo`15Ool00`00Oomoo`0=Ool01000
5:DDY@0013W>00<0048@@Q000T8@00<004icCW<00Tic00<005:DDY@00U:D00<005:DDY@00U:D00<0
05JeE[D00UJe00<005[FF]H00U[F00@005kgG_MNm`8000EW>@06001_NfmkKg/007>L0P002goo00<0
07ooOol0C7oo0P00>7oo0P0000<8@P007>L00QcW00<002U::DX00RU:00<003g_?Nl01Cg_2@002dic
0P000Ukg00H005:DDYABU000Ool30007Ool00`00Oomoo`0LOol002Moo`03001oogoo01Moo`8000@8
@P8000PYBP<000Tmk`030016<DHa00Y6<@03001:DTYB00Y:DP03001BU5:D00=BU0D00003CW<005Je
00EF]@<0009F]@8000Ioo`<000Moo`03001oogoo04Eoo`03001oogoo00eoo`03000DYABU008000<i
cP030012448@00800003BU8004ic009>L`03001BU5:D00=BU005001BU5:DDY@00004E[D00`00F]IJ
eP02F]H01`00G_MNmekg001W>@0000=W>@04001_NfmkKg/200000g>L001oo`0;Ool00`00Oomoo`1<
Ool00`00Ool0000POol5000BOol300027>L20005:DX80006AS400`00DYABU0040007CW<20003G_L0
0`00DYABU002000:Ool00`00Oomoo`0LOol002Moo`03001oogoo01Ioo`8000@8@P8000LYBP<000/m
k`030016<DHa00Y6<@03001:DTYB00Y:DP03001BU5:D00MBU0<000QF]@04001F]EJeE[D20002Ool3
000;Ool00`00Oomoo`14Ool00`00Oomoo`0>Ool20002<H`01000>LhicSW>0P000dYB00<004icCW<0
0Tic00<005:DDY@00U:D00D005:DDYABU00000=F]@03001JeU[F009JeP03001Nmekg009Nm`03001W
>FLi00800006IcT006mkKg/007>L0`0037oo00<007ooOol0?Woo100000=oo`00Ool00P001Goo0P00
00=oo`00Ool06Goo1P000bU:1P003Goo1@001BU:00<003F]00003DHa00<005:DDY@015:D1@000Tic
0P0015kg0P0000=BU000Ool02Woo00<007ooOol06goo000WOol00`00Oomoo`0FOol2000324800`00
6<H00005:DX20002<H`00`00?Nlmk`09?Nl00`00AS56<@04AS4=0006BU800`00DYABU007DY@00`00
DY@000020006E[D20003E[D3000>Ool00`00Oomoo`14Ool2000?Ool20002<H`01P00>LhicP00AS40
00A:DP05001>LdicCW<00004DY@01@00DYABU5:D00000eJe00<005[FF]H00U[F00@005kgG_MNm`80
00AW>@04001_Nfmk0002Li`2000=Ool00`00Oomoo`0iOol400001Goo001oogoo00000Woo00@007oo
Ool000Aoo`05001oogooOol0000GOol50003;F/400000bU:00000004000;Ool30005:DX20002=Jd0
0`00AS56<@0:AS400`00DYABU00:DY@40006G_L2000;Ool00`00Oomoo`0KOol002Ioo`03001oogoo
01Ioo`<0000324800000008HaP04000YBRU::DX20004<H`00`00?Nlmk`08?Nl80006AS400`00CW=>
L`04CW<80008DY@00`00DY@00002F]H20004E[D00`00G_L00003E[D00`00Oomoo`0=Ool00`00Oomo
o`14Ool00`00Oomoo`0>Ool20003<H`00`00>Lh00002AS400`00BU9:DP02BU801@00CW=>Ldic0000
0U:D2`000U[F00<005kgG_L00Ukg0P000fLi0P000Vmk00<007>LLi`00P003Goo00<007ooOol0?Woo
00@007ooOol000Eoo`04001oogooOol20004Ool00`00Oomoo`0AOol:0005;F/90008Ool40003:DX2
0004=Jd00`00AS56<@0:AS400`00DYABU00<DY@20007G_L00`00Oomoo`09Ool00`00Oomoo`0KOol0
02Ioo`03001oogoo01Ioo`800003248001S600<HaP04000YBP000006<H`00`00?Nlmk`04?Nl40000
0d8@0016<@0;AS400`00CW=>L`0:CW<00`00E[D000020005DY@00`00DY@00004F]H20002E[D00`00
G_MNm`0200000eJe001oo`0>Ool00`00Oomoo`14Ool00`00Oomoo`0?Ool20003<H`00`00AS56<@02
AS401@00BU9:DTYB00000Tic100000=F]@00F]H00U[F00H005[FF]IJeP00G_L30003G_L00`00IcT0
0002IcT00`00JeY[FP020003Li`2000=Ool00`00Oomoo`0nOol01000Oomoo`000Woo0`0017oo00<0
07ooOol017oo00<007ooOol03Woo1P001S6<1`0000A6<@000016<@D000Ioo`@00003:DX0000000He
[@030016<DHa00Y6<@03001BU5:D00]BU003001S6000008000ENm`03001oogoo00Uoo`03001oogoo
01]oo`009Woo0P005Woo10001AS60P00236<00<003g_?Nl010001D8@00<004HaAS402THa00<004ic
CW<02Tic00<005JeE[D00UJe0`000U:D00<005:D00001U[F0`0015kg0`003Woo00<007ooOol0A7oo
00<007ooOol03goo00<0024800000S6<0P0014Ha00@004YBBU9:DP80009BU005001F]EJeE[D00002
F]H01000F]IJeP000ekg00X006<H000005kg001W>@00IcT000=[FP04001cW7>LLi`2000>Ool00`00
Oomoo`0nOol01000Oomoo`000Woo00<007ooOol017oo0P001Woo00<007ooOol02goo0`0000<48@00
00003@0000=6<@0000000THa0`0000=6<@0000000P0017oo0P0000<48@0000001cF]400035:D00H0
06<HHaP006<H0004G_L2000:Ool00`00Oomoo`0KOol002Ioo`03001oogoo01Eoo`<000DHaP8000Ta
S0<000U240030016<DHa00Y6<@03001>Ldic00Y>L`03001F]EJe00EF]@<00003DY@005[F00MJeP03
001Nmekg00=Nm`8000ioo`03001oogoo04=oo`03001oogoo015oo`07000Q2000<H`0048@000014Ha
00D004YB001BU000009BU005001F]EJeE[D00002F]H01000F]IJeP000ekg00D006<HHaQS6000009W
>@8000=[FP80009cW0<000ioo`03001oogoo03ioo`@000=oo`<000=oo`03001oogoo00Ioo`03001o
ogoo00Uoo`0300048@@Q00<000/e[@03001:DTYB00800003AS4000000096<@800003AS4004YB0080
00=oo`@000De[@<00003>Lh004YB00]:DP03001F]@00008000UBU004001S66<H0002HaP01000G_MN
mekg0P002Woo00<007ooOol06goo000VOol00`00Oomoo`0DOol400046<H00`009BT00007<H`3000<
@Q000`00AS56<@09AS400`00CW=>L`09CW<00`00E[EF]@08E[D30008F]H20005G_L2000=Ool00`00
Oomoo`13Ool3000AOol00`008@PQ20020002@Q020003AS401@00DYABU5:D00000U:D00@005JeE[D0
009JeP04001JeU[F0002G_L02@00HaQS66<H001S6000IcT00003JeX00`00Kg]_N`0200000gNm001o
o`0>Ool00`00Oomoo`1;Ool20008Ool00`00Oomoo`07Ool01000124000000PaS00<003F]=Jd02CF]
00<004YBBU800TYB0P0000A6<@000016<@L00005Ool0000012400004=Jd20004>Lh00`00BU9:DP09
BU800`00E[EF]@03E[D40005DY@01000HaQS60000f<H0P0000=Nm`0000002Woo00<007ooOol06goo
000VOol00`00Oomoo`0DOol400026<H200029BT00`00<H`aS003<H`20002>Lh00`00@Q12400:@Q00
0`00AS56<@04AS480009CW<00`00E[EF]@08E[D00`00E[D000020006F]H00`00HaP00004G_L2000=
Ool00`00Oomoo`13Ool00`00Oomoo`0BOol010008@PQ2000148@00D004Ha001BU000009BU007001B
U5:DDY@005Je00000U[F00H005[FF]H005kg0003HaP01000HaQS66<H0P0000=[FP0000000fmk00@0
07NmMkd000moo`03001oogoo04]oo`03001oogoo00Qoo`03001oogoo00Eoo`<000<<H`l000E:DPD0
0003AS40000000P00004=Jd000000005>Lh00`00BU9:DP0:BU800`00E[EF]@07E[D300000e:D001S
6002HaP00`00HaQS6003HaP3000:Ool00`00Oomoo`0KOol002Ioo`8001Eoo`800004124001S60005
9BT00`00<H`aS0020004>Lh00`00@Q124009@Q080005BU800`00DYABU0060003CW<00`00E[EF]@08
E[D00`00E[D00002G_L20004F]H00`00HaQS60020002G_L3000=Ool00`00Oomoo`0jOol40004Ool0
0`00Oomoo`0COol00`008@P000020003@Q001@00DYABU5:D00000U:D00<005:DDY@020000Ukg00@0
06<H0000009S60D000=_N`<00003Mkd007oo00moo`03001oogoo04]oo`8000Yoo`03001oogoo00=o
o`8000<<H`03000HaP0000/icP03001:DTYB00@000A:DP@00003AS40000000<00004124000000008
>Lh00`00BU9:DP0:BU800`00E[EF]@0:E[D20003HaP00`00HaQS6004HaP30009Ool00`00Oomoo`0K
Ool002Eoo`03001oogoo01Eoo`8000031240000000HU:@<000LicP030012448@00A240@00003AS40
04YB00Y:DP03001BU5:D00IBU0D000UF]@8000ENm`80009JeP03001S66<H009S6003001Nm`000080
00eoo`03001oogoo03Yoo`04001oogoo0004Ool00`00Oomoo`0DOol20002?Nl00`00@Q1240030002
DY@8000248@00`005:DDY@0600000f<H001W>@02IcT02@00Kg]_Nfmk001g_@00Mkd0000AOol00`00
Oomoo`1;Ool00`00Oomoo`0:Ool01@00Oomoogoo00000`aS0P0000<HaP00>Lh023W>1P0014YB1000
0TYB0`0000=JeP0000001@001cW>20001TYB00<005JeE[D02eJe00<006<H00000`001V<H00<005kg
00002Woo00<007ooOol06Woo000MOol40004Ool00`00Oomoo`0DOol2000212400`009BTU:@059BT0
0`00>LhicP07>Lh00`00@Q1240040005AS400`00BU9:DP09BU800`00DYABU009DY@00`00E[D00003
0005E[D20007G_L30005HaP200000ekg001oo`0<Ool00`00Oomoo`0jOol01000Oomoo`0017oo00<0
07ooOol05Goo00@003g_?Nlmk`<0009BU0@000032480012400D@Q003000DYABU00@DY@03000HaP00
00<00003JeX006mk0080009g_@80019oo`03001oogoo03Qoo`@00003Ool0000000<00003Ool00000
00Ioo`03001oogoo00]oo`05001oogoo000<H`0200036<H01000>LhicSW>1P000cg_00<004YBBU80
100014YB0`0000AJeP00001JePH0008icPD0008mk`03001:DTYB00=:DPP000aF]@D00003HaP006<H
00=S60800003G_L007oo00Uoo`03001oogoo01Yoo`007Goo00@007ooOol000Aoo`03001oogoo01Ao
o`8000031240000000DU:@<000PicP@000U6<@03001:DTYB00U:DP03001BU5:D00UBU003001F]EJe
00=F]@@00003E[D0000000UNm`03001S66<H00AS6003001Nm`0000eoo`03001oogoo039oo`@000Ao
o`04001oogoo0004Ool2000FOol30003?Nl40003248B00026<H40004Mkd00`00Oomoo`0AOol00`00
Oomoo`0dOol50002Ool01@00Oomoogoo00000Woo00@007ooOol000Aoo`03001oogoo00eoo`@000DH
aP@000Tmk`03001:DTYB00A:DP@0009:DPT000045:D000000006?Nl00`00BU9:DP09BU800`00E[EF
]@030007E[D30004HaP30003HaP01000Li`000002Woo00<007ooOol06Woo000MOol01000Oomoo`00
17oo0P005Goo00<000@Q12400P0012DY00@002e[;F/000HicP800003?Nl004Ha00Y6<@03001:DTYB
00U:DP03001BU5:D00UBU003001F]EJe00MF]@<000UNm`8000ES6004001Nmekg000<Ool00`00Oomo
o`0cOol00`00Oomoo`04Ool01000Oomoo`0017oo00<007ooOol05Woo00<003W>00002@0000<Li`00
:DX01BU:00<0036<<H`01C6<00<003W>00002@005Goo00<007ooOol0=goo00@007ooOol000Aoo`03
001oogoo009oo`03001oogoo009oo`8000moo`03000HaQS600<HaP<000Xmk`T000A:DP<0009JePP0
00Pmk`03001:DTYB00U:DP03001F]EJe00=F]@L00003E[D006<H00IS6006001S66<HHaP007>L0P00
2Woo00<007ooOol06Woo000DOol40005Ool01000Oomoo`0017oo00<007ooOol057oo00<000@Q0000
0PQ200@002DY9BT000@]J`04000icSW>>Lh20004?Nl00`00AS56<@08AS400`00BU9:DP09BU800`00
DYABU009DY@00`00E[EF]@09E[D30006G_L01000IcUW>@0016<H0P0000=Nm`00Ool02goo00<007oo
Ool0<goo00<007ooOol017oo100017oo00<007ooOol05goo1`0011cW00<002U::DX01BU:00<0036<
<H`01C6<00<003W>>Lh00SW>0P000QBU0P005Woo00<007ooOol0=goo00@007ooOol000Eoo`@000Eo
o`03001oogoo00ioo`03000HaQS6008HaP04000YBRU:0006?Nl40002@Q000`00CW=>L`04CW<90000
0e[F0000000300000aS6000mk`06?Nl00`00BU9:DP09BU800`00E[EF]@04E[D600000eJe001S6008
HaP30002Li`2000:Ool00`00Oomoo`0JOol001Eoo`03001oogoo00Eoo`04001oogoo0004Ool00`00
Oomoo`0DOol00`001240000224830005;F/01000>Lh000001Sg_00<004HaAS4024Ha00<004YBBU80
14YB2`001U:D00<005JeE[D025Je0P000Ukg0P0015kg00D006LiIcUW>@00009S6005001[FP00G_L0
000<Ool00`00Oomoo`0dOol00`00Oomoo`0;Ool00`00Oomoo`0HOol300000`aS000Li`047>L00`00
:DXYBP06:DX00`00<H`aS005<H`00`00>LhicP02>Lh300000aBU001oo`0EOol00`00Oomoo`0gOol0
1000Oomoo`000Woo00<007ooOol00P001goo0P003Woo0P000QS60P000bU:00<003g_?Nl010001d8@
00<004icCW<02Dic00<005[F00001P0000<HaP00?Nl00cg_1P002DYB00@005JeE[EF]@<000IF]@80
00YS6080009cW08000Yoo`03001oogoo01Yoo`005Goo00<007ooOol01Goo10000goo0P005Woo0P00
0`Q200<002e[;F/01Be[0P0023g_00<004HaAS401THa2@001Dic00<005:DDY@00e:D20002EJe0P00
15kg0P0000=Nm`00IcT016Li00<006<H00000V]J0P003Goo00<007ooOol0<Goo100037oo00<007oo
Ool05Woo0`000PaS00<001cW7>L00acW3P00136<00<003W>>Lh00SW>0P0000@mk`00000001Eoo`03
001oogoo03Moo`@000=oo`<00004Ool000000004Ool00`00Oomoo`0=Ool010006<HHaP001BU:0`00
2T8@5@0000<HaP00000010000cg_00<004icCW<01P000TYB00<005Je00000P002EJe00<006<H0000
2F<H0P000g>L00<007ooOol027oo00<007ooOol06Woo000FOol00`00Oomoo`0;Ool2000FOol20002
24820006;F/2000:?Nl00`00AS56<@050002BU800`00CW=>L`09CW<00`00DYABU009DY@00`00F]H0
00020006E[D20006G_L20006IcT20002JeX2000=Ool00`00Oomoo`11Ool00`00Oomoo`0EOol20004
36<00`007>L0000300000c6<000icP07>Lh01000@Q12448@1P000cW>0P000cg_0P0057oo00<007oo
Ool0BGoo0P003Goo100012U:0`002D8@0`000THa00<004icCW<024ic2@0000<DY@00?Nl01Sg_00<0
04icCW<01Tic100035Je00<006<HHaP00P001f<H0`000W>L00<007ooOol027oo00<007ooOol06Woo
000DOol4000<Ool00`00Oomoo`0EOol010002488@PQ20P001Be[00@003F]=Jd000Pmk`@000M:DP03
001>Ldic00U>L`03001BU5:D00UBU003001JeU[F009JeP@00004E[D005kg0007G_L20006IcT01@00
JeY[FV]J00003Goo00<007ooOol0@Goo00<007ooOol04goo10000`aS0`001C6<00<003W>>Lh01SW>
00<0048@@Q001D8@00<004YB00000`0000<icP00?Nl00Sg_0`004Woo00<007ooOol0BGoo00<007oo
Ool037oo0P001BU:00D003F]=Jde[@0000E240<000E6<@03001>Ldic009>L`H0009BU004001JeU[F
F]H200025:D00`00?Nlmk`04?Nl00`00CW=>L`04CW<50002E[D30008E[D00`00HaQS6003HaP00`00
HaQS6003HaP01000Mkeg_@000W>L00<007ooOol027oo00<007ooOol06Woo000TOol00`00Oomoo`0E
Ool01P002488@P006<H000<]J`8000<e[@03000mkcg_00@mk`800003@Q0004YB00Q:DP03001>Ldic
00U>L`03001BU5:D00UBU003001JeU[F00IJeP@000MNm`03001W>@0000EW>@05001[FV]JJeX0000=
Ool00`00Oomoo`10Ool00`00Oomoo`0COol4000236<200000b48000aS005<H`00`00>LhicP06>Lh0
0`00@Q124005@Q000`00BU9:DP02BU830004?Nl3000AOol00`00Oomoo`19Ool2000<Ool30003:DX2
0004=Jd00`00@Q1240030008AS450008DY@<0003?Nl01000CW=>Ldic0`0014ic00<005JeE[D00eJe
0P001EJe00<006<HHaP01F<H00D006<HHaQS600000=g_@<000aoo`03001oogoo01Uoo`0097oo0P00
5Woo00@000Q22480008HaP03000]J`0000De[@03000mkcg_008mk`8000=24003001:DTYB00Q:DP03
001>Ldic00Q>L`03001BU5:D00QBU003001JeU[F00QJeP04001Nm`000005G_L01000IcUW>FLi0P00
0fLi0P000V]J00<007ooOol02goo00<007ooOol0@7oo00<007ooOol04Woo1`000b4800<0036<<H`0
136<00<003W>>Lh01SW>00<0048@@Q001D8@00<004YBBU800TYB00<004YB00000P000Sg_00@003W>
>Lh0011oo`03001oogoo04Uoo`03001oogoo00]oo`<0008YBP03000e[CF]00De[@8000U6<@80009:
DP03001BU5:D00ABU0@00003DY@0000000@000035:D003W>008icPL000I>L`03001F]EJe00IF]@<0
009F]@03001S66<H00IS60800003HaP007Nm009g_@03001cW00000aoo`03001oogoo01Uoo`0097oo
00<007ooOol05Goo00<48@0000000aS60P001cF]00@003g_000000E24003001:DTYB00Q:DP03001>
Ldic00Q>L`03001BU5:D00QBU003001JeU[F00MJeP05001Nm`00001Nm`020003G_L00`00IcUW>@03
IcT01@00IcT006mk00000V]J00<007ooOol02goo00<007ooOol0@7oo00<007ooOol04Woo1@001B48
00<0036<<H`0136<00<003W>>Lh01SW>00<0048@@Q001D8@00<004YBBU800TYB00<004YB00000Tic
0P0000Dmk`00>LhicP0000moo`03001oogoo045oo`@000=oo`03001oogoo00aoo`D000Le[@8000Q6
<@8000A:DP03001BU5:D00@000=BU080009F]@03001JeP0000@000<icP<0009:DP@000=>L`03001F
]EJe00UF]@8000]S6003001g_GNm00=g_@8000aoo`03001oogoo01Uoo`0097oo00<007ooOol05Goo
00<48@0000000aS600<003F]=Jd01SF]0P00248@00<004YBBU801dYB00<004icCW<00dic2`001E:D
00<005[FF]H01U[F0`000Ukg00<005kgG_L00P0000=Nm`00IcT01FLi0P000Vmk0`003Goo00<007oo
Ool0@7oo0P004Woo1@001B4800<0036<<H`01C6<00<003W>>Lh00SW>2000148@00<004YBBU800TYB
00<004YB000014ic0P000SW>00<007ooOol03Goo00<007ooOol0@Goo00@007ooOol000=oo`8000eo
o`0300048@0000Le[@80008mk`030016<DHa0096<@<000M:DP8000EBU080009F]@800003F]H00000
00<00006<H`00000000icP001dYB10002eJe00@006<H000000QS608000Eg_@03001oo`0000]oo`03
001oogoo01Uoo`008goo00<007ooOol05Woo00<48@006<H00QS60P001cF]0P002D8@00<004YBBU80
1DYB20001E:D00<005JeE[D00eJe1`0015[F0`0000=JeP00G_L00ekg00@005kgG_MNm`8000MW>@03
001_Nfmk00<000eoo`03001oogoo041oo`03001oogoo015oo`<0000336<0024800@Q2003000aS36<
00@aS0H000A24004001:DTYBBU860003BU801000BU9:DP0014ic0`0000<icP00Ool03Goo00<007oo
Ool0@Goo00@007ooOol000=oo`03001oogoo00]oo`800003124003F]00De[@03000mkcg_008mk`03
0016<DHa008000Q:DP800003CW<00000009BU080009F]@80009JePL0008aS0<000Q:DP03001F]EJe
008000QF]@03001S66<H009S6003001S66<H00AS6004001cW7>L0003Mkd01@00Oomoogoo00002Woo
00<007ooOol06Goo000SOol2000GOol00`@Q000HaP026<H00`009BT00005=Jd01000>LhicP00248@
00<004YBBU801@000TYB00<005:DDY@025:D00<005JeE[D025Je00<005[F00000`0015[F00<005kg
G_L00Ukg00L005kgG_MNm`00IcT00005IcT20003Kg/2000>Ool00`00Oomoo`0oOol00`00Oomoo`0@
Ool2000236<00`008@PQ20048@P00`00<H`aS00400000cW>00124007@Q000`00BU9:DP05BU800`00
DY@000030002BU800`00CW=>L`02CW<00`00E[EF]@02000>Ool00`00Oomoo`0jOol40003Ool01000
Oomoo`000goo0P0037oo00D000@Q000@Q00000@e[@03000mkcg_00@mk`8000Q:DP8000E>L`8000=F
]@03001JeU[F00X00004<H`004HaAS420005BU800`00E[EF]@03E[D20005E[D00`00HaQS6004HaP0
0`00HaQS6002HaP01@00LiacW7>L00000WNm00<007ooOol00goo00<007ooOol01goo00<007ooOol0
6Goo000SOol00`00Oomoo`0FOol01@@Q000HaQS600000RDY00@003F]=Jde[@8000<icP030012448@
00E240@000M:DP03001BU5:D00QBU003001F]EJe00MF]@<000=JeP<00003F]H005kg00ENm`04001N
mekg0002IcT00`00IcUW>@02IcT00`00Kg/00002Kg/2000>Ool00`00Oomoo`0oOol00`00Oomoo`0@
Ool01@0036<<H`aS00001B4810001CW>00<0048@@Q001T8@00<004YBBU801DYB00<005:DDY@00U:D
100014ic00<005JeE[D00UJe00<007ooOol02goo00<007ooOol0>goo00<007ooOol00goo00@007oo
Ool000=oo`03001oogoo00]oo`0500048@0048@00002=Jd20006?Nl00`00AS400006BU820005CW<2
0002DY@30002F]H=0004AS430002BU800`00E[EF]@05E[D20002E[D00`00HaQS6006HaP200000f<H
001cW003Li`01000Mkeg_@001Goo00<007ooOol01goo00<007ooOol06Goo000SOol00`00Oomoo`0F
Ool010@Q000000000bDY00@003F]=Jd000DicP030012448@00=240800003AS4004YB00Q:DP03001B
U5:D00QBU003001F]EJe00AF]@@000QJeP<000INm`8000=W>@05001W>FLiIcT00002Kg/00`00Kg]_
N`02000>Ool00`00Oomoo`0oOol00`00Oomoo`0?Ool2000336<00`008@P0000300000c6<000icP06
>Lh00`00@Q124006@Q000`00BU9:DP05BU800`00DYABU002DY@00`00DYABU0030003CW<01@00E[EF
]EJe00003Goo00<007ooOol0>goo00<007ooOol00goo10000goo0P0037oo0P000a240P001cg_00@0
04HaAS56<@80009:DP8000I>L`03001BU5:D009BU003001F]@0000H00003HaP0000000D00003>Lh0
000000E6<@8000YF]@8000]S6003001cW7>L00AcW08000Moo`03001oogoo00Ioo`03001oogoo01Uo
o`008goo00<007ooOol05Woo00@000@Q000000@U:@8000LicP0300124000008000A6<@03001:DTYB
00M:DP03001BU5:D00MBU004001F]EJeE[D30003E[D00`00F]IJeP07F]H01000HaP000001Ekg00<0
06LiIcT00VLi00@006LiIcT000=_N`@000ioo`03001oogoo03moo`03001oogoo00ioo`<000<<H`80
00DaS003000icSW>00DicP030012448@00I24003001:DTYB00E:DP03001BU5:D009BU004001BU5:D
0002E[D200000dic001F]@02E[D2000<Ool00`00Oomoo`0lOol00`00Oomoo`09Ool00`00Oomoo`0;
Ool2000348@00`00?Nlmk`05?Nl00`00AS56<@04AS420006CW<20004DY@300001UJe001JeU[F001N
m`P000047>L003W>>Lh20002AS420002DY@20006E[D200000ekg00000007HaP30006Li`00`00Oomo
o`07Ool00`00Oomoo`06Ool00`00Oomoo`0HOol002=oo`8001Moo`0300048@0000DU:@03000icSW>
00HicP8000M6<@03001:DTYB00M:DP03001BU5:D00MBU0@000IF]@03001JeU[F00IJeP03001S66<H
009S60<00004G_L006Li0004IcT20004Kg/4000>Ool00`00Oomoo`0oOol2000?Ool2000336<20006
<H`00`00>LhicP05>Lh00`00@Q124006@Q000`00BU9:DP05BU800`00DYABU002DY@01000DYABU000
15Je0P000eJe0P003Goo00<007ooOol0>Goo10002Woo00<007ooOol02goo0P000a240P0013g_0P00
1THa00<004icCW<00P000Tic0P000e:D0`000eJe00H005[FF]H005kg0002IcT90002>Lh30006DY@2
0002E[D20005G_L00`00HaQS6003HaP01@00Kg]_Nfmk000017>L00<007ooOol02Goo00<007ooOol0
1Goo00<007ooOol067oo000ROol00`00Oomoo`0GOol01000124000000bDY00<0036<00001SW>0P00
2DHa00<004YBBU801DYB400015Je00<005[FF]H01e[F00<006<HHaP01F<H0P0000=W>@00IcT016Li
00<006mkKg/00fmk0P003goo00<007ooOol0?goo00<007ooOol03Goo0`0000D<H`00000U:@0000Ha
S003000icSW>00DicP030012448@00I24003001:DTYB00E:DP03001BU5:D00=BU003001BU00000EF
]@8000=F]@8000aoo`03001oogoo04Moo`03001oogoo00]oo`07000@Q12448@002e[00000cg_00<0
04HaAS401DHa00<004icCW<00dic0P0000=BU00000000`0015Je00@005[FF]IJeP<00003IcT00000
00X00096<@8000MBU08000QNm`800003HaP0000000A_N`05001cW7>LLi`0000=Ool00`00Oomoo`04
Ool00`00Oomoo`0HOol001Aoo`@00003Ool007oo008000Eoo`03001oogoo01Moo`0400048@000003
9BT00`00<H`00005>Lh01000@Q12400024Ha00@004YBBU9:DP@00003CW<005:D009BU0<000=BU003
001F]EJe009F]@L000MJeP03001S66<H00ES6005001W>@00IcT00003IcT00`00Kg/00003Kg/3000?
Ool00`00Oomoo`0oOol00`00Oomoo`0=Ool400039BT00`00<H`aS004<H`00`00>LhicP05>Lh00`00
@Q124002@Q080004BU800`00DYABU003DY@00`00DY@00005E[D01@00G_L00000E[D00P0037oo00<0
07ooOol0AWoo0P003Goo00D0012448@@Q000008]J`03000mk`0000M6<@03001>Ldic00A>L`@0009B
U003001F]EJe009F]@03001JeP0000800005HaP006Li001[FP060002Ool30004AS420002DY@30002
F]H00`00G_MNm`06G_L20006Kg/01000LiacW0003Woo00<007ooOol017oo00<007ooOol067oo000D
Ool01000Oomoo`000Woo00@007ooOol000Aoo`03001oogoo01Moo`0700048@0036<002DY00000c6<
00<003W>>Lh00P000d8@00<004HaAS401THa10001Tic0`001U:D00<005JeE[D01UJe00<005[FF]H0
10000U[F00<006<HHaP01V<H00H006LiIcT006Li0002IcT00`00Kg/00003Kg/3000@Ool00`00Oomo
o`0nOol00`00Oomoo`0=Ool300049BT00`00<H`aS004<H`00`00>LhicP02>Lh80004BU801000CW=>
Ldic1P0015:D00<005:D00001EJe00@005kgG_MNm`<000aoo`03001oogoo04Ioo`03001oogoo00ao
o`04000@Q1240004;F/00`00AS56<@05AS400`00CW=>L`02CW<30004DY@9000015kg001S66<H0P00
00E[FP00000006Li00<000=oo`<000=6<@@000IJeP03001Nmekg00=Nm`800003IcT006mk00I_N`80
011oo`03001oogoo00=oo`03001oogoo01Qoo`0057oo00@007ooOol000Eoo`03001oogoo009oo`80
01Uoo`80000436<002DY0003<H`00`00>Lh00006@Q000`00AS56<@03AS4200000dYB001>L`04CW<4
0008DY@00`00E[EF]@06E[D00`00F]IJeP04F]H40006HaP00`00IcUW>@02IcT01000IcT000000fmk
00@006mk001cW080011oo`03001oogoo03eoo`03001oogoo00eoo`@000DU:@03000aS36<00<aS0D0
00=24003001:DTYB00I:DP03001>Ldic00E>L`03001F]@0000@00003DY@005Je00AF]@03001Nmekg
009Nm`03001BU00000]oo`03001oogoo03Aoo`@00003Ool0000000<00003Ool0000000Eoo`8000io
o`03000@Q00000@]J`<000A6<@03001>Ldic00<00003CW<005:D00ABU003001JeU[F009JeP03001N
mekg009Nm`03001S66<H00800003JeX0000000@000Eoo`D000=>L`03001JeU[F00EJeP03001Nmekg
008000=W>@03001_Nfmk00A_N`80019oo`03001oogoo009oo`03001oogoo01Qoo`0057oo00@007oo
Ool0009oo`<000Eoo`03001oogoo01Qoo`80008<H`03000aS36<00<aS0030012448@00E240030016
<DHa00<000A:DP03001>L`00008000=>L`03001BU5:D00IBU003001F]EJe00EF]@03001JeU[F00IJ
eP03001S66<H008000AS6003001W>FLi00=W>@03001W>@0000=_N`04001_N`00Li`2000@Ool00`00
Oomoo`0mOol3000=Ool200000`aS000U:@049BT01000<H`aS36<0`001d8@00<004YBBU801TYB00<0
04icCW<01Dic00<005JeE[D00eJe0`001EJe00<005kgG_L00Ukg0`002goo00<007ooOol0=7oo00@0
07ooOol000=oo`04001oogoo0002Ool00`00Oomoo`02Ool00`00Oomoo`0=Ool00`0048@00003;F/0
1000?Nlmkcg_0P0000=6<@0000000P000dic00<005:DDY@00e:D00<005[FF]H00U[F00<005kgG_L0
0Ukg00<006<H00000P0000=[FP00IcT01@001Woo100014ic0P0015[F10001VLi00@006mkKg]_N`<0
01Aoo`05001oogooOol0000JOol001Aoo`04001oogoo0002Ool00`00Oomoo`05Ool00`00Oomoo`0I
Ool00`0036<<H`020003<H`00`00>Lh00007@Q020005BU840006CW<00`00DYABU006DY@00`00E[EF
]@05E[D00`00F]IJeP06F]H00`00HaQS6002HaP200000f<H001W>@06IcT20004Kg/01000LiacW000
4Goo00<007ooOol0;goo100000=oo`00Ool00P001Goo00<007ooOol03Goo0P0000<<H`009BT012DY
00@0036<0000008mk`030012448@00E24003001:DTYB00I:DP03001>Ldic00E>L`03001F]EJe00=F
]@03001F]@00008000=F]@03001Nmekg009Nm`<000]oo`03001oogoo03Aoo`04001oogoo0004Ool0
0`00Oomoo`02Ool00`00Oomoo`02Ool00`00Oomoo`0>Ool00`00;F/]J`02;F/00`00?Nlmk`02?Nl3
0005CW<00`00DYABU003DY@00`00F]IJeP03F]H00`00G_MNm`02G_L200000fLi001[FP0500000f]J
00000007Ool=0003G_L00`00IcUW>@04IcT00`00Kg/00003000EOol01@00Oomoogoo00006Woo000D
Ool40003Ool30004Ool00`00Oomoo`0IOol00`0036<<H`020003<H`01000>LhicP001D8@00<004Ha
00000TYB0`000TYB00<004icCW<01Tic00<005:DDY@00U:D1`0015Je00<005[FF]H01E[F00<006<H
HaP01F<H0P001VLi00@006mk0000009_N`80009cW003001oogoo00moo`03001oogoo02moo`04001o
ogoo0002Ool01000Oomoo`0017oo00<007ooOol03Goo00@000aS36<000DU:@8000Dmk`030012448@
00A24003001:DTYB00E:DP03001>Ldic00E>L`03001F]EJe00AF]@8000=Nm`800003E[D005kg00=N
m`04001W>@00000:Ool00`00Oomoo`0dOol01000Oomoo`001Goo10001Goo0P003goo00<002e[;F/0
0Re[1@0000<mk`00BU800P000Tic00<005:DDY@015:D00<005[FF]H00U[F00<005kgG_L00`000VLi
0P0000=[FP00IcT01@002Woo1@000Tic00<005kgG_L015kg00<006LiIcT00VLi100000=[FP00Ool0
5goo00@007ooOol001Yoo`008Goo0P0077oo0P000QcW00<0036<00000cW>00@0048@@Q1240800096
<@<000I:DP03001>Ldic009>L`P000=F]@04001JeU[FF]H60006F]H00`00HaQS6005HaP00`00IcT0
0005IcT01P00Kg/006mk001_N`80009cW003001oogoo00moo`03001oogoo02moo`04001oogoo0005
Ool00`00Oomoo`02Ool00`00Oomoo`0=Ool01@0036<<H`aS00000bDY0P001Sg_00<0048@@Q001D8@
00<004YBBU8014YB00<004icCW<01Dic00<005JeE[D015Je0P001Ekg0P000ekg00D006LiIcUW>@00
00Yoo`03001oogoo03Aoo`04001oogoo0002Ool00`00Oomoo`020007Ool00`00Oomoo`0?Ool00`00
;F/000020004?Nl00`00BU9:DP02BU840004DY@00`00F]IJeP060002G_L30002IcT500000f]J0000
000<Ool200000dHa0000000@000KOol01000Oomoo`006Goo000QOol00`00Oomoo`0KOol200027>L0
0`00<H`00004>Lh01000@Q0004Ha0`0000=6<@00BU801TYB1@000e:D00<005JeE[D01EJe00<005[F
F]H01E[F00<005kg00000`0000=JeP00HaP01f<H00<006LiIcT00P000VLi00/006mkKg]_N`00Kg/0
07Nm001cW000019oo`03001oogoo02moo`04001oogoo0002Ool30005Ool00`00Oomoo`0=Ool01P00
36<<H`aS000U:@800003<H`003g_00Dmk`030012448@00E24003001:DTYB00A:DP03001>Ldic00E>
L`03001F]EJe00AF]@8000INm`80009Nm`05001W>FLiIcT0000:Ool00`00Oomoo`0dOol40003Ool3
000017oo0000000017oo00<007ooOol03Woo0`0000<]J`00?Nl013g_00<004YBBU800dYB00<005Je
E[D01`000U[F00<005kgG_L00Ukg0P000fLi0P0000=[FP00JeX010003goo100015[F0P0000=W>@00
00000P007goo00<007oo00006Goo000QOol00`00Oomoo`0KOol200037>L00`00>LhicP04>Lh30005
AS400`00BU9:DP02BU830007DY@00`00E[EF]@05E[D00`00F]IJeP04F]H00`00G_MNm`04G_L20007
HaP00`00IcUW>@03IcT00`00IcT00003Kg/00`00Kg/00004000BOol00`00Oomoo`0_Ool01000Oomo
o`000Woo00<007ooOol01Goo0P003Goo0P000`aS0P00136<00<003g_?Nl00cg_00<0048@@Q001D8@
00<004YBBU8014YB00<004icCW<01Dic00<005JeE[D015Je0P001Ekg00<006LiIcT00`000fLi00<0
07ooOol027oo00<007ooOol0AWoo00<007ooOol02goo0`000Woo0`0013g_00<004YBBU800dYB00<0
05JeE[D015Je00<005[FF]H00e[F00<005kgG_L00Ukg00P006LiIcUW>@00JeX006]J1@0057oo1`00
8goo00<007oo00006Goo000QOol00`00Oomoo`0KOol200037>L200000cW>0000000200000d8@0016
<@05AS400`00BU9:DP020003CW<00`00DYABU004DY@00`00E[EF]@05E[D00`00F]IJeP04F]H00`00
G_MNm`04G_L01000HaP0000016<H00<006LiIcT01FLi00<006mkKg/00fmk00<007Nm00000P0057oo
00<007ooOol0;Woo10000goo0`0017oo00<007ooOol037oo0P000`aS00<0036<<H`00c6<00<003g_
?Nl013g_00<0048@@Q000d8@4P001EJe00<005kgG_L015kg00<006LiIcT00VLi00D006LiIcUW>@00
00Yoo`03001oogoo04Eoo`8000Yoo`<000Eoo`03000<H`0000@mk`03001:DTYB00=:DP03001F]EJe
00AF]@03001JeU[F009JeP04001NmekgG_L30002IcT01000JeX006]J1P00?goo0P006Goo000QOol0
0`00Oomoo`0LOol200027>L30003>Lh01000@Q1240001THa0P001Dic00<005:DDY@01E:D00<005Je
E[D015Je00<005[FF]H015[F00<005kgG_L015kg00@006<HHaQS6080009S6003001W>FLi00AW>@80
00A_N`80009g_@8001Aoo`03001oogoo03aoo`03001oogoo00aoo`0400048@0036<20005<H`00`00
?Nlmk`04?Nl00`00@Q00000300000dYB001>L`05CW<00`00E[EF]@05E[D00`00F]H0000300000eJe
001Nm`05G_L00`00IcUW>@02IcT200000fLi0000000:Ool00`00Oomoo`15Ool00`00Oomoo`05Ool4
0009Ool30003?Nl00`00BU9:DP03BU800`00E[EF]@03E[D00`00F]IJeP03F]H01000G_L000000f<H
0P000V]J0`0000A_N`000000041oo`8001Uoo`008Goo00<007ooOol077oo10000Re[00@003W>>Lh0
00=240030016<DHa0096<@800003BU8004ic00E>L`03001BU5:D00ABU003001F]EJe00AF]@03001J
eU[F00=JeP03001Nmekg00ANm`03001S66<H00AS608000IW>@04001_Nfmk0002Kg/01000Mkd007Nm
0`0057oo00<007ooOol0?7oo00<007ooOol037oo00@000@Q0000008U:@03000aS36<008aS003000m
kcg_00<mk`<000E:DP03001>Ldic00A>L`03001F]EJe00EF]@03001JeU[F00=JeP8000INm`08001W
>FLiIcT006mk001W>@8000Yoo`03001oogoo04Eoo`8000=oo`<000eoo`03000<H`0000<000I:DP03
001F]EJe00=F]@03001JeU[F009JeP<000ES60H0009_N`8003eoo`@001]oo`008Goo00<007ooOol0
6Woo1P000be[00<003W>0000148@00@004Ha000000=:DP03001>Ldic00A>L`03001BU5:D009BU140
00ENm`03001S66<H00AS6004001W>@000004IcT01000Kg]_N`000Vmk00@007Nm001g_@800003Ool0
07oo019oo`03001oogoo03aoo`03001oogoo00aoo`0400048@0000029BT00`00<H`aS003<H`00`00
?Nl0000200000dHa001:DP05BU800`00CW=>L`04CW<00`00E[EF]@04E[D00`00F]IJeP05F]H30003
G_L00`00IcUW>@02IcT00`00Kg]_N`03000;Ool00`00Oomoo`14Ool00`00Ool00002000AOol20002
<H`30003BU800`00E[EF]@04E[D00`00F]H0000200000ekg001S6003HaP20002IcT01@00JeY[FP00
Kg/00P00>goo0`007goo000POol00`00Oomoo`0HOol30003Ool00`0024800004;F/00`00@Q124004
@Q000`00BU9:DP04BU800`00CW=>L`03CW<50002E[D00`00E[EF]@04E[D00`00F]IJeP03F]H00`00
G_L0000200000ekg001S6005HaP00`00IcUW>@02IcT200000fLi001_N`03Kg/20002Mkd30003Ool0
0`00Oomoo`0@Ool00`00Oomoo`0lOol00`00Oomoo`0<Ool00`00124000039BT00`00<H`aS003<H`2
0004AS400`00BU9:DP04BU800`00CW=>L`04CW<00`00E[EF]@04E[D00`00F]IJeP04F]H20002G_L2
00000ekg001W>@03IcT00`00Kg]_N`03000;Ool00`00Oomoo`0fOol400000goo001oo`020005Ool2
000EOol20002<H`00`00BU80000;0004G_L01@00HaQS66<H00000fLi00<006]JJeX01000>7oo1000
8Woo000POol00`00Oomoo`0DOol40006Ool40003;F/20004@Q000`00AS400005BU800`00CW=>L`03
00000e:D001F]@04E[D00`00E[EF]@04E[D00`00F]IJeP02F]H00`00G_MNm`03G_L30005HaP00`00
IcUW>@04IcT00`00Kg]_N`03Kg/01000Mkeg_GNm0`000goo00<007ooOol047oo00<007ooOol0?7oo
00<007ooOol03Goo00<48@009BT00RDY00@0036<<H`aS0<000I6<@03001:DTYB00A:DP03001>Ldic
00=>L`03001F]EJe00AF]@03001JeU[F00AJeP8000ANm`8000AW>@04001_NfmkKg/2000;Ool00`00
Oomoo`0fOol01000Oomoo`000Woo00@007ooOol000Eoo`03001oogoo01Aoo`05000aS36<<H`00004
BU800`00F]IJeP03F]H00`00G_MNm`04G_L30003IcT01000JeY[FV]J0`00=Goo10009Woo000POol2
000BOol3000;Ool01000248000000Re[00<003W>00000T8@00D004HaAS56<@0000E:DP8000EBU003
001F]EJe009F]@03001F]EJe00=F]@03001JeU[F00=JeP03001Nmekg00=Nm`03001S66<H0080009S
6003001W>FLi00AW>@8000A_N`8000=g_@8000Eoo`03001oogoo00moo`03001oogoo03]oo`03001o
ogoo00ioo`03000<H`0000<U:@<000<mk`030016<DHa00=6<@03001:DTYB00A:DP03001>Ldic00=>
L`03001F]EJe00AF]@03001JeU[F00=JeP03001Nm`0000ANm`03001W>@00009W>@03001_Nfmk009_
N`03001oogoo00Yoo`03001oogoo03Ioo`04001oogoo0005Ool00`00Oomoo`04Ool2000EOol20000
0c6<001:DP03BU800`00F]IJeP04F]H00`00G_MNm`02G_L30003IcT00`00JeX00005000bOol3000Z
Ool0021oo`03001oogoo00ioo`<000moo`<00003;F/003W>008icP030012400000A6<@07001:DTYB
BU8004ic00001E:D00<005JeE[D00eJe00<005JeE[D00UJe00<005[FF]H00e[F00<005kgG_L00ekg
00<006<HHaP00V<H0P001VLi00@006mkKg/0009_N`03001g_@00009g_@<000Ioo`03001oogoo00mo
o`03001oogoo03Yoo`03001oogoo00ioo`03000<H`0000<U:@03000mkcg_00<mk`030016<DHa00=6
<@03001:DTYB00A:DP03001>Ldic00=>L`03001F]EJe00=F]@03001JeU[F00AJeP<000ANm`03001W
>FLi00<000=_N`8000aoo`03001oogoo03Ioo`04001oogoo0002Ool30009Ool00`00Oomoo`0DOol3
0004BU800`00F]IJeP03F]H00`00G_MNm`020003IcT500000fmk00000002000_Ool4000]Ool0021o
o`03001oogoo00]oo`<0019oo`8000037>L0000000@icP030016<DHa00A6<@03001:DP0000=>L`03
001BU5:D00=BU003001F]EJe009F]@03001F]EJe00D0009JeP03001Nmekg00=Nm`03001S66<H00=S
6003001W>@0000AW>@07001_NfmkKg/006mk00000gNm100027oo00<007ooOol03Woo00<007ooOol0
>Woo00<007ooOol03Woo00P000aS36<002DY000e[@0013g_00<004HaAS400dHa00<004YBBU8014YB
00<004icCW<00dic00<005JeE[D00eJe00@005[FF]IJeP@0009Nm`04001Nmekg0005IcT01000Kg]_
Nfmk0P0037oo00<007ooOol0=Woo00@007ooOol0009oo`03001oogoo00Yoo`03001oogoo01Eoo`@0
0003BU8005[F00=JeP@000EW>@8000=_N`<002ioo`<0035oo`0087oo00<007ooOol027oo0`005Woo
0P0000<Li`00>Lh00cW>0P001THa00<004icCW<00dic00<005:DDY@00U:D00@005JeE[EF]@@0009J
eP03001Nmekg00D000=Nm`03001S66<H009S6004001W>FLiIcT200000fLi001_N`04Kg/00`00Mkeg
_@02Mkd00`00Mkd0000:Ool00`00Oomoo`0=Ool00`00Oomoo`0jOol00`00Oomoo`0>Ool0200036<<
H`009BT003F]0005?Nl00`00AS56<@03AS401000BU9:DTYB4P000U[F00<005kgG_L00Ukg00<005kg
000016Li0P000Vmk00<007>L000037oo00<007ooOol0=Woo10000goo0`002Woo0P005goo20000ekg
0P000VLi0P0000=_N`0000001000:goo1000=7oo000OOol20006Ool4000JOol00`007>L000020002
>Lh01000@Q0000000dHa00<004ic00001Dic00@005:DDYABU0@00003F]H005[F00=JeP03001Nmekg
009Nm`03001Nmekg00@000=S6003001W>FLi00AW>@03001_Nfmk00=_N`8000=g_@04001g_@00000;
Ool00`00Oomoo`0<Ool00`00Oomoo`0jOol2000?Ool2000236<01@00=Jde[CF]000013g_00<004Ha
AS400THa1@000U:D00<005JeE[D00eJe00<005[FF]H010000Ukg1@0015kg00<006LiIcT00fLi00@0
06mk001_N`<000aoo`03001oogoo04aoo`03001oogoo01Eoo`<00003BU80000000INm`D00004JeX0
0000000YOol4000hOol001moo`03001oogoo009oo`<001moo`04000Li`00;F/30003@Q000`00AS40
0003CW<20004CW<30002E[D01@00F]IJeU[F00000e[F00<005kgG_L00Ukg00D005kgG_MNm`00009S
60<000IW>@03001[FP00009_N`800003Li`007Nm00<00003Mkd007oo00]oo`03001oogoo00aoo`03
001oogoo03Yoo`03001oogoo00moo`04000<H`aS0004=Jd00`00?Nlmk`02?Nl01P00AS400000CW<0
00EBU003001F]EJe00=F]@<0009JeP03001Nmekg00=Nm`03001S66<H0080009Nm`8000=W>@03001_
Nfmk00800003Li`007oo00aoo`03001oogoo04eoo`03001oogoo01Aoo`03001oogoo00800003BU80
000000=Nm`800003JeX0000000<002Qoo`<003aoo`006Goo00<007ooOol00goo00<007oo00000P00
8goo0P000be[00<0048@@Q000T8@00<004icCW<014ic00H004ic000005Je0003E[D01@00F]IJeU[F
00000e[F00D005kgG_MNm`0000=Nm`03001S66<H009S6003001W>@0000=W>@80009[FP03001_N`00
00=cW0800004Mkd007Nm000>Ool00`00Oomoo`0;Ool00`00Oomoo`0jOol00`00Oomoo`0?Ool20000
0`aS00000004=Jd01000?Nlmkcg_0P0014ic00<005:DDY@00e:D1P0015[F00<005kgG_L00ekg00<0
06<HHaP00V<H0P000VLi00<006Li000016mk0`003Goo00<007ooOol0CWoo00<007ooOol04goo00<0
07ooOol00Woo2@00:Goo1000?goo000IOol00`00Oomoo`03Ool2000WOol010aS000]Jbe[0`000T8@
0`001Dic00<005JeE[D00UJe00D005JeE[EF]@0000=JeP03001JeP000080009Nm`05001NmekgG_L0
0002HaP20003IcT00`00IcT00005JeX00`00LiacW002Li`00`00Mkeg_@02000@Ool00`00Oomoo`0:
Ool00`00Oomoo`0jOol00`00Oomoo`0?Ool300000bU:000e[@03=Jd01P00?Nl00000BU8000A>L`03
001BU5:D00@000AF]@03001JeU[F009JeP03001Nmekg009Nm`03001S66<H00=S6003001W>@00009W
>@03001_Nfmk009_N`@000eoo`03001oogoo04moo`8001=oo`03001oogoo035oo`<004=oo`006Goo
00<007ooOol017oo0P009goo00@<H`00;F/0008icP<0009:DP80009>L`03001BU00000AF]@03001F
]EJe00H00003G_L005kg00H0009S6003001W>FLi00AW>@03001[FV]J009[FP<0009cW080009g_@03
001oogoo015oo`03001oogoo00Uoo`03001oogoo03Yoo`03001oogoo011oo`80008YBP03000e[CF]
008e[@05001:DTYBBU800004CW<30003DY@00`00E[EF]@02E[D00`00F]IJeP02F]H00`00G_MNm`02
G_L00`00HaQS6003HaP01@00IcUW>@00IcT00P000Vmk00@007Nm000000ioo`03001oogoo055oo`03
001oogoo011oo`03001oogoo02eoo`@004Ioo`006Goo00<007ooOol01Woo00<007ooOol09Goo1000
0cW>00<004YBBU800TYB0P000e:D0P000eJe0P000U[F00<005[F00000Ukg00@005kgG_L0009S6004
001S60000005IcT30002JeX20002Kg/3000017Nm001g_@005Goo00<007ooOol02Goo00<007ooOol0
>Goo00<007ooOol047oo0P000bU:00H003F]=Jd004Ha0003BU8400000dic001BU003DY@00`00E[EF
]@02E[D01@00F]IJeU[F000015kg00<006<HHaP00f<H00<006LiIcT00VLi00D006mk001_N`00009g
_@8000ioo`03001oogoo059oo`03001oogoo00moo`03001oogoo02Uoo`@004Yoo`0067oo0P002Goo
0P009Woo00@002480000008icP<000=:DP<000=BU005001F]@00F]H00002F]H00`00F]H00002G_L0
1P00G_MNm`00HaP0009S6003001W>@00009W>@8000=[FP8000A_N`03001g_GNm00<001Ioo`03001o
ogoo00Uoo`03001oogoo03Uoo`80019oo`80008YBP03000e[@0000=6<@<00003BU8004ic00=>L`03
001BU5:D009BU005001F]EJeE[D00003F]H00`00G_MNm`02G_L00`00HaQS6002HaP00`00IcUW>@02
IcT01000Kg/006mk0P0000=g_@0000003goo00<007ooOol0Dgoo0P003goo00<007ooOol09Woo0`00
CWoo000IOol00`00Oomoo`09Ool00`00Oomoo`0TOol00`008@P0000200000cW>0016<@030003DY@2
0002DY@20002F]H03000F]IJeP00F]H005kg001Nm`00HaP0009S6004001W>FLiIcT200001f]J0000
0000JeX006mk00<00004Li`007NmMkd2000HOol00`00Oomoo`08Ool00`00Oomoo`0iOol00`00Oomo
o`0AOol30002:DX500000dHa001:DP02BU801@00CW=>Ldic00000e:D2`000Ukg00<006<HHaP00f<H
00D006LiIcUW>@00009_N`05001_N`00001g_@02000?Ool00`00Oomoo`1EOol00`00Oomoo`0<Ool0
0`00Oomoo`0ROol4001AOol002Ioo`03001oogoo02Aoo`<00004<H`000000002AS420004DY@30000
1E[F000005[F00000U[F1`000V<H00@006Li0000009W>@8000A[FP8000=cW003001g_@00008001Yo
o`03001oogoo00Moo`03001oogoo03Uoo`03001oogoo019oo`03000DY@0000<000A6<@05001:DTYB
BU800003CW<400000eJe001JeP02F]H01P00F]IJeU[F001Nm`<000AS6005001W>FLiIcT00004Kg/0
0`00Mkeg_@02000AOol00`00Oomoo`1EOol00`00Oomoo`0:Ool00`00Oomoo`0POol3001EOol002Mo
o`8002Ioo`80008aS0@00006DY@00000001BU0000UJe1`0000=Nm`0000000`0000=S600000001@00
00=W>@00JeX00`0000=_N`00Li`00W>L1@0077oo00<007ooOol01Woo00<007ooOol0>7oo00<007oo
Ool04Goo100000DDY@00000mk`0000A6<@04001:DTYBBU830002E[D01@00E[EF]EJe00000U[F00@0
05[FF]H000=Nm`04001S60000002HaP01000IcUW>@0016mk0P0000Ag_@00Li`0015oo`03001oogoo
05Ioo`8000Yoo`03001oogoo01aoo`@005Qoo`00:Goo00<007ooOol09Goo0`000T8@100000=BU000
00000P000eJe3`0016Li0P000fmk00<007>L00000`0000=cW000Ool07Woo00<007ooOol01Goo00<0
07ooOol0>7oo00<007ooOol03Woo0`000goo0P0000@DY@00?Nlmk`8000=6<@05001:DP00DY@00003
E[D01@00E[EF]EJe00000U[F00@005[FF]H000=Nm`04001S66<HHaP20002IcT00`00Kg]_N`02Kg/0
0`00LiacW00300000goo001oo`0?Ool00`00Oomoo`1HOol00`00Oomoo`07Ool00`00Oomoo`0HOol4
001LOol002Yoo`8002Moo`@00003@Q0004ic01T0009[FP03001_Nfmk00@0009cW0<0021oo`03001o
ogoo00Eoo`03001oogoo03Qoo`03001oogoo00Yoo`@000Moo`@000<mk`050016<DHaAS400003DY@0
1@00E[EF]EJe00000UJe00@005[FF]H0009JeP04001Nmekg0004HaP01000IcT000000fmk0P000g>L
0P000goo00<007ooOol03Goo00<007ooOol0FGoo00<007ooOol01Woo00<007ooOol05Goo0`00H7oo
000/Ool00`00Oomoo`0VOol00`00?Nl00004000014ic001F]EJe600000=_N`00Li`010008goo00<0
07ooOol01Goo00<007ooOol0=goo0`001Woo100037oo10000cg_00D004Ha001>L`0000=BU007001F
]EJeE[D005Je00000U[F00H005[FF]H005kg0004HaP02000IcT00000JeX006mk0003Li`30005Ool0
0`00Oomoo`0<Ool00`00Oomoo`1JOol00`00Oomoo`05Ool00`00Oomoo`0AOol4001SOol002eoo`80
02Moo`80008mk`L000=JeP800003G_L0000001@002Ioo`03001oogoo00Aoo`03001oogoo035oo`03
001oogoo00=oo`03001oogoo00=oo`<0015oo`800003=Jd00000008mk`05001>LdicCW<00003DY@0
1000E[EF]@000UJe00D005[F001JeP00009Nm`03001S66<H00800003IcT006]J00=[FP03001cW7>L
00800003Mkd007oo00Ioo`03001oogoo00]oo`03001oogoo05]oo`8000Eoo`03001oogoo00ioo`<0
06Moo`00;goo00<007ooOol09Woo2`000U[F2@0000=W>@0000000P000f]J1@00:Goo00<007ooOol0
0goo00<007ooOol0<Goo00<007ooOol00goo00<007oo00000`005Goo0`0000@e[@00001>L`80009>
L`80009BU003001F]EJe00T0009S6004001W>FLi0003JeX50002Mkd00`00Oomoo`07Ool00`00Oomo
o`0:Ool00`00Oomoo`1MOol00`00Oomoo`02Ool00`00Oomoo`0:Ool4001ZOol0031oo`03001oogoo
02Eoo`05001oo`00000icP020002CW<400001e[F001Nmekg001S6000009S60<0009W>@D002eoo`03
001oogoo009oo`03001oogoo035oo`03001oogoo00=oo`8001Yoo`04000YBP00=Jd20002CW<20002
CW<00`00DYABU00200001Ukg000005kg001Nm`800004HaP00000IcT20002JeX20002Kg/00`00Mkeg
_@03000:Ool00`00Oomoo`09Ool00`00Oomoo`1NOol01@00Oomoogoo000027oo1000KWoo000aOol2
000UOol01000Oomoogoo1P000U:D2P000Ukg1P00;goo00<007ooOol00Woo00<007ooOol0<Goo00<0
07ooOol017oo00<007ooOol06Goo00D002U:000004ic0080009>L`800004CW<00000F]H200000ekg
0000000300000f<H000000020002JeX200002Fmk000006mk001g_@00Kg/0000<Ool00`00Oomoo`08
Ool00`00Oomoo`1OOol200000goo001oo`04Ool3001bOol003=oo`03001oogoo029oo`03001oogoo
00Eooa8003Aoo`05001oogooOol0000bOol20007Ool2000JOol2000012U:000004ic0`0000=>L`00
00000`0000=JeP0000001`0000=W>@00000010000Vmk0`0000=_N`00Ool03Goo00<007ooOol01goo
00<007ooOol0HGoo0P0000=oo`0000000P00MGoo000dOol2000ROol00`00Oomoo`1<Ool01000Oomo
o`00<goo00<007ooOol01goo00<007ooOol06Goo0P0000E:DP00000004ic01@000E_N`@0011oo`03
001oogoo00Ioo`03001oogoo069oo`8007Uoo`00=Woo00<007ooOol07goo00<007ooOol0CGoo00<0
07oo0000?Woo00<007ooOol06Goo1@0000=:DP000000500000=_N`0000000P004goo00<007ooOol0
1Goo00<007ooOol0gGoo000gOol2000OOol00`00Oomoo`1>Ool00`00Ool0000nOol2000KOol20000
0dYB0000000D000016]J000000005Woo00<007ooOol017oo00<007ooOol0gGoo000iOol00`00Oomo
o`0LOol00`00Oomoo`1>Ool00`00Ool00010Ool00`00Oomoo`0IOol=00000f<H0000000:000HOol0
0`00Oomoo`04Ool00`00Oomoo`3MOol003Yoo`03001oogoo01]oo`03001oogoo04moo`80045oo`03
001oogoo01Qoo`03001oogoo01@001aoo`03001oogoo00Aoo`03001oogoo075oo`03001oogoo06Qo
o`00>goo0P006goo00<007ooOol0CWoo0`00@Woo00<007ooOol05goo00<007ooOol00goo3@008Goo
00<007ooOol00goo00<007ooOol0LGoo00D007ooOomoo`0006Ioo`00?Goo00<007ooOol067oo00<0
07ooOol0BWoo1000AWoo0P005goo00<007ooOol0<Woo00<007ooOol00Woo00<007ooOol0LGoo0`00
0Woo00<007ooOol01Woo0P000Woo00<007oo00000P000Woo00<007oo0000100000=oo`00Ool0B7oo
000nOol2000HOol00`00Oomoo`15Ool5001<Ool00`00Oomoo`0DOol00`00Oomoo`0cOol01@00Oomo
ogoo0000Lgoo00<007ooOol00P0017oo1@000Woo00D007oo001oo`00009oo`04001oo`00Ool20002
Ool00`00Ool00019Ool0041oo`03001oogoo01Eoo`03001oogoo045oo`@0059oo`03001oogoo01=o
o`03001oogoo03Aoo`04001oogoo001cOol01000Oomoo`002Goo00P007ooOol007oo001oo`000Woo
00H007oo001oo`00Ool300000goo001oo`18Ool0045oo`8001Eoo`03001oogoo03eoo`@005Moo`80
01=oo`03001oogoo03Eoo`03001oo`0007=oo`04001oogoo0009Ool02000Oomoo`00Ool007oo0002
Ool01@00Ool007oo00000goo00<007oo0000BGoo0013Ool00`00Oomoo`0BOol00`00Oomoo`0hOol5
001MOol00`00Oomoo`0@Ool00`00Oomoo`0fOol2001cOol3000;Ool20002Ool50002Ool700000goo
001oo`18Ool004Aoo`80019oo`03001oogoo03Aoo`@006=oo`03001oogoo00moo`03001oogoo03Eo
o`<008Moo`03001oogoo009oo`03001oo`0000Eoo`03001oogoo04Moo`00AWoo00<007ooOol03goo
00<007ooOol0;goo1@00J7oo00<007ooOol03Woo00<007ooOol0<Goo1000hGoo0017Ool00`00Oomo
o`0=Ool00`00Oomoo`0/Ool4001^Ool2000>Ool00`00Oomoo`0]Ool4003UOol004Qoo`8000eoo`03
001oogoo02Qoo`@007Aoo`03001oogoo00]oo`03001oogoo02Uoo`@00>Uoo`00BWoo00<007ooOol0
2Woo00<007ooOol08goo1@00NGoo00<007ooOol02Woo00<007ooOol09Goo1000kGoo001;Ool2000:
Ool00`00Oomoo`0OOol4001oOol2000:Ool00`00Oomoo`0QOol4003aOol004eoo`03001oogoo00Mo
o`03001oogoo01Yoo`D008Eoo`03001oogoo00Moo`03001oogoo01eoo`@00?Eoo`00CWoo0P001goo
00<007ooOol05Woo1000Rgoo00<007ooOol01Woo00<007ooOol067oo1@00nGoo001@Ool00`00Oomo
o`04Ool00`00Oomoo`0AOol5002@Ool00`00Oomoo`05Ool00`00Oomoo`0DOol4003nOol0055oo`03
001oogoo00=oo`03001oogoo00eoo`@009Ioo`8000Eoo`03001oogoo011oo`@00?moo`=oo`00DWoo
0P000goo00<007ooOol02Goo1000W7oo00<007ooOol00Woo00<007ooOol037oo1000ogoo1goo001D
Ool01000Oomoo`001Woo1@00XGoo00D007ooOomoo`0000Yoo`@00?moo`]oo`00EGoo0`000Woo1000
Ygoo0P0000=oo`00Ool01Goo1000ogoo3goo001GOol3002]Ool20002Ool4003oOolCOol00?moo`Uo
o`<00?mooaMoo`00ogooogoo8goo003oOoooOolSOol00?moo`=oo`03001oogoo0?mooaeoo`00ogoo
0goo00<007ooOol0ogoo7Goo001FOol20008Ool20002Ool00`00Ool000020002Ool00`00Ool00004
00000goo001oo`2=Ool300000goo001oo`08Ool20002Ool00`00Ool000020002Ool00`00Ool00004
00000goo001oo`3lOol005Ioo`03001oo`00009oo`D0009oo`05001oo`00Ool00002Ool01000Ool0
07oo0P000Woo00<007oo0000SWoo00H007ooOol007oo0003Ool50002Ool01@00Ool007oo00000Woo
00@007oo001oo`80009oo`03001oo`000?eoo`00Egoo00<007ooOol01Goo00P007ooOol007oo001o
o`000Woo00H007oo001oo`00Ool300000goo001oo`2=Ool00`00Oomoo`030007Ool02000Oomoo`00
Ool007oo0002Ool01P00Ool007oo001oo`<00003Ool007oo0?aoo`00EWoo0P001goo00P007ooOol0
07oo001oo`000Woo00D007oo001oo`0000=oo`03001oo`0008ioo`04001oogoo0009Ool02000Oomo
o`00Ool007oo0002Ool01@00Ool007oo00000goo00<007oo0000oGoo001GOol20007Ool20002Ool5
0002Ool700000goo001oo`2=Ool3000;Ool20002Ool50002Ool700000goo001oo`3lOol006Ioo`03
001oogoo009oo`03001oo`0000Eoo`03001oogoo0:1oo`03001oogoo009oo`03001oo`0000Eoo`03
001oogoo0?]oo`00ogooogoo8goo003oOoooOolSOol00?mooomoob=oo`00ogooogoo8goo003oOooo
OolSOol00001\
\>"],
  ImageRangeCache->{{{0, 544.75}, {183.75, 0}} -> {-0.0839309, -0.0254467, 
  0.00581526, 0.00581526}, {{14.375, 175.625}, {179.375, 
  4.375}} -> {-0.143516, -0.0927963, 0.00610726, 0.00610726}, {{191.75, 
  352.938}, {179.375, 4.375}} -> {-1.30223, -0.0937697, 0.00650281, 
  0.00650281}, {{369.063, 530.313}, {179.375, 
  4.375}} -> {-3.04408, -0.236997, 0.00778435, 0.00778435}}]
}, Open  ]],

Cell[TextData[StyleBox[" "]], "Text"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .34766 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.31746 0.00827773 0.31746 [
[ 0 0 0 0 ]
[ 1 .34766 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .34766 L
0 .34766 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.00827773 0.321429 0.339387 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.066106 1.21212 1.11022e-016 1.21212 [
[.5 1.12503 -19.375 0 ]
[.5 1.12503 19.375 10.0625 ]
[.04889 .17135 -20.0812 -10 ]
[.04889 .17135 .54385 0 ]
[.15639 .12856 -23.7934 -10 ]
[.15639 .12856 1.2691 0 ]
[.26699 .08455 -7.80548 -10 ]
[.26699 .08455 .63202 0 ]
[.38085 .03925 -17.6224 -10 ]
[.38085 .03925 1.94011 0 ]
[.49809 -0.00739 -13.2579 -10 ]
[.49809 -0.00739 1.86709 0 ]
[.21792 .02683 -7.34293 -10 ]
[.21792 .02683 .59457 0 ]
[.54614 .01177 0 -6.96909 ]
[.54614 .01177 13.9375 3.03091 ]
[.66854 .17551 0 -6.83927 ]
[.66854 .17551 20.625 3.16073 ]
[.7789 .32314 0 -6.72552 ]
[.7789 .32314 8.4375 3.27448 ]
[.87891 .45694 0 -6.62501 ]
[.87891 .45694 15.125 3.37499 ]
[.96996 .57877 0 -6.53557 ]
[.96996 .57877 8.4375 3.46443 ]
[.85051 .29843 0 -6.72552 ]
[.85051 .29843 7.9375 3.27448 ]
[.03361 .20153 -20.625 -3.02288 ]
[.03361 .20153 0 6.97712 ]
[.02271 .29784 -25.0625 -3.08408 ]
[.02271 .29784 0 6.91592 ]
[.01132 .39849 -8.4375 -3.14821 ]
[.01132 .39849 0 6.85179 ]
[-0.00059 .50376 -19.5625 -3.21551 ]
[-0.00059 .50376 0 6.78449 ]
[-0.01306 .61401 -15.125 -3.28621 ]
[-0.01306 .61401 0 6.71379 ]
[-0.05972 .4248 -7.9375 -3.14821 ]
[-0.05972 .4248 0 6.85179 ]
[ 0 0 0 0 ]
[ 1 1.11253 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.12503 -80.375 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18.0625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.062500 moveto
 460.000000 14.062500 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.062500 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(p) show
67.437500 11.500000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
70.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
76.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
97.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.04875 .18654 m
.51908 0 L
s
.05931 .18235 m
.06452 .18785 L
s
gsave
.04889 .17135 -81.0812 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.16652 .13983 m
.17158 .14547 L
s
gsave
.15639 .12856 -84.7934 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.27681 .09609 m
.28172 .10186 L
s
gsave
.26699 .08455 -68.8055 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.39032 .05107 m
.39506 .05698 L
s
gsave
.38085 .03925 -78.6224 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.5072 .00471 m
.51176 .01076 L
s
gsave
.49809 -0.00739 -74.2579 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08051 .17394 m
.08362 .17726 L
s
.10183 .16549 m
.10492 .16882 L
s
.12327 .15699 m
.12634 .16033 L
s
.14483 .14843 m
.14789 .1518 L
s
.18832 .13118 m
.19134 .13458 L
s
.21025 .12249 m
.21326 .1259 L
s
.23231 .11374 m
.2353 .11717 L
s
.2545 .10494 m
.25746 .10839 L
s
.29925 .08719 m
.30217 .09067 L
s
.32182 .07824 m
.32472 .08173 L
s
.34452 .06923 m
.34741 .07275 L
s
.36736 .06018 m
.37022 .06371 L
s
.41342 .04191 m
.41625 .04547 L
s
.43666 .03269 m
.43946 .03627 L
s
.46004 .02342 m
.46282 .02701 L
s
.48355 .01409 m
.48631 .01771 L
s
gsave
.21792 .02683 -68.3429 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.51908 0 m
.96413 .59478 L
s
.53205 .01733 m
.525 .0201 L
s
gsave
.54614 .01177 -61 -10.9691 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.65432 .18074 m
.64721 .18336 L
s
gsave
.66854 .17551 -61 -10.8393 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.76458 .32809 m
.75741 .33056 L
s
gsave
.7789 .32314 -61 -10.7255 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.8645 .46163 m
.85729 .46397 L
s
gsave
.87891 .45694 -61 -10.625 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.95548 .58322 m
.94824 .58544 L
s
gsave
.96996 .57877 -61 -10.5356 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.55756 .05142 m
.55332 .05306 L
s
.58252 .08478 m
.57828 .08641 L
s
.60696 .11744 m
.60271 .11905 L
s
.63089 .14942 m
.62663 .15101 L
s
.67727 .21142 m
.673 .21297 L
s
.69976 .24147 m
.69548 .243 L
s
.7218 .27092 m
.71752 .27244 L
s
.7434 .29979 m
.73911 .30129 L
s
.78534 .35584 m
.78104 .3573 L
s
.8057 .38305 m
.80139 .3845 L
s
.82567 .40974 m
.82136 .41118 L
s
.84527 .43593 m
.84095 .43735 L
s
.88337 .48685 m
.87904 .48824 L
s
.90189 .51161 m
.89756 .51298 L
s
.92008 .53591 m
.91574 .53727 L
s
.93794 .55978 m
.9336 .56112 L
s
gsave
.85051 .29843 -61 -10.7255 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.04875 .18654 m
0 .62042 L
s
.0477 .19596 m
.05474 .19317 L
s
gsave
.03361 .20153 -81.625 -7.02288 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.03686 .29242 m
.04393 .28971 L
s
gsave
.02271 .29784 -86.0625 -7.08408 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.02553 .39322 m
.03263 .39059 L
s
gsave
.01132 .39849 -69.4375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.01368 .49867 m
.02082 .49613 L
s
gsave
-0.00059 .50376 -80.5625 -7.21551 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00127 .60909 m
.00844 .60664 L
s
gsave
-0.01306 .61401 -76.125 -7.28621 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.04557 .21492 m
.0498 .21325 L
s
.04342 .23404 m
.04765 .23239 L
s
.04125 .25333 m
.04549 .25168 L
s
.03906 .27279 m
.0433 .27115 L
s
.03463 .31222 m
.03888 .31061 L
s
.03239 .3322 m
.03664 .3306 L
s
.03012 .35236 m
.03438 .35076 L
s
.02784 .3727 m
.03209 .37111 L
s
.0232 .41393 m
.02747 .41236 L
s
.02085 .43483 m
.02512 .43327 L
s
.01849 .45591 m
.02276 .45437 L
s
.01609 .4772 m
.02037 .47566 L
s
.01125 .52035 m
.01553 .51883 L
s
.00879 .54222 m
.01308 .54072 L
s
.00631 .56431 m
.0106 .56281 L
s
.0038 .58659 m
.0081 .58511 L
s
gsave
-0.05972 .4248 -68.9375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.04875 .18654 m
0 .62042 L
s
0 .62042 m
.5574 1.11253 L
s
.5574 1.11253 m
.5561 .7194 L
s
.5561 .7194 m
.04875 .18654 L
s
.51908 0 m
.96413 .59478 L
s
.96413 .59478 m
1 .99897 L
s
1 .99897 m
.51687 .44401 L
s
.51687 .44401 m
.51908 0 L
s
.04875 .18654 m
0 .62042 L
s
0 .62042 m
.51687 .44401 L
s
.51687 .44401 m
.51908 0 L
s
.51908 0 m
.04875 .18654 L
s
.5561 .7194 m
.96413 .59478 L
s
.96413 .59478 m
1 .99897 L
s
1 .99897 m
.5574 1.11253 L
s
.5574 1.11253 m
.5561 .7194 L
s
0 0 m
1 0 L
1 1.11253 L
0 1.11253 L
closepath
clip
newpath
.5 Mabswid
.53101 .69057 m .53018 .74085 L .6324 .74871 L p .673 g
F P
0 g
s
.6324 .74871 m .62454 .69365 L .53101 .69057 L p .673 g
F P
0 g
s
.63773 .80318 m .6324 .74871 L .53018 .74085 L p .643 g
F P
0 g
s
.62454 .69365 m .6324 .74871 L .73703 .73168 L p .609 g
F P
0 g
s
.6324 .74871 m .63773 .80318 L .74853 .78715 L p .571 g
F P
0 g
s
.74853 .78715 m .73703 .73168 L .6324 .74871 L p .571 g
F P
0 g
s
.53206 .6424 m .53101 .69057 L .62454 .69365 L p .705 g
F P
0 g
s
.62454 .69365 m .61459 .6411 L .53206 .6424 L p .705 g
F P
0 g
s
.61459 .6411 m .62454 .69365 L .72014 .67627 L p .645 g
F P
0 g
s
.73703 .73168 m .72014 .67627 L .62454 .69365 L p .609 g
F P
0 g
s
.53018 .74085 m .52962 .79035 L .63773 .80318 L p .643 g
F P
0 g
s
.6402 .85394 m .63773 .80318 L .52962 .79035 L p .61 g
F P
0 g
s
.63773 .80318 m .6402 .85394 L .75389 .83951 L p .528 g
F P
0 g
s
.75389 .83951 m .74853 .78715 L .63773 .80318 L p .528 g
F P
0 g
s
.53328 .59916 m .53206 .6424 L .61459 .6411 L p .741 g
F P
0 g
s
.61459 .6411 m .60309 .59409 L .53328 .59916 L p .741 g
F P
0 g
s
.60309 .59409 m .61459 .6411 L .69881 .62407 L p .684 g
F P
0 g
s
.72014 .67627 m .69881 .62407 L .61459 .6411 L p .645 g
F P
0 g
s
.69881 .62407 m .72014 .67627 L .79541 .64136 L p .591 g
F P
0 g
s
.72014 .67627 m .73703 .73168 L .81971 .69272 L p .543 g
F P
0 g
s
.81971 .69272 m .79541 .64136 L .72014 .67627 L p .543 g
F P
0 g
s
.73703 .73168 m .74853 .78715 L .83623 .74521 L p .49 g
F P
0 g
s
.83623 .74521 m .81971 .69272 L .73703 .73168 L p .49 g
F P
0 g
s
.69881 .62407 m .67422 .57808 L .60309 .59409 L p .684 g
F P
0 g
s
.67422 .57808 m .69881 .62407 L .7648 .5941 L p .639 g
F P
0 g
s
.79541 .64136 m .7648 .5941 L .69881 .62407 L p .591 g
F P
0 g
s
.46175 .66985 m .45447 .71118 L .53018 .74085 L p .761 g
F P
0 g
s
.53018 .74085 m .53101 .69057 L .46175 .66985 L p .761 g
F P
0 g
s
.52962 .79035 m .53018 .74085 L .45447 .71118 L p .733 g
F P
0 g
s
.47103 .63045 m .46175 .66985 L .53101 .69057 L p .791 g
F P
0 g
s
.53101 .69057 m .53206 .6424 L .47103 .63045 L p .791 g
F P
0 g
s
.74853 .78715 m .75389 .83951 L .84389 .79574 L p .424 g
F P
0 g
s
.84389 .79574 m .83623 .74521 L .74853 .78715 L p .424 g
F P
0 g
s
.52962 .79035 m .52937 .83622 L .6402 .85394 L p .61 g
F P
0 g
s
.63959 .89795 m .6402 .85394 L .52937 .83622 L p .57 g
F P
0 g
s
.6402 .85394 m .63959 .89795 L .75266 .88559 L p .472 g
F P
0 g
s
.75266 .88559 m .75389 .83951 L .6402 .85394 L p .472 g
F P
0 g
s
.5346 .56352 m .53328 .59916 L .60309 .59409 L p .785 g
F P
0 g
s
.60309 .59409 m .59068 .55549 L .5346 .56352 L p .785 g
F P
0 g
s
.59068 .55549 m .60309 .59409 L .67422 .57808 L p .729 g
F P
0 g
s
.45447 .71118 m .44962 .75199 L .52962 .79035 L p .733 g
F P
0 g
s
.52937 .83622 m .52962 .79035 L .44962 .75199 L p .704 g
F P
0 g
s
.48177 .59536 m .47103 .63045 L .53206 .6424 L p .828 g
F P
0 g
s
.53206 .6424 m .53328 .59916 L .48177 .59536 L p .828 g
F P
0 g
s
.67422 .57808 m .64774 .54113 L .59068 .55549 L p .729 g
F P
0 g
s
.64774 .54113 m .67422 .57808 L .72962 .55371 L p .69 g
F P
0 g
s
.7648 .5941 m .72962 .55371 L .67422 .57808 L p .639 g
F P
0 g
s
.75389 .83951 m .75266 .88559 L .842 .84121 L p .329 g
F P
0 g
s
.842 .84121 m .84389 .79574 L .75389 .83951 L p .329 g
F P
0 g
s
.44962 .75199 m .44752 .78983 L .52937 .83622 L p .704 g
F P
0 g
s
.52944 .87566 m .52937 .83622 L .44752 .78983 L p .669 g
F P
0 g
s
.52937 .83622 m .52944 .87566 L .63959 .89795 L p .57 g
F P
0 g
s
.49334 .56679 m .48177 .59536 L .53328 .59916 L p .873 g
F P
0 g
s
.53328 .59916 m .5346 .56352 L .49334 .56679 L p .873 g
F P
0 g
s
.7648 .5941 m .79541 .64136 L .83094 .59759 L p .514 g
F P
0 g
s
.79541 .64136 m .81971 .69272 L .85904 .64096 L p .438 g
F P
0 g
s
.85904 .64096 m .83094 .59759 L .79541 .64136 L p .438 g
F P
0 g
s
.72962 .55371 m .7648 .5941 L .79554 .55915 L p .58 g
F P
0 g
s
.83094 .59759 m .79554 .55915 L .7648 .5941 L p .514 g
F P
0 g
s
.63584 .93232 m .63959 .89795 L .52944 .87566 L p .51 g
F P
0 g
s
.63959 .89795 m .63584 .93232 L .74468 .92235 L p .387 g
F P
0 g
s
.74468 .92235 m .75266 .88559 L .63959 .89795 L p .387 g
F P
0 g
s
.81971 .69272 m .83623 .74521 L .87806 .68666 L p .34 g
F P
0 g
s
.87806 .68666 m .85904 .64096 L .81971 .69272 L p .34 g
F P
0 g
s
.53593 .53792 m .5346 .56352 L .59068 .55549 L p .841 g
F P
0 g
s
.59068 .55549 m .57808 .52794 L .53593 .53792 L p .841 g
F P
0 g
s
.57808 .52794 m .59068 .55549 L .64774 .54113 L p .785 g
F P
0 g
s
.64774 .54113 m .62091 .51572 L .57808 .52794 L p .785 g
F P
0 g
s
.62091 .51572 m .64774 .54113 L .69187 .52266 L p .746 g
F P
0 g
s
.72962 .55371 m .69187 .52266 L .64774 .54113 L p .69 g
F P
0 g
s
.69187 .52266 m .72962 .55371 L .75497 .52793 L p .636 g
F P
0 g
s
.79554 .55915 m .75497 .52793 L .72962 .55371 L p .58 g
F P
0 g
s
.83623 .74521 m .84389 .79574 L .88668 .73193 L p .197 g
F P
0 g
s
.88668 .73193 m .87806 .68666 L .83623 .74521 L p .197 g
F P
0 g
s
.75266 .88559 m .74468 .92235 L .83037 .87858 L p .167 g
F P
0 g
s
.83037 .87858 m .842 .84121 L .75266 .88559 L p .167 g
F P
0 g
s
.44752 .78983 m .44833 .82235 L .52944 .87566 L p .669 g
F P
0 g
s
.52985 .90608 m .52944 .87566 L .44833 .82235 L p .614 g
F P
0 g
s
.52944 .87566 m .52985 .90608 L .63584 .93232 L p .51 g
F P
0 g
s
.43546 .63991 m .42569 .66884 L .45447 .71118 L p .88 g
F P
0 g
s
.45447 .71118 m .46175 .66985 L .43546 .63991 L p .88 g
F P
0 g
s
.44962 .75199 m .45447 .71118 L .42569 .66884 L p .849 g
F P
0 g
s
.44799 .61273 m .43546 .63991 L .46175 .66985 L p .906 g
F P
0 g
s
.46175 .66985 m .47103 .63045 L .45167 .61556 L p .906 g
F P
0 g
s
.46175 .66985 m .45167 .61556 L .44952 .61906 L .906 g
F
.4625 .589 m .44799 .61273 L .47103 .63045 L p .922 g
F P
0 g
s
.47103 .63045 m .48177 .59536 L .46873 .59105 L p .922 g
F P
0 g
s
.46873 .59105 m .4625 .589 L p .46434 .59794 L .922 g
F P
0 g
s
.47103 .63045 m .46873 .59105 L .46434 .59794 L .922 g
F
.47199 .60051 m .46264 .61376 L .44095 .63463 L p .015 g
F P
0 g
s
.44095 .63463 m .43846 .63703 L p .44161 .63359 L .015 g
F P
0 g
s
.44161 .63359 m .47199 .60051 L .44095 .63463 L .015 g
F
.43846 .63703 m .43546 .63991 L p .4391 .63599 L F P
0 g
s
.4391 .63599 m .47199 .60051 L .43846 .63703 L .015 g
F
.45167 .61556 m .44799 .61273 L p .44952 .61906 L .906 g
F P
0 g
s
.50505 .54669 m .49334 .56679 L .5346 .56352 L p .921 g
F P
0 g
s
.5346 .56352 m .53593 .53792 L .50505 .54669 L p .921 g
F P
0 g
s
.42569 .66884 m .41927 .69771 L .44962 .75199 L p .849 g
F P
0 g
s
.44752 .78983 m .44962 .75199 L .41927 .69771 L p .806 g
F P
0 g
s
.69187 .52266 m .65378 .50308 L .62091 .51572 L p .746 g
F P
0 g
s
.65378 .50308 m .69187 .52266 L .71164 .50586 L p .661 g
F P
0 g
s
.75497 .52793 m .71164 .50586 L .69187 .52266 L p .636 g
F P
0 g
s
.84389 .79574 m .842 .84121 L .88411 .77388 L p .039 g
F P
0 g
s
.88411 .77388 m .88668 .73193 L .84389 .79574 L p .039 g
F P
0 g
s
.62906 .95447 m .63584 .93232 L .52985 .90608 L p .402 g
F P
0 g
s
.63584 .93232 m .62906 .95447 L .73018 .94707 L p .228 g
F P
0 g
s
.73018 .94707 m .74468 .92235 L .63584 .93232 L p .228 g
F P
0 g
s
.5372 .52438 m .53593 .53792 L .57808 .52794 L p .9 g
F P
0 g
s
.57808 .52794 m .56608 .51365 L .5372 .52438 L p .9 g
F P
0 g
s
.56608 .51365 m .57808 .52794 L .62091 .51572 L p .844 g
F P
0 g
s
.62091 .51572 m .5954 .50392 L .56608 .51365 L p .844 g
F P
0 g
s
.5954 .50392 m .62091 .51572 L .65378 .50308 L p .771 g
F P
0 g
s
.41927 .69771 m .41665 .72473 L .44752 .78983 L p .806 g
F P
0 g
s
.44833 .82235 m .44752 .78983 L .41665 .72473 L p .734 g
F P
0 g
s
.4781 .57025 m .4625 .589 L .48177 .59536 L p .903 g
F P
0 g
s
.48177 .59536 m .48714 .5682 L .47928 .57831 L .903 g
F
.48284 .58934 m .47199 .60051 L .45552 .6089 L p .017 g
F P
0 g
s
.45552 .6089 m .44799 .61273 L p .45749 .60635 L .017 g
F P
0 g
s
.45749 .60635 m .48284 .58934 L .45552 .6089 L .017 g
F
.48177 .59536 m .49334 .56679 L .48714 .5682 L p .903 g
F P
0 g
s
.48714 .5682 m .4781 .57025 L p .47928 .57831 L .903 g
F P
0 g
s
.44833 .82235 m .45209 .84738 L .52985 .90608 L p .614 g
F P
0 g
s
.53058 .92519 m .52985 .90608 L .45209 .84738 L p .5 g
F P
0 g
s
.52985 .90608 m .53058 .92519 L .62906 .95447 L p .402 g
F P
0 g
s
.74468 .92235 m .73018 .94707 L .80938 .90511 L p .028 g
F P
0 g
s
.80938 .90511 m .83037 .87858 L .74468 .92235 L p .028 g
F P
0 g
s
.51863 .54284 m .53593 .53792 L p .52464 .53718 L .914 g
F P
0 g
s
.53593 .53792 m .5372 .52438 L .51613 .53663 L p .914 g
F P
0 g
s
.53832 .52432 m .52583 .53761 L p .53275 .52956 L F P
s
.5372 .52438 m .53832 .52432 L p .53275 .52956 L F P
s
.51613 .53663 m .50505 .54669 L .51863 .54284 L p .914 g
F P
0 g
s
.52464 .53718 m .51613 .53663 L .51863 .54284 L .914 g
F
.75497 .52793 m .79554 .55915 L .78082 .53136 L p .352 g
F P
0 g
s
.79554 .55915 m .83094 .59759 L .81468 .55857 L p .27 g
F P
0 g
s
.81468 .55857 m .78082 .53136 L .79554 .55915 L p .27 g
F P
0 g
s
.83094 .59759 m .85904 .64096 L .8415 .59105 L p .14 g
F P
0 g
s
.8415 .59105 m .81468 .55857 L .83094 .59759 L p .14 g
F P
0 g
s
.65378 .50308 m .61773 .49657 L .5954 .50392 L p .771 g
F P
0 g
s
.61773 .49657 m .65378 .50308 L .66815 .49444 L p .533 g
F P
0 g
s
.71164 .50586 m .66815 .49444 L .65378 .50308 L p .661 g
F P
0 g
s
.41665 .72473 m .41802 .74814 L .44833 .82235 L p .734 g
F P
0 g
s
.45209 .84738 m .44833 .82235 L .41802 .74814 L p .592 g
F P
0 g
s
.4938 .55777 m .4781 .57025 L .49334 .56679 L p .787 g
F P
0 g
s
.49334 .56679 m .50505 .54669 L .4938 .55777 L p .787 g
F P
0 g
s
.842 .84121 m .83037 .87858 L .87019 .8097 L p F P
s
.88411 .77388 m .842 .84121 L p .86431 .81628 L F P
s
.80938 .90511 m .84549 .83678 L p .83246 .8728 L .864 g
F P
0 g
s
.83037 .87858 m .82631 .88371 L p .83246 .8728 L .864 g
F P
0 g
s
.82631 .88371 m .80938 .90511 L p .83246 .8728 L .864 g
F P
0 g
s
.87019 .8097 m .87503 .79725 L p .86431 .81628 L F P
s
.87503 .79725 m .88411 .77388 L p .86431 .81628 L F P
s
.71164 .50586 m .75497 .52793 L .74206 .51107 L p .383 g
F P
0 g
s
.78082 .53136 m .74206 .51107 L .75497 .52793 L p .352 g
F P
0 g
s
.85904 .64096 m .87806 .68666 L .85947 .62681 L p .049 g
F P
0 g
s
.85947 .62681 m .8415 .59105 L .85904 .64096 L p .049 g
F P
0 g
s
.61956 .96237 m .62906 .95447 L .53058 .92519 L p .169 g
F P
0 g
s
.62906 .95447 m .61956 .96237 L .70983 .95752 L p .077 g
F P
0 g
s
.70983 .95752 m .73018 .94707 L .62906 .95447 L p .077 g
F P
0 g
s
.66815 .49444 m .71164 .50586 L .70081 .49896 L p .046 g
F P
0 g
s
.74206 .51107 m .70081 .49896 L .71164 .50586 L p .383 g
F P
0 g
s
.55529 .52584 m .54719 .53027 L p .56446 .51476 L .009 g
F P
0 g
s
.5372 .52438 m .56608 .51365 L p .54435 .522 L .824 g
F P
0 g
s
.53832 .52432 m .53785 .52435 L p .54435 .522 L .824 g
F P
0 g
s
.53785 .52435 m .5372 .52438 L p .54435 .522 L .824 g
F P
0 g
s
.56608 .51365 m .55552 .51419 L .53832 .52432 L p .824 g
F P
0 g
s
.56608 .51365 m .5954 .50392 L p .5598 .51309 L .768 g
F P
0 g
s
.55552 .51419 m .55708 .51411 L p .5598 .51309 L .768 g
F P
0 g
s
.55708 .51411 m .56608 .51365 L p .5598 .51309 L .768 g
F P
0 g
s
.5954 .50392 m .57296 .50713 L .55552 .51419 L p .768 g
F P
0 g
s
.57102 .50887 m .56642 .51405 L .56446 .51476 L .009 g
F
.57296 .50713 m .57148 .5087 L p .57102 .50887 L F P
0 g
s
.57148 .5087 m .56642 .51405 L p .57102 .50887 L .009 g
F P
0 g
s
.56642 .51405 m .55529 .52584 L p .56446 .51476 L .009 g
F P
0 g
s
.57296 .50713 m .5954 .50392 L .61773 .49657 L p F P
s
.87806 .68666 m .88668 .73193 L .86731 .66358 L p .528 g
F P
0 g
s
.86731 .66358 m .85947 .62681 L .87806 .68666 L p .528 g
F P
0 g
s
.45209 .84738 m .45863 .86308 L .53058 .92519 L p .5 g
F P
0 g
s
.53159 .93126 m .53058 .92519 L .45863 .86308 L p .255 g
F P
0 g
s
.53058 .92519 m .53159 .93126 L .61956 .96237 L p .169 g
F P
0 g
s
.41802 .74814 m .42339 .76638 L .45209 .84738 L p .592 g
F P
0 g
s
.45863 .86308 m .45209 .84738 L .42339 .76638 L p .033 g
F P
0 g
s
.50858 .5525 m .4938 .55777 L .50505 .54669 L p F P
s
.50505 .54669 m .51613 .53663 L .50858 .5525 L p F P
s
.43546 .63991 m .44799 .61273 L .47199 .60051 L p .015 g
F P
0 g
s
.42569 .66884 m .43546 .63991 L .46264 .61376 L p .017 g
F P
0 g
s
.73018 .94707 m .70983 .95752 L .78007 .91853 L p .68 g
F P
0 g
s
.78007 .91853 m .80938 .90511 L .73018 .94707 L p .68 g
F P
0 g
s
.52583 .53761 m .51613 .53663 L .5372 .52438 L p F P
s
.46264 .61376 m .45539 .62822 L .42569 .66884 L p .017 g
F P
0 g
s
.41927 .69771 m .42569 .66884 L .45539 .62822 L p .023 g
F P
0 g
s
.61773 .49657 m .58616 .50405 L .57296 .50713 L p F P
s
.66815 .49444 m .62725 .49462 L .61773 .49657 L p .533 g
F P
0 g
s
.58616 .50405 m .61773 .49657 L .62725 .49462 L p .065 g
F P
0 g
s
.44799 .61273 m .4625 .589 L .48284 .58934 L p .017 g
F P
0 g
s
.84549 .83678 m .87019 .8097 L .83037 .87858 L p .864 g
F P
0 g
s
.62725 .49462 m .66815 .49444 L .65963 .49584 L p .088 g
F P
0 g
s
.70081 .49896 m .65963 .49584 L .66815 .49444 L p .046 g
F P
0 g
s
.45539 .62822 m .45072 .64296 L .41927 .69771 L p .023 g
F P
0 g
s
.41665 .72473 m .41927 .69771 L .45072 .64296 L p .032 g
F P
0 g
s
.88668 .73193 m .88411 .77388 L .86427 .69898 L p .776 g
F P
0 g
s
.86427 .69898 m .86731 .66358 L .88668 .73193 L p .776 g
F P
0 g
s
.49449 .581 m .48284 .58934 L .4625 .589 L p .021 g
F P
0 g
s
.4625 .589 m .4781 .57025 L .49449 .581 L p .021 g
F P
0 g
s
.45072 .64296 m .44896 .657 L .41665 .72473 L p .032 g
F P
0 g
s
.41802 .74814 m .41665 .72473 L .44896 .657 L p .042 g
F P
0 g
s
.60785 .95474 m .61956 .96237 L .53159 .93126 L p .364 g
F P
0 g
s
.61956 .96237 m .60785 .95474 L .68477 .95227 L p .629 g
F P
0 g
s
.68477 .95227 m .70983 .95752 L .61956 .96237 L p .629 g
F P
0 g
s
.50615 .57604 m .49449 .581 L .4781 .57025 L p .053 g
F P
0 g
s
.4781 .57025 m .4938 .55777 L .50615 .57604 L p .053 g
F P
0 g
s
.42339 .76638 m .43247 .77812 L .45863 .86308 L p .033 g
F P
0 g
s
.46763 .86806 m .45863 .86308 L .43247 .77812 L p .065 g
F P
0 g
s
.45863 .86308 m .46763 .86806 L .53159 .93126 L p .255 g
F P
0 g
s
.52143 .55495 m .50858 .5525 L .51613 .53663 L p .006 g
F P
0 g
s
.51613 .53663 m .52583 .53761 L .52143 .55495 L p .006 g
F P
0 g
s
.62725 .49462 m .59164 .50671 L .58616 .50405 L p .065 g
F P
0 g
s
.59164 .50671 m .62725 .49462 L .62113 .502 L p .129 g
F P
0 g
s
.65963 .49584 m .62113 .502 L .62725 .49462 L p .088 g
F P
0 g
s
.54719 .53027 m .55552 .51419 L .57296 .50713 L p .009 g
F P
0 g
s
.55529 .52584 m .57296 .50713 L .58616 .50405 L p .086 g
F P
0 g
s
.53919 .53838 m .53832 .52432 L .55552 .51419 L p F P
s
.55552 .51419 m .54719 .53027 L .53919 .53838 L p F P
s
.70081 .49896 m .74206 .51107 L .69216 .51426 L p .153 g
F P
0 g
s
.74206 .51107 m .78082 .53136 L .72166 .52433 L p .207 g
F P
0 g
s
.72166 .52433 m .69216 .51426 L .74206 .51107 L p .207 g
F P
0 g
s
.78082 .53136 m .81468 .55857 L .74739 .54006 L p .315 g
F P
0 g
s
.74739 .54006 m .72166 .52433 L .78082 .53136 L p .315 g
F P
0 g
s
.53283 .92325 m .53159 .93126 L .46763 .86806 L p .253 g
F P
0 g
s
.53159 .93126 m .53283 .92325 L .60785 .95474 L p .364 g
F P
0 g
s
.44896 .657 m .45023 .66941 L .41802 .74814 L p .042 g
F P
0 g
s
.42339 .76638 m .41802 .74814 L .45023 .66941 L p .051 g
F P
0 g
s
.88411 .77388 m .87019 .8097 L .85031 .73057 L p .871 g
F P
0 g
s
.85031 .73057 m .86427 .69898 L .88411 .77388 L p .871 g
F P
0 g
s
.65963 .49584 m .70081 .49896 L .66081 .51048 L p .139 g
F P
0 g
s
.69216 .51426 m .66081 .51048 L .70081 .49896 L p .153 g
F P
0 g
s
.81468 .55857 m .8415 .59105 L .76764 .56049 L p .449 g
F P
0 g
s
.76764 .56049 m .74739 .54006 L .81468 .55857 L p .449 g
F P
0 g
s
.58616 .50405 m .56139 .52559 L .55529 .52584 L p .086 g
F P
0 g
s
.56139 .52559 m .58616 .50405 L .59164 .50671 L p .168 g
F P
0 g
s
.53342 .54999 m .52583 .53761 L .53832 .52432 L p .001 g
F P
0 g
s
.53832 .52432 m .53919 .53838 L .53342 .54999 L p .001 g
F P
0 g
s
.80938 .90511 m .78007 .91853 L .81137 .85293 L p .927 g
F P
0 g
s
.81137 .85293 m .84549 .83678 L .80938 .90511 L p .927 g
F P
0 g
s
.70983 .95752 m .68477 .95227 L .74417 .91733 L p .856 g
F P
0 g
s
.74417 .91733 m .78007 .91853 L .70983 .95752 L p .856 g
F P
0 g
s
.51707 .57484 m .50615 .57604 L .4938 .55777 L p .088 g
F P
0 g
s
.4938 .55777 m .50858 .5525 L .51707 .57484 L p .088 g
F P
0 g
s
.62113 .502 m .65963 .49584 L .62961 .51322 L p .157 g
F P
0 g
s
.66081 .51048 m .62961 .51322 L .65963 .49584 L p .139 g
F P
0 g
s
.8415 .59105 m .85947 .62681 L .781 .58433 L p .572 g
F P
0 g
s
.781 .58433 m .76764 .56049 L .8415 .59105 L p .572 g
F P
0 g
s
.45023 .66941 m .45449 .67934 L .42339 .76638 L p .051 g
F P
0 g
s
.43247 .77812 m .42339 .76638 L .45449 .67934 L p .126 g
F P
0 g
s
.59164 .50671 m .56386 .53028 L .56139 .52559 L p .168 g
F P
0 g
s
.56386 .53028 m .59164 .50671 L .58783 .5172 L p .266 g
F P
0 g
s
.62113 .502 m .58783 .5172 L .59164 .50671 L p .129 g
F P
0 g
s
.43247 .77812 m .44474 .78242 L .46763 .86806 L p .065 g
F P
0 g
s
.47853 .8616 m .46763 .86806 L .44474 .78242 L p .229 g
F P
0 g
s
.46763 .86806 m .47853 .8616 L .53283 .92325 L p .253 g
F P
0 g
s
.53141 .56514 m .52143 .55495 L .52583 .53761 L p .068 g
F P
0 g
s
.52583 .53761 m .53342 .54999 L .53141 .56514 L p .068 g
F P
0 g
s
.58783 .5172 m .62113 .502 L .60057 .52232 L p .226 g
F P
0 g
s
.62961 .51322 m .60057 .52232 L .62113 .502 L p .157 g
F P
0 g
s
.5265 .57751 m .51707 .57484 L .50858 .5525 L p .135 g
F P
0 g
s
.50858 .5525 m .52143 .55495 L .5265 .57751 L p .135 g
F P
0 g
s
.85947 .62681 m .86731 .66358 L .78649 .61002 L p .659 g
F P
0 g
s
.78649 .61002 m .781 .58433 L .85947 .62681 L p .659 g
F P
0 g
s
.87019 .8097 m .84549 .83678 L .82612 .7561 L p .865 g
F P
0 g
s
.82612 .7561 m .85031 .73057 L .87019 .8097 L p .865 g
F P
0 g
s
.5947 .9314 m .60785 .95474 L .53283 .92325 L p .593 g
F P
0 g
s
.60785 .95474 m .5947 .9314 L .65663 .93098 L p .735 g
F P
0 g
s
.65663 .93098 m .68477 .95227 L .60785 .95474 L p .735 g
F P
0 g
s
.54182 .56147 m .54719 .53027 L .55529 .52584 L p .117 g
F P
0 g
s
.55529 .52584 m .54391 .55945 L .54182 .56147 L p .117 g
F P
0 g
s
.54391 .55945 m .55529 .52584 L .56139 .52559 L p .279 g
F P
0 g
s
.53976 .56615 m .53919 .53838 L .54719 .53027 L p .051 g
F P
0 g
s
.54719 .53027 m .54182 .56147 L .53976 .56615 L p .051 g
F P
0 g
s
.53422 .90113 m .53283 .92325 L .47853 .8616 L p .451 g
F P
0 g
s
.53283 .92325 m .53422 .90113 L .5947 .9314 L p .593 g
F P
0 g
s
.45449 .67934 m .46147 .68611 L .43247 .77812 L p .126 g
F P
0 g
s
.44474 .78242 m .43247 .77812 L .46147 .68611 L p .213 g
F P
0 g
s
.56139 .52559 m .54548 .56028 L .54391 .55945 L p .279 g
F P
0 g
s
.54548 .56028 m .56139 .52559 L .56386 .53028 L p .371 g
F P
0 g
s
.78007 .91853 m .74417 .91733 L .76997 .85663 L p .857 g
F P
0 g
s
.76997 .85663 m .81137 .85293 L .78007 .91853 L p .857 g
F P
0 g
s
.46264 .61376 m .47199 .60051 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.45539 .62822 m .46264 .61376 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53827 .57326 m .53342 .54999 L .53919 .53838 L p .054 g
F P
0 g
s
.53919 .53838 m .53976 .56615 L .53827 .57326 L p .054 g
F P
0 g
s
.47199 .60051 m .48284 .58934 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.45072 .64296 m .45539 .62822 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.58783 .5172 m .56203 .54062 L .56386 .53028 L p .266 g
F P
0 g
s
.56203 .54062 m .58783 .5172 L .57558 .53722 L p .309 g
F P
0 g
s
.60057 .52232 m .57558 .53722 L .58783 .5172 L p .226 g
F P
0 g
s
.48284 .58934 m .49449 .581 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.86731 .66358 m .86427 .69898 L .78361 .63586 L p .708 g
F P
0 g
s
.78361 .63586 m .78649 .61002 L .86731 .66358 L p .708 g
F P
0 g
s
.68477 .95227 m .65663 .93098 L .70408 .90102 L p .797 g
F P
0 g
s
.70408 .90102 m .74417 .91733 L .68477 .95227 L p .797 g
F P
0 g
s
.44896 .657 m .45072 .64296 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53378 .58391 m .5265 .57751 L .52143 .55495 L p .194 g
F P
0 g
s
.52143 .55495 m .53141 .56514 L .53378 .58391 L p .194 g
F P
0 g
s
.62961 .51322 m .66081 .51048 L .60168 .54615 L p .262 g
F P
0 g
s
.66081 .51048 m .69216 .51426 L .61775 .54556 L p .292 g
F P
0 g
s
.61775 .54556 m .60168 .54615 L .66081 .51048 L p .292 g
F P
0 g
s
.56386 .53028 m .5461 .56408 L .54548 .56028 L p .371 g
F P
0 g
s
.5461 .56408 m .56386 .53028 L .56203 .54062 L p .402 g
F P
0 g
s
.69216 .51426 m .72166 .52433 L .63283 .54847 L p .343 g
F P
0 g
s
.63283 .54847 m .61775 .54556 L .69216 .51426 L p .343 g
F P
0 g
s
.49449 .581 m .50615 .57604 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.60057 .52232 m .62961 .51322 L .58568 .55018 L p .259 g
F P
0 g
s
.60168 .54615 m .58568 .55018 L .62961 .51322 L p .262 g
F P
0 g
s
.45023 .66941 m .44896 .657 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.41829 .45125 m .37549 .43876 L .41892 .50645 L p .623 g
F P
0 g
s
.41892 .50645 m .4502 .51023 L .41829 .45125 L p .623 g
F P
0 g
s
.38746 .50919 m .41892 .50645 L .37549 .43876 L p .652 g
F P
0 g
s
.4502 .51023 m .41892 .50645 L .47864 .56387 L p .559 g
F P
0 g
s
.41892 .50645 m .38746 .50919 L .46282 .56787 L p .581 g
F P
0 g
s
.46282 .56787 m .47864 .56387 L .41892 .50645 L p .581 g
F P
0 g
s
.44474 .78242 m .45942 .7788 L .47853 .8616 L p .229 g
F P
0 g
s
.49063 .84375 m .47853 .8616 L .45942 .7788 L p .368 g
F P
0 g
s
.47853 .8616 m .49063 .84375 L .53422 .90113 L p .451 g
F P
0 g
s
.45045 .42683 m .40694 .39368 L .41829 .45125 L p .633 g
F P
0 g
s
.41829 .45125 m .45787 .47199 L .45045 .42683 L p .633 g
F P
0 g
s
.37549 .43876 m .41829 .45125 L .40694 .39368 L p .672 g
F P
0 g
s
.45787 .47199 m .41829 .45125 L .4502 .51023 L p .59 g
F P
0 g
s
.53777 .58254 m .53141 .56514 L .53342 .54999 L p .101 g
F P
0 g
s
.53342 .54999 m .53827 .57326 L .53777 .58254 L p .101 g
F P
0 g
s
.48751 .46716 m .45045 .42683 L .45787 .47199 L p .593 g
F P
0 g
s
.45787 .47199 m .49176 .49968 L .48751 .46716 L p .593 g
F P
0 g
s
.49176 .49968 m .45787 .47199 L .47929 .5203 L p .555 g
F P
0 g
s
.4502 .51023 m .47929 .5203 L .45787 .47199 L p .59 g
F P
0 g
s
.47929 .5203 m .4502 .51023 L .49446 .56331 L p .533 g
F P
0 g
s
.47864 .56387 m .49446 .56331 L .4502 .51023 L p .559 g
F P
0 g
s
.72166 .52433 m .74739 .54006 L .64591 .55469 L p .405 g
F P
0 g
s
.64591 .55469 m .63283 .54847 L .72166 .52433 L p .405 g
F P
0 g
s
.37549 .43876 m .33211 .43533 L .38746 .50919 L p .652 g
F P
0 g
s
.35783 .51829 m .38746 .50919 L .33211 .43533 L p .677 g
F P
0 g
s
.38746 .50919 m .35783 .51829 L .44801 .57503 L p .598 g
F P
0 g
s
.44801 .57503 m .46282 .56787 L .38746 .50919 L p .598 g
F P
0 g
s
.84549 .83678 m .81137 .85293 L .79319 .77359 L p .826 g
F P
0 g
s
.79319 .77359 m .82612 .7561 L .84549 .83678 L p .826 g
F P
0 g
s
.57558 .53722 m .60057 .52232 L .57081 .55738 L p .28 g
F P
0 g
s
.58568 .55018 m .57081 .55738 L .60057 .52232 L p .259 g
F P
0 g
s
.49269 .44272 m .45913 .39139 L .45045 .42683 L p .617 g
F P
0 g
s
.45045 .42683 m .48751 .46716 L .49269 .44272 L p .617 g
F P
0 g
s
.40694 .39368 m .45045 .42683 L .45913 .39139 L p .661 g
F P
0 g
s
.46147 .68611 m .47073 .68921 L .44474 .78242 L p .213 g
F P
0 g
s
.45942 .7788 m .44474 .78242 L .47073 .68921 L p .283 g
F P
0 g
s
.40694 .39368 m .35953 .36977 L .37549 .43876 L p .672 g
F P
0 g
s
.33211 .43533 m .37549 .43876 L .35953 .36977 L p .711 g
F P
0 g
s
.51829 .49771 m .49269 .44272 L .48751 .46716 L p .575 g
F P
0 g
s
.48751 .46716 m .51588 .51228 L .51829 .49771 L p .575 g
F P
0 g
s
.51588 .51228 m .48751 .46716 L .49176 .49968 L p .554 g
F P
0 g
s
.50615 .57604 m .51707 .57484 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.47929 .5203 m .50431 .53605 L .49176 .49968 L p .555 g
F P
0 g
s
.50431 .53605 m .47929 .5203 L .5092 .56623 L p .503 g
F P
0 g
s
.49446 .56331 m .5092 .56623 L .47929 .5203 L p .533 g
F P
0 g
s
.49176 .49968 m .51782 .53264 L .51588 .51228 L p .554 g
F P
0 g
s
.51782 .53264 m .49176 .49968 L .50431 .53605 L p .519 g
F P
0 g
s
.45449 .67934 m .45023 .66941 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.33211 .43533 m .29089 .44129 L .35783 .51829 L p .677 g
F P
0 g
s
.33198 .53321 m .35783 .51829 L .29089 .44129 L p .697 g
F P
0 g
s
.35783 .51829 m .33198 .53321 L .43522 .58488 L p .609 g
F P
0 g
s
.43522 .58488 m .44801 .57503 L .35783 .51829 L p .609 g
F P
0 g
s
.52441 .49086 m .506 .43017 L .49269 .44272 L p .579 g
F P
0 g
s
.49269 .44272 m .51829 .49771 L .52441 .49086 L p .579 g
F P
0 g
s
.45913 .39139 m .49269 .44272 L .506 .43017 L p .623 g
F P
0 g
s
.56203 .54062 m .54561 .57088 L .5461 .56408 L p .402 g
F P
0 g
s
.54561 .57088 m .56203 .54062 L .55632 .55703 L p .401 g
F P
0 g
s
.57558 .53722 m .55632 .55703 L .56203 .54062 L p .309 g
F P
0 g
s
.74739 .54006 m .76764 .56049 L .6561 .56384 L p .467 g
F P
0 g
s
.6561 .56384 m .64591 .55469 L .74739 .54006 L p .467 g
F P
0 g
s
.45913 .39139 m .41947 .34668 L .40694 .39368 L p .661 g
F P
0 g
s
.35953 .36977 m .40694 .39368 L .41947 .34668 L p .707 g
F P
0 g
s
.55632 .55703 m .57558 .53722 L .55805 .56728 L p .323 g
F P
0 g
s
.57081 .55738 m .55805 .56728 L .57558 .53722 L p .28 g
F P
0 g
s
.506 .43017 m .48177 .37191 L .45913 .39139 L p .623 g
F P
0 g
s
.41947 .34668 m .45913 .39139 L .48177 .37191 L p .669 g
F P
0 g
s
.86427 .69898 m .85031 .73057 L .77242 .66006 L p .728 g
F P
0 g
s
.77242 .66006 m .78361 .63586 L .86427 .69898 L p .728 g
F P
0 g
s
.53262 .49292 m .52391 .43203 L .506 .43017 L p .555 g
F P
0 g
s
.506 .43017 m .52441 .49086 L .53262 .49292 L p .555 g
F P
0 g
s
.48177 .37191 m .506 .43017 L .52391 .43203 L p .602 g
F P
0 g
s
.53598 .55062 m .52441 .49086 L .51829 .49771 L p .537 g
F P
0 g
s
.51829 .49771 m .53441 .55324 L .53598 .55062 L p .537 g
F P
0 g
s
.53441 .55324 m .51829 .49771 L .51588 .51228 L p .543 g
F P
0 g
s
.53995 .60606 m .47864 .56387 L .46282 .56787 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .46282 .56787 L .44801 .57503 L closepath p .479 g
F P
0 g
s
.51588 .51228 m .5338 .55954 L .53441 .55324 L p .543 g
F P
0 g
s
.5338 .55954 m .51588 .51228 L .51782 .53264 L p .525 g
F P
0 g
s
.50431 .53605 m .52358 .55649 L .51782 .53264 L p .519 g
F P
0 g
s
.52358 .55649 m .50431 .53605 L .5219 .57246 L p .473 g
F P
0 g
s
.5092 .56623 m .5219 .57246 L .50431 .53605 L p .503 g
F P
0 g
s
.53995 .60606 m .44801 .57503 L .43522 .58488 L closepath p .479 g
F P
0 g
s
.51707 .57484 m .5265 .57751 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53995 .60606 m .49446 .56331 L .47864 .56387 L closepath p .479 g
F P
0 g
s
.53808 .55204 m .53262 .49292 L .52441 .49086 L p .489 g
F P
0 g
s
.52441 .49086 m .53598 .55062 L .53808 .55204 L p .489 g
F P
0 g
s
.35953 .36977 m .31104 .35669 L .33211 .43533 L p .711 g
F P
0 g
s
.29089 .44129 m .33211 .43533 L .31104 .35669 L p .749 g
F P
0 g
s
.53141 .56514 m .53714 .58856 L .53633 .5893 L .226 g
F
.53838 .59364 m .53773 .59226 L p .53714 .58856 L F P
0 g
s
.53773 .59226 m .53633 .5893 L p .53714 .58856 L .226 g
F P
0 g
s
.53141 .56514 m .53777 .58254 L .53838 .59364 L p .226 g
F P
0 g
s
.53378 .58391 m .53838 .59364 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53633 .5893 m .53378 .58391 L .53288 .57682 L p .226 g
F P
0 g
s
.53288 .57682 m .53141 .56514 L p .53633 .5893 L .226 g
F P
0 g
s
.5265 .57751 m .53378 .58391 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53167 .5816 m .53784 .59305 L .53577 .58035 L p .451 g
F P
0 g
s
.53577 .58035 m .53784 .59305 L .53836 .59627 L p .498 g
F P
0 g
s
.53995 .60606 m .53784 .59305 L .53167 .5816 L closepath p .479 g
F P
0 g
s
.53836 .59627 m .53995 .60606 L .53852 .59724 L p .498 g
F P
0 g
s
.53852 .59724 m .53577 .58035 L p .53836 .59627 L .498 g
F P
0 g
s
.53429 .56886 m .53577 .58035 L .53852 .59724 L p .047 g
F P
0 g
s
.53852 .59724 m .53995 .60606 L .53873 .59799 L p .047 g
F P
0 g
s
.53873 .59799 m .53429 .56886 L p .53852 .59724 L .047 g
F P
0 g
s
.46147 .68611 m .45449 .67934 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.42527 .59678 m .43522 .58488 L .33198 .53321 L p .613 g
F P
0 g
s
.53995 .60606 m .43522 .58488 L .42527 .59678 L closepath p .479 g
F P
0 g
s
.29089 .44129 m .25451 .45641 L .33198 .53321 L p .697 g
F P
0 g
s
.31162 .55302 m .33198 .53321 L .25451 .45641 L p .71 g
F P
0 g
s
.33198 .53321 m .31162 .55302 L .42527 .59678 L p .613 g
F P
0 g
s
.51782 .53264 m .53429 .56886 L .5338 .55954 L p .525 g
F P
0 g
s
.53429 .56886 m .51782 .53264 L .52358 .55649 L p .494 g
F P
0 g
s
.58104 .89341 m .5947 .9314 L .53422 .90113 L p .628 g
F P
0 g
s
.5947 .9314 m .58104 .89341 L .62743 .89458 L p .699 g
F P
0 g
s
.62743 .89458 m .65663 .93098 L .5947 .9314 L p .699 g
F P
0 g
s
.53565 .86599 m .53422 .90113 L .49063 .84375 L p .532 g
F P
0 g
s
.53422 .90113 m .53565 .86599 L .58104 .89341 L p .628 g
F P
0 g
s
.54015 .55747 m .54072 .50402 L .53262 .49292 L p .398 g
F P
0 g
s
.53262 .49292 m .53808 .55204 L .54015 .55747 L p .398 g
F P
0 g
s
.52391 .43203 m .53262 .49292 L .54072 .50402 L p .5 g
F P
0 g
s
.53995 .60606 m .5092 .56623 L .49446 .56331 L closepath p .479 g
F P
0 g
s
.54072 .50402 m .54164 .44877 L .52391 .43203 L p .5 g
F P
0 g
s
.51244 .37271 m .52391 .43203 L .54164 .44877 L p .549 g
F P
0 g
s
.52391 .43203 m .51244 .37271 L .48177 .37191 L p .602 g
F P
0 g
s
.54391 .55945 m .53995 .60606 L .54182 .56147 L closepath p .403 g
F P
0 g
s
.54548 .56028 m .53995 .60606 L .54391 .55945 L closepath p .312 g
F P
0 g
s
.54182 .56147 m .53995 .60606 L .53976 .56615 L closepath p .47 g
F P
0 g
s
.76764 .56049 m .781 .58433 L .66268 .57532 L p .522 g
F P
0 g
s
.66268 .57532 m .6561 .56384 L .76764 .56049 L p .522 g
F P
0 g
s
.41886 .60995 m .42527 .59678 L .31162 .55302 L p .609 g
F P
0 g
s
.53995 .60606 m .42527 .59678 L .41886 .60995 L closepath p .479 g
F P
0 g
s
.5461 .56408 m .53995 .60606 L .54548 .56028 L closepath p .17 g
F P
0 g
s
.54164 .56647 m .54656 .52315 L .54072 .50402 L p .313 g
F P
0 g
s
.54072 .50402 m .54015 .55747 L .54164 .56647 L p .313 g
F P
0 g
s
.54164 .44877 m .54072 .50402 L .54656 .52315 L p .427 g
F P
0 g
s
.74417 .91733 m .70408 .90102 L .72413 .84727 L p .781 g
F P
0 g
s
.72413 .84727 m .76997 .85663 L .74417 .91733 L p .781 g
F P
0 g
s
.54656 .52315 m .5544 .47867 L .54164 .44877 L p .427 g
F P
0 g
s
.54284 .39483 m .54164 .44877 L .5544 .47867 L p .473 g
F P
0 g
s
.54164 .44877 m .54284 .39483 L .51244 .37271 L p .549 g
F P
0 g
s
.45303 .31937 m .48177 .37191 L .51244 .37271 L p .645 g
F P
0 g
s
.48177 .37191 m .45303 .31937 L .41947 .34668 L p .669 g
F P
0 g
s
.53995 .60606 m .5219 .57246 L .5092 .56623 L closepath p .479 g
F P
0 g
s
.53976 .56615 m .53995 .60606 L .53827 .57326 L closepath p .526 g
F P
0 g
s
.55632 .55703 m .54414 .58048 L .54561 .57088 L p .401 g
F P
0 g
s
.54414 .58048 m .55632 .55703 L .54824 .57923 L p .384 g
F P
0 g
s
.55805 .56728 m .54824 .57923 L .55632 .55703 L p .323 g
F P
0 g
s
.47073 .68921 m .48166 .68839 L .45942 .7788 L p .283 g
F P
0 g
s
.47554 .76734 m .45942 .7788 L .48166 .68839 L p .336 g
F P
0 g
s
.45942 .7788 m .47554 .76734 L .49063 .84375 L p .368 g
F P
0 g
s
.52358 .55649 m .53577 .58035 L .53429 .56886 L p .494 g
F P
0 g
s
.53577 .58035 m .52358 .55649 L .53167 .5816 L p .451 g
F P
0 g
s
.5219 .57246 m .53167 .5816 L .52358 .55649 L p .473 g
F P
0 g
s
.41947 .34668 m .37595 .31132 L .35953 .36977 L p .707 g
F P
0 g
s
.31104 .35669 m .35953 .36977 L .37595 .31132 L p .758 g
F P
0 g
s
.54561 .57088 m .53995 .60606 L .5461 .56408 L closepath p .031 g
F P
0 g
s
.47073 .68921 m .46147 .68611 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.54215 .57828 m .54855 .54833 L .54656 .52315 L p .28 g
F P
0 g
s
.54656 .52315 m .54164 .56647 L .54215 .57828 L p .28 g
F P
0 g
s
.5544 .47867 m .54656 .52315 L .54855 .54833 L p .366 g
F P
0 g
s
.54855 .54833 m .55874 .51807 L .5544 .47867 L p .366 g
F P
0 g
s
.56469 .43563 m .5544 .47867 L .55874 .51807 L p .401 g
F P
0 g
s
.5544 .47867 m .56469 .43563 L .54284 .39483 L p .473 g
F P
0 g
s
.31162 .55302 m .29818 .57649 L .41886 .60995 L p .609 g
F P
0 g
s
.41644 .62351 m .41886 .60995 L .29818 .57649 L p .596 g
F P
0 g
s
.53995 .60606 m .41886 .60995 L .41644 .62351 L closepath p .479 g
F P
0 g
s
.37595 .31132 m .41947 .34668 L .45303 .31937 L p .719 g
F P
0 g
s
.53827 .57326 m .53995 .60606 L .53777 .58254 L closepath p .577 g
F P
0 g
s
.53598 .55062 m .53995 .60606 L .53808 .55204 L closepath p .484 g
F P
0 g
s
.53777 .58254 m .53995 .60606 L .53838 .59364 L closepath p .628 g
F P
0 g
s
.53441 .55324 m .53995 .60606 L .53598 .55062 L closepath p .42 g
F P
0 g
s
.50307 .81547 m .49063 .84375 L .47554 .76734 L p .442 g
F P
0 g
s
.49063 .84375 m .50307 .81547 L .53565 .86599 L p .532 g
F P
0 g
s
.5338 .55954 m .53995 .60606 L .53441 .55324 L closepath p .331 g
F P
0 g
s
.25451 .45641 m .22546 .47988 L .31162 .55302 L p .71 g
F P
0 g
s
.29818 .57649 m .31162 .55302 L .22546 .47988 L p .713 g
F P
0 g
s
.53995 .60606 m .53167 .5816 L .5219 .57246 L closepath p .479 g
F P
0 g
s
.53808 .55204 m .53995 .60606 L .54015 .55747 L closepath p .536 g
F P
0 g
s
.53429 .56886 m .53995 .60606 L .5338 .55954 L closepath p .186 g
F P
0 g
s
.65663 .93098 m .62743 .89458 L .6627 .87031 L p .726 g
F P
0 g
s
.6627 .87031 m .70408 .90102 L .65663 .93098 L p .726 g
F P
0 g
s
.31104 .35669 m .26449 .35552 L .29089 .44129 L p .749 g
F P
0 g
s
.25451 .45641 m .29089 .44129 L .26449 .35552 L p .786 g
F P
0 g
s
.49881 .3183 m .51244 .37271 L .54284 .39483 L p .581 g
F P
0 g
s
.51244 .37271 m .49881 .3183 L .45303 .31937 L p .645 g
F P
0 g
s
.54561 .57088 m .54414 .58048 L .54139 .59729 L p F P
s
.54118 .59842 m .54561 .57088 L p .54139 .59729 L F P
s
.54414 .58048 m .54207 .59245 L .54154 .59582 L p .801 g
F P
0 g
s
.54139 .59729 m .53995 .60606 L .54118 .59842 L p F P
s
.54154 .59582 m .53995 .60606 L .54139 .59729 L p .801 g
F P
0 g
s
.54139 .59729 m .54414 .58048 L p .54154 .59582 L .801 g
F P
0 g
s
.54824 .57923 m .54207 .59245 L .54414 .58048 L p .384 g
F P
0 g
s
.54855 .54833 m .54278 .58415 L .54359 .58521 L .292 g
F
.54153 .59186 m .54219 .58975 L p .54278 .58415 L F P
0 g
s
.54219 .58975 m .54359 .58521 L p .54278 .58415 L .292 g
F P
0 g
s
.54359 .58521 m .54615 .57691 L .54705 .56614 L p .292 g
F P
0 g
s
.54705 .56614 m .54855 .54833 L p .54359 .58521 L .292 g
F P
0 g
s
.53995 .60606 m .57081 .55738 L .58568 .55018 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .58568 .55018 L .60168 .54615 L closepath p .479 g
F P
0 g
s
.54015 .55747 m .53995 .60606 L .54164 .56647 L closepath p .584 g
F P
0 g
s
.54615 .57691 m .55349 .56219 L .55874 .51807 L p .333 g
F P
0 g
s
.55874 .51807 m .54855 .54833 L .54615 .57691 L p .333 g
F P
0 g
s
.57208 .48937 m .55874 .51807 L .55349 .56219 L p .351 g
F P
0 g
s
.55874 .51807 m .57208 .48937 L .56469 .43563 L p .401 g
F P
0 g
s
.53995 .60606 m .55805 .56728 L .57081 .55738 L closepath p .479 g
F P
0 g
s
.81137 .85293 m .76997 .85663 L .7537 .78164 L p .778 g
F P
0 g
s
.7537 .78164 m .79319 .77359 L .81137 .85293 L p .778 g
F P
0 g
s
.53995 .60606 m .60168 .54615 L .61775 .54556 L closepath p .479 g
F P
0 g
s
.54855 .54833 m .54215 .57828 L .54153 .59186 L p .292 g
F P
0 g
s
.781 .58433 m .78649 .61002 L .66517 .58837 L p .565 g
F P
0 g
s
.66517 .58837 m .66268 .57532 L .781 .58433 L p .565 g
F P
0 g
s
.29818 .57649 m .29265 .60209 L .41644 .62351 L p .596 g
F P
0 g
s
.41819 .63656 m .41644 .62351 L .29265 .60209 L p .57 g
F P
0 g
s
.53995 .60606 m .41644 .62351 L .41819 .63656 L closepath p .479 g
F P
0 g
s
.85031 .73057 m .82612 .7561 L .75357 .68089 L p .728 g
F P
0 g
s
.75357 .68089 m .77242 .66006 L .85031 .73057 L p .728 g
F P
0 g
s
.54427 .34528 m .54284 .39483 L .56469 .43563 L p .491 g
F P
0 g
s
.54284 .39483 m .54427 .34528 L .49881 .3183 L p .581 g
F P
0 g
s
.53995 .60606 m .54824 .57923 L .55805 .56728 L closepath p .479 g
F P
0 g
s
.54164 .56647 m .53995 .60606 L .54215 .57828 L closepath p .632 g
F P
0 g
s
.57685 .39661 m .56469 .43563 L .57208 .48937 L p .408 g
F P
0 g
s
.56469 .43563 m .57685 .39661 L .54427 .34528 L p .491 g
F P
0 g
s
.53995 .60606 m .61775 .54556 L .63283 .54847 L closepath p .479 g
F P
0 g
s
.55349 .56219 m .56306 .54865 L .57208 .48937 L p .351 g
F P
0 g
s
.58776 .46403 m .57208 .48937 L .56306 .54865 L p .354 g
F P
0 g
s
.57208 .48937 m .58776 .46403 L .57685 .39661 L p .408 g
F P
0 g
s
.48166 .68839 m .47073 .68921 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53995 .60606 m .54207 .59245 L .54824 .57923 L closepath p .479 g
F P
0 g
s
.55349 .56219 m .54615 .57691 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.56306 .54865 m .55349 .56219 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.54215 .57828 m .53995 .60606 L .54153 .59186 L closepath p .683 g
F P
0 g
s
.57426 .53719 m .56306 .54865 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.56306 .54865 m .57426 .53719 L .58776 .46403 L p .354 g
F P
0 g
s
.53995 .60606 m .63283 .54847 L .64591 .55469 L closepath p .479 g
F P
0 g
s
.54615 .57691 m .54153 .59186 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.29265 .60209 m .29556 .6281 L .41819 .63656 L p .57 g
F P
0 g
s
.42404 .6482 m .41819 .63656 L .29556 .6281 L p .531 g
F P
0 g
s
.53995 .60606 m .41819 .63656 L .42404 .6482 L closepath p .479 g
F P
0 g
s
.22546 .47988 m .20585 .51034 L .29818 .57649 L p .713 g
F P
0 g
s
.29265 .60209 m .29818 .57649 L .20585 .51034 L p .702 g
F P
0 g
s
.48166 .68839 m .49351 .68366 L .47554 .76734 L p .336 g
F P
0 g
s
.49194 .7487 m .47554 .76734 L .49351 .68366 L p .376 g
F P
0 g
s
.47554 .76734 m .49194 .7487 L .50307 .81547 L p .442 g
F P
0 g
s
.42132 .27569 m .45303 .31937 L .49881 .3183 L p .687 g
F P
0 g
s
.45303 .31937 m .42132 .27569 L .37595 .31132 L p .719 g
F P
0 g
s
.58634 .52856 m .57426 .53719 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.57426 .53719 m .58634 .52856 L .60481 .44368 L p .344 g
F P
0 g
s
.60481 .44368 m .58776 .46403 L .57426 .53719 L p .344 g
F P
0 g
s
.37595 .31132 m .33107 .28767 L .31104 .35669 L p .758 g
F P
0 g
s
.26449 .35552 m .31104 .35669 L .33107 .28767 L p .814 g
F P
0 g
s
.49351 .68366 m .48166 .68839 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.58776 .46403 m .60481 .44368 L .59018 .36402 L p .393 g
F P
0 g
s
.59018 .36402 m .57685 .39661 L .58776 .46403 L p .393 g
F P
0 g
s
.53995 .60606 m .64591 .55469 L .6561 .56384 L closepath p .479 g
F P
0 g
s
.53703 .82011 m .53565 .86599 L .50307 .81547 L p .556 g
F P
0 g
s
.53565 .86599 m .53703 .82011 L .56792 .84324 L p .616 g
F P
0 g
s
.56792 .84324 m .58104 .89341 L .53565 .86599 L p .616 g
F P
0 g
s
.78649 .61002 m .78361 .63586 L .66337 .6021 L p .595 g
F P
0 g
s
.66337 .6021 m .66517 .58837 L .78649 .61002 L p .595 g
F P
0 g
s
.26449 .35552 m .22293 .36667 L .25451 .45641 L p .786 g
F P
0 g
s
.22546 .47988 m .25451 .45641 L .22293 .36667 L p .819 g
F P
0 g
s
.29556 .6281 m .30682 .65272 L .42404 .6482 L p .531 g
F P
0 g
s
.43362 .65761 m .42404 .6482 L .30682 .65272 L p .475 g
F P
0 g
s
.53995 .60606 m .42404 .6482 L .43362 .65761 L closepath p .479 g
F P
0 g
s
.58104 .89341 m .56792 .84324 L .59944 .84545 L p .654 g
F P
0 g
s
.59944 .84545 m .62743 .89458 L .58104 .89341 L p .654 g
F P
0 g
s
.59852 .52335 m .58634 .52856 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.58634 .52856 m .59852 .52335 L .62217 .4297 L p .321 g
F P
0 g
s
.62217 .4297 m .60481 .44368 L .58634 .52856 L p .321 g
F P
0 g
s
.33107 .28767 m .37595 .31132 L .42132 .27569 L p .776 g
F P
0 g
s
.51491 .77863 m .50307 .81547 L .49194 .7487 L p .481 g
F P
0 g
s
.50307 .81547 m .51491 .77863 L .53703 .82011 L p .556 g
F P
0 g
s
.60999 .52193 m .59852 .52335 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53995 .60606 m .43362 .65761 L .44629 .66414 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .5456 .65985 L .544 .66251 L closepath p .42 g
F P
0 g
s
.53995 .60606 m .54622 .65337 L .5456 .65985 L closepath p .331 g
F P
0 g
s
.53995 .60606 m .54217 .62987 L .54154 .61856 L closepath p .628 g
F P
0 g
s
.53995 .60606 m .544 .66251 L .54186 .66102 L closepath p .484 g
F P
0 g
s
.53995 .60606 m .54167 .63945 L .54217 .62987 L closepath p .577 g
F P
0 g
s
.53995 .60606 m .54186 .66102 L .53975 .6554 L closepath p .536 g
F P
0 g
s
.53995 .60606 m .54015 .64686 L .54167 .63945 L closepath p .526 g
F P
0 g
s
.53995 .60606 m .53975 .6554 L .53825 .64615 L closepath p .584 g
F P
0 g
s
.53322 .69057 m .53825 .64615 L .53975 .6554 L p .548 g
F P
0 g
s
.53995 .60606 m .53804 .65176 L .54015 .64686 L closepath p .47 g
F P
0 g
s
.53995 .60606 m .53774 .63409 L .53837 .62033 L closepath p .683 g
F P
0 g
s
.53995 .60606 m .53825 .64615 L .53774 .63409 L closepath p .632 g
F P
0 g
s
.53621 .63461 m .53488 .63505 L .53755 .62379 L .467 g
F
.5363 .62704 m .53837 .62033 L p .53488 .63505 L F P
0 g
s
.53837 .62033 m .53371 .63544 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53371 .63544 m .5263 .65034 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.5263 .65034 m .51665 .66396 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53995 .60606 m .53158 .63316 L .53783 .61974 L closepath p .479 g
F P
0 g
s
.53371 .63544 m .5363 .62704 L p .53488 .63505 L .467 g
F P
0 g
s
.5263 .65034 m .53371 .63544 L .53123 .66462 L p .439 g
F P
0 g
s
.53774 .63409 m .53123 .66462 L .53278 .6464 L p .467 g
F P
0 g
s
.53278 .6464 m .53371 .63544 L p .53488 .63505 L .467 g
F P
0 g
s
.53774 .63409 m .53278 .6464 L .53488 .63505 L .467 g
F
.53572 .63192 m .53783 .61974 L p .53758 .62037 L .061 g
F P
0 g
s
.51665 .66396 m .5263 .65034 L .52088 .6954 L p .423 g
F P
0 g
s
.52324 .65613 m .53572 .63192 L p .53758 .62037 L .061 g
F P
0 g
s
.53158 .63316 m .52324 .65613 L p .53758 .62037 L .061 g
F P
0 g
s
.53783 .61974 m .53759 .62025 L p .53758 .62037 L .061 g
F P
0 g
s
.53759 .62025 m .53158 .63316 L p .53758 .62037 L .061 g
F P
0 g
s
.53572 .63192 m .5342 .64181 L .53732 .62245 L p F P
s
.53783 .61974 m .53572 .63192 L .53908 .6114 L p .801 g
F P
0 g
s
.53732 .62245 m .53809 .61764 L p .53739 .62186 L F P
s
.53739 .62186 m .53572 .63192 L .53732 .62245 L F
.53809 .61764 m .53923 .61054 L p .53788 .61878 L F P
s
.53788 .61878 m .53572 .63192 L .53809 .61764 L F
.53908 .6114 m .53995 .60606 L .53891 .61281 L p .801 g
F P
0 g
s
.53891 .61281 m .53783 .61974 L p .53908 .6114 L .801 g
F P
0 g
s
.53837 .62033 m .53774 .63409 L p .53621 .63461 L .467 g
F P
0 g
s
.53621 .63461 m .53755 .62379 L .53837 .62033 L .467 g
F
.53923 .61054 m .53995 .60606 L .53908 .6114 L p 0 g
F P
s
.53908 .6114 m .53572 .63192 L p .53923 .61054 L F P
s
.53123 .66462 m .53774 .63409 L .53825 .64615 L p .53 g
F P
0 g
s
.53825 .64615 m .53322 .69057 L .53123 .66462 L p .53 g
F P
0 g
s
.53995 .60606 m .5359 .65385 L .53804 .65176 L closepath p .403 g
F P
0 g
s
.53995 .60606 m .5343 .65293 L .5359 .65385 L closepath p .312 g
F P
0 g
s
.53995 .60606 m .53368 .6489 L .5343 .65293 L closepath p .17 g
F P
0 g
s
.53995 .60606 m .5342 .64181 L .53368 .6489 L closepath p .031 g
F P
0 g
s
.53995 .60606 m .5457 .6438 L .54622 .65337 L closepath p .186 g
F P
0 g
s
.54833 .63081 m .55667 .65666 L p .54233 .61979 L .369 g
F P
0 g
s
.54208 .61915 m .54232 .61959 L p .54233 .61979 L .369 g
F P
0 g
s
.54232 .61959 m .54833 .63081 L p .54233 .61979 L .369 g
F P
0 g
s
.54418 .63205 m .5457 .6438 L .54259 .62337 L p .047 g
F P
0 g
s
.54259 .62337 m .54067 .61079 L p .54244 .62136 L .047 g
F P
0 g
s
.54083 .61142 m .54244 .62136 L p .54067 .61079 L .047 g
F P
0 g
s
.54244 .62136 m .54418 .63205 L p .54259 .62337 L .047 g
F P
0 g
s
.54208 .61915 m .54418 .63205 L .54083 .61142 L p .498 g
F P
0 g
s
.54067 .61079 m .53995 .60606 L .54083 .61142 L p .047 g
F P
0 g
s
.54083 .61142 m .53995 .60606 L .541 .61251 L p .498 g
F P
0 g
s
.541 .61251 m .54208 .61915 L p .54083 .61142 L .498 g
F P
0 g
s
.54154 .61856 m .54217 .62987 L p .54371 .62936 L .322 g
F P
0 g
s
.55667 .65666 m .54418 .63205 L .54208 .61915 L p .369 g
F P
0 g
s
.54371 .62936 m .54505 .62891 L .54236 .62099 L .322 g
F
.54371 .62936 m .54236 .62099 L .54154 .61856 L F
.54622 .62851 m .54361 .62298 L p .54505 .62891 L F P
0 g
s
.54361 .62298 m .54154 .61856 L p .54505 .62891 L .322 g
F P
0 g
s
.53995 .60606 m .6561 .56384 L .66268 .57532 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .66268 .57532 L .66517 .58837 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .66517 .58837 L .66337 .6021 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .66337 .6021 L .65736 .61556 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .65736 .61556 L .64755 .62782 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .64755 .62782 L .63459 .638 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .63459 .638 L .61939 .64539 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .61939 .64539 L .60303 .64946 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .60303 .64946 L .58664 .64993 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .58664 .64993 L .57139 .64678 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .57139 .64678 L .55832 .64025 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .55832 .64025 L .54833 .63081 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .54833 .63081 L .54208 .61915 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .44629 .66414 L .4612 .6673 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .4612 .6673 L .47731 .66686 L closepath p .479 g
F P
0 g
s
.50545 .67533 m .49351 .68366 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.51665 .66396 m .50545 .67533 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.53995 .60606 m .52162 .64536 L .53158 .63316 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .47731 .66686 L .4935 .66283 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .50863 .65549 L .52162 .64536 L closepath p .479 g
F P
0 g
s
.53995 .60606 m .4935 .66283 L .50863 .65549 L closepath p .479 g
F P
0 g
s
.61997 .52445 m .60999 .52193 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.62775 .53078 m .61997 .52445 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.54154 .61856 m .54622 .62851 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.63279 .54051 m .62775 .53078 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.54622 .62851 m .5537 .63522 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.6347 .55302 m .63279 .54051 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.5537 .63522 m .5635 .63819 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.63332 .56745 m .6347 .55302 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.5635 .63819 m .57498 .63717 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.62872 .5828 m .63332 .56745 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.57498 .63717 m .58736 .63219 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.62121 .59797 m .62872 .5828 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.58736 .63219 m .59977 .62357 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.61133 .61189 m .62121 .59797 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.59977 .62357 m .61133 .61189 L .53995 .60606 L closepath p .323 g
F P
0 g
s
.57685 .39661 m .59018 .36402 L .54584 .30314 L p .485 g
F P
0 g
s
.54584 .30314 m .54427 .34528 L .57685 .39661 L p .485 g
F P
0 g
s
.70408 .90102 m .6627 .87031 L .67722 .82522 L p .717 g
F P
0 g
s
.67722 .82522 m .72413 .84727 L .70408 .90102 L p .717 g
F P
0 g
s
.49351 .68366 m .50545 .67533 L .49194 .7487 L p .376 g
F P
0 g
s
.48375 .27205 m .49881 .3183 L .54427 .34528 L p .6 g
F P
0 g
s
.49881 .3183 m .48375 .27205 L .42132 .27569 L p .687 g
F P
0 g
s
.82612 .7561 m .79319 .77359 L .72832 .69684 L p .716 g
F P
0 g
s
.72832 .69684 m .75357 .68089 L .82612 .7561 L p .716 g
F P
0 g
s
.20585 .51034 m .19727 .54585 L .29265 .60209 L p .702 g
F P
0 g
s
.29556 .6281 m .29265 .60209 L .19727 .54585 L p .668 g
F P
0 g
s
.54427 .34528 m .54584 .30314 L .48375 .27205 L p .6 g
F P
0 g
s
.30682 .65272 m .32576 .67421 L .43362 .65761 L p .475 g
F P
0 g
s
.44629 .66414 m .43362 .65761 L .32576 .67421 L p .403 g
F P
0 g
s
.50744 .72414 m .49194 .7487 L .50545 .67533 L p .403 g
F P
0 g
s
.49194 .7487 m .50744 .72414 L .51491 .77863 L p .481 g
F P
0 g
s
.59852 .52335 m .60999 .52193 L .63871 .42314 L p .285 g
F P
0 g
s
.63871 .42314 m .62217 .4297 L .59852 .52335 L p .285 g
F P
0 g
s
.76997 .85663 m .72413 .84727 L .71044 .77948 L p .731 g
F P
0 g
s
.71044 .77948 m .7537 .78164 L .76997 .85663 L p .731 g
F P
0 g
s
.50545 .67533 m .51665 .66396 L .50744 .72414 L p .403 g
F P
0 g
s
.62743 .89458 m .59944 .84545 L .62323 .82722 L p .667 g
F P
0 g
s
.62323 .82722 m .6627 .87031 L .62743 .89458 L p .667 g
F P
0 g
s
.60481 .44368 m .62217 .4297 L .60389 .34008 L p .349 g
F P
0 g
s
.60389 .34008 m .59018 .36402 L .60481 .44368 L p .349 g
F P
0 g
s
.78361 .63586 m .77242 .66006 L .65736 .61556 L p .613 g
F P
0 g
s
.65736 .61556 m .66337 .6021 L .78361 .63586 L p .613 g
F P
0 g
s
.32576 .67421 m .35115 .69099 L .44629 .66414 L p .403 g
F P
0 g
s
.4612 .6673 m .44629 .66414 L .35115 .69099 L p .319 g
F P
0 g
s
.52088 .6954 m .50744 .72414 L .51665 .66396 L p .423 g
F P
0 g
s
.53123 .66462 m .52088 .6954 L .5263 .65034 L p .439 g
F P
0 g
s
.50325 .67698 m .52324 .65613 L .53158 .63316 L p .094 g
F P
0 g
s
.53158 .63316 m .52162 .64536 L .50325 .67698 L p .094 g
F P
0 g
s
.56268 .68144 m .5457 .6438 L .54418 .63205 L p .407 g
F P
0 g
s
.54418 .63205 m .55667 .65666 L .56268 .68144 L p .407 g
F P
0 g
s
.51715 .67366 m .5342 .64181 L .53572 .63192 L p .06 g
F P
0 g
s
.53572 .63192 m .52324 .65613 L .51715 .67366 L p .06 g
F P
0 g
s
.60999 .52193 m .61997 .52445 L .65331 .42461 L p .234 g
F P
0 g
s
.65331 .42461 m .63871 .42314 L .60999 .52193 L p .234 g
F P
0 g
s
.57665 .67813 m .55667 .65666 L .54833 .63081 L p .46 g
F P
0 g
s
.54833 .63081 m .55832 .64025 L .57665 .67813 L p .46 g
F P
0 g
s
.50744 .72414 m .52088 .6954 L .52524 .7359 L p .501 g
F P
0 g
s
.52524 .7359 m .51491 .77863 L .50744 .72414 L p .501 g
F P
0 g
s
.35115 .69099 m .3812 .70181 L .4612 .6673 L p .319 g
F P
0 g
s
.47731 .66686 m .4612 .6673 L .3812 .70181 L p .232 g
F P
0 g
s
.54217 .62987 m .54873 .64811 L .54622 .62851 L p .322 g
F P
0 g
s
.56474 .70261 m .54622 .65337 L .5457 .6438 L p .445 g
F P
0 g
s
.5457 .6438 m .56268 .68144 L .56474 .70261 L p .445 g
F P
0 g
s
.54873 .64811 m .54217 .62987 L .54167 .63945 L p .318 g
F P
0 g
s
.51491 .77863 m .52524 .7359 L .53823 .76687 L p .558 g
F P
0 g
s
.53823 .76687 m .53703 .82011 L .51491 .77863 L p .558 g
F P
0 g
s
.47707 .69294 m .50325 .67698 L .52162 .64536 L p .107 g
F P
0 g
s
.52162 .64536 m .50863 .65549 L .47707 .69294 L p .107 g
F P
0 g
s
.53703 .82011 m .53823 .76687 L .55641 .7846 L p .595 g
F P
0 g
s
.55641 .7846 m .56792 .84324 L .53703 .82011 L p .595 g
F P
0 g
s
.22293 .36667 m .18929 .38982 L .22546 .47988 L p .819 g
F P
0 g
s
.20585 .51034 m .22546 .47988 L .18929 .38982 L p .838 g
F P
0 g
s
.5537 .63522 m .54622 .62851 L .54873 .64811 L p .212 g
F P
0 g
s
.19727 .54585 m .2006 .58403 L .29556 .6281 L p .668 g
F P
0 g
s
.30682 .65272 m .29556 .6281 L .2006 .58403 L p .6 g
F P
0 g
s
.52088 .6954 m .53123 .66462 L .53322 .69057 L p .513 g
F P
0 g
s
.3812 .70181 m .41377 .70585 L .47731 .66686 L p .232 g
F P
0 g
s
.4935 .66283 m .47731 .66686 L .41377 .70585 L p .155 g
F P
0 g
s
.33107 .28767 m .28757 .27762 L .26449 .35552 L p .814 g
F P
0 g
s
.22293 .36667 m .26449 .35552 L .28757 .27762 L p .877 g
F P
0 g
s
.44652 .70282 m .47707 .69294 L .50863 .65549 L p .123 g
F P
0 g
s
.50863 .65549 m .4935 .66283 L .44652 .70282 L p .123 g
F P
0 g
s
.53322 .69057 m .52524 .7359 L .52088 .6954 L p .513 g
F P
0 g
s
.53975 .6554 m .53917 .71046 L .53322 .69057 L p .548 g
F P
0 g
s
.53917 .71046 m .53975 .6554 L .54186 .66102 L p .544 g
F P
0 g
s
.41377 .70585 m .44652 .70282 L .4935 .66283 L p .155 g
F P
0 g
s
.515 .68517 m .53368 .6489 L .5342 .64181 L p .086 g
F P
0 g
s
.5342 .64181 m .51715 .67366 L .515 .68517 L p .086 g
F P
0 g
s
.56792 .84324 m .55641 .7846 L .57493 .78723 L p .612 g
F P
0 g
s
.57493 .78723 m .59944 .84545 L .56792 .84324 L p .612 g
F P
0 g
s
.79319 .77359 m .7537 .78164 L .6984 .70671 L p .695 g
F P
0 g
s
.6984 .70671 m .72832 .69684 L .79319 .77359 L p .695 g
F P
0 g
s
.38843 .2438 m .42132 .27569 L .48375 .27205 L p .729 g
F P
0 g
s
.42132 .27569 m .38843 .2438 L .33107 .28767 L p .776 g
F P
0 g
s
.56226 .71765 m .5456 .65985 L .54622 .65337 L p .478 g
F P
0 g
s
.54622 .65337 m .56474 .70261 L .56226 .71765 L p .478 g
F P
0 g
s
.54186 .66102 m .54748 .72217 L .53917 .71046 L p .544 g
F P
0 g
s
.54748 .72217 m .54186 .66102 L .544 .66251 L p .528 g
F P
0 g
s
.60279 .69489 m .57665 .67813 L .55832 .64025 L p .511 g
F P
0 g
s
.55832 .64025 m .57139 .64678 L .60279 .69489 L p .511 g
F P
0 g
s
.5456 .65985 m .56226 .71765 L .55594 .72455 L p .506 g
F P
0 g
s
.54167 .63945 m .54676 .66441 L .54873 .64811 L p .318 g
F P
0 g
s
.55594 .72455 m .544 .66251 L .5456 .65985 L p .506 g
F P
0 g
s
.54676 .66441 m .54167 .63945 L .54015 .64686 L p .304 g
F P
0 g
s
.77242 .66006 m .75357 .68089 L .64755 .62782 L p .621 g
F P
0 g
s
.64755 .62782 m .65736 .61556 L .77242 .66006 L p .621 g
F P
0 g
s
.59018 .36402 m .60389 .34008 L .54748 .27122 L p .442 g
F P
0 g
s
.54748 .27122 m .54584 .30314 L .59018 .36402 L p .442 g
F P
0 g
s
.544 .66251 m .55594 .72455 L .54748 .72217 L p .528 g
F P
0 g
s
.6627 .87031 m .62323 .82722 L .6328 .79199 L p .66 g
F P
0 g
s
.6328 .79199 m .67722 .82522 L .6627 .87031 L p .66 g
F P
0 g
s
.52524 .7359 m .53322 .69057 L .53917 .71046 L p .552 g
F P
0 g
s
.53917 .71046 m .53823 .76687 L .52524 .7359 L p .552 g
F P
0 g
s
.61997 .52445 m .62775 .53078 L .66491 .43421 L p .167 g
F P
0 g
s
.66491 .43421 m .65331 .42461 L .61997 .52445 L p .167 g
F P
0 g
s
.5174 .69072 m .5343 .65293 L .53368 .6489 L p .118 g
F P
0 g
s
.53368 .6489 m .515 .68517 L .5174 .69072 L p .118 g
F P
0 g
s
.72413 .84727 m .67722 .82522 L .66657 .76716 L p .684 g
F P
0 g
s
.66657 .76716 m .71044 .77948 L .72413 .84727 L p .684 g
F P
0 g
s
.58986 .71622 m .56268 .68144 L .55667 .65666 L p .499 g
F P
0 g
s
.55667 .65666 m .57665 .67813 L .58986 .71622 L p .499 g
F P
0 g
s
.62217 .4297 m .63871 .42314 L .6171 .32661 L p .264 g
F P
0 g
s
.6171 .32661 m .60389 .34008 L .62217 .4297 L p .264 g
F P
0 g
s
.4897 .69932 m .51715 .67366 L .52324 .65613 L p .066 g
F P
0 g
s
.52324 .65613 m .50325 .67698 L .4897 .69932 L p .066 g
F P
0 g
s
.5635 .63819 m .5537 .63522 L .55931 .65945 L p .115 g
F P
0 g
s
.54873 .64811 m .55931 .65945 L .5537 .63522 L p .212 g
F P
0 g
s
.54015 .64686 m .54075 .67715 L .54676 .66441 L p .304 g
F P
0 g
s
.54075 .67715 m .54015 .64686 L .53804 .65176 L p .276 g
F P
0 g
s
.63328 .7057 m .60279 .69489 L .57139 .64678 L p .548 g
F P
0 g
s
.57139 .64678 m .58664 .64993 L .63328 .7057 L p .548 g
F P
0 g
s
.59944 .84545 m .57493 .78723 L .58883 .77497 L p .615 g
F P
0 g
s
.58883 .77497 m .62323 .82722 L .59944 .84545 L p .615 g
F P
0 g
s
.53823 .76687 m .53917 .71046 L .54748 .72217 L p .569 g
F P
0 g
s
.54748 .72217 m .55641 .7846 L .53823 .76687 L p .569 g
F P
0 g
s
.28757 .27762 m .33107 .28767 L .38843 .2438 L p .842 g
F P
0 g
s
.52375 .69081 m .5359 .65385 L .5343 .65293 L p .177 g
F P
0 g
s
.5343 .65293 m .5174 .69072 L .52375 .69081 L p .177 g
F P
0 g
s
.2006 .58403 m .21586 .62219 L .30682 .65272 L p .6 g
F P
0 g
s
.32576 .67421 m .30682 .65272 L .21586 .62219 L p .489 g
F P
0 g
s
.53804 .65176 m .53232 .68609 L .54075 .67715 L p .276 g
F P
0 g
s
.53232 .68609 m .53804 .65176 L .5359 .65385 L p .234 g
F P
0 g
s
.5359 .65385 m .52375 .69081 L .53232 .68609 L p .234 g
F P
0 g
s
.55641 .7846 m .54748 .72217 L .55594 .72455 L p .572 g
F P
0 g
s
.55594 .72455 m .57493 .78723 L .55641 .7846 L p .572 g
F P
0 g
s
.66594 .70974 m .63328 .7057 L .58664 .64993 L p .576 g
F P
0 g
s
.58664 .64993 m .60303 .64946 L .66594 .70974 L p .576 g
F P
0 g
s
.54584 .30314 m .54748 .27122 L .4681 .23705 L p .595 g
F P
0 g
s
.4681 .23705 m .48375 .27205 L .54584 .30314 L p .595 g
F P
0 g
s
.48375 .27205 m .4681 .23705 L .38843 .2438 L p .729 g
F P
0 g
s
.59432 .75004 m .56474 .70261 L .56268 .68144 L p .528 g
F P
0 g
s
.56268 .68144 m .58986 .71622 L .59432 .75004 L p .528 g
F P
0 g
s
.75357 .68089 m .72832 .69684 L .63459 .638 L p .62 g
F P
0 g
s
.63459 .638 m .64755 .62782 L .75357 .68089 L p .62 g
F P
0 g
s
.62775 .53078 m .63279 .54051 L .67262 .45147 L p .085 g
F P
0 g
s
.67262 .45147 m .66491 .43421 L .62775 .53078 L p .085 g
F P
0 g
s
.55931 .65945 m .54873 .64811 L .54676 .66441 L p .115 g
F P
0 g
s
.7537 .78164 m .71044 .77948 L .66594 .70974 L p .669 g
F P
0 g
s
.66594 .70974 m .6984 .70671 L .7537 .78164 L p .669 g
F P
0 g
s
.57493 .78723 m .55594 .72455 L .56226 .71765 L p .564 g
F P
0 g
s
.6984 .70671 m .66594 .70974 L .60303 .64946 L p .597 g
F P
0 g
s
.60303 .64946 m .61939 .64539 L .6984 .70671 L p .597 g
F P
0 g
s
.57498 .63717 m .5635 .63819 L .57328 .66295 L p .074 g
F P
0 g
s
.55931 .65945 m .57328 .66295 L .5635 .63819 L p .115 g
F P
0 g
s
.61939 .64539 m .63459 .638 L .72832 .69684 L p .612 g
F P
0 g
s
.62535 .74554 m .58986 .71622 L .57665 .67813 L p .556 g
F P
0 g
s
.57665 .67813 m .60279 .69489 L .62535 .74554 L p .556 g
F P
0 g
s
.58883 .77497 m .56226 .71765 L .56474 .70261 L p .549 g
F P
0 g
s
.56474 .70261 m .59432 .75004 L .58883 .77497 L p .549 g
F P
0 g
s
.72832 .69684 m .6984 .70671 L .61939 .64539 L p .612 g
F P
0 g
s
.56226 .71765 m .58883 .77497 L .57493 .78723 L p .564 g
F P
0 g
s
.18929 .38982 m .16618 .4238 L .20585 .51034 L p .838 g
F P
0 g
s
.19727 .54585 m .20585 .51034 L .16618 .4238 L p .822 g
F P
0 g
s
.62323 .82722 m .58883 .77497 L .59432 .75004 L p .606 g
F P
0 g
s
.59432 .75004 m .6328 .79199 L .62323 .82722 L p .606 g
F P
0 g
s
.45355 .71682 m .4897 .69932 L .50325 .67698 L p .062 g
F P
0 g
s
.50325 .67698 m .47707 .69294 L .45355 .71682 L p .062 g
F P
0 g
s
.67722 .82522 m .6328 .79199 L .62535 .74554 L p .637 g
F P
0 g
s
.62535 .74554 m .66657 .76716 L .67722 .82522 L p .637 g
F P
0 g
s
.21586 .62219 m .24214 .65754 L .32576 .67421 L p .489 g
F P
0 g
s
.35115 .69099 m .32576 .67421 L .24214 .65754 L p .337 g
F P
0 g
s
.48474 .7122 m .515 .68517 L .51715 .67366 L p .053 g
F P
0 g
s
.51715 .67366 m .4897 .69932 L .48474 .7122 L p .053 g
F P
0 g
s
.63279 .54051 m .6347 .55302 L .6758 .47533 L p .047 g
F P
0 g
s
.6758 .47533 m .67262 .45147 L .63279 .54051 L p .047 g
F P
0 g
s
.6328 .79199 m .59432 .75004 L .58986 .71622 L p .586 g
F P
0 g
s
.58986 .71622 m .62535 .74554 L .6328 .79199 L p .586 g
F P
0 g
s
.66657 .76716 m .62535 .74554 L .60279 .69489 L p .6 g
F P
0 g
s
.60279 .69489 m .63328 .7057 L .66657 .76716 L p .6 g
F P
0 g
s
.58736 .63219 m .57498 .63717 L .58975 .65817 L p .033 g
F P
0 g
s
.57328 .66295 m .58975 .65817 L .57498 .63717 L p .074 g
F P
0 g
s
.71044 .77948 m .66657 .76716 L .63328 .7057 L p .637 g
F P
0 g
s
.63328 .7057 m .66594 .70974 L .71044 .77948 L p .637 g
F P
0 g
s
.63871 .42314 m .65331 .42461 L .62892 .32489 L p .131 g
F P
0 g
s
.62892 .32489 m .6171 .32661 L .63871 .42314 L p .131 g
F P
0 g
s
.41111 .72472 m .45355 .71682 L .47707 .69294 L p .073 g
F P
0 g
s
.47707 .69294 m .44652 .70282 L .41111 .72472 L p .073 g
F P
0 g
s
.28757 .27762 m .24835 .28238 L .22293 .36667 L p .877 g
F P
0 g
s
.18929 .38982 m .22293 .36667 L .24835 .28238 L p .929 g
F P
0 g
s
.24214 .65754 m .27764 .68741 L .35115 .69099 L p .337 g
F P
0 g
s
.3812 .70181 m .35115 .69099 L .27764 .68741 L p .186 g
F P
0 g
s
.54676 .66441 m .55502 .67904 L .55931 .65945 L p .115 g
F P
0 g
s
.55502 .67904 m .54676 .66441 L .54075 .67715 L p .083 g
F P
0 g
s
.60389 .34008 m .6171 .32661 L .54906 .25199 L p .319 g
F P
0 g
s
.54906 .25199 m .54748 .27122 L .60389 .34008 L p .319 g
F P
0 g
s
.6347 .55302 m .63332 .56745 L .67412 .50417 L p .037 g
F P
0 g
s
.67412 .50417 m .6758 .47533 L .6347 .55302 L p .037 g
F P
0 g
s
.3654 .72228 m .41111 .72472 L .44652 .70282 L p .095 g
F P
0 g
s
.38738 .71481 m .3654 .72228 L p .38818 .71681 L .095 g
F P
0 g
s
.31978 .70954 m .3654 .72228 L .41377 .70585 L p .13 g
F P
0 g
s
.27764 .68741 m .31978 .70954 L .3812 .70181 L p .186 g
F P
0 g
s
.33826 .70722 m .31978 .70954 L p .34005 .70875 L .13 g
F P
0 g
s
.2006 .58403 m .15877 .51519 L p .19841 .58949 L .498 g
F P
0 g
s
.21586 .62219 m .20293 .58984 L p .19841 .58949 L .498 g
F P
0 g
s
.20293 .58984 m .2006 .58403 L p .19841 .58949 L .498 g
F P
0 g
s
.15877 .51519 m .17577 .56623 L .21586 .62219 L p .498 g
F P
0 g
s
.24214 .65754 m .21586 .62219 L .20819 .61149 L p .05 g
F P
0 g
s
.24214 .65754 m .20819 .61149 L .21491 .62026 L .05 g
F
.19287 .5901 m .17577 .56623 L p .24214 .65754 L F P
0 g
s
.20819 .61149 m .19287 .5901 L p .21491 .62026 L .05 g
F P
0 g
s
.20558 .61579 m .24214 .65754 L p .19234 .58902 L .05 g
F P
0 g
s
.27764 .68741 m .24214 .65754 L .23 .64368 L p .187 g
F P
0 g
s
.27764 .68741 m .23 .64368 L .24809 .65804 L .187 g
F
.23 .64368 m .20558 .61579 L p .24809 .65804 L F P
0 g
s
.24602 .66 m .27764 .68741 L p .23024 .6403 L .187 g
F P
0 g
s
.31978 .70954 m .27764 .68741 L .26476 .67625 L p .325 g
F P
0 g
s
.26476 .67625 m .24602 .66 L p .28777 .68805 L .325 g
F P
0 g
s
.31978 .70954 m .26476 .67625 L .28777 .68805 L .325 g
F
.24602 .66 m .29397 .69544 L .29786 .69757 L p F P
0 g
s
.29786 .69757 m .31978 .70954 L p .27547 .67978 L .325 g
F P
0 g
s
.27547 .67978 m .24602 .66 L .29786 .69757 L .325 g
F
.20558 .61579 m .24171 .6553 L p .23024 .6403 L .187 g
F P
0 g
s
.24171 .6553 m .24602 .66 L p .23024 .6403 L .187 g
F P
0 g
s
.17577 .56623 m .18934 .58879 L p .19234 .58902 L .05 g
F P
0 g
s
.18934 .58879 m .20558 .61579 L p .19234 .58902 L .05 g
F P
0 g
s
.3654 .72228 m .31978 .70954 L .30834 .70329 L p .409 g
F P
0 g
s
.30834 .70329 m .29397 .69544 L p .33379 .7104 L .409 g
F P
0 g
s
.3654 .72228 m .30834 .70329 L .33379 .7104 L .409 g
F
.29397 .69544 m .34568 .71944 L .35029 .72011 L p F P
0 g
s
.35029 .72011 m .35395 .72063 L p .33939 .71452 L .409 g
F P
0 g
s
.33939 .71452 m .29397 .69544 L .35029 .72011 L .409 g
F
.34568 .71944 m .39328 .72956 L p .39213 .72669 L .443 g
F P
0 g
s
.41111 .72472 m .39091 .72364 L p .39068 .72307 L .443 g
F P
0 g
s
.39091 .72364 m .3654 .72228 L p .39068 .72307 L .443 g
F P
0 g
s
.35395 .72063 m .3654 .72228 L p .34841 .71589 L .409 g
F P
0 g
s
.34841 .71589 m .29397 .69544 L .35395 .72063 L .409 g
F
.3654 .72228 m .35511 .7208 L p .39068 .72307 L .443 g
F P
0 g
s
.39068 .72307 m .35511 .7208 L .36348 .72088 L .443 g
F
.39068 .72307 m .37451 .72177 L .39213 .72669 L F
.39716 .73038 m .41111 .72472 L p .39068 .72307 L F P
0 g
s
.39328 .72956 m .39716 .73038 L p .39068 .72307 L .443 g
F P
0 g
s
.40944 .72414 m .39716 .73038 L p .38465 .72108 L .682 g
F P
0 g
s
.41377 .70585 m .3812 .70181 L .33826 .70722 L p .13 g
F P
0 g
s
.41377 .70585 m .33826 .70722 L .34005 .70875 L .13 g
F
.44652 .70282 m .38738 .71481 L .38818 .71681 L .095 g
F
.36163 .70396 m .40261 .72125 L p .38465 .72108 L .682 g
F P
0 g
s
.40261 .72125 m .40944 .72414 L p .38465 .72108 L .682 g
F P
0 g
s
.35511 .7208 m .34568 .71944 L p .36348 .72088 L .443 g
F P
0 g
s
.37451 .72177 m .34568 .71944 L .39213 .72669 L .443 g
F
.44652 .70282 m .41377 .70585 L .38738 .71481 L p .095 g
F P
0 g
s
.59977 .62357 m .58736 .63219 L .60763 .64524 L p .014 g
F P
0 g
s
.58975 .65817 m .60763 .64524 L .58736 .63219 L p .033 g
F P
0 g
s
.4899 .71659 m .5174 .69072 L .515 .68517 L p .055 g
F P
0 g
s
.515 .68517 m .48474 .7122 L .4899 .71659 L p .055 g
F P
0 g
s
.57328 .66295 m .55931 .65945 L .55502 .67904 L p .004 g
F P
0 g
s
.63332 .56745 m .62872 .5828 L .66766 .5359 L p .028 g
F P
0 g
s
.66766 .5359 m .67412 .50417 L .63332 .56745 L p .028 g
F P
0 g
s
.35635 .22619 m .38843 .2438 L .4681 .23705 L p .745 g
F P
0 g
s
.38843 .2438 m .35635 .22619 L .28757 .27762 L p .842 g
F P
0 g
s
.16618 .4238 m .15561 .46654 L .19727 .54585 L p .822 g
F P
0 g
s
.2006 .58403 m .19727 .54585 L .15561 .46654 L p .721 g
F P
0 g
s
.61133 .61189 m .59977 .62357 L .62565 .62494 L p .012 g
F P
0 g
s
.60763 .64524 m .62565 .62494 L .59977 .62357 L p .014 g
F P
0 g
s
.44461 .72777 m .48474 .7122 L .4897 .69932 L p .028 g
F P
0 g
s
.4897 .69932 m .45355 .71682 L .44461 .72777 L p .028 g
F P
0 g
s
.62872 .5828 m .62121 .59797 L .65622 .56872 L p .019 g
F P
0 g
s
.62121 .59797 m .61133 .61189 L .64247 .59863 L p .014 g
F P
0 g
s
.62565 .62494 m .64247 .59863 L .61133 .61189 L p .012 g
F P
0 g
s
.64247 .59863 m .65684 .5682 L .62121 .59797 L p .014 g
F P
0 g
s
.60783 .61854 m .62139 .57724 L .64142 .59756 L p .926 g
F P
0 g
s
.64142 .59756 m .64247 .59863 L p .64019 .59994 L .926 g
F P
0 g
s
.60783 .61854 m .64142 .59756 L .64019 .59994 L .926 g
F
.65622 .56872 m .65684 .5682 L p .65636 .56845 L .019 g
F P
0 g
s
.65636 .56845 m .62872 .5828 L .65622 .56872 L .019 g
F
.65684 .5682 m .66766 .5359 L .62872 .5828 L p F P
0 g
s
.54075 .67715 m .54173 .69458 L .55502 .67904 L p .083 g
F P
0 g
s
.54173 .69458 m .54075 .67715 L .53232 .68609 L p .071 g
F P
0 g
s
.24835 .28238 m .28757 .27762 L .35635 .22619 L p .888 g
F P
0 g
s
.50391 .71411 m .52375 .69081 L .5174 .69072 L p .06 g
F P
0 g
s
.5174 .69072 m .4899 .71659 L .50391 .71411 L p .06 g
F P
0 g
s
.65331 .42461 m .66491 .43421 L .63846 .33552 L p .054 g
F P
0 g
s
.63846 .33552 m .62892 .32489 L .65331 .42461 L p .054 g
F P
0 g
s
.54748 .27122 m .54906 .25199 L .45282 .21602 L p .51 g
F P
0 g
s
.45282 .21602 m .45983 .22567 L p .47671 .22996 L .51 g
F P
0 g
s
.4681 .23705 m .45282 .21602 L .35635 .22619 L p .745 g
F P
0 g
s
.32725 .22475 m .35635 .22619 L .44012 .21736 L p .586 g
F P
0 g
s
.44012 .21736 m .45282 .21602 L p .44049 .21688 L .586 g
F P
0 g
s
.32725 .22475 m .44012 .21736 L .44049 .21688 L .586 g
F
.45282 .21602 m .43894 .2111 L .32725 .22475 L p F P
0 g
s
.45282 .21602 m .54906 .25199 L p .46201 .21966 L .077 g
F P
0 g
s
.43894 .2111 m .44363 .21276 L p .46201 .21966 L .077 g
F P
0 g
s
.44363 .21276 m .45282 .21602 L p .46201 .21966 L .077 g
F P
0 g
s
.4681 .23705 m .54748 .27122 L p .47671 .22996 L .51 g
F P
0 g
s
.45983 .22567 m .4681 .23705 L p .47671 .22996 L .51 g
F P
0 g
s
.55166 .25832 m .42752 .22351 L p .51994 .24141 L .231 g
F P
0 g
s
.54906 .25199 m .5501 .24859 L p .51791 .24042 L .077 g
F P
0 g
s
.5501 .24859 m .55049 .24733 L p .51791 .24042 L .077 g
F P
0 g
s
.54906 .25199 m .6171 .32661 L p .57327 .27444 L .14 g
F P
0 g
s
.55049 .24733 m .54921 .2515 L p .57327 .27444 L .14 g
F P
0 g
s
.54921 .2515 m .54906 .25199 L p .57327 .27444 L .14 g
F P
0 g
s
.6171 .32661 m .62892 .32489 L .55049 .24733 L p .14 g
F P
0 g
s
.62892 .32489 m .63846 .33552 L .62167 .32059 L p .354 g
F P
0 g
s
.62167 .32059 m .55166 .25832 L p .6005 .3004 L .354 g
F P
0 g
s
.6005 .3004 m .62892 .32489 L .62167 .32059 L .354 g
F
.55049 .24733 m .55111 .25312 L p .51994 .24141 L .231 g
F P
0 g
s
.55111 .25312 m .55166 .25832 L p .51994 .24141 L .231 g
F P
0 g
s
.55049 .24733 m .44374 .21265 L p .51791 .24042 L .077 g
F P
0 g
s
.44374 .21265 m .43894 .2111 L p .4436 .21283 L .077 g
F P
0 g
s
.4436 .21283 m .51791 .24042 L .44374 .21265 L .077 g
F
.53232 .68609 m .52296 .70639 L .54173 .69458 L p .071 g
F P
0 g
s
.52296 .70639 m .53232 .68609 L .52375 .69081 L p .066 g
F P
0 g
s
.52375 .69081 m .50391 .71411 L .52296 .70639 L p .066 g
F P
0 g
s
.24835 .28238 m .21626 .30231 L .18929 .38982 L p .929 g
F P
0 g
s
.16618 .4238 m .18929 .38982 L .21626 .30231 L p .886 g
F P
0 g
s
.15561 .46654 m .15877 .51519 L .2006 .58403 L p .721 g
F P
0 g
s
.58975 .65817 m .57328 .66295 L .56601 .68199 L p F P
s
.55502 .67904 m .56601 .68199 L .57328 .66295 L p .004 g
F P
0 g
s
.39716 .73038 m .44461 .72777 L .45355 .71682 L p .004 g
F P
0 g
s
.48474 .7122 m .44461 .72777 L .45318 .72822 L p F P
s
.45318 .72822 m .44461 .72777 L p .42746 .726 L .634 g
F P
0 g
s
.45355 .71682 m .41111 .72472 L .41037 .72502 L p .004 g
F P
0 g
s
.45355 .71682 m .41037 .72502 L .4177 .72544 L .004 g
F
.40944 .72414 m .43242 .72628 L p .42746 .726 L .634 g
F P
0 g
s
.43242 .72628 m .45318 .72822 L p .42746 .726 L .634 g
F P
0 g
s
.41037 .72502 m .39716 .73038 L p .4177 .72544 L .004 g
F P
0 g
s
.56601 .68199 m .55502 .67904 L .54173 .69458 L p F P
s
.45318 .72822 m .4899 .71659 L .48474 .7122 L p F P
s
.66491 .43421 m .67262 .45147 L .64495 .3582 L p .023 g
F P
0 g
s
.63846 .33552 m .66491 .43421 L p .64847 .37162 L .023 g
F P
0 g
s
.6758 .47533 m .6478 .39165 L p .66517 .43353 L .633 g
F P
0 g
s
.67262 .45147 m .67516 .47054 L p .66517 .43353 L .633 g
F P
0 g
s
.67516 .47054 m .6758 .47533 L p .66517 .43353 L .633 g
F P
0 g
s
.64495 .3582 m .64327 .35233 L p .64847 .37162 L .023 g
F P
0 g
s
.64327 .35233 m .63846 .33552 L p .64847 .37162 L .023 g
F P
0 g
s
.60763 .64524 m .58975 .65817 L .57907 .67259 L p .827 g
F P
0 g
s
.56601 .68199 m .57907 .67259 L .58975 .65817 L p F P
s
.35635 .22619 m .32725 .22475 L .24835 .28238 L p .888 g
F P
0 g
s
.21626 .30231 m .24835 .28238 L .32725 .22475 L p .728 g
F P
0 g
s
.54173 .69458 m .54303 .69739 L .56601 .68199 L p F P
s
.54303 .69739 m .54173 .69458 L .52296 .70639 L p .627 g
F P
0 g
s
.47731 .72146 m .50391 .71411 L .4899 .71659 L p F P
s
.4899 .71659 m .45318 .72822 L .47731 .72146 L p F P
s
.21626 .30231 m .19394 .33668 L .16618 .4238 L p .886 g
F P
0 g
s
.15561 .46654 m .16618 .4238 L .19394 .33668 L p F P
s
.6478 .39165 m .64495 .3582 L .67262 .45147 L p .633 g
F P
0 g
s
.52296 .70639 m .5104 .71058 L .54303 .69739 L p .627 g
F P
0 g
s
.5104 .71058 m .52296 .70639 L .50391 .71411 L p .61 g
F P
0 g
s
.50391 .71411 m .47731 .72146 L .5104 .71058 L p .61 g
F P
0 g
s
.44461 .72777 m .39716 .73038 L .40944 .72414 L p .634 g
F P
0 g
s
.57907 .67259 m .56601 .68199 L .54303 .69739 L p .874 g
F P
0 g
s
.55166 .25832 m .55049 .24733 L .62892 .32489 L p .354 g
F P
0 g
s
.62565 .62494 m .60763 .64524 L .59335 .65105 L p .916 g
F P
0 g
s
.57907 .67259 m .59335 .65105 L .60763 .64524 L p .827 g
F P
0 g
s
.6758 .47533 m .67412 .50417 L .64669 .43363 L p .766 g
F P
0 g
s
.64669 .43363 m .6478 .39165 L .6758 .47533 L p .766 g
F P
0 g
s
.64247 .59863 m .62565 .62494 L .60783 .61854 L p .926 g
F P
0 g
s
.59335 .65105 m .60783 .61854 L .62565 .62494 L p .916 g
F P
0 g
s
.19394 .33668 m .18344 .38357 L .15561 .46654 L p F P
s
.15877 .51519 m .15561 .46654 L .18344 .38357 L p .04 g
F P
0 g
s
.67412 .50417 m .66766 .5359 L .64163 .48104 L p .838 g
F P
0 g
s
.64163 .48104 m .64669 .43363 L .67412 .50417 L p .838 g
F P
0 g
s
.39716 .73038 m .34568 .71944 L .36163 .70396 L p .682 g
F P
0 g
s
.32725 .22475 m .3033 .24044 L .21626 .30231 L p .728 g
F P
0 g
s
.19394 .33668 m .21626 .30231 L .3033 .24044 L p F P
s
.65684 .5682 m .64247 .59863 L .62139 .57724 L p .909 g
F P
0 g
s
.44547 .71151 m .47731 .72146 L .45318 .72822 L p .763 g
F P
0 g
s
.45318 .72822 m .40944 .72414 L .44547 .71151 L p .763 g
F P
0 g
s
.66766 .5359 m .65684 .5682 L .63297 .5302 L p .88 g
F P
0 g
s
.63297 .5302 m .64163 .48104 L .66766 .5359 L p .88 g
F P
0 g
s
.54303 .69739 m .54459 .68465 L .57907 .67259 L p .874 g
F P
0 g
s
.54459 .68465 m .54303 .69739 L .5104 .71058 L p .87 g
F P
0 g
s
.3033 .24044 m .32725 .22475 L .43894 .2111 L p .078 g
F P
0 g
s
.62139 .57724 m .63297 .5302 L .65684 .5682 L p .909 g
F P
0 g
s
.59335 .65105 m .57907 .67259 L .54459 .68465 L p .915 g
F P
0 g
s
.63846 .33552 m .64495 .3582 L .55247 .2849 L p .556 g
F P
0 g
s
.55247 .2849 m .55166 .25832 L .63846 .33552 L p .556 g
F P
0 g
s
.42752 .22351 m .43894 .2111 L .55049 .24733 L p .231 g
F P
0 g
s
.18344 .38357 m .18604 .43985 L .15877 .51519 L p .04 g
F P
0 g
s
.17577 .56623 m .15877 .51519 L .18604 .43985 L p .161 g
F P
0 g
s
.49535 .69746 m .5104 .71058 L .47731 .72146 L p .826 g
F P
0 g
s
.47731 .72146 m .44547 .71151 L .49535 .69746 L p .826 g
F P
0 g
s
.5104 .71058 m .49535 .69746 L .54459 .68465 L p .87 g
F P
0 g
s
.31327 .66863 m .36163 .70396 L .34568 .71944 L p .635 g
F P
0 g
s
.34568 .71944 m .29397 .69544 L .31327 .66863 L p .635 g
F P
0 g
s
.43894 .2111 m .42752 .22351 L .3033 .24044 L p .078 g
F P
0 g
s
.18604 .43985 m .20191 .50134 L .17577 .56623 L p .161 g
F P
0 g
s
.20558 .61579 m .17577 .56623 L .20191 .50134 L p .342 g
F P
0 g
s
.26817 .62051 m .31327 .66863 L .29397 .69544 L p .559 g
F P
0 g
s
.29397 .69544 m .24602 .66 L .26817 .62051 L p .559 g
F P
0 g
s
.3033 .24044 m .28653 .27303 L .19394 .33668 L p F P
s
.18344 .38357 m .19394 .33668 L .28653 .27303 L p .123 g
F P
0 g
s
.60783 .61854 m .59335 .65105 L .54631 .65652 L p .88 g
F P
0 g
s
.54459 .68465 m .54631 .65652 L .59335 .65105 L p .915 g
F P
0 g
s
.64495 .3582 m .6478 .39165 L .55285 .32573 L p .657 g
F P
0 g
s
.55285 .32573 m .55247 .2849 L .64495 .3582 L p .657 g
F P
0 g
s
.41046 .68407 m .44547 .71151 L .40944 .72414 L p .756 g
F P
0 g
s
.40944 .72414 m .36163 .70396 L .41046 .68407 L p .756 g
F P
0 g
s
.20191 .50134 m .23001 .56323 L .20558 .61579 L p .342 g
F P
0 g
s
.24602 .66 m .20558 .61579 L .23001 .56323 L p .465 g
F P
0 g
s
.23001 .56323 m .26817 .62051 L .24602 .66 L p .465 g
F P
0 g
s
.54631 .65652 m .54459 .68465 L .49535 .69746 L p .848 g
F P
0 g
s
.28653 .27303 m .3033 .24044 L .42752 .22351 L p .228 g
F P
0 g
s
.55166 .25832 m .55247 .2849 L .41954 .25329 L p .436 g
F P
0 g
s
.41954 .25329 m .42752 .22351 L .55166 .25832 L p .436 g
F P
0 g
s
.62139 .57724 m .60783 .61854 L .54805 .61444 L p .84 g
F P
0 g
s
.54631 .65652 m .54805 .61444 L .60783 .61854 L p .88 g
F P
0 g
s
.6478 .39165 m .64669 .43363 L .55274 .3781 L p .709 g
F P
0 g
s
.55274 .3781 m .55285 .32573 L .6478 .39165 L p .709 g
F P
0 g
s
.47877 .66706 m .49535 .69746 L .44547 .71151 L p .797 g
F P
0 g
s
.44547 .71151 m .41046 .68407 L .47877 .66706 L p .797 g
F P
0 g
s
.49535 .69746 m .47877 .66706 L .54631 .65652 L p .848 g
F P
0 g
s
.42752 .22351 m .41954 .25329 L .28653 .27303 L p .228 g
F P
0 g
s
.28653 .27303 m .27857 .32084 L .18344 .38357 L p .123 g
F P
0 g
s
.18604 .43985 m .18344 .38357 L .27857 .32084 L p .316 g
F P
0 g
s
.37485 .64046 m .41046 .68407 L .36163 .70396 L p .698 g
F P
0 g
s
.36163 .70396 m .31327 .66863 L .37485 .64046 L p .698 g
F P
0 g
s
.63297 .5302 m .62139 .57724 L .5497 .56121 L p .806 g
F P
0 g
s
.54805 .61444 m .5497 .56121 L .62139 .57724 L p .84 g
F P
0 g
s
.64669 .43363 m .64163 .48104 L .55214 .43806 L p .745 g
F P
0 g
s
.55214 .43806 m .55274 .3781 L .64669 .43363 L p .745 g
F P
0 g
s
.54805 .61444 m .54631 .65652 L .47877 .66706 L p .797 g
F P
0 g
s
.64163 .48104 m .63297 .5302 L .5511 .5008 L p .775 g
F P
0 g
s
.5497 .56121 m .5511 .5008 L .63297 .5302 L p .806 g
F P
0 g
s
.5511 .5008 m .55214 .43806 L .64163 .48104 L p .775 g
F P
0 g
s
.55247 .2849 m .55285 .32573 L .41578 .299 L p .539 g
F P
0 g
s
.41578 .299 m .41954 .25329 L .55247 .2849 L p .539 g
F P
0 g
s
.27857 .32084 m .28653 .27303 L .41954 .25329 L p .393 g
F P
0 g
s
.27857 .32084 m .28041 .38072 L .18604 .43985 L p .316 g
F P
0 g
s
.20191 .50134 m .18604 .43985 L .28041 .38072 L p .431 g
F P
0 g
s
.34148 .58356 m .37485 .64046 L .31327 .66863 L p .638 g
F P
0 g
s
.31327 .66863 m .26817 .62051 L .34148 .58356 L p .638 g
F P
0 g
s
.46186 .62087 m .47877 .66706 L .41046 .68407 L p .742 g
F P
0 g
s
.41046 .68407 m .37485 .64046 L .46186 .62087 L p .742 g
F P
0 g
s
.47877 .66706 m .46186 .62087 L .54805 .61444 L p .797 g
F P
0 g
s
.28041 .38072 m .2922 .44815 L .20191 .50134 L p .431 g
F P
0 g
s
.23001 .56323 m .20191 .50134 L .2922 .44815 L p .512 g
F P
0 g
s
.41954 .25329 m .41578 .299 L .27857 .32084 L p .393 g
F P
0 g
s
.31312 .51767 m .34148 .58356 L .26817 .62051 L p .578 g
F P
0 g
s
.26817 .62051 m .23001 .56323 L .31312 .51767 L p .578 g
F P
0 g
s
.2922 .44815 m .31312 .51767 L .23001 .56323 L p .512 g
F P
0 g
s
.5497 .56121 m .54805 .61444 L .46186 .62087 L p .753 g
F P
0 g
s
.55285 .32573 m .55274 .3781 L .41671 .3576 L p .599 g
F P
0 g
s
.41671 .3576 m .41578 .299 L .55285 .32573 L p .599 g
F P
0 g
s
.28041 .38072 m .27857 .32084 L .41578 .299 L p .487 g
F P
0 g
s
.44597 .56193 m .46186 .62087 L .37485 .64046 L p .691 g
F P
0 g
s
.37485 .64046 m .34148 .58356 L .44597 .56193 L p .691 g
F P
0 g
s
.5511 .5008 m .5497 .56121 L .44597 .56193 L p .714 g
F P
0 g
s
.46186 .62087 m .44597 .56193 L .5497 .56121 L p .753 g
F P
0 g
s
.55274 .3781 m .55214 .43806 L .42239 .42466 L p .642 g
F P
0 g
s
.42239 .42466 m .41671 .3576 L .55274 .3781 L p .642 g
F P
0 g
s
.41578 .299 m .41671 .3576 L .28041 .38072 L p .487 g
F P
0 g
s
.2922 .44815 m .28041 .38072 L .41671 .3576 L p .55 g
F P
0 g
s
.55214 .43806 m .5511 .5008 L .43242 .49471 L p .679 g
F P
0 g
s
.44597 .56193 m .43242 .49471 L .5511 .5008 L p .714 g
F P
0 g
s
.43242 .49471 m .42239 .42466 L .55214 .43806 L p .679 g
F P
0 g
s
.43242 .49471 m .44597 .56193 L .34148 .58356 L p .646 g
F P
0 g
s
.34148 .58356 m .31312 .51767 L .43242 .49471 L p .646 g
F P
0 g
s
.41671 .3576 m .42239 .42466 L .2922 .44815 L p .55 g
F P
0 g
s
.31312 .51767 m .2922 .44815 L .42239 .42466 L p .6 g
F P
0 g
s
.42239 .42466 m .43242 .49471 L .31312 .51767 L p .6 g
F P
0 g
s
.25 Mabswid
.51908 0 m
.96413 .59478 L
s
.96413 .59478 m
1 .99897 L
s
1 .99897 m
.51687 .44401 L
s
.51687 .44401 m
.51908 0 L
s
.04875 .18654 m
0 .62042 L
s
0 .62042 m
.51687 .44401 L
s
.51687 .44401 m
.51908 0 L
s
.51908 0 m
.04875 .18654 L
s
.04875 .18654 m
.51908 0 L
s
.05931 .18235 m
.06452 .18785 L
s
gsave
.04889 .17135 -81.0812 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.16652 .13983 m
.17158 .14547 L
s
gsave
.15639 .12856 -84.7934 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.27681 .09609 m
.28172 .10186 L
s
gsave
.26699 .08455 -68.8055 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.39032 .05107 m
.39506 .05698 L
s
gsave
.38085 .03925 -78.6224 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.25) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.5072 .00471 m
.51176 .01076 L
s
gsave
.49809 -0.00739 -74.2579 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08051 .17394 m
.08362 .17726 L
s
.10183 .16549 m
.10492 .16882 L
s
.12327 .15699 m
.12634 .16033 L
s
.14483 .14843 m
.14789 .1518 L
s
.18832 .13118 m
.19134 .13458 L
s
.21025 .12249 m
.21326 .1259 L
s
.23231 .11374 m
.2353 .11717 L
s
.2545 .10494 m
.25746 .10839 L
s
.29925 .08719 m
.30217 .09067 L
s
.32182 .07824 m
.32472 .08173 L
s
.34452 .06923 m
.34741 .07275 L
s
.36736 .06018 m
.37022 .06371 L
s
.41342 .04191 m
.41625 .04547 L
s
.43666 .03269 m
.43946 .03627 L
s
.46004 .02342 m
.46282 .02701 L
s
.48355 .01409 m
.48631 .01771 L
s
gsave
.21792 .02683 -68.3429 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.35119 0.00827773 0.64881 0.339387 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.30443 1.63518 -0.0106411 1.63518 [
[.5 1.63703 -21 0 ]
[.5 1.63703 21 10.625 ]
[.08347 .2009 -13.133 -10 ]
[.08347 .2009 .80453 0 ]
[.21284 .14994 -18.984 -10 ]
[.21284 .14994 1.64105 0 ]
[.3456 .09765 -7.58187 -10 ]
[.3456 .09765 .85563 0 ]
[.4819 .04399 -13.2609 -10 ]
[.4819 .04399 1.86415 0 ]
[.62186 -0.01111 -7.21329 -10 ]
[.62186 -0.01111 1.22421 0 ]
[.2819 .01774 -7.13257 -10 ]
[.2819 .01774 .80493 0 ]
[.67411 .0028 0 -6.95638 ]
[.67411 .0028 13.9375 3.04362 ]
[.74455 .10274 0 -6.89654 ]
[.74455 .10274 20.625 3.10346 ]
[.81167 .19797 0 -6.84024 ]
[.81167 .19797 8.4375 3.15976 ]
[.87569 .28881 0 -6.7872 ]
[.87569 .28881 15.125 3.2128 ]
[.93683 .37557 0 -6.73712 ]
[.93683 .37557 8.4375 3.26288 ]
[.90758 .16267 0 -6.84024 ]
[.90758 .16267 7.9375 3.15976 ]
[.06357 .25073 -13.9375 -3.04815 ]
[.06357 .25073 0 6.95185 ]
[.04583 .48129 -13.9375 -3.16049 ]
[.04583 .48129 0 6.83951 ]
[.02659 .73148 -8.4375 -3.28285 ]
[.02659 .73148 0 6.71715 ]
[.00564 1.00394 -8.4375 -3.41663 ]
[.00564 1.00394 0 6.58337 ]
[-0.01725 1.30176 -8.4375 -3.56352 ]
[-0.01725 1.30176 0 6.43648 ]
[-0.07007 .76468 -7.9375 -3.28285 ]
[-0.07007 .76468 0 6.71715 ]
[ 0 0 0 0 ]
[ 1 1.62453 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.63703 -82 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18.625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.625000 moveto
 460.000000 14.625000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.625000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(d) show
67.437500 13.187500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
70.437500 11.937500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(2) show
74.187500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
79.312500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
101.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
.64785 0 L
s
.09701 .21621 m
.10379 .22387 L
s
gsave
.08347 .2009 -74.133 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.226 .16558 m
.23257 .17341 L
s
gsave
.21284 .14994 -79.984 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.35834 .11363 m
.36471 .12163 L
s
gsave
.3456 .09765 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4942 .06031 m
.50035 .06847 L
s
gsave
.4819 .04399 -74.2609 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.63369 .00556 m
.63961 .01389 L
s
gsave
.62186 -0.01111 -68.2133 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12255 .20619 m
.12659 .2108 L
s
.14821 .19612 m
.15223 .20075 L
s
.17401 .18599 m
.178 .19064 L
s
.19993 .17581 m
.2039 .18049 L
s
.25219 .1553 m
.25611 .16001 L
s
.27852 .14497 m
.28242 .1497 L
s
.30499 .13458 m
.30886 .13933 L
s
.3316 .12413 m
.33545 .12891 L
s
.38523 .10308 m
.38903 .1079 L
s
.41226 .09247 m
.41603 .09731 L
s
.43943 .08181 m
.44317 .08666 L
s
.46674 .07109 m
.47046 .07596 L
s
.5218 .04948 m
.52546 .05439 L
s
.54955 .03858 m
.55318 .04352 L
s
.57745 .02763 m
.58105 .03259 L
s
.60549 .01662 m
.60907 .0216 L
s
gsave
.2819 .01774 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.64785 0 m
.92348 .39072 L
s
.65508 .01025 m
.64556 .01397 L
s
gsave
.67411 .0028 -61 -10.9564 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.72544 .10999 m
.71588 .11361 L
s
gsave
.74455 .10274 -61 -10.8965 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.79249 .20503 m
.78289 .20856 L
s
gsave
.81167 .19797 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.85644 .29569 m
.84682 .29913 L
s
gsave
.87569 .28881 -61 -10.7872 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.91753 .38228 m
.90787 .38563 L
s
gsave
.93683 .37557 -61 -10.7371 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.66943 .03059 m
.66371 .03281 L
s
.68363 .05073 m
.67791 .05294 L
s
.69771 .07067 m
.69198 .07287 L
s
.71164 .09043 m
.70591 .09261 L
s
.73911 .12936 m
.73337 .13152 L
s
.75264 .14855 m
.7469 .1507 L
s
.76605 .16756 m
.7603 .1697 L
s
.77933 .18638 m
.77358 .18851 L
s
.80552 .2235 m
.79976 .22561 L
s
.81843 .2418 m
.81266 .2439 L
s
.83122 .25993 m
.82545 .26202 L
s
.84389 .2779 m
.83812 .27997 L
s
.86888 .31333 m
.86311 .31538 L
s
.88121 .3308 m
.87543 .33284 L
s
.89343 .34812 m
.88764 .35015 L
s
.90553 .36527 m
.89974 .3673 L
s
gsave
.90758 .16267 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
0 1.32748 L
s
.08261 .2433 m
.09213 .23959 L
s
gsave
.06357 .25073 -74.9375 -7.04815 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.06502 .47423 m
.07461 .4707 L
s
gsave
.04583 .48129 -74.9375 -7.16049 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.04592 .72484 m
.05559 .72152 L
s
gsave
.02659 .73148 -69.4375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.02512 .99776 m
.03487 .99468 L
s
gsave
.00564 1.00394 -69.4375 -7.41663 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00239 1.29611 m
.01221 1.29329 L
s
gsave
-0.01725 1.30176 -69.4375 -7.56352 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.07921 .28803 m
.08493 .28582 L
s
.07574 .33346 m
.08147 .33128 L
s
.07222 .37963 m
.07796 .37747 L
s
.06865 .42655 m
.0744 .42441 L
s
.06132 .5227 m
.06709 .52061 L
s
.05757 .57198 m
.06334 .56991 L
s
.05375 .62208 m
.05953 .62004 L
s
.04987 .67303 m
.05566 .67101 L
s
.0419 .77755 m
.04771 .77559 L
s
.03782 .83117 m
.04364 .82924 L
s
.03366 .88573 m
.03949 .88382 L
s
.02943 .94125 m
.03527 .93937 L
s
.02074 1.05529 m
.0266 1.05347 L
s
.01628 1.11386 m
.02214 1.11207 L
s
.01173 1.1735 m
.01761 1.17174 L
s
.0071 1.23424 m
.01299 1.23251 L
s
gsave
-0.07007 .76468 -68.9375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.37669 1.62453 L
s
.37669 1.62453 m
.3971 .57309 L
s
.3971 .57309 m
.0843 .2212 L
s
.64785 0 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.64785 0 m
.0843 .2212 L
s
.3971 .57309 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.37669 1.62453 L
s
.37669 1.62453 m
.3971 .57309 L
s
0 0 m
1 0 L
1 1.62453 L
0 1.62453 L
closepath
clip
newpath
.5 Mabswid
.49193 .39279 m .47443 .4748 L .50947 .47842 L p .668 g
F P
0 g
s
.50947 .47842 m .51114 .39482 L .49193 .39279 L p .668 g
F P
0 g
s
.5084 .56965 m .50947 .47842 L .47443 .4748 L p .627 g
F P
0 g
s
.51114 .39482 m .50947 .47842 L .54474 .47604 L p .641 g
F P
0 g
s
.50947 .47842 m .5084 .56965 L .55386 .56665 L p .59 g
F P
0 g
s
.55386 .56665 m .54474 .47604 L .50947 .47842 L p .59 g
F P
0 g
s
.54474 .47604 m .53047 .39349 L .51114 .39482 L p .641 g
F P
0 g
s
.53047 .39349 m .54474 .47604 L .57801 .46779 L p .616 g
F P
0 g
s
.54474 .47604 m .55386 .56665 L .59678 .55627 L p .553 g
F P
0 g
s
.59678 .55627 m .57801 .46779 L .54474 .47604 L p .553 g
F P
0 g
s
.47412 .38752 m .44185 .46537 L .47443 .4748 L p .697 g
F P
0 g
s
.47443 .4748 m .49193 .39279 L .47412 .38752 L p .697 g
F P
0 g
s
.46325 .56509 m .47443 .4748 L .44185 .46537 L p .666 g
F P
0 g
s
.47443 .4748 m .46325 .56509 L .5084 .56965 L p .627 g
F P
0 g
s
.57801 .46779 m .54867 .38886 L .53047 .39349 L p .616 g
F P
0 g
s
.54867 .38886 m .57801 .46779 L .60714 .45415 L p .592 g
F P
0 g
s
.57801 .46779 m .59678 .55627 L .63447 .53909 L p .512 g
F P
0 g
s
.63447 .53909 m .60714 .45415 L .57801 .46779 L p .512 g
F P
0 g
s
.51114 .39482 m .51316 .32978 L .49193 .39279 L closepath p .709 g
F P
0 g
s
.53047 .39349 m .51316 .32978 L .51114 .39482 L closepath p .697 g
F P
0 g
s
.45888 .37934 m .41383 .45071 L .44185 .46537 L p .73 g
F P
0 g
s
.44185 .46537 m .47412 .38752 L .45888 .37934 L p .73 g
F P
0 g
s
.42119 .55324 m .44185 .46537 L .41383 .45071 L p .71 g
F P
0 g
s
.44185 .46537 m .42119 .55324 L .46325 .56509 L p .666 g
F P
0 g
s
.54867 .38886 m .51316 .32978 L .53047 .39349 L closepath p .687 g
F P
0 g
s
.49193 .39279 m .51316 .32978 L .47412 .38752 L closepath p .722 g
F P
0 g
s
.50812 .65752 m .5084 .56965 L .46325 .56509 L p .578 g
F P
0 g
s
.5084 .56965 m .50812 .65752 L .55628 .65443 L p .531 g
F P
0 g
s
.55628 .65443 m .55386 .56665 L .5084 .56965 L p .531 g
F P
0 g
s
.56453 .38125 m .51316 .32978 L .54867 .38886 L closepath p .678 g
F P
0 g
s
.60714 .45415 m .56453 .38125 L .54867 .38886 L p .592 g
F P
0 g
s
.55386 .56665 m .55628 .65443 L .60177 .6437 L p .48 g
F P
0 g
s
.60177 .6437 m .59678 .55627 L .55386 .56665 L p .48 g
F P
0 g
s
.47412 .38752 m .51316 .32978 L .45888 .37934 L closepath p .736 g
F P
0 g
s
.46027 .65281 m .46325 .56509 L .42119 .55324 L p .624 g
F P
0 g
s
.46325 .56509 m .46027 .65281 L .50812 .65752 L p .578 g
F P
0 g
s
.56453 .38125 m .60714 .45415 L .6302 .43595 L p .569 g
F P
0 g
s
.60714 .45415 m .63447 .53909 L .66445 .51611 L p .464 g
F P
0 g
s
.66445 .51611 m .6302 .43595 L .60714 .45415 L p .464 g
F P
0 g
s
.57699 .37113 m .51316 .32978 L .56453 .38125 L closepath p .672 g
F P
0 g
s
.6302 .43595 m .57699 .37113 L .56453 .38125 L p .569 g
F P
0 g
s
.44725 .36877 m .39225 .4317 L .41383 .45071 L p .766 g
F P
0 g
s
.41383 .45071 m .45888 .37934 L .44725 .36877 L p .766 g
F P
0 g
s
.38491 .53476 m .41383 .45071 L .39225 .4317 L p .761 g
F P
0 g
s
.41383 .45071 m .38491 .53476 L .42119 .55324 L p .71 g
F P
0 g
s
.45888 .37934 m .51316 .32978 L .44725 .36877 L closepath p .751 g
F P
0 g
s
.59678 .55627 m .60177 .6437 L .64175 .62593 L p .418 g
F P
0 g
s
.64175 .62593 m .63447 .53909 L .59678 .55627 L p .418 g
F P
0 g
s
.4157 .64056 m .42119 .55324 L .38491 .53476 L p .674 g
F P
0 g
s
.42119 .55324 m .4157 .64056 L .46027 .65281 L p .624 g
F P
0 g
s
.58519 .35916 m .51316 .32978 L .57699 .37113 L closepath p .668 g
F P
0 g
s
.57699 .37113 m .6302 .43595 L .64558 .41432 L p .546 g
F P
0 g
s
.64558 .41432 m .58519 .35916 L .57699 .37113 L p .546 g
F P
0 g
s
.44725 .36877 m .51316 .32978 L .44002 .35651 L closepath p .765 g
F P
0 g
s
.39225 .4317 m .44725 .36877 L .44002 .35651 L p .807 g
F P
0 g
s
.6302 .43595 m .66445 .51611 L .68463 .48871 L p .401 g
F P
0 g
s
.68463 .48871 m .64558 .41432 L .6302 .43595 L p .401 g
F P
0 g
s
.44002 .35651 m .37862 .40953 L .39225 .4317 L p .807 g
F P
0 g
s
.35681 .51074 m .39225 .4317 L .37862 .40953 L p .822 g
F P
0 g
s
.39225 .4317 m .35681 .51074 L .38491 .53476 L p .761 g
F P
0 g
s
.63447 .53909 m .64175 .62593 L .67358 .60215 L p .331 g
F P
0 g
s
.67358 .60215 m .66445 .51611 L .63447 .53909 L p .331 g
F P
0 g
s
.58854 .34613 m .51316 .32978 L .58519 .35916 L closepath p .668 g
F P
0 g
s
.58519 .35916 m .64558 .41432 L .65214 .39064 L p .521 g
F P
0 g
s
.65214 .39064 m .58854 .34613 L .58519 .35916 L p .521 g
F P
0 g
s
.44002 .35651 m .51316 .32978 L .43772 .34337 L closepath p .778 g
F P
0 g
s
.37862 .40953 m .44002 .35651 L .43772 .34337 L p .852 g
F P
0 g
s
.37721 .62145 m .38491 .53476 L .35681 .51074 L p .731 g
F P
0 g
s
.38491 .53476 m .37721 .62145 L .4157 .64056 L p .674 g
F P
0 g
s
.58676 .33291 m .51316 .32978 L .58854 .34613 L closepath p .671 g
F P
0 g
s
.43772 .34337 m .51316 .32978 L .44057 .33022 L closepath p .788 g
F P
0 g
s
.57991 .32041 m .51316 .32978 L .58676 .33291 L closepath p .678 g
F P
0 g
s
.56844 .30952 m .51316 .32978 L .57991 .32041 L closepath p .69 g
F P
0 g
s
.44057 .33022 m .51316 .32978 L .44842 .31799 L closepath p .795 g
F P
0 g
s
.55309 .30099 m .51316 .32978 L .56844 .30952 L closepath p .705 g
F P
0 g
s
.44842 .31799 m .51316 .32978 L .46078 .30753 L closepath p .796 g
F P
0 g
s
.53493 .29546 m .51316 .32978 L .55309 .30099 L closepath p .723 g
F P
0 g
s
.46078 .30753 m .51316 .32978 L .4768 .29958 L closepath p .791 g
F P
0 g
s
.51524 .29331 m .51316 .32978 L .53493 .29546 L closepath p .743 g
F P
0 g
s
.49539 .29472 m .51316 .32978 L .51524 .29331 L closepath p .763 g
F P
0 g
s
.4768 .29958 m .51316 .32978 L .49539 .29472 L closepath p .779 g
F P
0 g
s
.58854 .34613 m .65214 .39064 L .64925 .36649 L p .494 g
F P
0 g
s
.64925 .36649 m .58676 .33291 L .58854 .34613 L p .494 g
F P
0 g
s
.50872 .7317 m .50812 .65752 L .46027 .65281 L p .504 g
F P
0 g
s
.50812 .65752 m .50872 .7317 L .55118 .72903 L p .447 g
F P
0 g
s
.55118 .72903 m .55628 .65443 L .50812 .65752 L p .447 g
F P
0 g
s
.37399 .38562 m .43772 .34337 L .44057 .33022 L p .895 g
F P
0 g
s
.43772 .34337 m .37399 .38562 L .37862 .40953 L p .852 g
F P
0 g
s
.64558 .41432 m .68463 .48871 L .69345 .45861 L p .304 g
F P
0 g
s
.69345 .45861 m .65214 .39064 L .64558 .41432 L p .304 g
F P
0 g
s
.55628 .65443 m .55118 .72903 L .59127 .71978 L p .377 g
F P
0 g
s
.59127 .71978 m .60177 .6437 L .55628 .65443 L p .377 g
F P
0 g
s
.46653 .72764 m .46027 .65281 L .4157 .64056 L p .555 g
F P
0 g
s
.46027 .65281 m .46653 .72764 L .50872 .7317 L p .504 g
F P
0 g
s
.33888 .48262 m .37862 .40953 L .37399 .38562 L p .889 g
F P
0 g
s
.37862 .40953 m .33888 .48262 L .35681 .51074 L p .822 g
F P
0 g
s
.66445 .51611 m .67358 .60215 L .69507 .57377 L p .191 g
F P
0 g
s
.69507 .57377 m .68463 .48871 L .66445 .51611 L p .191 g
F P
0 g
s
.58676 .33291 m .64925 .36649 L .63694 .34353 L p .46 g
F P
0 g
s
.63694 .34353 m .57991 .32041 L .58676 .33291 L p .46 g
F P
0 g
s
.37886 .36157 m .44057 .33022 L .44842 .31799 L p .92 g
F P
0 g
s
.44057 .33022 m .37886 .36157 L .37399 .38562 L p .895 g
F P
0 g
s
.60177 .6437 m .59127 .71978 L .62644 .70447 L p .279 g
F P
0 g
s
.62644 .70447 m .64175 .62593 L .60177 .6437 L p .279 g
F P
0 g
s
.34736 .59658 m .35681 .51074 L .33888 .48262 L p .791 g
F P
0 g
s
.35681 .51074 m .34736 .59658 L .37721 .62145 L p .731 g
F P
0 g
s
.42726 .71707 m .4157 .64056 L .37721 .62145 L p .603 g
F P
0 g
s
.4157 .64056 m .42726 .71707 L .46653 .72764 L p .555 g
F P
0 g
s
.57991 .32041 m .63694 .34353 L .61591 .32339 L p .416 g
F P
0 g
s
.61591 .32339 m .56844 .30952 L .57991 .32041 L p .416 g
F P
0 g
s
.65214 .39064 m .69345 .45861 L .69005 .42778 L p .151 g
F P
0 g
s
.69005 .42778 m .64925 .36649 L .65214 .39064 L p .151 g
F P
0 g
s
.39306 .33906 m .44842 .31799 L .46078 .30753 L p .89 g
F P
0 g
s
.44842 .31799 m .39306 .33906 L .37886 .36157 L p .92 g
F P
0 g
s
.33255 .4522 m .37399 .38562 L .37886 .36157 L p .913 g
F P
0 g
s
.37399 .38562 m .33255 .4522 L .33888 .48262 L p .889 g
F P
0 g
s
.64175 .62593 m .62644 .70447 L .65437 .684 L p .123 g
F P
0 g
s
.65437 .684 m .67358 .60215 L .64175 .62593 L p .123 g
F P
0 g
s
.56844 .30952 m .61591 .32339 L .58753 .30757 L p .373 g
F P
0 g
s
.58753 .30757 m .55309 .30099 L .56844 .30952 L p .373 g
F P
0 g
s
.41575 .31971 m .46078 .30753 L .4768 .29958 L p .775 g
F P
0 g
s
.46078 .30753 m .41575 .31971 L .39306 .33906 L p .89 g
F P
0 g
s
.68463 .48871 m .69507 .57377 L .70452 .54256 L p .047 g
F P
0 g
s
.70452 .54256 m .69345 .45861 L .68463 .48871 L p .047 g
F P
0 g
s
.39341 .70061 m .37721 .62145 L .34736 .59658 L p .642 g
F P
0 g
s
.37721 .62145 m .39341 .70061 L .42726 .71707 L p .603 g
F P
0 g
s
.55309 .30099 m .58753 .30757 L .55376 .29727 L p .367 g
F P
0 g
s
.55376 .29727 m .53493 .29546 L .55309 .30099 L p .367 g
F P
0 g
s
.32826 .56746 m .33888 .48262 L .33255 .4522 L p .798 g
F P
0 g
s
.33888 .48262 m .32826 .56746 L .34736 .59658 L p .791 g
F P
0 g
s
.44543 .30495 m .4768 .29958 L .49539 .29472 L p .606 g
F P
0 g
s
.4768 .29958 m .44543 .30495 L .41575 .31971 L p .775 g
F P
0 g
s
.53493 .29546 m .55376 .29727 L .51703 .29328 L p .385 g
F P
0 g
s
.51703 .29328 m .51524 .29331 L .53493 .29546 L p .385 g
F P
0 g
s
.49539 .29472 m .48002 .2959 L .44543 .30495 L p .606 g
F P
0 g
s
.48002 .2959 m .49539 .29472 L .51524 .29331 L p F P
s
.64925 .36649 m .69005 .42778 L .67434 .39837 L p .056 g
F P
0 g
s
.67434 .39837 m .63694 .34353 L .64925 .36649 L p .056 g
F P
0 g
s
.51524 .29331 m .51703 .29328 L .48002 .2959 L p F P
s
.33853 .42149 m .37886 .36157 L .39306 .33906 L p .725 g
F P
0 g
s
.37886 .36157 m .33853 .42149 L .33255 .4522 L p .913 g
F P
0 g
s
.67358 .60215 m .65437 .684 L .67312 .65962 L p .022 g
F P
0 g
s
.67312 .65962 m .69507 .57377 L .67358 .60215 L p .022 g
F P
0 g
s
.36724 .67921 m .34736 .59658 L .32826 .56746 L p .626 g
F P
0 g
s
.34736 .59658 m .36724 .67921 L .39341 .70061 L p .642 g
F P
0 g
s
.69345 .45861 m .70452 .54256 L .701 .51057 L p .013 g
F P
0 g
s
.701 .51057 m .69005 .42778 L .69345 .45861 L p .013 g
F P
0 g
s
.51018 .78332 m .50872 .7317 L .46653 .72764 L p .367 g
F P
0 g
s
.50872 .7317 m .51018 .78332 L .53862 .78156 L p .297 g
F P
0 g
s
.53862 .78156 m .55118 .72903 L .50872 .7317 L p .297 g
F P
0 g
s
.63694 .34353 m .67434 .39837 L .64717 .37248 L p .384 g
F P
0 g
s
.64717 .37248 m .61591 .32339 L .63694 .34353 L p .384 g
F P
0 g
s
.55118 .72903 m .53862 .78156 L .56542 .77546 L p .194 g
F P
0 g
s
.56542 .77546 m .59127 .71978 L .55118 .72903 L p .194 g
F P
0 g
s
.33255 .4522 m .32144 .53591 L .32826 .56746 L p .798 g
F P
0 g
s
.32144 .53591 m .33255 .4522 L .33853 .42149 L p F P
s
.48194 .78064 m .46653 .72764 L .42726 .71707 L p .414 g
F P
0 g
s
.46653 .72764 m .48194 .78064 L .51018 .78332 L p .367 g
F P
0 g
s
.44609 .88942 m .43185 .9383 L .50542 .94494 L p .698 g
F P
0 g
s
.50542 .94494 m .50677 .89499 L .44609 .88942 L p .698 g
F P
0 g
s
.50492 .98608 m .50542 .94494 L .43185 .9383 L p .63 g
F P
0 g
s
.50677 .89499 m .50542 .94494 L .57944 .94058 L p .677 g
F P
0 g
s
.50542 .94494 m .50492 .98608 L .58374 .98149 L p .593 g
F P
0 g
s
.58374 .98149 m .57944 .94058 L .50542 .94494 L p .593 g
F P
0 g
s
.39306 .33906 m .35672 .39263 L .33853 .42149 L p .725 g
F P
0 g
s
.35672 .39263 m .39306 .33906 L .41575 .31971 L p F P
s
.57944 .94058 m .56784 .89133 L .50677 .89499 L p .677 g
F P
0 g
s
.56784 .89133 m .57944 .94058 L .6496 .92541 L p .659 g
F P
0 g
s
.57944 .94058 m .58374 .98149 L .6585 .96554 L p .556 g
F P
0 g
s
.6585 .96554 m .6496 .92541 L .57944 .94058 L p .556 g
F P
0 g
s
.43185 .9383 m .42657 .97909 L .50492 .98608 L p .63 g
F P
0 g
s
.50534 1.00737 m .50492 .98608 L .42657 .97909 L p .422 g
F P
0 g
s
.50492 .98608 m .50534 1.00737 L .5801 1.00305 L p .359 g
F P
0 g
s
.5801 1.00305 m .58374 .98149 L .50492 .98608 L p .359 g
F P
0 g
s
.38943 .87492 m .36302 .92097 L .43185 .9383 L p .721 g
F P
0 g
s
.43185 .9383 m .44609 .88942 L .38943 .87492 L p .721 g
F P
0 g
s
.42657 .97909 m .43185 .9383 L .36302 .92097 L p .669 g
F P
0 g
s
.59127 .71978 m .56542 .77546 L .58883 .7654 L p .094 g
F P
0 g
s
.58883 .7654 m .62644 .70447 L .59127 .71978 L p .094 g
F P
0 g
s
.4557 .77369 m .42726 .71707 L .39341 .70061 L p .433 g
F P
0 g
s
.42726 .71707 m .4557 .77369 L .48194 .78064 L p .414 g
F P
0 g
s
.58374 .98149 m .5801 1.00305 L .65095 .98803 L p .273 g
F P
0 g
s
.65095 .98803 m .6585 .96554 L .58374 .98149 L p .273 g
F P
0 g
s
.69507 .57377 m .67312 .65962 L .68126 .63287 L p F P
s
.68126 .63287 m .70452 .54256 L .69507 .57377 L p F P
s
.36302 .92097 m .35322 .96086 L .42657 .97909 L p .669 g
F P
0 g
s
.43105 1.00079 m .42657 .97909 L .35322 .96086 L p .47 g
F P
0 g
s
.42657 .97909 m .43105 1.00079 L .50534 1.00737 L p .422 g
F P
0 g
s
.61591 .32339 m .64717 .37248 L .61028 .35209 L p .43 g
F P
0 g
s
.61028 .35209 m .58753 .30757 L .61591 .32339 L p .43 g
F P
0 g
s
.32826 .56746 m .35059 .65421 L .36724 .67921 L p .626 g
F P
0 g
s
.35059 .65421 m .32826 .56746 L .32144 .53591 L p .024 g
F P
0 g
s
.38608 .36774 m .41575 .31971 L .44543 .30495 L p .076 g
F P
0 g
s
.41575 .31971 m .38608 .36774 L .35672 .39263 L p F P
s
.46756 .84414 m .44609 .88942 L .50677 .89499 L p .738 g
F P
0 g
s
.50677 .89499 m .50881 .84798 L .46756 .84414 L p .738 g
F P
0 g
s
.50881 .84798 m .50677 .89499 L .56784 .89133 L p .736 g
F P
0 g
s
.6496 .92541 m .62563 .87863 L .56784 .89133 L p .659 g
F P
0 g
s
.62563 .87863 m .6496 .92541 L .71169 .90019 L p .643 g
F P
0 g
s
.6496 .92541 m .6585 .96554 L .72474 .93899 L p .516 g
F P
0 g
s
.72474 .93899 m .71169 .90019 L .6496 .92541 L p .516 g
F P
0 g
s
.69005 .42778 m .701 .51057 L .68441 .48001 L p .693 g
F P
0 g
s
.68441 .48001 m .67434 .39837 L .69005 .42778 L p .693 g
F P
0 g
s
.62644 .70447 m .58883 .7654 L .60731 .752 L p F P
s
.60731 .752 m .65437 .684 L .62644 .70447 L p F P
s
.56784 .89133 m .55034 .84545 L .50881 .84798 L p .736 g
F P
0 g
s
.55034 .84545 m .56784 .89133 L .62563 .87863 L p .735 g
F P
0 g
s
.34032 .85225 m .30311 .8938 L .36302 .92097 L p .745 g
F P
0 g
s
.36302 .92097 m .38943 .87492 L .34032 .85225 L p .745 g
F P
0 g
s
.35322 .96086 m .36302 .92097 L .30311 .8938 L p .713 g
F P
0 g
s
.42916 .83416 m .38943 .87492 L .44609 .88942 L p .74 g
F P
0 g
s
.44609 .88942 m .46756 .84414 L .42916 .83416 L p .74 g
F P
0 g
s
.43319 .76287 m .39341 .70061 L .36724 .67921 L p .4 g
F P
0 g
s
.39341 .70061 m .43319 .76287 L .4557 .77369 L p .433 g
F P
0 g
s
.3277 .50404 m .33853 .42149 L .35672 .39263 L p .032 g
F P
0 g
s
.33853 .42149 m .3277 .50404 L .32144 .53591 L p F P
s
.6585 .96554 m .65095 .98803 L .71367 .96305 L p .143 g
F P
0 g
s
.71367 .96305 m .72474 .93899 L .6585 .96554 L p .143 g
F P
0 g
s
.58753 .30757 m .61028 .35209 L .56623 .33878 L p .414 g
F P
0 g
s
.56623 .33878 m .55376 .29727 L .58753 .30757 L p .414 g
F P
0 g
s
.42469 .3487 m .44543 .30495 L .48002 .2959 L p .172 g
F P
0 g
s
.44543 .30495 m .42469 .3487 L .38608 .36774 L p .076 g
F P
0 g
s
.30311 .8938 m .28928 .93227 L .35322 .96086 L p .713 g
F P
0 g
s
.36154 .98363 m .35322 .96086 L .28928 .93227 L p .502 g
F P
0 g
s
.35322 .96086 m .36154 .98363 L .43105 1.00079 L p .47 g
F P
0 g
s
.50667 .99971 m .50534 1.00737 L .43105 1.00079 L p .615 g
F P
0 g
s
.50534 1.00737 m .50667 .99971 L .56875 .99611 L p .671 g
F P
0 g
s
.56875 .99611 m .5801 1.00305 L .50534 1.00737 L p .671 g
F P
0 g
s
.62563 .87863 m .58954 .83671 L .55034 .84545 L p .735 g
F P
0 g
s
.58954 .83671 m .62563 .87863 L .67659 .85757 L p .733 g
F P
0 g
s
.71169 .90019 m .67659 .85757 L .62563 .87863 L p .643 g
F P
0 g
s
.55376 .29727 m .56623 .33878 L .51823 .33362 L p .363 g
F P
0 g
s
.51823 .33362 m .51703 .29328 L .55376 .29727 L p .363 g
F P
0 g
s
.5801 1.00305 m .56875 .99611 L .62751 .98362 L p .744 g
F P
0 g
s
.62751 .98362 m .65095 .98803 L .5801 1.00305 L p .744 g
F P
0 g
s
.46984 .33702 m .48002 .2959 L .51703 .29328 L p .287 g
F P
0 g
s
.48002 .2959 m .46984 .33702 L .42469 .3487 L p .172 g
F P
0 g
s
.44497 .99423 m .43105 1.00079 L .36154 .98363 L p .583 g
F P
0 g
s
.43105 1.00079 m .44497 .99423 L .50667 .99971 L p .615 g
F P
0 g
s
.70452 .54256 m .68126 .63287 L .67799 .6055 L p .841 g
F P
0 g
s
.67799 .6055 m .701 .51057 L .70452 .54256 L p .841 g
F P
0 g
s
.39608 .81861 m .34032 .85225 L .38943 .87492 L p .743 g
F P
0 g
s
.38943 .87492 m .42916 .83416 L .39608 .81861 L p .743 g
F P
0 g
s
.51703 .29328 m .51823 .33362 L .46984 .33702 L p .287 g
F P
0 g
s
.65437 .684 m .60731 .752 L .61956 .7361 L p F P
s
.61956 .7361 m .67312 .65962 L .65437 .684 L p F P
s
.34478 .62717 m .32144 .53591 L .3277 .50404 L p .055 g
F P
0 g
s
.32144 .53591 m .34478 .62717 L .35059 .65421 L p .024 g
F P
0 g
s
.36724 .67921 m .41592 .74887 L .43319 .76287 L p .4 g
F P
0 g
s
.41592 .74887 m .36724 .67921 L .35059 .65421 L p .048 g
F P
0 g
s
.67659 .85757 m .71169 .90019 L .76176 .86621 L p .629 g
F P
0 g
s
.71169 .90019 m .72474 .93899 L .77829 .90319 L p .469 g
F P
0 g
s
.77829 .90319 m .76176 .86621 L .71169 .90019 L p .469 g
F P
0 g
s
.67434 .39837 m .68441 .48001 L .6556 .45311 L p .69 g
F P
0 g
s
.6556 .45311 m .64717 .37248 L .67434 .39837 L p .69 g
F P
0 g
s
.34693 .47406 m .35672 .39263 L .38608 .36774 L p .165 g
F P
0 g
s
.35672 .39263 m .34693 .47406 L .3277 .50404 L p .032 g
F P
0 g
s
.30195 .82265 m .256 .85822 L .30311 .8938 L p .772 g
F P
0 g
s
.30311 .8938 m .34032 .85225 L .30195 .82265 L p .772 g
F P
0 g
s
.28928 .93227 m .30311 .8938 L .256 .85822 L p .765 g
F P
0 g
s
.65095 .98803 m .62751 .98362 L .67934 .96289 L p .815 g
F P
0 g
s
.67934 .96289 m .71367 .96305 L .65095 .98803 L p .815 g
F P
0 g
s
.72474 .93899 m .71367 .96305 L .76427 .9294 L p .047 g
F P
0 g
s
.76427 .9294 m .77829 .90319 L .72474 .93899 L p .047 g
F P
0 g
s
.67659 .85757 m .62392 .82225 L .58954 .83671 L p .733 g
F P
0 g
s
.62392 .82225 m .67659 .85757 L .71742 .82929 L p .733 g
F P
0 g
s
.76176 .86621 m .71742 .82929 L .67659 .85757 L p .629 g
F P
0 g
s
.28928 .93227 m .30102 .95673 L .36154 .98363 L p .502 g
F P
0 g
s
.38736 .97996 m .36154 .98363 L .30102 .95673 L p .573 g
F P
0 g
s
.36154 .98363 m .38736 .97996 L .44497 .99423 L p .583 g
F P
0 g
s
.256 .85822 m .23886 .89476 L .28928 .93227 L p .765 g
F P
0 g
s
.30102 .95673 m .28928 .93227 L .23886 .89476 L p .498 g
F P
0 g
s
.37051 .79841 m .30195 .82265 L .34032 .85225 L p .745 g
F P
0 g
s
.34032 .85225 m .39608 .81861 L .37051 .79841 L p .745 g
F P
0 g
s
.49372 .81382 m .46756 .84414 L .50881 .84798 L p .737 g
F P
0 g
s
.50881 .84798 m .51131 .81548 L .49372 .81382 L p .737 g
F P
0 g
s
.51131 .81548 m .50881 .84798 L .55034 .84545 L p .765 g
F P
0 g
s
.67312 .65962 m .61956 .7361 L .62469 .71874 L p .898 g
F P
0 g
s
.62469 .71874 m .68126 .63287 L .67312 .65962 L p .898 g
F P
0 g
s
.55034 .84545 m .52901 .81439 L .51131 .81548 L p .765 g
F P
0 g
s
.52901 .81439 m .55034 .84545 L .58954 .83671 L p .791 g
F P
0 g
s
.47742 .80954 m .42916 .83416 L .46756 .84414 L p .712 g
F P
0 g
s
.46756 .84414 m .49372 .81382 L .47742 .80954 L p .712 g
F P
0 g
s
.701 .51057 m .67799 .6055 L .66328 .57941 L p .824 g
F P
0 g
s
.66328 .57941 m .68441 .48001 L .701 .51057 L p .824 g
F P
0 g
s
.64717 .37248 m .6556 .45311 L .61642 .43189 L p .639 g
F P
0 g
s
.61642 .43189 m .61028 .35209 L .64717 .37248 L p .639 g
F P
0 g
s
.40508 .73258 m .35059 .65421 L .34478 .62717 L p .132 g
F P
0 g
s
.35059 .65421 m .40508 .73258 L .41592 .74887 L p .048 g
F P
0 g
s
.37807 .44817 m .38608 .36774 L .42469 .3487 L p .313 g
F P
0 g
s
.38608 .36774 m .37807 .44817 L .34693 .47406 L p .165 g
F P
0 g
s
.50874 .95737 m .50667 .99971 L .44497 .99423 L p .745 g
F P
0 g
s
.50667 .99971 m .50874 .95737 L .55099 .95488 L p .748 g
F P
0 g
s
.55099 .95488 m .56875 .99611 L .50667 .99971 L p .748 g
F P
0 g
s
.35045 .59991 m .3277 .50404 L .34693 .47406 L p .229 g
F P
0 g
s
.3277 .50404 m .35045 .59991 L .34478 .62717 L p .055 g
F P
0 g
s
.58954 .83671 m .54567 .81063 L .52901 .81439 L p .791 g
F P
0 g
s
.54567 .81063 m .58954 .83671 L .62392 .82225 L p .811 g
F P
0 g
s
.51239 .80615 m .51018 .78332 L .48194 .78064 L p .136 g
F P
0 g
s
.51018 .78332 m .51239 .80615 L .51971 .8057 L p .103 g
F P
0 g
s
.51971 .8057 m .53862 .78156 L .51018 .78332 L p .103 g
F P
0 g
s
.53862 .78156 m .51971 .8057 L .52659 .80415 L p .05 g
F P
0 g
s
.52659 .80415 m .56542 .77546 L .53862 .78156 L p .05 g
F P
0 g
s
.56875 .99611 m .55099 .95488 L .59088 .94628 L p .75 g
F P
0 g
s
.59088 .94628 m .62751 .98362 L .56875 .99611 L p .75 g
F P
0 g
s
.46347 .80289 m .39608 .81861 L .42916 .83416 L p .691 g
F P
0 g
s
.42916 .83416 m .47742 .80954 L .46347 .80289 L p .691 g
F P
0 g
s
.50512 .80547 m .48194 .78064 L .4557 .77369 L p .151 g
F P
0 g
s
.48194 .78064 m .50512 .80547 L .51239 .80615 L p .136 g
F P
0 g
s
.71367 .96305 m .67934 .96289 L .72089 .93505 L p .862 g
F P
0 g
s
.72089 .93505 m .76427 .9294 L .71367 .96305 L p .862 g
F P
0 g
s
.46676 .95359 m .44497 .99423 L .38736 .97996 L p .743 g
F P
0 g
s
.44497 .99423 m .46676 .95359 L .50874 .95737 L p .745 g
F P
0 g
s
.71742 .82929 m .76176 .86621 L .7963 .82533 L p .617 g
F P
0 g
s
.76176 .86621 m .77829 .90319 L .81537 .86005 L p .406 g
F P
0 g
s
.81537 .86005 m .7963 .82533 L .76176 .86621 L p .406 g
F P
0 g
s
.56542 .77546 m .52659 .80415 L .53257 .80159 L p F P
s
.53257 .80159 m .58883 .7654 L .56542 .77546 L p F P
s
.4557 .77369 m .43319 .76287 L p .49759 .80319 L .14 g
F P
0 g
s
.43319 .76287 m .49266 .80095 L .49839 .80369 L p .14 g
F P
0 g
s
.43319 .76287 m .41592 .74887 L p .4895 .7988 L .203 g
F P
0 g
s
.49266 .80095 m .4904 .7995 L p .4895 .7988 L .203 g
F P
0 g
s
.4904 .7995 m .43319 .76287 L p .4895 .7988 L .203 g
F P
0 g
s
.49791 .80336 m .4557 .77369 L p .49759 .80319 L .14 g
F P
0 g
s
.4557 .77369 m .49839 .80369 L .50512 .80547 L p .151 g
F P
0 g
s
.48831 .79741 m .41592 .74887 L .40508 .73258 L p .279 g
F P
0 g
s
.41592 .74887 m .48831 .79741 L .49266 .80095 L p .203 g
F P
0 g
s
.35417 .72359 m .3486 .7493 L p .44251 .76244 L .659 g
F P
0 g
s
.48564 .79332 m .40508 .73258 L .40151 .71506 L p .382 g
F P
0 g
s
.40508 .73258 m .48564 .79332 L .48831 .79741 L p .279 g
F P
0 g
s
.44678 .76303 m .439 .75972 L p .44251 .76244 L .659 g
F P
0 g
s
.439 .75972 m .35417 .72359 L p .44251 .76244 L .659 g
F P
0 g
s
.44415 .77369 m .44678 .76303 L p .40906 .75776 L .659 g
F P
0 g
s
.3486 .7493 m .4361 .77163 L p .40906 .75776 L .659 g
F P
0 g
s
.4361 .77163 m .44415 .77369 L p .40906 .75776 L .659 g
F P
0 g
s
.49839 .80369 m .49791 .80336 L p .49759 .80319 L .14 g
F P
0 g
s
.23886 .89476 m .2534 .92148 L .30102 .95673 L p .498 g
F P
0 g
s
.3374 .95765 m .30102 .95673 L .2534 .92148 L p .575 g
F P
0 g
s
.30102 .95673 m .3374 .95765 L .38736 .97996 L p .573 g
F P
0 g
s
.71742 .82929 m .65121 .80293 L .62392 .82225 L p .733 g
F P
0 g
s
.65121 .80293 m .71742 .82929 L .74528 .79539 L p .732 g
F P
0 g
s
.7963 .82533 m .74528 .79539 L .71742 .82929 L p .617 g
F P
0 g
s
.61028 .35209 m .61642 .43189 L .5696 .41803 L p .586 g
F P
0 g
s
.5696 .41803 m .56623 .33878 L .61028 .35209 L p .586 g
F P
0 g
s
.27709 .78784 m .22507 .81619 L .256 .85822 L p .8 g
F P
0 g
s
.256 .85822 m .30195 .82265 L .27709 .78784 L p .8 g
F P
0 g
s
.23886 .89476 m .256 .85822 L .22507 .81619 L p .826 g
F P
0 g
s
.62392 .82225 m .56019 .80445 L .54567 .81063 L p .811 g
F P
0 g
s
.56019 .80445 m .62392 .82225 L .65121 .80293 L p .823 g
F P
0 g
s
.41909 .42836 m .42469 .3487 L .46984 .33702 L p .407 g
F P
0 g
s
.42469 .3487 m .41909 .42836 L .37807 .44817 L p .313 g
F P
0 g
s
.68126 .63287 m .62469 .71874 L .62223 .70108 L p .907 g
F P
0 g
s
.62223 .70108 m .67799 .6055 L .68126 .63287 L p .907 g
F P
0 g
s
.62751 .98362 m .59088 .94628 L .62587 .93206 L p .751 g
F P
0 g
s
.62587 .93206 m .67934 .96289 L .62751 .98362 L p .751 g
F P
0 g
s
.77829 .90319 m .76427 .9294 L .79919 .8889 L p F P
s
.79919 .8889 m .81537 .86005 L .77829 .90319 L p F P
s
.35426 .77481 m .27709 .78784 L .30195 .82265 L p .746 g
F P
0 g
s
.30195 .82265 m .37051 .79841 L .35426 .77481 L p .746 g
F P
0 g
s
.58883 .7654 m .5347 .80021 L p .54022 .7963 L .709 g
F P
0 g
s
.53723 .7982 m .60731 .752 L .58883 .7654 L p .709 g
F P
0 g
s
.60731 .752 m .53723 .7982 L .54027 .7942 L p .899 g
F P
0 g
s
.54027 .7942 m .61956 .7361 L .60731 .752 L p .899 g
F P
0 g
s
.67423 .72886 m .58835 .76215 L p .58212 .77593 L .804 g
F P
0 g
s
.58835 .76215 m .58046 .76521 L .58082 .7675 L p .804 g
F P
0 g
s
.58082 .7675 m .58212 .77593 L p .58835 .76215 L .804 g
F P
0 g
s
.53257 .80159 m .53723 .7982 L p .54022 .7963 L .709 g
F P
0 g
s
.5347 .80021 m .53257 .80159 L p .54022 .7963 L .709 g
F P
0 g
s
.45283 .79431 m .37051 .79841 L .39608 .81861 L p .676 g
F P
0 g
s
.39608 .81861 m .46347 .80289 L .45283 .79431 L p .676 g
F P
0 g
s
.42769 .94377 m .38736 .97996 L .3374 .95765 L p .741 g
F P
0 g
s
.38736 .97996 m .42769 .94377 L .46676 .95359 L p .743 g
F P
0 g
s
.40151 .71506 m .34478 .62717 L .35045 .59991 L p .286 g
F P
0 g
s
.34478 .62717 m .40151 .71506 L .40508 .73258 L p .132 g
F P
0 g
s
.68441 .48001 m .66328 .57941 L .63791 .55648 L p .758 g
F P
0 g
s
.63791 .55648 m .6556 .45311 L .68441 .48001 L p .758 g
F P
0 g
s
.56623 .33878 m .5696 .41803 L .51855 .41265 L p .533 g
F P
0 g
s
.51855 .41265 m .51823 .33362 L .56623 .33878 L p .533 g
F P
0 g
s
.22507 .81619 m .20561 .8504 L .23886 .89476 L p .826 g
F P
0 g
s
.2534 .92148 m .23886 .89476 L .20561 .8504 L p .402 g
F P
0 g
s
.46709 .41619 m .46984 .33702 L .51823 .33362 L p .476 g
F P
0 g
s
.46984 .33702 m .46709 .41619 L .41909 .42836 L p .407 g
F P
0 g
s
.36747 .57433 m .34693 .47406 L .37807 .44817 L p .359 g
F P
0 g
s
.34693 .47406 m .36747 .57433 L .35045 .59991 L p .229 g
F P
0 g
s
.51823 .33362 m .51855 .41265 L .46709 .41619 L p .476 g
F P
0 g
s
.65121 .80293 m .57159 .79622 L .56019 .80445 L p .823 g
F P
0 g
s
.57159 .79622 m .65121 .80293 L .66952 .77992 L p .824 g
F P
0 g
s
.74528 .79539 m .66952 .77992 L .65121 .80293 L p .732 g
F P
0 g
s
.67934 .96289 m .62587 .93206 L .65366 .91304 L p .752 g
F P
0 g
s
.65366 .91304 m .72089 .93505 L .67934 .96289 L p .752 g
F P
0 g
s
.44623 .78436 m .35426 .77481 L .37051 .79841 L p .666 g
F P
0 g
s
.37051 .79841 m .45283 .79431 L .44623 .78436 L p .666 g
F P
0 g
s
.76427 .9294 m .72089 .93505 L .74926 .90167 L p .878 g
F P
0 g
s
.74926 .90167 m .79919 .8889 L .76427 .9294 L p .878 g
F P
0 g
s
.67799 .6055 m .62223 .70108 L .61225 .68432 L p .842 g
F P
0 g
s
.61225 .68432 m .66328 .57941 L .67799 .6055 L p .842 g
F P
0 g
s
.2534 .92148 m .29837 .92851 L .3374 .95765 L p .575 g
F P
0 g
s
.39401 .92847 m .3374 .95765 L .29837 .92851 L p .739 g
F P
0 g
s
.3374 .95765 m .39401 .92847 L .42769 .94377 L p .741 g
F P
0 g
s
.66952 .77992 m .74528 .79539 L .75794 .75795 L p .732 g
F P
0 g
s
.74528 .79539 m .7963 .82533 L .81248 .77996 L p .609 g
F P
0 g
s
.81248 .77996 m .75794 .75795 L .74528 .79539 L p .609 g
F P
0 g
s
.6556 .45311 m .63791 .55648 L .60354 .53843 L p .697 g
F P
0 g
s
.60354 .53843 m .61642 .43189 L .6556 .45311 L p .697 g
F P
0 g
s
.40555 .69748 m .35045 .59991 L .36747 .57433 L p .395 g
F P
0 g
s
.35045 .59991 m .40555 .69748 L .40151 .71506 L p .286 g
F P
0 g
s
.51129 .87972 m .50874 .95737 L .46676 .95359 L p .717 g
F P
0 g
s
.50874 .95737 m .51129 .87972 L .52918 .87864 L p .702 g
F P
0 g
s
.52918 .87864 m .55099 .95488 L .50874 .95737 L p .702 g
F P
0 g
s
.61956 .7361 m .54027 .7942 L .54147 .78987 L p .932 g
F P
0 g
s
.54147 .78987 m .62469 .71874 L .61956 .7361 L p .932 g
F P
0 g
s
.7963 .82533 m .81537 .86005 L .83293 .81208 L p .312 g
F P
0 g
s
.83293 .81208 m .81248 .77996 L .7963 .82533 L p .312 g
F P
0 g
s
.20561 .8504 m .22212 .87984 L .2534 .92148 L p .402 g
F P
0 g
s
.29837 .92851 m .2534 .92148 L .22212 .87984 L p .581 g
F P
0 g
s
.55099 .95488 m .52918 .87864 L .54601 .87492 L p .688 g
F P
0 g
s
.54601 .87492 m .59088 .94628 L .55099 .95488 L p .688 g
F P
0 g
s
.39487 .55228 m .37807 .44817 L .41909 .42836 L p .441 g
F P
0 g
s
.37807 .44817 m .39487 .55228 L .36747 .57433 L p .359 g
F P
0 g
s
.49352 .87808 m .46676 .95359 L .42769 .94377 L p .734 g
F P
0 g
s
.46676 .95359 m .49352 .87808 L .51129 .87972 L p .717 g
F P
0 g
s
.3486 .7493 m .26779 .74995 L .27709 .78784 L p .748 g
F P
0 g
s
.27709 .78784 m .35426 .77481 L .3486 .7493 L p .748 g
F P
0 g
s
.22507 .81619 m .27709 .78784 L .26779 .74995 L p .828 g
F P
0 g
s
.26779 .74995 m .21297 .77023 L .22507 .81619 L p .828 g
F P
0 g
s
.20561 .8504 m .22507 .81619 L .21297 .77023 L p .893 g
F P
0 g
s
.50512 .80547 m .49372 .81382 L .51131 .81548 L p .683 g
F P
0 g
s
.51131 .81548 m .51239 .80615 L .50512 .80547 L p .683 g
F P
0 g
s
.51239 .80615 m .51131 .81548 L .52901 .81439 L p .73 g
F P
0 g
s
.66952 .77992 m .57907 .7865 L .57159 .79622 L p .824 g
F P
0 g
s
.57907 .7865 m .66952 .77992 L .67745 .75467 L p .817 g
F P
0 g
s
.75794 .75795 m .67745 .75467 L .66952 .77992 L p .732 g
F P
0 g
s
.52901 .81439 m .51971 .8057 L .51239 .80615 L p .73 g
F P
0 g
s
.51971 .8057 m .52901 .81439 L .54567 .81063 L p .783 g
F P
0 g
s
.49839 .80369 m .47742 .80954 L .49372 .81382 L p .649 g
F P
0 g
s
.49372 .81382 m .50512 .80547 L .49839 .80369 L p .649 g
F P
0 g
s
.59088 .94628 m .54601 .87492 L .56068 .86879 L p .677 g
F P
0 g
s
.56068 .86879 m .62587 .93206 L .59088 .94628 L p .677 g
F P
0 g
s
.81537 .86005 m .79919 .8889 L .81557 .84393 L p .814 g
F P
0 g
s
.81557 .84393 m .83293 .81208 L .81537 .86005 L p .814 g
F P
0 g
s
.47705 .87383 m .42769 .94377 L .39401 .92847 L p .752 g
F P
0 g
s
.42769 .94377 m .47705 .87383 L .49352 .87808 L p .734 g
F P
0 g
s
.54567 .81063 m .52659 .80415 L .51971 .8057 L p .783 g
F P
0 g
s
.52659 .80415 m .54567 .81063 L .56019 .80445 L p .829 g
F P
0 g
s
.44415 .77369 m .3486 .7493 L .35426 .77481 L p .66 g
F P
0 g
s
.35426 .77481 m .44623 .78436 L .44415 .77369 L p .66 g
F P
0 g
s
.49266 .80095 m .46347 .80289 L .47742 .80954 L p .627 g
F P
0 g
s
.47742 .80954 m .49839 .80369 L .49266 .80095 L p .627 g
F P
0 g
s
.62469 .71874 m .54147 .78987 L .54074 .78549 L p .89 g
F P
0 g
s
.54074 .78549 m .62223 .70108 L .62469 .71874 L p .89 g
F P
0 g
s
.61642 .43189 m .60354 .53843 L .56253 .52666 L p .645 g
F P
0 g
s
.56253 .52666 m .5696 .41803 L .61642 .43189 L p .645 g
F P
0 g
s
.66328 .57941 m .61225 .68432 L .59533 .66967 L p .777 g
F P
0 g
s
.59533 .66967 m .63791 .55648 L .66328 .57941 L p .777 g
F P
0 g
s
.72089 .93505 m .65366 .91304 L .67232 .89039 L p .752 g
F P
0 g
s
.67232 .89039 m .74926 .90167 L .72089 .93505 L p .752 g
F P
0 g
s
.48484 .78895 m .40151 .71506 L .40555 .69748 L p .453 g
F P
0 g
s
.40151 .71506 m .48484 .78895 L .48564 .79332 L p .382 g
F P
0 g
s
.56019 .80445 m .53257 .80159 L .52659 .80415 L p .829 g
F P
0 g
s
.53257 .80159 m .56019 .80445 L .57159 .79622 L p .856 g
F P
0 g
s
.43084 .53543 m .41909 .42836 L .46709 .41619 L p .501 g
F P
0 g
s
.41909 .42836 m .43084 .53543 L .39487 .55228 L p .441 g
F P
0 g
s
.48831 .79741 m .45283 .79431 L .46347 .80289 L p .616 g
F P
0 g
s
.46347 .80289 m .49266 .80095 L .48831 .79741 L p .616 g
F P
0 g
s
.41704 .68107 m .36747 .57433 L .39487 .55228 L p .465 g
F P
0 g
s
.36747 .57433 m .41704 .68107 L .40555 .69748 L p .395 g
F P
0 g
s
.21297 .77023 m .1924 .80179 L .20561 .8504 L p .893 g
F P
0 g
s
.22212 .87984 m .20561 .8504 L .1924 .80179 L p .06 g
F P
0 g
s
.62587 .93206 m .56068 .86879 L .57219 .86064 L p .668 g
F P
0 g
s
.57219 .86064 m .65366 .91304 L .62587 .93206 L p .668 g
F P
0 g
s
.22212 .87984 m .27304 .89423 L .29837 .92851 L p .581 g
F P
0 g
s
.36798 .90859 m .29837 .92851 L .27304 .89423 L p .737 g
F P
0 g
s
.29837 .92851 m .36798 .90859 L .39401 .92847 L p .739 g
F P
0 g
s
.57159 .79622 m .53723 .7982 L .53257 .80159 L p .856 g
F P
0 g
s
.53723 .7982 m .57159 .79622 L .57907 .7865 L p .862 g
F P
0 g
s
.46296 .86725 m .39401 .92847 L .36798 .90859 L p .77 g
F P
0 g
s
.39401 .92847 m .46296 .86725 L .47705 .87383 L p .752 g
F P
0 g
s
.48564 .79332 m .44623 .78436 L .45283 .79431 L p .612 g
F P
0 g
s
.45283 .79431 m .48831 .79741 L .48564 .79332 L p .612 g
F P
0 g
s
.5696 .41803 m .56253 .52666 L .51787 .52209 L p .598 g
F P
0 g
s
.51787 .52209 m .51855 .41265 L .5696 .41803 L p .598 g
F P
0 g
s
.47286 .5251 m .46709 .41619 L .51855 .41265 L p .551 g
F P
0 g
s
.46709 .41619 m .47286 .5251 L .43084 .53543 L p .501 g
F P
0 g
s
.79919 .8889 m .74926 .90167 L .76219 .86479 L p .869 g
F P
0 g
s
.76219 .86479 m .81557 .84393 L .79919 .8889 L p .869 g
F P
0 g
s
.62223 .70108 m .54074 .78549 L .53813 .78138 L p .834 g
F P
0 g
s
.53813 .78138 m .61225 .68432 L .62223 .70108 L p .834 g
F P
0 g
s
.67745 .75467 m .75794 .75795 L .75404 .71939 L p .733 g
F P
0 g
s
.75794 .75795 m .81248 .77996 L .80845 .73299 L p .604 g
F P
0 g
s
.80845 .73299 m .75404 .71939 L .75794 .75795 L p .604 g
F P
0 g
s
.67745 .75467 m .58212 .77593 L .57907 .7865 L p .817 g
F P
0 g
s
.58212 .77593 m .67745 .75467 L .67423 .72886 L p .804 g
F P
0 g
s
.75404 .71939 m .67423 .72886 L .67745 .75467 L p .733 g
F P
0 g
s
.51855 .41265 m .51787 .52209 L .47286 .5251 L p .551 g
F P
0 g
s
.48596 .78461 m .40555 .69748 L .41704 .68107 L p .501 g
F P
0 g
s
.40555 .69748 m .48596 .78461 L .48484 .78895 L p .453 g
F P
0 g
s
.57907 .7865 m .54027 .7942 L .53723 .7982 L p .862 g
F P
0 g
s
.54027 .7942 m .57907 .7865 L .58212 .77593 L p .851 g
F P
0 g
s
.63791 .55648 m .59533 .66967 L .57258 .65817 L p .723 g
F P
0 g
s
.57258 .65817 m .60354 .53843 L .63791 .55648 L p .723 g
F P
0 g
s
.26779 .74995 m .3486 .7493 L .35417 .72359 L p .749 g
F P
0 g
s
.48484 .78895 m .44415 .77369 L .44623 .78436 L p .613 g
F P
0 g
s
.44623 .78436 m .48564 .79332 L .48484 .78895 L p .613 g
F P
0 g
s
.35417 .72359 m .27521 .71148 L .26779 .74995 L p .749 g
F P
0 g
s
.21297 .77023 m .26779 .74995 L .27521 .71148 L p .852 g
F P
0 g
s
.1924 .80179 m .20986 .83429 L .22212 .87984 L p .06 g
F P
0 g
s
.27304 .89423 m .22212 .87984 L .20986 .83429 L p .59 g
F P
0 g
s
.51239 .84124 m .51129 .87972 L .49352 .87808 L p .704 g
F P
0 g
s
.51129 .87972 m .51239 .84124 L .51975 .84079 L p .684 g
F P
0 g
s
.51975 .84079 m .52918 .87864 L .51129 .87972 L p .684 g
F P
0 g
s
.43528 .66699 m .39487 .55228 L .43084 .53543 L p .516 g
F P
0 g
s
.39487 .55228 m .43528 .66699 L .41704 .68107 L p .465 g
F P
0 g
s
.52918 .87864 m .51975 .84079 L .52667 .83924 L p .666 g
F P
0 g
s
.52667 .83924 m .54601 .87492 L .52918 .87864 L p .666 g
F P
0 g
s
.81248 .77996 m .83293 .81208 L .82893 .76234 L p .161 g
F P
0 g
s
.82893 .76234 m .80845 .73299 L .81248 .77996 L p .161 g
F P
0 g
s
.50508 .84056 m .49352 .87808 L .47705 .87383 L p .726 g
F P
0 g
s
.49352 .87808 m .50508 .84056 L .51239 .84124 L p .704 g
F P
0 g
s
.65366 .91304 m .57219 .86064 L .57976 .85101 L p .662 g
F P
0 g
s
.57976 .85101 m .67232 .89039 L .65366 .91304 L p .662 g
F P
0 g
s
.54601 .87492 m .52667 .83924 L .53268 .8367 L p .65 g
F P
0 g
s
.53268 .8367 m .56068 .86879 L .54601 .87492 L p .65 g
F P
0 g
s
.58212 .77593 m .54147 .78987 L .54027 .7942 L p .851 g
F P
0 g
s
.54147 .78987 m .58212 .77593 L .58046 .76521 L p .829 g
F P
0 g
s
.50508 .84056 m .48102 .96179 L .5101 .9644 L p .664 g
F P
0 g
s
.5101 .9644 m .51239 .84124 L .50508 .84056 L p .664 g
F P
0 g
s
.50846 1.09102 m .5101 .9644 L .48102 .96179 L p .635 g
F P
0 g
s
.51239 .84124 m .5101 .9644 L .53936 .96268 L p .632 g
F P
0 g
s
.5101 .9644 m .50846 1.09102 L .55341 1.08849 L p .597 g
F P
0 g
s
.55341 1.08849 m .53936 .96268 L .5101 .9644 L p .597 g
F P
0 g
s
.49831 .83879 m .47705 .87383 L .46296 .86725 L p .751 g
F P
0 g
s
.47705 .87383 m .49831 .83879 L .50508 .84056 L p .726 g
F P
0 g
s
.61225 .68432 m .53813 .78138 L .5338 .7778 L p .782 g
F P
0 g
s
.5338 .7778 m .59533 .66967 L .61225 .68432 L p .782 g
F P
0 g
s
.27304 .89423 m .35142 .88536 L .36798 .90859 L p .737 g
F P
0 g
s
.45221 .85874 m .36798 .90859 L .35142 .88536 L p .789 g
F P
0 g
s
.36798 .90859 m .45221 .85874 L .46296 .86725 L p .77 g
F P
0 g
s
.48596 .78461 m .44678 .76303 L .44415 .77369 L p .618 g
F P
0 g
s
.44415 .77369 m .48484 .78895 L .48596 .78461 L p .618 g
F P
0 g
s
.53936 .96268 m .51975 .84079 L .51239 .84124 L p .632 g
F P
0 g
s
.51975 .84079 m .53936 .96268 L .56695 .95674 L p .602 g
F P
0 g
s
.53936 .96268 m .55341 1.08849 L .59586 1.07972 L p .559 g
F P
0 g
s
.59586 1.07972 m .56695 .95674 L .53936 .96268 L p .559 g
F P
0 g
s
.49831 .83879 m .45402 .95501 L .48102 .96179 L p .698 g
F P
0 g
s
.48102 .96179 m .50508 .84056 L .49831 .83879 L p .698 g
F P
0 g
s
.4638 1.08717 m .48102 .96179 L .45402 .95501 L p .676 g
F P
0 g
s
.48102 .96179 m .4638 1.08717 L .50846 1.09102 L p .635 g
F P
0 g
s
.48895 .78058 m .41704 .68107 L .43528 .66699 L p .539 g
F P
0 g
s
.41704 .68107 m .48895 .78058 L .48596 .78461 L p .501 g
F P
0 g
s
.74926 .90167 m .67232 .89039 L .68041 .86554 L p .752 g
F P
0 g
s
.68041 .86554 m .76219 .86479 L .74926 .90167 L p .752 g
F P
0 g
s
.56068 .86879 m .53268 .8367 L .53737 .83332 L p .637 g
F P
0 g
s
.53737 .83332 m .57219 .86064 L .56068 .86879 L p .637 g
F P
0 g
s
.27521 .71148 m .22131 .72333 L .21297 .77023 L p .852 g
F P
0 g
s
.1924 .80179 m .21297 .77023 L .22131 .72333 L p .916 g
F P
0 g
s
.49254 .83606 m .46296 .86725 L .45221 .85874 L p .777 g
F P
0 g
s
.46296 .86725 m .49254 .83606 L .49831 .83879 L p .751 g
F P
0 g
s
.60354 .53843 m .57258 .65817 L .54558 .6507 L p .676 g
F P
0 g
s
.54558 .6507 m .56253 .52666 L .60354 .53843 L p .676 g
F P
0 g
s
.58046 .76521 m .54074 .78549 L .54147 .78987 L p .829 g
F P
0 g
s
.56695 .95674 m .52667 .83924 L .51975 .84079 L p .602 g
F P
0 g
s
.52667 .83924 m .56695 .95674 L .59106 .94694 L p .571 g
F P
0 g
s
.56695 .95674 m .59586 1.07972 L .63312 1.06521 L p .518 g
F P
0 g
s
.63312 1.06521 m .59106 .94694 L .56695 .95674 L p .518 g
F P
0 g
s
.49254 .83606 m .43084 .94448 L .45402 .95501 L p .737 g
F P
0 g
s
.45402 .95501 m .49831 .83879 L .49254 .83606 L p .737 g
F P
0 g
s
.42221 1.07716 m .45402 .95501 L .43084 .94448 L p .721 g
F P
0 g
s
.45402 .95501 m .42221 1.07716 L .4638 1.08717 L p .676 g
F P
0 g
s
.54074 .78549 m .58046 .76521 L .57419 .75508 L p .803 g
F P
0 g
s
.58046 .76521 m .67423 .72886 L .65982 .70426 L p .788 g
F P
0 g
s
.65982 .70426 m .57419 .75508 L .58046 .76521 L p .788 g
F P
0 g
s
.45905 .65627 m .43084 .53543 L .47286 .5251 L p .558 g
F P
0 g
s
.43084 .53543 m .45905 .65627 L .43528 .66699 L p .516 g
F P
0 g
s
.67423 .72886 m .75404 .71939 L .73328 .68236 L p .735 g
F P
0 g
s
.73328 .68236 m .65982 .70426 L .67423 .72886 L p .735 g
F P
0 g
s
.48895 .78058 m .45398 .75312 L .44678 .76303 L p .626 g
F P
0 g
s
.44678 .76303 m .48596 .78461 L .48895 .78058 L p .626 g
F P
0 g
s
.35417 .72359 m .44678 .76303 L .45398 .75312 L p .662 g
F P
0 g
s
.57419 .75508 m .53813 .78138 L .54074 .78549 L p .803 g
F P
0 g
s
.59106 .94694 m .53268 .8367 L .52667 .83924 L p .571 g
F P
0 g
s
.20986 .83429 m .26353 .85691 L .27304 .89423 L p .59 g
F P
0 g
s
.35142 .88536 m .27304 .89423 L .26353 .85691 L p .736 g
F P
0 g
s
.83293 .81208 m .81557 .84393 L .81155 .79737 L p .908 g
F P
0 g
s
.81155 .79737 m .82893 .76234 L .83293 .81208 L p .908 g
F P
0 g
s
.50766 1.20886 m .50846 1.09102 L .4638 1.08717 L p .6 g
F P
0 g
s
.50846 1.09102 m .50766 1.20886 L .5602 1.20601 L p .556 g
F P
0 g
s
.5602 1.20601 m .55341 1.08849 L .50846 1.09102 L p .556 g
F P
0 g
s
.57219 .86064 m .53737 .83332 L .54043 .82935 L p .626 g
F P
0 g
s
.54043 .82935 m .57976 .85101 L .57219 .86064 L p .626 g
F P
0 g
s
.59533 .66967 m .5338 .7778 L .52805 .77501 L p .738 g
F P
0 g
s
.52805 .77501 m .57258 .65817 L .59533 .66967 L p .738 g
F P
0 g
s
.53268 .8367 m .59106 .94694 L .61009 .93389 L p .539 g
F P
0 g
s
.61009 .93389 m .53737 .83332 L .53268 .8367 L p .539 g
F P
0 g
s
.35142 .88536 m .44554 .84888 L .45221 .85874 L p .789 g
F P
0 g
s
.48817 .83254 m .45221 .85874 L .44554 .84888 L p .804 g
F P
0 g
s
.45221 .85874 m .48817 .83254 L .49254 .83606 L p .777 g
F P
0 g
s
.43084 .94448 m .49254 .83606 L .48817 .83254 L p .782 g
F P
0 g
s
.45398 .75312 m .37083 .69947 L .35417 .72359 L p .662 g
F P
0 g
s
.27521 .71148 m .35417 .72359 L .37083 .69947 L p .749 g
F P
0 g
s
.55341 1.08849 m .5602 1.20601 L .60987 1.19615 L p .509 g
F P
0 g
s
.60987 1.19615 m .59586 1.07972 L .55341 1.08849 L p .509 g
F P
0 g
s
.53813 .78138 m .57419 .75508 L .56368 .74625 L p .776 g
F P
0 g
s
.56368 .74625 m .5338 .7778 L .53813 .78138 L p .776 g
F P
0 g
s
.43528 .66699 m .49361 .77715 L .48895 .78058 L p .539 g
F P
0 g
s
.45398 .75312 m .48895 .78058 L .49361 .77715 L p .637 g
F P
0 g
s
.49361 .77715 m .43528 .66699 L .45905 .65627 L p .571 g
F P
0 g
s
.45546 1.20453 m .4638 1.08717 L .42221 1.07716 L p .645 g
F P
0 g
s
.4638 1.08717 m .45546 1.20453 L .50766 1.20886 L p .6 g
F P
0 g
s
.56253 .52666 m .54558 .6507 L .51625 .6478 L p .635 g
F P
0 g
s
.51625 .6478 m .51787 .52209 L .56253 .52666 L p .635 g
F P
0 g
s
.59106 .94694 m .63312 1.06521 L .66276 1.04581 L p .47 g
F P
0 g
s
.66276 1.04581 m .61009 .93389 L .59106 .94694 L p .47 g
F P
0 g
s
.53737 .83332 m .61009 .93389 L .62272 .9184 L p .502 g
F P
0 g
s
.62272 .9184 m .54043 .82935 L .53737 .83332 L p .502 g
F P
0 g
s
.44554 .84888 m .48548 .82847 L .48817 .83254 L p .804 g
F P
0 g
s
.41304 .93084 m .48817 .83254 L .48548 .82847 L p .834 g
F P
0 g
s
.48817 .83254 m .41304 .93084 L .43084 .94448 L p .782 g
F P
0 g
s
.4867 .64971 m .47286 .5251 L .51787 .52209 L p .597 g
F P
0 g
s
.47286 .5251 m .4867 .64971 L .45905 .65627 L p .558 g
F P
0 g
s
.67232 .89039 m .57976 .85101 L .58284 .84053 L p .659 g
F P
0 g
s
.58284 .84053 m .68041 .86554 L .67232 .89039 L p .659 g
F P
0 g
s
.5338 .7778 m .56368 .74625 L .54965 .73935 L p .75 g
F P
0 g
s
.54965 .73935 m .52805 .77501 L .5338 .7778 L p .75 g
F P
0 g
s
.57419 .75508 m .65982 .70426 L .63503 .68265 L p .769 g
F P
0 g
s
.63503 .68265 m .56368 .74625 L .57419 .75508 L p .769 g
F P
0 g
s
.45905 .65627 m .49961 .77455 L .49361 .77715 L p .571 g
F P
0 g
s
.46528 .74464 m .49361 .77715 L .49961 .77455 L p .65 g
F P
0 g
s
.49361 .77715 m .46528 .74464 L .45398 .75312 L p .637 g
F P
0 g
s
.38634 1.06155 m .43084 .94448 L .41304 .93084 L p .774 g
F P
0 g
s
.43084 .94448 m .38634 1.06155 L .42221 1.07716 L p .721 g
F P
0 g
s
.37083 .69947 m .45398 .75312 L .46528 .74464 L p .668 g
F P
0 g
s
.51787 .52209 m .51625 .6478 L .4867 .64971 L p .597 g
F P
0 g
s
.57976 .85101 m .54043 .82935 L .54163 .82504 L p .617 g
F P
0 g
s
.54043 .82935 m .62272 .9184 L .62801 .90147 L p .454 g
F P
0 g
s
.62801 .90147 m .54163 .82504 L .54043 .82935 L p .454 g
F P
0 g
s
.57258 .65817 m .52805 .77501 L .52127 .7732 L p .699 g
F P
0 g
s
.52127 .7732 m .54558 .6507 L .57258 .65817 L p .699 g
F P
0 g
s
.52805 .77501 m .54965 .73935 L .53306 .73487 L p .726 g
F P
0 g
s
.53306 .73487 m .52127 .7732 L .52805 .77501 L p .726 g
F P
0 g
s
.22131 .72333 m .20101 .7521 L .1924 .80179 L p .916 g
F P
0 g
s
.20986 .83429 m .1924 .80179 L .20101 .7521 L p .194 g
F P
0 g
s
.75404 .71939 m .80845 .73299 L .78367 .68767 L p .603 g
F P
0 g
s
.78367 .68767 m .73328 .68236 L .75404 .71939 L p .603 g
F P
0 g
s
.4867 .64971 m .50654 .77296 L .49961 .77455 L p .602 g
F P
0 g
s
.49961 .77455 m .45905 .65627 L .4867 .64971 L p .602 g
F P
0 g
s
.47993 .73821 m .49961 .77455 L .50654 .77296 L p .666 g
F P
0 g
s
.49961 .77455 m .47993 .73821 L .46528 .74464 L p .65 g
F P
0 g
s
.44343 .83831 m .48467 .82413 L .48548 .82847 L p .83 g
F P
0 g
s
.48548 .82847 m .44554 .84888 L .44343 .83831 L p .83 g
F P
0 g
s
.40187 .91496 m .48548 .82847 L .48467 .82413 L p .89 g
F P
0 g
s
.48548 .82847 m .40187 .91496 L .41304 .93084 L p .834 g
F P
0 g
s
.54163 .82504 m .58284 .84053 L .57976 .85101 L p .617 g
F P
0 g
s
.54558 .6507 m .52127 .7732 L .51393 .7725 L p .664 g
F P
0 g
s
.51393 .7725 m .51625 .6478 L .54558 .6507 L p .664 g
F P
0 g
s
.52127 .7732 m .53306 .73487 L .51506 .73314 L p .704 g
F P
0 g
s
.51506 .73314 m .51393 .7725 L .52127 .7732 L p .704 g
F P
0 g
s
.51625 .6478 m .51393 .7725 L .50654 .77296 L p .632 g
F P
0 g
s
.50654 .77296 m .4867 .64971 L .51625 .6478 L p .632 g
F P
0 g
s
.49692 .73428 m .50654 .77296 L .51393 .7725 L p .684 g
F P
0 g
s
.50654 .77296 m .49692 .73428 L .47993 .73821 L p .666 g
F P
0 g
s
.26353 .85691 m .34564 .86025 L .35142 .88536 L p .736 g
F P
0 g
s
.46478 .80952 m .4935 .81237 L .48882 .81579 L p .857 g
F P
0 g
s
.48882 .81579 m .4935 .81237 L p .49034 .81446 L .714 g
F P
0 g
s
.35142 .88536 m .35022 .88015 L p .44125 .84937 L .806 g
F P
0 g
s
.35022 .88015 m .34564 .86025 L p .40747 .85321 L .806 g
F P
0 g
s
.44125 .84937 m .35022 .88015 L .40747 .85321 L .806 g
F
.4935 .81237 m .49953 .80979 L p .49874 .81028 L 0 g
F P
s
.43762 .85195 m .35142 .88536 L p .44125 .84937 L .806 g
F P
0 g
s
.44343 .83831 m .44509 .84664 L p .41175 .8512 L .806 g
F P
0 g
s
.44125 .84937 m .34564 .86025 L .44509 .84664 L .806 g
F
.44609 .82775 m .48581 .8198 L .48467 .82413 L p .851 g
F P
0 g
s
.48467 .82413 m .44343 .83831 L .44609 .82775 L p .851 g
F P
0 g
s
.45336 .81792 m .48882 .81579 L .48581 .8198 L p .863 g
F P
0 g
s
.48581 .8198 m .44609 .82775 L .45336 .81792 L p .863 g
F P
0 g
s
.48882 .81579 m .45336 .81792 L .46478 .80952 L p .857 g
F P
0 g
s
.43292 .851 m .41414 .86473 L p .49034 .81446 L .714 g
F P
0 g
s
.41414 .86473 m .43919 .84832 L p .43872 .84852 L .714 g
F P
0 g
s
.43919 .84832 m .4843 .81875 L p .43872 .84852 L .714 g
F P
0 g
s
.43872 .84852 m .4843 .81875 L .49034 .81446 L .714 g
F
.44509 .84664 m .44554 .84888 L p .44125 .84937 L .806 g
F P
0 g
s
.44554 .84888 m .43762 .85195 L p .44125 .84937 L .806 g
F P
0 g
s
.40232 .88074 m .48581 .8198 L .48882 .81579 L p .901 g
F P
0 g
s
.48882 .81579 m .41414 .86473 L .40232 .88074 L p .901 g
F P
0 g
s
.4574 .84053 m .43292 .851 L p .49874 .81028 L F P
s
.49953 .80979 m .49908 .81012 L p .49874 .81028 L F P
s
.49908 .81012 m .4574 .84053 L p .49874 .81028 L F P
s
.34564 .86025 m .43524 .84015 L p .41175 .8512 L .806 g
F P
0 g
s
.43524 .84015 m .44343 .83831 L p .41175 .8512 L .806 g
F P
0 g
s
.4935 .81237 m .49129 .81378 L p .49034 .81446 L .714 g
F P
0 g
s
.49129 .81378 m .43292 .851 L p .49034 .81446 L .714 g
F P
0 g
s
.4843 .81875 m .48882 .81579 L p .49034 .81446 L .714 g
F P
0 g
s
.43292 .851 m .49129 .81378 L p .49874 .81028 L F P
s
.49129 .81378 m .4935 .81237 L p .49874 .81028 L F P
s
.51393 .7725 m .51506 .73314 L .49692 .73428 L p .684 g
F P
0 g
s
.59586 1.07972 m .60987 1.19615 L .65355 1.17982 L p .454 g
F P
0 g
s
.65355 1.17982 m .63312 1.06521 L .59586 1.07972 L p .454 g
F P
0 g
s
.58284 .84053 m .54163 .82504 L .5409 .82068 L p .612 g
F P
0 g
s
.54163 .82504 m .62801 .90147 L .6255 .88425 L p .384 g
F P
0 g
s
.6255 .88425 m .5409 .82068 L .54163 .82504 L p .384 g
F P
0 g
s
.39817 .89788 m .48467 .82413 L .48581 .8198 L p .932 g
F P
0 g
s
.48467 .82413 m .39817 .89788 L .40187 .91496 L p .89 g
F P
0 g
s
.40679 1.19327 m .42221 1.07716 L .38634 1.06155 L p .694 g
F P
0 g
s
.42221 1.07716 m .40679 1.19327 L .45546 1.20453 L p .645 g
F P
0 g
s
.56368 .74625 m .63503 .68265 L .60143 .66564 L p .751 g
F P
0 g
s
.60143 .66564 m .54965 .73935 L .56368 .74625 L p .751 g
F P
0 g
s
.81557 .84393 m .76219 .86479 L .75826 .82679 L p .844 g
F P
0 g
s
.75826 .82679 m .81155 .79737 L .81557 .84393 L p .844 g
F P
0 g
s
.39761 .67869 m .46528 .74464 L .47993 .73821 L p .677 g
F P
0 g
s
.46528 .74464 m .39761 .67869 L .37083 .69947 L p .668 g
F P
0 g
s
.37083 .69947 m .29939 .67513 L .27521 .71148 L p .749 g
F P
0 g
s
.22131 .72333 m .27521 .71148 L .29939 .67513 L p .866 g
F P
0 g
s
.5409 .82068 m .58117 .8299 L .58284 .84053 L p .612 g
F P
0 g
s
.58117 .8299 m .5409 .82068 L .53827 .81658 L p .611 g
F P
0 g
s
.5409 .82068 m .6255 .88425 L .61523 .86791 L p .281 g
F P
0 g
s
.61523 .86791 m .53827 .81658 L .5409 .82068 L p .281 g
F P
0 g
s
.61009 .93389 m .66276 1.04581 L .6827 1.02267 L p .404 g
F P
0 g
s
.6827 1.02267 m .62272 .9184 L .61009 .93389 L p .404 g
F P
0 g
s
.48581 .8198 m .40232 .88074 L .39817 .89788 L p .932 g
F P
0 g
s
.68041 .86554 m .58284 .84053 L .58117 .8299 L p .66 g
F P
0 g
s
.65982 .70426 m .73328 .68236 L .69664 .64962 L p .736 g
F P
0 g
s
.69664 .64962 m .63503 .68265 L .65982 .70426 L p .736 g
F P
0 g
s
.35856 1.04127 m .41304 .93084 L .40187 .91496 L p .837 g
F P
0 g
s
.41304 .93084 m .35856 1.04127 L .38634 1.06155 L p .774 g
F P
0 g
s
.53827 .81658 m .57483 .81987 L .58117 .8299 L p .611 g
F P
0 g
s
.57483 .81987 m .53827 .81658 L .53392 .81302 L p .615 g
F P
0 g
s
.53827 .81658 m .61523 .86791 L .59781 .85361 L p .206 g
F P
0 g
s
.59781 .85361 m .53392 .81302 L .53827 .81658 L p .206 g
F P
0 g
s
.54965 .73935 m .60143 .66564 L .56138 .65454 L p .733 g
F P
0 g
s
.56138 .65454 m .53306 .73487 L .54965 .73935 L p .733 g
F P
0 g
s
.35129 .83493 m .44609 .82775 L .44343 .83831 L p .819 g
F P
0 g
s
.44343 .83831 m .34564 .86025 L .35129 .83493 L p .819 g
F P
0 g
s
.43276 .66281 m .47993 .73821 L .49692 .73428 L p .688 g
F P
0 g
s
.47993 .73821 m .43276 .66281 L .39761 .67869 L p .677 g
F P
0 g
s
.53392 .81302 m .56422 .81112 L .57483 .81987 L p .615 g
F P
0 g
s
.56422 .81112 m .53392 .81302 L .52813 .81025 L p .626 g
F P
0 g
s
.53392 .81302 m .59781 .85361 L .57439 .84239 L p .138 g
F P
0 g
s
.57439 .84239 m .52813 .81025 L .53392 .81302 L p .138 g
F P
0 g
s
.29939 .67513 m .37083 .69947 L .39761 .67869 L p .749 g
F P
0 g
s
.47958 .80314 m .49953 .80979 L .4935 .81237 L p .829 g
F P
0 g
s
.4935 .81237 m .46478 .80952 L .47958 .80314 L p .829 g
F P
0 g
s
.53306 .73487 m .56138 .65454 L .51776 .65024 L p .716 g
F P
0 g
s
.51776 .65024 m .51506 .73314 L .53306 .73487 L p .716 g
F P
0 g
s
.20101 .7521 m .21824 .7878 L .20986 .83429 L p .194 g
F P
0 g
s
.26353 .85691 m .20986 .83429 L .21824 .7878 L p .6 g
F P
0 g
s
.4738 .65307 m .49692 .73428 L .51506 .73314 L p .701 g
F P
0 g
s
.49692 .73428 m .4738 .65307 L .43276 .66281 L p .688 g
F P
0 g
s
.63312 1.06521 m .65355 1.17982 L .68842 1.15793 L p .382 g
F P
0 g
s
.68842 1.15793 m .66276 1.04581 L .63312 1.06521 L p .382 g
F P
0 g
s
.51506 .73314 m .51776 .65024 L .4738 .65307 L p .701 g
F P
0 g
s
.52813 .81025 m .55004 .80427 L .56422 .81112 L p .626 g
F P
0 g
s
.55004 .80427 m .52813 .81025 L .52132 .80845 L p .646 g
F P
0 g
s
.52813 .81025 m .57439 .84239 L .54657 .8351 L p .149 g
F P
0 g
s
.54657 .8351 m .52132 .80845 L .52813 .81025 L p .149 g
F P
0 g
s
.49675 .79924 m .5065 .80821 L .49953 .80979 L p .782 g
F P
0 g
s
.49953 .80979 m .47958 .80314 L .49675 .79924 L p .782 g
F P
0 g
s
.4574 .84053 m .49953 .80979 L .5065 .80821 L p .048 g
F P
0 g
s
.80845 .73299 m .82893 .76234 L .80269 .71425 L p .057 g
F P
0 g
s
.80269 .71425 m .78367 .68767 L .80845 .73299 L p .057 g
F P
0 g
s
.76219 .86479 m .68041 .86554 L .67715 .84011 L p .751 g
F P
0 g
s
.67715 .84011 m .75826 .82679 L .76219 .86479 L p .751 g
F P
0 g
s
.52132 .80845 m .53327 .79983 L .55004 .80427 L p .646 g
F P
0 g
s
.53327 .79983 m .52132 .80845 L .51394 .80775 L p .681 g
F P
0 g
s
.52132 .80845 m .54657 .8351 L .51635 .83227 L p .135 g
F P
0 g
s
.51635 .83227 m .51394 .80775 L .52132 .80845 L p .135 g
F P
0 g
s
.51508 .79811 m .51394 .80775 L .5065 .80821 L p .728 g
F P
0 g
s
.5065 .80821 m .49675 .79924 L .51508 .79811 L p .728 g
F P
0 g
s
.48589 .83413 m .5065 .80821 L .51394 .80775 L p .102 g
F P
0 g
s
.5065 .80821 m .48589 .83413 L .4574 .84053 L p .048 g
F P
0 g
s
.51394 .80775 m .51508 .79811 L .53327 .79983 L p .681 g
F P
0 g
s
.51394 .80775 m .51635 .83227 L .48589 .83413 L p .102 g
F P
0 g
s
.58117 .8299 m .67715 .84011 L .68041 .86554 L p .66 g
F P
0 g
s
.67715 .84011 m .58117 .8299 L .57483 .81987 L p .665 g
F P
0 g
s
.36471 1.17569 m .38634 1.06155 L .35856 1.04127 L p .752 g
F P
0 g
s
.38634 1.06155 m .36471 1.17569 L .40679 1.19327 L p .694 g
F P
0 g
s
.36822 .81116 m .45336 .81792 L .44609 .82775 L p .826 g
F P
0 g
s
.44609 .82775 m .35129 .83493 L .36822 .81116 L p .826 g
F P
0 g
s
.62272 .9184 m .6827 1.02267 L .69141 .99726 L p .302 g
F P
0 g
s
.69141 .99726 m .62801 .90147 L .62272 .9184 L p .302 g
F P
0 g
s
.50782 1.30563 m .50766 1.20886 L .45546 1.20453 L p .547 g
F P
0 g
s
.50766 1.20886 m .50782 1.30563 L .55888 1.30295 L p .496 g
F P
0 g
s
.55888 1.30295 m .5602 1.20601 L .50766 1.20886 L p .496 g
F P
0 g
s
.27103 .819 m .35129 .83493 L .34564 .86025 L p .735 g
F P
0 g
s
.34564 .86025 m .26353 .85691 L .27103 .819 L p .735 g
F P
0 g
s
.21824 .7878 m .27103 .819 L .26353 .85691 L p .6 g
F P
0 g
s
.34085 1.01753 m .40187 .91496 L .39817 .89788 L p .903 g
F P
0 g
s
.40187 .91496 m .34085 1.01753 L .35856 1.04127 L p .837 g
F P
0 g
s
.29939 .67513 m .25033 .67879 L .22131 .72333 L p .866 g
F P
0 g
s
.20101 .7521 m .22131 .72333 L .25033 .67879 L p .73 g
F P
0 g
s
.5602 1.20601 m .55888 1.30295 L .60713 1.29369 L p .437 g
F P
0 g
s
.60713 1.29369 m .60987 1.19615 L .5602 1.20601 L p .437 g
F P
0 g
s
.45709 1.30156 m .45546 1.20453 L .40679 1.19327 L p .595 g
F P
0 g
s
.45546 1.20453 m .45709 1.30156 L .50782 1.30563 L p .547 g
F P
0 g
s
.57483 .81987 m .6625 .81588 L .67715 .84011 L p .665 g
F P
0 g
s
.6625 .81588 m .57483 .81987 L .56422 .81112 L p .675 g
F P
0 g
s
.63503 .68265 m .69664 .64962 L .6464 .6237 L p .738 g
F P
0 g
s
.6464 .6237 m .60143 .66564 L .63503 .68265 L p .738 g
F P
0 g
s
.39548 .79069 m .46478 .80952 L .45336 .81792 L p .824 g
F P
0 g
s
.45336 .81792 m .36822 .81116 L .39548 .79069 L p .824 g
F P
0 g
s
.73328 .68236 m .78367 .68767 L .73913 .6474 L p .607 g
F P
0 g
s
.73913 .6474 m .69664 .64962 L .73328 .68236 L p .607 g
F P
0 g
s
.33909 .6436 m .39761 .67869 L .43276 .66281 L p .748 g
F P
0 g
s
.39761 .67869 m .33909 .6436 L .29939 .67513 L p .749 g
F P
0 g
s
.82893 .76234 m .81155 .79737 L .78654 .75244 L p .852 g
F P
0 g
s
.78654 .75244 m .80269 .71425 L .82893 .76234 L p .852 g
F P
0 g
s
.66276 1.04581 m .68842 1.15793 L .71204 1.13177 L p .271 g
F P
0 g
s
.71204 1.13177 m .6827 1.02267 L .66276 1.04581 L p .271 g
F P
0 g
s
.56422 .81112 m .63727 .79459 L .6625 .81588 L p .675 g
F P
0 g
s
.63727 .79459 m .56422 .81112 L .55004 .80427 L p .69 g
F P
0 g
s
.60987 1.19615 m .60713 1.29369 L .64955 1.27835 L p .361 g
F P
0 g
s
.64955 1.27835 m .65355 1.17982 L .60987 1.19615 L p .361 g
F P
0 g
s
.43127 .77504 m .47958 .80314 L .46478 .80952 L p .812 g
F P
0 g
s
.46478 .80952 m .39548 .79069 L .43127 .77504 L p .812 g
F P
0 g
s
.62801 .90147 m .69141 .99726 L .68802 .97124 L p .152 g
F P
0 g
s
.68802 .97124 m .6255 .88425 L .62801 .90147 L p .152 g
F P
0 g
s
.25033 .67879 m .29939 .67513 L .33909 .6436 L p .862 g
F P
0 g
s
.33199 1.1528 m .35856 1.04127 L .34085 1.01753 L p .818 g
F P
0 g
s
.35856 1.04127 m .33199 1.1528 L .36471 1.17569 L p .752 g
F P
0 g
s
.40981 1.29098 m .40679 1.19327 L .36471 1.17569 L p .645 g
F P
0 g
s
.40679 1.19327 m .40981 1.29098 L .45709 1.30156 L p .595 g
F P
0 g
s
.3346 .99185 m .39817 .89788 L .40232 .88074 L p .909 g
F P
0 g
s
.39817 .89788 m .3346 .99185 L .34085 1.01753 L p .903 g
F P
0 g
s
.55004 .80427 m .60306 .77782 L .63727 .79459 L p .69 g
F P
0 g
s
.60306 .77782 m .55004 .80427 L .53327 .79983 L p .711 g
F P
0 g
s
.75826 .82679 m .67715 .84011 L .6625 .81588 L p .75 g
F P
0 g
s
.47307 .76544 m .49675 .79924 L .47958 .80314 L p .791 g
F P
0 g
s
.47958 .80314 m .43127 .77504 L .47307 .76544 L p .791 g
F P
0 g
s
.60143 .66564 m .6464 .6237 L .58606 .60672 L p .74 g
F P
0 g
s
.58606 .60672 m .56138 .65454 L .60143 .66564 L p .74 g
F P
0 g
s
.25033 .67879 m .2318 .70481 L .20101 .7521 L p .73 g
F P
0 g
s
.21824 .7878 m .20101 .7521 L .2318 .70481 L p .346 g
F P
0 g
s
.39175 .61938 m .43276 .66281 L .4738 .65307 L p .747 g
F P
0 g
s
.43276 .66281 m .39175 .61938 L .33909 .6436 L p .748 g
F P
0 g
s
.53327 .79983 m .56227 .76689 L .60306 .77782 L p .711 g
F P
0 g
s
.56227 .76689 m .53327 .79983 L .51508 .79811 L p .736 g
F P
0 g
s
.51785 .76265 m .51508 .79811 L .49675 .79924 L p .764 g
F P
0 g
s
.49675 .79924 m .47307 .76544 L .51785 .76265 L p .764 g
F P
0 g
s
.29559 .78315 m .36822 .81116 L .35129 .83493 L p .735 g
F P
0 g
s
.35129 .83493 m .27103 .819 L .29559 .78315 L p .735 g
F P
0 g
s
.51508 .79811 m .51785 .76265 L .56227 .76689 L p .736 g
F P
0 g
s
.81155 .79737 m .75826 .82679 L .73718 .79029 L p .814 g
F P
0 g
s
.73718 .79029 m .78654 .75244 L .81155 .79737 L p .814 g
F P
0 g
s
.6625 .81588 m .73718 .79029 L .75826 .82679 L p .75 g
F P
0 g
s
.56138 .65454 m .58606 .60672 L .52012 .60013 L p .743 g
F P
0 g
s
.52012 .60013 m .51776 .65024 L .56138 .65454 L p .743 g
F P
0 g
s
.65355 1.17982 m .64955 1.27835 L .68339 1.25779 L p .249 g
F P
0 g
s
.68339 1.25779 m .68842 1.15793 L .65355 1.17982 L p .249 g
F P
0 g
s
.6255 .88425 m .68802 .97124 L .67248 .94642 L p .054 g
F P
0 g
s
.67248 .94642 m .61523 .86791 L .6255 .88425 L p .054 g
F P
0 g
s
.45365 .60447 m .4738 .65307 L .51776 .65024 L p .745 g
F P
0 g
s
.4738 .65307 m .45365 .60447 L .39175 .61938 L p .747 g
F P
0 g
s
.78367 .68767 m .80269 .71425 L .75522 .67145 L p .367 g
F P
0 g
s
.75522 .67145 m .73913 .6474 L .78367 .68767 L p .367 g
F P
0 g
s
.2318 .70481 m .24753 .74363 L .21824 .7878 L p .346 g
F P
0 g
s
.27103 .819 m .21824 .7878 L .24753 .74363 L p .613 g
F P
0 g
s
.40232 .88074 m .34054 .96593 L .3346 .99185 L p .909 g
F P
0 g
s
.34054 .96593 m .40232 .88074 L .41414 .86473 L p F P
s
.51776 .65024 m .52012 .60013 L .45365 .60447 L p .745 g
F P
0 g
s
.6827 1.02267 m .71204 1.13177 L .72255 1.10296 L p .076 g
F P
0 g
s
.72255 1.10296 m .69141 .99726 L .6827 1.02267 L p .076 g
F P
0 g
s
.36894 1.27447 m .36471 1.17569 L .33199 1.1528 L p .698 g
F P
0 g
s
.36471 1.17569 m .36894 1.27447 L .40981 1.29098 L p .645 g
F P
0 g
s
.24753 .74363 m .29559 .78315 L .27103 .819 L p .613 g
F P
0 g
s
.69664 .64962 m .73913 .6474 L .67753 .61541 L p .619 g
F P
0 g
s
.67753 .61541 m .6464 .6237 L .69664 .64962 L p .619 g
F P
0 g
s
.31097 1.12595 m .34085 1.01753 L .3346 .99185 L p .86 g
F P
0 g
s
.34085 1.01753 m .31097 1.12595 L .33199 1.1528 L p .818 g
F P
0 g
s
.33909 .6436 m .29869 .63998 L .25033 .67879 L p .862 g
F P
0 g
s
.2318 .70481 m .25033 .67879 L .29869 .63998 L p F P
s
.73718 .79029 m .6625 .81588 L .63727 .79459 L p .748 g
F P
0 g
s
.29869 .63998 m .33909 .6436 L .39175 .61938 L p .835 g
F P
0 g
s
.61523 .86791 m .67248 .94642 L .64561 .92458 L p .376 g
F P
0 g
s
.64561 .92458 m .59781 .85361 L .61523 .86791 L p .376 g
F P
0 g
s
.33596 .75206 m .39548 .79069 L .36822 .81116 L p .735 g
F P
0 g
s
.36822 .81116 m .29559 .78315 L .33596 .75206 L p .735 g
F P
0 g
s
.35852 .94158 m .41414 .86473 L .43292 .851 L p F P
s
.41414 .86473 m .35852 .94158 L .34054 .96593 L p F P
s
.50891 1.36999 m .50782 1.30563 L .45709 1.30156 L p .442 g
F P
0 g
s
.50782 1.30563 m .50891 1.36999 L .54953 1.36791 L p .382 g
F P
0 g
s
.54953 1.36791 m .55888 1.30295 L .50782 1.30563 L p .382 g
F P
0 g
s
.68842 1.15793 m .68339 1.25779 L .70628 1.23324 L p .075 g
F P
0 g
s
.70628 1.23324 m .71204 1.13177 L .68842 1.15793 L p .075 g
F P
0 g
s
.55888 1.30295 m .54953 1.36791 L .58787 1.36071 L p .301 g
F P
0 g
s
.58787 1.36071 m .60713 1.29369 L .55888 1.30295 L p .301 g
F P
0 g
s
.80269 .71425 m .78654 .75244 L .74156 .7125 L p .783 g
F P
0 g
s
.74156 .7125 m .75522 .67145 L .80269 .71425 L p .783 g
F P
0 g
s
.46855 1.36683 m .45709 1.30156 L .40981 1.29098 L p .49 g
F P
0 g
s
.45709 1.30156 m .46855 1.36683 L .50891 1.36999 L p .442 g
F P
0 g
s
.3372 1.25298 m .33199 1.1528 L .31097 1.12595 L p .738 g
F P
0 g
s
.33199 1.1528 m .3372 1.25298 L .36894 1.27447 L p .698 g
F P
0 g
s
.69141 .99726 m .72255 1.10296 L .71887 1.07338 L p .037 g
F P
0 g
s
.71887 1.07338 m .68802 .97124 L .69141 .99726 L p .037 g
F P
0 g
s
.59781 .85361 m .64561 .92458 L .60914 .90738 L p .416 g
F P
0 g
s
.60914 .90738 m .57439 .84239 L .59781 .85361 L p .416 g
F P
0 g
s
.6464 .6237 m .67753 .61541 L .60319 .59439 L p .642 g
F P
0 g
s
.60319 .59439 m .58606 .60672 L .6464 .6237 L p .642 g
F P
0 g
s
.63727 .79459 m .69993 .758 L .73718 .79029 L p .748 g
F P
0 g
s
.69993 .758 m .63727 .79459 L .60306 .77782 L p .746 g
F P
0 g
s
.38756 .92058 m .43292 .851 L .4574 .84053 L p .083 g
F P
0 g
s
.43292 .851 m .38756 .92058 L .35852 .94158 L p F P
s
.78654 .75244 m .73718 .79029 L .69993 .758 L p .784 g
F P
0 g
s
.29869 .63998 m .28338 .66355 L .2318 .70481 L p F P
s
.24753 .74363 m .2318 .70481 L .28338 .66355 L p .435 g
F P
0 g
s
.36334 .61007 m .39175 .61938 L .45365 .60447 L p .787 g
F P
0 g
s
.39175 .61938 m .36334 .61007 L .29869 .63998 L p .835 g
F P
0 g
s
.30335 1.09682 m .3346 .99185 L .34054 .96593 L p .702 g
F P
0 g
s
.3346 .99185 m .30335 1.09682 L .31097 1.12595 L p .86 g
F P
0 g
s
.60713 1.29369 m .58787 1.36071 L .62149 1.34881 L p .182 g
F P
0 g
s
.62149 1.34881 m .64955 1.27835 L .60713 1.29369 L p .182 g
F P
0 g
s
.38956 .72817 m .43127 .77504 L .39548 .79069 L p .736 g
F P
0 g
s
.39548 .79069 m .33596 .75206 L .38956 .72817 L p .736 g
F P
0 g
s
.69993 .758 m .74156 .7125 L .78654 .75244 L p .784 g
F P
0 g
s
.431 1.35861 m .40981 1.29098 L .36894 1.27447 L p .526 g
F P
0 g
s
.40981 1.29098 m .431 1.35861 L .46855 1.36683 L p .49 g
F P
0 g
s
.29638 .70514 m .33596 .75206 L .29559 .78315 L p .627 g
F P
0 g
s
.29559 .78315 m .24753 .74363 L .29638 .70514 L p .627 g
F P
0 g
s
.73913 .6474 m .75522 .67145 L .68935 .63739 L p .415 g
F P
0 g
s
.68935 .63739 m .67753 .61541 L .73913 .6474 L p .415 g
F P
0 g
s
.57439 .84239 m .60914 .90738 L .5656 .89615 L p .401 g
F P
0 g
s
.5656 .89615 m .54657 .8351 L .57439 .84239 L p .401 g
F P
0 g
s
.42573 .90452 m .4574 .84053 L .48589 .83413 L p .173 g
F P
0 g
s
.4574 .84053 m .42573 .90452 L .38756 .92058 L p .083 g
F P
0 g
s
.28338 .66355 m .29638 .70514 L .24753 .74363 L p .435 g
F P
0 g
s
.58606 .60672 m .60319 .59439 L .52176 .58621 L p .679 g
F P
0 g
s
.52176 .58621 m .52012 .60013 L .58606 .60672 L p .679 g
F P
0 g
s
.28338 .66355 m .29869 .63998 L .36334 .61007 L p .072 g
F P
0 g
s
.54657 .8351 m .5656 .89615 L .51817 .89179 L p .354 g
F P
0 g
s
.51817 .89179 m .51635 .83227 L .54657 .8351 L p .354 g
F P
0 g
s
.60306 .77782 m .6488 .73243 L .69993 .758 L p .746 g
F P
0 g
s
.6488 .73243 m .60306 .77782 L .56227 .76689 L p .744 g
F P
0 g
s
.43965 .5916 m .45365 .60447 L .52012 .60013 L p .73 g
F P
0 g
s
.45365 .60447 m .43965 .5916 L .36334 .61007 L p .787 g
F P
0 g
s
.47035 .89466 m .48589 .83413 L .51635 .83227 L p .282 g
F P
0 g
s
.48589 .83413 m .47035 .89466 L .42573 .90452 L p .173 g
F P
0 g
s
.51635 .83227 m .51817 .89179 L .47035 .89466 L p .282 g
F P
0 g
s
.45257 .71346 m .47307 .76544 L .43127 .77504 L p .737 g
F P
0 g
s
.43127 .77504 m .38956 .72817 L .45257 .71346 L p .737 g
F P
0 g
s
.71204 1.13177 m .70628 1.23324 L .71643 1.20621 L p .021 g
F P
0 g
s
.71643 1.20621 m .72255 1.10296 L .71204 1.13177 L p .021 g
F P
0 g
s
.52012 .60013 m .52176 .58621 L .43965 .5916 L p .73 g
F P
0 g
s
.64955 1.27835 m .62149 1.34881 L .64816 1.33291 L p .069 g
F P
0 g
s
.64816 1.33291 m .68339 1.25779 L .64955 1.27835 L p .069 g
F P
0 g
s
.68802 .97124 m .71887 1.07338 L .70085 1.04508 L p .57 g
F P
0 g
s
.70085 1.04508 m .67248 .94642 L .68802 .97124 L p .57 g
F P
0 g
s
.39865 1.34581 m .36894 1.27447 L .3372 1.25298 L p .535 g
F P
0 g
s
.36894 1.27447 m .39865 1.34581 L .431 1.35861 L p .526 g
F P
0 g
s
.31683 1.22778 m .31097 1.12595 L .30335 1.09682 L p .653 g
F P
0 g
s
.31097 1.12595 m .31683 1.22778 L .3372 1.25298 L p .738 g
F P
0 g
s
.56227 .76689 m .58738 .71568 L .6488 .73243 L p .744 g
F P
0 g
s
.58738 .71568 m .56227 .76689 L .51785 .76265 L p .741 g
F P
0 g
s
.52025 .70917 m .51785 .76265 L .47307 .76544 L p .739 g
F P
0 g
s
.47307 .76544 m .45257 .71346 L .52025 .70917 L p .739 g
F P
0 g
s
.34054 .96593 m .31002 1.06733 L .30335 1.09682 L p .702 g
F P
0 g
s
.31002 1.06733 m .34054 .96593 L .35852 .94158 L p .011 g
F P
0 g
s
.51785 .76265 m .52025 .70917 L .58738 .71568 L p .741 g
F P
0 g
s
.74156 .7125 m .69993 .758 L .6488 .73243 L p .754 g
F P
0 g
s
.75522 .67145 m .74156 .7125 L .67931 .68076 L p .724 g
F P
0 g
s
.67931 .68076 m .68935 .63739 L .75522 .67145 L p .724 g
F P
0 g
s
.67753 .61541 m .68935 .63739 L .60971 .61499 L p .401 g
F P
0 g
s
.60971 .61499 m .60319 .59439 L .67753 .61541 L p .401 g
F P
0 g
s
.36171 .67546 m .38956 .72817 L .33596 .75206 L p .643 g
F P
0 g
s
.33596 .75206 m .29638 .70514 L .36171 .67546 L p .643 g
F P
0 g
s
.36334 .61007 m .35255 .6317 L .28338 .66355 L p .072 g
F P
0 g
s
.29638 .70514 m .28338 .66355 L .35255 .6317 L p .496 g
F P
0 g
s
.35255 .6317 m .36334 .61007 L .43965 .5916 L p .161 g
F P
0 g
s
.6488 .73243 m .67931 .68076 L .74156 .7125 L p .754 g
F P
0 g
s
.68339 1.25779 m .64816 1.33291 L .66605 1.31398 L p F P
s
.66605 1.31398 m .70628 1.23324 L .68339 1.25779 L p F P
s
.37366 1.32919 m .3372 1.25298 L .31683 1.22778 L p .461 g
F P
0 g
s
.3372 1.25298 m .37366 1.32919 L .39865 1.34581 L p .535 g
F P
0 g
s
.35255 .6317 m .36171 .67546 L .29638 .70514 L p .496 g
F P
0 g
s
.67248 .94642 m .70085 1.04508 L .66939 1.02012 L p .625 g
F P
0 g
s
.66939 1.02012 m .64561 .92458 L .67248 .94642 L p .625 g
F P
0 g
s
.72255 1.10296 m .71643 1.20621 L .7128 1.17848 L p .696 g
F P
0 g
s
.7128 1.17848 m .71887 1.07338 L .72255 1.10296 L p .696 g
F P
0 g
s
.33094 1.03956 m .35852 .94158 L .38756 .92058 L p .091 g
F P
0 g
s
.35852 .94158 m .33094 1.03956 L .31002 1.06733 L p .011 g
F P
0 g
s
.51079 1.39308 m .50891 1.36999 L .46855 1.36683 L p .145 g
F P
0 g
s
.50891 1.36999 m .51079 1.39308 L .53345 1.39193 L p .117 g
F P
0 g
s
.53345 1.39193 m .54953 1.36791 L .50891 1.36999 L p .117 g
F P
0 g
s
.67931 .68076 m .6488 .73243 L .58738 .71568 L p .727 g
F P
0 g
s
.30335 1.09682 m .30948 1.20045 L .31683 1.22778 L p .653 g
F P
0 g
s
.30948 1.20045 m .30335 1.09682 L .31002 1.06733 L p .029 g
F P
0 g
s
.60319 .59439 m .60971 .61499 L .52239 .60627 L p .353 g
F P
0 g
s
.52239 .60627 m .52176 .58621 L .60319 .59439 L p .353 g
F P
0 g
s
.54953 1.36791 m .53345 1.39193 L .55479 1.38796 L p .072 g
F P
0 g
s
.55479 1.38796 m .58787 1.36071 L .54953 1.36791 L p .072 g
F P
0 g
s
.43432 .61201 m .43965 .5916 L .52176 .58621 L p .276 g
F P
0 g
s
.43965 .5916 m .43432 .61201 L .35255 .6317 L p .161 g
F P
0 g
s
.48828 1.39133 m .46855 1.36683 L .431 1.35861 L p .16 g
F P
0 g
s
.46855 1.36683 m .48828 1.39133 L .51079 1.39308 L p .145 g
F P
0 g
s
.43885 .65714 m .45257 .71346 L .38956 .72817 L p .661 g
F P
0 g
s
.38956 .72817 m .36171 .67546 L .43885 .65714 L p .661 g
F P
0 g
s
.52176 .58621 m .52239 .60627 L .43432 .61201 L p .276 g
F P
0 g
s
.68935 .63739 m .67931 .68076 L .60418 .65991 L p .674 g
F P
0 g
s
.60418 .65991 m .60971 .61499 L .68935 .63739 L p .674 g
F P
0 g
s
.58787 1.36071 m .55479 1.38796 L .57341 1.3814 L p .003 g
F P
0 g
s
.57341 1.3814 m .62149 1.34881 L .58787 1.36071 L p .003 g
F P
0 g
s
.46739 1.3868 m .431 1.35861 L .39865 1.34581 L p .156 g
F P
0 g
s
.431 1.35861 m .46739 1.3868 L .48828 1.39133 L p .16 g
F P
0 g
s
.36171 .67546 m .35255 .6317 L .43432 .61201 L p .545 g
F P
0 g
s
.70628 1.23324 m .66605 1.31398 L .67377 1.29322 L p .759 g
F P
0 g
s
.67377 1.29322 m .71643 1.20621 L .70628 1.23324 L p .759 g
F P
0 g
s
.64561 .92458 m .66939 1.02012 L .62648 1.00041 L p .593 g
F P
0 g
s
.62648 1.00041 m .60914 .90738 L .64561 .92458 L p .593 g
F P
0 g
s
.58738 .71568 m .60418 .65991 L .67931 .68076 L p .727 g
F P
0 g
s
.60418 .65991 m .58738 .71568 L .52025 .70917 L p .703 g
F P
0 g
s
.52186 .65179 m .52025 .70917 L .45257 .71346 L p .681 g
F P
0 g
s
.45257 .71346 m .43885 .65714 L .52186 .65179 L p .681 g
F P
0 g
s
.36498 1.01554 m .38756 .92058 L .42573 .90452 L p .243 g
F P
0 g
s
.38756 .92058 m .36498 1.01554 L .33094 1.03956 L p .091 g
F P
0 g
s
.31683 1.22778 m .35779 1.30978 L .37366 1.32919 L p .461 g
F P
0 g
s
.35779 1.30978 m .31683 1.22778 L .30948 1.20045 L p .052 g
F P
0 g
s
.43432 .61201 m .43885 .65714 L .36171 .67546 L p .545 g
F P
0 g
s
.52025 .70917 m .52186 .65179 L .60418 .65991 L p .703 g
F P
0 g
s
.62149 1.34881 m .57341 1.3814 L .58806 1.37269 L p F P
s
.58806 1.37269 m .64816 1.33291 L .62149 1.34881 L p F P
s
.71887 1.07338 m .7128 1.17848 L .69527 1.15196 L p .784 g
F P
0 g
s
.69527 1.15196 m .70085 1.04508 L .71887 1.07338 L p .784 g
F P
0 g
s
.4495 1.37976 m .39865 1.34581 L .37366 1.32919 L p .118 g
F P
0 g
s
.39865 1.34581 m .4495 1.37976 L .46739 1.3868 L p .156 g
F P
0 g
s
.60971 .61499 m .60418 .65991 L .52186 .65179 L p .63 g
F P
0 g
s
.52186 .65179 m .52239 .60627 L .60971 .61499 L p .63 g
F P
0 g
s
.31602 1.17281 m .31002 1.06733 L .33094 1.03956 L p .08 g
F P
0 g
s
.31002 1.06733 m .31602 1.17281 L .30948 1.20045 L p .029 g
F P
0 g
s
.43885 .65714 m .43432 .61201 L .52239 .60627 L p .588 g
F P
0 g
s
.60914 .90738 m .62648 1.00041 L .57512 .98753 L p .545 g
F P
0 g
s
.57512 .98753 m .5656 .89615 L .60914 .90738 L p .545 g
F P
0 g
s
.40992 .99713 m .42573 .90452 L .47035 .89466 L p .354 g
F P
0 g
s
.42573 .90452 m .40992 .99713 L .36498 1.01554 L p .243 g
F P
0 g
s
.52239 .60627 m .52186 .65179 L .43885 .65714 L p .588 g
F P
0 g
s
.64816 1.33291 m .58806 1.37269 L .59773 1.36236 L p .84 g
F P
0 g
s
.59773 1.36236 m .66605 1.31398 L .64816 1.33291 L p .84 g
F P
0 g
s
.5656 .89615 m .57512 .98753 L .51908 .98253 L p .492 g
F P
0 g
s
.51908 .98253 m .51817 .89179 L .5656 .89615 L p .492 g
F P
0 g
s
.71643 1.20621 m .67377 1.29322 L .67059 1.272 L p .901 g
F P
0 g
s
.67059 1.272 m .7128 1.17848 L .71643 1.20621 L p .901 g
F P
0 g
s
.46259 .98582 m .47035 .89466 L .51817 .89179 L p .431 g
F P
0 g
s
.47035 .89466 m .46259 .98582 L .40992 .99713 L p .354 g
F P
0 g
s
.37366 1.32919 m .43581 1.37065 L .4495 1.37976 L p .118 g
F P
0 g
s
.43581 1.37065 m .37366 1.32919 L .35779 1.30978 L p .255 g
F P
0 g
s
.51817 .89179 m .51908 .98253 L .46259 .98582 L p .431 g
F P
0 g
s
.35229 1.2888 m .30948 1.20045 L .31602 1.17281 L p .145 g
F P
0 g
s
.30948 1.20045 m .35229 1.2888 L .35779 1.30978 L p .052 g
F P
0 g
s
.70085 1.04508 m .69527 1.15196 L .66471 1.12858 L p .736 g
F P
0 g
s
.66471 1.12858 m .66939 1.02012 L .70085 1.04508 L p .736 g
F P
0 g
s
.33637 1.14679 m .33094 1.03956 L .36498 1.01554 L p .273 g
F P
0 g
s
.33094 1.03956 m .33637 1.14679 L .31602 1.17281 L p .08 g
F P
0 g
s
.66605 1.31398 m .59773 1.36236 L .60171 1.35111 L p .932 g
F P
0 g
s
.60171 1.35111 m .67377 1.29322 L .66605 1.31398 L p .932 g
F P
0 g
s
.51316 1.3707 m .51079 1.39308 L .48828 1.39133 L closepath p .707 g
F P
0 g
s
.51316 1.3707 m .53345 1.39193 L .51079 1.39308 L closepath p .737 g
F P
0 g
s
.42728 1.36008 m .35779 1.30978 L .35229 1.2888 L p .355 g
F P
0 g
s
.35779 1.30978 m .42728 1.36008 L .43581 1.37065 L p .255 g
F P
0 g
s
.51316 1.3707 m .55479 1.38796 L .53345 1.39193 L closepath p .771 g
F P
0 g
s
.51316 1.3707 m .48828 1.39133 L .46739 1.3868 L closepath p .684 g
F P
0 g
s
.51316 1.3707 m .57341 1.3814 L .55479 1.38796 L closepath p .802 g
F P
0 g
s
.7128 1.17848 m .67059 1.272 L .65649 1.25178 L p .854 g
F P
0 g
s
.65649 1.25178 m .69527 1.15196 L .7128 1.17848 L p .854 g
F P
0 g
s
.51316 1.3707 m .46739 1.3868 L .4495 1.37976 L closepath p .669 g
F P
0 g
s
.66939 1.02012 m .66471 1.12858 L .62306 1.11013 L p .675 g
F P
0 g
s
.62306 1.11013 m .62648 1.00041 L .66939 1.02012 L p .675 g
F P
0 g
s
.35777 1.26767 m .31602 1.17281 L .33637 1.14679 L p .32 g
F P
0 g
s
.31602 1.17281 m .35777 1.26767 L .35229 1.2888 L p .145 g
F P
0 g
s
.51316 1.3707 m .58806 1.37269 L .57341 1.3814 L closepath p .825 g
F P
0 g
s
.51316 1.3707 m .4495 1.37976 L .43581 1.37065 L closepath p .661 g
F P
0 g
s
.36943 1.12429 m .36498 1.01554 L .40992 .99713 L p .384 g
F P
0 g
s
.36498 1.01554 m .36943 1.12429 L .33637 1.14679 L p .273 g
F P
0 g
s
.67377 1.29322 m .60171 1.35111 L .59967 1.33969 L p .911 g
F P
0 g
s
.59967 1.33969 m .67059 1.272 L .67377 1.29322 L p .911 g
F P
0 g
s
.51316 1.3707 m .59773 1.36236 L .58806 1.37269 L closepath p .836 g
F P
0 g
s
.42453 1.34873 m .35229 1.2888 L .35777 1.26767 L p .445 g
F P
0 g
s
.35229 1.2888 m .42453 1.34873 L .42728 1.36008 L p .355 g
F P
0 g
s
.51316 1.3707 m .43581 1.37065 L .42728 1.36008 L closepath p .658 g
F P
0 g
s
.62648 1.00041 m .62306 1.11013 L .57324 1.09808 L p .621 g
F P
0 g
s
.57324 1.09808 m .57512 .98753 L .62648 1.00041 L p .621 g
F P
0 g
s
.69527 1.15196 m .65649 1.25178 L .63224 1.23403 L p .784 g
F P
0 g
s
.63224 1.23403 m .66471 1.12858 L .69527 1.15196 L p .784 g
F P
0 g
s
.41304 1.10707 m .40992 .99713 L .46259 .98582 L p .459 g
F P
0 g
s
.40992 .99713 m .41304 1.10707 L .36943 1.12429 L p .384 g
F P
0 g
s
.51316 1.3707 m .60171 1.35111 L .59773 1.36236 L closepath p .835 g
F P
0 g
s
.51316 1.3707 m .42728 1.36008 L .42453 1.34873 L closepath p .659 g
F P
0 g
s
.37406 1.24785 m .33637 1.14679 L .36943 1.12429 L p .42 g
F P
0 g
s
.33637 1.14679 m .37406 1.24785 L .35777 1.26767 L p .32 g
F P
0 g
s
.67059 1.272 m .59967 1.33969 L .59167 1.32888 L p .857 g
F P
0 g
s
.59167 1.32888 m .65649 1.25178 L .67059 1.272 L p .857 g
F P
0 g
s
.57512 .98753 m .57324 1.09808 L .5189 1.0934 L p .57 g
F P
0 g
s
.5189 1.0934 m .51908 .98253 L .57512 .98753 L p .57 g
F P
0 g
s
.46412 1.09648 m .46259 .98582 L .51908 .98253 L p .518 g
F P
0 g
s
.46259 .98582 m .46412 1.09648 L .41304 1.10707 L p .459 g
F P
0 g
s
.42783 1.33737 m .35777 1.26767 L .37406 1.24785 L p .502 g
F P
0 g
s
.35777 1.26767 m .42783 1.33737 L .42453 1.34873 L p .445 g
F P
0 g
s
.51316 1.3707 m .59967 1.33969 L .60171 1.35111 L closepath p .825 g
F P
0 g
s
.51908 .98253 m .5189 1.0934 L .46412 1.09648 L p .518 g
F P
0 g
s
.51316 1.3707 m .42453 1.34873 L .42783 1.33737 L closepath p .663 g
F P
0 g
s
.51316 1.3707 m .5782 1.31944 L .59167 1.32888 L closepath p .79 g
F P
0 g
s
.51316 1.3707 m .56017 1.31204 L .5782 1.31944 L closepath p .77 g
F P
0 g
s
.51316 1.3707 m .5388 1.30724 L .56017 1.31204 L closepath p .751 g
F P
0 g
s
.51316 1.3707 m .5156 1.30538 L .5388 1.30724 L closepath p .732 g
F P
0 g
s
.51316 1.3707 m .49224 1.30661 L .5156 1.30538 L closepath p .716 g
F P
0 g
s
.51316 1.3707 m .47036 1.31082 L .49224 1.30661 L closepath p .701 g
F P
0 g
s
.51316 1.3707 m .45152 1.31771 L .47036 1.31082 L closepath p .688 g
F P
0 g
s
.51316 1.3707 m .43702 1.32678 L .45152 1.31771 L closepath p .677 g
F P
0 g
s
.51316 1.3707 m .42783 1.33737 L .43702 1.32678 L closepath p .669 g
F P
0 g
s
.51316 1.3707 m .59167 1.32888 L .59967 1.33969 L closepath p .809 g
F P
0 g
s
.66471 1.12858 m .63224 1.23403 L .5994 1.22006 L p .723 g
F P
0 g
s
.5994 1.22006 m .62306 1.11013 L .66471 1.12858 L p .723 g
F P
0 g
s
.37406 1.24785 m .43702 1.32678 L .42783 1.33737 L p .502 g
F P
0 g
s
.40025 1.23077 m .36943 1.12429 L .41304 1.10707 L p .486 g
F P
0 g
s
.36943 1.12429 m .40025 1.23077 L .37406 1.24785 L p .42 g
F P
0 g
s
.65649 1.25178 m .59167 1.32888 L .5782 1.31944 L p .803 g
F P
0 g
s
.5782 1.31944 m .63224 1.23403 L .65649 1.25178 L p .803 g
F P
0 g
s
.43702 1.32678 m .37406 1.24785 L .40025 1.23077 L p .544 g
F P
0 g
s
.40025 1.23077 m .45152 1.31771 L .43702 1.32678 L p .544 g
F P
0 g
s
.62306 1.11013 m .5994 1.22006 L .56027 1.21094 L p .672 g
F P
0 g
s
.56027 1.21094 m .57324 1.09808 L .62306 1.11013 L p .672 g
F P
0 g
s
.4346 1.21773 m .41304 1.10707 L .46412 1.09648 L p .538 g
F P
0 g
s
.41304 1.10707 m .4346 1.21773 L .40025 1.23077 L p .486 g
F P
0 g
s
.63224 1.23403 m .5782 1.31944 L .56017 1.31204 L p .755 g
F P
0 g
s
.56017 1.31204 m .5994 1.22006 L .63224 1.23403 L p .755 g
F P
0 g
s
.4346 1.21773 m .47036 1.31082 L .45152 1.31771 L p .579 g
F P
0 g
s
.45152 1.31771 m .40025 1.23077 L .4346 1.21773 L p .579 g
F P
0 g
s
.57324 1.09808 m .56027 1.21094 L .51766 1.20741 L p .626 g
F P
0 g
s
.51766 1.20741 m .5189 1.0934 L .57324 1.09808 L p .626 g
F P
0 g
s
.5994 1.22006 m .56017 1.31204 L .5388 1.30724 L p .714 g
F P
0 g
s
.47471 1.20974 m .46412 1.09648 L .5189 1.0934 L p .583 g
F P
0 g
s
.46412 1.09648 m .47471 1.20974 L .4346 1.21773 L p .538 g
F P
0 g
s
.47471 1.20974 m .49224 1.30661 L .47036 1.31082 L p .612 g
F P
0 g
s
.47036 1.31082 m .4346 1.21773 L .47471 1.20974 L p .612 g
F P
0 g
s
.5189 1.0934 m .51766 1.20741 L .47471 1.20974 L p .583 g
F P
0 g
s
.5388 1.30724 m .56027 1.21094 L .5994 1.22006 L p .714 g
F P
0 g
s
.56027 1.21094 m .5388 1.30724 L .5156 1.30538 L p .677 g
F P
0 g
s
.51766 1.20741 m .5156 1.30538 L .49224 1.30661 L p .643 g
F P
0 g
s
.49224 1.30661 m .47471 1.20974 L .51766 1.20741 L p .643 g
F P
0 g
s
.5156 1.30538 m .51766 1.20741 L .56027 1.21094 L p .677 g
F P
0 g
s
.25 Mabswid
.64785 0 m
.92348 .39072 L
s
.92348 .39072 m
1 1.47188 L
s
1 1.47188 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.0843 .2212 m
0 1.32748 L
s
0 1.32748 m
.67613 1.13533 L
s
.67613 1.13533 m
.64785 0 L
s
.64785 0 m
.0843 .2212 L
s
.0843 .2212 m
.64785 0 L
s
.09701 .21621 m
.10379 .22387 L
s
gsave
.08347 .2009 -74.133 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.226 .16558 m
.23257 .17341 L
s
gsave
.21284 .14994 -79.984 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.35834 .11363 m
.36471 .12163 L
s
gsave
.3456 .09765 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4942 .06031 m
.50035 .06847 L
s
gsave
.4819 .04399 -74.2609 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.63369 .00556 m
.63961 .01389 L
s
gsave
.62186 -0.01111 -68.2133 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.12255 .20619 m
.12659 .2108 L
s
.14821 .19612 m
.15223 .20075 L
s
.17401 .18599 m
.178 .19064 L
s
.19993 .17581 m
.2039 .18049 L
s
.25219 .1553 m
.25611 .16001 L
s
.27852 .14497 m
.28242 .1497 L
s
.30499 .13458 m
.30886 .13933 L
s
.3316 .12413 m
.33545 .12891 L
s
.38523 .10308 m
.38903 .1079 L
s
.41226 .09247 m
.41603 .09731 L
s
.43943 .08181 m
.44317 .08666 L
s
.46674 .07109 m
.47046 .07596 L
s
.5218 .04948 m
.52546 .05439 L
s
.54955 .03858 m
.55318 .04352 L
s
.57745 .02763 m
.58105 .03259 L
s
.60549 .01662 m
.60907 .0216 L
s
gsave
.2819 .01774 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.678571 0.00827773 0.97619 0.339387 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.0651548 1.12552 2.22045e-016 1.12552 [
[.5 1.11398 -20.875 0 ]
[.5 1.11398 20.875 10 ]
[.07154 .25839 -20.4146 -10 ]
[.07154 .25839 .21037 0 ]
[.22889 .19635 -19.474 -10 ]
[.22889 .19635 1.15096 0 ]
[.39487 .13095 -7.58187 -10 ]
[.39487 .13095 .85563 0 ]
[.5702 .06189 -12.9015 -10 ]
[.5702 .06189 2.22355 0 ]
[.7557 -0.01112 -12.2117 -10 ]
[.7557 -0.01112 2.91331 0 ]
[.35102 .07594 -7.13257 -10 ]
[.35102 .07594 .80493 0 ]
[.77886 -0.00434 0 -6.96183 ]
[.77886 -0.00434 20.625 3.03817 ]
[.82009 .06289 0 -6.89909 ]
[.82009 .06289 20.625 3.10091 ]
[.85923 .12672 0 -6.84024 ]
[.85923 .12672 8.4375 3.15976 ]
[.89644 .18741 0 -6.78494 ]
[.89644 .18741 15.125 3.21506 ]
[.93186 .24517 0 -6.73285 ]
[.93186 .24517 15.125 3.26715 ]
[.92525 .10243 0 -6.84024 ]
[.92525 .10243 7.9375 3.15976 ]
[.06633 .27587 -20.625 -3.03872 ]
[.06633 .27587 0 6.96128 ]
[.0489 .41979 -20.625 -3.15535 ]
[.0489 .41979 0 6.84465 ]
[.02996 .57614 -8.4375 -3.28285 ]
[.02996 .57614 0 6.71715 ]
[.00933 .7466 -15.125 -3.4228 ]
[.00933 .7466 0 6.5772 ]
[-0.01326 .93319 -15.125 -3.57713 ]
[-0.01326 .93319 0 6.42287 ]
[-0.03657 .59899 -7.9375 -3.28285 ]
[-0.03657 .59899 0 6.71715 ]
[ 0 0 0 0 ]
[ 1 1.10148 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.11398 -81.875 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(d) show
67.437500 11.500000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(zx) show
73.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
79.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
100.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.07962 .26913 m
.76528 0 L
s
.08139 .26844 m
.08631 .27346 L
s
gsave
.07154 .25839 -81.4146 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.23824 .20687 m
.24291 .21213 L
s
gsave
.22889 .19635 -80.474 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.40364 .14195 m
.40803 .14745 L
s
gsave
.39487 .13095 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.57832 .07338 m
.58238 .07913 L
s
gsave
.5702 .06189 -73.9015 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.76307 .00087 m
.76676 .00686 L
s
gsave
.7557 -0.01112 -73.2117 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.11983 .25335 m
.12275 .2564 L
s
.15879 .23806 m
.16167 .24114 L
s
.19825 .22257 m
.20109 .22569 L
s
.27876 .19096 m
.28152 .19416 L
s
.31983 .17485 m
.32255 .17807 L
s
.36145 .15851 m
.36413 .16177 L
s
.44641 .12516 m
.449 .1285 L
s
.48977 .10814 m
.49231 .11151 L
s
.53374 .09088 m
.53622 .09429 L
s
.62353 .05564 m
.62591 .05912 L
s
.66938 .03764 m
.67171 .04116 L
s
.71589 .01938 m
.71816 .02294 L
s
gsave
.35102 .07594 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.76528 0 m
.91897 .25043 L
s
.76577 .0008 m
.75922 .00337 L
s
gsave
.77886 -0.00434 -61 -10.9618 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.80694 .06789 m
.80036 .07038 L
s
gsave
.82009 .06289 -61 -10.8991 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.84603 .13158 m
.83943 .13401 L
s
gsave
.85923 .12672 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.88319 .19214 m
.87657 .1945 L
s
gsave
.89644 .18741 -61 -10.7849 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.91857 .24978 m
.91192 .25208 L
s
gsave
.93186 .24517 -61 -10.7329 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.77417 .0145 m
.77024 .01603 L
s
.7825 .02806 m
.77856 .02958 L
s
.79073 .04147 m
.78679 .04299 L
s
.79888 .05475 m
.79493 .05626 L
s
.81492 .08089 m
.81097 .08238 L
s
.82282 .09376 m
.81886 .09524 L
s
.83063 .10649 m
.82668 .10797 L
s
.83837 .1191 m
.83441 .12057 L
s
.85361 .14394 m
.84965 .14539 L
s
.86112 .15617 m
.85715 .15761 L
s
.86855 .16828 m
.86458 .16971 L
s
.87591 .18027 m
.87194 .18169 L
s
.89041 .20389 m
.88643 .20531 L
s
.89755 .21553 m
.89357 .21694 L
s
.90463 .22706 m
.90064 .22846 L
s
.91163 .23848 m
.90765 .23986 L
s
gsave
.92525 .10243 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.07962 .26913 m
0 .93161 L
s
.07942 .27074 m
.08597 .26817 L
s
gsave
.06633 .27587 -81.625 -7.03872 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.0621 .41492 m
.0687 .41248 L
s
gsave
.0489 .41979 -81.625 -7.15535 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.04327 .57157 m
.04992 .56928 L
s
gsave
.02996 .57614 -69.4375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.02274 .74237 m
.02945 .74025 L
s
gsave
.00933 .7466 -76.125 -7.4228 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00027 .92934 m
.00704 .92741 L
s
gsave
-0.01326 .93319 -76.125 -7.57713 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.07522 .30569 m
.07916 .30417 L
s
.07094 .34136 m
.07488 .33986 L
s
.06656 .37776 m
.07051 .37628 L
s
.05754 .45285 m
.06151 .45141 L
s
.05288 .49159 m
.05686 .49017 L
s
.04813 .53115 m
.05211 .52975 L
s
.03831 .61287 m
.04231 .61152 L
s
.03323 .65508 m
.03724 .65376 L
s
.02805 .69824 m
.03206 .69694 L
s
.01732 .78751 m
.02135 .78627 L
s
.01177 .83369 m
.01581 .83248 L
s
.00609 .88096 m
.01014 .87977 L
s
gsave
-0.03657 .59899 -68.9375 -7.28285 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.07962 .26913 m
0 .93161 L
s
0 .93161 m
.24138 1.10148 L
s
.24138 1.10148 m
.27878 .47223 L
s
.27878 .47223 m
.07962 .26913 L
s
.76528 0 m
.91897 .25043 L
s
.91897 .25043 m
1 .91568 L
s
1 .91568 m
.82346 .69758 L
s
.82346 .69758 m
.76528 0 L
s
.07962 .26913 m
0 .93161 L
s
0 .93161 m
.82346 .69758 L
s
.82346 .69758 m
.76528 0 L
s
.76528 0 m
.07962 .26913 L
s
.27878 .47223 m
.91897 .25043 L
s
.91897 .25043 m
1 .91568 L
s
1 .91568 m
.24138 1.10148 L
s
.24138 1.10148 m
.27878 .47223 L
s
0 0 m
1 0 L
1 1.10148 L
0 1.10148 L
closepath
clip
newpath
.5 Mabswid
.31373 .39946 m .27755 .44147 L .3554 .48043 L p .625 g
F P
0 g
s
.3554 .48043 m .37858 .43822 L .31373 .39946 L p .625 g
F P
0 g
s
.34285 .52867 m .3554 .48043 L .27755 .44147 L p .622 g
F P
0 g
s
.37858 .43822 m .3554 .48043 L .43334 .50862 L p .527 g
F P
0 g
s
.3554 .48043 m .34285 .52867 L .42776 .54992 L p .528 g
F P
0 g
s
.42776 .54992 m .43334 .50862 L .3554 .48043 L p .528 g
F P
0 g
s
.26429 .35852 m .21753 .3955 L .27755 .44147 L p .714 g
F P
0 g
s
.27755 .44147 m .31373 .39946 L .26429 .35852 L p .714 g
F P
0 g
s
.25773 .49255 m .27755 .44147 L .21753 .3955 L p .718 g
F P
0 g
s
.27755 .44147 m .25773 .49255 L .34285 .52867 L p .622 g
F P
0 g
s
.34394 .5739 m .34285 .52867 L .25773 .49255 L p .576 g
F P
0 g
s
.34285 .52867 m .34394 .5739 L .42842 .58725 L p .48 g
F P
0 g
s
.42842 .58725 m .42776 .54992 L .34285 .52867 L p .48 g
F P
0 g
s
.32383 .34228 m .26429 .35852 L .31373 .39946 L p .672 g
F P
0 g
s
.31373 .39946 m .36015 .3752 L .32383 .34228 L p .672 g
F P
0 g
s
.36015 .3752 m .31373 .39946 L .37858 .43822 L p .585 g
F P
0 g
s
.21753 .3955 m .1917 .44536 L .25773 .49255 L p .718 g
F P
0 g
s
.25909 .54329 m .25773 .49255 L .1917 .44536 L p .669 g
F P
0 g
s
.25773 .49255 m .25909 .54329 L .34394 .5739 L p .576 g
F P
0 g
s
.37858 .43822 m .4085 .41064 L .36015 .3752 L p .585 g
F P
0 g
s
.4085 .41064 m .37858 .43822 L .44381 .47127 L p .466 g
F P
0 g
s
.43334 .50862 m .44381 .47127 L .37858 .43822 L p .527 g
F P
0 g
s
.24271 .32327 m .19046 .35142 L .21753 .3955 L p .83 g
F P
0 g
s
.21753 .3955 m .26429 .35852 L .24271 .32327 L p .83 g
F P
0 g
s
.1917 .44536 m .21753 .3955 L .19046 .35142 L p .853 g
F P
0 g
s
.44381 .47127 m .43334 .50862 L .49314 .52726 L p .356 g
F P
0 g
s
.43334 .50862 m .42776 .54992 L .49275 .55774 L p .381 g
F P
0 g
s
.49275 .55774 m .49314 .52726 L .43334 .50862 L p .381 g
F P
0 g
s
.30872 .31812 m .24271 .32327 L .26429 .35852 L p .755 g
F P
0 g
s
.26429 .35852 m .32383 .34228 L .30872 .31812 L p .755 g
F P
0 g
s
.36061 .60714 m .34394 .5739 L .25909 .54329 L p .413 g
F P
0 g
s
.34394 .5739 m .36061 .60714 L .43611 .61292 L p .326 g
F P
0 g
s
.43611 .61292 m .42842 .58725 L .34394 .5739 L p .326 g
F P
0 g
s
.42776 .54992 m .42842 .58725 L .49281 .58483 L p .329 g
F P
0 g
s
.49281 .58483 m .49275 .55774 L .42776 .54992 L p .329 g
F P
0 g
s
.38682 .35287 m .32383 .34228 L .36015 .3752 L p .548 g
F P
0 g
s
.36015 .3752 m .40966 .3761 L .38682 .35287 L p .548 g
F P
0 g
s
.40966 .3761 m .36015 .3752 L .4085 .41064 L p .451 g
F P
0 g
s
.1917 .44536 m .19332 .49902 L .25909 .54329 L p .669 g
F P
0 g
s
.2848 .58396 m .25909 .54329 L .19332 .49902 L p .423 g
F P
0 g
s
.25909 .54329 m .2848 .58396 L .36061 .60714 L p .413 g
F P
0 g
s
.19046 .35142 m .16152 .39636 L .1917 .44536 L p .853 g
F P
0 g
s
.19332 .49902 m .1917 .44536 L .16152 .39636 L p .748 g
F P
0 g
s
.44381 .47127 m .45739 .44554 L .4085 .41064 L p .466 g
F P
0 g
s
.45739 .44554 m .44381 .47127 L .49387 .49946 L p .219 g
F P
0 g
s
.49314 .52726 m .49387 .49946 L .44381 .47127 L p .356 g
F P
0 g
s
.4085 .41064 m .44056 .40537 L .40966 .3761 L p .451 g
F P
0 g
s
.44056 .40537 m .4085 .41064 L .45739 .44554 L p .268 g
F P
0 g
s
.37787 .33993 m .30872 .31812 L .32383 .34228 L p .566 g
F P
0 g
s
.3855 .34256 m .31964 .31116 L .31635 .31326 L p .067 g
F P
0 g
s
.38682 .35287 m .37787 .33993 L p .34749 .34125 L .566 g
F P
0 g
s
.43912 .38999 m .37787 .33993 L .38682 .35287 L p .171 g
F P
0 g
s
.44321 .39393 m .38682 .35287 L .40966 .3761 L p .228 g
F P
0 g
s
.40966 .3761 m .45429 .40741 L .44321 .39393 L p .228 g
F P
0 g
s
.44321 .39393 m .43912 .38999 L p .39016 .35524 L .171 g
F P
0 g
s
.38682 .35287 m .38993 .35514 L p .39016 .35524 L .171 g
F P
0 g
s
.38993 .35514 m .44321 .39393 L p .39016 .35524 L .171 g
F P
0 g
s
.38916 .34997 m .43661 .38798 L .41803 .37565 L .296 g
F
.38531 .34601 m .43912 .38999 L p .38916 .34997 L F P
0 g
s
.43912 .38999 m .4432 .39803 L p .41803 .37565 L .296 g
F P
0 g
s
.43661 .38798 m .43912 .38999 L .41803 .37565 L .296 g
F
.4432 .39803 m .3855 .34256 L p .38916 .34997 L F P
0 g
s
.45429 .40741 m .40966 .3761 L .44056 .40537 L p .148 g
F P
0 g
s
.44056 .40537 m .46951 .4282 L .45429 .40741 L p .148 g
F P
0 g
s
.44056 .40537 m .47192 .43829 L p .47143 .43625 L .053 g
F P
0 g
s
.47143 .43625 m .46419 .42458 L .44056 .40537 L .053 g
F
.46821 .42717 m .46323 .42325 L p .46972 .42907 L F P
0 g
s
.46323 .42325 m .44056 .40537 L p .46418 .42457 L .053 g
F P
0 g
s
.46972 .42907 m .46323 .42325 L .46418 .42457 L .053 g
F
.45739 .44554 m .47192 .43829 L .44056 .40537 L p .268 g
F P
0 g
s
.47192 .43829 m .45739 .44554 L .49482 .48018 L p .058 g
F P
0 g
s
.48194 .46627 m .43912 .38999 L .43997 .39081 L p .488 g
F P
0 g
s
.43997 .39081 m .44321 .39393 L p .44616 .39944 L .488 g
F P
0 g
s
.44616 .39944 m .48194 .46627 L .43997 .39081 L .488 g
F
.49387 .49946 m .49482 .48018 L .45739 .44554 L p .219 g
F P
0 g
s
.47192 .43829 m .485 .45468 L .46951 .4282 L p .053 g
F P
0 g
s
.485 .45468 m .47461 .44167 L p .49584 .47451 L .486 g
F P
0 g
s
.47523 .4433 m .49584 .47451 L p .47461 .44167 L .486 g
F P
0 g
s
.49482 .48018 m .49584 .47451 L .47192 .43829 L p .058 g
F P
0 g
s
.46951 .4282 m .485 .45468 L p .48971 .47972 L .522 g
F P
0 g
s
.49005 .48153 m .48923 .47939 L p .48971 .47972 L .522 g
F P
0 g
s
.48923 .47939 m .46951 .4282 L p .48971 .47972 L .522 g
F P
0 g
s
.49005 .48153 m .48594 .4708 L p .4711 .43231 L .029 g
F P
0 g
s
.47116 .43248 m .49005 .48153 L p .4711 .43231 L .029 g
F P
0 g
s
.48297 .46548 m .48594 .4708 L .48682 .47776 L p .676 g
F P
0 g
s
.49314 .52726 m .49275 .55774 L .52 .55852 L p .053 g
F P
0 g
s
.49275 .55774 m .49281 .58483 L .51963 .5731 L p .037 g
F P
0 g
s
.51963 .5731 m .52 .55852 L .49275 .55774 L p .037 g
F P
0 g
s
.48682 .47776 m .49143 .51412 L p .48459 .47625 L .676 g
F P
0 g
s
.48954 .50911 m .48459 .47625 L p .49143 .51412 L .676 g
F P
0 g
s
.48531 .48703 m .48194 .46627 L p .48674 .49052 L .683 g
F P
0 g
s
.48311 .47369 m .48194 .46627 L .48531 .48703 L p .498 g
F P
0 g
s
.48531 .48703 m .4883 .50546 L p .48514 .48612 L .498 g
F P
0 g
s
.48805 .50405 m .48514 .48612 L p .4883 .50546 L .498 g
F P
0 g
s
.48514 .48612 m .48311 .47369 L p .48531 .48703 L .498 g
F P
0 g
s
.48674 .49052 m .48954 .50911 L p .48531 .48703 L .683 g
F P
0 g
s
.48954 .50911 m .49283 .53097 L p .4883 .50546 L .683 g
F P
0 g
s
.49194 .52783 m .4883 .50546 L p .49283 .53097 L .683 g
F P
0 g
s
.4883 .50546 m .48531 .48703 L p .48954 .50911 L .683 g
F P
0 g
s
.4883 .50546 m .49194 .52783 L p .48805 .50405 L .498 g
F P
0 g
s
.49194 .52783 m .49761 .56276 L .49172 .52659 L p .498 g
F P
0 g
s
.49172 .52659 m .48805 .50405 L p .49194 .52783 L .498 g
F P
0 g
s
.48805 .50405 m .49172 .52659 L p .48884 .50515 L .047 g
F P
0 g
s
.49225 .5276 m .48884 .50515 L p .49172 .52659 L .047 g
F P
0 g
s
.4862 .48784 m .48311 .47369 L .48805 .50405 L p .047 g
F P
0 g
s
.48884 .50515 m .4862 .48784 L p .48805 .50405 L .047 g
F P
0 g
s
.48311 .47369 m .4862 .48784 L p .45403 .42065 L .576 g
F P
0 g
s
.48232 .47974 m .48498 .48528 L .48157 .4779 L .576 g
F
.4862 .48784 m .48495 .48517 L p .48498 .48528 L F P
0 g
s
.48495 .48517 m .48157 .4779 L p .48498 .48528 L .576 g
F P
0 g
s
.4432 .39803 m .45692 .42405 L p .45403 .42065 L .576 g
F P
0 g
s
.45692 .42405 m .48311 .47369 L p .45403 .42065 L .576 g
F P
0 g
s
.45452 .41988 m .4432 .39803 L p .48232 .47974 L .576 g
F P
0 g
s
.48157 .4779 m .45452 .41988 L p .48232 .47974 L .576 g
F P
0 g
s
.48194 .46627 m .48297 .46548 L .48674 .49052 L p .683 g
F P
0 g
s
.49143 .51412 m .49761 .56276 L .48954 .50911 L p .676 g
F P
0 g
s
.49039 .50827 m .4862 .48784 L .49225 .5276 L p .186 g
F P
0 g
s
.49283 .53097 m .49761 .56276 L .49194 .52783 L p .683 g
F P
0 g
s
.49225 .5276 m .49761 .56276 L .49331 .53033 L p .186 g
F P
0 g
s
.49331 .53033 m .49039 .50827 L p .49225 .5276 L .186 g
F P
0 g
s
.4862 .48784 m .49039 .50827 L p .4683 .45383 L .61 g
F P
0 g
s
.45452 .41988 m .47492 .46365 L p .4683 .45383 L .61 g
F P
0 g
s
.47492 .46365 m .4862 .48784 L p .4683 .45383 L .61 g
F P
0 g
s
.47365 .46702 m .48356 .49143 L .47242 .46309 L .61 g
F
.47365 .46702 m .47242 .46309 L .46424 .44226 L F
.47299 .46538 m .47365 .46702 L .46424 .44226 L F
.48356 .49143 m .49039 .50827 L .48262 .48865 L F
.48356 .49143 m .48262 .48865 L .46424 .44226 L F
.48752 .50101 m .48956 .50616 L .48746 .50067 L F
.49039 .50827 m .4896 .50623 L p .48956 .50616 L F P
0 g
s
.4896 .50623 m .48746 .50067 L p .48956 .50616 L .61 g
F P
0 g
s
.47021 .45601 m .46424 .44226 L p .48752 .50101 L .61 g
F P
0 g
s
.48746 .50067 m .47021 .45601 L p .48752 .50101 L .61 g
F P
0 g
s
.49387 .49946 m .49314 .52726 L .51825 .54219 L p .051 g
F P
0 g
s
.52 .55852 m .51825 .54219 L .49314 .52726 L p .053 g
F P
0 g
s
.49005 .48153 m .49761 .56276 L .48594 .4708 L closepath p .628 g
F P
0 g
s
.49039 .50827 m .49761 .56276 L .49458 .53385 L closepath p .331 g
F P
0 g
s
.49172 .51641 m .49458 .53385 L p .48447 .50156 L .648 g
F P
0 g
s
.47021 .45601 m .49172 .51641 L .48447 .50156 L .648 g
F
.48466 .50217 m .49414 .53245 L .48113 .48972 L F
.49414 .53245 m .49419 .53259 L .48113 .48972 L F
.48603 .50487 m .48113 .48972 L p .49089 .52175 L F P
0 g
s
.48113 .48972 m .47021 .45601 L p .48466 .50217 L .648 g
F P
0 g
s
.48603 .50487 m .47343 .47919 L .44841 .42779 L p .634 g
F P
0 g
s
.46424 .44226 m .45452 .41988 L p .47299 .46538 L .61 g
F P
0 g
s
.4432 .39803 m .45452 .41988 L p .40752 .36723 L .478 g
F P
0 g
s
.3855 .34256 m .4197 .37544 L p .40752 .36723 L .478 g
F P
0 g
s
.4197 .37544 m .4432 .39803 L p .40752 .36723 L .478 g
F P
0 g
s
.4035 .36134 m .45452 .41988 L .40456 .36182 L .478 g
F
.40141 .36039 m .42933 .39166 L .4035 .36134 L F
.42933 .39166 m .45452 .41988 L .4035 .36134 L F
.41017 .36835 m .43439 .39648 L .40828 .36559 L F
.40828 .36559 m .40456 .36182 L p .41017 .36835 L F P
0 g
s
.45452 .41988 m .43607 .39821 L p .43439 .39648 L .478 g
F P
0 g
s
.43607 .39821 m .40828 .36559 L p .43439 .39648 L .478 g
F P
0 g
s
.45756 .42688 m .47021 .45601 L p .43707 .40763 L .571 g
F P
0 g
s
.40828 .36559 m .45756 .42688 L .43707 .40763 L .571 g
F
.44052 .41158 m .42102 .38376 L p .43796 .40865 L F P
0 g
s
.47021 .45601 m .44052 .41158 L p .43796 .40865 L .571 g
F P
0 g
s
.42102 .38376 m .40828 .36559 L p .44067 .41289 L .571 g
F P
0 g
s
.44067 .41289 m .47021 .45601 L .42102 .38376 L .571 g
F
.35434 .34741 m .38682 .35287 L p .34749 .34125 L .566 g
F P
0 g
s
.44052 .41158 m .40616 .37964 L .37478 .3491 L p .51 g
F P
0 g
s
.37787 .33993 m .38531 .34601 L p .38916 .34997 L .296 g
F P
0 g
s
.3855 .34256 m .37787 .33993 L p .38916 .34997 L .296 g
F P
0 g
s
.37787 .33993 m .3855 .34256 L p .34283 .32898 L .067 g
F P
0 g
s
.30872 .31812 m .33788 .32732 L p .34283 .32898 L .067 g
F P
0 g
s
.33788 .32732 m .37787 .33993 L p .34283 .32898 L .067 g
F P
0 g
s
.32891 .32454 m .3855 .34256 L .31635 .31326 L .067 g
F
.40456 .36182 m .3855 .34256 L p .40141 .36039 L .478 g
F P
0 g
s
.3855 .34256 m .40828 .36559 L p .36739 .34049 L .304 g
F P
0 g
s
.31964 .31116 m .3551 .32807 L p .35927 .33228 L .304 g
F P
0 g
s
.31964 .31116 m .35927 .33228 L .34961 .32957 L .304 g
F
.34961 .32957 m .35927 .33228 L .36739 .34049 L F
.35502 .32988 m .34153 .32275 L p .36043 .33488 L F P
0 g
s
.3551 .32807 m .3855 .34256 L p .36739 .34049 L .304 g
F P
0 g
s
.34153 .32275 m .32584 .31445 L p .36824 .34075 L .304 g
F P
0 g
s
.36824 .34075 m .40828 .36559 L .34153 .32275 L .304 g
F
.33643 .32147 m .40828 .36559 L .32584 .31445 L F
.31635 .31326 m .30872 .31812 L p .32891 .32454 L .067 g
F P
0 g
s
.31964 .31116 m .25615 .30497 L .24271 .32327 L p .642 g
F P
0 g
s
.30872 .31812 m .31964 .31116 L p .29222 .31548 L .642 g
F P
0 g
s
.24271 .32327 m .30055 .31876 L p .29222 .31548 L .642 g
F P
0 g
s
.30055 .31876 m .30872 .31812 L p .29222 .31548 L .642 g
F P
0 g
s
.32584 .31445 m .31964 .31116 L p .33643 .32147 L .304 g
F P
0 g
s
.31964 .31116 m .35502 .32988 L p .28534 .31232 L .153 g
F P
0 g
s
.25615 .30497 m .29702 .30896 L p .28534 .31232 L .153 g
F P
0 g
s
.29702 .30896 m .31964 .31116 L p .28534 .31232 L .153 g
F P
0 g
s
.28577 .31163 m .27794 .30987 L p .31328 .31936 L .153 g
F P
0 g
s
.2821 .31151 m .31328 .31936 L .27794 .30987 L .153 g
F
.35502 .32988 m .30335 .31559 L .28577 .31163 L p F P
0 g
s
.31328 .31936 m .35502 .32988 L .28577 .31163 L .153 g
F
.30335 .31559 m .26218 .32327 L .24803 .32307 L p .121 g
F P
0 g
s
.19046 .35142 m .24271 .32327 L .25615 .30497 L p .839 g
F P
0 g
s
.25615 .30497 m .20562 .32246 L .19046 .35142 L p .839 g
F P
0 g
s
.24803 .32307 m .20562 .32246 L p .26453 .31832 L .121 g
F P
0 g
s
.26453 .31832 m .30335 .31559 L .24803 .32307 L .121 g
F
.27794 .30987 m .25615 .30497 L p .2821 .31151 L .153 g
F P
0 g
s
.37478 .3491 m .35502 .32988 L p .38961 .36293 L .51 g
F P
0 g
s
.38961 .36293 m .39626 .36929 L .39111 .36445 L .51 g
F
.38961 .36293 m .39111 .36445 L .37478 .3491 L F
.40828 .36559 m .36452 .33625 L p .36043 .33488 L .304 g
F P
0 g
s
.36452 .33625 m .35502 .32988 L p .36043 .33488 L .304 g
F P
0 g
s
.39626 .36929 m .44052 .41158 L .37478 .3491 L .51 g
F
.40828 .36559 m .45452 .41988 L .45756 .42688 L p .571 g
F P
0 g
s
.44841 .42779 m .44052 .41158 L p .45877 .44899 L .634 g
F P
0 g
s
.45877 .44899 m .48603 .50487 L .44841 .42779 L .634 g
F
.47021 .45601 m .49039 .50827 L .49172 .51641 L p .648 g
F P
0 g
s
.49414 .53245 m .49458 .53385 L .49419 .53259 L .648 g
F
.49089 .52175 m .49324 .52946 L .49076 .52099 L F
.49089 .52175 m .49076 .52099 L .48603 .50487 L F
.49172 .52659 m .49761 .56276 L .49225 .5276 L p .047 g
F P
0 g
s
.49458 .53385 m .49334 .52965 L p .49324 .52946 L .648 g
F P
0 g
s
.49334 .52965 m .48603 .50487 L p .49324 .52946 L .648 g
F P
0 g
s
.48459 .47625 m .48297 .46548 L p .48682 .47776 L .676 g
F P
0 g
s
.46951 .4282 m .47116 .43248 L p .4711 .43231 L .029 g
F P
0 g
s
.47461 .44167 m .47192 .43829 L .47523 .4433 L p .486 g
F P
0 g
s
.48594 .4708 m .48297 .46548 L p .45722 .41333 L .384 g
F P
0 g
s
.45429 .40741 m .45737 .41359 L p .45722 .41333 L .384 g
F P
0 g
s
.45737 .41359 m .48594 .4708 L p .45722 .41333 L .384 g
F P
0 g
s
.47143 .43625 m .46972 .42907 L .46419 .42458 L .053 g
F
.46951 .4282 m .46821 .42717 L p .46972 .42907 L F P
0 g
s
.44321 .39393 m .45429 .40741 L p .45722 .41333 L .384 g
F P
0 g
s
.48297 .46548 m .44649 .39983 L p .45722 .41333 L .384 g
F P
0 g
s
.44649 .39983 m .44321 .39393 L p .45722 .41333 L .384 g
F P
0 g
s
.32383 .34228 m .35434 .34741 L p .34749 .34125 L .566 g
F P
0 g
s
.42842 .58725 m .43611 .61292 L .49337 .6028 L p .151 g
F P
0 g
s
.49337 .6028 m .49281 .58483 L .42842 .58725 L p .151 g
F P
0 g
s
.16152 .39636 m .16352 .44999 L .19332 .49902 L p .748 g
F P
0 g
s
.19332 .49902 m .22653 .54654 L .2848 .58396 L p .423 g
F P
0 g
s
.22653 .54654 m .19332 .49902 L .16352 .44999 L p .052 g
F P
0 g
s
.16152 .39636 m .19046 .35142 L .20562 .32246 L p .751 g
F P
0 g
s
.39329 .62007 m .36061 .60714 L .28977 .58547 L p .087 g
F P
0 g
s
.39329 .62007 m .28977 .58547 L .29211 .58639 L .087 g
F
.2848 .58396 m .3356 .60522 L .39329 .62007 L p F P
0 g
s
.22653 .54654 m .29198 .57805 L .3276 .60023 L p .309 g
F P
0 g
s
.3276 .60023 m .3356 .60522 L p .323 .59844 L .309 g
F P
0 g
s
.323 .59844 m .22653 .54654 L .3276 .60023 L .309 g
F
.28977 .58547 m .2848 .58396 L p .29211 .58639 L .087 g
F P
0 g
s
.36061 .60714 m .39329 .62007 L .45094 .62014 L p .075 g
F P
0 g
s
.43611 .61292 m .36061 .60714 L p .43711 .61815 L .075 g
F P
0 g
s
.44037 .60623 m .47209 .60399 L p .40454 .61777 L .689 g
F P
0 g
s
.39329 .62007 m .40124 .61774 L p .40454 .61777 L .689 g
F P
0 g
s
.40124 .61774 m .44037 .60623 L p .40454 .61777 L .689 g
F P
0 g
s
.45094 .62014 m .44709 .61827 L p .43711 .61815 L .075 g
F P
0 g
s
.44709 .61827 m .43611 .61292 L p .43711 .61815 L .075 g
F P
0 g
s
.3356 .60522 m .2848 .58396 L .22653 .54654 L p .309 g
F P
0 g
s
.20562 .32246 m .17774 .35957 L .16152 .39636 L p .751 g
F P
0 g
s
.16352 .44999 m .16152 .39636 L .17774 .35957 L p .032 g
F P
0 g
s
.43611 .61292 m .45094 .62014 L .49441 .6068 L p .005 g
F P
0 g
s
.49337 .6028 m .43611 .61292 L p .4813 .60818 L .005 g
F P
0 g
s
.49281 .58483 m .49337 .6028 L .51694 .58278 L p F P
s
.47209 .60399 m .49587 .59368 L p .46149 .61393 L .759 g
F P
0 g
s
.45094 .62014 m .45759 .61506 L p .46149 .61393 L .759 g
F P
0 g
s
.45759 .61506 m .47209 .60399 L p .46149 .61393 L .759 g
F P
0 g
s
.49441 .6068 m .49382 .60454 L p .4813 .60818 L .005 g
F P
0 g
s
.49382 .60454 m .49337 .6028 L p .4813 .60818 L .005 g
F P
0 g
s
.49482 .48018 m .49387 .49946 L .51482 .5275 L p .385 g
F P
0 g
s
.51171 .52334 m .49387 .49946 L p .51738 .54066 L .051 g
F P
0 g
s
.51482 .5275 m .51036 .51767 L .50553 .50601 L p .385 g
F P
0 g
s
.50553 .50601 m .49482 .48018 L p .50843 .51239 L .385 g
F P
0 g
s
.51482 .5275 m .50553 .50601 L .50843 .51239 L .385 g
F
.49584 .47451 m .49482 .48018 L .50553 .50601 L p .543 g
F P
0 g
s
.50553 .50601 m .51036 .51767 L p .49584 .47451 L .543 g
F P
0 g
s
.49675 .48605 m .49588 .47497 L p .50097 .496 L .647 g
F P
0 g
s
.50564 .51533 m .49738 .48813 L .50097 .496 L .647 g
F
.49738 .48813 m .49675 .48605 L .50097 .496 L F
.49588 .47497 m .49584 .47451 L .49616 .47583 L p F P
0 g
s
.49616 .47583 m .50564 .51533 L p .49588 .47497 L .647 g
F P
0 g
s
.51825 .54219 m .51482 .5275 L .51171 .52334 L p .051 g
F P
0 g
s
.51825 .54219 m .51171 .52334 L .51738 .54066 L .051 g
F
.51694 .58278 m .51963 .5731 L .49281 .58483 L p 0 g
F P
s
.58615 .44973 m .61707 .45876 L .68553 .38969 L p .648 g
F P
0 g
s
.68553 .38969 m .63675 .3844 L .58615 .44973 L p .648 g
F P
0 g
s
.72379 .41217 m .68553 .38969 L .61707 .45876 L p .609 g
F P
0 g
s
.63675 .3844 m .68553 .38969 L .73814 .31368 L p .642 g
F P
0 g
s
.68553 .38969 m .72379 .41217 L .78825 .33291 L p .584 g
F P
0 g
s
.78825 .33291 m .73814 .31368 L .68553 .38969 L p .584 g
F P
0 g
s
.16352 .44999 m .20099 .50277 L .22653 .54654 L p .052 g
F P
0 g
s
.29198 .57805 m .22653 .54654 L .20099 .50277 L p .401 g
F P
0 g
s
.61707 .45876 m .64107 .47904 L .72379 .41217 L p .609 g
F P
0 g
s
.74443 .44676 m .72379 .41217 L .64107 .47904 L p .554 g
F P
0 g
s
.72379 .41217 m .74443 .44676 L .81586 .37082 L p .492 g
F P
0 g
s
.81586 .37082 m .78825 .33291 L .72379 .41217 L p .492 g
F P
0 g
s
.73814 .31368 m .67508 .31758 L .63675 .3844 L p .642 g
F P
0 g
s
.67508 .31758 m .73814 .31368 L .7619 .23976 L p .657 g
F P
0 g
s
.73814 .31368 m .78825 .33291 L .81851 .25129 L p .575 g
F P
0 g
s
.81851 .25129 m .7619 .23976 L .73814 .31368 L p .575 g
F P
0 g
s
.58557 .39951 m .63675 .3844 L .67508 .31758 L p .678 g
F P
0 g
s
.67508 .31758 m .60952 .34654 L .58557 .39951 L p .678 g
F P
0 g
s
.60952 .34654 m .67508 .31758 L .6916 .25668 L p .703 g
F P
0 g
s
.7619 .23976 m .6916 .25668 L .67508 .31758 L p .657 g
F P
0 g
s
.55355 .45522 m .58615 .44973 L .63675 .3844 L p .672 g
F P
0 g
s
.63675 .3844 m .58557 .39951 L .55355 .45522 L p .672 g
F P
0 g
s
.17774 .35957 m .18018 .40978 L .16352 .44999 L p .032 g
F P
0 g
s
.20099 .50277 m .16352 .44999 L .18018 .40978 L p .247 g
F P
0 g
s
.20562 .32246 m .25615 .30497 L .30335 .31559 L p .121 g
F P
0 g
s
.78825 .33291 m .81586 .37082 L .85023 .28854 L p .41 g
F P
0 g
s
.85023 .28854 m .81851 .25129 L .78825 .33291 L p .41 g
F P
0 g
s
.64107 .47904 m .65369 .50605 L .74443 .44676 L p .554 g
F P
0 g
s
.74201 .48697 m .74443 .44676 L .65369 .50605 L p .466 g
F P
0 g
s
.74443 .44676 m .74201 .48697 L .81345 .42069 L p .313 g
F P
0 g
s
.81345 .42069 m .81586 .37082 L .74443 .44676 L p .313 g
F P
0 g
s
.54878 .51647 m .55905 .52903 L .64107 .47904 L p .645 g
F P
0 g
s
.64107 .47904 m .61707 .45876 L .54878 .51647 L p .645 g
F P
0 g
s
.65369 .50605 m .64107 .47904 L .55905 .52903 L p .616 g
F P
0 g
s
.44037 .60623 m .39329 .62007 L .3356 .60522 L p .647 g
F P
0 g
s
.47209 .60399 m .45094 .62014 L .39329 .62007 L p .689 g
F P
0 g
s
.53547 .50947 m .54878 .51647 L .61707 .45876 L p .666 g
F P
0 g
s
.61707 .45876 m .58615 .44973 L .53547 .50947 L p .666 g
F P
0 g
s
.6916 .25668 m .61938 .30174 L .60952 .34654 L p .703 g
F P
0 g
s
.61938 .30174 m .6916 .25668 L .68149 .21172 L p .752 g
F P
0 g
s
.6916 .25668 m .7619 .23976 L .74951 .18096 L p .705 g
F P
0 g
s
.74951 .18096 m .68149 .21172 L .6916 .25668 L p .705 g
F P
0 g
s
.17774 .35957 m .20562 .32246 L .26218 .32327 L p .161 g
F P
0 g
s
.48594 .4708 m .45429 .40741 L .46951 .4282 L p .029 g
F P
0 g
s
.51204 .58504 m .51694 .58278 L .51157 .58734 L p .731 g
F P
0 g
s
.51157 .58734 m .49337 .6028 L p .50601 .59077 L .731 g
F P
0 g
s
.51204 .58504 m .51157 .58734 L .50601 .59077 L .731 g
F
.49587 .59368 m .50536 .57839 L p .4989 .59517 L .845 g
F P
0 g
s
.49337 .6028 m .49441 .6068 L .51204 .58504 L p .731 g
F P
0 g
s
.49441 .6068 m .49547 .59729 L p .4989 .59517 L .845 g
F P
0 g
s
.49547 .59729 m .49587 .59368 L p .4989 .59517 L .845 g
F P
0 g
s
.54037 .43597 m .58557 .39951 L .60952 .34654 L p .691 g
F P
0 g
s
.60952 .34654 m .55197 .39992 L .54037 .43597 L p .691 g
F P
0 g
s
.55197 .39992 m .60952 .34654 L .61938 .30174 L p .721 g
F P
0 g
s
.55905 .52903 m .5643 .54435 L .65369 .50605 L p .616 g
F P
0 g
s
.65168 .53449 m .65369 .50605 L .5643 .54435 L p .574 g
F P
0 g
s
.65369 .50605 m .65168 .53449 L .74201 .48697 L p .466 g
F P
0 g
s
.44321 .39393 m .48297 .46548 L .48194 .46627 L p .488 g
F P
0 g
s
.40883 .59947 m .3356 .60522 L .29198 .57805 L p .625 g
F P
0 g
s
.3356 .60522 m .40883 .59947 L .44037 .60623 L p .647 g
F P
0 g
s
.7619 .23976 m .81851 .25129 L .8049 .18248 L p .597 g
F P
0 g
s
.8049 .18248 m .74951 .18096 L .7619 .23976 L p .597 g
F P
0 g
s
.61938 .30174 m .55658 .37191 L .55197 .39992 L p .721 g
F P
0 g
s
.55658 .37191 m .61938 .30174 L .61241 .27174 L p .772 g
F P
0 g
s
.68149 .21172 m .61241 .27174 L .61938 .30174 L p .752 g
F P
0 g
s
.49587 .59368 m .49441 .6068 L .45094 .62014 L p .759 g
F P
0 g
s
.52474 .47675 m .55355 .45522 L .58557 .39951 L p .68 g
F P
0 g
s
.58557 .39951 m .54037 .43597 L .52474 .47675 L p .68 g
F P
0 g
s
.81586 .37082 m .81345 .42069 L .84803 .34546 L p .04 g
F P
0 g
s
.84803 .34546 m .85023 .28854 L .81586 .37082 L p .04 g
F P
0 g
s
.52143 .51016 m .53547 .50947 L .58615 .44973 L p .679 g
F P
0 g
s
.58615 .44973 m .55355 .45522 L .52143 .51016 L p .679 g
F P
0 g
s
.71363 .52542 m .74201 .48697 L .65168 .53449 L p .287 g
F P
0 g
s
.74201 .48697 m .71363 .52542 L .77689 .4739 L p .017 g
F P
0 g
s
.77689 .4739 m .81345 .42069 L .74201 .48697 L p .017 g
F P
0 g
s
.49584 .47451 m .49675 .48605 L .485 .45468 L p .486 g
F P
0 g
s
.51036 .51767 m .50564 .51533 L .49584 .47451 L p .543 g
F P
0 g
s
.48311 .47369 m .4432 .39803 L .43912 .38999 L p .545 g
F P
0 g
s
.43912 .38999 m .48194 .46627 L .48311 .47369 L p .545 g
F P
0 g
s
.26218 .32327 m .23974 .35094 L .17774 .35957 L p .161 g
F P
0 g
s
.18018 .40978 m .17774 .35957 L .23974 .35094 L p .301 g
F P
0 g
s
.5643 .54435 m .56318 .55928 L .65168 .53449 L p .574 g
F P
0 g
s
.63343 .55881 m .65168 .53449 L .56318 .55928 L p .504 g
F P
0 g
s
.65168 .53449 m .63343 .55881 L .71363 .52542 L p .287 g
F P
0 g
s
.61241 .27174 m .55296 .35553 L .55658 .37191 L p .772 g
F P
0 g
s
.55296 .35553 m .61241 .27174 L .59005 .26307 L p .828 g
F P
0 g
s
.61241 .27174 m .68149 .21172 L .64637 .19323 L p .818 g
F P
0 g
s
.64637 .19323 m .59005 .26307 L .61241 .27174 L p .818 g
F P
0 g
s
.68149 .21172 m .74951 .18096 L .7023 .15182 L p .792 g
F P
0 g
s
.7023 .15182 m .64637 .19323 L .68149 .21172 L p .792 g
F P
0 g
s
.81851 .25129 m .85023 .28854 L .83628 .21594 L p .227 g
F P
0 g
s
.83628 .21594 m .8049 .18248 L .81851 .25129 L p .227 g
F P
0 g
s
.18018 .40978 m .21708 .46449 L .20099 .50277 L p .247 g
F P
0 g
s
.27396 .54466 m .20099 .50277 L .21708 .46449 L p .483 g
F P
0 g
s
.20099 .50277 m .27396 .54466 L .29198 .57805 L p .401 g
F P
0 g
s
.485 .45468 m .49423 .49687 L .49005 .48153 L p .522 g
F P
0 g
s
.49423 .49687 m .485 .45468 L .49675 .48605 L p .657 g
F P
0 g
s
.38554 .58511 m .29198 .57805 L .27396 .54466 L p .617 g
F P
0 g
s
.29198 .57805 m .38554 .58511 L .40883 .59947 L p .625 g
F P
0 g
s
.5121 .47417 m .55197 .39992 L .55658 .37191 L p .695 g
F P
0 g
s
.55658 .37191 m .51331 .46163 L .5121 .47417 L p .695 g
F P
0 g
s
.51331 .46163 m .55658 .37191 L .55296 .35553 L p .76 g
F P
0 g
s
.49761 .56276 m .51482 .5275 L .51825 .54219 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .51825 .54219 L .52 .55852 L closepath p .751 g
F P
0 g
s
.33135 .70532 m .27314 .7917 L .35394 .79274 L p .705 g
F P
0 g
s
.35394 .79274 m .39085 .70959 L .33135 .70532 L p .705 g
F P
0 g
s
.33193 .86963 m .35394 .79274 L .27314 .7917 L p .672 g
F P
0 g
s
.39085 .70959 m .35394 .79274 L .43366 .76486 L p .651 g
F P
0 g
s
.35394 .79274 m .33193 .86963 L .424 .82938 L p .603 g
F P
0 g
s
.424 .82938 m .43366 .76486 L .35394 .79274 L p .603 g
F P
0 g
s
.49761 .56276 m .51036 .51767 L .51482 .5275 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .50564 .51533 L .51036 .51767 L closepath p .751 g
F P
0 g
s
.5091 .51963 m .52143 .51016 L .55355 .45522 L p .681 g
F P
0 g
s
.55355 .45522 m .52474 .47675 L .5091 .51963 L p .681 g
F P
0 g
s
.5121 .47417 m .51331 .46163 L .50993 .48342 L p .801 g
F P
0 g
s
.50899 .49199 m .51023 .48977 L p .51021 .48937 L .665 g
F P
0 g
s
.55197 .39992 m .5121 .47417 L .50899 .49199 L p .665 g
F P
0 g
s
.52474 .47675 m .54037 .43597 L p .50761 .5075 L .66 g
F P
0 g
s
.50149 .52211 m .49675 .48605 L p .50412 .51032 L .647 g
F P
0 g
s
.50564 .51533 m .50326 .51921 L p .50412 .51032 L .647 g
F P
0 g
s
.50326 .51921 m .50149 .52211 L p .50412 .51032 L .647 g
F P
0 g
s
.50761 .5075 m .50568 .51172 L .50772 .50824 L .66 g
F
.5048 .51364 m .50559 .51218 L p .50568 .51172 L F P
0 g
s
.50559 .51218 m .50772 .50824 L p .50568 .51172 L .66 g
F P
0 g
s
.54037 .43597 m .50899 .49199 L .5048 .51364 L p .66 g
F P
0 g
s
.50899 .49199 m .5121 .47417 L .50716 .50437 L p F P
s
.49761 .56276 m .50149 .52211 L .50564 .51533 L closepath p .751 g
F P
0 g
s
.52474 .47675 m .5048 .51364 L .50063 .53768 L p .67 g
F P
0 g
s
.50716 .50437 m .50349 .52678 L p .50646 .50784 L F P
s
.50349 .52678 m .5032 .52829 L .50646 .50784 L F
.50716 .50437 m .50646 .50784 L .50899 .49199 L F
.50419 .52871 m .50206 .53409 L .50439 .52929 L .67 g
F
.50187 .53505 m .5091 .51963 L p .50206 .53409 L F P
0 g
s
.5091 .51963 m .50063 .53768 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.50126 .54008 m .50899 .49199 L p .50117 .541 L F P
s
.5048 .51364 m .50899 .49199 L .50296 .52951 L p .031 g
F P
0 g
s
.50206 .53409 m .50107 .53657 L .50187 .53505 L .67 g
F
.50097 .53698 m .50187 .53505 L p .50107 .53657 L F P
0 g
s
.50296 .52951 m .49761 .56276 L .5019 .53349 L p .031 g
F P
0 g
s
.5019 .53349 m .5048 .51364 L p .50296 .52951 L .031 g
F P
0 g
s
.50063 .53768 m .50097 .53698 L p .50107 .53657 L .67 g
F P
0 g
s
.50692 .50275 m .50328 .5262 L p .50716 .50437 L .801 g
F P
0 g
s
.50349 .52678 m .50117 .541 L p .5032 .52829 L F P
s
.50117 .541 m .49761 .56276 L .50126 .54008 L p F P
s
.50328 .5262 m .49761 .56276 L .50349 .52678 L p .801 g
F P
0 g
s
.50349 .52678 m .50716 .50437 L p .50328 .5262 L .801 g
F P
0 g
s
.5091 .51963 m .52474 .47675 L p .50419 .52871 L .67 g
F P
0 g
s
.50419 .52871 m .50439 .52929 L .5091 .51963 L .67 g
F
.50772 .50824 m .52474 .47675 L p .50761 .5075 L .66 g
F P
0 g
s
.50993 .48342 m .50692 .50275 L p .51008 .48653 L .801 g
F P
0 g
s
.50716 .50437 m .51008 .48653 L p .50692 .50275 L .801 g
F P
0 g
s
.51008 .48653 m .5121 .47417 L p .50993 .48342 L .801 g
F P
0 g
s
.54037 .43597 m .55197 .39992 L p .51021 .48937 L .665 g
F P
0 g
s
.51023 .48977 m .54037 .43597 L p .51021 .48937 L .665 g
F P
0 g
s
.49761 .56276 m .52 .55852 L .51963 .5731 L closepath p .751 g
F P
0 g
s
.43366 .76486 m .45005 .69338 L .39085 .70959 L p .651 g
F P
0 g
s
.45005 .69338 m .43366 .76486 L .49325 .7111 L p .592 g
F P
0 g
s
.43366 .76486 m .424 .82938 L .4926 .7576 L p .519 g
F P
0 g
s
.4926 .7576 m .49325 .7111 L .43366 .76486 L p .519 g
F P
0 g
s
.35502 .32988 m .40828 .36559 L .44052 .41158 L p .51 g
F P
0 g
s
.30335 .31559 m .35502 .32988 L .40616 .37964 L p .456 g
F P
0 g
s
.81345 .42069 m .77689 .4739 L .80703 .41275 L p .848 g
F P
0 g
s
.80703 .41275 m .84803 .34546 L .81345 .42069 L p .848 g
F P
0 g
s
.55296 .35553 m .51231 .45562 L .51331 .46163 L p .76 g
F P
0 g
s
.51231 .45562 m .55296 .35553 L .54199 .35392 L p .836 g
F P
0 g
s
.59005 .26307 m .54199 .35392 L .55296 .35553 L p .828 g
F P
0 g
s
.50536 .57839 m .51204 .58504 L .49441 .6068 L p .845 g
F P
0 g
s
.42165 .87873 m .424 .82938 L .33193 .86963 L p .52 g
F P
0 g
s
.424 .82938 m .42165 .87873 L .49245 .79232 L p .389 g
F P
0 g
s
.49245 .79232 m .4926 .7576 L .424 .82938 L p .389 g
F P
0 g
s
.54199 .35392 m .59005 .26307 L .55793 .28024 L p .863 g
F P
0 g
s
.59005 .26307 m .64637 .19323 L .59496 .2092 L p .874 g
F P
0 g
s
.59496 .2092 m .55793 .28024 L .59005 .26307 L p .874 g
F P
0 g
s
.74951 .18096 m .8049 .18248 L .74815 .14418 L p .672 g
F P
0 g
s
.74815 .14418 m .7023 .15182 L .74951 .18096 L p .672 g
F P
0 g
s
.27314 .7917 m .23803 .87457 L .33193 .86963 L p .672 g
F P
0 g
s
.32622 .93039 m .33193 .86963 L .23803 .87457 L p .611 g
F P
0 g
s
.33193 .86963 m .32622 .93039 L .42165 .87873 L p .52 g
F P
0 g
s
.40616 .37964 m .37281 .36349 L .30335 .31559 L p .456 g
F P
0 g
s
.26218 .32327 m .30335 .31559 L .37281 .36349 L p .432 g
F P
0 g
s
.40821 .62588 m .33135 .70532 L .39085 .70959 L p .725 g
F P
0 g
s
.39085 .70959 m .44002 .62987 L .40821 .62588 L p .725 g
F P
0 g
s
.44002 .62987 m .39085 .70959 L .45005 .69338 L p .687 g
F P
0 g
s
.65979 .55469 m .71363 .52542 L .63343 .55881 L p .026 g
F P
0 g
s
.71363 .52542 m .65979 .55469 L .7069 .52087 L p .844 g
F P
0 g
s
.7069 .52087 m .77689 .4739 L .71363 .52542 L p .844 g
F P
0 g
s
.49761 .56276 m .51963 .5731 L .51694 .58278 L closepath p .751 g
F P
0 g
s
.56318 .55928 m .55513 .57071 L .63343 .55881 L p .504 g
F P
0 g
s
.59942 .5739 m .63343 .55881 L .55513 .57071 L p .333 g
F P
0 g
s
.63343 .55881 m .59942 .5739 L .65979 .55469 L p .026 g
F P
0 g
s
.55905 .52903 m .54878 .51647 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.5643 .54435 m .55905 .52903 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.45005 .69338 m .47196 .62331 L .44002 .62987 L p .687 g
F P
0 g
s
.47196 .62331 m .45005 .69338 L .49437 .6589 L p .647 g
F P
0 g
s
.49325 .7111 m .49437 .6589 L .45005 .69338 L p .592 g
F P
0 g
s
.23974 .35094 m .24239 .39391 L .18018 .40978 L p .301 g
F P
0 g
s
.21708 .46449 m .18018 .40978 L .24239 .39391 L p .429 g
F P
0 g
s
.54199 .35392 m .50935 .45706 L .51231 .45562 L p .836 g
F P
0 g
s
.50935 .45706 m .54199 .35392 L .52647 .36895 L p .862 g
F P
0 g
s
.55793 .28024 m .52647 .36895 L .54199 .35392 L p .863 g
F P
0 g
s
.54878 .51647 m .53547 .50947 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.56318 .55928 m .5643 .54435 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.85023 .28854 m .84803 .34546 L .83429 .27665 L p .737 g
F P
0 g
s
.83429 .27665 m .83628 .21594 L .85023 .28854 L p .737 g
F P
0 g
s
.64637 .19323 m .7023 .15182 L .63182 .164 L p .888 g
F P
0 g
s
.63182 .164 m .59496 .2092 L .64637 .19323 L p .888 g
F P
0 g
s
.28632 .68235 m .21087 .76352 L .27314 .7917 L p .763 g
F P
0 g
s
.27314 .7917 m .33135 .70532 L .28632 .68235 L p .763 g
F P
0 g
s
.23803 .87457 m .27314 .7917 L .21087 .76352 L p .748 g
F P
0 g
s
.49675 .48605 m .49738 .51607 L .49423 .49687 L p .657 g
F P
0 g
s
.49738 .51607 m .49675 .48605 L .50149 .52211 L p .709 g
F P
0 g
s
.38473 .61253 m .28632 .68235 L .33135 .70532 L p .765 g
F P
0 g
s
.33135 .70532 m .40821 .62588 L .38473 .61253 L p .765 g
F P
0 g
s
.49437 .6589 m .49325 .7111 L .51723 .63948 L p .492 g
F P
0 g
s
.49325 .7111 m .4926 .7576 L .52015 .66368 L p .346 g
F P
0 g
s
.52015 .66368 m .51723 .63948 L .49325 .7111 L p .346 g
F P
0 g
s
.37281 .36349 m .34637 .36469 L .26218 .32327 L p .432 g
F P
0 g
s
.23974 .35094 m .26218 .32327 L .34637 .36469 L p .442 g
F P
0 g
s
.53547 .50947 m .52143 .51016 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.55513 .57071 m .56318 .55928 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.4926 .7576 m .49245 .79232 L .52073 .6815 L p .062 g
F P
0 g
s
.52073 .6815 m .52015 .66368 L .4926 .7576 L p .062 g
F P
0 g
s
.49761 .56276 m .51694 .58278 L .51204 .58504 L closepath p .751 g
F P
0 g
s
.52647 .36895 m .55793 .28024 L .52466 .32412 L p .861 g
F P
0 g
s
.55793 .28024 m .59496 .2092 L .54128 .26186 L p .89 g
F P
0 g
s
.54128 .26186 m .52466 .32412 L .55793 .28024 L p .89 g
F P
0 g
s
.42648 .90549 m .42165 .87873 L .32622 .93039 L p .325 g
F P
0 g
s
.42165 .87873 m .42648 .90549 L .4928 .81006 L p .092 g
F P
0 g
s
.4928 .81006 m .49245 .79232 L .47433 .81444 L p .092 g
F P
0 g
s
.4928 .81006 m .47433 .81444 L .46319 .83864 L .092 g
F
.49245 .79232 m .4928 .81006 L .51906 .69025 L p 0 g
F P
s
.51906 .69025 m .52073 .6815 L .50516 .74251 L p F P
s
.51906 .69025 m .50516 .74251 L .50469 .74538 L F
.51556 .68831 m .51906 .69025 L .49723 .78984 L p .891 g
F P
0 g
s
.49723 .78984 m .4928 .81006 L p .49608 .79253 L .891 g
F P
0 g
s
.51556 .68831 m .49723 .78984 L .49608 .79253 L .891 g
F
.44447 .8796 m .42648 .90549 L p .44469 .87884 L .75 g
F P
0 g
s
.4928 .81006 m .49356 .8073 L .50204 .76141 L p .891 g
F P
0 g
s
.50318 .75451 m .4928 .81006 L .50204 .76141 L .891 g
F
.49356 .8073 m .44447 .8796 L .44469 .87884 L .75 g
F
.49356 .8073 m .4928 .81006 L .44447 .8796 L p F P
0 g
s
.43749 .90407 m .47875 .83286 L p .44469 .87884 L .75 g
F P
0 g
s
.47875 .83286 m .49356 .8073 L p .48324 .82241 L .75 g
F P
0 g
s
.48324 .82241 m .44469 .87884 L .47875 .83286 L .75 g
F
.50204 .76141 m .51556 .68831 L p .50318 .75451 L .891 g
F P
0 g
s
.50516 .74251 m .49245 .79232 L p .50469 .74538 L F P
s
.42648 .90549 m .4328 .90467 L p .44469 .87884 L .75 g
F P
0 g
s
.4328 .90467 m .43749 .90407 L p .44469 .87884 L .75 g
F P
0 g
s
.47433 .81444 m .42165 .87873 L p .46319 .83864 L .092 g
F P
0 g
s
.52647 .36895 m .50522 .46628 L .50935 .45706 L p .862 g
F P
0 g
s
.50522 .46628 m .52647 .36895 L .51053 .40055 L p .846 g
F P
0 g
s
.52466 .32412 m .51053 .40055 L .52647 .36895 L p .861 g
F P
0 g
s
.49761 .56276 m .44037 .60623 L .40883 .59947 L closepath p .728 g
F P
0 g
s
.49761 .56276 m .47209 .60399 L .44037 .60623 L closepath p .738 g
F P
0 g
s
.21708 .46449 m .28761 .51398 L .27396 .54466 L p .483 g
F P
0 g
s
.37672 .56664 m .27396 .54466 L .28761 .51398 L p .616 g
F P
0 g
s
.27396 .54466 m .37672 .56664 L .38554 .58511 L p .617 g
F P
0 g
s
.49761 .56276 m .40883 .59947 L .38554 .58511 L closepath p .719 g
F P
0 g
s
.49437 .6589 m .49587 .60738 L .47196 .62331 L p .647 g
F P
0 g
s
.49587 .60738 m .49437 .6589 L .51215 .61223 L p .587 g
F P
0 g
s
.51723 .63948 m .51215 .61223 L .49437 .6589 L p .492 g
F P
0 g
s
.23803 .87457 m .22855 .94274 L .32622 .93039 L p .611 g
F P
0 g
s
.3367 .96562 m .32622 .93039 L .22855 .94274 L p .455 g
F P
0 g
s
.32622 .93039 m .3367 .96562 L .42648 .90549 L p .325 g
F P
0 g
s
.21087 .76352 m .16488 .84551 L .23803 .87457 L p .748 g
F P
0 g
s
.22855 .94274 m .23803 .87457 L .16488 .84551 L p .704 g
F P
0 g
s
.49761 .56276 m .49587 .59368 L .47209 .60399 L closepath p .748 g
F P
0 g
s
.49423 .49687 m .49761 .56276 L .49005 .48153 L closepath p .577 g
F P
0 g
s
.77689 .4739 m .7069 .52087 L .72846 .47885 L p .919 g
F P
0 g
s
.72846 .47885 m .80703 .41275 L .77689 .4739 L p .919 g
F P
0 g
s
.8049 .18248 m .83628 .21594 L .77417 .17219 L p .262 g
F P
0 g
s
.77417 .17219 m .74815 .14418 L .8049 .18248 L p .262 g
F P
0 g
s
.52143 .51016 m .5091 .51963 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.5405 .57605 m .55513 .57071 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.55513 .57071 m .5405 .57605 L .59942 .5739 L p .333 g
F P
0 g
s
.5048 .51364 m .49761 .56276 L .50063 .53768 L closepath p .17 g
F P
0 g
s
.51231 .45562 m .49761 .56276 L .51331 .46163 L closepath p .805 g
F P
0 g
s
.55245 .57584 m .59942 .5739 L .5405 .57605 L p .002 g
F P
0 g
s
.59942 .5739 m .55245 .57584 L .58498 .56844 L p .83 g
F P
0 g
s
.58498 .56844 m .65979 .55469 L .59942 .5739 L p .83 g
F P
0 g
s
.50935 .45706 m .49761 .56276 L .51231 .45562 L closepath p .743 g
F P
0 g
s
.44052 .41158 m .47021 .45601 L .48603 .50487 L p .634 g
F P
0 g
s
.40616 .37964 m .44052 .41158 L .47343 .47919 L p .613 g
F P
0 g
s
.44002 .62987 m .49761 .56276 L .40821 .62588 L closepath p .749 g
F P
0 g
s
.47196 .62331 m .49761 .56276 L .44002 .62987 L closepath p .74 g
F P
0 g
s
.65979 .55469 m .58498 .56844 L .60989 .55264 L p .868 g
F P
0 g
s
.60989 .55264 m .7069 .52087 L .65979 .55469 L p .868 g
F P
0 g
s
.51053 .40055 m .50101 .48292 L .50522 .46628 L p .846 g
F P
0 g
s
.50101 .48292 m .51053 .40055 L .49849 .44644 L p .815 g
F P
0 g
s
.51053 .40055 m .52466 .32412 L .49945 .39112 L p .833 g
F P
0 g
s
.49945 .39112 m .49849 .44644 L .51053 .40055 L p .833 g
F P
0 g
s
.47343 .47919 m .45885 .45982 L .40616 .37964 L p .613 g
F P
0 g
s
.37281 .36349 m .40616 .37964 L .45885 .45982 L p .593 g
F P
0 g
s
.50149 .52211 m .49863 .53833 L .49738 .51607 L p .709 g
F P
0 g
s
.49761 .56276 m .49863 .53833 L .50149 .52211 L closepath p .751 g
F P
0 g
s
.50522 .46628 m .49761 .56276 L .50935 .45706 L closepath p .683 g
F P
0 g
s
.49761 .56276 m .50536 .57839 L .49587 .59368 L closepath p .759 g
F P
0 g
s
.49587 .60738 m .49761 .56276 L .47196 .62331 L closepath p .731 g
F P
0 g
s
.59496 .2092 m .63182 .164 L .55776 .2213 L p .931 g
F P
0 g
s
.55776 .2213 m .54128 .26186 L .59496 .2092 L p .931 g
F P
0 g
s
.24239 .39391 m .27317 .44509 L .21708 .46449 L p .429 g
F P
0 g
s
.28761 .51398 m .21708 .46449 L .27317 .44509 L p .534 g
F P
0 g
s
.49738 .51607 m .49761 .56276 L .49423 .49687 L closepath p .526 g
F P
0 g
s
.52466 .32412 m .54128 .26186 L .50057 .3462 L p .866 g
F P
0 g
s
.50057 .3462 m .49945 .39112 L .52466 .32412 L p .866 g
F P
0 g
s
.7023 .15182 m .74815 .14418 L .66204 .15108 L p .72 g
F P
0 g
s
.66204 .15108 m .63182 .164 L .7023 .15182 L p .72 g
F P
0 g
s
.34637 .36469 m .33227 .38265 L .23974 .35094 L p .442 g
F P
0 g
s
.24239 .39391 m .23974 .35094 L .33227 .38265 L p .477 g
F P
0 g
s
.49761 .56276 m .38554 .58511 L .37672 .56664 L closepath p .713 g
F P
0 g
s
.40821 .62588 m .49761 .56276 L .38473 .61253 L closepath p .758 g
F P
0 g
s
.49761 .56276 m .51204 .58504 L .50536 .57839 L closepath p .751 g
F P
0 g
s
.84803 .34546 m .80703 .41275 L .79409 .35518 L p .88 g
F P
0 g
s
.79409 .35518 m .83429 .27665 L .84803 .34546 L p .88 g
F P
0 g
s
.50101 .48292 m .49761 .56276 L .50522 .46628 L closepath p .632 g
F P
0 g
s
.37587 .59325 m .26791 .64626 L .28632 .68235 L p .808 g
F P
0 g
s
.28632 .68235 m .38473 .61253 L .37587 .59325 L p .808 g
F P
0 g
s
.21087 .76352 m .28632 .68235 L .26791 .64626 L p .834 g
F P
0 g
s
.45885 .45982 m .44477 .44901 L .37281 .36349 L p .593 g
F P
0 g
s
.34637 .36469 m .37281 .36349 L .44477 .44901 L p .58 g
F P
0 g
s
.50538 .58552 m .49761 .56276 L .49587 .60738 L closepath p .724 g
F P
0 g
s
.51215 .61223 m .50538 .58552 L .49587 .60738 L p .587 g
F P
0 g
s
.5206 .57356 m .5405 .57605 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.5405 .57605 m .5206 .57356 L .55245 .57584 L p .002 g
F P
0 g
s
.49761 .56276 m .51215 .61223 L .51723 .63948 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .51723 .63948 L .52015 .66368 L closepath p .751 g
F P
0 g
s
.43749 .90407 m .42648 .90549 L .3367 .96562 L p F P
s
.26791 .64626 m .18405 .71478 L .21087 .76352 L p .834 g
F P
0 g
s
.16488 .84551 m .21087 .76352 L .18405 .71478 L p .848 g
F P
0 g
s
.49849 .44644 m .49784 .50582 L .50101 .48292 L p .815 g
F P
0 g
s
.49784 .50582 m .49849 .44644 L .49369 .50235 L p .775 g
F P
0 g
s
.49849 .44644 m .49945 .39112 L .48941 .47385 L p .79 g
F P
0 g
s
.48941 .47385 m .49369 .50235 L .49849 .44644 L p .79 g
F P
0 g
s
.49863 .53833 m .49761 .56276 L .49738 .51607 L closepath p .47 g
F P
0 g
s
.49784 .50582 m .49761 .56276 L .50101 .48292 L closepath p .584 g
F P
0 g
s
.49761 .56276 m .52015 .66368 L .52073 .6815 L closepath p .751 g
F P
0 g
s
.49945 .39112 m .50057 .3462 L .48445 .45115 L p .818 g
F P
0 g
s
.48445 .45115 m .48941 .47385 L .49945 .39112 L p .818 g
F P
0 g
s
.38473 .61253 m .49761 .56276 L .37587 .59325 L closepath p .766 g
F P
0 g
s
.49761 .56276 m .50538 .58552 L .51215 .61223 L closepath p .751 g
F P
0 g
s
.54128 .26186 m .55776 .2213 L .50167 .31727 L p .914 g
F P
0 g
s
.50167 .31727 m .50057 .3462 L .54128 .26186 L p .914 g
F P
0 g
s
.7069 .52087 m .60989 .55264 L .62057 .53197 L p .844 g
F P
0 g
s
.62057 .53197 m .72846 .47885 L .7069 .52087 L p .844 g
F P
0 g
s
.16488 .84551 m .15226 .91642 L .22855 .94274 L p .704 g
F P
0 g
s
.24477 .98518 m .22855 .94274 L .15226 .91642 L p .542 g
F P
0 g
s
.22855 .94274 m .24477 .98518 L .3367 .96562 L p .455 g
F P
0 g
s
.28761 .51398 m .38512 .54901 L .37672 .56664 L p .616 g
F P
0 g
s
.49761 .56276 m .37672 .56664 L .38512 .54901 L closepath p .708 g
F P
0 g
s
.49761 .56276 m .52073 .6815 L .51906 .69025 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .55245 .57584 L .5206 .57356 L closepath p .778 g
F P
0 g
s
.49761 .56276 m .58498 .56844 L .55245 .57584 L closepath p .779 g
F P
0 g
s
.52064 .5807 m .54081 .6033 L .60083 .62642 L p .663 g
F P
0 g
s
.60083 .62642 m .55265 .5896 L .52064 .5807 L p .663 g
F P
0 g
s
.63738 .66831 m .60083 .62642 L .54081 .6033 L p .63 g
F P
0 g
s
.55265 .5896 m .60083 .62642 L .663 .62912 L p .637 g
F P
0 g
s
.60083 .62642 m .63738 .66831 L .72261 .68073 L p .584 g
F P
0 g
s
.72261 .68073 m .663 .62912 L .60083 .62642 L p .584 g
F P
0 g
s
.49658 .5332 m .49761 .56276 L .49784 .50582 L closepath p .536 g
F P
0 g
s
.49369 .50235 m .49658 .5332 L .49784 .50582 L p .775 g
F P
0 g
s
.44477 .44901 m .43375 .448 L .34637 .36469 L p .58 g
F P
0 g
s
.33227 .38265 m .34637 .36469 L .43375 .448 L p .577 g
F P
0 g
s
.33227 .38265 m .33453 .41443 L .24239 .39391 L p .477 g
F P
0 g
s
.27317 .44509 m .24239 .39391 L .33453 .41443 L p .524 g
F P
0 g
s
.36129 .96775 m .3367 .96562 L .24477 .98518 L p .047 g
F P
0 g
s
.3367 .96562 m .36129 .96775 L .43749 .90407 L p F P
s
.47343 .47919 m .48603 .50487 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.45885 .45982 m .47343 .47919 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.54081 .6033 m .556 .62757 L .63738 .66831 L p .63 g
F P
0 g
s
.65888 .70945 m .63738 .66831 L .556 .62757 L p .605 g
F P
0 g
s
.63738 .66831 m .65888 .70945 L .75843 .73531 L p .538 g
F P
0 g
s
.75843 .73531 m .72261 .68073 L .63738 .66831 L p .538 g
F P
0 g
s
.556 .62757 m .54081 .6033 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.56477 .65016 m .556 .62757 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.556 .62757 m .56477 .65016 L .65888 .70945 L p .605 g
F P
0 g
s
.27317 .44509 m .33073 .49581 L .28761 .51398 L p .534 g
F P
0 g
s
.38512 .54901 m .28761 .51398 L .33073 .49581 L p .623 g
F P
0 g
s
.49761 .56276 m .60989 .55264 L .58498 .56844 L closepath p .777 g
F P
0 g
s
.48603 .50487 m .49458 .53385 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.54081 .6033 m .52064 .5807 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.83628 .21594 m .83429 .27665 L .77236 .23274 L p .716 g
F P
0 g
s
.77236 .23274 m .77417 .17219 L .83628 .21594 L p .716 g
F P
0 g
s
.44477 .44901 m .45885 .45982 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .49658 .5332 L .49369 .50235 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .49369 .50235 L .48941 .47385 L closepath p .751 g
F P
0 g
s
.56649 .66776 m .56477 .65016 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.56477 .65016 m .56649 .66776 L .66368 .74367 L p .587 g
F P
0 g
s
.66368 .74367 m .65888 .70945 L .56477 .65016 L p .587 g
F P
0 g
s
.50057 .3462 m .50167 .31727 L .47964 .43712 L p .856 g
F P
0 g
s
.47964 .43712 m .48445 .45115 L .50057 .3462 L p .856 g
F P
0 g
s
.49761 .56276 m .48941 .47385 L .48445 .45115 L closepath p .751 g
F P
0 g
s
.55265 .5896 m .49761 .56276 L .52064 .5807 L closepath p .708 g
F P
0 g
s
.58543 .58792 m .49761 .56276 L .55265 .5896 L closepath p .706 g
F P
0 g
s
.663 .62912 m .58543 .58792 L .55265 .5896 L p .637 g
F P
0 g
s
.57115 .20838 m .55776 .2213 L .63182 .164 L p .749 g
F P
0 g
s
.63182 .164 m .66204 .15108 L .57115 .20838 L p .749 g
F P
0 g
s
.80703 .41275 m .72846 .47885 L .71754 .43829 L p .848 g
F P
0 g
s
.71754 .43829 m .79409 .35518 L .80703 .41275 L p .848 g
F P
0 g
s
.49761 .56276 m .51906 .69025 L .51556 .68831 L closepath p .751 g
F P
0 g
s
.18405 .71478 m .13255 .78914 L .16488 .84551 L p .848 g
F P
0 g
s
.15226 .91642 m .16488 .84551 L .13255 .78914 L p .819 g
F P
0 g
s
.65888 .70945 m .66368 .74367 L .76721 .78423 L p .496 g
F P
0 g
s
.76721 .78423 m .75843 .73531 L .65888 .70945 L p .496 g
F P
0 g
s
.43375 .448 m .44477 .44901 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.61061 .57658 m .49761 .56276 L .58543 .58792 L closepath p .706 g
F P
0 g
s
.58543 .58792 m .663 .62912 L .71205 .61257 L p .622 g
F P
0 g
s
.71205 .61257 m .61061 .57658 L .58543 .58792 L p .622 g
F P
0 g
s
.37587 .59325 m .49761 .56276 L .38441 .57297 L closepath p .772 g
F P
0 g
s
.26791 .64626 m .37587 .59325 L .38441 .57297 L p .852 g
F P
0 g
s
.49761 .56276 m .48445 .45115 L .47964 .43712 L closepath p .751 g
F P
0 g
s
.56149 .67756 m .56649 .66776 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.56649 .66776 m .56149 .67756 L .6522 .76522 L p .576 g
F P
0 g
s
.6522 .76522 m .66368 .74367 L .56649 .66776 L p .576 g
F P
0 g
s
.74815 .14418 m .77417 .17219 L .67905 .17356 L p .298 g
F P
0 g
s
.67905 .17356 m .66204 .15108 L .74815 .14418 L p .298 g
F P
0 g
s
.663 .62912 m .72261 .68073 L .79144 .66569 L p .547 g
F P
0 g
s
.79144 .66569 m .71205 .61257 L .663 .62912 L p .547 g
F P
0 g
s
.49761 .56276 m .62057 .53197 L .60989 .55264 L closepath p .772 g
F P
0 g
s
.38441 .57297 m .28243 .60583 L .26791 .64626 L p .852 g
F P
0 g
s
.18405 .71478 m .26791 .64626 L .28243 .60583 L p .914 g
F P
0 g
s
.33073 .49581 m .40889 .53734 L .38512 .54901 L p .623 g
F P
0 g
s
.49761 .56276 m .38512 .54901 L .40889 .53734 L closepath p .706 g
F P
0 g
s
.43375 .448 m .42806 .45675 L .33227 .38265 L p .577 g
F P
0 g
s
.33453 .41443 m .33227 .38265 L .42806 .45675 L p .584 g
F P
0 g
s
.46301 .86003 m .43749 .90407 L p .45334 .87049 L .921 g
F P
0 g
s
.4946 .78311 m .46301 .86003 L .45334 .87049 L .921 g
F
.43749 .90407 m .45269 .87227 L .4946 .78311 L p F P
0 g
s
.45269 .87227 m .43749 .90407 L .36129 .96775 L p .929 g
F P
0 g
s
.39564 .93309 m .45269 .87227 L p .42379 .90246 L .929 g
F P
0 g
s
.36129 .96775 m .39319 .93557 L p .42379 .90246 L .929 g
F P
0 g
s
.39319 .93557 m .39564 .93309 L p .42379 .90246 L .929 g
F P
0 g
s
.4946 .78311 m .49356 .8073 L .46301 .86003 L p .921 g
F P
0 g
s
.49356 .8073 m .4946 .78311 L .5109 .67548 L p .863 g
F P
0 g
s
.5109 .67548 m .51556 .68831 L .49356 .8073 L p .863 g
F P
0 g
s
.33453 .41443 m .35483 .45495 L .27317 .44509 L p .524 g
F P
0 g
s
.33073 .49581 m .27317 .44509 L .35483 .45495 L p .577 g
F P
0 g
s
.49761 .56276 m .51556 .68831 L .5109 .67548 L closepath p .751 g
F P
0 g
s
.42806 .45675 m .43375 .448 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.551 .6777 m .56149 .67756 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.62145 .55879 m .49761 .56276 L .61061 .57658 L closepath p .708 g
F P
0 g
s
.49761 .56276 m .5109 .67548 L .50594 .65316 L closepath p .751 g
F P
0 g
s
.50068 .59204 m .50941 .62182 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.50941 .62182 m .50068 .59204 L .50501 .61864 L p .655 g
F P
0 g
s
.49761 .56276 m .50501 .61864 L .50068 .59204 L closepath p .331 g
F P
0 g
s
.50545 .61896 m .50784 .62532 L .507 .62008 L .655 g
F
.50582 .62078 m .5262 .67414 L p .50545 .61896 L F P
0 g
s
.5262 .67414 m .50501 .61864 L .50945 .64054 L p .497 g
F P
0 g
s
.54365 .71546 m .5262 .67414 L p .51071 .64306 L .497 g
F P
0 g
s
.50945 .64054 m .51076 .64341 L p .51071 .64306 L .497 g
F P
0 g
s
.51076 .64341 m .54365 .71546 L p .51071 .64306 L .497 g
F P
0 g
s
.54365 .71546 m .50945 .64054 L .51282 .65622 L p .383 g
F P
0 g
s
.51282 .65622 m .5127 .6557 L p .51265 .65534 L .047 g
F P
0 g
s
.5127 .6557 m .50945 .64054 L p .51265 .65534 L .047 g
F P
0 g
s
.50945 .64054 m .50822 .63245 L p .51265 .65534 L .047 g
F P
0 g
s
.50822 .63245 m .49761 .56276 L .51173 .64957 L p .047 g
F P
0 g
s
.51173 .64957 m .51282 .65622 L p .50822 .63245 L .047 g
F P
0 g
s
.51418 .66482 m .51396 .66342 L p .51126 .64681 L .498 g
F P
0 g
s
.49761 .56276 m .5113 .6471 L p .51126 .64681 L .498 g
F P
0 g
s
.5113 .6471 m .51418 .66482 L p .51126 .64681 L .498 g
F P
0 g
s
.51396 .66342 m .51282 .65622 L .51173 .64957 L p .498 g
F P
0 g
s
.51173 .64957 m .49761 .56276 L p .51396 .66342 L .498 g
F P
0 g
s
.5262 .67414 m .50941 .62182 L p .507 .62008 L .655 g
F P
0 g
s
.50784 .62532 m .5262 .67414 L .507 .62008 L .655 g
F
.50501 .61864 m .50582 .62078 L p .50545 .61896 L F P
0 g
s
.49761 .56276 m .50945 .64054 L .50501 .61864 L closepath p .186 g
F P
0 g
s
.51001 .66045 m .51315 .66611 L .51131 .65388 L p .676 g
F P
0 g
s
.51131 .65388 m .50897 .6383 L p .50965 .65293 L .676 g
F P
0 g
s
.50965 .65293 m .51001 .66045 L .51131 .65388 L .676 g
F
.50722 .63846 m .51001 .66045 L p .50897 .6383 L F P
0 g
s
.49761 .56276 m .5056 .64865 L .51001 .66045 L closepath p .628 g
F P
0 g
s
.51315 .66611 m .51418 .66482 L .51217 .65245 L p .683 g
F P
0 g
s
.51217 .65245 m .49761 .56276 L .51131 .65388 L p .683 g
F P
0 g
s
.51131 .65388 m .51315 .66611 L p .51217 .65245 L .683 g
F P
0 g
s
.5629 .75562 m .51418 .66482 L .51315 .66611 L p .438 g
F P
0 g
s
.51315 .66611 m .51001 .66045 L p .52982 .69808 L .487 g
F P
0 g
s
.55914 .75374 m .54773 .73199 L p .54767 .73195 L .487 g
F P
0 g
s
.51282 .65622 m .55714 .743 L .54365 .71546 L p .383 g
F P
0 g
s
.55714 .743 m .51282 .65622 L .51418 .66482 L p .384 g
F P
0 g
s
.51418 .66482 m .5629 .75562 L .55714 .743 L p .384 g
F P
0 g
s
.54678 .73913 m .55914 .75374 L p .54767 .73195 L .487 g
F P
0 g
s
.52982 .69808 m .54005 .72472 L .54767 .73195 L .487 g
F
.52626 .69523 m .54005 .72472 L p .52982 .69808 L F P
0 g
s
.51001 .66045 m .52626 .69523 L p .52982 .69808 L .487 g
F P
0 g
s
.52947 .69719 m .51315 .66611 L p .52982 .69808 L .487 g
F P
0 g
s
.55914 .75374 m .5629 .75562 L p .5387 .71208 L .438 g
F P
0 g
s
.51315 .66611 m .53415 .70612 L p .5387 .71208 L .438 g
F P
0 g
s
.53415 .70612 m .55914 .75374 L p .5387 .71208 L .438 g
F P
0 g
s
.51001 .66045 m .5056 .64865 L p .5279 .69764 L .524 g
F P
0 g
s
.52626 .69523 m .51001 .66045 L p .5279 .69764 L .524 g
F P
0 g
s
.54005 .72472 m .54678 .73913 L p .54767 .73195 L .487 g
F P
0 g
s
.54678 .73913 m .52626 .69523 L p .5279 .69764 L .524 g
F P
0 g
s
.52926 .71437 m .54678 .73913 L p .50942 .65704 L .524 g
F P
0 g
s
.50906 .65764 m .52926 .71437 L .50942 .65704 L .524 g
F
.50649 .65112 m .52926 .71437 L p .50681 .6513 L F P
0 g
s
.50681 .6513 m .50906 .65764 L .50942 .65704 L .524 g
F
.5056 .64865 m .50649 .65112 L p .50681 .6513 L F P
0 g
s
.49761 .56276 m .50159 .62424 L .49865 .59268 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .50594 .65316 L .50159 .62424 L closepath p .751 g
F P
0 g
s
.49865 .59268 m .50159 .62424 L .4967 .68302 L p .693 g
F P
0 g
s
.50159 .62424 m .50594 .65316 L .49571 .73978 L p .752 g
F P
0 g
s
.49571 .73978 m .4967 .68302 L .49995 .644 L p .752 g
F P
0 g
s
.49995 .644 m .50159 .62424 L p .50028 .65001 L .752 g
F P
0 g
s
.50028 .65001 m .49571 .73978 L .49995 .644 L .752 g
F
.50114 .63177 m .5056 .64865 L .50278 .61836 L p .577 g
F P
0 g
s
.5003 .61531 m .50114 .63177 L p .50278 .61836 L .577 g
F P
0 g
s
.49761 .56276 m .49865 .59268 L .49737 .62102 L closepath p .536 g
F P
0 g
s
.4967 .68302 m .49737 .62102 L .4982 .60267 L p .693 g
F P
0 g
s
.4982 .60267 m .49865 .59268 L p .4984 .60388 L .693 g
F P
0 g
s
.4984 .60388 m .4967 .68302 L .4982 .60267 L .693 g
F
.49761 .56276 m .49737 .62102 L .4941 .64526 L closepath p .584 g
F P
0 g
s
.49853 .64517 m .49784 .61101 L .49702 .59585 L p .574 g
F P
0 g
s
.49702 .59585 m .49657 .58761 L p .49681 .59464 L .574 g
F P
0 g
s
.49681 .59464 m .49853 .64517 L .49702 .59585 L .574 g
F
.49761 .56276 m .49784 .61101 L .50114 .63177 L closepath p .526 g
F P
0 g
s
.50114 .63177 m .49826 .61364 L p .49853 .61504 L .553 g
F P
0 g
s
.52184 .69256 m .52566 .70379 L .52121 .69157 L .541 g
F
.52742 .70925 m .5056 .64865 L p .52566 .70379 L F P
0 g
s
.52926 .71437 m .52742 .70925 L p .52566 .70379 L .541 g
F P
0 g
s
.51154 .68224 m .52926 .71437 L p .52184 .69256 L .541 g
F P
0 g
s
.5056 .64865 m .50114 .63177 L p .52184 .69256 L .541 g
F P
0 g
s
.52184 .69256 m .52121 .69157 L .5056 .64865 L .541 g
F
.50594 .65316 m .5109 .67548 L .4946 .78311 L p .805 g
F P
0 g
s
.50354 .6388 m .51154 .68224 L .52184 .69256 L .541 g
F
.59478 .77725 m .54365 .71546 L .55714 .743 L p .275 g
F P
0 g
s
.55714 .743 m .60182 .78855 L p .59478 .77725 L .275 g
F P
0 g
s
.60182 .78855 m .62415 .81131 L .6092 .79397 L p .275 g
F P
0 g
s
.6092 .79397 m .59478 .77725 L p .60182 .78855 L .275 g
F P
0 g
s
.5629 .75562 m .55914 .75374 L p .58215 .77409 L .136 g
F P
0 g
s
.65352 .81741 m .59478 .77725 L .62415 .81131 L p .171 g
F P
0 g
s
.62415 .81131 m .70195 .85165 L .65352 .81741 L p .171 g
F P
0 g
s
.62415 .81131 m .55714 .743 L .5629 .75562 L p .09 g
F P
0 g
s
.70195 .85165 m .62415 .81131 L .63746 .82303 L p .315 g
F P
0 g
s
.5629 .75562 m .63746 .82303 L .62415 .81131 L p .09 g
F P
0 g
s
.63746 .82303 m .58848 .77874 L p .58215 .77409 L .136 g
F P
0 g
s
.58848 .77874 m .5629 .75562 L p .58215 .77409 L .136 g
F P
0 g
s
.54678 .73913 m .52926 .71437 L p .58877 .77349 L .161 g
F P
0 g
s
.60405 .78866 m .58795 .77474 L p .58877 .77349 L .161 g
F P
0 g
s
.58795 .77474 m .54678 .73913 L p .58877 .77349 L .161 g
F P
0 g
s
.56149 .67756 m .551 .6777 L .62719 .76966 L p .573 g
F P
0 g
s
.62719 .76966 m .6522 .76522 L .56149 .67756 L p .573 g
F P
0 g
s
.59353 .75486 m .62719 .76966 L .551 .6777 L p .578 g
F P
0 g
s
.6522 .76522 m .62719 .76966 L .70858 .83121 L p .434 g
F P
0 g
s
.62719 .76966 m .59353 .75486 L .65352 .81741 L p .423 g
F P
0 g
s
.65352 .81741 m .70858 .83121 L .62719 .76966 L p .423 g
F P
0 g
s
.59478 .77725 m .65352 .81741 L .59353 .75486 L p .436 g
F P
0 g
s
.61051 .75552 m .67299 .80207 L p .59435 .76499 L .41 g
F P
0 g
s
.56608 .75166 m .60405 .78866 L p .54583 .73083 L .161 g
F P
0 g
s
.551 .6777 m .53707 .66758 L .59353 .75486 L p .578 g
F P
0 g
s
.55762 .72169 m .59353 .75486 L .53707 .66758 L p .593 g
F P
0 g
s
.59353 .75486 m .55762 .72169 L .59478 .77725 L p .436 g
F P
0 g
s
.54365 .71546 m .59478 .77725 L .55762 .72169 L p .475 g
F P
0 g
s
.56608 .75166 m .60109 .7547 L p .59435 .76499 L .41 g
F P
0 g
s
.60109 .7547 m .61051 .75552 L p .59435 .76499 L .41 g
F P
0 g
s
.53174 .71683 m .54504 .73034 L .54583 .73083 L .161 g
F
.52926 .71437 m .53099 .71612 L p .53174 .71683 L F P
0 g
s
.53099 .71612 m .54504 .73034 L p .53174 .71683 L .161 g
F P
0 g
s
.54504 .73034 m .56608 .75166 L p .54583 .73083 L .161 g
F P
0 g
s
.52982 .69808 m .54767 .73195 L .52947 .69719 L .487 g
F
.54773 .73199 m .52947 .69719 L p .54767 .73195 L F P
0 g
s
.52926 .71437 m .51154 .68224 L p .54502 .72485 L .289 g
F P
0 g
s
.56608 .75166 m .55517 .74061 L p .54502 .72485 L .289 g
F P
0 g
s
.55517 .74061 m .52926 .71437 L p .54502 .72485 L .289 g
F P
0 g
s
.51154 .68224 m .53915 .72117 L .53797 .71824 L .289 g
F
.51154 .68224 m .53797 .71824 L .53057 .70645 L F
.50273 .63643 m .50319 .63883 L .50354 .6388 L .541 g
F
.50114 .63177 m .50319 .63883 L .50273 .63643 L F
.50114 .63177 m .50261 .63889 L p .50319 .63883 L F P
0 g
s
.4946 .78311 m .49571 .73978 L .50594 .65316 L p .805 g
F P
0 g
s
.50261 .63889 m .51154 .68224 L p .50354 .6388 L .541 g
F P
0 g
s
.51154 .68224 m .50114 .63177 L p .50066 .62868 L .553 g
F P
0 g
s
.49853 .61504 m .51154 .68224 L .50066 .62868 L .553 g
F
.49823 .613 m .51154 .68224 L .49826 .61364 L F
.49853 .64517 m .51154 .68224 L p .5009 .62688 L F P
0 g
s
.5276 .70809 m .53915 .72117 L p .51355 .68507 L .289 g
F P
0 g
s
.51154 .68224 m .51305 .68468 L p .51355 .68507 L .289 g
F P
0 g
s
.51305 .68468 m .5276 .70809 L p .51355 .68507 L .289 g
F P
0 g
s
.5276 .70809 m .51154 .68224 L .50897 .6749 L p .42 g
F P
0 g
s
.50897 .6749 m .49853 .64517 L p .51362 .67783 L .42 g
F P
0 g
s
.51362 .67783 m .5276 .70809 L .50897 .6749 L .42 g
F
.49958 .66138 m .5276 .70809 L p .51082 .67176 L F P
0 g
s
.49853 .64517 m .49907 .65353 L p .51082 .67176 L .42 g
F P
0 g
s
.49907 .65353 m .49958 .66138 L p .51082 .67176 L .42 g
F P
0 g
s
.50083 .65831 m .52852 .6872 L p .5011 .66283 L .08 g
F P
0 g
s
.55762 .72169 m .5262 .67414 L .54365 .71546 L p .475 g
F P
0 g
s
.52852 .6872 m .54687 .70634 L p .52325 .68388 L .08 g
F P
0 g
s
.5011 .66283 m .52852 .6872 L .52325 .68388 L .08 g
F
.53057 .70645 m .53797 .71824 L .53601 .71338 L .289 g
F
.49816 .61268 m .49823 .61861 L .5009 .62688 L .553 g
F
.49823 .61861 m .49853 .64517 L .5009 .62688 L F
.49826 .61364 m .49784 .61101 L p .49823 .613 L F P
0 g
s
.49784 .61101 m .49787 .61231 L p .49816 .61268 L .553 g
F P
0 g
s
.49787 .61231 m .49853 .64517 L p .49816 .61268 L .553 g
F P
0 g
s
.50278 .61836 m .49761 .56276 L .5003 .61531 L p .577 g
F P
0 g
s
.49958 .66138 m .49973 .66103 L p .5011 .66283 L .08 g
F P
0 g
s
.49973 .66103 m .50083 .65831 L p .5011 .66283 L .08 g
F P
0 g
s
.53915 .72117 m .56608 .75166 L p .53601 .71338 L .289 g
F P
0 g
s
.50897 .6383 m .49761 .56276 L .50722 .63846 L p .676 g
F P
0 g
s
.49761 .56276 m .49657 .58761 L .49784 .61101 L closepath p .47 g
F P
0 g
s
.49761 .56276 m .48214 .67551 L .48107 .66925 L closepath p .805 g
F P
0 g
s
.49761 .56276 m .48529 .67358 L .48214 .67551 L closepath p .743 g
F P
0 g
s
.49761 .56276 m .49019 .61341 L .49453 .58825 L closepath p .17 g
F P
0 g
s
.49761 .56276 m .48107 .66925 L .48129 .66706 L p .801 g
F P
0 g
s
.48129 .66706 m .4824 .65575 L .48348 .64914 L p .801 g
F P
0 g
s
.45119 .70038 m .46872 .65432 L p .48447 .63785 L .83 g
F P
0 g
s
.48419 .64481 m .49761 .56276 L p .48398 .64988 L .801 g
F P
0 g
s
.48348 .64914 m .48419 .64481 L p .48129 .66706 L .801 g
F P
0 g
s
.48398 .64988 m .48129 .66706 L .48419 .64481 L .801 g
F
.4824 .65575 m .48574 .63653 L .48698 .62885 L p 0 g
F P
s
.49761 .56276 m .48967 .66327 L .48529 .67358 L closepath p .683 g
F P
0 g
s
.49761 .56276 m .4941 .64526 L .48967 .66327 L closepath p .632 g
F P
0 g
s
.48405 .73298 m .4941 .64526 L .49696 .62409 L p .71 g
F P
0 g
s
.49696 .62409 m .49737 .62102 L p .49699 .62421 L .71 g
F P
0 g
s
.49699 .62421 m .48405 .73298 L .49696 .62409 L .71 g
F
.49737 .62102 m .4967 .68302 L .48405 .73298 L p F P
0 g
s
.48698 .62885 m .49304 .59117 L p .48348 .64914 L F P
s
.49311 .59026 m .48348 .64914 L p .49304 .59117 L F P
s
.49761 .56276 m .48574 .63653 L .49019 .61341 L closepath p .031 g
F P
0 g
s
.48974 .61281 m .48713 .61924 L .48817 .6107 L .772 g
F
.49019 .61341 m .48937 .61497 L p .48974 .61281 L F P
0 g
s
.48937 .61497 m .48713 .61924 L p .48974 .61281 L .772 g
F P
0 g
s
.48571 .60741 m .49453 .58825 L .49019 .61341 L p .772 g
F P
0 g
s
.48348 .64914 m .4824 .65575 L p .48698 .62885 L F P
s
.46872 .65432 m .49019 .61341 L .48574 .63653 L p .83 g
F P
0 g
s
.49169 .60833 m .4891 .61298 L .49073 .62051 L p .542 g
F P
0 g
s
.49073 .62051 m .49958 .66138 L p .49169 .60833 L .542 g
F P
0 g
s
.46872 .65432 m .48571 .60741 L p .48817 .6107 L .772 g
F P
0 g
s
.48713 .61924 m .46872 .65432 L p .48817 .6107 L .772 g
F P
0 g
s
.47252 .61816 m .48571 .60741 L .46872 .65432 L p .73 g
F P
0 g
s
.49958 .66138 m .49853 .64517 L .49754 .63711 L p .542 g
F P
0 g
s
.49754 .63711 m .49533 .61916 L p .49697 .63547 L .542 g
F P
0 g
s
.49697 .63547 m .49958 .66138 L .49754 .63711 L .542 g
F
.49535 .61949 m .49958 .66138 L .49533 .61916 L F
.49489 .61722 m .49958 .66138 L .49524 .61837 L F
.49358 .60494 m .49169 .60833 L p .49958 .66138 L F P
0 g
s
.50083 .65831 m .49958 .66138 L .49202 .62646 L p .442 g
F P
0 g
s
.49202 .62646 m .4891 .61298 L p .49268 .62681 L .442 g
F P
0 g
s
.49268 .62681 m .50083 .65831 L .49202 .62646 L .442 g
F
.4891 .61298 m .48387 .61134 L .49668 .64683 L p F P
0 g
s
.4891 .61298 m .49668 .64683 L .49641 .64123 L .442 g
F
.49304 .59117 m .49761 .56276 L .49311 .59026 L p 0 g
F P
s
.49453 .58825 m .48571 .60741 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.48571 .60741 m .47252 .61816 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .48387 .61134 L .4891 .61298 L closepath p .751 g
F P
0 g
s
.47252 .61816 m .45716 .61971 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47884 .60123 L .48387 .61134 L closepath p .751 g
F P
0 g
s
.48387 .61134 m .47884 .60123 L .4927 .62311 L p .044 g
F P
0 g
s
.49657 .58761 m .49358 .60494 L .49853 .64517 L p .574 g
F P
0 g
s
.4927 .62311 m .50204 .63785 L p .49348 .62535 L .044 g
F P
0 g
s
.48387 .61134 m .4927 .62311 L .49348 .62535 L .044 g
F
.49668 .64683 m .50083 .65831 L p .49641 .64123 L .442 g
F P
0 g
s
.49533 .61916 m .49524 .61837 L p .49535 .61949 L .542 g
F P
0 g
s
.49524 .61837 m .49358 .60494 L p .49489 .61722 L .542 g
F P
0 g
s
.48574 .63653 m .48443 .63896 L p .48447 .63785 L .83 g
F P
0 g
s
.48443 .63896 m .45119 .70038 L p .48447 .63785 L .83 g
F P
0 g
s
.53707 .66758 m .551 .6777 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47964 .43712 L .47587 .43361 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .48264 .51185 L .4897 .53961 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .49358 .60494 L .49657 .58761 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .47587 .43361 L .47392 .44113 L closepath p .751 g
F P
0 g
s
.42927 .47383 m .42806 .45675 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .40889 .53734 L .44167 .53548 L closepath p .706 g
F P
0 g
s
.44146 .54938 m .49761 .56276 L .47424 .55179 L closepath p .778 g
F P
0 g
s
.40842 .55697 m .49761 .56276 L .44146 .54938 L closepath p .779 g
F P
0 g
s
.40842 .55697 m .44146 .54938 L p .4134 .55475 L .68 g
F P
0 g
s
.44146 .54938 m .39042 .55104 L .32733 .57124 L p .68 g
F P
0 g
s
.44146 .54938 m .47424 .55179 L p .42574 .55136 L .152 g
F P
0 g
s
.42778 .54982 m .44146 .54938 L p .42574 .55136 L .152 g
F P
0 g
s
.44161 .54939 m .4302 .54976 L p .47424 .55179 L .152 g
F P
0 g
s
.47424 .55179 m .45323 .54901 L .44161 .54939 L p .152 g
F P
0 g
s
.39042 .55104 m .42778 .54982 L p .42574 .55136 L .152 g
F P
0 g
s
.42485 .55135 m .47424 .55179 L .4302 .54976 L .152 g
F
.45323 .54901 m .43704 .55439 L .418 .55717 L p .41 g
F P
0 g
s
.32733 .57124 m .40234 .55804 L p .4134 .55475 L .68 g
F P
0 g
s
.40234 .55804 m .40842 .55697 L p .4134 .55475 L .68 g
F P
0 g
s
.38441 .57297 m .49761 .56276 L .40842 .55697 L closepath p .777 g
F P
0 g
s
.28243 .60583 m .38441 .57297 L .40842 .55697 L p .87 g
F P
0 g
s
.40842 .55697 m .32733 .57124 L .28243 .60583 L p .87 g
F P
0 g
s
.2024 .65669 m .28243 .60583 L .30958 .58492 L p .82 g
F P
0 g
s
.30958 .58492 m .32733 .57124 L p .3046 .58678 L .82 g
F P
0 g
s
.2024 .65669 m .30958 .58492 L .3046 .58678 L .82 g
F
.36746 .55839 m .39042 .55104 L p .3691 .5598 L .07 g
F P
0 g
s
.32733 .57124 m .26346 .60324 L .2024 .65669 L p .82 g
F P
0 g
s
.29346 .5909 m .36746 .55839 L .3691 .5598 L .07 g
F
.26346 .60324 m .29135 .59121 L .29346 .5909 L F
.29135 .59121 m .36746 .55839 L .29346 .5909 L F
.28243 .60583 m .2024 .65669 L .18405 .71478 L p .914 g
F P
0 g
s
.32643 .59418 m .22442 .64749 L p .31885 .58722 L .278 g
F P
0 g
s
.39042 .55104 m .35046 .56704 L .28827 .59291 L p .07 g
F P
0 g
s
.29165 .59164 m .39042 .55104 L .28827 .59291 L .07 g
F
.35046 .56704 m .33462 .58493 L p .31885 .58722 L .278 g
F P
0 g
s
.33462 .58493 m .32643 .59418 L p .31885 .58722 L .278 g
F P
0 g
s
.28827 .59291 m .26346 .60324 L p .29165 .59164 L .07 g
F P
0 g
s
.26346 .60324 m .32733 .57124 L .36746 .55839 L p .07 g
F P
0 g
s
.418 .55717 m .37358 .56366 L p .42144 .55459 L .41 g
F P
0 g
s
.37358 .56366 m .35046 .56704 L p .37293 .5631 L .41 g
F P
0 g
s
.37293 .5631 m .42144 .55459 L .37358 .56366 L .41 g
F
.42144 .55459 m .45323 .54901 L .418 .55717 L F
.4302 .54976 m .39042 .55104 L p .42485 .55135 L .152 g
F P
0 g
s
.45323 .54901 m .47424 .55179 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.52232 .64816 m .53707 .66758 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47732 .48343 L .48264 .51185 L closepath p .751 g
F P
0 g
s
.47429 .54461 m .45355 .52142 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47392 .44113 L .47434 .4586 L closepath p .751 g
F P
0 g
s
.43797 .49657 m .42927 .47383 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.50941 .62182 m .52232 .64816 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47434 .4586 L .47732 .48343 L closepath p .751 g
F P
0 g
s
.45355 .52142 m .43797 .49657 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.61467 .53942 m .49761 .56276 L .62145 .55879 L closepath p .713 g
F P
0 g
s
.49761 .56276 m .4897 .53961 L .49941 .51658 L closepath p .724 g
F P
0 g
s
.49761 .56276 m .44167 .53548 L .47429 .54461 L closepath p .708 g
F P
0 g
s
.49761 .56276 m .4891 .61298 L .49358 .60494 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .48969 .5468 L .48253 .5395 L closepath p .751 g
F P
0 g
s
.43704 .55439 m .45323 .54901 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .48253 .5395 L .477 .54142 L closepath p .751 g
F P
0 g
s
.42755 .56649 m .43704 .55439 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .6138 .51153 L .62057 .53197 L closepath p .766 g
F P
0 g
s
.59165 .52388 m .49761 .56276 L .61467 .53942 L closepath p .719 g
F P
0 g
s
.49761 .56276 m .49941 .51658 L .52447 .49934 L closepath p .731 g
F P
0 g
s
.48969 .5468 m .49761 .56276 L .49941 .53053 L closepath p .759 g
F P
0 g
s
.49761 .56276 m .477 .54142 L .47375 .55156 L closepath p .751 g
F P
0 g
s
.42567 .58262 m .42755 .56649 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.45716 .61971 m .4424 .6127 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .47497 .58533 L .47884 .60123 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .47375 .55156 L .4731 .56741 L closepath p .751 g
F P
0 g
s
.43111 .59927 m .42567 .58262 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.4424 .6127 m .43111 .59927 L .49761 .56276 L closepath p .688 g
F P
0 g
s
.49761 .56276 m .4731 .56741 L .47497 .58533 L closepath p .751 g
F P
0 g
s
.49761 .56276 m .59097 .49685 L .6138 .51153 L closepath p .758 g
F P
0 g
s
.49761 .56276 m .55814 .49223 L .59097 .49685 L closepath p .749 g
F P
0 g
s
.55854 .51649 m .49761 .56276 L .59165 .52388 L closepath p .728 g
F P
0 g
s
.49761 .56276 m .52447 .49934 L .55814 .49223 L closepath p .74 g
F P
0 g
s
.49941 .53053 m .49761 .56276 L .52462 .51913 L closepath p .748 g
F P
0 g
s
.52462 .51913 m .49761 .56276 L .55854 .51649 L closepath p .738 g
F P
0 g
s
.50255 .30845 m .50167 .31727 L .55776 .2213 L p .861 g
F P
0 g
s
.55776 .2213 m .57115 .20838 L .50255 .30845 L p .861 g
F P
0 g
s
.47587 .43361 m .47964 .43712 L .50167 .31727 L p .882 g
F P
0 g
s
.72261 .68073 m .75843 .73531 L .84017 .72786 L p .468 g
F P
0 g
s
.84017 .72786 m .79144 .66569 L .72261 .68073 L p .468 g
F P
0 g
s
.50167 .31727 m .50255 .30845 L .47587 .43361 L p .882 g
F P
0 g
s
.66368 .74367 m .6522 .76522 L .74919 .81868 L p .461 g
F P
0 g
s
.74919 .81868 m .76721 .78423 L .66368 .74367 L p .461 g
F P
0 g
s
.61061 .57658 m .71205 .61257 L .73491 .58197 L p .616 g
F P
0 g
s
.73491 .58197 m .62145 .55879 L .61061 .57658 L p .616 g
F P
0 g
s
.72846 .47885 m .62057 .53197 L .6138 .51153 L p .809 g
F P
0 g
s
.6138 .51153 m .71754 .43829 L .72846 .47885 L p .809 g
F P
0 g
s
.35483 .45495 m .39194 .4976 L .33073 .49581 L p .577 g
F P
0 g
s
.40889 .53734 m .33073 .49581 L .39194 .4976 L p .637 g
F P
0 g
s
.15226 .91642 m .17335 .96404 L .24477 .98518 L p .542 g
F P
0 g
s
.24477 .98518 m .2835 .99261 L .36129 .96775 L p .047 g
F P
0 g
s
.2835 .99261 m .24477 .98518 L .17335 .96404 L p .384 g
F P
0 g
s
.42806 .45675 m .42927 .47383 L .33453 .41443 L p .584 g
F P
0 g
s
.39194 .4976 m .44167 .53548 L .40889 .53734 L p .637 g
F P
0 g
s
.35483 .45495 m .33453 .41443 L .42927 .47383 L p .6 g
F P
0 g
s
.83429 .27665 m .79409 .35518 L .73879 .31677 L p .77 g
F P
0 g
s
.73879 .31677 m .77236 .23274 L .83429 .27665 L p .77 g
F P
0 g
s
.45355 .52142 m .47429 .54461 L .44167 .53548 L p .661 g
F P
0 g
s
.47392 .44113 m .47587 .43361 L .50255 .30845 L p .668 g
F P
0 g
s
.13255 .78914 m .18405 .71478 L .2024 .65669 L p .924 g
F P
0 g
s
.39564 .93309 m .36129 .96775 L .2835 .99261 L p .888 g
F P
0 g
s
.44167 .53548 m .39194 .4976 L .45355 .52142 L p .661 g
F P
0 g
s
.50104 .46094 m .49941 .51658 L .4897 .53961 L p .639 g
F P
0 g
s
.4897 .53961 m .48264 .51185 L .50104 .46094 L p .639 g
F P
0 g
s
.49571 .73978 m .4946 .78311 L .45269 .87227 L p .84 g
F P
0 g
s
.42927 .47383 m .43797 .49657 L .35483 .45495 L p .6 g
F P
0 g
s
.53707 .66758 m .52232 .64816 L .55762 .72169 L p .593 g
F P
0 g
s
.13255 .78914 m .11853 .85756 L .15226 .91642 L p .819 g
F P
0 g
s
.17335 .96404 m .15226 .91642 L .11853 .85756 L p .428 g
F P
0 g
s
.43797 .49657 m .45355 .52142 L .39194 .4976 L p .625 g
F P
0 g
s
.52232 .64816 m .50941 .62182 L .5262 .67414 L p .62 g
F P
0 g
s
.71205 .61257 m .79144 .66569 L .82551 .62908 L p .51 g
F P
0 g
s
.82551 .62908 m .73491 .58197 L .71205 .61257 L p .51 g
F P
0 g
s
.48253 .5395 m .48969 .5468 L .49941 .53053 L p .898 g
F P
0 g
s
.39194 .4976 m .35483 .45495 L .43797 .49657 L p .625 g
F P
0 g
s
.5262 .67414 m .55762 .72169 L .52232 .64816 L p .62 g
F P
0 g
s
.66204 .15108 m .67905 .17356 L .57855 .22603 L p .168 g
F P
0 g
s
.57855 .22603 m .57115 .20838 L .66204 .15108 L p .168 g
F P
0 g
s
.75843 .73531 m .76721 .78423 L .85299 .78852 L p .382 g
F P
0 g
s
.85299 .78852 m .84017 .72786 L .75843 .73531 L p .382 g
F P
0 g
s
.70858 .83121 m .74919 .81868 L .6522 .76522 L p .434 g
F P
0 g
s
.50255 .30845 m .50303 .32152 L .47392 .44113 L p .668 g
F P
0 g
s
.47434 .4586 m .47392 .44113 L .50303 .32152 L p .048 g
F P
0 g
s
.62145 .55879 m .73491 .58197 L .72403 .54593 L p .62 g
F P
0 g
s
.72403 .54593 m .61467 .53942 L .62145 .55879 L p .62 g
F P
0 g
s
.77417 .17219 m .77236 .23274 L .67755 .22946 L p .603 g
F P
0 g
s
.67755 .22946 m .67905 .17356 L .77417 .17219 L p .603 g
F P
0 g
s
.50303 .32152 m .50255 .30845 L .57115 .20838 L p .017 g
F P
0 g
s
.46928 .81263 m .45269 .87227 L .39564 .93309 L p .857 g
F P
0 g
s
.45269 .87227 m .46928 .81263 L .49571 .73978 L p .84 g
F P
0 g
s
.50227 .40408 m .50104 .46094 L .48264 .51185 L p .538 g
F P
0 g
s
.48264 .51185 m .47732 .48343 L .50227 .40408 L p .538 g
F P
0 g
s
.71754 .43829 m .6138 .51153 L .59097 .49685 L p .773 g
F P
0 g
s
.4967 .68302 m .49571 .73978 L .46928 .81263 L p .774 g
F P
0 g
s
.50303 .32152 m .50295 .35503 L .47434 .4586 L p .048 g
F P
0 g
s
.47732 .48343 m .47434 .4586 L .50295 .35503 L p .375 g
F P
0 g
s
.46688 .76908 m .48967 .66327 L .4941 .64526 L p .733 g
F P
0 g
s
.4941 .64526 m .48405 .73298 L .46688 .76908 L p .733 g
F P
0 g
s
.46928 .81263 m .48405 .73298 L .4967 .68302 L p .774 g
F P
0 g
s
.79409 .35518 m .71754 .43829 L .67555 .41018 L p .778 g
F P
0 g
s
.67555 .41018 m .73879 .31677 L .79409 .35518 L p .778 g
F P
0 g
s
.57115 .20838 m .57855 .22603 L .50303 .32152 L p .017 g
F P
0 g
s
.50295 .35503 m .50227 .40408 L .47732 .48343 L p .375 g
F P
0 g
s
.35046 .56704 m .39042 .55104 L .45323 .54901 L p .41 g
F P
0 g
s
.54898 .42167 m .52447 .49934 L .49941 .51658 L p .675 g
F P
0 g
s
.49941 .51658 m .50104 .46094 L .54898 .42167 L p .675 g
F P
0 g
s
.59097 .49685 m .67555 .41018 L .71754 .43829 L p .773 g
F P
0 g
s
.44973 .78806 m .48529 .67358 L .48967 .66327 L p .76 g
F P
0 g
s
.48967 .66327 m .46688 .76908 L .44973 .78806 L p .76 g
F P
0 g
s
.2024 .65669 m .15317 .7177 L .13255 .78914 L p .924 g
F P
0 g
s
.11853 .85756 m .13255 .78914 L .15317 .7177 L p F P
s
.45119 .70038 m .48574 .63653 L .4824 .65575 L p .849 g
F P
0 g
s
.50109 .51479 m .49941 .53053 L .52462 .51913 L p .813 g
F P
0 g
s
.49941 .53053 m .50109 .51479 L .48253 .5395 L p .898 g
F P
0 g
s
.477 .54142 m .48253 .5395 L .50109 .51479 L p .826 g
F P
0 g
s
.67555 .41018 m .59097 .49685 L .55814 .49223 L p .739 g
F P
0 g
s
.61467 .53942 m .72403 .54593 L .68069 .51478 L p .635 g
F P
0 g
s
.68069 .51478 m .59165 .52388 L .61467 .53942 L p .635 g
F P
0 g
s
.76721 .78423 m .74919 .81868 L .82953 .83593 L p .286 g
F P
0 g
s
.82953 .83593 m .85299 .78852 L .76721 .78423 L p .286 g
F P
0 g
s
.17335 .96404 m .22379 .97779 L .2835 .99261 L p .384 g
F P
0 g
s
.33772 .95992 m .2835 .99261 L .22379 .97779 L p .815 g
F P
0 g
s
.2835 .99261 m .33772 .95992 L .39564 .93309 L p .888 g
F P
0 g
s
.39564 .93309 m .43332 .86386 L .46928 .81263 L p .857 g
F P
0 g
s
.48405 .73298 m .46928 .81263 L .43332 .86386 L p .792 g
F P
0 g
s
.79144 .66569 m .84017 .72786 L .88226 .6926 L p .366 g
F P
0 g
s
.88226 .6926 m .82551 .62908 L .79144 .66569 L p .366 g
F P
0 g
s
.43332 .86386 m .39564 .93309 L .33772 .95992 L p .853 g
F P
0 g
s
.6134 .4035 m .55814 .49223 L .52447 .49934 L p .707 g
F P
0 g
s
.52447 .49934 m .54898 .42167 L .6134 .4035 L p .707 g
F P
0 g
s
.43724 .78875 m .48214 .67551 L .48529 .67358 L p .79 g
F P
0 g
s
.48529 .67358 m .44973 .78806 L .43724 .78875 L p .79 g
F P
0 g
s
.43332 .86386 m .46688 .76908 L .48405 .73298 L p .792 g
F P
0 g
s
.4824 .65575 m .43802 .74126 L .45119 .70038 L p .849 g
F P
0 g
s
.43802 .74126 m .4824 .65575 L .48107 .66925 L p .841 g
F P
0 g
s
.50295 .35503 m .50303 .32152 L .57855 .22603 L p .355 g
F P
0 g
s
.55814 .49223 m .6134 .4035 L .67555 .41018 L p .739 g
F P
0 g
s
.43291 .77216 m .48107 .66925 L .48214 .67551 L p .819 g
F P
0 g
s
.48214 .67551 m .43724 .78875 L .43291 .77216 L p .819 g
F P
0 g
s
.48107 .66925 m .43291 .77216 L .43802 .74126 L p .841 g
F P
0 g
s
.43704 .55439 m .42755 .56649 L .32643 .59418 L p .514 g
F P
0 g
s
.32643 .59418 m .35046 .56704 L .43704 .55439 L p .514 g
F P
0 g
s
.45716 .61971 m .47252 .61816 L .43628 .68066 L p .703 g
F P
0 g
s
.46872 .65432 m .43628 .68066 L .47252 .61816 L p .73 g
F P
0 g
s
.11853 .85756 m .14249 .90745 L .17335 .96404 L p .428 g
F P
0 g
s
.22379 .97779 m .17335 .96404 L .14249 .90745 L p .525 g
F P
0 g
s
.15317 .7177 m .2024 .65669 L .26346 .60324 L p F P
s
.55006 .49826 m .52462 .51913 L .55854 .51649 L p .725 g
F P
0 g
s
.52462 .51913 m .55006 .49826 L .50109 .51479 L p .813 g
F P
0 g
s
.77236 .23274 m .73879 .31677 L .65512 .31062 L p .684 g
F P
0 g
s
.65512 .31062 m .67755 .22946 L .77236 .23274 L p .684 g
F P
0 g
s
.67905 .17356 m .67755 .22946 L .57768 .27273 L p .498 g
F P
0 g
s
.57768 .27273 m .57855 .22603 L .67905 .17356 L p .498 g
F P
0 g
s
.56755 .3417 m .54898 .42167 L .50104 .46094 L p .614 g
F P
0 g
s
.50104 .46094 m .50227 .40408 L .56755 .3417 L p .614 g
F P
0 g
s
.59165 .52388 m .68069 .51478 L .61642 .49749 L p .667 g
F P
0 g
s
.61642 .49749 m .55854 .51649 L .59165 .52388 L p .667 g
F P
0 g
s
.73491 .58197 m .82551 .62908 L .81293 .58163 L p .462 g
F P
0 g
s
.81293 .58163 m .72403 .54593 L .73491 .58197 L p .462 g
F P
0 g
s
.55854 .51649 m .61642 .49749 L .55006 .49826 L p .725 g
F P
0 g
s
.43628 .68066 m .46872 .65432 L .45119 .70038 L p .762 g
F P
0 g
s
.39708 .88865 m .44973 .78806 L .46688 .76908 L p .806 g
F P
0 g
s
.46688 .76908 m .43332 .86386 L .39708 .88865 L p .806 g
F P
0 g
s
.57855 .22603 m .57768 .27273 L .50295 .35503 L p .355 g
F P
0 g
s
.33772 .95992 m .39708 .88865 L .43332 .86386 L p .853 g
F P
0 g
s
.50109 .51479 m .50242 .51734 L .477 .54142 L p .826 g
F P
0 g
s
.47375 .55156 m .477 .54142 L .50242 .51734 L p F P
s
.50227 .40408 m .50295 .35503 L .57768 .27273 L p .527 g
F P
0 g
s
.42755 .56649 m .42567 .58262 L .32102 .62693 L p .572 g
F P
0 g
s
.32102 .62693 m .32643 .59418 L .42755 .56649 L p .572 g
F P
0 g
s
.4424 .6127 m .45716 .61971 L .39827 .68958 L p .676 g
F P
0 g
s
.43628 .68066 m .39827 .68958 L .45716 .61971 L p .703 g
F P
0 g
s
.74919 .81868 m .70858 .83121 L .7754 .85936 L p .187 g
F P
0 g
s
.7754 .85936 m .82953 .83593 L .74919 .81868 L p .187 g
F P
0 g
s
.57768 .27273 m .56755 .3417 L .50227 .40408 L p .527 g
F P
0 g
s
.73879 .31677 m .67555 .41018 L .6134 .4035 L p .719 g
F P
0 g
s
.6134 .4035 m .65512 .31062 L .73879 .31677 L p .719 g
F P
0 g
s
.22379 .97779 m .29402 .95143 L .33772 .95992 L p .815 g
F P
0 g
s
.39708 .88865 m .33772 .95992 L .29402 .95143 L p .829 g
F P
0 g
s
.15317 .7177 m .14021 .77831 L .11853 .85756 L p F P
s
.14249 .90745 m .11853 .85756 L .14021 .77831 L p .239 g
F P
0 g
s
.50204 .63785 m .50083 .65831 L .48387 .61134 L p .044 g
F P
0 g
s
.42567 .58262 m .43111 .59927 L .33394 .65828 L p .613 g
F P
0 g
s
.33394 .65828 m .32102 .62693 L .42567 .58262 L p .613 g
F P
0 g
s
.43111 .59927 m .4424 .6127 L .36171 .68108 L p .647 g
F P
0 g
s
.39827 .68958 m .36171 .68108 L .4424 .6127 L p .676 g
F P
0 g
s
.37025 .88559 m .43724 .78875 L .44973 .78806 L p .814 g
F P
0 g
s
.44973 .78806 m .39708 .88865 L .37025 .88559 L p .814 g
F P
0 g
s
.84017 .72786 m .85299 .78852 L .89786 .76093 L p .151 g
F P
0 g
s
.89786 .76093 m .88226 .6926 L .84017 .72786 L p .151 g
F P
0 g
s
.65512 .31062 m .6134 .4035 L .54898 .42167 L p .667 g
F P
0 g
s
.54898 .42167 m .56755 .3417 L .65512 .31062 L p .667 g
F P
0 g
s
.36171 .68108 m .33394 .65828 L .43111 .59927 L p .647 g
F P
0 g
s
.22442 .64749 m .26346 .60324 L .35046 .56704 L p .278 g
F P
0 g
s
.50242 .51734 m .50322 .53691 L .47375 .55156 L p F P
s
.4731 .56741 m .47375 .55156 L .50322 .53691 L p .031 g
F P
0 g
s
.26346 .60324 m .22442 .64749 L .15317 .7177 L p F P
s
.45119 .70038 m .39877 .74619 L .43628 .68066 L p .762 g
F P
0 g
s
.39877 .74619 m .45119 .70038 L .43802 .74126 L p .788 g
F P
0 g
s
.70858 .83121 m .65352 .81741 L .70195 .85165 L p .15 g
F P
0 g
s
.70195 .85165 m .7754 .85936 L .70858 .83121 L p .15 g
F P
0 g
s
.67755 .22946 m .65512 .31062 L .56755 .3417 L p .61 g
F P
0 g
s
.56755 .3417 m .57768 .27273 L .67755 .22946 L p .61 g
F P
0 g
s
.14249 .90745 m .1991 .92757 L .22379 .97779 L p .525 g
F P
0 g
s
.29402 .95143 m .22379 .97779 L .1991 .92757 L p .738 g
F P
0 g
s
.29402 .95143 m .37025 .88559 L .39708 .88865 L p .829 g
F P
0 g
s
.50242 .51734 m .50109 .51479 L .55006 .49826 L p F P
s
.36043 .85681 m .43291 .77216 L .43724 .78875 L p .815 g
F P
0 g
s
.43724 .78875 m .37025 .88559 L .36043 .85681 L p .815 g
F P
0 g
s
.14021 .77831 m .15317 .7177 L .22442 .64749 L p .207 g
F P
0 g
s
.50297 .60535 m .50204 .63785 L .47884 .60123 L p .058 g
F P
0 g
s
.47884 .60123 m .47497 .58533 L .50297 .60535 L p .058 g
F P
0 g
s
.50322 .53691 m .50341 .56876 L .4731 .56741 L p .031 g
F P
0 g
s
.47497 .58533 m .4731 .56741 L .50341 .56876 L p .052 g
F P
0 g
s
.43802 .74126 m .37081 .80774 L .39877 .74619 L p .788 g
F P
0 g
s
.37081 .80774 m .43802 .74126 L .43291 .77216 L p .806 g
F P
0 g
s
.39827 .68958 m .43628 .68066 L .39877 .74619 L p .713 g
F P
0 g
s
.50341 .56876 m .50297 .60535 L .47497 .58533 L p .052 g
F P
0 g
s
.43291 .77216 m .36043 .85681 L .37081 .80774 L p .806 g
F P
0 g
s
.72403 .54593 m .81293 .58163 L .75387 .53724 L p .381 g
F P
0 g
s
.75387 .53724 m .68069 .51478 L .72403 .54593 L p .381 g
F P
0 g
s
.82551 .62908 m .88226 .6926 L .86945 .64066 L p .179 g
F P
0 g
s
.86945 .64066 m .81293 .58163 L .82551 .62908 L p .179 g
F P
0 g
s
.14021 .77831 m .16393 .82665 L .14249 .90745 L p .239 g
F P
0 g
s
.1991 .92757 m .14249 .90745 L .16393 .82665 L p .555 g
F P
0 g
s
.1991 .92757 m .27698 .91086 L .29402 .95143 L p .738 g
F P
0 g
s
.37025 .88559 m .29402 .95143 L .27698 .91086 L p .794 g
F P
0 g
s
.27698 .91086 m .36043 .85681 L .37025 .88559 L p .794 g
F P
0 g
s
.32643 .59418 m .32102 .62693 L .21477 .6957 L p .445 g
F P
0 g
s
.21477 .6957 m .22442 .64749 L .32643 .59418 L p .445 g
F P
0 g
s
.85299 .78852 m .82953 .83593 L .87124 .81985 L p .014 g
F P
0 g
s
.87124 .81985 m .89786 .76093 L .85299 .78852 L p .014 g
F P
0 g
s
.54687 .70634 m .5276 .70809 L .49958 .66138 L p .08 g
F P
0 g
s
.22442 .64749 m .21477 .6957 L .14021 .77831 L p .207 g
F P
0 g
s
.36171 .68108 m .39827 .68958 L .33736 .76817 L p .665 g
F P
0 g
s
.39877 .74619 m .33736 .76817 L .39827 .68958 L p .713 g
F P
0 g
s
.16393 .82665 m .14021 .77831 L .21477 .6957 L p .448 g
F P
0 g
s
.33736 .76817 m .39877 .74619 L .37081 .80774 L p .731 g
F P
0 g
s
.57073 .50312 m .55006 .49826 L .61642 .49749 L p .037 g
F P
0 g
s
.55006 .49826 m .57073 .50312 L .50242 .51734 L p F P
s
.2926 .846 m .37081 .80774 L .36043 .85681 L p .759 g
F P
0 g
s
.36043 .85681 m .27698 .91086 L .2926 .846 L p .759 g
F P
0 g
s
.16393 .82665 m .21859 .8517 L .1991 .92757 L p .555 g
F P
0 g
s
.27698 .91086 m .1991 .92757 L .21859 .8517 L p .687 g
F P
0 g
s
.68069 .51478 m .75387 .53724 L .66399 .50886 L p .335 g
F P
0 g
s
.66399 .50886 m .61642 .49749 L .68069 .51478 L p .335 g
F P
0 g
s
.55914 .75374 m .63031 .81417 L .63746 .82303 L p .136 g
F P
0 g
s
.63031 .81417 m .55914 .75374 L .54678 .73913 L p .117 g
F P
0 g
s
.32102 .62693 m .33394 .65828 L .23454 .73791 L p .541 g
F P
0 g
s
.23454 .73791 m .21477 .6957 L .32102 .62693 L p .541 g
F P
0 g
s
.37081 .80774 m .2926 .846 L .33736 .76817 L p .731 g
F P
0 g
s
.50322 .53691 m .50242 .51734 L .57073 .50312 L p .096 g
F P
0 g
s
.54678 .73913 m .60405 .78866 L .63031 .81417 L p .117 g
F P
0 g
s
.82953 .83593 m .7754 .85936 L .80909 .85542 L p F P
s
.80909 .85542 m .87124 .81985 L .82953 .83593 L p F P
s
.21859 .8517 m .2926 .846 L .27698 .91086 L p .687 g
F P
0 g
s
.33394 .65828 m .36171 .68108 L .27861 .76445 L p .61 g
F P
0 g
s
.33736 .76817 m .27861 .76445 L .36171 .68108 L p .665 g
F P
0 g
s
.61642 .49749 m .66399 .50886 L .57073 .50312 L p .037 g
F P
0 g
s
.27861 .76445 m .23454 .73791 L .33394 .65828 L p .61 g
F P
0 g
s
.21477 .6957 m .23454 .73791 L .16393 .82665 L p .448 g
F P
0 g
s
.21859 .8517 m .16393 .82665 L .23454 .73791 L p .579 g
F P
0 g
s
.7754 .85936 m .70195 .85165 L .72524 .8576 L p .498 g
F P
0 g
s
.63746 .82303 m .72524 .8576 L .70195 .85165 L p .315 g
F P
0 g
s
.72524 .8576 m .80909 .85542 L .7754 .85936 L p .498 g
F P
0 g
s
.88226 .6926 m .89786 .76093 L .88528 .71154 L p .475 g
F P
0 g
s
.88528 .71154 m .86945 .64066 L .88226 .6926 L p .475 g
F P
0 g
s
.27861 .76445 m .33736 .76817 L .2926 .846 L p .666 g
F P
0 g
s
.2926 .846 m .21859 .8517 L .27861 .76445 L p .666 g
F P
0 g
s
.56578 .6787 m .54687 .70634 L .50083 .65831 L p .268 g
F P
0 g
s
.50083 .65831 m .50204 .63785 L .56578 .6787 L p .268 g
F P
0 g
s
.23454 .73791 m .27861 .76445 L .21859 .8517 L p .579 g
F P
0 g
s
.81293 .58163 m .86945 .64066 L .80068 .58751 L p .247 g
F P
0 g
s
.80068 .58751 m .75387 .53724 L .81293 .58163 L p .247 g
F P
0 g
s
.61051 .75552 m .56608 .75166 L .5276 .70809 L p .27 g
F P
0 g
s
.5276 .70809 m .54687 .70634 L .61051 .75552 L p .27 g
F P
0 g
s
.72524 .8576 m .63746 .82303 L .63031 .81417 L p .463 g
F P
0 g
s
.57073 .50312 m .58365 .53232 L .50322 .53691 L p .096 g
F P
0 g
s
.50341 .56876 m .50322 .53691 L .58365 .53232 L p .269 g
F P
0 g
s
.58028 .63241 m .56578 .6787 L .50204 .63785 L p .365 g
F P
0 g
s
.50204 .63785 m .50297 .60535 L .58028 .63241 L p .365 g
F P
0 g
s
.58365 .53232 m .57073 .50312 L .66399 .50886 L p .231 g
F P
0 g
s
.89786 .76093 m .87124 .81985 L .85891 .77854 L p .896 g
F P
0 g
s
.85891 .77854 m .88528 .71154 L .89786 .76093 L p .896 g
F P
0 g
s
.58365 .53232 m .58687 .57918 L .50341 .56876 L p .269 g
F P
0 g
s
.50297 .60535 m .50341 .56876 L .58687 .57918 L p .356 g
F P
0 g
s
.63031 .81417 m .7152 .83884 L .72524 .8576 L p .463 g
F P
0 g
s
.7152 .83884 m .63031 .81417 L .60405 .78866 L p .478 g
F P
0 g
s
.58687 .57918 m .58028 .63241 L .50297 .60535 L p .356 g
F P
0 g
s
.67299 .80207 m .60405 .78866 L .56608 .75166 L p .41 g
F P
0 g
s
.75387 .53724 m .80068 .58751 L .69425 .54851 L p .303 g
F P
0 g
s
.69425 .54851 m .66399 .50886 L .75387 .53724 L p .303 g
F P
0 g
s
.80909 .85542 m .72524 .8576 L .7152 .83884 L p .744 g
F P
0 g
s
.60405 .78866 m .67299 .80207 L .7152 .83884 L p .478 g
F P
0 g
s
.66399 .50886 m .69425 .54851 L .58365 .53232 L p .231 g
F P
0 g
s
.87124 .81985 m .80909 .85542 L .79738 .82518 L p .899 g
F P
0 g
s
.79738 .82518 m .85891 .77854 L .87124 .81985 L p .899 g
F P
0 g
s
.7152 .83884 m .79738 .82518 L .80909 .85542 L p .744 g
F P
0 g
s
.86945 .64066 m .88528 .71154 L .81368 .65528 L p .644 g
F P
0 g
s
.81368 .65528 m .80068 .58751 L .86945 .64066 L p .644 g
F P
0 g
s
.65421 .72662 m .61051 .75552 L .54687 .70634 L p .484 g
F P
0 g
s
.54687 .70634 m .56578 .6787 L .65421 .72662 L p .484 g
F P
0 g
s
.58687 .57918 m .58365 .53232 L .69425 .54851 L p .437 g
F P
0 g
s
.79738 .82518 m .7152 .83884 L .67299 .80207 L p .687 g
F P
0 g
s
.74046 .77844 m .67299 .80207 L .61051 .75552 L p .6 g
F P
0 g
s
.61051 .75552 m .65421 .72662 L .74046 .77844 L p .6 g
F P
0 g
s
.68757 .67274 m .65421 .72662 L .56578 .6787 L p .533 g
F P
0 g
s
.56578 .6787 m .58028 .63241 L .68757 .67274 L p .533 g
F P
0 g
s
.88528 .71154 m .85891 .77854 L .79153 .72477 L p .757 g
F P
0 g
s
.79153 .72477 m .81368 .65528 L .88528 .71154 L p .757 g
F P
0 g
s
.80068 .58751 m .81368 .65528 L .70235 .60787 L p .548 g
F P
0 g
s
.70235 .60787 m .69425 .54851 L .80068 .58751 L p .548 g
F P
0 g
s
.67299 .80207 m .74046 .77844 L .79738 .82518 L p .687 g
F P
0 g
s
.69425 .54851 m .70235 .60787 L .58687 .57918 L p .437 g
F P
0 g
s
.58028 .63241 m .58687 .57918 L .70235 .60787 L p .514 g
F P
0 g
s
.85891 .77854 m .79738 .82518 L .74046 .77844 L p .758 g
F P
0 g
s
.70235 .60787 m .68757 .67274 L .58028 .63241 L p .514 g
F P
0 g
s
.74046 .77844 m .79153 .72477 L .85891 .77854 L p .758 g
F P
0 g
s
.79153 .72477 m .74046 .77844 L .65421 .72662 L p .643 g
F P
0 g
s
.65421 .72662 m .68757 .67274 L .79153 .72477 L p .643 g
F P
0 g
s
.81368 .65528 m .79153 .72477 L .68757 .67274 L p .627 g
F P
0 g
s
.68757 .67274 m .70235 .60787 L .81368 .65528 L p .627 g
F P
0 g
s
.25 Mabswid
.76528 0 m
.91897 .25043 L
s
.91897 .25043 m
1 .91568 L
s
1 .91568 m
.82346 .69758 L
s
.82346 .69758 m
.76528 0 L
s
.07962 .26913 m
0 .93161 L
s
0 .93161 m
.82346 .69758 L
s
.82346 .69758 m
.76528 0 L
s
.76528 0 m
.07962 .26913 L
s
.07962 .26913 m
.76528 0 L
s
.08139 .26844 m
.08631 .27346 L
s
gsave
.07154 .25839 -81.4146 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.23824 .20687 m
.24291 .21213 L
s
gsave
.22889 .19635 -80.474 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.40364 .14195 m
.40803 .14745 L
s
gsave
.39487 .13095 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.57832 .07338 m
.58238 .07913 L
s
gsave
.5702 .06189 -73.9015 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.76307 .00087 m
.76676 .00686 L
s
gsave
.7557 -0.01112 -73.2117 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.11983 .25335 m
.12275 .2564 L
s
.15879 .23806 m
.16167 .24114 L
s
.19825 .22257 m
.20109 .22569 L
s
.27876 .19096 m
.28152 .19416 L
s
.31983 .17485 m
.32255 .17807 L
s
.36145 .15851 m
.36413 .16177 L
s
.44641 .12516 m
.449 .1285 L
s
.48977 .10814 m
.49231 .11151 L
s
.53374 .09088 m
.53622 .09429 L
s
.62353 .05564 m
.62591 .05912 L
s
.66938 .03764 m
.67171 .04116 L
s
.71589 .01938 m
.71816 .02294 L
s
gsave
.35102 .07594 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  PageBreakAbove->Automatic,
  ImageSize->{549.25, 190.5},
  ImageMargins->{{43.75, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40008U0000_Q000`40O003h00Oogooogoo9goo003o
OoooOolWOol00?mooomoobMoo`00ogooogoo9goo003oOoooOolWOol00?mooomoobMoo`00ogooogoo
9goo003oOoooOolWOol00?mooomoobMoo`00ogoo8Goo00<007ooOol0ogoo0goo003oOolQOol00`00
Oomoo`30Ool400001Goo001oogoo0000>Goo003oOolQOol00`00Oomoo`30Ool01000Oomoo`000goo
0`00>7oo001LOol400000goo001oo`02002HOol00`00Ool0000POol00`00Oomoo`30Ool01000Oomo
o`000goo0P00>Goo001LOol01000Oomoo`000Woo00@007ooOol009Qoo`03001oogoo01ioo`80015o
o`03001oogoo0:ioo`04001oogoo0003Ool2000iOol005aoo`04001oogoo0005Ool00`00Oomoo`2F
Ool00`00Oomoo`09Ool400000goo001oo`02000=Ool00`00Oomoo`08Ool40003Ool00`00Oomoo`2^
Ool01000Oomoo`0017oo00<007ooOol02Woo100000=oo`0000000`008goo001LOol01000Oomoo`00
0Woo0`00VGoo00<007ooOol02Goo00@007ooOol0009oo`04001oogoo000NOol00`00Oomoo`2^Ool4
0004Ool00`00Oomoo`06Ool50002Ool01@00Oomoogoo00009Goo001LOol01000Oomoo`000Woo00<0
07ooOol0V7oo00<007oo00002Woo00@007ooOol000Eoo`03001oogoo00ioo`<000]oo`03001oogoo
0<=oo`04001oogoo0004Ool00`00Oomoo`0ROol003aoo`03001oo`0001eoo`@000=oo`<00:Aoo`04
001oogoo0002Ool3000?Ool60009Ool20035Ool01000Oomoo`001Goo00<007ooOol08Goo000mOol0
0`00Oomoo`08Ool400000goo0000000300000goo0000000SOol00`00Oomoo`2?Ool01000Oomoo`00
0Woo00<007ooOol037oo10000goo0P002Woo00<007ooOol0^Goo0P0027oo00@007ooOol0009oo`04
001oogoo000SOol003eoo`03001oogoo00Qoo`04001oogoo0003Ool01000Oomoo`000Woo00<007oo
Ool04Goo0`001Goo10000goo00<007ooOol0R7oo10000goo10000goo0`0027oo10001Woo0`00`goo
0P0000=oo`0000001goo10000goo0`008goo000mOol00`00Oomoo`08Ool01000Oomoo`0017oo00<0
07ooOol00Woo00<007ooOol047oo1@002goo00<007ooOol0R7oo00@007ooOol001=oo`@000Qoo`04
001oogoo002oOol30003Ool2000dOol003aoo`03001oo`0000Uoo`04001oogoo0005Ool4000@Ool3
0003Ool2000;Ool00`00Oomoo`28Ool01000Oomoo`0047oo10002goo00@007ooOol009moo`@00003
Ool0000000<001=oo`<000Ioo`03001oo`0003=oo`00B7oo00@007ooOol0009oo`03001oogoo0080
011oo`8000Ioo`03001oo`0000Uoo`8008Yoo`04001oogoo000>Ool3000>Ool01000Oomoogoo0P00
WGoo00@007ooOol000=oo`03001oogoo015oo`8000Uoo`03001oo`0003=oo`00B7oo10000goo0`00
00Aoo`00000000Uoo`D000Moo`04001oogoo0009Ool00`00Oomoo`28Ool01000Oomoo`0037oo0`00
47oo00@007ooOomoo`8000]oo`@00003Ool007oo0080071oo`03001oo`0001Ioo`04001oogoo0004
Ool00`00Oomoo`0=Ool3000;Ool01000Oomoo`00<Woo001MOol4000:Ool01@00Oomoogoo0000P7oo
100000=oo`00Ool00P002Woo10002Goo0`004goo00<007ooOol00Woo0P001Goo100000Eoo`00Oomo
o`00009oo`04001oogoo001`Ool00`00Oomoo`0EOol01000Oomoo`001Goo00<007ooOol02Woo0P00
00=oo`00Ool02goo00<007ooOol00P002Woo100000Eoo`00Oomoo`0001ioo`00@goo100057oo0`00
3Goo00D007ooOomoo`0007aoo`D0009oo`04001oogoo0002Ool00`00Oomoo`0BOol2000FOol00`00
Oomoo`02Ool2000:Ool01000Oomoo`001Goo00<007ooOol0KWoo00<007ooOol05Goo00@007ooOol0
009oo`04001oogoo0009Ool3000AOol01000Oomoo`001Goo100000Eoo`00Oomoo`0000Aoo`03001o
ogoo01aoo`00@goo00@007ooOol0015oo`<0011oo`04001oogooOol2001oOol01000Oomoo`001Goo
00<007ooOol03goo0`0067oo00<007ooOol017oo00<007ooOol01goo00@007ooOol0009oo`<0075o
o`03001oogoo01Eoo`@000=oo`<000Ioo`<001Aoo`03001oogoo008000Uoo`04001oogoo0004Ool0
0`00Oomoo`0LOol004=oo`04001oogoo000?Ool20002Ool00`00Oomoo`0>Ool00`00Oomoo`03Ool0
0`00Oomoo`1lOol01000Oomoo`000Woo0`003goo0`006goo00<007ooOol00goo00<007oo000027oo
00@007ooOol0009oo`03001oogoo009oo`03001oogoo06]oo`03001oo`0002Aoo`8001Moo`04001o
ogooOol20008Ool01000Oomoo`0017oo00<007ooOol077oo0013Ool01000Oomoo`0037oo0`005Goo
00<007ooOol017oo00<007ooOol0Ngoo00@007ooOol0009oo`03001oogoo00eoo`8001ioo`03001o
ogoo00Eoo`8000Moo`@000=oo`<00003Ool007oo091oo`<001Uoo`04001oogooOol20008Ool01000
Oomoo`000goo0P0000=oo`00Ool06goo0013Ool01000Oomoo`002Goo10005goo00<007ooOol017oo
00<007ooOol0MWoo00<007ooOol00Woo10000goo0`002Goo0`0087oo00<007ooOol01Woo00<007oo
Ool047oo00<007ooOol0SGoo0P0000=oo`00Ool06Goo00<007ooOol00Woo0P001goo100017oo00<0
07oo000077oo0013Ool40007Ool3000JOol00`00Oomoo`05Ool00`00Oomoo`1_Ool40002Ool00`00
Oomoo`0COol2000SOol00`00Oomoo`06Ool2000@Ool00`00Ool0001aOol4000EOol4000NOol00`00
Oomoo`03Ool00`00Oomoo`0?Ool00`00Oomoo`0JOol002Uoo`@00003Ool0000000<00003Ool00000
01Ioo`8001eoo`03001oogoo00Eoo`8007Ioo`03001oogoo011oo`<002Eoo`03001oogoo00Moo`03
001oogoo00ioo`03001oo`00075oo`04001oogoo000COol4000POol00`00Oomoo`03Ool2000?Ool0
0`00Ool0000KOol002Eoo`D0009oo`05001oogooOol00002Ool01000Oomoo`004Woo0`007goo00<0
07ooOol01goo00<007ooOol0Lgoo00<007ooOol03Goo0`00:7oo00<007ooOol027oo00<007ooOol0
17oo10001Goo00<007oo0000LGoo00@007ooOol0015oo`8002Aoo`03001oogoo00Aoo`03001oogoo
00eoo`03001oo`0001]oo`00:Goo00@007ooOol000Aoo`03001oogoo009oo`03001oogoo00eoo`<0
0003Ool007oo01moo`03001oogoo00Moo`03001oogoo079oo`8000eoo`<002Yoo`03001oogoo00Uo
o`03001oogoo00=oo`04001oogoo0004Ool01@00Oomoogoo0000L7oo00@007ooOol000ioo`<002Io
o`03001oogoo00Aoo`8000Eoo`@000Eoo`03001oo`0001]oo`00:Goo00@007ooOol000Eoo`@000io
o`800003Ool007oo029oo`03001oogoo00Qoo`03001oogoo079oo`03001oogoo00Qoo`<002eoo`03
001oogoo00Uoo`8000Aoo`04001oogoo001iOol01000Oomoo`0037oo0P00:Goo00<007ooOol01Woo
00<007ooOol00Woo00@007ooOol000Aoo`05001oogooOol0000JOol002Uoo`04001oogoo0002Ool0
0`00Oomoo`02000=Ool4000VOol00`00Oomoo`08Ool2001kOol4000`Ool00`00Oomoo`09Ool00`00
Oomoo`02Ool01000Oomoo`00NGoo10002Goo0`00:goo00<007ooOol01Goo0P0017oo00@007ooOol0
02=oo`007Woo100000=oo`00Ool00P000Woo10000goo0`0000Aoo`00000000Qoo`<002Uoo`03001o
ogoo00Yoo`03001oogoo07Ioo`<003=oo`03001oogoo00Uoo`8000=oo`04001oogoo0024Ool20000
0goo001oo`0[Ool00`00Oomoo`07Ool01@00Oomoogoo00000Woo00<007ooOol08Goo000JOol50002
Ool01000Oomoo`000Woo00<007ooOol04Woo0`00;7oo00<007ooOol02Woo00<007ooOol0M7oo0`00
=Goo00<007ooOol02Woo0P000Woo1000PGoo0`00<Goo00<007ooOol01Goo0P000goo00@007ooOol0
02=oo`007Woo00@007ooOol000Eoo`03001oogoo011oo`8002moo`03001oogoo00]oo`03001oogoo
07=oo`8003Ioo`03001oogoo00Yoo`8008Aoo`@002Uoo`H000Aoo`03001oogoo00Moo`03001oogoo
00@002=oo`007Woo00@007ooOol0009oo`<0011oo`<00003Ool007oo02ioo`03001oogoo00]oo`80
07Aoo`<003Eoo`03001oogoo00aoo`03001oogoo05moo`@00003Ool0000000<001Ioo`8002Ioo`L0
00HU:@80009oo`03001oogoo00Qoo`03001oogoo02Ioo`007Woo00@007ooOol0009oo`03001oogoo
00eoo`<003Aoo`03001oogoo00eoo`03001oogoo00Qoo`@00003Ool007oo0080061oo`04001oogoo
000dOol00`00Oomoo`0<Ool00`00Oomoo`1JOol400001Goo001oogoo00000goo00<007ooOol04goo
0`009Woo0P0000<DY@009BT02bDY0P0000=oo`00Ool02Goo00<007ooOol09Woo000NOol40003Ool3
000:Ool200000goo001oo`0dOol00`00Oomoo`0=Ool00`00Oomoo`04Ool50002Ool01000Oomoo`00
0Woo00<007ooOol0EGoo100017oo00D007ooOomoo`0003=oo`03001oogoo00eoo`8000Eoo`@00003
Ool007oo0080055oo`04001oogoo0004Ool00`00Oomoo`0@Ool2000WOol200045:D00`009BTU:@0:
9BT3000;Ool00`00Oomoo`03Ool400001Goo001oogoo00006Goo000_Ool4000hOol00`00Oomoo`0>
Ool00`00Oomoo`07Ool01000Oomoo`001Goo00<007ooOol0D7oo10000Woo00<007ooOol017oo00D0
07ooOomoo`0003=oo`03001oogoo00ioo`03001oogoo00=oo`04001oogoo0002Ool01000Oomoo`00
D7oo00@007ooOol000Eoo`03001oogoo00aoo`<002Qoo`8000HDYA4000Uoo`03001oogoo00=oo`04
001oogoo0004Ool00`00Oomoo`0GOol002eoo`<003]oo`03001oogoo00moo`03001oogoo00Ioo`04
001oogoo0002Ool3001JOol00`00Oomoo`03Ool00`00Oomoo`02Ool00`00Oomoo`0`Ool00`00Oomo
o`0>Ool20004Ool01000Oomoo`001Goo00<007ooOol0CWoo00@007ooOol0009oo`04001oogoo000;
Ool300000goo001oo`0VOol200065:D3000<CW<300000b48001oo`08Ool00`00Oomoo`02Ool01000
Oomoo`0017oo00<007ooOol05goo000[Ool3000mOol00`00Oomoo`0>Ool00`00Ool00007Ool01000
Oomoo`000Woo00<007ooOol0Fgoo00@007ooOomoo`8000Eoo`03001oogoo01aoo`/000Qoo`03001o
ogoo00moo`8000=oo`04001oogoo0002Ool3001AOol40003Ool30009Ool2000[Ool200055:D20002
?Nl00`00CW=>L`0:CW<2000015[F000Q20002Goo00D007ooOomoo`00009oo`03001oogoo009oo`03
001oogoo01Moo`00:goo0P00?Woo00<007ooOol047oo00<007ooOol01Goo10000goo0`00Egoo00@0
07ooOol000=oo`03001oogoo00Eoo`03001oogoo01Uoo`8000035:D002DY008U:@06000]Jbe[;F/0
03F]0P001Woo00<007ooOol03goo0P000goo00@007ooOol0009oo`03001oogoo065oo`@002]oo`03
000DYABU00<DY@8000@mk`03001>Ldic00Y>L`80009JeP<000Moo`8000=oo`04001oogoo0003Ool2
000IOol002]oo`03001oo`0003eoo`03001oogoo015oo`03001oogoo06Ioo`<000=oo`03001oogoo
00Ioo`03001oogoo01Ioo`8000<DY@05000U:BDY9BT00003;F/00`00=Jde[@030004Ool00`00Oomo
o`0AOol00`00Oomoo`040003Ool3001NOol2000]Ool200045:D20006?Nl00`00CW=>L`0:CW<20004
F]H20008Ool00`00Oomoo`040004Ool00`00Oomoo`0GOol001aoo`@00003Ool007oo008000Ioo`04
001oogoo000lOol00`00Oomoo`0BOol00`00Oomoo`1[Ool00`00Oomoo`07Ool00`00Oomoo`0DOol0
0`00248000035:D01@009BTU:BDY000012e[00H003F]=Jd003F]0003Ool00`00Oomoo`0AOol00`00
Oomoo`1UOol4000]Ool00`005:DDY@025:D20008?Nl00`00CW=>L`0:CW<20005F]H20007Ool00`00
Oomoo`0ROol001Moo`@00005Ool007ooOol00002Ool01000Oomoo`0017oo0P000goo00<007ooOol0
9goo2`001goo00<007ooOol04Goo0P00KGoo0P002Goo00<007ooOol04Woo00<000Q200000QBU3000
0SF]00@003F]=Jd0009oo`03001oogoo019oo`03001oogoo049oo`@00005Ool007ooOol0000FOol3
000`Ool200025:D2000:?Nl00`00CW=>L`0:CW<20007F]H00`00Oomoo`05Ool00`00Oomoo`0QOol0
01aoo`04001oogoo0005Ool00`00Oomoo`02Ool00`00Oomoo`03Ool00`00Oomoo`0NOol800000`Q2
0000000400027>L40004Ool00`00Oomoo`0DOol00`00Oomoo`1ZOol00`00Oomoo`09Ool00`00Oomo
o`0@Ool00`002488@P030002=Jd01@00?Nlmkcg_000014Ha0`0000<e[@00=Jd00P0000=oo`00Ool0
57oo0P00?Goo100000Eoo`00Oomoo`0000=oo`<001=oo`<0035oo`D000`mk`03001>Ldic00Y>L`03
001JeP0000IJeP03001oogoo00Ioo`03001oogoo021oo`0077oo00@007ooOol0009oo`<000Eoo`80
00Eoo`03001oogoo01]oo`8000032480000000H000@Li`03000e[@0000<0008Li`D001Moo`03001o
ogoo06Uoo`03001oogoo00Uoo`03001oogoo00moo`030008@P00008000@e[@05000mkcg_?Nl00004
AS400`00BU9:DP020002=Jd3000EOol00`00Oomoo`06Ool00`00Oomoo`0hOol01000Oomoo`000goo
0P004Goo0`00<goo1@0033g_3`0000=JeP00F]H01E[F00<007ooOol01Woo00<007ooOol00goo1000
00=oo`0000000`004goo000LOol01000Oomoo`000Woo00<007ooOol01Goo00<007ooOol01Goo00<0
07ooOol067oo1@002acW00<003F]=Jd013F]0`000T8@1@005Goo00<007ooOol0J7oo00<007ooOol0
2Woo00<007ooOol03Woo0P000RU:00<003F]=Jd00SF]00D003g_?Nlmk`0000A6<@04001:DTYBBU86
000FOol20006Ool00`00Oomoo`0hOol01000Oomoo`000goo0P003goo0P00=Woo0`0000<]J`00?Nl0
2Cg_0P0000=>L`00E[D02eJe0`001U[F0P0027oo00<007ooOol00Woo00@007ooOol000=oo`03001o
ogoo01=oo`0077oo10000goo0`0017oo00<007ooOol01Woo00<007ooOol05Woo0P003acW00<003F]
=Jd01cF]100000=2400024800P004goo00<007ooOol0J7oo0P0037oo00<007ooOol037oo0P000bU:
00D003F]=Jde[@0000@mk`030016<DHa0096<@05001:DTYBBU800002DY@3000GOol00`00Oomoo`04
Ool00`00Oomoo`0hOol01000Oomoo`0017oo00<007ooOol02Woo0`00=goo100000<]J`00?Nl01cg_
0P000Tic00<005JeE[D02UJe00<006<HHaP00P0015[F00<005kg000027oo00<007ooOol00Woo00@0
07ooOol000Aoo`03001oogoo019oo`00:Woo00<007ooOol01goo00<007ooOol04goo0`003QcW1000
2cF]100000<Li`0000004goo00<007ooOol0Igoo00<007ooOol037oo00<007ooOol02goo0P000bU:
00D003F]=Jde[@0000@mk`030016<DHa0096<@05001:DTYBBU800003DY@2000GOol20005Ool00`00
Oomoo`0hOol40004Ool00`00Oomoo`07Ool3000jOol30002;F/00`00?Nlmk`03?Nl30004CW<00`00
E[EF]@0:E[D01P00HaQS6000HaP000=JeP03001Nm`0000Qoo`8000=oo`04001oogoo0005Ool00`00
Oomoo`0AOol002Yoo`03001oogoo00Moo`03001oogoo019oo`@000PLi`H0008aS0030016<@0000<0
00Le[@04000e[CF]=Jd200000be[001oo`0BOol00`00Oomoo`1VOol00`00Oomoo`0=Ool00`00Oomo
o`09Ool00`005:D00003:DX01@00=Jde[CF]000013g_00<004HaAS400THa00D004YBBU9:DP0000=B
U0<001Ioo`8000Aoo`8004Yoo`80009oo`03001oogoo03Moo`<000<]J`04000mkcg_?Nl20007CW<0
0`00E[EF]@09E[D01@00HaQS66<H00000V<H00H005[FF]H005kg0005Ool300000goo001oo`02Ool0
1000Oomoo`000Woo00@007ooOol001=oo`00:Woo00<007ooOol027oo00<007ooOol047oo10000acW
1P00236<00<004HaAS400dHa0`0013F]00<003F]=Jd00SF]0P0000<]J`00Ool04Woo00<007ooOol0
4Woo00<007ooOol0D7oo00<007ooOol03Woo00<007ooOol01goo0P0000<DY@00:DX00RU:00D003F]
=Jde[@0000@mk`030016<DHa0096<@03001:DTYB009:DP03001BU5:D00@0019oo`<00003Ool007oo
00Aoo`03001oogoo04Ioo`8003eoo`<000@]J`04000mk`000008CW<00`00E[EF]@0:E[D01@00HaQS
66<H00000f<H0`0000ANm`00Oomoo`<000Aoo`04001oogooOol40003Ool3000COol002Uoo`8000]o
o`03001oogoo00eoo`T000haS0030016<DHa00I6<@D000@e[@050016<@00;F/0000BOol00`00Oomo
o`0BOol00`00Oomoo`1@Ool00`00Oomoo`0?Ool00`00Oomoo`06Ool200000aBU000YBP02:DX00`00
=Jde[@0=0003BU800`00DYABU004000?Ool30004Ool00`00Oomoo`1<Ool2000mOol20005;F/2000:
CW<00`00E[EF]@09E[D00`00HaQS6002HaP00`00HaQS6003HaP00`00G_MNm`020007Ool00`00Oomo
o`0NOol002Uoo`03001oogoo00]oo`03001oogoo00]oo`L0014aS0030016<DHa00Y6<@@00004=Jd0
04HaAS43000BOol00`00Oomoo`0AOol00`00Oomoo`1@Ool00`00Oomoo`0@Ool00`00Oomoo`04Ool0
1@00124001BU00000bU:0`0000=24000AS400dHa00<004icCW<00Tic00H005:D000004YB0003DY@2
00000e[F001oo`0:Ool30007Ool00`00Oomoo`1<Ool00`00Ool0000kOol30004;F/2000:CW<00`00
E[EF]@0:E[D00`00HaQS6002HaP00`00HaQS6003HaP01000G_MNm`002Goo00<007ooOol07Goo000Y
Ool2000=Ool00`00Oomoo`09Ool5000236<00`00<H`aS00?<H`00`00AS56<@0:AS401000AS56<DHa
0P0014Ha00<007ooOol04Goo00<007ooOol03goo00<007oo0000D7oo00<007ooOol04Woo00<007oo
Ool00Woo0`000QBU00<002U:00000P000d8@00<004HaAS400THa00<004icCW<00Tic00@005:DDYAB
U08000=BU0800003F]H007oo00Moo`<000Yoo`03001oogoo03eoo`@00005Ool007ooOol00006Ool0
1000Oomoo`00>Goo0`0012e[00<0048@00002Tic00<005JeE[D02UJe00<006<HHaP00V<H00<006<H
HaP00V<H0P000Ukg00<007ooOol01goo00<007ooOol07Goo000YOol00`00Oomoo`0=Ool00`00Oomo
o`06Ool6000236<00`00<H`aS00=<H`6000:AS400`00AS56<@02AS430003AS400`00Oomoo`0AOol0
0`00Oomoo`0>Ool00`00Ool0001@Ool00`00Oomoo`0BOol00`00Oomoo`02Ool300025:D20002>Lh0
1@00@Q12448@000014Ha00<004icCW<00Tic00<005:DDY@00U:D0`0000=BU000F]H00P001Goo0`00
3Goo00<007ooOol0>Goo1@000Woo00@007ooOomoo`<000Eoo`05001oogooOol0000gOol30004;F/0
0`00@Q00000;CW<00`00E[EF]@09E[D00`00HaQS6003HaP00`00HaQS6002HaP01@00Li`005kg0000
2Goo00<007ooOol07Goo000FOol400000goo0000000300000goo00000006Ool00`00Oomoo`0>Ool0
0`00Oomoo`04Ool5000436<00`00<H`aS007<H`60003?Nl00`00CW=>L`040006AS400`00AS56<@02
AS401000E[EF]@000dHa00<007ooOol03goo0P001Woo10001Woo00<007oo0000D7oo00<007ooOol0
4goo00@007ooOomoo`<000035:D0000000<icP050012448@@Q000004AS400`00CW=>L`02CW<00`00
DYABU002DY@00`00F]IJeP0200000e[F00000002Ool3000@Ool00`00Oomoo`0mOol01000Oomoo`00
0goo0P001Goo0P0017oo00<007ooOol047oo1P007Woo0P0012e[00@0048@@Q0000Y>L`03001F]EJe
00YF]@03001S66<H00=S6005001S66<HHaP00002Li`00`00G_L00009Ool00`00Oomoo`0MOol0015o
o`@00005Ool007ooOol00003Ool01000Oomoo`000Woo00<007ooOol00goo00<007ooOol03goo00<0
07ooOol00Woo1@001@aS00@0036<<H`aS0H000Tmk`03001>Ldic00A>L`D00003AS4004Ha00=6<@04
001F]EJeE[D20002AS400`00Oomoo`0@Ool00`00Oomoo`03Ool01000Oomoo`001Goo00D007ooOomo
o`0004moo`8001Eoo`05001oogoo00048@0200000be[000icP02>Lh01@00@Q12448@000014Ha00<0
04icCW<00Tic00D005:DDYABU00000AJePH001Aoo`03001oogoo03aoo`04001oogoo0003Ool20005
Ool00`00Oomoo`04Ool00`00Oomoo`0=Ool2000336<8000HOol30003;F/01@00@Q12448@00002Dic
1`001UJe00<006<HHaP01F<H00@006<HHaP0009cW003001Nm`0000Uoo`03001oogoo01eoo`005Woo
00@007ooOol000Aoo`03001oogoo009oo`03001oogoo00=oo`03001oogoo011oo`04001oogooOol4
000636<4000??Nl00`00CW=>L`09CW<40002AS400`00E[EF]@03E[D00`00AS56<@02000AOol00`00
Oomoo`02Ool01000Oomoo`00FGoo00<007ooOol05Goo00D007oo00048@00008]J`05000icSW>>Lh0
0004@Q001@00AS56<DHa00000dic00<005:DDY@00U:D00<005[FF]H00U[F00D006<H001JeP0001Eo
o`03001oogoo03aoo`04001oogoo0004Ool00`00Oomoo`03Ool00`00Oomoo`05Ool00`00Oomoo`0:
Ool2000736<4000248@2000FOol20003;F/00`00@Q124002@Q000`00CW=>L`05CW<200000eJe001N
m`04G_L70007HaP00`00HaP00003Li`2000:Ool00`00Oomoo`0MOol001Ioo`04001oogoo0005Ool4
0006Ool00`00Oomoo`0AOol00`00Ool00002000636<3000B?Nl00`00CW=>L`0:CW<00`00CW=>L`03
0006E[D01@00AS56<DHa00004Goo00D007ooOomoo`00009oo`03001oogoo055oo`03001oogoo00=o
o`03001oogoo01Ioo`@0008]J`03000icSW>008icP050012448@@Q000003AS401@00CW=>Ldic0000
15:D00@005[FF]IJeP800004HaP00000000EOol00`00Oomoo`0lOol40004Ool00`00Oomoo`03Ool0
0`00Oomoo`06Ool00`00Oomoo`07Ool;0003=Jd7000DOol01@00;F/]Jbe[00001D8@00<004icCW<0
0dic0P000UJe00<005kgG_L02Ekg00@006<H000000ES6003001S600000=cW08000Yoo`03001oogoo
01eoo`005Woo00@007ooOol0009oo`03001oogoo008000Moo`8001Aoo`<000H<H`04000YBRU:000B
?Nl00`00CW=>L`0:CW<00`00CW=>L`02CW<30005E[D20002AS400`00Oomoo`0=Ool20003Ool01000
Oomoo`00C7oo10000goo00<007ooOol00goo00<007ooOol05Woo10000Re[00<003W>>Lh00SW>00D0
048@@Q12400000=6<@05001>LdicCW<00004DY@01000F]IJeU[F0P0000AS6000000001Eoo`03001o
ogoo04Yoo`03001oogoo00Moo`03001oogoo00Eoo`03000DYABU00DDY@04000icP000005=Jd00`00
>LhicP04000AOol20002;F/00`00@Q124003@Q001000CW=>Ldic0`0015Je00<005kgG_L02Ekg00D0
06<HHaQS600000AS608000EcW003001oogoo00Qoo`03001oogoo01eoo`005Woo10000goo0`0000Ao
o`00000000Eoo`03001oogoo019oo`03000<H`aS00@<H`8000<YBP03000mkcg_010mk`03001>Ldic
00Y>L`03001>Ldic009>L`04001JeU[F0006E[D00`00AS40000AOol00`00Oomoo`04001COol00`00
Oomoo`03Ool00`00Oomoo`0FOol200027>L01000;F/]J`000cW>00D0048@@Q12400000=6<@05001>
LdicCW<00004DY@01000F]IJeU[F0`000V]J00<007ooOol04goo00<007ooOol0BWoo00<007ooOol0
27oo00<007ooOol00goo00<001BU5:D01ABU00<003W>>Lh00SW>0P0013F]00@003W>>LhicP@000mo
o`800003;F/0048@00E24004001>L`000006E[D00`00G_MNm`09G_L00`00HaQS6003HaP20002HaP2
0005Li`00`00Oomoo`09Ool00`00Oomoo`0LOol002Qoo`8001Aoo`03000<H`aS00<<H`03000YBRU:
00<YBP03000mkcg_010mk`03001>Ldic00Y>L`03001>Ldic009>L`04001JeU[FF]H20005E[D3000A
Ool00`00Oomoo`1FOol00`00Oomoo`03Ool00`00Oomoo`0FOol200027>L01000;F/]J`000cW>00D0
048@@Q1240000096<@<0009>L`05001BU5:DDY@00004F]H30002JeX00`00Oomoo`0COol00`00Oomo
o`1:Ool00`00Oomoo`09Ool00`00Oomoo`02Ool00`005:DDY@045:D00`00>LhicP05>Lh20003=Jd2
0002>Lh4000>Ool30007@Q020008E[D00`00G_MNm`08G_L00`00HaQS6006HaP30006Li`00`00Oomo
o`09Ool00`00Oomoo`0LOol002Qoo`03001oogoo019oo`8000@<H`03000YBRU:00@YBP03000mkcg_
00hmk`D000Y>L`03001>Ldic009>L`03001JeU[F00=JeP03001F]EJe00AF]@03001oogoo00moo`80
05Ioo`8000Eoo`03001oogoo01Ioo`80008Li`04000]Jbe[0003>Lh00`00@Q12400400000dic001B
U0040002DY@00`00F]IJeP02F]H30002JeX00`00Oomoo`0COol00`00Oomoo`1:Ool00`00Oomoo`0:
Ool01000Oomoo`000aBU1@002CW>0P000cF]00<003W>>Lh010003Goo0P001d8@0P002EJe00<005kg
G_L025kg00<006<HHaP026<H00<007>LLi`00g>L0P002goo00<007ooOol077oo000XOol00`00Oomo
o`0BOol0100036<<H`aS0P001bU:00<003g_?Nl02Cg_1@000THa00<005:DDY@01P0014ic00<004ic
CW<00Tic00<005[FF]H015[F00<005JeE[D00eJe00<007ooOol047oo00<007ooOol0EGoo00<007oo
Ool00goo00<007ooOol05Woo00D000@Q000Li`00008]J`03000icSW>00@00003AS4004ic009>L`06
001BU5:DDY@005Je0`000e[F00D006<H000006]J008001Eoo`03001oogoo04Uoo`03001oogoo00ao
o`H000<U:@03000mk`00008000PicP80008e[@800003>Lh0000000<000Yoo`0300048@0000I24003
001>L`0000QF]@03001Nmekg00QNm`03001S66<H00QS608000EcW003001oogoo00Yoo`03001oogoo
01aoo`00:7oo00<007ooOol04Woo00@000aS36<000TYBP03000mkcg_00Dmk`@000M6<@03001BU5:D
00IBU0H000=>L`03001JeU[F00EJeP8000=F]@03001oogoo015oo`03001oogoo05Yoo`03001oogoo
01Ioo`0300048@00008Li`03000]J`0000800003@Q0004Ha0096<@05001>LdicCW<00003DY@01000
E[EF]EJe0`0000=JeP00HaP00P0000=[FP0000005Goo00<007ooOol0BGoo00<007ooOol037oo0P00
1bDY00<003g_?Nl00Sg_10001SW>1P0000<icP00@Q000P002Goo0`001D8@00<004ic00002EJe00<0
05kgG_L01ekg00<006<HHaP026<H00<006mk000017>L0P0037oo00<007ooOol077oo000XOol00`00
Oomoo`0BOol3000::DX00`00?Nlmk`05000;AS400`00DYABU00:DY@00`00DY@000030009F]H00`00
E[EF]@02000COol00`00Oomoo`1IOol00`00Oomoo`0FOol00`00124000027>L30003@Q001@00AS56
<DHa00000dic00D005:DDYABU00000=F]@=Nm`800004HaP006]JJeX2000EOol00`00Oomoo`19Ool2
000=Ool200079BT00`00?Nlmk`06?Nl30002>Lh00`00BU800003BU8400000d8@00000008Ool30003
@Q020002CW<00`00E[EF]@07E[D00`00G_MNm`06G_L00`00HaQS6008HaP00`00Kg]_N`020003Li`0
0`00Oomoo`0;Ool00`00Oomoo`0LOol002Qoo`03001oogoo015oo`8000`YBP<00116<@03001BU5:D
00YBU003001BU5:D009BU0<000QJeP@001=oo`8005Uoo`8001Uoo`80008Li`04000icSW>0003@Q00
1000AS56<@000dic00D005:DDYABU000009F]@03001Nmekg009Nm`<0009[FP03001oogoo01Aoo`03
001oogoo04Uoo`03001oogoo00]oo`0300048@0000PU:@03000mkcg_00Pmk`8000=:DP06001:DTYB
BU8004YB1@001goo0`000T8@00<004icCW<00Tic00<005JeE[D01UJe00<005kgG_L01ekg00<006<H
HaP01f<H00@006mkKg]_N`80009cW08000eoo`03001oogoo01aoo`00:7oo00<007ooOol04Goo0P00
2RU:0`004THa00<005:DDY@02U:D00<005:DDY@00U:D00<005kgG_L00P0025[F0P0057oo00<007oo
Ool0Egoo00<007ooOol067oo00<000@Q7>L00P000SW>00D0048@@Q1240000096<@03001>L`0000H0
009F]@05001NmekgG_L00002HaP4000FOol00`00Oomoo`19Ool00`00Oomoo`0;Ool00`0012400008
9BT00`00?Nlmk`09?Nl20003BU801@00BU9:DTYB00000TYB10001Goo100000=24000CW<014ic00<0
05JeE[D01UJe00<005kgG_L01Ukg00<006<HHaP01f<H00<006mkKg/00Vmk0P000W>L00<007ooOol0
37oo00<007ooOol077oo000XOol00`00Oomoo`0AOol20008:DX20002=Jd00`00AS56<@0@AS400`00
DYABU00:DY@00`00DYABU002DY@00`00G_MNm`02G_L00`00F]IJeP05F]H2000EOol00`00Oomoo`1F
Ool00`00Oomoo`0HOol02000124Li`00<H`003W>0002@Q0600038@P01000;F/]Jbe[1@0000ENm`00
HaQS66]J00<001Ioo`03001oogoo04Uoo`03001oogoo00Yoo`0400048@@Q00069BT50009?Nl01000
BU8000000TYB00H004YBBU9:DP00BU850004Ool200000be[00000006CW<00`00E[EF]@05E[D00`00
G_MNm`06G_L00`00HaQS6007HaP00`00Kg]_N`04Kg/01000Li`000003goo00<007ooOol06goo000O
Ool40004Ool00`00Oomoo`0BOol20007:DX00`00=Jde[@03=Jd00`00AS56<@0?AS400`00DYABU00:
DY@00`00DYABU002DY@00`00G_MNm`03G_L20005F]H3000EOol00`00Oomoo`1EOol00`00Oomoo`0H
Ool00`0012400002<H`400000aBU000Q20038@P80003;F/00`00<H`00003000016]J000000005goo
00<007ooOol0B7oo00<007ooOol02Woo00@000@Q1240008U:@@000<e[@030012400000<000Hmk`04
001:DTYB0002BU801P00BU9:DTYB001:DP@000Aoo`800003;F/004ic00I>L`03001F]EJe00AF]@D0
00ENm`03001S66<H00MS6003001_Nfmk00A_N`<0011oo`03001oogoo01]oo`007goo00@007ooOol0
00Aoo`80019oo`<000DYBP8000He[@030016<DHa00m6<@03001BU5:D00YBU003001BU5:D009BU003
001Nmekg00ENm`03001JeU[F009JeP<001Eoo`03001oogoo05Eoo`03001oogoo01Uoo`03000U:@00
008000<DY@H000A:DP04001BU5:DDY@60003<H`3000HOol00`00Oomoo`18Ool2000;Ool00`001244
8@030007=Jd00`00@Q124003@Q040002?Nl01@00BU9:DTYB00000TYB0P0000A:DP00BU9:DPT00003
;F/004ic00E>L`03001F]EJe009F]@@000ANm`D000US6003001_Nfmk00E_N`<0011oo`03001oogoo
01]oo`007goo00@007ooOol000Aoo`03001oogoo015oo`0300048@0000@YBP03000e[CF]00He[@03
0016<DHa00m6<@03001BU5:D00YBU003001BU5:D009BU003001Nmekg00INm`80009JeP<001Ioo`03
001oogoo05Aoo`8001Yoo`80008DY@@00003AS4004YB00M:DP03001BU5:D00EBU003001F]@0000L0
01Ioo`03001oogoo04Qoo`03001oogoo00Yoo`0300048@@Q008000Te[@030012448@00I240<000A:
DP08001:DTYBBU8004YB001:DPD00003Ool0000000<000I>L`03001F]@00008000=JeP03001Nmekg
00ENm`04001S60000006HaP00`00Kg]_N`06Kg/2000AOol00`00Oomoo`0KOol001moo`04001oogoo
0004Ool00`00Oomoo`0AOol00`0012400002:DX20009=Jd00`00AS56<@0=AS4;0004DY@00`00DYAB
U002DY@00`00G_MNm`08G_L300000f<H001oo`0FOol00`00Oomoo`1COol00`00Oomoo`0GOol00`00
248000030005AS400`00BU9:DP06BU800`00DYABU005DY@00`00E[EF]@03E[D300000cF]0000000D
Ool00`00Oomoo`0iOol400000goo000000030004Ool00`00Oomoo`0;Ool01@00124001cW00002CF]
00<0048@@Q00248@0`000dYB0P000TYB00@004YB001:DPL00003;F/0000000I>L`8000EJeP03001N
mekg00ENm`03001S66<H009S6003001S66<H009S6003001_Nfmk00I_N`<0015oo`03001oogoo01]o
o`005Woo10001Goo00@007ooOol000Aoo`03001oogoo011oo`0300048@@Q00<000`e[@030016<DHa
00I6<@H0009>L`03001F]EJe00IF]@X000aNm`04001S6000000FOol2001DOol00`00Oomoo`0EOol4
00000cg_0016<@06AS400`00BU9:DP06BU800`00DYABU005DY@00`00E[EF]@03E[D00`00E[D00002
00000cF]0000000BOol00`00Oomoo`0dOol400001Goo001oogoo00000goo00<007ooOol017oo00<0
07ooOol037oo0P000QcW00<003F]=Jd01cF]00<0048@@Q00248@00@004ic000000=:DP05001:DTYB
001:DP0800000be[00000004CW<30006F]H00`00G_MNm`04G_L00`00HaQS6004HaP200000f<H001_
N`07Kg/3000BOol00`00Oomoo`0KOol001Moo`03001oogoo00Eoo`@000Aoo`03001oogoo011oo`04
00048@@Q000>=Jd00`00AS56<@060008CW<00`00E[EF]@09E[D20004E[D3000:G_L01000HaQS6000
5goo00<007ooOol00goo100000=oo`00Ool00P00AWoo00<007ooOol04goo10000cg_00<004HaAS40
14Ha3P0015:D00<005JeE[D00eJe00<005Je00000Ukg00@003F]0000015oo`03001oogoo03Uoo`04
001oogoo0004Ool00`00Oomoo`03Ool00`00Oomoo`0<Ool200027>L00`00=Jde[@07=Jd00`00@Q12
4008@Q001@00CW=>Ldic00000dYB00D004YBBU8004YB00L00003;F/0000000=>L`03001F]@0000IJ
eP03001Nmekg00ANm`03001S66<H00MS6003001_Nfmk00I_N`04001g_@00000BOol00`00Oomoo`0K
Ool001Moo`03001oogoo00eoo`03001oogoo011oo`0400048@@Q000>=Jd3000?CW<00`00E[EF]@05
E[D30002E[D01@00E[EF]EJe00000V<H0P0025kg0P0000=S6000Ool05goo00<007ooOol00Woo00@0
07ooOol0009oo`04001oogoo0015Ool00`00Oomoo`0AOol40006?Nl00`00AS40000300000eJe001F
]@07E[D01000F]IJeU[F1P0015Je0P0015kg100047oo00<007ooOol0>Goo00@007ooOol000Eoo`03
001oogoo009oo`03001oogoo00aoo`03000LiacW008Li`03000e[CF]00He[@L000E24004001>Ldic
CW<20002BU8200000dYB0000000200000e:D0000000200001DHa001>Ldic00000UJe00<005[FF]H0
0e[F00<005kgG_L015kg00<006<HHaP01V<H0`001Vmk0P0000Ag_@000000019oo`03001oogoo01]o
o`0067oo00<007ooOol037oo00<007ooOol03goo00<000@Q12400P0033F]0`004Dic00<005JeE[D0
0UJe0`001UJe00@005JeE[D000AS608000ENm`05001[FP00HaP0000IOol01@00Oomoogoo00000Woo
00<007ooOol00goo00<007ooOol0@goo00<007ooOol047oo10001cg_0P001UJe00<005JeE[D01EJe
00<005[FF]H01E[F00<005kg00001@001Ukg00<002e[00003goo00<007ooOol0>Goo00@007ooOol0
009oo`04001oogoo0004Ool2000=Ool00`007>LLi`037>L00`00=Jde[@03=Jd20002AS400`00BU9:
DP02BU860005CW<01000BU9:DTYB1P0000EBU000;F/004Ha00<000=F]@03001JeU[F00=JeP03001N
mekg00=Nm`03001S66<H00IS6005001W>FLiIcT00004Kg/01`00Mkeg_GNm001oo`0001=oo`03001o
ogoo01Yoo`005Woo100037oo00<007ooOol047oo00H000@Q124001BU0009=Jd20003@Q000`00CW=>
L`0>CW<00`00E[D00002000:E[D00`00E[D00006HaP20003G_L01P00JeY[FP00HaP001Qoo`80009o
o`04001oogoo0002Ool30016Ool00`00Oomoo`0?Ool30002=Jd00`00?Nlmk`02?Nl200000e:D001F
]@05E[D00`00E[EF]@04E[D00`00F]IJeP05F]H00`00G_MNm`04G_L20006G_L200000be[001oo`0=
Ool00`00Oomoo`0iOol40003Ool30004Ool00`00Oomoo`0=Ool00`007>LLi`027>L00`00=Jde[@03
0005AS400`00BU9:DP07BU820004CW<20002BU8200002E[F000005:D000]J`00AS400004E[D00`00
F]IJeP03F]H00`00G_MNm`03G_L00`00HaQS6006HaP00`00IcUW>@02IcT01@00Kg]_Nfmk00000gNm
00@007ooOol001=oo`03001oogoo01Yoo`009Woo0P004Goo00H000@Q124001BU0007=Jd20005@Q00
0`00CW=>L`0=CW<3000=E[D00`00E[D00008HaP200001ekg001[FV]J001S600001Uoo`04001oogoo
0002Ool01000Oomoo`00B7oo00<007ooOol03goo00<003F]=Jd00SF]00<003g_?Nl00P000e:D00<0
05JeE[D015Je00<005JeE[D015Je00<005[FF]H01E[F00<005kgG_L015kg00@006<H000000=Nm`03
001W>FLi00<000eoo`03001oogoo04Moo`03001oogoo00eoo`03000LiacW00<Li`8000U6<@03001:
DTYB00M:DP<000A>L`T00004;F/004Ha0004E[D00`00F]IJeP02F]H00`00G_MNm`03G_L00`00HaQS
6005HaP20005IcT01000Kg]_N`0017Nm00@007ooOol001=oo`03001oogoo01Yoo`009Woo00<007oo
Ool047oo00<000@Q00000QBU00<003F]=Jd00cF]0P001d8@00<004icCW<02Tic0`000Tic00<005Je
E[D035Je0P002V<H0P000f]J0P006Woo00<007oo00000`000goo0`00A7oo00<007ooOol03goo00<0
03F]=Jd013F]0P001U:D00<005JeE[D015Je00@005JeE[EF]@H000=JeP03001Nmekg00ENm`04001S
66<HHaP200000ekg001W>@03IcT00`00Oomoo`0;Ool00`00Oomoo`17Ool00`00Oomoo`0>Ool00`00
7>LLi`027>L30009AS420008BU801000CW<0000014ic0`0000=JeP0000000P0000=6<@0000000eJe
00<005[FF]H00e[F00<005kgG_L00Ukg00<006<HHaP01F<H00<006LiIcT01VLi0P0017Nm00D007oo
Oomoo`0001=oo`03001oogoo01Yoo`009Woo00<007ooOol047oo00<000@Q00000QBU00@003F]=Jde
[@8000Y24003001>Ldic00M>L`8000E>L`03001F]EJe00eF]@03001S66<H00US6004001[FV]JJeX2
000JOol2001>Ool00`00Oomoo`0>Ool20005=Jd00`00DY@00006DY@00`00E[EF]@02E[D60002G_L0
1000G_MNmekg1@001Ekg0P001F<H0P001FLi00<007ooOol02Woo00<007ooOol0AWoo00<007ooOol0
47oo00@001cW7>L000<mk`030016<DHa00Q6<@03001:DTYB00I:DP03001>Ldic0080009>L`P00003
AS40000000=F]@03001JeU[F009JeP03001Nmekg009Nm`03001S66<H00AS608000UW>@03001g_GNm
009g_@03001oogoo00=oo`03001oogoo011oo`03001oogoo01Yoo`009Woo0P004Woo0P000aBU0`00
348@00<004icCW<014ic0`001Dic2P0025Je0P002V<H0P000f]J00<007ooOol06Woo00<007ooOol0
Bgoo0P003goo00<002e[00000SF]0P000e:D00<005:DDY@015:D10000Ukg00<005kgG_L00ekg00<0
05kgG_L00ekg00<005kgG_L00`000Ukg0P001F<H00<006Li00000fLi0P0037oo00<007ooOol0AWoo
0P004Goo0P0000<Li`00?Nl00cg_00<004HaAS4024Ha00<004YBBU801TYB00@004icCW=>L`800005
CW<005[F001JeP0400000e:D001F]@02E[D00`00F]IJeP02F]H01@00G_MNmekg00001V<H00<006Li
IcT01VLi0`000gNm0P001Woo00<007ooOol047oo00<007ooOol06Woo000VOol00`00Oomoo`0AOol2
00035:D00`00@Q12400<@Q001000CW=>Ldic0`000dic1@000U:D00<005[FF]H01E[F2`002F<H0P00
0f]J1P005goo00<007ooOol0Bgoo00<007ooOol03Goo00H002e[;F/003F]0005DY@00`00DYABU002
DY@200000ekg001Nm`04G_L00`00G_MNm`02G_L00`00G_MNm`03G_L00`00G_MNm`03G_L40004HaP0
1@00IcUW>FLi00000VLi00<006mk000037oo00<007ooOol0AGoo00<007ooOol04Goo0P001Sg_0P00
2DHa2P0014ic2@0000EBU000E[EF]@00009JeP`00003HaP006Li00IW>@80009_N`03001g_GNm0080
00Moo`03001oogoo011oo`03001oogoo01Yoo`004Woo100000=oo`0000000`0000=oo`0000001goo
00<007ooOol04Goo00<001BU5:D00QBU00<0048@@Q003D8@2@001e:D00<005[FF]H03E[F00<006Li
00000P001V<H00@007>LLi`0009[FP03000@Q12400<@Q0@001Aoo`8004=oo`@000Aoo`03001oogoo
00eoo`05000]Jbe[;F/00007DY@01000DY@0000015kg00<005kgG_L00Ukg00<005kgG_L00Ukg00<0
05kgG_L00Ukg00<005kgG_L015kg00<005kgG_L00P000V<H00<006LiIcT00fLi0P000Vmk00<007oo
Ool02Woo00<007ooOol0AGoo00<007ooOol04Woo00<003g_?Nl01Sg_00<004HaAS4014Ha0P0000=>
L`0000001e:D0P000dic1`000U:D00<005JeE[D03P0000<48@0000000P000fLi0P000fmk00<007Nm
Mkd00P0027oo00<007ooOol04Goo00<007ooOol06Goo000BOol01000Oomoo`000goo00@007ooOol0
009oo`03001oogoo00Aoo`03001oogoo015oo`07000DYABU5:D002U:00002d8@1P003U:D00<005[F
F]H035[F00@006LiIcUW>@<000AS6004001cW7>L0002JeX600046<H2000BOol20013Ool01000Oomo
o`0017oo00<007ooOol037oo00L002e[;F/]J`00CW<00007DY@00`00G_MNm`04G_L00`00G_MNm`03
G_L01@00G_MNmekg000015kg00<005kgG_L00Ukg0P000ekg00@005kg000000MW>@05001_NfmkKg/0
000;Ool00`00Oomoo`15Ool00`00Oomoo`0BOol30007?Nl20002AS420005CW<20006DY@30002CW<6
0002DY@8000236<00`007>LLi`047>L80005Kg/3000:Ool00`00Oomoo`0AOol00`00Oomoo`0IOol0
019oo`04001oogoo0004Ool00`00Oomoo`02Ool00`00Oomoo`04Ool00`00Oomoo`0AOol01`005:DD
YABU000YBP0000Q240<00004@Q000000000@DY@00`00F]IJeP0<F]H00`00IcUW>@02IcT00`00IcT0
000200000f<H001cW002Li`30003:DX600026<H2000BOol00`00Oomoo`10Ool01000Oomoo`0017oo
00<007ooOol037oo0P0000<]J`00CW<00Tic0P0015:D00@005kg000000ENm`04001NmekgG_L;0003
G_L00`00G_MNm`03G_L300000ekg001W>@03IcT200001F]J001_Nfmk00002goo00<007ooOol0AGoo
00<007ooOol04goo00@0036<000000Lmk`8000U>L`8000EBU0D00003F]H000000080009BU0<000L<
H`03000LiacW00/Li`@00004Kg/00000000;Ool00`00Oomoo`0AOol00`00Oomoo`0IOol0019oo`04
001oogoo0005Ool40006Ool00`00Oomoo`0COol010005:DDY@000RU:00<0048@@Q000T8@0`000T8@
0P000dYB00<005:DDY@03E:D00<005[FF]H035[F00<006LiIcT00VLi00<006LiIcT00VLi0P000g>L
0P0000<YBP0000000P000S6<0P0000<YBP0000000`004Goo00<007ooOol0=goo10001Goo00@007oo
Ool000Aoo`8000eoo`800003;F/004ic00A>L`04001BU0000004G_L20003G_L40002IcT01000HaQS
60000Ukg00<005[FF]H00`001Ukg00<005kgG_L00Ukg00@006Li000000=[FP04001_Nfmk000;Ool0
0`00Oomoo`15Ool00`00Oomoo`0DOol01000<H`aS0001Cg_00<004icCW<010001dic0P001E:D00D0
05Je000005[F00<00003DY@0000000X<H`03000LiacW00/Li`03000U:BDY00<000aoo`03001oogoo
015oo`03001oogoo01Uoo`004Woo00@007ooOol0009oo`03001oogoo008000Qoo`8001Aoo`03000D
Y@0000<YBP0300124000008000=2408000E:DP03001BU5:D00eBU003001JeU[F00aJeP03001W>FLi
00=W>@03001W>FLi009W>@8000=cW08000HaS0D000<YBP03001oogoo00ioo`8003Uoo`03001oogoo
00Eoo`04001oogoo0004Ool00`00Oomoo`0<Ool00`006<H00007CW<20007G_L30002IcT01@00IcUW
>FLi00000V<H00@005kgG_L0009JeP800003F]H00000009Nm`<000ENm`03001[FV]J00A[FP<000]o
o`03001oogoo04Eoo`03001oogoo01Eoo`03000aS36<008000<mk`03001>Ldic00A>L`<000I>L`80
00ABU003001F]EJe00L000D<H`T000XLi`03000U:BDY00<U:@<000Uoo`03001oogoo015oo`03001o
ogoo01Uoo`004Woo10000goo0`0000Aoo`00000000Eoo`03001oogoo01=oo`03000DY@00008YBP<0
00A2408000M:DP03001BU5:D00iBU003001JeU[F00]JeP03001W>FLi00AW>@06001W>FLi001g_@00
0W>L00<0036<00001P0013W>100000<YBP00000047oo00<007ooOol0=Woo00<007ooOol01Goo1000
17oo00<007ooOol037oo00<001S600000P0014ic00<005kgG_L010000Ukg00<006]JJeX01P000VLi
1@0000=JeP0000001E[F0P000Ukg00<005kgG_L00P0000=Nm`00JeX01V]J00<007Nm00002Woo00<0
07ooOol0AGoo00<007ooOol05Goo00<0036<<H`00S6<0`002Tic10000dic200000=F]@0000002@00
1B4800@003F]=Jde[@P000<Li`03000U:BDY00DU:@8000Moo`03001oogoo015oo`03001oogoo01Uo
o`009Goo00<007ooOol04goo00<001BU00000P000RU:00<0048@@Q000P002TYB00<005:DDY@03E:D
00<005[FF]H02e[F00<006LiIcT01FLi00<006Li00000WNm00@007>L000000/icP04000icSW>>Lh4
000@Ool00`00Oomoo`0fOol00`00Oomoo`0<Ool00`00Oomoo`0<Ool00`006<H00002CW<200000dic
001Nm`06G_L50004JeX800000e:D000000040004E[D00`00G_L000020004G_L00`00JeY[FP02JeX3
00000gNm001oo`09Ool00`00Oomoo`14Ool00`00Oomoo`0GOol20004<H`3000<CW<500000e[F0000
000:000:8@P00`00=Jde[@09=Jd600079BT00`00Oomoo`04Ool00`00Oomoo`0AOol00`00Oomoo`0I
Ool002Eoo`03001oogoo019oo`<000HYBP8000a:DP03001BU5:D00]BU1<000IW>@80009g_A8000@i
cP8000moo`03001oogoo00Ioo`03001oogoo02]oo`@000eoo`03001oogoo00aoo`8000E>L`8000MN
m`03001[FV]J009[FPT0000336<0000000h000INm`03001[FV]J0080009_N`<000Yoo`03001oogoo
04Aoo`03001oogoo01Qoo`<000<aS003001>L`0000@000M>La8000XQ2003000e[CF]00de[@040016
<@0000069BT20004Ool00`00Oomoo`0AOol00`00Oomoo`0IOol002Eoo`03001oogoo00moo`<0009o
o`03000YBRU:00@YBP8000a:DP03001BU5:D00IBU0D0009JeP03001S66<H00]S6003001[FV]J00@0
009W>@8000=g_@030012448@00e240<000@icP03001oogoo00eoo`03001oogoo00Eoo`03001oogoo
03aoo`03001oogoo00eoo`03001>Ldic00=>L`03001Nm`0000<000=Nm`04001[FV]JJeX700021240
0`aS0000000200000a24000000050004E[D00`00G_MNm`02G_L50004Kg/2000;Ool00`00Oomoo`14
Ool00`00Oomoo`0JOol300000c6<001>L`05CW<=00000d8@001JeP03F]H6000;8@P00`00=Jde[@0<
=Jd01000AS56<DHa0`0012DY0P000goo00<007ooOol04Woo00<007ooOol067oo000UOol00`00Oomo
o`0=Ool20006Ool00`00:DXYBP02:DX00`00>Lh0000=BU800`00DYABU0050007F]H00`00HaQS600;
HaP00`00JeY[FP04JeX300000gNm001g_@0:0007@Q000`00BU9:DP0300000cW>001oo`0?Ool00`00
Oomoo`04Ool00`00Oomoo`0lOol00`00Oomoo`0=Ool20003CW<00`00G_MNm`04G_L70002JeX80000
0bDY000]J`060003E[D90003HaP00`00Kg]_N`02Kg/3000;Ool00`00Oomoo`14Ool2000LOol50007
CW<50006@Q0>00098@P00`00=Jde[@0<=Jd00`00AS56<@04AS4200039BT20003Ool00`00Oomoo`0A
Ool00`00Oomoo`0HOol002Eoo`03001oogoo00Yoo`<000Qoo`03000YBRU:008YBP04000icSW>000<
BU83000=F]H00`00HaQS600:HaP00`00JeY[FP06JeX3000017Nm000mk`001dYB2@0014YB0`003goo
0P001Goo00<007ooOol0>goo00<007ooOol03Woo00H004YB000004ic0007G_L00`00JeY[FP080003
5:D010009BTU:@000Re[00<0036<<H`01@0015Je0P001f<H00H006mk000006mk000<Ool00`00Oomo
o`14Ool00`00Oomoo`0MOol;0006@Q0500000e[F00000004000312400`00;F/]J`0600038@P00`00
=Jde[@0<=Jd00`00AS56<@06AS4300001RDY00000000Ool001=oo`03001oogoo01Qoo`0097oo0P00
2Goo0`0037oo00@002U::DX000<icP03001:DTYB00Q:DP800003DY@005[F00eJeP03001S66<H00US
6003001[FV]J00M[FP03001g_@0000H000e:DP<000A:DP8000moo`03001oogoo009oo`8003eoo`03
001oogoo00ioo`04001:DTYBBU8<0003IcT300000`Q2000DY@025:D010009BTU:@000be[00L0036<
<H`003F]001F]@0300000eJe00000004HaP50002Kg/2000<Ool00`00Oomoo`0lOol40004Ool00`00
Oomoo`0MOol00`00Ool0000500000b4800124002@Q040003BU830002F]H200000e[F00048@021240
0`00;F/]J`06;F/5000>=Jd00`00AS56<@08AS4300000acW0000000COol00`00Oomoo`0HOol002Ao
o`03001oogoo00Eoo`<0011oo`03000YBP0000@icP03001:DTYB00E:DP8000=BU003001JeU[F00aJ
eP03001S66<H00US6003001[FV]J00I[FP04001g_GNmMkd2000013g_001>Ldic1P001dYB00<005Je
E[D00P0014YB00<007ooOol037oo0P0017oo00<007ooOol0>goo00<007ooOol03goo00D004YBBU9:
DP0000MNm`03001W>FLi00L000<DY@05000U:BDY9BT00003;F/01@00<H`aS000=Jd00`000eJe1P00
0f<H00D006mkKg]_N`0000ioo`03001oogoo03]oo`04001oogoo0004Ool00`00Oomoo`0MOol00`00
Ool0000300028@P50005BU8200000dic001JeP020002F]H2000312400`00;F/]J`09;F/00`00@Q00
0003000:=Jd00`00AS56<@0:AS4200000acW001oo`0BOol00`00Oomoo`0HOol002Aoo`8000=oo`<0
01=oo`8000HicP03001:DTYB009:DP8000IBU003001JeU[F00]JeP03001S66<H00US6003001[FV]J
00I[FP03001g_GNm00H000M>L`H0009:DP04001F]EJeE[D20002BU800`00Oomoo`09Ool300000goo
001oo`11Ool2000@Ool30002BU800`00G_MNm`02G_L50003IcT200000`Q2000DY@025:D01@009BTU
:BDY00000be[00H0036<<H`aS000=Jd400000eJe001S6006HaP01000Kg]_Nfmk0P003Woo00<007oo
Ool0>goo00@007ooOol000Aoo`03001oogoo01eoo`<000@Q208000M:DP8000A>L`06001JeU[FF]H0
05[F0`0000<48@00;F/02Re[00<0048@@Q000d8@10001SF]00<004HaAS402dHa10004Woo00<007oo
Ool067oo000TOol00`00Ool00002000GOol00`00>LhicP04>Lh00`00BU9:DP020008DY@00`00F]IJ
eP0<F]H00`00HaQS6003HaP:0004JeX00`00Mkeg_@0500000dic000000020009CW<30005E[D20000
0dYB001oo`05Ool40004Ool00`00Oomoo`10Ool00`00Oomoo`0@Ool00`00BU8000060004G_L00`00
IcUW>@0400025:D90002;F/01000<H`aS0000SF]0P0000=F]@0000000`0016<H00<006mkKg/00P00
3goo00<007ooOol0<Woo10001Goo00@007ooOol000=oo`03001oogoo01eoo`8000@Q20<000E:DP<0
00A>L`800009DY@005[F001JeP00F]H000@Q00<000X]J`030012448@00M240@0008e[@030016<DHa
00]6<@03001BU0000080015oo`03001oogoo01Qoo`005Goo100000=oo`00Ool00P001Woo0P006goo
00<003W>>Lh013W>0P002e:D00<005[FF]H02U[F1`001FLi00<007>LLi`00W>L1@0000Eg_@00?Nlm
k`00009>L`8000=>L`@000M>L`<000EF]@8000=oo`<000Qoo`03001oogoo041oo`03001oogoo015o
o`05001:DTYBBU800007G_L00`00IcT00002000010Q2000000000Re[00D003F]=Jde[@00008mk`@0
008aS005000e[CF]001F]@020003HaP8000@Ool00`00Oomoo`0cOol00`00Oomoo`05Ool01000Oomo
o`000goo0P007Goo00<002488@P00R480P000SW>00@004YBBU9:DP8000I>L`03001BU5:D009BU003
001JeU[F00@00003124002e[00<000H]J`030012448@00a240@000e6<@800003DY@00000011oo`03
001oogoo01Qoo`005Goo00@007ooOol0009oo`04001oogoo0006Ool00`00Oomoo`0IOol20004>Lh0
0`00AS40000<DY@00`00F]IJeP05F]H400000ekg001W>@09IcT00`00LiacW005Li`00`00Mkd00004
0003CW<30005CW<40002CW<01@00F]IJeU[F00001UJe0`002goo00<007ooOol0@7oo00<007ooOol0
4Woo00@004YBBU8000MNm`L000@]J`05000e[CF]=Jd00004?Nl00`00AS40000300000cF]00000002
0006HaP01000LiacW0004Goo00<007ooOol0<goo00<007ooOol01Goo10000goo00<007ooOol06Woo
0P000b480P0013W>00@004YB000000I>L`8000ABU003001F]@0000D00003124002e[00<]J`8000@]
J`030012448@00i24004001BU000000;AS420002DY@2000@Ool00`00Oomoo`0GOol001Eoo`04001o
ogoo0005Ool00`00Oomoo`05Ool00`00Oomoo`0IOol00`00>LhicP02>Lh01000AS56<@002e:D00@0
05[FF]IJeP@000INm`03001W>FLi00MW>@03001cW7>L00EcW003001g_@00008mk`04001>L`000002
CW<01000CW<000001dic0`0015[F0P0015Je00<007ooOol02goo00<007ooOol0@7oo00<007ooOol0
4goo0P0000A:DP00G_MNm`@00004G_L0000000027>L00`00;F/]J`02;F/01@00=Jde[CF]000013g_
00H004HaAS56<@00BU880002HaP01@00LiacW7>L00004Goo00<007ooOol0=7oo00<007ooOol02goo
00<007ooOol06Goo0P000R480P001SW>0P001dic00<005:DDY@00e:D00L005JeE[D005[F001Nm`04
0006;F/300000be[0012400?@Q001000DYABU5:D0`0024Ha0P000e:D0P003goo00<007ooOol05goo
000EOol01000Oomoo`000Woo0`002Goo00<007ooOol06Goo00@003W>>Lh000A6<@03001BU5:D00UB
U0<000YNm`03001W>FLi00MW>@03001cW7>L00AcW004001g_@00?Nl20004CW<30003CW<30006CW<2
0005F]H20002E[D00`00Oomoo`0<Ool00`00Oomoo`0oOol00`00Oomoo`0DOol50004G_L400027>L0
0`00;F/]J`02;F/01@00=Jde[CF]000013g_00D004HaAS56<@0000=:DP<000=S60H0019oo`03001o
ogoo039oo`@000aoo`03001oogoo01Qoo`8000038@P0000000HicP<000M>L`03001BU5:D009BU080
009F]@03001JeU[F00H000T]J`<000i24003001BU5:D00EBU0<000E6<@8000EBU003001oogoo00ao
o`03001oogoo01Moo`005Goo00@007ooOol0009oo`03001oogoo00Yoo`8001Yoo`03000icP0000E6
<@03001BU5:D00IBU0800003F]H005kg00YNm`03001W>FLi00MW>@04001cW7>LLi`800000dic0000
0004CW<20005CW<20003CW<00`00G_MNm`020005F]H01000E[EF]@003Goo00<007ooOol0?goo00<0
07ooOol05Goo0P0000=:DP0000000Ukg0P0000<8@P007>L00QcW00D002e[;F/]J`0000@e[@03000m
kcg_008mk`030016<DHa0096<@06001:DTYB001>L`000f<H00<006mkKg/00P004goo00<007ooOol0
@Woo00<007ooOol05Woo1@001cW>00D004HaAS56<@0000A>L`8000ABU003001F]EJe009F]@06001J
eP00G_L006]J0`002Re[00@004Ha000000a24003001BU5:D00QBU08000A6<@03001BU5:D00ABU003
001oogoo00]oo`03001oogoo01Moo`005Goo10000goo0`002goo00<007ooOol067oo00<004HaAS40
1DHa00<005:DDY@00e:D0P0015[F00<005kgG_L02Ekg00<006LiIcT00VLi20000WNm0P0000=:DP00
CW<00P000Tic0P0014ic0P001Dic0P0000=>L`00G_L00ekg00<005[FF]H00e[F0`003Goo00<007oo
Ool0?goo00<007ooOol05goo1`0000D8@P007>LLi`0000@]J`03000e[CF]008e[@03000mkcg_00<m
k`050016<DHaAS400003BU8500000fmk0000000EOol00`00Oomoo`12Ool00`00Oomoo`0EOol40007
>Lh20004AS401@00CW=>Ldic00001E:D00<005JeE[D00UJe00D005[FF]H005kg00L000L]J`030016
<DHa0096<@<000U24003001BU5:D00YBU0<00003AS4005:D00EBU003001oogoo00]oo`03001oogoo
01Moo`00:goo00<007ooOol067oo0P001THa00<005:DDY@00P001U[F00<005kgG_L025kg1P0016mk
00<007NmMkd00gNm0P000TYB00@004icCW<000=>L`8000=>L`<000E>L`8000ENm`03001JeU[F00AJ
eP03001oogoo00]oo`03001oogoo03moo`03001oogoo01Qoo`800003BU8000000088@P04000LiacW
0004;F/00`00=Jde[@02=Jd00`00?Nlmk`03?Nl01@00AS56<DHa00000dYB00<004ic00000`005goo
00<007ooOol0@Woo00<007ooOol057oo10001cW>00<004HaAS401DHa0`001E:D00<005JeE[D00UJe
00L005[FF]H005kg001[FP0400028@P00`00;F/]J`04;F/00`00AS56<@05AS420007@Q000`00DYAB
U00=DY@30005DY@2000<Ool00`00Oomoo`0GOol002aoo`03001oogoo01Qoo`8000I6<@8000UJeP03
001Nmekg00ANm`<00003IcT006mk00M_N`04001g_GNmMkd300001DHa001:DTYB00000Tic00<004ic
CW<00Tic0P000Tic00@004ic0000009>L`800003G_L005kg00ANm`03001JeU[F00=JeP03001oogoo
00]oo`03001oogoo03moo`03001oogoo01Yoo`<00003248001cW008Li`03000]Jbe[008]J`03000e
[CF]008e[@03000mkcg_00<mk`050016<DHaAS400003BU801000CW<000006Goo00<007ooOol0@Goo
00<007ooOol057oo0`0000<HaP00>Lh013W>0P0024Ha00<005:DDY@015:D00<005JeE[D00UJe00@0
05[FF]IJePP000<Q208000@]J`030016<DHa00M6<@<000=24003001BU5:D00mBU08000IBU003001o
ogoo00Yoo`03001oogoo01Moo`00;Goo00<007ooOol067oo0P0014Ha00@005:D000000UJeP03001N
mekg00<000EW>@03001_Nfmk00E_N`03001g_@000080009240030016<@00009:DP800003CW<00000
00A>L`800004CW<004icCW<20004G_L20004G_L00`00F]IJeP02F]H2000<Ool00`00Oomoo`0nOol0
0`00Oomoo`0KOol300000`Q2000Li`027>L00`00;F/]J`0?0003AS401`00BU9:DP00CW<004YB01Uo
o`03001oogoo045oo`03001oogoo01=oo`<0008HaP03000icSW>008icP030016<DHa00M6<@03001>
L`0000=BU08000AF]@04001JeU[FF]H300000f]J0000000200000f<H000Q20038@P01@00;F/]Jbe[
000034Ha0P0000=24000DY@03e:D00@006<H000000EBU003001oogoo00Yoo`03001oogoo01Moo`00
;Woo00<007ooOol067oo0`0000=6<@00DY@00e:D00<005[FF]H01U[F0`0026Li00<006mkKg/016mk
0`0000=_N`0000000`000THa00D004YBBU9:DP00009>L`03001>Ldic00=>L`D00003F]H005kg00AN
m`03001Nmekg009Nm`800004F]H006<H000<Ool00`00Oomoo`0nOol2000LOol2000224800`007>LL
i`040002>Lh00`00@Q124002@Q000`00BU9:DP03BU801P00CW<00000AS40009:DP05001>L`00BU80
000HOol00`00Oomoo`11Ool00`00Oomoo`0COol200026<H01000>LhicSW>0P0024Ha0P000dic00<0
05:D00001UJe00H005[FF]IJeP00HaP200000f]J0000000400048@P3000?AS42000@DY@01@00HaP0
06<H000015:D00<007ooOol02goo00<007ooOol05Woo000_Ool00`00Oomoo`0IOol20006DY@20002
F]H40002HaP20008IcT00`00Kg/0000300001Fmk001_N`00@Q000P0000E6<@000016<@0000A:DP04
001>Ldic0006CW<20003F]H00`00G_MNm`03G_L00`00G_MNm`03G_L200000f<H001oo`0;Ool00`00
Oomoo`0nOol00`00Oomoo`0KOol200000`Q2000Li`0200000c6<000icP03>Lh00`00@Q124002@Q00
0`00BU9:DP03BU801000CW=>Ldic0P000dYB10006Goo00<007ooOol0@7oo00<007ooOol04Woo0P00
0aS600@003W>>Lh000U6<@03001>Ldic00A>L`<000AF]@03001JeP000080009S60X000DQ20030016
<DHa00e6<@04001BU000000=DY@01000HaQS60000V<H0P000U:D0P0037oo00<007ooOol05Woo000`
Ool2000KOol20006DY@20008HaP20004IcT40003Kg/4000013g_000000000THa100014YB00<004ic
CW<00P000Tic0P0000=JeP00F]H00e[F00<005kgG_L00ekg00<005kgG_L00ekg00@006<HHaP000]o
o`03001oogoo03Qoo`03001oogoo00=oo`03001oogoo01Yoo`<000032480000000<aS003000icSW>
008icP030012448@00924003001:DTYB00=:DP03001>Ldic009>L`<00003BU800000008001Uoo`03
001oogoo041oo`03001oogoo015oo`8000@HaP<000U6<@03001>Ldic00A>L`04001F]EJeE[D60000
0e[F001S6002HaP00`00JeX0000600000`Q2000Q20028@P01000?Nl000003DHa00@005:DDYABU080
00]BU004001S66<H0004HaP01000DY@0000037oo00<007ooOol05Woo000bOol00`00Oomoo`0IOol3
0003DY@00`00E[EF]@020007HaP50003JeX60000148@000004Ha0`000THa00<004ic00002`0015[F
00<005[FF]H00U[F00<005kgG_L00ekg00<005kgG_L00Ukg00@006<HHaP000]oo`03001oogoo03Qo
o`03001oogoo00=oo`03001oogoo01Yoo`D000@aS003000icSW>008icP030012448@00924003001:
DTYB00=:DP03001>Ldic009>L`03001F]EJe00D001Uoo`03001oogoo041oo`03001oogoo011oo`80
00DHaP030016<DHa00Q6<@03001>Ldic00A>L`03001F]EJe00AF]@03001JeU[F009JeP03001S66<H
00/00004248002480004?Nl2000:AS400`00DYABU004DY@20009DY@01000HaP000001F<H00<005:D
000037oo00<007ooOol05Woo000cOol00`00Oomoo`0KOol200000e:D001F]@02E[D30003HaP20002
HaP20003JeX30000148@0016<@000T8@00D004Ha001240000096<@05001>LdicCW<00004DY@01@00
E[EF]EJe000015[F0P0015[F00<005kgG_L00ekg0P0000INm`00Kg/006<H000;Ool00`00Oomoo`0h
Ool00`00Oomoo`03Ool00`00Oomoo`0JOol200028@P00`00<H`aS002<H`00`00>LhicP02>Lh00`00
@Q124002@Q000`00BU9:DP03BU800`00CW=>L`02CW<01000E[EF]EJe1@0067oo00<007ooOol0@7oo
00<007ooOol047oo00<001S66<H00aS60P0024Ha0P001Tic00<005JeE[D015Je00<005[FF]H00U[F
00P006<HHaQS6000JeX006Li100000=_N`0000000PQ200<003g_?Nl01Cg_0P001dHa00<005:DDY@0
1e:D0P001E:D10000V<H00<006<HHaP00f<H0P0037oo00<007ooOol05Woo000dOol00`00Oomoo`0L
Ool40004E[D40005HaP300001S6<0000048@0016<@@00096<@<00096<@04001>Ldic0004DY@01@00
E[EF]EJe00001E[F0P000e[F00<005kgG_L01Ekg0P0000E_N`00HaQS600000]oo`03001oogoo03Mo
o`03001oogoo00=oo`03001oogoo01Uoo`<0008Q2003000aS36<008aS003000icSW>008icP030012
448@00924003001:DTYB00=:DP03001>Ldic009>L`04001F]EJeE[D200000ekg0000000HOol00`00
Oomoo`10Ool00`00Oomoo`0?Ool00`006<HHaP036<H01000>LhicP001THa00<004icCW<01Dic00<0
05JeE[D01EJe00<005[FF]H00U[F00<006<HHaP03P0000<Q2000?Nl01cg_0P0014Ha00<005:DDY@0
2U:D1@0000=BU000HaP016<H00<006<HHaP00f<H0P002goo00<007ooOol05Woo000eOol00`00Oomo
o`0NOol?00000bU:000aS00200000d8@0016<@02000014YB000004Ha0P0000A:DP00000000=>L`03
001BU5:D009BU005001F]EJeE[D00005F]H20003F]H00`00G_MNm`05G_L00`00Kg/00002HaP00`00
Oomoo`09Ool00`00Oomoo`0fOol20005Ool2000JOol300028@P00`00<H`aS002<H`00`00>LhicP02
>Lh00`00@Q124002@Q000`00BU9:DP03BU801@00CW=>Ldic000015Je0P0000=Nm`00000067oo00<0
07ooOol0@7oo00<007ooOol03Woo00<001S66<H00aS600D003W>>LhicP0000E6<@03001>Ldic00E>
L`03001F]EJe00EF]@03001JeU[F009JeP03001S66<H00800003JeX0000000L00003Ool00000008Q
208000Pmk`800096<@03001BU5:D00QBU0H00003DY@006<H00ES6003001S66<H00=S608000]oo`03
001oogoo01Ioo`00=Woo00<007ooOol097oo00@000@Q000@Q0<0008YBP@00003@Q0004Ha00800003
BU8000000096<@80009:DP80009>L`03001BU5:D009BU003001F]EJe009F]@03001JeU[F00=JeP80
00=JeP03001Nmekg00=Nm`800005Kg/006<HHaP0000;Ool00`00Oomoo`0gOol00`00Oomoo`03Ool0
0`00Oomoo`0IOol00`00248000028@P00`00<H`aS002<H`00`00>LhicP02>Lh00`00@Q124002@Q00
0`00BU9:DP03BU801@00CW=>Ldic000015Je0P0000ANm`00000001Moo`03001oogoo03moo`03001o
ogoo00moo`03000HaQS6008HaP03000icSW>00<icP030016<DHa008000M>L`03001F]EJe00EF]@03
001JeU[F00=JeP<00005IcT00000001W>@02000016<H000000000goo0P000b4800<003g_?Nl01cg_
0P0025:D0`001U:D0P001f<H00L006<HHaQS6000Li`0000<Ool00`00Oomoo`0EOol003Moo`03001o
ogoo02=oo`0400048@0048@30002:DX2000023g_00124000AS4004YB0002BU801000AS4000000dYB
00D004icCW=>L`0000ABU005001F]EJeE[D00006F]H20003F]H01`00G_MNmekg001_N`00009_N`03
001S600000]oo`03001oogoo03eoo`03001oogoo01Uoo`030008@P0000<Q2005000aS36<<H`00004
>Lh00`00@Q124002@Q000`00BU9:DP03BU801@00CW=>Ldic00000eJe00L005kg001Nm`00IcT0000G
Ool00`00Oomoo`0oOol00`00Oomoo`0>Ool00`006<HHaP026<H00`00>LhicP04>Lh00`00AS400009
CW<00`00E[EF]@04E[D00`00F]IJeP03F]H20002IcT00`00JeX00004000016<H000000000goo0P00
124800<003g_?Nl01Cg_0P0000=>L`0000000U:D10002E:D00@006<H000000IS6006001S66<H001c
W00037oo00<007ooOol05Goo000hOol00`00Oomoo`0ROol00`001240000200002BDY000YBP00<H`0
03g_0012400300001DYB001:DTYB00000THa00@004YBBU9:DP<0009>L`03001BU5:D009BU005001F
]EJeE[D00006F]H00`00F]IJeP02F]H00`00G_L00003Kg/01000Kg/0000037oo00<007ooOol0?Goo
00<007ooOol067oo0P0000<8@P008@P00R4800D0036<<H`aS00000@icP030012448@00924003001:
DTYB00=:DP05001>LdicCW<00003E[D01`00G_L005kg001W>@0001Moo`03001oogoo03moo`8000mo
o`05000HaQS66<H00008>Lh00`00CW=>L`07CW<00`00E[EF]@04E[D00`00F]IJeP04F]H01000IcUW
>FLi0P0000=[FP0000000P0000AS6000000000Aoo`8000@Q208000@mk`03001>Ldic009>L`<000aB
U003001S66<H009S6003001S66<H00AS6005001S6000Li`0000<Ool00`00Oomoo`0EOol003Uoo`80
02=oo`07000@Q000000U:@00:DX00`000Sg_00/004Ha000004YB001:DP00BU8004Ha0080009:DP03
001BU5:D00<000ABU003001F]EJe009F]@03001JeU[F00=JeP8000AJeP03001_Nfmk009_N`04001_
N`00000<Ool00`00Oomoo`0mOol00`00Oomoo`0HOol20002248010008@PQ20000c6<00<003W>>Lh0
3`000Tic00H005JeE[EF]@00G_L20002IcT00`00Oomoo`0EOol00`00Oomoo`0oOol00`00Oomoo`0>
Ool010006<HHaP0023W>00@004YB000000I>L`03001F]EJe00AF]@03001JeU[F00=JeP8000=W>@@0
0005IcT00000001S600200000`Q2Oomoo`03Ool200058@P01@00?Nl00000CW<010000dic00<005:D
DY@025:D00<006<HHaP016<H00<006<HHaP016<H00<007>LLi`00P002goo00<007ooOol05Goo000k
Ool00`00Oomoo`0POol00`0048@000020002:DX00`00>Lh00002?Nl02@00AS56<@00BU8004YB001:
DP020002CW<00`00BU800004DY@30003DY@00`00E[EF]@02E[D00`00F]IJeP03F]H2000015[F0000
000016mk00@006mkKg/000aoo`03001oogoo03eoo`03001oogoo01Qoo`800088@P04000Q22480003
<H`30002AS400`00BU9:DP02BU800`00DYABU003DY@00`00E[D000020003E[D01000G_L000000VLi
00<007ooOol05Goo00<007ooOol0<7oo100000=oo`0000000`001Goo00<007ooOol03Goo0P0000<H
aP00>Lh01cW>00<004YBBU800TYB00<004icCW<00Tic00<005JeE[D01EJe00<005[FF]H00`0000YN
m`00IcUW>@00IcT006]J001W>@800004HaP000000007Ool;0008CW<20007DY@00`00HaQS6006HaP2
0004HaP01@00LiacW7>L00002goo00<007ooOol05Goo000lOol00`00Oomoo`0OOol300000bDY0000
000200001CW>000mkcg_00000THa0`0014YB00L004icCW=>L`00BU800004DY@00`00E[D000030004
E[D00`00F]IJeP02F]H40003JeX01@00Kg]_Nfmk00000Vmk00<007ooOol02Woo00<007ooOol0?7oo
0P006Woo0P000PQ200D002488@P0036<00<000A6<@03001:DTYB009:DP03001BU5:D00=BU004001F
]EJeE[D300000eJe001Nm`020002IcT00`00Oomoo`0EOol00`00Oomoo`0`Ool01000Oomoo`000goo
00<007ooOol01Goo00<007ooOol03Goo00<001S6000023W>00<004YBBU8014YB0P0000=>L`00E[D0
1UJe10000ekg00D006LiIcT006Li00<00005IcT00000001S60030008Ool200000cF]00000003000<
CW<20004DY@00`00HaQS6009HaP01000HaQS66<H0P000W>L00<007ooOol02Goo00<007ooOol05Goo
000mOol00`00Oomoo`0OOol40002;F/00`00>Lh00002?Nl01000AS56<@000Tic00H004YBBU8004ic
0003CW<20004DY@00`00E[EF]@02E[D:0002HaP00`00JeX00002JeX01@00Kg]_Nfmk00000Vmk00<0
07ooOol02Woo00<007ooOol0?7oo00<007ooOol06Goo0`0000@8@P008@PQ2080008mk`030016<DHa
0096<@03001:DTYB009:DP03001BU5:D009BU003001F]EJe009F]@03001Nmekg00800004G_L00000
IcT2000GOol00`00Oomoo`0`Ool01000Oomoo`0017oo00<007ooOol017oo00<007ooOol03Goo0P00
2CW>00<004YBBU801TYB2P001Ukg00<006Li00000VLi00@006]J001W>@@00003JeX0000000Qoo`<0
00De[@8000a>L`800003DY@006<H00aS60<0009cW003001cW00000]oo`03001oogoo01Eoo`00?Woo
00<007ooOol07Woo0P0000<Li`00;F/00Re[00@003W>000mk`800004AS4004icCW<30002CW<00`00
CW=>L`02CW<00`00DYABU002DY@00`00E[EF]@03E[D00`00G_MNm`02G_L00`00HaQS6003HaP01P00
JeY[FP00JeX000=_N`03001_N`0000eoo`03001oogoo03aoo`03001oogoo01Uoo`<0000324800000
008000@mk`050016<DHaAS400004BU800`00DYABU002DY@01@00E[EF]EJe000015kg100000=W>@00
000067oo00<007ooOol0;goo00@007ooOol000Eoo`03001oogoo00=oo`03001oogoo00eoo`03000i
cSW>00LicP03001:DTYB00I:DP03001F]EJe00EF]@03001Nmekg00ENm`03001W>@0000=W>@04001[
FP00IcT400000f]J00000004Ool40002Ool20006=Jd00`00CW=>L`0:CW<2000@HaP01@00LiacW7>L
000037oo00<007ooOol05Goo000oOol00`00Oomoo`0MOol01@00248LiacW00000Re[00P003W>0000
04Ha0016<@000Tic00<004ic00000dic00@004icCW=>L`8000=BU003001F]EJe00=F]@03001Nmekg
00=Nm`03001S66<H00=S6003001[FV]J008000A_N`8000eoo`03001oogoo03aoo`03001oogoo01Uo
o`<00003248003F]008e[@05000mkcg_?Nl00003AS400`00BU9:DP02BU800`00DYABU002DY@01P00
E[EF]EJe001Nm`@0009W>@04001W>@00000HOol00`00Oomoo`0_Ool01000Oomoo`000Woo00@007oo
Ool000Aoo`03001oogoo00eoo`<000LicP03001:DTYB00I:DP03001F]EJe00IF]@03001Nmekg00EN
m`03001W>@00009W>@<00004IcT00000IcT2000016]J000007oo0`001goo0P001SF]00<004icCW<0
1dic0P000UJe00<006<HHaP036<H0P000g>L0P0037oo00<007ooOol057oo0010Ool00`00Oomoo`0L
Ool300001QcW000]Jbe[000icP800096<@80009>L`04001>Ldic0003CW<01P00CW=>L`00E[D000=B
U003001F]EJe009F]@03001Nmekg00=Nm`03001S66<H00=S6004001[FV]JJeX20003Kg/00`00Oomo
o`0=Ool00`00Oomoo`0kOol00`00Oomoo`0JOol40003=Jd01@00?Nlmkcg_000014Ha00D004YBBU9:
DP0000ABU003001F]@0000<0009Nm`04001W>FLiIcT3000IOol00`00Oomoo`0_Ool40003Ool30004
Ool2000>Ool00`007>L00006>Lh00`00BU9:DP07BU800`00E[EF]@05E[D00`00G_MNm`05G_L00`00
IcT00002IcT00`00JeY[FP0400000fLi001[FP04000;Ool40004=Jd20005CW<20005E[D00`00HaQS
6009HaP20002Li`01@00Li`007oo000037oo00<007ooOol057oo0011Ool2000MOol200001acW000]
Jbe[000icP0000=6<@80009>L`04001>Ldic0004CW<20003E[D200000e:D001F]@03E[D00`00G_MN
m`03G_L00`00HaQS6003HaP01000JeY[FV]J0P000fmk00<007ooOol03Goo00<007ooOol0>goo0P00
6goo0P000RU:00@003F]=Jd000<mk`030016<DHa0096<@05001:DTYBBU800003DY@30002E[D01@00
G_MNmekg00000VLi00@006mk000001Uoo`03001oogoo03eoo`03001oogoo00eoo`04000LiacW0004
>Lh00`00BU9:DP07BU800`00E[EF]@06E[D00`00G_MNm`05G_L20002IcT01@00JeY[FP00JeX00P00
00=W>@0000000`003Woo0P0000<mk`0000000`0000@e[@00CW=>L`8000QF]@03001S66<H00MS6003
001cW7>L009cW003001cW000009oo`03001oogoo00Uoo`03001oogoo01Aoo`00@goo00<007ooOol0
6Woo00@000Q20000008]J`03000icP0000A6<@07001>LdicCW<004ic000014ic0P001EJe0P001EJe
00<005kgG_L00Ukg00<006<HHaP00f<H00<006]JJeX00V]J00@006mkKg/0011oo`03001oogoo03]o
o`03001oogoo01Yoo`80008YBP04000e[CF]0003?Nl00`00AS56<@02AS401000BU9:DTYB100000=B
U000E[D00UJe00@005kgG_L000=W>@03001_Nfmk008001Qoo`03001oogoo03eoo`03001oogoo00eo
o`04000LiacW0003>Lh00`00BU9:DP07BU800`00E[EF]@06E[D00`00G_MNm`05G_L20002IcT01000
JeY[FV]J1P000V]J0P003goo0`0013g_10002eJe0P001V<H00<007>LLi`00g>L00<007>L00000goo
00<007ooOol027oo00<007ooOol057oo0014Ool00`00Oomoo`0IOol3000011cW000]Jbe[0`000dHa
0P000Tic00@004icCW<000A>L`03001F]EJe00AF]@03001F]EJe00=F]@03001Nmekg00=Nm`03001S
66<H009S6003001[FV]J009[FP@0011oo`03001oogoo03]oo`03001oogoo01Yoo`05000@Q000:DX0
0003=Jd01000?Nlmk`0014Ha1@000e:D00<005JeE[D00UJe00@005kgG_L000=W>@06001_Nfmk001o
o`005goo00<007ooOol0?Goo00<007ooOol037oo0P000acW00@003W>>Lh000Q:DP03001F]EJe00MF
]@03001Nmekg00ANm`80009W>@03001[FV]J00@00005IcT006Li001[FP03000AOol00a2400000005
?Nl00`00E[EF]@0;E[D01000HaQS66<H0P001g>L00<007ooOol00goo00<007ooOol01goo00<007oo
Ool057oo0015Ool00`00Oomoo`0HOol300001AcW000]Jbe[00000THa00D004HaAS4004ic00<0009>
L`07001>LdicCW<005Je00001EJe0`000eJe00<005kgG_L00ekg00<006<HHaP00V<H00<006]JJeX0
0f]J0P004Goo00<007ooOol0>goo00<007ooOol06Woo00D00124000YBP0000<e[@04000mkcg_?Nl5
0003BU801@00DYABU5:D00000eJe00D005kgG_MNm`00009W>@03001_Nfmk0080009oo`03001oogoo
01Aoo`03001oogoo03eoo`03001oogoo00aoo`03000LiacW008Li`03000icP0000U:DP03001F]EJe
00IF]@03001Nmekg00=Nm`800004HaP006LiIcT40002JeX2000016Li000006]J0P004goo00<003W>
00000P000cg_00<005JeE[D02eJe00<006<H00002G>L00<007ooOol017oo00<007ooOol01Woo00<0
07ooOol057oo0016Ool00`00Oomoo`0HOol01000248000000Re[00@004HaAS400096<@04001>Ldic
CW<20003CW<20003E[D01@00E[EF]EJe00000e[F0P0000=F]@00G_L015kg00<006<HHaP00f<H00@0
06]JJeY[FP@0011oo`03001oogoo03]oo`03001oogoo01Yoo`06000@Q124000YBP000SF]1@000THa
2`000UJe00D005kgG_MNm`00009W>@03001_Nfmk008000=oo`03001oogoo01=oo`03001oogoo03eo
o`03001oogoo00aoo`03000LiacW00<Li`03001:DTYB00M:DP03001F]EJe00MF]@04001NmekgG_L2
0002HaP40003JeX01P00JeX006Li001W>@D001Aoo`80008icP<00003?Nl0000000YF]@<000QcW080
00Qoo`03001oogoo00Eoo`03001oogoo01Aoo`00Agoo00<007ooOol05goo00P000Q2000002e[000i
cP000THa00<004Ha00000dic00<005:D00000P001EJe00D005JeE[EF]@0000EJePD000=Nm`03001S
66<H009S6003001[FP0000<0009oo`03001oogoo00eoo`03001oogoo03]oo`03001oogoo01]oo`04
000@Q000:DX500000cg_0000000200000dic001BU002DY@01P00E[EF]EJe001JeP@00003G_L006Li
009W>@04001_N`00Li`5Ool00`00Oomoo`0BOol00`00Oomoo`0mOol00`00Oomoo`0<Ool00`007>LL
i`037>L30006BU800`00E[EF]@07E[D00`00G_MNm`0200000f<H0000000200000fLi001[FP02JeX0
1@00JeY[FP00IcT00`0000=[FP0000005goo0P000cW>0P0000<mk`00E[D00UJe1P000Ukg00<007>L
Li`017>L0P002Woo00<007ooOol01Goo00<007ooOol057oo0018Ool00`00Oomoo`0FOol300001AcW
000003W>00000dHa0P0014ic00D005:DDYABU00000EF]@04001F]EJe0006F]H01000HaQS66<H2P00
0WNm0P000goo00<007ooOol037oo00<007ooOol0>goo00<007ooOol06goo1@000cF]0P0000=:DP00
CW<00Tic00D005:DDYABU00000=F]@06001JeU[FF]H006<H0P000fLi00@006mk000000Ioo`03001o
ogoo015oo`03001oogoo03eoo`03001oogoo00aoo`03000LiacW008Li`040016<DHaAS420003BU80
0`00E[EF]@08E[D60003HaP20003JeX01@00JeY[FP00IcT00P0000A_N`00000001Yoo`8000@icP@0
00UNm`04001cW7>LLi`2000=Ool00`00Oomoo`05Ool00`00Oomoo`0COol004Uoo`03001oogoo01Io
o`040008@P000003>Lh01@00AS56<DHa000014ic00D005:DDYABU00000IF]@8000IJeP03001S66<H
00AS6003001_Nfmk00=_N`03001g_GNm008000Aoo`03001oogoo00aoo`03001oogoo03]oo`03001o
ogoo01Qoo`D0000448@002U:=Jd200001DHa001:DTYB00000dic00@005:DDY@0009F]@05001JeU[F
F]H00003HaP01FLi00000000Kg/00`001goo00<007ooOol047oo00<007ooOol0?7oo00<007ooOol0
37oo00<001cW7>L00acW00<004HaAS400dHa0P0000=:DP00E[D00eJe20001V<H0P000f]J00@006]J
JeX0009W>@03001_Nfmk00<001aoo`<0008icP03001Nmekg00UNm`04001cW000000@Ool00`00Oomo
o`04Ool00`00Oomoo`0COol004Yoo`8001Moo`8000038@P003W>008icP030016<DHa00<0009>L`03
001BU5:D009BU003001F]EJe00AF]@03001JeU[F00AJeP03001S66<H00AS6003001_Nfmk009_N`04
001g_GNmMkd20005Ool00`00Oomoo`0;Ool00`00Oomoo`0kOol00`00Oomoo`0DOol40004Ool01@00
48@002U:00000dHa00@004YBBU8000=>L`04001BU5:D0002E[D01@00F]IJeU[F00000V<H00H006Li
IcUW>@00Kg/20009Ool00`00Oomoo`0?Ool00`00Oomoo`0lOol00`00Oomoo`0<Ool00`007>LLi`03
7>L00`00AS56<@05AS4600000eJe000000030008HaP00`00JeY[FP02JeX01@00JeY[FP00IcT00P00
0Vmk0P0087oo0P0000<icP00G_L01Ekg1P004goo00<007ooOol00goo00<007ooOol04goo001<Ool0
0`00Oomoo`0EOol200000b48000icP02>Lh00`00AS400002DY@00`00CW<00005DY@01000E[EF]EJe
0P0000=Nm`00000015[F00<006<HHaP00f<H00<006mkKg/00fmk00<007NmMkd00P001goo00<007oo
Ool02Woo00<007ooOol0>Woo00<007ooOol04Woo0`0027oo0P0000@@Q000@Q0000=6<@04001:DTYB
0002CW<01000DYABU0000UJe00@005[FF]H0009S6004001W>FLiIcT5000:Ool00`00Oomoo`0>Ool0
0`00Oomoo`0lOol00`00Oomoo`0<Ool00`007>LLi`037>L00`00AS56<@05AS401000E[EF]EJe0`00
15Je00<006<HHaP016<H0`000f]J00@006]JJeX0009W>@03001_Nfmk00<0029oo`/001Moo`03001o
ogoo009oo`03001oogoo01=oo`00CGoo00<007ooOol05Goo0P0000LQ2000>LhicP00AS400003DY@3
0004DY@01000E[D000001Ekg00D005[FF]IJeP0000ES6003001_Nfmk009_N`05001g_GNmMkd00009
Ool00`00Oomoo`0:Ool00`00Oomoo`0iOol00`00Oomoo`0>Ool4000<Ool00`0048@00002@Q001000
AS56<@000TYB00@004icCW<0009BU004001F]EJe0002F]H01000HaQS60000VLi00<006mkKg/00P00
37oo00<007ooOol03Woo00<007ooOol0>goo00<007ooOol03Goo00D001cW7>LLi`0000Q6<@@000IF
]@03001S66<H00AS6007001W>FLiIcT006]J00000f]J00<006Li00000fmk0P00AGoo00<007ooOol0
0Woo00<007ooOol04goo001>Ool00`00Oomoo`0EOol30003>Lh20005DY@70008G_L30005HaP00`00
Kg]_N`03Kg/01000Mkeg_@002goo00<007ooOol02Goo00<007ooOol0>Goo00<007ooOol02Woo1000
47oo0P0000DicP00@Q1240000096<@04001:DTYB0002CW<01@00DY@005Je00000U[F00@006<HHaP0
009W>@03001_Nfmk00<000eoo`03001oogoo00eoo`03001oogoo03]oo`03001oogoo00eoo`05000L
iacW7>L00005AS43000:E[D00`00HaQS6003HaP00`00IcUW>@02IcT00`00JeX00002JeX40003Kg/2
0016Ool01@00Oomoogoo00005Goo001?Ool00`00Oomoo`0EOol30003>Lh00`00DYABU003DY@00`00
F]IJeP04F]H00`00G_MNm`07G_L20004HaP00`00Kg]_N`02Kg/01@00Mkeg_GNm000037oo00<007oo
Ool027oo00<007ooOol0>Goo0P0027oo0`005Goo00H003W>>Lh0048@0003AS400`00BU800002CW<0
1@00DY@005Je00000U[F00H006<HHaP006Li0002Kg/01000Mkd000003Woo00<007ooOol037oo00<0
07ooOol0>goo00<007ooOol03Woo00D001cW7>L004Ha00@00096<@03001F]EJe00QF]@04001S66<H
HaP20006IcT20002JeX00`00Kg/00003Kg/20018Ool01000Oomoo`005Goo001@Ool00`00Oomoo`0F
Ool20002>Lh30004DY@00`00F]IJeP04F]H00`00G_MNm`04G_L20002IcT200000f<H001_N`04Kg/0
1000Mkeg_@003Woo00<007ooOol01goo00<007ooOol0>Goo00<007ooOol00goo100067oo0`0000@i
cP00@Q000096<@05001:DTYB001>L`0500001e[F001S66<H001W>@00009_N`04001g_@00000?Ool0
0`00Oomoo`0<Ool00`00Oomoo`0kOol00`00Oomoo`0>Ool50006AS400`00E[EF]@08E[D01000HaQS
60001fLi00<006]JJeX00P000Vmk00<006mkKg/00P00BWoo00<007oo00005Goo001AOol00`00Oomo
o`0FOol30003BU820002DY@00`00F]IJeP04F]H00`00G_MNm`02G_L20006IcT40002Kg/01000Mkd0
000047oo00<007ooOol01Woo00<007ooOol0<Goo100017oo00<007oo00000`007Goo00L002e[000i
cP00@Q000002AS4500000e[F001Nm`030002HaP20002Kg/01@00Mkd007Nm000047oo00<007ooOol0
2goo00<007ooOol0>goo00<007ooOol037oo1@0024Ha00<005JeE[D01eJe00@006<HHaP000MW>@05
001[FV]JJeX00002Kg/00`00Kg]_N`03001;Ool00`00Ool0000DOol0059oo`8001Moo`8000E:DP80
0003DY@005[F00EJeP04001Nm`000007IcT20003Kg/4000BOol00`00Oomoo`06Ool00`00Oomoo`0b
Ool00`00Oomoo`04Ool2000QOol200001Be[000icP00@Q000P0000EF]@00F]H005[F00<00003HaP0
06<H00<0009_N`<00003Mkd007oo015oo`03001oogoo00Yoo`03001oogoo03Yoo`03001oogoo00Yo
o`<000Aoo`8000M6<@03001F]EJe00MF]@<000MW>@05001[FV]JJeX00003Kg/01@00Kg/007Nm0000
C7oo00<007oo000057oo001DOol00`00Oomoo`0EOol01000;F/0000014YB2P0026Li00<006mkKg/0
0fmk0P005Goo00<007ooOol01Goo00<007ooOol0<goo00<007ooOol017oo00<007ooOol087oo0P00
00<]J`000000100000AF]@00001JeP<00003HaP006<H00<00004Kg/00000Mkd2000DOol00`00Oomo
o`09Ool00`00Oomoo`0jOol00`00Oomoo`06Ool40007Ool01000@Q00000014Ha00<005JeE[D01UJe
0`0026Li00<006]JJeX00V]J00P006mkKg/006mk001g_@00CWoo0P0057oo001EOol00`00Oomoo`0F
Ool01000;F/000000dYB00<005kgG_L01Ukg00<006LiIcT00fLi0P0016mk0P0067oo00<007ooOol0
17oo00<007ooOol0=7oo00<007ooOol017oo0P008Goo0P000Re[1@0000=F]@000000100000AS6000
IcUW>@<00003Mkd0000001Ioo`03001oogoo00Qoo`03001oogoo03Yoo`03001oogoo00=oo`<000ao
o`030012448@00<00004AS4005JeE[D60002G_L00`00IcUW>@04IcT20004JeX01000Kg]_Nfmk1000
CWoo0`0057oo001FOol00`00Oomoo`0GOol40002BU800`00G_MNm`06G_L200000fLi00000004Kg/2
000KOol00`00Oomoo`03Ool00`00Oomoo`0aOol01000Oomoo`0027oo00<007ooOol07Woo00H007oo
000]J`00E[D;00000fLi00000004000IOol00`00Oomoo`07Ool00`00Oomoo`0jOol6000?Ool00`00
@Q124003@Q040009G_L00`00IcUW>@02IcT00`00JeY[FP03JeX01000Kg]_Nfmk0`00C7oo10005goo
001GOol00`00Oomoo`0FOol00`00Ool000040006G_L60003Kg/2000NOol00`00Oomoo`02Ool00`00
Oomoo`0bOol30009Ool00`00Oomoo`0LOol01000Oomoogoo50006goo00<007ooOol01Woo00<007oo
Ool0>Woo0P0057oo00<0048@@Q000d8@00<005kgG_L02Ekg00D006LiIcUW>@0000E[FP03001_Nfmk
009_N`8004Uoo`@001]oo`00F7oo00<007ooOol05Goo00<007ooOol00Woo20000U:D1`008Goo00D0
07ooOomoo`00045oo`03001oogoo01]oo`03001oogoo00=ooa0001ioo`03001oogoo00Eoo`03001o
ogoo02]oo`@00005Ool007ooOol00007Ool2000COol00`00@Q124004@Q000`00G_MNm`08G_L01000
IcUW>@001F]J00@006mkKg]_N`<004Ioo`@001moo`00FGoo00<007ooOol057oo00<007ooOol017oo
2000:Goo00D007ooOomoo`00045oo`8001]oo`03001oogoo00Ioo`D00003G_L0000000@0021oo`03
001oogoo00Aoo`03001oogoo02]oo`04001oogoo0003Ool30008Ool00`00Oomoo`0AOol00`00@Q12
4003@Q000`00G_MNm`09G_L20006JeX00`00Kg/000030013Ool4000SOol005Yoo`8001Aoo`03001o
ogoo03Ioo`04001oogoo0013Ool00`00Oomoo`0HOol00`00Oomoo`07Ool8000TOol00`00Oomoo`03
Ool00`00Oomoo`0[Ool01000Oomoo`000goo0P002Woo00<007ooOol047oo00<0048@@Q00148@00<0
05kgG_L01ekg100016]J0`0000=cW0000000@7oo10009goo001LOol00`00Oomoo`0AOol00`00Oomo
o`0fOol01000Oomoo`00A7oo00<007ooOol05goo00<007ooOol0=7oo00<007ooOol00goo00<007oo
Ool0:Woo00@007ooOol000=oo`8000]oo`80015oo`8000A24003001Nmekg00ANm`<000A[FPD0009c
W003001oogoo03aoo`@002]oo`00GGoo00<007ooOol047oo00<007ooOol0=goo00<007oo0000AGoo
00<007ooOol05Woo00<007ooOol0=Goo00<007ooOol00Woo00<007ooOol0:Woo00@007ooOol000Ao
o`03001oogoo00]oo`03001oogoo011oo`@00003@Q0005kg00@000Q[FP80009cW08003]oo`@002mo
o`00GWoo00<007ooOol03goo00<007ooOol0>7oo0P00AWoo0P005Woo00<007ooOol0=Woo00D007oo
Oomoo`0002aoo`@000Aoo`03001oogoo00aoo`8001=oo`@000Y[FPH003Uoo`@003=oo`00Ggoo00<0
07ooOol03Woo00<007ooOol0>7oo0P00B7oo00<007ooOol04goo00<007ooOol0=goo00@007ooOol0
04Eoo`03001oogoo01Aoo`@000A[FP@003Uoo`@003Moo`00H7oo00<007ooOol03Goo00<007ooOol0
=7oo1000Bgoo00<007ooOol04Woo00<007ooOol0>7oo00<007oo0000AWoo0P005Woo00<007oo0000
0`00>Goo1000>goo001QOol00`00Oomoo`0<Ool00`00Oomoo`0`Ool4001@Ool00`00Oomoo`0AOol0
0`00Oomoo`0iOol20018Ool00`00Oomoo`0COol00`00Oomoo`0hOol4000oOol0069oo`03001oogoo
00]oo`03001oogoo02aoo`@005Eoo`03001oogoo011oo`03001oogoo03Qoo`<004Uoo`03001oogoo
019oo`03001oogoo03Aoo`@004=oo`00Hgoo0P002goo00<007ooOol0:7oo1000FWoo0P0047oo00<0
07ooOol0=7oo1000CGoo0P004Woo00<007ooOol0;goo1@00Agoo001UOol00`00Oomoo`08Ool00`00
Oomoo`0TOol4001POol00`00Oomoo`0=Ool00`00Oomoo`0`Ool4001COol00`00Oomoo`0?Ool00`00
Oomoo`0[Ool4001<Ool006Ioo`03001oogoo00Moo`03001oogoo021oo`@006Eoo`03001oogoo00ao
o`03001oogoo02aoo`@005Qoo`8000moo`03001oogoo02Moo`@0051oo`00Igoo00<007ooOol01Woo
00<007ooOol077oo1000JWoo00<007ooOol02goo00<007ooOol0:7oo1000GWoo00<007ooOol037oo
00<007ooOol08goo1000E7oo001XOol00`00Oomoo`05Ool00`00Oomoo`0HOol4001_Ool2000;Ool0
0`00Oomoo`0TOol4001SOol00`00Oomoo`0;Ool00`00Oomoo`0OOol4001HOol006Uoo`03001oogoo
00Aoo`03001oogoo01Aoo`@007Eoo`03001oogoo00Qoo`03001oogoo021oo`@006Qoo`8000Yoo`03
001oogoo01aoo`@005aoo`00JWoo00<007ooOol00goo00<007ooOol047oo1000NWoo00<007ooOol0
1goo00<007ooOol077oo1000KWoo00<007ooOol01goo00<007ooOol067oo1000H7oo001[Ool20003
Ool00`00Oomoo`0<Ool4001oOol00`00Oomoo`06Ool00`00Oomoo`0HOol4001cOol20007Ool00`00
Oomoo`0DOol4001TOol006eoo`04001oogoo000:Ool40024Ool20006Ool00`00Oomoo`0DOol4001i
Ool00`00Oomoo`04Ool00`00Oomoo`0@Ool4001XOol006ioo`03001oo`0000Ioo`@008Yoo`03001o
ogoo00=oo`03001oogoo011oo`@007ioo`8000Aoo`03001oogoo00aoo`@006aoo`00Kgoo0P000Woo
1000Sgoo00<007ooOol00Woo00<007ooOol037oo1000Q7oo00D007ooOomoo`0000Yoo`@0071oo`00
L7oo0`00U7oo00D007ooOomoo`0000Yoo`@008Uoo`04001oogoo0006Ool4001dOol00?moo`Uoo`80
0003Ool007oo00Eoo`@008ioo`<0009oo`@007Qoo`00ogoo2goo0P000Woo1000U7oo0`00O7oo003o
Ool<Ool3003oOolHOol005Moo`05001oogooOol0003oOoo:Ool005Moo`03001oogoo009oo`03001o
ogoo0?moolMoo`00Egoo0`0000=oo`00000027oo0P000Woo00<007oo00000P000Woo00<007oo0000
100000=oo`00Ool0ogoo[7oo001GOol00`00Oomoo`020004Ool50002Ool01@00Ool007oo00000Woo
00@007oo001oo`80009oo`03001oo`00091oo`03001oogoo0:moo`04001oogoo001VOol005Moo`04
001oogoo0009Ool02000Oomoo`00Ool007oo0002Ool01P00Ool007oo001oo`<00003Ool007oo091o
o`04001oo`00002ZOol300001Goo001oogoo000027oo0P000Woo00<007oo00000P000Woo00<007oo
0000100000=oo`00Ool0B7oo001GOol01000Oomoo`002Goo00P007ooOol007oo001oo`000Woo00D0
07oo001oo`0000=oo`03001oo`0008aoo`<00003Ool00000009oo`03001oogoo00Ioo`80009oo`03
001oo`000080009oo`03001oo`0000@00003Ool007oo08aoo`03001oogoo00<00003Ool007oo00=o
o`D0009oo`05001oo`00Ool00002Ool01000Ool007oo0P000Woo00<007oo0000BGoo001GOol3000;
Ool20002Ool50002Ool700000goo001oo`2:Ool01000Oomoo`001Woo00<007ooOol01@000Woo00D0
07oo001oo`00009oo`04001oo`00Ool20002Ool00`00Ool0002=Ool01000Oomoo`0037oo00P007oo
Ool007oo001oo`000Woo00H007oo001oo`00Ool300000goo001oo`18Ool006]oo`03001oogoo009o
o`03001oo`0000Eoo`03001oogoo08Uoo`04001oogoo0005Ool00`00Oomoo`05Ool02000Oomoo`00
Ool007oo0002Ool01P00Ool007oo001oo`<00003Ool007oo08aoo`04001oogoo000<Ool02000Oomo
o`00Ool007oo0002Ool01@00Ool007oo00000goo00<007oo0000BGoo003oOol5Ool01000Oomoo`00
3Goo00P007ooOol007oo001oo`000Woo00D007oo001oo`0000=oo`03001oo`0008ioo`<000eoo`80
009oo`D0009oo`L00003Ool007oo04Qoo`00ogoo1Woo0`003Woo0P000Woo1@000Woo1`0000=oo`00
Ool0Sgoo00<007ooOol04Goo00<007ooOol00Woo00<007oo00001Goo00<007ooOol0Agoo003oOol8
Ool00`00Oomoo`0BOol00`00Oomoo`02Ool00`00Ool00005Ool00`00Oomoo`3iOol00?mooomoobMo
o`00ogooogoo9goo003oOoooOolWOol00?mooomoobMoo`00ogooogoo9goo0000\
\>"],
  ImageRangeCache->{{{0, 548.25}, {189.5, 0}} -> {-0.0841965, -0.0260804, 
  0.00577911, 0.00577911}, {{14.5625, 176.75}, {184.938, 
  4.5}} -> {-0.167364, -0.114368, 0.00607277, 0.00607277}, {{193, 355.188}, {
  184.938, 4.5}} -> {-1.27353, -0.0921373, 0.00630699, 0.00630699}, {{
  371.438, 533.625}, {184.938, 4.5}} -> {-2.41421, -0.113309, 0.00636458, 
  0.00636458}}],

Cell[TextData[StyleBox[" "]], "Text"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .45459 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 0.31746 0.0108235 0.31746 [
[ 0 0 0 0 ]
[ 1 .45459 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 0 m
1 0 L
1 .45459 L
0 .45459 L
closepath
clip
newpath
% Start of sub-graphic
p
0.0238095 0.0108235 0.321429 0.443762 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-0.218821 1.52137 0 1.52137 [
[.5 1.46717 -20.875 0 ]
[.5 1.46717 20.875 10.0625 ]
[.08216 .16577 -19.474 -10 ]
[.08216 .16577 1.15096 0 ]
[.1847 .1225 -19.0038 -10 ]
[.1847 .1225 1.62125 0 ]
[.29019 .07798 -7.58187 -10 ]
[.29019 .07798 .85563 0 ]
[.39878 .03217 -13.2463 -10 ]
[.39878 .03217 1.87867 0 ]
[.51059 -0.01498 -12.9015 -10 ]
[.51059 -0.01498 2.22355 0 ]
[.23092 .00363 -7.13257 -10 ]
[.23092 .00363 .80493 0 ]
[.54184 -0.00533 0 -7.10061 ]
[.54184 -0.00533 20.625 2.89939 ]
[.65925 .16129 0 -6.96183 ]
[.65925 .16129 20.625 3.03817 ]
[.76511 .31152 0 -6.84024 ]
[.76511 .31152 8.4375 3.15976 ]
[.86103 .44765 0 -6.73285 ]
[.86103 .44765 15.125 3.26715 ]
[.94834 .57159 0 -6.63731 ]
[.94834 .57159 15.125 3.36269 ]
[.85434 .27867 0 -6.84024 ]
[.85434 .27867 7.9375 3.15976 ]
[.07587 .18984 -20.625 -2.89898 ]
[.07587 .18984 0 7.10102 ]
[.05559 .36867 -20.625 -3.01764 ]
[.05559 .36867 0 6.98236 ]
[.03341 .5643 -8.4375 -3.14821 ]
[.03341 .5643 0 6.85179 ]
[.00904 .77922 -15.125 -3.2926 ]
[.00904 .77922 0 6.7074 ]
[-0.01784 1.01642 -15.125 -3.45311 ]
[-0.01784 1.01642 0 6.54689 ]
[-0.05576 .59732 -7.9375 -3.14821 ]
[-0.05576 .59732 0 6.85179 ]
[ 0 0 0 0 ]
[ 1 1.45467 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 1.46717 -81.875 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18.0625 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.062500 moveto
 460.000000 14.062500 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.062500 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(d) show
67.437500 11.500000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(yz) show
73.937500 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
79.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
100.750000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.09363 .18048 m
.52287 0 L
s
.09479 .17999 m
.10111 .1871 L
s
gsave
.08216 .16577 -80.474 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.19695 .13704 m
.20308 .14431 L
s
gsave
.1847 .1225 -80.0038 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.30205 .09285 m
.30798 .10028 L
s
gsave
.29019 .07798 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4102 .04737 m
.41592 .05497 L
s
gsave
.39878 .03217 -74.2463 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.52156 .00055 m
.52704 .00832 L
s
gsave
.51059 -0.01498 -73.9015 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.115 .1715 m
.11876 .17578 L
s
.13531 .16295 m
.13906 .16726 L
s
.15575 .15436 m
.15947 .15869 L
s
.17629 .14572 m
.17999 .15007 L
s
.21773 .1283 m
.22139 .13268 L
s
.23863 .11951 m
.24226 .12391 L
s
.25965 .11067 m
.26326 .1151 L
s
.28079 .10179 m
.28437 .10623 L
s
.32343 .08386 m
.32696 .08834 L
s
.34494 .07482 m
.34844 .07932 L
s
.36657 .06572 m
.37005 .07024 L
s
.38832 .05657 m
.39178 .06111 L
s
.43222 .03812 m
.43562 .0427 L
s
.45436 .02881 m
.45773 .03341 L
s
.47662 .01944 m
.47997 .02407 L
s
.49903 .01003 m
.50234 .01467 L
s
gsave
.23092 .00363 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.52287 0 m
.93124 .57888 L
s
.5243 .00203 m
.51554 .00572 L
s
gsave
.54184 -0.00533 -61 -11.1006 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.64155 .16824 m
.6327 .17171 L
s
gsave
.65925 .16129 -61 -10.9618 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.74726 .31808 m
.73834 .32137 L
s
gsave
.76511 .31152 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.84306 .45388 m
.83407 .45699 L
s
gsave
.86103 .44765 -61 -10.7329 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.93027 .57751 m
.92124 .58047 L
s
gsave
.94834 .57159 -61 -10.6373 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.5548 .04526 m
.54953 .04744 L
s
.58448 .08734 m
.5792 .08949 L
s
.61339 .12832 m
.60809 .13043 L
s
.66899 .20713 m
.66367 .20919 L
s
.69574 .24505 m
.69041 .24707 L
s
.72182 .28202 m
.71648 .28402 L
s
.77208 .35327 m
.76672 .35521 L
s
.79631 .38761 m
.79094 .38953 L
s
.81996 .42114 m
.81458 .42303 L
s
.86562 .48586 m
.86022 .4877 L
s
.88766 .51711 m
.88226 .51893 L
s
.90921 .54765 m
.90379 .54945 L
s
gsave
.85434 .27867 -61 -10.8402 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.09363 .18048 m
0 1.0137 L
s
.0934 .18247 m
.10217 .17879 L
s
gsave
.07587 .18984 -81.625 -6.89898 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.07327 .36166 m
.08211 .35815 L
s
gsave
.05559 .36867 -81.625 -7.01764 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.05124 .5577 m
.06016 .55439 L
s
gsave
.03341 .5643 -69.4375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.02704 .77307 m
.03604 .77 L
s
gsave
.00904 .77922 -76.125 -7.2926 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00033 1.0108 m
.00941 1.00799 L
s
gsave
-0.01784 1.01642 -76.125 -7.45311 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.08853 .22581 m
.0938 .22362 L
s
.08356 .2701 m
.08884 .26794 L
s
.07847 .31537 m
.08376 .31324 L
s
.06795 .409 m
.07326 .40693 L
s
.06251 .45743 m
.06783 .45538 L
s
.05694 .50698 m
.06228 .50497 L
s
.04541 .60962 m
.05077 .60767 L
s
.03943 .66279 m
.04481 .66088 L
s
.03331 .71726 m
.0387 .71538 L
s
.02061 .83028 m
.02602 .82847 L
s
.01402 .88893 m
.01944 .88716 L
s
.00726 .94909 m
.0127 .94736 L
s
gsave
-0.05576 .59732 -68.9375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.09363 .18048 m
0 1.0137 L
s
0 1.0137 m
.5592 1.45467 L
s
.5592 1.45467 m
.55672 .70144 L
s
.55672 .70144 m
.09363 .18048 L
s
.52287 0 m
.93124 .57888 L
s
.93124 .57888 m
1 1.3537 L
s
1 1.3537 m
.51861 .85346 L
s
.51861 .85346 m
.52287 0 L
s
.09363 .18048 m
0 1.0137 L
s
0 1.0137 m
.51861 .85346 L
s
.51861 .85346 m
.52287 0 L
s
.52287 0 m
.09363 .18048 L
s
.55672 .70144 m
.93124 .57888 L
s
.93124 .57888 m
1 1.3537 L
s
1 1.3537 m
.5592 1.45467 L
s
.5592 1.45467 m
.55672 .70144 L
s
0 0 m
1 0 L
1 1.45467 L
0 1.45467 L
closepath
clip
newpath
.5 Mabswid
.53445 .63506 m .53302 .69047 L .62513 .67768 L p .745 g
F P
0 g
s
.62513 .67768 m .61172 .61755 L .53445 .63506 L p .745 g
F P
0 g
s
.63243 .74351 m .62513 .67768 L .53302 .69047 L p .663 g
F P
0 g
s
.61172 .61755 m .62513 .67768 L .71902 .65396 L p .677 g
F P
0 g
s
.62513 .67768 m .63243 .74351 L .73472 .7205 L p .588 g
F P
0 g
s
.73472 .7205 m .71902 .65396 L .62513 .67768 L p .588 g
F P
0 g
s
.71902 .65396 m .69034 .595 L .61172 .61755 L p .677 g
F P
0 g
s
.69034 .595 m .71902 .65396 L .79349 .62255 L p .619 g
F P
0 g
s
.71902 .65396 m .73472 .7205 L .81608 .68499 L p .504 g
F P
0 g
s
.81608 .68499 m .79349 .62255 L .71902 .65396 L p .504 g
F P
0 g
s
.53302 .69047 m .53225 .75074 L .63243 .74351 L p .663 g
F P
0 g
s
.63193 .80297 m .63243 .74351 L .53225 .75074 L p .568 g
F P
0 g
s
.63243 .74351 m .63193 .80297 L .73373 .78244 L p .475 g
F P
0 g
s
.73373 .78244 m .73472 .7205 L .63243 .74351 L p .475 g
F P
0 g
s
.73472 .7205 m .73373 .78244 L .81453 .74619 L p .339 g
F P
0 g
s
.81453 .74619 m .81608 .68499 L .73472 .7205 L p .339 g
F P
0 g
s
.5363 .59559 m .53445 .63506 L .61172 .61755 L p .841 g
F P
0 g
s
.61172 .61755 m .59436 .57502 L .5363 .59559 L p .841 g
F P
0 g
s
.59436 .57502 m .61172 .61755 L .69034 .595 L p .777 g
F P
0 g
s
.69034 .595 m .65331 .55538 L .59436 .57502 L p .777 g
F P
0 g
s
.65331 .55538 m .69034 .595 L .75225 .57036 L p .728 g
F P
0 g
s
.79349 .62255 m .75225 .57036 L .69034 .595 L p .619 g
F P
0 g
s
.4767 .64984 m .46401 .6951 L .53302 .69047 L p .845 g
F P
0 g
s
.53302 .69047 m .53445 .63506 L .4767 .64984 L p .845 g
F P
0 g
s
.53225 .75074 m .53302 .69047 L .46401 .6951 L p .761 g
F P
0 g
s
.46401 .6951 m .45725 .74513 L .53225 .75074 L p .761 g
F P
0 g
s
.53232 .80464 m .53225 .75074 L .45725 .74513 L p .665 g
F P
0 g
s
.53225 .75074 m .53232 .80464 L .63193 .80297 L p .568 g
F P
0 g
s
.62245 .84414 m .63193 .80297 L .53232 .80464 L p .393 g
F P
0 g
s
.63193 .80297 m .62245 .84414 L .71352 .82758 L p .254 g
F P
0 g
s
.71352 .82758 m .73373 .78244 L .63193 .80297 L p .254 g
F P
0 g
s
.75225 .57036 m .79349 .62255 L .82988 .5926 L p .511 g
F P
0 g
s
.79349 .62255 m .81608 .68499 L .85604 .64653 L p .331 g
F P
0 g
s
.85604 .64653 m .82988 .5926 L .79349 .62255 L p .331 g
F P
0 g
s
.49314 .61867 m .4767 .64984 L .53445 .63506 L p .926 g
F P
0 g
s
.53445 .63506 m .5363 .59559 L .49314 .61867 L p .926 g
F P
0 g
s
.73373 .78244 m .71352 .82758 L .78523 .79432 L p .079 g
F P
0 g
s
.78523 .79432 m .81453 .74619 L .73373 .78244 L p .079 g
F P
0 g
s
.75225 .57036 m .69924 .53914 L .65331 .55538 L p .728 g
F P
0 g
s
.69924 .53914 m .75225 .57036 L .7819 .5517 L p .566 g
F P
0 g
s
.82988 .5926 m .7819 .5517 L .75225 .57036 L p .511 g
F P
0 g
s
.81608 .68499 m .81453 .74619 L .85376 .70345 L p .053 g
F P
0 g
s
.85376 .70345 m .85604 .64653 L .81608 .68499 L p .053 g
F P
0 g
s
.5363 .59559 m .59436 .57502 L p .55392 .58032 L .922 g
F P
0 g
s
.55392 .58032 m .54899 .58096 L .54197 .58906 L .922 g
F
.55392 .58032 m .54197 .58906 L .5363 .59559 L F
.59436 .57502 m .57564 .56128 L .53829 .58237 L p F P
0 g
s
.55859 .58553 m .5401 .6037 L p .56883 .56941 L F P
s
.57564 .56128 m .57108 .56776 L p .56883 .56941 L F P
s
.57108 .56776 m .55859 .58553 L p .56883 .56941 L F P
s
.57564 .56128 m .59436 .57502 L .65331 .55538 L p .866 g
F P
0 g
s
.65331 .55538 m .6135 .54595 L .57564 .56128 L p .866 g
F P
0 g
s
.57731 .57535 m .55859 .58553 L p .60064 .55523 L F P
s
.6135 .54595 m .60439 .55335 L p .60064 .55523 L F P
s
.60439 .55335 m .57731 .57535 L p .60064 .55523 L F P
s
.53829 .58237 m .53765 .58661 L p .54899 .58096 L .922 g
F P
0 g
s
.53765 .58661 m .5363 .59559 L p .54899 .58096 L .922 g
F P
0 g
s
.6135 .54595 m .65331 .55538 L .69924 .53914 L p .706 g
F P
0 g
s
.45725 .74513 m .45804 .79035 L .53232 .80464 L p .665 g
F P
0 g
s
.53334 .84119 m .53232 .80464 L .45804 .79035 L p .488 g
F P
0 g
s
.53232 .80464 m .53334 .84119 L .62245 .84414 L p .393 g
F P
0 g
s
.5401 .6037 m .5266 .62975 L p .5346 .59731 L F P
s
.50959 .60988 m .5363 .59559 L p .52118 .60406 L .836 g
F P
0 g
s
.52118 .60406 m .51075 .6099 L .50959 .60988 L .836 g
F
.5363 .59559 m .53829 .58237 L .51075 .6099 L p F P
0 g
s
.53829 .58237 m .53936 .59493 L p .5346 .59731 L F P
s
.53936 .59493 m .5401 .6037 L p .5346 .59731 L F P
s
.51075 .6099 m .49314 .61867 L .50959 .60988 L p .836 g
F P
0 g
s
.45416 .66892 m .43682 .69987 L .46401 .6951 L p .863 g
F P
0 g
s
.46401 .6951 m .4767 .64984 L .45416 .66892 L p .863 g
F P
0 g
s
.45725 .74513 m .46401 .6951 L .43682 .69987 L p .851 g
F P
0 g
s
.69924 .53914 m .64264 .53819 L .6135 .54595 L p .706 g
F P
0 g
s
.64264 .53819 m .69924 .53914 L .72054 .53259 L p .053 g
F P
0 g
s
.7819 .5517 m .72054 .53259 L .69924 .53914 L p .566 g
F P
0 g
s
.81453 .74619 m .78523 .79432 L .81907 .75262 L p F P
s
.81907 .75262 m .85376 .70345 L .81453 .74619 L p F P
s
.43682 .69987 m .42774 .73554 L .45725 .74513 L p .851 g
F P
0 g
s
.45804 .79035 m .45725 .74513 L .42774 .73554 L p .746 g
F P
0 g
s
.60356 .85589 m .62245 .84414 L .53334 .84119 L p .095 g
F P
0 g
s
.62245 .84414 m .60356 .85589 L .67319 .84439 L p .019 g
F P
0 g
s
.67319 .84439 m .71352 .82758 L .62245 .84414 L p .019 g
F P
0 g
s
.47664 .64933 m .45416 .66892 L .4767 .64984 L p .687 g
F P
0 g
s
.4767 .64984 m .49314 .61867 L .47664 .64933 L p .687 g
F P
0 g
s
.71352 .82758 m .67319 .84439 L .72728 .81812 L p .651 g
F P
0 g
s
.72728 .81812 m .78523 .79432 L .71352 .82758 L p .651 g
F P
0 g
s
.7819 .5517 m .82988 .5926 L .81589 .57824 L p .145 g
F P
0 g
s
.82988 .5926 m .85604 .64653 L .84088 .62015 L p .36 g
F P
0 g
s
.84088 .62015 m .81589 .57824 L .82988 .5926 L p .36 g
F P
0 g
s
.72054 .53259 m .7819 .5517 L .76953 .55102 L p .13 g
F P
0 g
s
.81589 .57824 m .76953 .55102 L .7819 .5517 L p .145 g
F P
0 g
s
.45804 .79035 m .46736 .82145 L .53334 .84119 L p .488 g
F P
0 g
s
.53536 .8504 m .53334 .84119 L .46736 .82145 L p .128 g
F P
0 g
s
.53334 .84119 m .53536 .8504 L .60356 .85589 L p .095 g
F P
0 g
s
.42774 .73554 m .42918 .76886 L .45804 .79035 L p .746 g
F P
0 g
s
.46736 .82145 m .45804 .79035 L .42918 .76886 L p .494 g
F P
0 g
s
.85604 .64653 m .85376 .70345 L .83795 .66886 L p .742 g
F P
0 g
s
.83795 .66886 m .84088 .62015 L .85604 .64653 L p .742 g
F P
0 g
s
.55859 .58553 m .57564 .56128 L .6135 .54595 L p F P
s
.57731 .57535 m .6135 .54595 L .64264 .53819 L p .07 g
F P
0 g
s
.5401 .6037 m .53829 .58237 L .57564 .56128 L p F P
s
.5005 .64658 m .47664 .64933 L .49314 .61867 L p F P
s
.49314 .61867 m .51075 .6099 L .5005 .64658 L p F P
s
.72054 .53259 m .65559 .54206 L .64264 .53819 L p .053 g
F P
0 g
s
.65559 .54206 m .72054 .53259 L .71043 .54461 L p .144 g
F P
0 g
s
.76953 .55102 m .71043 .54461 L .72054 .53259 L p .13 g
F P
0 g
s
.64264 .53819 m .59153 .57425 L .57731 .57535 L p .07 g
F P
0 g
s
.59153 .57425 m .64264 .53819 L .65559 .54206 L p .158 g
F P
0 g
s
.78523 .79432 m .72728 .81812 L .75165 .78363 L p .896 g
F P
0 g
s
.75165 .78363 m .81907 .75262 L .78523 .79432 L p .896 g
F P
0 g
s
.5266 .62975 m .51075 .6099 L .53829 .58237 L p F P
s
.85376 .70345 m .81907 .75262 L .80357 .71545 L p .922 g
F P
0 g
s
.80357 .71545 m .83795 .66886 L .85376 .70345 L p .922 g
F P
0 g
s
.42918 .76886 m .44233 .79293 L .46736 .82145 L p .494 g
F P
0 g
s
.46736 .82145 m .48533 .8302 L .53536 .8504 L p .128 g
F P
0 g
s
.48533 .8302 m .46736 .82145 L .44233 .79293 L p .183 g
F P
0 g
s
.47635 .70251 m .4632 .71732 L .43682 .69987 L p .031 g
F P
0 g
s
.43682 .69987 m .45416 .66892 L .47635 .70251 L p .031 g
F P
0 g
s
.42774 .73554 m .43682 .69987 L .4632 .71732 L p .031 g
F P
0 g
s
.57596 .82941 m .60356 .85589 L .53536 .8504 L p .697 g
F P
0 g
s
.60356 .85589 m .57596 .82941 L .61432 .82359 L p .773 g
F P
0 g
s
.61432 .82359 m .67319 .84439 L .60356 .85589 L p .773 g
F P
0 g
s
.4632 .71732 m .45646 .73569 L .42774 .73554 L p .031 g
F P
0 g
s
.42918 .76886 m .42774 .73554 L .45646 .73569 L p .039 g
F P
0 g
s
.65559 .54206 m .59757 .58412 L .59153 .57425 L p .158 g
F P
0 g
s
.59757 .58412 m .65559 .54206 L .64843 .56292 L p .277 g
F P
0 g
s
.71043 .54461 m .64843 .56292 L .65559 .54206 L p .144 g
F P
0 g
s
.49341 .69457 m .47635 .70251 L .45416 .66892 L p .071 g
F P
0 g
s
.45416 .66892 m .47664 .64933 L .49341 .69457 L p .071 g
F P
0 g
s
.52174 .66427 m .5005 .64658 L .51075 .6099 L p .051 g
F P
0 g
s
.51075 .6099 m .5266 .62975 L .52174 .66427 L p .051 g
F P
0 g
s
.67319 .84439 m .61432 .82359 L .64359 .80846 L p .833 g
F P
0 g
s
.64359 .80846 m .72728 .81812 L .67319 .84439 L p .833 g
F P
0 g
s
.71043 .54461 m .76953 .55102 L .71516 .58124 L p .283 g
F P
0 g
s
.76953 .55102 m .81589 .57824 L .75085 .59541 L p .38 g
F P
0 g
s
.75085 .59541 m .71516 .58124 L .76953 .55102 L p .38 g
F P
0 g
s
.81589 .57824 m .84088 .62015 L .76974 .62325 L p .552 g
F P
0 g
s
.76974 .62325 m .75085 .59541 L .81589 .57824 L p .552 g
F P
0 g
s
.53828 .8246 m .53536 .8504 L .48533 .8302 L p .629 g
F P
0 g
s
.53536 .8504 m .53828 .8246 L .57596 .82941 L p .697 g
F P
0 g
s
.45646 .73569 m .45789 .75383 L .42918 .76886 L p .039 g
F P
0 g
s
.44233 .79293 m .42918 .76886 L .45789 .75383 L p .057 g
F P
0 g
s
.64843 .56292 m .71043 .54461 L .66962 .58418 L p .275 g
F P
0 g
s
.71516 .58124 m .66962 .58418 L .71043 .54461 L p .283 g
F P
0 g
s
.81907 .75262 m .75165 .78363 L .73835 .75091 L p .91 g
F P
0 g
s
.73835 .75091 m .80357 .71545 L .81907 .75262 L p .91 g
F P
0 g
s
.54639 .65271 m .55859 .58553 L .57731 .57535 L p .108 g
F P
0 g
s
.57731 .57535 m .55145 .64786 L .54639 .65271 L p .108 g
F P
0 g
s
.55145 .64786 m .57731 .57535 L .59153 .57425 L p .273 g
F P
0 g
s
.54139 .66386 m .5401 .6037 L .55859 .58553 L p .038 g
F P
0 g
s
.55859 .58553 m .54639 .65271 L .54139 .66386 L p .038 g
F P
0 g
s
.51138 .69598 m .49341 .69457 L .47664 .64933 L p .113 g
F P
0 g
s
.47664 .64933 m .5005 .64658 L .51138 .69598 L p .113 g
F P
0 g
s
.84088 .62015 m .83795 .66886 L .76672 .65954 L p .716 g
F P
0 g
s
.76672 .65954 m .76974 .62325 L .84088 .62015 L p .716 g
F P
0 g
s
.59153 .57425 m .55525 .64974 L .55145 .64786 L p .273 g
F P
0 g
s
.55525 .64974 m .59153 .57425 L .59757 .58412 L p .372 g
F P
0 g
s
.72728 .81812 m .64359 .80846 L .65597 .78789 L p .858 g
F P
0 g
s
.65597 .78789 m .75165 .78363 L .72728 .81812 L p .858 g
F P
0 g
s
.53521 .95837 m .53293 1.0647 L .62675 1.09259 L p .674 g
F P
0 g
s
.62675 1.09259 m .60515 .97602 L .53521 .95837 L p .674 g
F P
0 g
s
.63965 1.19836 m .62675 1.09259 L .53293 1.0647 L p .642 g
F P
0 g
s
.60515 .97602 m .62675 1.09259 L .72308 1.08554 L p .63 g
F P
0 g
s
.62675 1.09259 m .63965 1.19836 L .7509 1.19317 L p .584 g
F P
0 g
s
.7509 1.19317 m .72308 1.08554 L .62675 1.09259 L p .584 g
F P
0 g
s
.53778 .68086 m .5266 .62975 L .5401 .6037 L p .045 g
F P
0 g
s
.5401 .6037 m .54139 .66386 L .53778 .68086 L p .045 g
F P
0 g
s
.53293 1.0647 m .53158 1.16044 L .63965 1.19836 L p .642 g
F P
0 g
s
.64311 1.28061 m .63965 1.19836 L .53158 1.16044 L p .602 g
F P
0 g
s
.63965 1.19836 m .64311 1.28061 L .75848 1.27855 L p .521 g
F P
0 g
s
.75848 1.27855 m .7509 1.19317 L .63965 1.19836 L p .521 g
F P
0 g
s
.44233 .79293 m .46697 .80183 L .48533 .8302 L p .183 g
F P
0 g
s
.51095 .81063 m .48533 .8302 L .46697 .80183 L p .575 g
F P
0 g
s
.48533 .8302 m .51095 .81063 L .53828 .8246 L p .629 g
F P
0 g
s
.48425 .91893 m .4644 1.00552 L .53293 1.0647 L p .731 g
F P
0 g
s
.53293 1.0647 m .53521 .95837 L .48425 .91893 L p .731 g
F P
0 g
s
.53158 1.16044 m .53293 1.0647 L .4644 1.00552 L p .715 g
F P
0 g
s
.64843 .56292 m .59367 .60708 L .59757 .58412 L p .277 g
F P
0 g
s
.59367 .60708 m .64843 .56292 L .62213 .60561 L p .323 g
F P
0 g
s
.66962 .58418 m .62213 .60561 L .64843 .56292 L p .275 g
F P
0 g
s
.45789 .75383 m .46821 .76802 L .44233 .79293 L p .057 g
F P
0 g
s
.46697 .80183 m .44233 .79293 L .46821 .76802 L p .251 g
F P
0 g
s
.72308 1.08554 m .67671 .96901 L .60515 .97602 L p .63 g
F P
0 g
s
.67671 .96901 m .72308 1.08554 L .79852 1.04556 L p .591 g
F P
0 g
s
.72308 1.08554 m .7509 1.19317 L .8385 1.14672 L p .523 g
F P
0 g
s
.8385 1.14672 m .79852 1.04556 L .72308 1.08554 L p .523 g
F P
0 g
s
.83795 .66886 m .80357 .71545 L .73946 .69782 L p .806 g
F P
0 g
s
.73946 .69782 m .76672 .65954 L .83795 .66886 L p .806 g
F P
0 g
s
.4644 1.00552 m .4527 1.08369 L .53158 1.16044 L p .715 g
F P
0 g
s
.53124 1.23398 m .53158 1.16044 L .4527 1.08369 L p .694 g
F P
0 g
s
.53158 1.16044 m .53124 1.23398 L .64311 1.28061 L p .602 g
F P
0 g
s
.59757 .58412 m .5568 .65873 L .55525 .64974 L p .372 g
F P
0 g
s
.5568 .65873 m .59757 .58412 L .59367 .60708 L p .404 g
F P
0 g
s
.53826 .85318 m .53521 .95837 L .60515 .97602 L p .704 g
F P
0 g
s
.60515 .97602 m .57619 .86124 L .53826 .85318 L p .704 g
F P
0 g
s
.57619 .86124 m .60515 .97602 L .67671 .96901 L p .674 g
F P
0 g
s
.52716 .70796 m .51138 .69598 L .5005 .64658 L p .184 g
F P
0 g
s
.5005 .64658 m .52174 .66427 L .52716 .70796 L p .184 g
F P
0 g
s
.7509 1.19317 m .75848 1.27855 L .84929 1.22927 L p .416 g
F P
0 g
s
.84929 1.22927 m .8385 1.14672 L .7509 1.19317 L p .416 g
F P
0 g
s
.53652 .70321 m .52174 .66427 L .5266 .62975 L p .097 g
F P
0 g
s
.5266 .62975 m .53778 .68086 L .53652 .70321 L p .097 g
F P
0 g
s
.51079 .83404 m .48425 .91893 L .53521 .95837 L p .745 g
F P
0 g
s
.53521 .95837 m .53826 .85318 L .51079 .83404 L p .745 g
F P
0 g
s
.67671 .96901 m .61484 .85658 L .57619 .86124 L p .674 g
F P
0 g
s
.61484 .85658 m .67671 .96901 L .73213 .93923 L p .65 g
F P
0 g
s
.79852 1.04556 m .73213 .93923 L .67671 .96901 L p .591 g
F P
0 g
s
.63718 1.32709 m .64311 1.28061 L .53124 1.23398 L p .52 g
F P
0 g
s
.64311 1.28061 m .63718 1.32709 L .7458 1.32861 L p .385 g
F P
0 g
s
.7458 1.32861 m .75848 1.27855 L .64311 1.28061 L p .385 g
F P
0 g
s
.46594 .86511 m .43948 .92459 L .4644 1.00552 L p .825 g
F P
0 g
s
.4644 1.00552 m .48425 .91893 L .46594 .86511 L p .825 g
F P
0 g
s
.4527 1.08369 m .4644 1.00552 L .43948 .92459 L p .831 g
F P
0 g
s
.42418 .44901 m .35321 .38289 L .36913 .47855 L p .61 g
F P
0 g
s
.36913 .47855 m .43365 .52182 L .42418 .44901 L p .61 g
F P
0 g
s
.30653 .46124 m .36913 .47855 L .35321 .38289 L p .663 g
F P
0 g
s
.43365 .52182 m .36913 .47855 L .4139 .57977 L p .568 g
F P
0 g
s
.36913 .47855 m .30653 .46124 L .36818 .57525 L p .61 g
F P
0 g
s
.36818 .57525 m .4139 .57977 L .36913 .47855 L p .61 g
F P
0 g
s
.75165 .78363 m .65597 .78789 L .64772 .76756 L p .848 g
F P
0 g
s
.64772 .76756 m .73835 .75091 L .75165 .78363 L p .848 g
F P
0 g
s
.49091 .50582 m .43565 .39359 L .42418 .44901 L p .585 g
F P
0 g
s
.42418 .44901 m .48523 .54004 L .49091 .50582 L p .585 g
F P
0 g
s
.35321 .38289 m .42418 .44901 L .43565 .39359 L p .639 g
F P
0 g
s
.48523 .54004 m .42418 .44901 L .43365 .52182 L p .562 g
F P
0 g
s
.4527 1.08369 m .44985 1.14348 L .53124 1.23398 L p .694 g
F P
0 g
s
.53188 1.27441 m .53124 1.23398 L .44985 1.14348 L p .626 g
F P
0 g
s
.53124 1.23398 m .53188 1.27441 L .63718 1.32709 L p .52 g
F P
0 g
s
.66962 .58418 m .71516 .58124 L .63142 .65125 L p .341 g
F P
0 g
s
.71516 .58124 m .75085 .59541 L .64984 .65588 L p .408 g
F P
0 g
s
.64984 .65588 m .63142 .65125 L .71516 .58124 L p .408 g
F P
0 g
s
.43948 .92459 m .42401 .97898 L .4527 1.08369 L p .831 g
F P
0 g
s
.44985 1.14348 m .4527 1.08369 L .42401 .97898 L p .8 g
F P
0 g
s
.43365 .52182 m .48967 .58759 L .48523 .54004 L p .562 g
F P
0 g
s
.48967 .58759 m .43365 .52182 L .46151 .60286 L p .526 g
F P
0 g
s
.4139 .57977 m .46151 .60286 L .43365 .52182 L p .568 g
F P
0 g
s
.62213 .60561 m .66962 .58418 L .60775 .65674 L p .318 g
F P
0 g
s
.63142 .65125 m .60775 .65674 L .66962 .58418 L p .341 g
F P
0 g
s
.43565 .39359 m .37079 .30633 L .35321 .38289 L p .639 g
F P
0 g
s
.28348 .34813 m .35321 .38289 L .37079 .30633 L p .706 g
F P
0 g
s
.35321 .38289 m .28348 .34813 L .30653 .46124 L p .663 g
F P
0 g
s
.25665 .47055 m .30653 .46124 L .28348 .34813 L p .724 g
F P
0 g
s
.30653 .46124 m .25665 .47055 L .33231 .58822 L p .649 g
F P
0 g
s
.33231 .58822 m .36818 .57525 L .30653 .46124 L p .649 g
F P
0 g
s
.59367 .60708 m .55566 .67495 L .5568 .65873 L p .404 g
F P
0 g
s
.55566 .67495 m .59367 .60708 L .58061 .64463 L p .404 g
F P
0 g
s
.62213 .60561 m .58061 .64463 L .59367 .60708 L p .323 g
F P
0 g
s
.4139 .57977 m .36818 .57525 L .45323 .67741 L p .544 g
F P
0 g
s
.36818 .57525 m .33231 .58822 L .43522 .6873 L p .571 g
F P
0 g
s
.43522 .6873 m .45323 .67741 L .36818 .57525 L p .571 g
F P
0 g
s
.50527 .48946 m .46535 .36436 L .43565 .39359 L p .588 g
F P
0 g
s
.43565 .39359 m .49091 .50582 L .50527 .48946 L p .588 g
F P
0 g
s
.37079 .30633 m .43565 .39359 L .46535 .36436 L p .646 g
F P
0 g
s
.80357 .71545 m .73835 .75091 L .68911 .73122 L p .83 g
F P
0 g
s
.68911 .73122 m .73946 .69782 L .80357 .71545 L p .83 g
F P
0 g
s
.46151 .60286 m .4139 .57977 L .47648 .67599 L p .512 g
F P
0 g
s
.45323 .67741 m .47648 .67599 L .4139 .57977 L p .544 g
F P
0 g
s
.75085 .59541 m .76974 .62325 L .65933 .66934 L p .503 g
F P
0 g
s
.65933 .66934 m .64984 .65588 L .75085 .59541 L p .503 g
F P
0 g
s
.73213 .93923 m .79852 1.04556 L .83292 .98046 L p .549 g
F P
0 g
s
.79852 1.04556 m .8385 1.14672 L .87891 1.06724 L p .434 g
F P
0 g
s
.87891 1.06724 m .83292 .98046 L .79852 1.04556 L p .434 g
F P
0 g
s
.54187 .76069 m .57596 .82941 L .53828 .8246 L closepath p .759 g
F P
0 g
s
.54187 .76069 m .61432 .82359 L .57596 .82941 L closepath p .768 g
F P
0 g
s
.46697 .80183 m .50131 .79148 L .51095 .81063 L p .575 g
F P
0 g
s
.50117 .80811 m .46594 .86511 L .48425 .91893 L p .813 g
F P
0 g
s
.48425 .91893 m .51079 .83404 L .50117 .80811 L p .813 g
F P
0 g
s
.4632 .71732 m .47635 .70251 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.45646 .73569 m .4632 .71732 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.46821 .76802 m .48696 .77508 L .46697 .80183 L p .251 g
F P
0 g
s
.50131 .79148 m .46697 .80183 L .48696 .77508 L p .514 g
F P
0 g
s
.5322 .62579 m .50527 .48946 L .49091 .50582 L p .54 g
F P
0 g
s
.49091 .50582 m .52839 .63216 L .5322 .62579 L p .54 g
F P
0 g
s
.52839 .63216 m .49091 .50582 L .48523 .54004 L p .546 g
F P
0 g
s
.54187 .76069 m .64359 .80846 L .61432 .82359 L closepath p .774 g
F P
0 g
s
.47635 .70251 m .49341 .69457 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.31331 .6155 m .33231 .58822 L .25665 .47055 L p .682 g
F P
0 g
s
.33231 .58822 m .31331 .6155 L .42597 .7033 L p .587 g
F P
0 g
s
.42597 .7033 m .43522 .6873 L .33231 .58822 L p .587 g
F P
0 g
s
.45789 .75383 m .45646 .73569 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.48523 .54004 m .52687 .64738 L .52839 .63216 L p .546 g
F P
0 g
s
.52687 .64738 m .48523 .54004 L .48967 .58759 L p .528 g
F P
0 g
s
.73213 .93923 m .64429 .84043 L .61484 .85658 L p .65 g
F P
0 g
s
.64429 .84043 m .73213 .93923 L .75665 .89313 L p .631 g
F P
0 g
s
.83292 .98046 m .75665 .89313 L .73213 .93923 L p .549 g
F P
0 g
s
.54187 .76069 m .53828 .8246 L .51095 .81063 L closepath p .748 g
F P
0 g
s
.58061 .64463 m .62213 .60561 L .58309 .67232 L p .335 g
F P
0 g
s
.60775 .65674 m .58309 .67232 L .62213 .60561 L p .318 g
F P
0 g
s
.75848 1.27855 m .7458 1.32861 L .83083 1.27999 L p .122 g
F P
0 g
s
.83083 1.27999 m .84929 1.22927 L .75848 1.27855 L p .122 g
F P
0 g
s
.52456 .49385 m .50562 .36747 L .46535 .36436 L p .56 g
F P
0 g
s
.46535 .36436 m .50527 .48946 L .52456 .49385 L p .56 g
F P
0 g
s
.41816 .26281 m .46535 .36436 L .50562 .36747 L p .616 g
F P
0 g
s
.46535 .36436 m .41816 .26281 L .37079 .30633 L p .646 g
F P
0 g
s
.5373 .62915 m .52456 .49385 L .50527 .48946 L p .49 g
F P
0 g
s
.50527 .48946 m .5322 .62579 L .5373 .62915 L p .49 g
F P
0 g
s
.46151 .60286 m .5031 .64333 L .48967 .58759 L p .526 g
F P
0 g
s
.5031 .64333 m .46151 .60286 L .50084 .68433 L p .478 g
F P
0 g
s
.47648 .67599 m .50084 .68433 L .46151 .60286 L p .512 g
F P
0 g
s
.52174 .66427 m .535 .71804 L .53309 .72011 L .223 g
F
.53799 .73018 m .53642 .72695 L p .535 .71804 L F P
0 g
s
.53642 .72695 m .53309 .72011 L p .535 .71804 L .223 g
F P
0 g
s
.52174 .66427 m .53652 .70321 L .53799 .73018 L p .223 g
F P
0 g
s
.52716 .70796 m .53799 .73018 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.53309 .72011 m .52716 .70796 L .5251 .69135 L p .223 g
F P
0 g
s
.5251 .69135 m .52174 .66427 L p .53309 .72011 L .223 g
F P
0 g
s
.51138 .69598 m .52716 .70796 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.52213 .70238 m .53668 .72873 L .53163 .69776 L p .453 g
F P
0 g
s
.54187 .76069 m .53668 .72873 L .52213 .70238 L closepath p .479 g
F P
0 g
s
.53163 .69776 m .53668 .72873 L .53797 .73668 L p .498 g
F P
0 g
s
.53797 .73668 m .54187 .76069 L .53834 .73904 L p .498 g
F P
0 g
s
.53834 .73904 m .53163 .69776 L p .53797 .73668 L .498 g
F P
0 g
s
.52805 .66992 m .53163 .69776 L .53834 .73904 L p .047 g
F P
0 g
s
.53834 .73904 m .54187 .76069 L .53885 .74085 L p .047 g
F P
0 g
s
.53885 .74085 m .52805 .66992 L p .53834 .73904 L .047 g
F P
0 g
s
.30627 .25348 m .37079 .30633 L .41816 .26281 L p .722 g
F P
0 g
s
.37079 .30633 m .30627 .25348 L .28348 .34813 L p .706 g
F P
0 g
s
.76974 .62325 m .76672 .65954 L .65731 .6891 L p .601 g
F P
0 g
s
.65731 .6891 m .65933 .66934 L .76974 .62325 L p .601 g
F P
0 g
s
.48967 .58759 m .52805 .66992 L .52687 .64738 L p .528 g
F P
0 g
s
.52805 .66992 m .48967 .58759 L .5031 .64333 L p .496 g
F P
0 g
s
.28348 .34813 m .22701 .34835 L .25665 .47055 L p .724 g
F P
0 g
s
.22947 .50407 m .25665 .47055 L .22701 .34835 L p .794 g
F P
0 g
s
.25665 .47055 m .22947 .50407 L .31331 .6155 L p .682 g
F P
0 g
s
.54235 .64219 m .54369 .51949 L .52456 .49385 L p .396 g
F P
0 g
s
.52456 .49385 m .5373 .62915 L .54235 .64219 L p .396 g
F P
0 g
s
.50562 .36747 m .52456 .49385 L .54369 .51949 L p .496 g
F P
0 g
s
.22701 .34835 m .28348 .34813 L .30627 .25348 L p .8 g
F P
0 g
s
.8385 1.14672 m .84929 1.22927 L .8909 1.14092 L p .19 g
F P
0 g
s
.8909 1.14092 m .87891 1.06724 L .8385 1.14672 L p .19 g
F P
0 g
s
.49341 .69457 m .51138 .69598 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.55145 .64786 m .54187 .76069 L .54639 .65271 L closepath p .403 g
F P
0 g
s
.55525 .64974 m .54187 .76069 L .55145 .64786 L closepath p .312 g
F P
0 g
s
.57619 .86124 m .54187 .76069 L .53826 .85318 L closepath p .724 g
F P
0 g
s
.61484 .85658 m .54187 .76069 L .57619 .86124 L closepath p .717 g
F P
0 g
s
.46821 .76802 m .45789 .75383 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.42401 .97898 m .42057 1.02087 L .44985 1.14348 L p .8 g
F P
0 g
s
.4557 1.17589 m .44985 1.14348 L .42057 1.02087 L p .515 g
F P
0 g
s
.44985 1.14348 m .4557 1.17589 L .53188 1.27441 L p .626 g
F P
0 g
s
.54639 .65271 m .54187 .76069 L .54139 .66386 L closepath p .47 g
F P
0 g
s
.54369 .51949 m .54568 .40448 L .50562 .36747 L p .496 g
F P
0 g
s
.48316 .26271 m .50562 .36747 L .54568 .40448 L p .54 g
F P
0 g
s
.50562 .36747 m .48316 .26271 L .41816 .26281 L p .616 g
F P
0 g
s
.5568 .65873 m .54187 .76069 L .55525 .64974 L closepath p .17 g
F P
0 g
s
.31634 .652 m .31331 .6155 L .22947 .50407 L p .681 g
F P
0 g
s
.31331 .6155 m .31634 .652 L .42793 .72223 L p .58 g
F P
0 g
s
.42793 .72223 m .42597 .7033 L .31331 .6155 L p .58 g
F P
0 g
s
.54187 .76069 m .65597 .78789 L .64359 .80846 L closepath p .778 g
F P
0 g
s
.64429 .84043 m .54187 .76069 L .61484 .85658 L closepath p .711 g
F P
0 g
s
.53826 .85318 m .54187 .76069 L .51079 .83404 L closepath p .731 g
F P
0 g
s
.54598 .66395 m .55752 .56417 L .54369 .51949 L p .308 g
F P
0 g
s
.54369 .51949 m .54235 .64219 L .54598 .66395 L p .308 g
F P
0 g
s
.54568 .40448 m .54369 .51949 L .55752 .56417 L p .414 g
F P
0 g
s
.54139 .66386 m .54187 .76069 L .53778 .68086 L closepath p .526 g
F P
0 g
s
.54187 .76069 m .51095 .81063 L .50131 .79148 L closepath p .738 g
F P
0 g
s
.54187 .76069 m .47648 .67599 L .45323 .67741 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .45323 .67741 L .43522 .6873 L closepath p .479 g
F P
0 g
s
.55752 .56417 m .57464 .47158 L .54568 .40448 L p .414 g
F P
0 g
s
.54804 .30909 m .54568 .40448 L .57464 .47158 L p .44 g
F P
0 g
s
.54568 .40448 m .54804 .30909 L .48316 .26271 L p .54 g
F P
0 g
s
.48657 .8081 m .46711 .83694 L .43948 .92459 L p .86 g
F P
0 g
s
.46711 .83694 m .45587 .86399 L .44731 .89486 L p F P
s
.43948 .92459 m .46453 .86828 L p .45844 .87768 L .86 g
F P
0 g
s
.43948 .92459 m .46711 .83694 L p .43037 .95802 L F P
s
.42401 .97898 m .42706 .96825 L p .43037 .95802 L F P
s
.42706 .96825 m .43948 .92459 L p .43037 .95802 L F P
s
.45149 .88841 m .46711 .83694 L .44731 .89486 L F
.45587 .86399 m .45365 .88527 L .4529 .88835 L p .03 g
F P
0 g
s
.4529 .88835 m .42057 1.02087 L p .44464 .9139 L .03 g
F P
0 g
s
.44731 .89486 m .42401 .97898 L p .45149 .88841 L F P
s
.44464 .9139 m .45587 .86399 L .4529 .88835 L .03 g
F
.46594 .86511 m .48657 .8081 L p .45844 .87768 L .86 g
F P
0 g
s
.46453 .86828 m .46594 .86511 L p .45844 .87768 L .86 g
F P
0 g
s
.58061 .64463 m .5521 .6981 L .55566 .67495 L p .404 g
F P
0 g
s
.5521 .6981 m .58061 .64463 L .56165 .69666 L p .386 g
F P
0 g
s
.58309 .67232 m .56165 .69666 L .58061 .64463 L p .335 g
F P
0 g
s
.55566 .67495 m .54187 .76069 L .5568 .65873 L closepath p .031 g
F P
0 g
s
.54187 .76069 m .43522 .6873 L .42597 .7033 L closepath p .479 g
F P
0 g
s
.5031 .64333 m .53163 .69776 L .52805 .66992 L p .496 g
F P
0 g
s
.53163 .69776 m .5031 .64333 L .52213 .70238 L p .453 g
F P
0 g
s
.50084 .68433 m .52213 .70238 L .5031 .64333 L p .478 g
F P
0 g
s
.76672 .65954 m .73946 .69782 L .6428 .71175 L p .681 g
F P
0 g
s
.6428 .71175 m .65731 .6891 L .76672 .65954 L p .681 g
F P
0 g
s
.3708 .19645 m .41816 .26281 L .48316 .26271 L p .678 g
F P
0 g
s
.41816 .26281 m .3708 .19645 L .30627 .25348 L p .722 g
F P
0 g
s
.54187 .76069 m .50084 .68433 L .47648 .67599 L closepath p .479 g
F P
0 g
s
.48696 .77508 m .46821 .76802 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.73835 .75091 m .64772 .76756 L .62054 .75359 L p .815 g
F P
0 g
s
.62054 .75359 m .68911 .73122 L .73835 .75091 L p .815 g
F P
0 g
s
.62293 1.3277 m .63718 1.32709 L .53188 1.27441 L p .154 g
F P
0 g
s
.63718 1.32709 m .62293 1.3277 L .7152 1.33241 L p .037 g
F P
0 g
s
.7152 1.33241 m .7458 1.32861 L .63718 1.32709 L p .037 g
F P
0 g
s
.53339 1.27309 m .53188 1.27441 L .4557 1.17589 L p .057 g
F P
0 g
s
.53188 1.27441 m .53339 1.27309 L .62293 1.3277 L p .154 g
F P
0 g
s
.54724 .69263 m .56232 .62336 L .55752 .56417 L p .277 g
F P
0 g
s
.55752 .56417 m .54598 .66395 L .54724 .69263 L p .277 g
F P
0 g
s
.57464 .47158 m .55752 .56417 L .56232 .62336 L p .352 g
F P
0 g
s
.42057 1.02087 m .42401 .97898 L .45587 .86399 L p .03 g
F P
0 g
s
.54187 .76069 m .50131 .79148 L .51244 .77285 L closepath p .728 g
F P
0 g
s
.51239 .7815 m .48657 .8081 L .46594 .86511 L p .936 g
F P
0 g
s
.46594 .86511 m .50117 .80811 L .51239 .7815 L p .936 g
F P
0 g
s
.53778 .68086 m .54187 .76069 L .53652 .70321 L closepath p .577 g
F P
0 g
s
.5322 .62579 m .54187 .76069 L .5373 .62915 L closepath p .484 g
F P
0 g
s
.53652 .70321 m .54187 .76069 L .53799 .73018 L closepath p .628 g
F P
0 g
s
.52839 .63216 m .54187 .76069 L .5322 .62579 L closepath p .42 g
F P
0 g
s
.52687 .64738 m .54187 .76069 L .52839 .63216 L closepath p .331 g
F P
0 g
s
.54187 .76069 m .42597 .7033 L .42793 .72223 L closepath p .479 g
F P
0 g
s
.51079 .83404 m .54187 .76069 L .50117 .80811 L closepath p .74 g
F P
0 g
s
.65668 .81667 m .54187 .76069 L .64429 .84043 L closepath p .708 g
F P
0 g
s
.75665 .89313 m .65668 .81667 L .64429 .84043 L p .631 g
F P
0 g
s
.56232 .62336 m .58462 .56047 L .57464 .47158 L p .352 g
F P
0 g
s
.59477 .39615 m .57464 .47158 L .58462 .56047 L p .364 g
F P
0 g
s
.57464 .47158 m .59477 .39615 L .54804 .30909 L p .44 g
F P
0 g
s
.22701 .34835 m .19549 .38335 L .22947 .50407 L p .794 g
F P
0 g
s
.23267 .55613 m .22947 .50407 L .19549 .38335 L p .722 g
F P
0 g
s
.22947 .50407 m .23267 .55613 L .31634 .652 L p .681 g
F P
0 g
s
.5373 .62915 m .54187 .76069 L .54235 .64219 L closepath p .536 g
F P
0 g
s
.34376 .69129 m .31634 .652 L .23267 .55613 L p .411 g
F P
0 g
s
.31634 .652 m .34376 .69129 L .4421 .74051 L p .497 g
F P
0 g
s
.4421 .74051 m .42793 .72223 L .31634 .652 L p .497 g
F P
0 g
s
.52805 .66992 m .54187 .76069 L .52687 .64738 L closepath p .186 g
F P
0 g
s
.54187 .76069 m .52213 .70238 L .50084 .68433 L closepath p .479 g
F P
0 g
s
.55566 .67495 m .5521 .6981 L .54539 .73916 L p F P
s
.54488 .74192 m .55566 .67495 L p .54539 .73916 L F P
s
.5521 .6981 m .54706 .72724 L .54577 .73557 L p .801 g
F P
0 g
s
.54539 .73916 m .54187 .76069 L .54488 .74192 L p F P
s
.54577 .73557 m .54187 .76069 L .54539 .73916 L p .801 g
F P
0 g
s
.54539 .73916 m .5521 .6981 L p .54577 .73557 L .801 g
F P
0 g
s
.56165 .69666 m .54706 .72724 L .5521 .6981 L p .386 g
F P
0 g
s
.56232 .62336 m .54876 .70713 L .55068 .71008 L .291 g
F
.54574 .72578 m .54733 .72075 L p .54876 .70713 L F P
0 g
s
.54733 .72075 m .55068 .71008 L p .54876 .70713 L .291 g
F P
0 g
s
.55068 .71008 m .55667 .69103 L .5588 .66546 L p .291 g
F P
0 g
s
.5588 .66546 m .56232 .62336 L p .55068 .71008 L .291 g
F P
0 g
s
.54187 .76069 m .64772 .76756 L .65597 .78789 L closepath p .779 g
F P
0 g
s
.65668 .81667 m .75665 .89313 L .74223 .84108 L p .615 g
F P
0 g
s
.75665 .89313 m .83292 .98046 L .81446 .9036 L p .483 g
F P
0 g
s
.81446 .9036 m .74223 .84108 L .75665 .89313 L p .483 g
F P
0 g
s
.54235 .64219 m .54187 .76069 L .54598 .66395 L closepath p .584 g
F P
0 g
s
.25329 .24208 m .30627 .25348 L .3708 .19645 L p .849 g
F P
0 g
s
.30627 .25348 m .25329 .24208 L .22701 .34835 L p .8 g
F P
0 g
s
.19549 .38335 m .22701 .34835 L .25329 .24208 L p .921 g
F P
0 g
s
.54187 .76069 m .42793 .72223 L .4421 .74051 L closepath p .479 g
F P
0 g
s
.83292 .98046 m .87891 1.06724 L .85784 .96973 L p .212 g
F P
0 g
s
.85784 .96973 m .81446 .9036 L .83292 .98046 L p .212 g
F P
0 g
s
.55667 .69103 m .57278 .66045 L .58462 .56047 L p .325 g
F P
0 g
s
.58462 .56047 m .56232 .62336 L .55667 .69103 L p .325 g
F P
0 g
s
.61061 .511 m .58462 .56047 L .57278 .66045 L p .33 g
F P
0 g
s
.58462 .56047 m .61061 .511 L .59477 .39615 L p .364 g
F P
0 g
s
.46056 .19198 m .48316 .26271 L .54804 .30909 L p .55 g
F P
0 g
s
.48316 .26271 m .46056 .19198 L .3708 .19645 L p .678 g
F P
0 g
s
.56232 .62336 m .54724 .69263 L .54574 .72578 L p .291 g
F P
0 g
s
.48696 .77508 m .51244 .77285 L .50131 .79148 L p .514 g
F P
0 g
s
.84929 1.22927 m .83083 1.27999 L .86891 1.18899 L p F P
s
.86891 1.18899 m .8909 1.14092 L .84929 1.22927 L p F P
s
.54187 .76069 m .58309 .67232 L .60775 .65674 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .60775 .65674 L .63142 .65125 L closepath p .479 g
F P
0 g
s
.42057 1.02087 m .42881 1.04375 L .4557 1.17589 L p .515 g
F P
0 g
s
.46903 1.17417 m .4557 1.17589 L .42881 1.04375 L p .1 g
F P
0 g
s
.4557 1.17589 m .46903 1.17417 L .53339 1.27309 L p .057 g
F P
0 g
s
.42881 1.04375 m .42057 1.02087 L .45365 .88527 L p .079 g
F P
0 g
s
.51244 .77285 m .48696 .77508 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54598 .66395 m .54187 .76069 L .54724 .69263 L closepath p .632 g
F P
0 g
s
.54187 .76069 m .63142 .65125 L .64984 .65588 L closepath p .479 g
F P
0 g
s
.73946 .69782 m .68911 .73122 L .61667 .7335 L p .736 g
F P
0 g
s
.61667 .7335 m .6428 .71175 L .73946 .69782 L p .736 g
F P
0 g
s
.54187 .76069 m .56165 .69666 L .58309 .67232 L closepath p .479 g
F P
0 g
s
.7458 1.32861 m .7152 1.33241 L .78672 1.28781 L p .845 g
F P
0 g
s
.78672 1.28781 m .83083 1.27999 L .7458 1.32861 L p .845 g
F P
0 g
s
.5504 .24482 m .54804 .30909 L .59477 .39615 L p .403 g
F P
0 g
s
.54804 .30909 m .5504 .24482 L .46056 .19198 L p .55 g
F P
0 g
s
.50117 .80811 m .54187 .76069 L .51239 .7815 L closepath p .749 g
F P
0 g
s
.61473 .34702 m .59477 .39615 L .61061 .511 L p .322 g
F P
0 g
s
.59477 .39615 m .61473 .34702 L .5504 .24482 L p .403 g
F P
0 g
s
.64826 .79117 m .54187 .76069 L .65668 .81667 L closepath p .706 g
F P
0 g
s
.74223 .84108 m .64826 .79117 L .65668 .81667 L p .615 g
F P
0 g
s
.54187 .76069 m .64984 .65588 L .65933 .66934 L closepath p .479 g
F P
0 g
s
.46771 .75468 m .4421 .74051 L .34376 .69129 L p .139 g
F P
0 g
s
.54187 .76069 m .4421 .74051 L .46771 .75468 L closepath p .479 g
F P
0 g
s
.57278 .66045 m .59142 .6376 L .61061 .511 L p .33 g
F P
0 g
s
.63605 .48087 m .61061 .511 L .59142 .6376 L p .308 g
F P
0 g
s
.61061 .511 m .63605 .48087 L .61473 .34702 L p .322 g
F P
0 g
s
.23267 .55613 m .26997 .61813 L .34376 .69129 L p .411 g
F P
0 g
s
.34376 .69129 m .39434 .72642 L .46771 .75468 L p .139 g
F P
0 g
s
.54187 .76069 m .46771 .75468 L .50228 .76196 L closepath p .479 g
F P
0 g
s
.50228 .76196 m .46771 .75468 L .39434 .72642 L p .6 g
F P
0 g
s
.39434 .72642 m .46311 .75106 L .50228 .76196 L p .6 g
F P
0 g
s
.43439 .72989 m .46311 .75106 L p .3696 .6969 L .614 g
F P
0 g
s
.34376 .69129 m .26997 .61813 L p .36735 .70292 L .358 g
F P
0 g
s
.39434 .72642 m .36594 .70669 L p .36735 .70292 L .358 g
F P
0 g
s
.36594 .70669 m .34376 .69129 L p .36735 .70292 L .358 g
F P
0 g
s
.26997 .61813 m .33969 .67957 L .39434 .72642 L p .358 g
F P
0 g
s
.33969 .67957 m .37005 .6957 L p .3696 .6969 L .614 g
F P
0 g
s
.37005 .6957 m .43439 .72989 L p .3696 .6969 L .614 g
F P
0 g
s
.54187 .76069 m .54706 .72724 L .56165 .69666 L closepath p .479 g
F P
0 g
s
.54724 .69263 m .54187 .76069 L .54574 .72578 L closepath p .683 g
F P
0 g
s
.57278 .66045 m .55667 .69103 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.59142 .6376 m .57278 .66045 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.19549 .38335 m .19825 .44816 L .23267 .55613 L p .722 g
F P
0 g
s
.26997 .61813 m .23267 .55613 L .19825 .44816 L p .152 g
F P
0 g
s
.54187 .76069 m .62054 .75359 L .64772 .76756 L closepath p .777 g
F P
0 g
s
.55667 .69103 m .54574 .72578 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .65933 .66934 L .65731 .6891 L closepath p .479 g
F P
0 g
s
.6094 .62511 m .59142 .6376 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.59142 .6376 m .6094 .62511 L .63605 .48087 L p .308 g
F P
0 g
s
.87891 1.06724 m .8909 1.14092 L .86849 1.02899 L p .521 g
F P
0 g
s
.86849 1.02899 m .85784 .96973 L .87891 1.06724 L p .521 g
F P
0 g
s
.46711 .83694 m .48657 .8081 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.45587 .86399 m .46711 .83694 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.48657 .8081 m .51239 .7815 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.68911 .73122 m .62054 .75359 L .58163 .75073 L p .766 g
F P
0 g
s
.58163 .75073 m .61667 .7335 L .68911 .73122 L p .766 g
F P
0 g
s
.45365 .88527 m .45587 .86399 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.33157 .17506 m .3708 .19645 L .46056 .19198 L p .723 g
F P
0 g
s
.3708 .19645 m .33157 .17506 L .25329 .24208 L p .849 g
F P
0 g
s
.45365 .88527 m .46007 .89735 L .42881 1.04375 L p .079 g
F P
0 g
s
.44684 1.04302 m .42881 1.04375 L .46007 .89735 L p .235 g
F P
0 g
s
.42881 1.04375 m .44684 1.04302 L .46903 1.17417 L p .1 g
F P
0 g
s
.46311 .75106 m .39434 .72642 L .33969 .67957 L p .614 g
F P
0 g
s
.62347 .62436 m .6094 .62511 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.6094 .62511 m .62347 .62436 L .65635 .47407 L p .25 g
F P
0 g
s
.65635 .47407 m .63605 .48087 L .6094 .62511 L p .25 g
F P
0 g
s
.54187 .76069 m .65731 .6891 L .6428 .71175 L closepath p .479 g
F P
0 g
s
.62083 .77034 m .54187 .76069 L .64826 .79117 L closepath p .706 g
F P
0 g
s
.64826 .79117 m .74223 .84108 L .69119 .79537 L p .591 g
F P
0 g
s
.69119 .79537 m .62083 .77034 L .64826 .79117 L p .591 g
F P
0 g
s
.74223 .84108 m .81446 .9036 L .74535 .83197 L p .305 g
F P
0 g
s
.74535 .83197 m .69119 .79537 L .74223 .84108 L p .305 g
F P
0 g
s
.63605 .48087 m .65635 .47407 L .63096 .33136 L p .173 g
F P
0 g
s
.63096 .33136 m .61473 .34702 L .63605 .48087 L p .173 g
F P
0 g
s
.25329 .24208 m .22319 .27496 L .19549 .38335 L p .921 g
F P
0 g
s
.19825 .44816 m .19549 .38335 L .22319 .27496 L p .04 g
F P
0 g
s
.46007 .89735 m .45365 .88527 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.53551 1.22601 m .53339 1.27309 L .46903 1.17417 L p .423 g
F P
0 g
s
.53339 1.27309 m .53551 1.22601 L .60277 1.2773 L p .56 g
F P
0 g
s
.60277 1.2773 m .62293 1.3277 L .53339 1.27309 L p .56 g
F P
0 g
s
.54187 .76069 m .50228 .76196 L .46311 .75106 L closepath p .708 g
F P
0 g
s
.54187 .76069 m .58163 .75073 L .62054 .75359 L closepath p .772 g
F P
0 g
s
.54187 .76069 m .6428 .71175 L .61667 .7335 L closepath p .479 g
F P
0 g
s
.62293 1.3277 m .60277 1.2773 L .67192 1.2841 L p .692 g
F P
0 g
s
.67192 1.2841 m .7152 1.33241 L .62293 1.3277 L p .692 g
F P
0 g
s
.5022 .77062 m .54187 .76069 L .46282 .76781 L closepath p .772 g
F P
0 g
s
.46717 .78784 m .5022 .77062 L .46282 .76781 L p .723 g
F P
0 g
s
.54187 .76069 m .5022 .77062 L .46717 .78784 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .46311 .75106 L .43439 .72989 L closepath p .706 g
F P
0 g
s
.19825 .44816 m .23989 .53266 L .26997 .61813 L p .152 g
F P
0 g
s
.33969 .67957 m .26997 .61813 L .23989 .53266 L p .489 g
F P
0 g
s
.63073 .63518 m .62347 .62436 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.62347 .62436 m .63073 .63518 L .66722 .4918 L p .13 g
F P
0 g
s
.66722 .4918 m .65635 .47407 L .62347 .62436 L p .13 g
F P
0 g
s
.22319 .27496 m .25329 .24208 L .33157 .17506 L p F P
s
.46282 .76781 m .39225 .79063 L .46717 .78784 L p .723 g
F P
0 g
s
.44058 .80979 m .46717 .78784 L .39225 .79063 L p .703 g
F P
0 g
s
.54187 .76069 m .46717 .78784 L .44058 .80979 L closepath p .479 g
F P
0 g
s
.48745 1.13564 m .46903 1.17417 L .44684 1.04302 L p .349 g
F P
0 g
s
.46903 1.17417 m .48745 1.13564 L .53551 1.22601 L p .423 g
F P
0 g
s
.62919 .6558 m .63073 .63518 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .44058 .80979 L .42515 .83306 L closepath p .479 g
F P
0 g
s
.58171 .75941 m .54187 .76069 L .62083 .77034 L closepath p .708 g
F P
0 g
s
.54187 .76069 m .55599 .89537 L .552 .90206 L closepath p .42 g
F P
0 g
s
.54187 .76069 m .55754 .87905 L .55599 .89537 L closepath p .331 g
F P
0 g
s
.54187 .76069 m .54739 .82004 L .5458 .79169 L closepath p .628 g
F P
0 g
s
.54187 .76069 m .552 .90206 L .54664 .8982 L closepath p .484 g
F P
0 g
s
.54187 .76069 m .54615 .84428 L .54739 .82004 L closepath p .577 g
F P
0 g
s
.54187 .76069 m .54664 .8982 L .54137 .88397 L closepath p .536 g
F P
0 g
s
.54187 .76069 m .54237 .86319 L .54615 .84428 L closepath p .526 g
F P
0 g
s
.54187 .76069 m .54137 .88397 L .53761 .86063 L closepath p .584 g
F P
0 g
s
.52544 .9669 m .53761 .86063 L .54137 .88397 L p .544 g
F P
0 g
s
.54187 .76069 m .53705 .87572 L .54237 .86319 L closepath p .47 g
F P
0 g
s
.54187 .76069 m .53637 .83037 L .53794 .79603 L closepath p .683 g
F P
0 g
s
.54187 .76069 m .53761 .86063 L .53637 .83037 L closepath p .632 g
F P
0 g
s
.53265 .83091 m .52948 .83138 L .53592 .80446 L .459 g
F
.53291 .81212 m .53794 .79603 L p .52948 .83138 L F P
0 g
s
.53794 .79603 m .52676 .83177 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.51025 .86319 m .49135 .88617 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.52676 .83177 m .51025 .86319 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .52159 .82633 L .5366 .7946 L closepath p .479 g
F P
0 g
s
.52676 .83177 m .53291 .81212 L p .52948 .83138 L .459 g
F P
0 g
s
.49135 .88617 m .51025 .86319 L .49752 .9684 L p .382 g
F P
0 g
s
.51025 .86319 m .52676 .83177 L .52066 .90306 L p .419 g
F P
0 g
s
.53637 .83037 m .52066 .90306 L .52448 .85845 L p .459 g
F P
0 g
s
.52448 .85845 m .52676 .83177 L p .52948 .83138 L .459 g
F P
0 g
s
.53637 .83037 m .52448 .85845 L .52948 .83138 L .459 g
F
.53134 .82505 m .5366 .7946 L p .53598 .79614 L .073 g
F P
0 g
s
.52066 .90306 m .49752 .9684 L .51025 .86319 L p .419 g
F P
0 g
s
.50102 .88305 m .53134 .82505 L p .53598 .79614 L .073 g
F P
0 g
s
.52159 .82633 m .50102 .88305 L p .53598 .79614 L .073 g
F P
0 g
s
.5366 .7946 m .53602 .79582 L p .53598 .79614 L .073 g
F P
0 g
s
.53602 .79582 m .52159 .82633 L p .53598 .79614 L .073 g
F P
0 g
s
.5366 .7946 m .53134 .82505 L .5397 .7739 L p .801 g
F P
0 g
s
.53386 .81047 m .53526 .80175 L p .53134 .82505 L F P
s
.53134 .82505 m .52749 .85005 L .53386 .81047 L p F P
s
.53526 .80175 m .53728 .78919 L p .53547 .80011 L F P
s
.53547 .80011 m .53134 .82505 L .53526 .80175 L F
.53728 .78919 m .54008 .77179 L p .53677 .79194 L F P
s
.53677 .79194 m .53134 .82505 L .53728 .78919 L F
.5397 .7739 m .54187 .76069 L .53927 .7774 L p .801 g
F P
0 g
s
.53927 .7774 m .5366 .7946 L p .5397 .7739 L .801 g
F P
0 g
s
.53794 .79603 m .53637 .83037 L p .53265 .83091 L .459 g
F P
0 g
s
.53265 .83091 m .53592 .80446 L .53794 .79603 L .459 g
F
.54008 .77179 m .54187 .76069 L .5397 .7739 L p 0 g
F P
s
.5397 .7739 m .53134 .82505 L p .54008 .77179 L F P
s
.52066 .90306 m .53637 .83037 L .53761 .86063 L p .524 g
F P
0 g
s
.53761 .86063 m .52544 .9669 L .52066 .90306 L p .524 g
F P
0 g
s
.54187 .76069 m .53164 .88104 L .53705 .87572 L closepath p .403 g
F P
0 g
s
.54187 .76069 m .52764 .87859 L .53164 .88104 L closepath p .312 g
F P
0 g
s
.54187 .76069 m .52612 .86817 L .52764 .87859 L closepath p .17 g
F P
0 g
s
.54187 .76069 m .52749 .85005 L .52612 .86817 L closepath p .031 g
F P
0 g
s
.54187 .76069 m .55621 .85491 L .55754 .87905 L closepath p .186 g
F P
0 g
s
.56219 .82073 m .58269 .8843 L p .54776 .79472 L .376 g
F P
0 g
s
.54847 .80403 m .54365 .77242 L p .54807 .79878 L .047 g
F P
0 g
s
.54365 .77242 m .54187 .76069 L .54403 .77397 L p .047 g
F P
0 g
s
.54403 .77397 m .54807 .79878 L p .54365 .77242 L .047 g
F P
0 g
s
.55239 .82538 m .55621 .85491 L .54847 .80403 L p .047 g
F P
0 g
s
.54807 .79878 m .55239 .82538 L p .54847 .80403 L .047 g
F P
0 g
s
.54713 .79313 m .55239 .82538 L .54403 .77397 L p .498 g
F P
0 g
s
.54403 .77397 m .54187 .76069 L .54446 .77667 L p .498 g
F P
0 g
s
.54446 .77667 m .54713 .79313 L p .54403 .77397 L .498 g
F P
0 g
s
.5458 .79169 m .54739 .82004 L p .55113 .81818 L .316 g
F P
0 g
s
.58269 .8843 m .55239 .82538 L .54713 .79313 L p .376 g
F P
0 g
s
.55113 .81818 m .55433 .8166 L .54783 .79761 L .316 g
F
.55113 .81818 m .54783 .79761 L .5458 .79169 L F
.55708 .81524 m .55086 .80226 L p .55433 .8166 L F P
0 g
s
.55086 .80226 m .5458 .79169 L p .55433 .8166 L .316 g
F P
0 g
s
.54713 .79313 m .54771 .79419 L p .54776 .79472 L .376 g
F P
0 g
s
.54771 .79419 m .56219 .82073 L p .54776 .79472 L .376 g
F P
0 g
s
.54187 .76069 m .61667 .7335 L .58163 .75073 L closepath p .479 g
F P
0 g
s
.61721 .76679 m .58171 .75941 L .62083 .77034 L p .475 g
F P
0 g
s
.54187 .76069 m .58171 .75941 L .61721 .76679 L closepath p .479 g
F P
0 g
s
.62083 .77034 m .69119 .79537 L .61721 .76679 L p .475 g
F P
0 g
s
.54187 .76069 m .61721 .76679 L .64435 .78141 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .64435 .78141 L .66018 .80062 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .66018 .80062 L .66331 .82081 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .66331 .82081 L .65425 .83801 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .65425 .83801 L .63538 .84855 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .63538 .84855 L .6106 .84971 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .6106 .84971 L .58464 .84028 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .58464 .84028 L .56219 .82073 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .56219 .82073 L .54713 .79313 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .42515 .83306 L .4221 .85382 L closepath p .479 g
F P
0 g
s
.61808 .68295 m .62919 .6558 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.59818 .7124 m .61808 .68295 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.57169 .73963 m .59818 .7124 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .58375 .71188 L .57169 .73963 L closepath p .749 g
F P
0 g
s
.59818 .7124 m .57169 .73963 L .58375 .71188 L p .874 g
F P
0 g
s
.58391 .72876 m .54187 .76069 L .57174 .74833 L closepath p .728 g
F P
0 g
s
.54187 .76069 m .4221 .85382 L .43092 .86837 L closepath p .479 g
F P
0 g
s
.47351 .89791 m .46007 .89735 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .43439 .72989 L .42378 .7031 L closepath p .706 g
F P
0 g
s
.46282 .76781 m .54187 .76069 L .43382 .75367 L closepath p .777 g
F P
0 g
s
.54187 .76069 m .43092 .86837 L .44935 .87375 L closepath p .479 g
F P
0 g
s
.49135 .88617 m .47351 .89791 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.54187 .76069 m .49931 .85191 L .52159 .82633 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .44935 .87375 L .47367 .86828 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .47367 .86828 L .49931 .85191 L closepath p .479 g
F P
0 g
s
.54187 .76069 m .57441 .68385 L .58375 .71188 L closepath p .74 g
F P
0 g
s
.57459 .70783 m .54187 .76069 L .58391 .72876 L closepath p .738 g
F P
0 g
s
.54187 .76069 m .42378 .7031 L .43459 .67717 L closepath p .708 g
F P
0 g
s
.43382 .75367 m .54187 .76069 L .42301 .73235 L closepath p .779 g
F P
0 g
s
.54187 .76069 m .54569 .66253 L .57441 .68385 L closepath p .731 g
F P
0 g
s
.54572 .69199 m .54187 .76069 L .57459 .70783 L closepath p .748 g
F P
0 g
s
.54187 .76069 m .43459 .67717 L .46442 .65891 L closepath p .711 g
F P
0 g
s
.43439 .72989 m .33969 .67957 L .3182 .62276 L p .623 g
F P
0 g
s
.3182 .62276 m .42378 .7031 L .43439 .72989 L p .623 g
F P
0 g
s
.42301 .73235 m .54187 .76069 L .43381 .70994 L closepath p .778 g
F P
0 g
s
.54187 .76069 m .46442 .65891 L .50526 .65345 L closepath p .717 g
F P
0 g
s
.54187 .76069 m .50526 .65345 L .54569 .66253 L closepath p .724 g
F P
0 g
s
.50499 .68635 m .54187 .76069 L .54572 .69199 L closepath p .759 g
F P
0 g
s
.43381 .70994 m .54187 .76069 L .46383 .69293 L closepath p .774 g
F P
0 g
s
.5986 .74582 m .57174 .74833 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.46383 .69293 m .54187 .76069 L .50499 .68635 L closepath p .768 g
F P
0 g
s
.61927 .75298 m .5986 .74582 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.5458 .79169 m .55708 .81524 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.64435 .78141 m .61721 .76679 L .69119 .79537 L p .196 g
F P
0 g
s
.63138 .768 m .61927 .75298 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.55708 .81524 m .57395 .82878 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.63378 .78759 m .63138 .768 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.57395 .82878 m .59358 .83124 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.62684 .80752 m .63378 .78759 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.59358 .83124 m .61242 .82334 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.61242 .82334 m .62684 .80752 L .54187 .76069 L closepath p .323 g
F P
0 g
s
.61473 .34702 m .63096 .33136 L .55235 .22147 L p .191 g
F P
0 g
s
.5504 .24482 m .61473 .34702 L p .56318 .24327 L .191 g
F P
0 g
s
.63993 .35274 m .63096 .33136 L .65635 .47407 L p .038 g
F P
0 g
s
.63993 .35274 m .55345 .24486 L p .60776 .30547 L .52 g
F P
0 g
s
.63096 .33136 m .63751 .34698 L p .60776 .30547 L .52 g
F P
0 g
s
.63751 .34698 m .63993 .35274 L p .60776 .30547 L .52 g
F P
0 g
s
.55235 .22147 m .55185 .22746 L p .56318 .24327 L .191 g
F P
0 g
s
.55185 .22746 m .5504 .24482 L p .56318 .24327 L .191 g
F P
0 g
s
.83083 1.27999 m .78672 1.28781 L .81764 1.20101 L p .872 g
F P
0 g
s
.81764 1.20101 m .86891 1.18899 L .83083 1.27999 L p .872 g
F P
0 g
s
.46007 .89735 m .47351 .89791 L .44684 1.04302 L p .235 g
F P
0 g
s
.81446 .9036 m .85784 .96973 L .77753 .87426 L p .384 g
F P
0 g
s
.77753 .87426 m .74535 .83197 L .81446 .9036 L p .384 g
F P
0 g
s
.46056 .19198 m .5504 .24482 L p .45235 .17375 L .367 g
F P
0 g
s
.5504 .24482 m .55235 .22147 L .44181 .16611 L p .367 g
F P
0 g
s
.55345 .24486 m .43108 .19177 L p .52871 .21568 L .415 g
F P
0 g
s
.55235 .22147 m .5527 .22885 L p .52871 .21568 L .415 g
F P
0 g
s
.5527 .22885 m .55345 .24486 L p .52871 .21568 L .415 g
F P
0 g
s
.44181 .16611 m .44404 .16918 L p .45235 .17375 L .367 g
F P
0 g
s
.44404 .16918 m .46056 .19198 L p .45235 .17375 L .367 g
F P
0 g
s
.46056 .19198 m .44181 .16611 L .33157 .17506 L p .723 g
F P
0 g
s
.39225 .79063 m .33784 .8256 L .44058 .80979 L p .703 g
F P
0 g
s
.42515 .83306 m .44058 .80979 L .33784 .8256 L p .663 g
F P
0 g
s
.39225 .79063 m .46282 .76781 L .43382 .75367 L p .786 g
F P
0 g
s
.63073 .63518 m .62919 .6558 L .66548 .53173 L p .036 g
F P
0 g
s
.66548 .53173 m .66722 .4918 L .63073 .63518 L p .036 g
F P
0 g
s
.8909 1.14092 m .86891 1.18899 L .84665 1.07072 L p .841 g
F P
0 g
s
.84665 1.07072 m .86849 1.02899 L .8909 1.14092 L p .841 g
F P
0 g
s
.47126 1.01714 m .44684 1.04302 L .47351 .89791 L p .327 g
F P
0 g
s
.44684 1.04302 m .47126 1.01714 L .48745 1.13564 L p .349 g
F P
0 g
s
.23989 .53266 m .3182 .62276 L .33969 .67957 L p .489 g
F P
0 g
s
.33784 .8256 m .30547 .86702 L .42515 .83306 L p .663 g
F P
0 g
s
.4221 .85382 m .42515 .83306 L .30547 .86702 L p .601 g
F P
0 g
s
.47351 .89791 m .49135 .88617 L .47126 1.01714 L p .327 g
F P
0 g
s
.69119 .79537 m .74535 .83197 L .64435 .78141 L p .196 g
F P
0 g
s
.65635 .47407 m .66722 .4918 L .63993 .35274 L p .038 g
F P
0 g
s
.7152 1.33241 m .67192 1.2841 L .72486 1.24669 L p .743 g
F P
0 g
s
.72486 1.24669 m .78672 1.28781 L .7152 1.33241 L p .743 g
F P
0 g
s
.64984 .58785 m .66548 .53173 L .62919 .6558 L p F P
s
.62919 .6558 m .61808 .68295 L .64984 .58785 L p F P
s
.57174 .74833 m .5986 .74582 L .6226 .71634 L p .441 g
F P
0 g
s
.6226 .71634 m .58391 .72876 L .57174 .74833 L p .441 g
F P
0 g
s
.43382 .75367 m .33558 .77095 L .39225 .79063 L p .786 g
F P
0 g
s
.33784 .8256 m .39225 .79063 L .33558 .77095 L p .795 g
F P
0 g
s
.61808 .68295 m .59818 .7124 L .62144 .65125 L p .923 g
F P
0 g
s
.58375 .71188 m .62144 .65125 L .59818 .7124 L p .874 g
F P
0 g
s
.62144 .65125 m .64984 .58785 L .61808 .68295 L p .923 g
F P
0 g
s
.22319 .27496 m .22529 .34871 L .19825 .44816 L p .04 g
F P
0 g
s
.23989 .53266 m .19825 .44816 L .22529 .34871 L p .392 g
F P
0 g
s
.49752 .9684 m .47126 1.01714 L .49135 .88617 L p .382 g
F P
0 g
s
.30547 .86702 m .29822 .90763 L .4221 .85382 L p .601 g
F P
0 g
s
.43092 .86837 m .4221 .85382 L .29822 .90763 L p .512 g
F P
0 g
s
.33558 .77095 m .43382 .75367 L .42301 .73235 L p .832 g
F P
0 g
s
.85784 .96973 m .86849 1.02899 L .78474 .91497 L p .673 g
F P
0 g
s
.78474 .91497 m .77753 .87426 L .85784 .96973 L p .673 g
F P
0 g
s
.4556 .92735 m .50102 .88305 L .52159 .82633 L p .12 g
F P
0 g
s
.52159 .82633 m .49931 .85191 L .4556 .92735 L p .12 g
F P
0 g
s
.74535 .83197 m .77753 .87426 L .66018 .80062 L p .491 g
F P
0 g
s
.66018 .80062 m .64435 .78141 L .74535 .83197 L p .491 g
F P
0 g
s
.47126 1.01714 m .49752 .9684 L .50755 1.06339 L p .448 g
F P
0 g
s
.50755 1.06339 m .48745 1.13564 L .47126 1.01714 L p .448 g
F P
0 g
s
.29822 .90763 m .31556 .93966 L .43092 .86837 L p .512 g
F P
0 g
s
.44935 .87375 m .43092 .86837 L .31556 .93966 L p .402 g
F P
0 g
s
.59765 .94568 m .55621 .85491 L .55239 .82538 L p .413 g
F P
0 g
s
.55239 .82538 m .58269 .8843 L .59765 .94568 L p .413 g
F P
0 g
s
.48564 .9274 m .52749 .85005 L .53134 .82505 L p .071 g
F P
0 g
s
.53134 .82505 m .50102 .88305 L .48564 .9274 L p .071 g
F P
0 g
s
.33714 .56857 m .43459 .67717 L .42378 .7031 L p .636 g
F P
0 g
s
.42378 .7031 m .3182 .62276 L .33714 .56857 L p .636 g
F P
0 g
s
.48745 1.13564 m .50755 1.06339 L .53784 1.13644 L p .52 g
F P
0 g
s
.53784 1.13644 m .53551 1.22601 L .48745 1.13564 L p .52 g
F P
0 g
s
.62144 .65125 m .58375 .71188 L .57441 .68385 L p .776 g
F P
0 g
s
.62803 .92993 m .58269 .8843 L .56219 .82073 L p .469 g
F P
0 g
s
.56219 .82073 m .58464 .84028 L .62803 .92993 L p .469 g
F P
0 g
s
.53551 1.22601 m .53784 1.13644 L .58042 1.17892 L p .584 g
F P
0 g
s
.58042 1.17892 m .60277 1.2773 L .53551 1.22601 L p .584 g
F P
0 g
s
.30908 .20416 m .33157 .17506 L .44181 .16611 L p .144 g
F P
0 g
s
.33157 .17506 m .30908 .20416 L .22319 .27496 L p F P
s
.40303 .95251 m .4556 .92735 L .49931 .85191 L p .189 g
F P
0 g
s
.49931 .85191 m .47367 .86828 L .40303 .95251 L p .189 g
F P
0 g
s
.33558 .77095 m .2582 .80972 L .33784 .8256 L p .795 g
F P
0 g
s
.30547 .86702 m .33784 .8256 L .2582 .80972 L p .776 g
F P
0 g
s
.54739 .82004 m .56343 .86396 L .55708 .81524 L p .316 g
F P
0 g
s
.60282 .99815 m .55754 .87905 L .55621 .85491 L p .449 g
F P
0 g
s
.55621 .85491 m .59765 .94568 L .60282 .99815 L p .449 g
F P
0 g
s
.56343 .86396 m .54739 .82004 L .54615 .84428 L p .31 g
F P
0 g
s
.31556 .93966 m .35309 .95617 L .44935 .87375 L p .402 g
F P
0 g
s
.47367 .86828 m .44935 .87375 L .35309 .95617 L p .288 g
F P
0 g
s
.5986 .74582 m .61927 .75298 L .6532 .72462 L p .258 g
F P
0 g
s
.6532 .72462 m .6226 .71634 L .5986 .74582 L p .258 g
F P
0 g
s
.66722 .4918 m .66548 .53173 L .63887 .40932 L p .769 g
F P
0 g
s
.63887 .40932 m .63993 .35274 L .66722 .4918 L p .769 g
F P
0 g
s
.55345 .24486 m .55235 .22147 L .63096 .33136 L p .52 g
F P
0 g
s
.57395 .82878 m .55708 .81524 L .56343 .86396 L p .2 g
F P
0 g
s
.35309 .95617 m .40303 .95251 L .47367 .86828 L p .288 g
F P
0 g
s
.22529 .34871 m .22319 .27496 L .30908 .20416 L p .313 g
F P
0 g
s
.58391 .72876 m .6226 .71634 L .60541 .68254 L p .514 g
F P
0 g
s
.60541 .68254 m .57459 .70783 L .58391 .72876 L p .514 g
F P
0 g
s
.49752 .9684 m .52066 .90306 L .52544 .9669 L p .491 g
F P
0 g
s
.60277 1.2773 m .58042 1.17892 L .62404 1.18633 L p .626 g
F P
0 g
s
.62404 1.18633 m .67192 1.2841 L .60277 1.2773 L p .626 g
F P
0 g
s
.52544 .9669 m .50755 1.06339 L .49752 .9684 L p .491 g
F P
0 g
s
.22529 .34871 m .26411 .45238 L .23989 .53266 L p .392 g
F P
0 g
s
.3182 .62276 m .23989 .53266 L .26411 .45238 L p .547 g
F P
0 g
s
.66331 .82081 m .66018 .80062 L .77753 .87426 L p .575 g
F P
0 g
s
.86891 1.18899 m .81764 1.20101 L .79746 1.08591 L p .761 g
F P
0 g
s
.79746 1.08591 m .84665 1.07072 L .86891 1.18899 L p .761 g
F P
0 g
s
.78672 1.28781 m .72486 1.24669 L .74683 1.17114 L p .725 g
F P
0 g
s
.74683 1.17114 m .81764 1.20101 L .78672 1.28781 L p .725 g
F P
0 g
s
.54137 .88397 m .53994 1.01619 L .52544 .9669 L p .544 g
F P
0 g
s
.53994 1.01619 m .54137 .88397 L .54664 .8982 L p .542 g
F P
0 g
s
.26411 .45238 m .33714 .56857 L .3182 .62276 L p .547 g
F P
0 g
s
.31251 .7356 m .42301 .73235 L .43381 .70994 L p .855 g
F P
0 g
s
.42301 .73235 m .31251 .7356 L .33558 .77095 L p .832 g
F P
0 g
s
.47979 .95745 m .52612 .86817 L .52749 .85005 L p .093 g
F P
0 g
s
.52749 .85005 m .48564 .9274 L .47979 .95745 L p .093 g
F P
0 g
s
.68045 .95661 m .62803 .92993 L .58464 .84028 L p .522 g
F P
0 g
s
.58464 .84028 m .6106 .84971 L .68045 .95661 L p .522 g
F P
0 g
s
.59676 1.03524 m .55599 .89537 L .55754 .87905 L p .48 g
F P
0 g
s
.55754 .87905 m .60282 .99815 L .59676 1.03524 L p .48 g
F P
0 g
s
.77753 .87426 m .78474 .91497 L .66331 .82081 L p .575 g
F P
0 g
s
.54664 .8982 m .56034 1.04553 L .53994 1.01619 L p .542 g
F P
0 g
s
.56034 1.04553 m .54664 .8982 L .552 .90206 L p .528 g
F P
0 g
s
.86849 1.02899 m .84665 1.07072 L .7675 .94639 L p .697 g
F P
0 g
s
.7675 .94639 m .78474 .91497 L .86849 1.02899 L p .697 g
F P
0 g
s
.50755 1.06339 m .52544 .9669 L .53994 1.01619 L p .539 g
F P
0 g
s
.53994 1.01619 m .53784 1.13644 L .50755 1.06339 L p .539 g
F P
0 g
s
.55599 .89537 m .59676 1.03524 L .58116 1.05192 L p .507 g
F P
0 g
s
.54615 .84428 m .55874 .90543 L .56343 .86396 L p .31 g
F P
0 g
s
.58116 1.05192 m .552 .90206 L .55599 .89537 L p .507 g
F P
0 g
s
.55874 .90543 m .54615 .84428 L .54237 .86319 L p .296 g
F P
0 g
s
.39317 .53096 m .46442 .65891 L .43459 .67717 L p .652 g
F P
0 g
s
.43459 .67717 m .33714 .56857 L .39317 .53096 L p .652 g
F P
0 g
s
.57441 .68385 m .6041 .58975 L .62144 .65125 L p .776 g
F P
0 g
s
.6041 .58975 m .57441 .68385 L .54569 .66253 L p .729 g
F P
0 g
s
.65425 .83801 m .66331 .82081 L .78474 .91497 L p .592 g
F P
0 g
s
.552 .90206 m .58116 1.05192 L .56034 1.04553 L p .528 g
F P
0 g
s
.64984 .58785 m .62144 .65125 L .6041 .58975 L p .805 g
F P
0 g
s
.66548 .53173 m .64984 .58785 L .6266 .49288 L p .83 g
F P
0 g
s
.6266 .49288 m .63887 .40932 L .66548 .53173 L p .83 g
F P
0 g
s
.43108 .19177 m .44181 .16611 L .55235 .22147 L p .415 g
F P
0 g
s
.44181 .16611 m .43108 .19177 L .30908 .20416 L p .144 g
F P
0 g
s
.2582 .80972 m .33558 .77095 L .31251 .7356 L p .88 g
F P
0 g
s
.61927 .75298 m .63138 .768 L .67173 .75127 L p .086 g
F P
0 g
s
.67173 .75127 m .6532 .72462 L .61927 .75298 L p .086 g
F P
0 g
s
.73018 .96174 m .68045 .95661 L .6106 .84971 L p .559 g
F P
0 g
s
.6106 .84971 m .63538 .84855 L .73018 .96174 L p .559 g
F P
0 g
s
.67192 1.2841 m .62404 1.18633 L .65702 1.15868 L p .641 g
F P
0 g
s
.65702 1.15868 m .72486 1.24669 L .67192 1.2841 L p .641 g
F P
0 g
s
.2582 .80972 m .21091 .86333 L .30547 .86702 L p .776 g
F P
0 g
s
.29822 .90763 m .30547 .86702 L .21091 .86333 L p .71 g
F P
0 g
s
.59358 .83124 m .57395 .82878 L .58782 .88745 L p .106 g
F P
0 g
s
.56343 .86396 m .58782 .88745 L .57395 .82878 L p .2 g
F P
0 g
s
.65958 1.02053 m .59765 .94568 L .58269 .8843 L p .51 g
F P
0 g
s
.58269 .8843 m .62803 .92993 L .65958 1.02053 L p .51 g
F P
0 g
s
.78474 .91497 m .7675 .94639 L .65425 .83801 L p .592 g
F P
0 g
s
.63538 .84855 m .65425 .83801 L .7675 .94639 L p .583 g
F P
0 g
s
.4854 .97265 m .52764 .87859 L .52612 .86817 L p .122 g
F P
0 g
s
.52612 .86817 m .47979 .95745 L .4854 .97265 L p .122 g
F P
0 g
s
.42213 .98291 m .48564 .9274 L .50102 .88305 L p .096 g
F P
0 g
s
.50102 .88305 m .4556 .92735 L .42213 .98291 L p .096 g
F P
0 g
s
.6041 .58975 m .6266 .49288 L .64984 .58785 L p .805 g
F P
0 g
s
.53784 1.13644 m .53994 1.01619 L .56034 1.04553 L p .565 g
F P
0 g
s
.56034 1.04553 m .58042 1.17892 L .53784 1.13644 L p .565 g
F P
0 g
s
.7675 .94639 m .73018 .96174 L .63538 .84855 L p .583 g
F P
0 g
s
.54569 .66253 m .54922 .54229 L .6041 .58975 L p .729 g
F P
0 g
s
.57459 .70783 m .60541 .68254 L .54941 .65662 L p .588 g
F P
0 g
s
.54941 .65662 m .54572 .69199 L .57459 .70783 L p .588 g
F P
0 g
s
.54237 .86319 m .54386 .93843 L .55874 .90543 L p .296 g
F P
0 g
s
.54386 .93843 m .54237 .86319 L .53705 .87572 L p .271 g
F P
0 g
s
.54922 .54229 m .54569 .66253 L .50526 .65345 L p .697 g
F P
0 g
s
.47141 .52095 m .50526 .65345 L .46442 .65891 L p .672 g
F P
0 g
s
.46442 .65891 m .39317 .53096 L .47141 .52095 L p .672 g
F P
0 g
s
.33121 .69543 m .43381 .70994 L .46383 .69293 L p .838 g
F P
0 g
s
.43381 .70994 m .33121 .69543 L .31251 .7356 L p .855 g
F P
0 g
s
.50115 .97361 m .53164 .88104 L .52764 .87859 L p .181 g
F P
0 g
s
.52764 .87859 m .4854 .97265 L .50115 .97361 L p .181 g
F P
0 g
s
.58042 1.17892 m .56034 1.04553 L .58116 1.05192 L p .574 g
F P
0 g
s
.58116 1.05192 m .62404 1.18633 L .58042 1.17892 L p .574 g
F P
0 g
s
.53705 .87572 m .52268 .96167 L .54386 .93843 L p .271 g
F P
0 g
s
.52268 .96167 m .53705 .87572 L .53164 .88104 L p .233 g
F P
0 g
s
.63993 .35274 m .63887 .40932 L .55335 .31397 L p .697 g
F P
0 g
s
.55335 .31397 m .55345 .24486 L .63993 .35274 L p .697 g
F P
0 g
s
.63138 .768 m .63378 .78759 L .67516 .79017 L p .045 g
F P
0 g
s
.67512 .78982 m .63138 .768 L .67516 .79017 L .045 g
F
.67592 .79022 m .67173 .75127 L .63138 .768 L p F P
0 g
s
.65001 .78057 m .6461 .7238 L .67123 .75074 L p .67 g
F P
0 g
s
.67123 .75074 m .67173 .75127 L p .67134 .7518 L .67 g
F P
0 g
s
.65001 .78057 m .67123 .75074 L .67134 .7518 L .67 g
F
.67516 .79017 m .67592 .79022 L p .67512 .78982 L .045 g
F P
0 g
s
.50526 .65345 m .47141 .52095 L .54922 .54229 L p .697 g
F P
0 g
s
.53164 .88104 m .50115 .97361 L .52268 .96167 L p .233 g
F P
0 g
s
.61242 .82334 m .59358 .83124 L .61662 .88831 L p .065 g
F P
0 g
s
.58782 .88745 m .61662 .88831 L .59358 .83124 L p .106 g
F P
0 g
s
.30908 .20416 m .31051 .28202 L .22529 .34871 L p .313 g
F P
0 g
s
.26411 .45238 m .22529 .34871 L .31051 .28202 L p .502 g
F P
0 g
s
.81764 1.20101 m .74683 1.17114 L .73075 1.06918 L p .68 g
F P
0 g
s
.73075 1.06918 m .79746 1.08591 L .81764 1.20101 L p .68 g
F P
0 g
s
.84665 1.07072 m .79746 1.08591 L .73018 .96174 L p .667 g
F P
0 g
s
.73018 .96174 m .7675 .94639 L .84665 1.07072 L p .667 g
F P
0 g
s
.67019 1.10054 m .60282 .99815 L .59765 .94568 L p .539 g
F P
0 g
s
.59765 .94568 m .65958 1.02053 L .67019 1.10054 L p .539 g
F P
0 g
s
.21091 .86333 m .19921 .92174 L .29822 .90763 L p .71 g
F P
0 g
s
.31556 .93966 m .29822 .90763 L .19921 .92174 L p .578 g
F P
0 g
s
.72486 1.24669 m .65702 1.15868 L .67019 1.10054 L p .633 g
F P
0 g
s
.67019 1.10054 m .74683 1.17114 L .72486 1.24669 L p .633 g
F P
0 g
s
.58782 .88745 m .56343 .86396 L .55874 .90543 L p .101 g
F P
0 g
s
.73075 1.06918 m .65958 1.02053 L .62803 .92993 L p .573 g
F P
0 g
s
.62803 .92993 m .68045 .95661 L .73075 1.06918 L p .573 g
F P
0 g
s
.62404 1.18633 m .58116 1.05192 L .59676 1.03524 L p .571 g
F P
0 g
s
.59676 1.03524 m .65702 1.15868 L .62404 1.18633 L p .571 g
F P
0 g
s
.31051 .28202 m .30908 .20416 L .43108 .19177 L p .479 g
F P
0 g
s
.34773 1.00954 m .42213 .98291 L .4556 .92735 L p .121 g
F P
0 g
s
.4556 .92735 m .40303 .95251 L .34773 1.00954 L p .121 g
F P
0 g
s
.54572 .69199 m .54941 .65662 L .46933 .64875 L p .679 g
F P
0 g
s
.50499 .68635 m .54572 .69199 L p .47582 .65243 L .679 g
F P
0 g
s
.47262 .65223 m .50499 .68635 L p .47582 .65243 L .679 g
F P
0 g
s
.50499 .68635 m .46933 .64875 L .38865 .66303 L p .774 g
F P
0 g
s
.55248 .66044 m .43945 .65463 L p .54001 .65645 L .065 g
F P
0 g
s
.54941 .65662 m .54977 .65706 L p .54001 .65645 L .065 g
F P
0 g
s
.54977 .65706 m .55248 .66044 L p .54001 .65645 L .065 g
F P
0 g
s
.46933 .64875 m .47262 .65223 L p .47582 .65243 L .679 g
F P
0 g
s
.33927 .3975 m .39317 .53096 L .33714 .56857 L p .589 g
F P
0 g
s
.33714 .56857 m .26411 .45238 L .33927 .3975 L p .589 g
F P
0 g
s
.65702 1.15868 m .59676 1.03524 L .60282 .99815 L p .559 g
F P
0 g
s
.60282 .99815 m .67019 1.10054 L .65702 1.15868 L p .559 g
F P
0 g
s
.63378 .78759 m .62684 .80752 L .66593 .83251 L p .034 g
F P
0 g
s
.66593 .83251 m .67592 .79022 L .63378 .78759 L p .034 g
F P
0 g
s
.62684 .80752 m .61242 .82334 L .64455 .86813 L p .032 g
F P
0 g
s
.61662 .88831 m .64455 .86813 L .61242 .82334 L p .065 g
F P
0 g
s
.38865 .66303 m .46383 .69293 L .50499 .68635 L p .774 g
F P
0 g
s
.46383 .69293 m .38865 .66303 L .33121 .69543 L p .838 g
F P
0 g
s
.6226 .71634 m .6532 .72462 L .63045 .68843 L p .204 g
F P
0 g
s
.63045 .68843 m .60541 .68254 L .6226 .71634 L p .204 g
F P
0 g
s
.64455 .86813 m .66593 .83251 L .62684 .80752 L p .032 g
F P
0 g
s
.6266 .49288 m .6041 .58975 L .54922 .54229 L p .723 g
F P
0 g
s
.31051 .28202 m .33927 .3975 L .26411 .45238 L p .502 g
F P
0 g
s
.19921 .92174 m .22317 .97288 L .31556 .93966 L p .578 g
F P
0 g
s
.35309 .95617 m .31556 .93966 L .22317 .97288 L p .39 g
F P
0 g
s
.79746 1.08591 m .73075 1.06918 L .68045 .95661 L p .624 g
F P
0 g
s
.68045 .95661 m .73018 .96174 L .79746 1.08591 L p .624 g
F P
0 g
s
.31251 .7356 m .22339 .76995 L .2582 .80972 L p .88 g
F P
0 g
s
.21091 .86333 m .2582 .80972 L .22339 .76995 L p .897 g
F P
0 g
s
.40847 1.01711 m .47979 .95745 L .48564 .9274 L p .078 g
F P
0 g
s
.48564 .9274 m .42213 .98291 L .40847 1.01711 L p .078 g
F P
0 g
s
.74683 1.17114 m .67019 1.10054 L .65958 1.02053 L p .609 g
F P
0 g
s
.65958 1.02053 m .73075 1.06918 L .74683 1.17114 L p .609 g
F P
0 g
s
.27656 1.00503 m .34773 1.00954 L .40303 .95251 L p .205 g
F P
0 g
s
.40303 .95251 m .35309 .95617 L .27656 1.00503 L p .205 g
F P
0 g
s
.63887 .40932 m .6266 .49288 L .5519 .41901 L p .716 g
F P
0 g
s
.5519 .41901 m .55335 .31397 L .63887 .40932 L p .716 g
F P
0 g
s
.22317 .97288 m .27656 1.00503 L .35309 .95617 L p .39 g
F P
0 g
s
.55345 .24486 m .55335 .31397 L .43183 .26815 L p .589 g
F P
0 g
s
.43183 .26815 m .43108 .19177 L .55345 .24486 L p .589 g
F P
0 g
s
.22339 .76995 m .31251 .7356 L .33121 .69543 L p .918 g
F P
0 g
s
.54922 .54229 m .5519 .41901 L .6266 .49288 L p .723 g
F P
0 g
s
.43108 .19177 m .43183 .26815 L .31051 .28202 L p .479 g
F P
0 g
s
.44565 .38449 m .47141 .52095 L .39317 .53096 L p .628 g
F P
0 g
s
.39317 .53096 m .33927 .3975 L .44565 .38449 L p .628 g
F P
0 g
s
.5519 .41901 m .54922 .54229 L .47141 .52095 L p .67 g
F P
0 g
s
.55874 .90543 m .5782 .93675 L .58782 .88745 L p .101 g
F P
0 g
s
.5782 .93675 m .55874 .90543 L .54386 .93843 L p .061 g
F P
0 g
s
.33927 .3975 m .31051 .28202 L .43183 .26815 L p .574 g
F P
0 g
s
.61662 .88831 m .58782 .88745 L .5782 .93675 L p F P
s
.47141 .52095 m .44565 .38449 L .5519 .41901 L p .67 g
F P
0 g
s
.41954 1.03077 m .4854 .97265 L .47979 .95745 L p .068 g
F P
0 g
s
.47979 .95745 m .40847 1.01711 L .41954 1.03077 L p .068 g
F P
0 g
s
.55335 .31397 m .5519 .41901 L .44565 .38449 L p .639 g
F P
0 g
s
.44565 .38449 m .43183 .26815 L .55335 .31397 L p .639 g
F P
0 g
s
.22339 .76995 m .16748 .82939 L .21091 .86333 L p .897 g
F P
0 g
s
.19921 .92174 m .21091 .86333 L .16748 .82939 L p .823 g
F P
0 g
s
.43183 .26815 m .44565 .38449 L .33927 .3975 L p .574 g
F P
0 g
s
.6532 .72462 m .67173 .75127 L .6461 .7238 L p .381 g
F P
0 g
s
.6461 .7238 m .63045 .68843 L .6532 .72462 L p .381 g
F P
0 g
s
.32367 1.03923 m .40847 1.01711 L .42213 .98291 L p .081 g
F P
0 g
s
.42213 .98291 m .34773 1.00954 L .32367 1.03923 L p .081 g
F P
0 g
s
.55248 .66044 m .54941 .65662 L .60541 .68254 L p .102 g
F P
0 g
s
.60541 .68254 m .63045 .68843 L .55248 .66044 L p .102 g
F P
0 g
s
.54386 .93843 m .5462 .97693 L .5782 .93675 L p .061 g
F P
0 g
s
.5462 .97693 m .54386 .93843 L .52268 .96167 L p .055 g
F P
0 g
s
.24667 .71989 m .33121 .69543 L .38865 .66303 L p .786 g
F P
0 g
s
.33121 .69543 m .24667 .71989 L .22339 .76995 L p .918 g
F P
0 g
s
.45321 1.02642 m .50115 .97361 L .4854 .97265 L p .063 g
F P
0 g
s
.4854 .97265 m .41954 1.03077 L .45321 1.02642 L p .063 g
F P
0 g
s
.16748 .82939 m .15263 .90212 L .19921 .92174 L p .823 g
F P
0 g
s
.22317 .97288 m .19921 .92174 L .15263 .90212 L p .562 g
F P
0 g
s
.64455 .86813 m .61662 .88831 L .60154 .93462 L p F P
s
.5782 .93675 m .60154 .93462 L .61662 .88831 L p F P
s
.52268 .96167 m .50007 1.00745 L .5462 .97693 L p .055 g
F P
0 g
s
.50007 1.00745 m .52268 .96167 L .50115 .97361 L p .057 g
F P
0 g
s
.50115 .97361 m .45321 1.02642 L .50007 1.00745 L p .057 g
F P
0 g
s
.16748 .82939 m .22339 .76995 L .24667 .71989 L p .864 g
F P
0 g
s
.24158 1.02213 m .32367 1.03923 L .34773 1.00954 L p .114 g
F P
0 g
s
.40847 1.01711 m .32367 1.03923 L .34009 1.04473 L p .02 g
F P
0 g
s
.34009 1.04473 m .32367 1.03923 L p .30281 1.03058 L .496 g
F P
0 g
s
.34773 1.00954 m .27656 1.00503 L .2583 1.01396 L p .114 g
F P
0 g
s
.34773 1.00954 m .2583 1.01396 L .27088 1.01865 L .114 g
F
.26217 1.01372 m .33067 1.04098 L p .30281 1.03058 L .496 g
F P
0 g
s
.33067 1.04098 m .34009 1.04473 L p .30281 1.03058 L .496 g
F P
0 g
s
.2583 1.01396 m .24158 1.02213 L p .27088 1.01865 L .114 g
F P
0 g
s
.67173 .75127 m .67592 .79022 L .65001 .78057 L p .67 g
F P
0 g
s
.43945 .65463 m .46933 .64875 L .54941 .65662 L p .065 g
F P
0 g
s
.15263 .90212 m .17978 .97201 L .22317 .97288 L p .562 g
F P
0 g
s
.27656 1.00503 m .22317 .97288 L .18838 .97218 L p .218 g
F P
0 g
s
.27656 1.00503 m .18838 .97218 L .19482 .97714 L .218 g
F
.17978 .97201 m .24158 1.02213 L .27656 1.00503 L p F P
0 g
s
.2031 .94666 m .26217 1.01372 L .24919 1.01902 L p .387 g
F P
0 g
s
.24919 1.01902 m .24158 1.02213 L p .23405 1.00735 L .387 g
F P
0 g
s
.23405 1.00735 m .2031 .94666 L .24919 1.01902 L .387 g
F
.18838 .97218 m .17978 .97201 L p .19482 .97714 L .218 g
F P
0 g
s
.38865 .66303 m .32583 .67651 L .24667 .71989 L p .786 g
F P
0 g
s
.32583 .67651 m .38865 .66303 L .46933 .64875 L p F P
s
.60154 .93462 m .5782 .93675 L .5462 .97693 L p .685 g
F P
0 g
s
.60154 .93462 m .62445 .90115 L .64455 .86813 L p F P
s
.66593 .83251 m .64455 .86813 L .62445 .90115 L p .832 g
F P
0 g
s
.34009 1.04473 m .41954 1.03077 L .40847 1.01711 L p .02 g
F P
0 g
s
.67592 .79022 m .66593 .83251 L .64202 .84506 L p .811 g
F P
0 g
s
.64202 .84506 m .65001 .78057 L .67592 .79022 L p .811 g
F P
0 g
s
.46933 .64875 m .43945 .65463 L .32583 .67651 L p F P
s
.62445 .90115 m .64202 .84506 L .66593 .83251 L p .832 g
F P
0 g
s
.63045 .68843 m .6461 .7238 L .55444 .70279 L p .426 g
F P
0 g
s
.55444 .70279 m .55248 .66044 L .63045 .68843 L p .426 g
F P
0 g
s
.24667 .71989 m .1925 .77927 L .16748 .82939 L p .864 g
F P
0 g
s
.15263 .90212 m .16748 .82939 L .1925 .77927 L p F P
s
.5462 .97693 m .54903 .9734 L .60154 .93462 L p .685 g
F P
0 g
s
.54903 .9734 m .5462 .97693 L .50007 1.00745 L p .787 g
F P
0 g
s
.1925 .77927 m .24667 .71989 L .32583 .67651 L p F P
s
.3948 1.03162 m .45321 1.02642 L .41954 1.03077 L p .574 g
F P
0 g
s
.41954 1.03077 m .34009 1.04473 L .3948 1.03162 L p .574 g
F P
0 g
s
.32367 1.03923 m .24158 1.02213 L .26217 1.01372 L p .496 g
F P
0 g
s
.62445 .90115 m .60154 .93462 L .54903 .9734 L p .935 g
F P
0 g
s
.50007 1.00745 m .47244 1.00625 L .54903 .9734 L p .787 g
F P
0 g
s
.47244 1.00625 m .50007 1.00745 L .45321 1.02642 L p .732 g
F P
0 g
s
.45321 1.02642 m .3948 1.03162 L .47244 1.00625 L p .732 g
F P
0 g
s
.1925 .77927 m .17744 .86128 L .15263 .90212 L p F P
s
.17978 .97201 m .15263 .90212 L .17744 .86128 L p .093 g
F P
0 g
s
.6461 .7238 m .65001 .78057 L .55496 .77424 L p .657 g
F P
0 g
s
.55496 .77424 m .55444 .70279 L .6461 .7238 L p .657 g
F P
0 g
s
.55248 .66044 m .55444 .70279 L .42008 .7035 L p .311 g
F P
0 g
s
.42008 .7035 m .43945 .65463 L .55248 .66044 L p .311 g
F P
0 g
s
.24158 1.02213 m .17978 .97201 L .2031 .94666 L p .387 g
F P
0 g
s
.17744 .86128 m .2031 .94666 L .17978 .97201 L p .093 g
F P
0 g
s
.28508 .73157 m .32583 .67651 L .43945 .65463 L p .128 g
F P
0 g
s
.32583 .67651 m .28508 .73157 L .1925 .77927 L p F P
s
.64202 .84506 m .62445 .90115 L .55185 .92976 L p .873 g
F P
0 g
s
.54903 .9734 m .55185 .92976 L .62445 .90115 L p .935 g
F P
0 g
s
.43945 .65463 m .42008 .7035 L .28508 .73157 L p .128 g
F P
0 g
s
.65001 .78057 m .64202 .84506 L .554 .85709 L p .778 g
F P
0 g
s
.554 .85709 m .55496 .77424 L .65001 .78057 L p .778 g
F P
0 g
s
.17744 .86128 m .1925 .77927 L .28508 .73157 L p .166 g
F P
0 g
s
.33693 .98939 m .3948 1.03162 L .34009 1.04473 L p .73 g
F P
0 g
s
.34009 1.04473 m .26217 1.01372 L .33693 .98939 L p .73 g
F P
0 g
s
.55185 .92976 m .554 .85709 L .64202 .84506 L p .873 g
F P
0 g
s
.55185 .92976 m .54903 .9734 L .47244 1.00625 L p .875 g
F P
0 g
s
.55444 .70279 m .55496 .77424 L .41465 .78481 L p .541 g
F P
0 g
s
.41465 .78481 m .42008 .7035 L .55444 .70279 L p .541 g
F P
0 g
s
.44497 .95924 m .47244 1.00625 L .3948 1.03162 L p .809 g
F P
0 g
s
.3948 1.03162 m .33693 .98939 L .44497 .95924 L p .809 g
F P
0 g
s
.47244 1.00625 m .44497 .95924 L .55185 .92976 L p .875 g
F P
0 g
s
.28508 .73157 m .27355 .81659 L .17744 .86128 L p .166 g
F P
0 g
s
.2031 .94666 m .17744 .86128 L .27355 .81659 L p .43 g
F P
0 g
s
.29273 .91077 m .33693 .98939 L .26217 1.01372 L p .605 g
F P
0 g
s
.26217 1.01372 m .2031 .94666 L .29273 .91077 L p .605 g
F P
0 g
s
.27355 .81659 m .28508 .73157 L .42008 .7035 L p .406 g
F P
0 g
s
.554 .85709 m .55185 .92976 L .44497 .95924 L p .771 g
F P
0 g
s
.27355 .81659 m .29273 .91077 L .2031 .94666 L p .43 g
F P
0 g
s
.42008 .7035 m .41465 .78481 L .27355 .81659 L p .406 g
F P
0 g
s
.55496 .77424 m .554 .85709 L .42388 .87831 L p .667 g
F P
0 g
s
.42388 .87831 m .41465 .78481 L .55496 .77424 L p .667 g
F P
0 g
s
.44497 .95924 m .42388 .87831 L .554 .85709 L p .771 g
F P
0 g
s
.42388 .87831 m .44497 .95924 L .33693 .98939 L p .69 g
F P
0 g
s
.33693 .98939 m .29273 .91077 L .42388 .87831 L p .69 g
F P
0 g
s
.29273 .91077 m .27355 .81659 L .41465 .78481 L p .567 g
F P
0 g
s
.41465 .78481 m .42388 .87831 L .29273 .91077 L p .567 g
F P
0 g
s
.25 Mabswid
.52287 0 m
.93124 .57888 L
s
.93124 .57888 m
1 1.3537 L
s
1 1.3537 m
.51861 .85346 L
s
.51861 .85346 m
.52287 0 L
s
.09363 .18048 m
0 1.0137 L
s
0 1.0137 m
.51861 .85346 L
s
.51861 .85346 m
.52287 0 L
s
.52287 0 m
.09363 .18048 L
s
.09363 .18048 m
.52287 0 L
s
.09479 .17999 m
.10111 .1871 L
s
gsave
.08216 .16577 -80.474 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.19695 .13704 m
.20308 .14431 L
s
gsave
.1847 .1225 -80.0038 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.30205 .09285 m
.30798 .10028 L
s
gsave
.29019 .07798 -68.5819 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4102 .04737 m
.41592 .05497 L
s
gsave
.39878 .03217 -74.2463 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.52156 .00055 m
.52704 .00832 L
s
gsave
.51059 -0.01498 -73.9015 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.115 .1715 m
.11876 .17578 L
s
.13531 .16295 m
.13906 .16726 L
s
.15575 .15436 m
.15947 .15869 L
s
.17629 .14572 m
.17999 .15007 L
s
.21773 .1283 m
.22139 .13268 L
s
.23863 .11951 m
.24226 .12391 L
s
.25965 .11067 m
.26326 .1151 L
s
.28079 .10179 m
.28437 .10623 L
s
.32343 .08386 m
.32696 .08834 L
s
.34494 .07482 m
.34844 .07932 L
s
.36657 .06572 m
.37005 .07024 L
s
.38832 .05657 m
.39178 .06111 L
s
.43222 .03812 m
.43562 .0427 L
s
.45436 .02881 m
.45773 .03341 L
s
.47662 .01944 m
.47997 .02407 L
s
.49903 .01003 m
.50234 .01467 L
s
gsave
.23092 .00363 -68.1326 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
MathSubEnd
P
% End of sub-graphic
% Start of sub-graphic
p
0.35119 0.0108235 0.64881 0.443762 MathSubStart
%% Graphics3D
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8  scalefont  setfont
% Scaling calculations
-1.11022e-016 1.05015 -0.0697213 1.05015 [
[.5 .82045 -26.0313 0 ]
[.5 .82045 26.0313 11.75 ]
[.03101 .24391 -13.9375 -9.68197 ]
[.03101 .24391 0 .31803 ]
[.17566 .18707 -20.2869 -10 ]
[.17566 .18707 .33806 0 ]
[.32887 .12688 -7.87434 -10 ]
[.32887 .12688 .56316 0 ]
[.49142 .06307 -13.3539 -10 ]
[.49142 .06307 1.77111 0 ]
[.66418 -0.00472 -7.02463 -10 ]
[.66418 -0.00472 1.41287 0 ]
[.29055 .07541 -7.39613 -10 ]
[.29055 .07541 .54137 0 ]
[.70593 .00492 0 -6.94437 ]
[.70593 .00492 13.9375 3.05563 ]
[.78524 .12593 0 -6.81327 ]
[.78524 .12593 20.625 3.18673 ]
[.85636 .23443 0 -6.69873 ]
[.85636 .23443 8.4375 3.30127 ]
[.92048 .33228 0 -6.59781 ]
[.92048 .33228 15.125 3.40219 ]
[.97859 .42096 0 -6.5082 ]
[.97859 .42096 8.4375 3.4918 ]
[.9185 .21332 0 -6.69873 ]
[.9185 .21332 7.9375 3.30127 ]
[.02382 .26035 -20.625 -3.04587 ]
[.02382 .26035 0 6.95413 ]
[.01528 .3141 -20.625 -3.09606 ]
[.01528 .3141 0 6.90394 ]
[.00644 .36975 -8.4375 -3.14821 ]
[.00644 .36975 0 6.85179 ]
[-0.00272 .4274 -15.125 -3.20243 ]
[-0.00272 .4274 0 6.79757 ]
[-0.01222 .48717 -15.125 -3.25885 ]
[-0.01222 .48717 0 6.74115 ]
[-0.05511 .39255 -7.9375 -3.14821 ]
[-0.05511 .39255 0 6.85179 ]
[ 0 0 0 0 ]
[ 1 .80795 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
gsave
.5 .82045 -87.0313 -4 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 19.75 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 15.750000 moveto
 460.000000 15.750000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 15.750000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(d) show
67.437500 13.187500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
70.437500 11.937500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(2) show
73.687500 13.187500 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
77.500000 13.187500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.687500 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
80.500000 11.937500 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 5.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(2) show
84.250000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
89.375000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(orbital) show
111.062500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.25 Mabswid
.0362 .2546 m
.68734 0 L
s
.0406 .25288 m
.04539 .25737 L
s
gsave
.03101 .24391 -74.9375 -13.682 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.18479 .1965 m
.18935 .20122 L
s
gsave
.17566 .18707 -81.2869 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.33747 .1368 m
.34177 .14176 L
s
gsave
.32887 .12688 -68.8743 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.4994 .07349 m
.50339 .0787 L
s
gsave
.49142 .06307 -74.3539 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.67145 .00621 m
.67509 .01168 L
s
gsave
.66418 -0.00472 -68.0246 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.06879 .24186 m
.07164 .24458 L
s
.0973 .23071 m
.10012 .23346 L
s
.12614 .21944 m
.12893 .22221 L
s
.1553 .20803 m
.15806 .21084 L
s
.21462 .18484 m
.21733 .1877 L
s
.2448 .17304 m
.24748 .17593 L
s
.27533 .1611 m
.27798 .16402 L
s
.30622 .14902 m
.30883 .15197 L
s
.36909 .12444 m
.37163 .12745 L
s
.40108 .11193 m
.40359 .11497 L
s
.43346 .09927 m
.43593 .10234 L
s
.46623 .08646 m
.46867 .08955 L
s
.53297 .06036 m
.53532 .06352 L
s
.56695 .04707 m
.56927 .05026 L
s
.60136 .03362 m
.60363 .03684 L
s
.63619 .02 m
.63841 .02325 L
s
gsave
.29055 .07541 -68.3961 -14 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(x) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.68734 0 m
.97019 .4311 L
s
.69369 .00968 m
.68758 .01206 L
s
gsave
.70593 .00492 -61 -10.9444 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.7729 .1304 m
.76673 .13264 L
s
gsave
.78524 .12593 -61 -10.8133 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.84393 .23866 m
.83771 .24077 L
s
gsave
.85636 .23443 -61 -10.6987 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.90797 .33628 m
.90172 .33827 L
s
gsave
.92048 .33228 -61 -10.5978 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.5) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.96603 .42476 m
.95974 .42665 L
s
gsave
.97859 .42096 -61 -10.5082 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(1) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.71026 .03492 m
.70658 .03633 L
s
.72645 .0596 m
.72276 .06099 L
s
.74228 .08373 m
.73859 .0851 L
s
.75776 .10732 m
.75406 .10868 L
s
.78772 .15299 m
.78401 .15431 L
s
.80222 .17509 m
.79851 .1764 L
s
.81642 .19673 m
.8127 .19802 L
s
.83031 .21791 m
.82659 .21919 L
s
.85726 .25898 m
.85353 .26023 L
s
.87032 .27889 m
.86659 .28013 L
s
.88313 .2984 m
.87938 .29963 L
s
.89567 .31753 m
.89193 .31874 L
s
.92004 .35466 m
.91628 .35585 L
s
.93187 .37269 m
.92811 .37386 L
s
.94347 .39038 m
.93971 .39154 L
s
.95485 .40773 m
.95109 .40888 L
s
gsave
.9185 .21332 -61 -10.6987 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(y) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0362 .2546 m
0 .48397 L
s
.03605 .25557 m
.04216 .25318 L
s
gsave
.02382 .26035 -81.625 -7.04587 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.02755 .30943 m
.03368 .30709 L
s
gsave
.01528 .3141 -81.625 -7.09606 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Math1
%%IncludeFont: Math1
/Math1 findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(-) show
68.500000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Math1
%%+ font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Math1
%%+ Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Math1
%%+ font Helvetica
grestore
.01875 .36519 m
.0249 .36291 L
s
gsave
.00644 .36975 -69.4375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00963 .42296 m
.01581 .42074 L
s
gsave
-0.00272 .4274 -76.125 -7.20243 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.2) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.00018 .48285 m
.00638 .48069 L
s
gsave
-0.01222 .48717 -76.125 -7.25885 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(0.4) show
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.125 Mabswid
.03395 .26886 m
.03762 .26744 L
s
.03183 .28227 m
.03551 .28085 L
s
.0297 .29579 m
.03338 .29438 L
s
.02538 .32319 m
.02906 .32179 L
s
.02319 .33707 m
.02687 .33568 L
s
.02098 .35107 m
.02467 .34969 L
s
.0165 .37944 m
.02019 .37808 L
s
.01423 .39382 m
.01793 .39247 L
s
.01194 .40832 m
.01564 .40698 L
s
.0073 .43773 m
.01101 .43641 L
s
.00495 .45263 m
.00866 .45132 L
s
.00257 .46767 m
.00629 .46637 L
s
gsave
-0.05511 .39255 -68.9375 -7.14821 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 18 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 14.000000 moveto
 460.000000 14.000000 lineto
 460.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 14.000000 lineto
closepath
clip newpath
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(z) show
66.937500 10.750000 moveto
%%IncludeResource: font Helvetica
%%IncludeFont: Helvetica
/Helvetica findfont 8.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Helvetica
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Helvetica
%%DocumentSuppliedFonts:
%%DocumentFonts: font Helvetica
grestore
.25 Mabswid
.0362 .2546 m
0 .48397 L
s
0 .48397 m
.39943 .80795 L
s
.39943 .80795 m
.40657 .60039 L
s
.40657 .60039 m
.0362 .2546 L
s
.68734 0 m
.97019 .4311 L
s
.97019 .4311 m
1 .65038 L
s
1 .65038 m
.70104 .23999 L
s
.70104 .23999 m
.68734 0 L
s
.0362 .2546 m
0 .48397 L
s
0 .48397 m
.70104 .23999 L
s
.70104 .23999 m
.68734 0 L
s
.68734 0 m
.0362 .2546 L
s
.40657 .60039 m
.97019 .4311 L
s
.97019 .4311 m
1 .65038 L
s
1 .65038 m
.39943 .80795 L
s
.39943 .80795 m
.40657 .60039 L
s
0 0 m
1 0 L
1 .80795 L
0 .80795 L
closepath
clip
newpath
.5 Mabswid
.54261 .5056 m .54504 .53108 L .60708 .54431 L p .742 g
F P
0 g
s
.60708 .54431 m .59841 .51251 L .54261 .5056 L p .742 g
F P
0 g
s
.61349 .5767 m .60708 .54431 L .54504 .53108 L p .714 g
F P
0 g
s
.59841 .51251 m .60708 .54431 L .66555 .53385 L p .645 g
F P
0 g
s
.60708 .54431 m .61349 .5767 L .67643 .56689 L p .605 g
F P
0 g
s
.67643 .56689 m .66555 .53385 L .60708 .54431 L p .605 g
F P
0 g
s
.54504 .53108 m .54682 .55681 L .61349 .5767 L p .714 g
F P
0 g
s
.6172 .60751 m .61349 .5767 L .54682 .55681 L p .687 g
F P
0 g
s
.61349 .5767 m .6172 .60751 L .68278 .59877 L p .561 g
F P
0 g
s
.68278 .59877 m .67643 .56689 L .61349 .5767 L p .561 g
F P
0 g
s
.53968 .48207 m .54261 .5056 L .59841 .51251 L p .773 g
F P
0 g
s
.59841 .51251 m .58805 .48334 L .53968 .48207 L p .773 g
F P
0 g
s
.58805 .48334 m .59841 .51251 L .65091 .50185 L p .685 g
F P
0 g
s
.66555 .53385 m .65091 .50185 L .59841 .51251 L p .645 g
F P
0 g
s
.65091 .50185 m .66555 .53385 L .69024 .50577 L p .487 g
F P
0 g
s
.66555 .53385 m .67643 .56689 L .70311 .53357 L p .409 g
F P
0 g
s
.70311 .53357 m .69024 .50577 L .66555 .53385 L p .409 g
F P
0 g
s
.67643 .56689 m .68278 .59877 L .71053 .56088 L p .295 g
F P
0 g
s
.71053 .56088 m .70311 .53357 L .67643 .56689 L p .295 g
F P
0 g
s
.54682 .55681 m .54782 .58101 L .6172 .60751 L p .687 g
F P
0 g
s
.61793 .63454 m .6172 .60751 L .54782 .58101 L p .656 g
F P
0 g
s
.6172 .60751 m .61793 .63454 L .68408 .62721 L p .504 g
F P
0 g
s
.68408 .62721 m .68278 .59877 L .6172 .60751 L p .504 g
F P
0 g
s
.65091 .50185 m .63346 .47297 L .58805 .48334 L p .685 g
F P
0 g
s
.63346 .47297 m .65091 .50185 L .67287 .47942 L p .545 g
F P
0 g
s
.69024 .50577 m .67287 .47942 L .65091 .50185 L p .487 g
F P
0 g
s
.68278 .59877 m .68408 .62721 L .71188 .58571 L p .102 g
F P
0 g
s
.71188 .58571 m .71053 .56088 L .68278 .59877 L p .102 g
F P
0 g
s
.53646 .46197 m .53968 .48207 L .58805 .48334 L p .807 g
F P
0 g
s
.58805 .48334 m .57667 .45859 L .53646 .46197 L p .807 g
F P
0 g
s
.57667 .45859 m .58805 .48334 L .63346 .47297 L p .727 g
F P
0 g
s
.54782 .58101 m .54796 .60192 L .61793 .63454 L p .656 g
F P
0 g
s
.61553 .65576 m .61793 .63454 L .54796 .60192 L p .618 g
F P
0 g
s
.61793 .63454 m .61553 .65576 L .68012 .65004 L p .418 g
F P
0 g
s
.68012 .65004 m .68408 .62721 L .61793 .63454 L p .418 g
F P
0 g
s
.63346 .47297 m .61432 .44894 L .57667 .45859 L p .727 g
F P
0 g
s
.61432 .44894 m .63346 .47297 L .65219 .45624 L p .589 g
F P
0 g
s
.67287 .47942 m .65219 .45624 L .63346 .47297 L p .545 g
F P
0 g
s
.68408 .62721 m .68012 .65004 L .70693 .60612 L p .012 g
F P
0 g
s
.70693 .60612 m .71188 .58571 L .68408 .62721 L p .012 g
F P
0 g
s
.51118 .48277 m .51008 .49635 L .54504 .53108 L p .837 g
F P
0 g
s
.51346 .48443 m .51118 .48277 L p .51312 .48554 L .837 g
F P
0 g
s
.5125 .47024 m .51118 .48277 L .54261 .5056 L p .854 g
F P
0 g
s
.54261 .5056 m .53968 .48207 L .51682 .47212 L p .854 g
F P
0 g
s
.51639 .47194 m .51539 .4715 L p .51593 .47427 L .854 g
F P
0 g
s
.51539 .4715 m .51434 .47104 L p .5148 .47295 L .854 g
F P
0 g
s
.5148 .47295 m .51434 .47104 L .51421 .47225 L .854 g
F
.51434 .47104 m .5125 .47024 L p .51421 .47225 L F P
0 g
s
.51393 .45958 m .5125 .47024 L .53968 .48207 L p .869 g
F P
0 g
s
.51954 .46017 m .51731 .45994 L p .51826 .46336 L .869 g
F P
0 g
s
.51731 .45994 m .51393 .45958 L p .51643 .46176 L .869 g
F P
0 g
s
.51826 .46336 m .51731 .45994 L .51643 .46176 L .869 g
F
.5154 .45141 m .51393 .45958 L .53646 .46197 L p .878 g
F P
0 g
s
.5125 .47024 m .51714 .45714 L .517 .4579 L .005 g
F
.51692 .45835 m .51733 .457 L .51716 .45768 L F
.5125 .47024 m .517 .4579 L .51692 .45835 L F
.517 .4579 m .51733 .457 L .51692 .45835 L F
.51665 .45874 m .51662 .45884 L p .51726 .45742 L F P
0 g
s
.44609 .51221 m .4432 .52789 L .51198 .46888 L p .234 g
F P
0 g
s
.51659 .45891 m .51658 .45893 L p .51719 .45762 L .005 g
F P
0 g
s
.51658 .45893 m .513 .46883 L p .51718 .45765 L .005 g
F P
0 g
s
.51718 .45765 m .51719 .45762 L .51658 .45893 L .005 g
F
.51307 .4687 m .51727 .45739 L .513 .46883 L F
.51727 .45739 m .51727 .45739 L .513 .46883 L F
.51734 .45721 m .51734 .45721 L .513 .46883 L F
.513 .46883 m .5125 .47024 L p .51307 .4687 L F P
0 g
s
.51593 .47427 m .51539 .4715 L .5148 .47295 L .854 g
F
.51682 .47212 m .51639 .47194 L p .5161 .47447 L F P
0 g
s
.5161 .47447 m .51639 .47194 L .51593 .47427 L .854 g
F
.54261 .5056 m .51682 .47212 L .5161 .47447 L F
.54504 .53108 m .54261 .5056 L .51346 .48443 L p .837 g
F P
0 g
s
.54504 .53108 m .51346 .48443 L .51312 .48554 L .837 g
F
.54682 .55681 m .54504 .53108 L .51008 .49635 L p .817 g
F P
0 g
s
.51008 .49635 m .50928 .51003 L .54682 .55681 L p .817 g
F P
0 g
s
.54782 .58101 m .54682 .55681 L .50928 .51003 L p .795 g
F P
0 g
s
.50928 .51003 m .50883 .52279 L .54782 .58101 L p .795 g
F P
0 g
s
.54796 .60192 m .54782 .58101 L .50883 .52279 L p .769 g
F P
0 g
s
.50883 .52279 m .50878 .53366 L .54796 .60192 L p .769 g
F P
0 g
s
.54722 .61794 m .54796 .60192 L .50878 .53366 L p .732 g
F P
0 g
s
.50878 .53366 m .50913 .54178 L .54722 .61794 L p .732 g
F P
0 g
s
.51814 .46702 m .51743 .46844 L .51727 .46875 L p .69 g
F P
0 g
s
.51673 .46943 m .5167 .46989 L p .51607 .47028 L .022 g
F P
0 g
s
.51602 .47125 m .51607 .47028 L .5167 .46989 L .022 g
F
.51673 .46943 m .51743 .46844 L .5174 .46899 L p .026 g
F P
0 g
s
.5174 .46899 m .51367 .53768 L p .51665 .47104 L .026 g
F P
0 g
s
.51665 .47104 m .51673 .46943 L .5174 .46899 L .026 g
F
.51882 .46531 m .51814 .46702 L .51799 .46736 L p .819 g
F P
0 g
s
.51743 .46844 m .51673 .46943 L p .51714 .46888 L F P
s
.51517 .528 m .51367 .53768 L .51743 .46844 L p .029 g
F P
0 g
s
.5167 .46989 m .51673 .46943 L p .51671 .46988 L .026 g
F P
0 g
s
.51118 .48277 m .5125 .47024 L .51432 .46518 L p .005 g
F P
0 g
s
.51118 .48277 m .51432 .46518 L .51461 .46604 L .005 g
F
.51118 .48277 m .51538 .46226 L .51494 .46639 L F
.51538 .46226 m .51627 .4598 L p .51494 .46639 L F P
0 g
s
.51627 .4598 m .51675 .45846 L p .51597 .46186 L .005 g
F P
0 g
s
.51494 .46639 m .51627 .4598 L .51597 .46186 L .005 g
F
.51213 .47853 m .51483 .46682 L .51209 .47864 L F
.51483 .46682 m .51646 .45974 L .51481 .46686 L F
.51483 .46682 m .51481 .46686 L .51209 .47864 L F
.51646 .45974 m .51646 .45974 L .51631 .46039 L F
.51608 .46138 m .51608 .46138 L .51568 .46311 L F
.51568 .46311 m .51568 .46311 L .51209 .47864 L F
.51608 .46138 m .51568 .46311 L .51568 .46311 L F
.51656 .45929 m .51656 .4593 L .51209 .47864 L F
.51214 .47851 m .51219 .47831 L .51213 .47853 L F
.5125 .47024 m .51393 .45958 L .51714 .45714 L p F P
0 g
s
.51267 .46985 m .51198 .46888 L .51081 .4714 L .187 g
F
.51219 .47831 m .51656 .45929 L .51213 .47853 L .005 g
F
.51656 .45929 m .51656 .45929 L .51213 .47853 L F
.51664 .45896 m .51664 .45896 L .51657 .45925 L F
.51646 .45974 m .51646 .45974 L .51213 .47853 L F
.51675 .45846 m .51611 .46031 L .51606 .46056 L p F P
0 g
s
.4432 .52789 m .44287 .54162 L .51242 .47054 L p .187 g
F P
0 g
s
.51601 .4608 m .51209 .47864 L p .51656 .4593 L .005 g
F P
0 g
s
.44287 .54162 m .44515 .55232 L .51118 .47386 L p .117 g
F P
0 g
s
.51167 .48061 m .51657 .45925 L .51164 .48068 L .005 g
F
.51657 .45925 m .51657 .45925 L .51164 .48068 L F
.51477 .46834 m .51475 .46827 L .51563 .46315 L F
.51557 .46353 m .51563 .46315 L .51611 .46031 L F
.51477 .46834 m .51563 .46315 L .51557 .46353 L F
.51452 .46754 m .51512 .46483 L p .51475 .46827 L F P
0 g
s
.51512 .46483 m .51611 .46031 L p .51475 .46827 L .005 g
F P
0 g
s
.51008 .49635 m .51452 .46754 L .51477 .46834 L .005 g
F
.51085 .49171 m .51202 .4847 L .51081 .49181 L F
.51202 .4847 m .51562 .46326 L .51191 .48531 L F
.51562 .46326 m .51611 .46031 L .51561 .46327 L F
.51562 .46326 m .51561 .46327 L .51191 .48531 L F
.51202 .4847 m .51191 .48531 L .51081 .49181 L F
.51611 .46031 m .51558 .46234 L .51554 .46262 L p F P
0 g
s
.51242 .47054 m .51367 .46926 L p .51267 .46985 L .187 g
F P
0 g
s
.51552 .46268 m .51546 .46305 L p .51599 .46102 L .005 g
F P
0 g
s
.51546 .46305 m .51158 .48707 L p .51591 .46152 L .005 g
F P
0 g
s
.51591 .46152 m .51599 .46102 L .51546 .46305 L .005 g
F
.51462 .46829 m .51522 .46454 L p .51475 .46815 L .006 g
F P
0 g
s
.51376 .46894 m .51317 .4685 L .51267 .46985 L .187 g
F
.51191 .48531 m .51591 .46151 L .51158 .48707 L .005 g
F
.51591 .46151 m .51599 .46101 L .51591 .46151 L F
.51591 .46151 m .51591 .46151 L .51158 .48707 L F
.51312 .47118 m .51242 .47054 L .51162 .47268 L .117 g
F
.51599 .46101 m .51599 .46102 L .51158 .48707 L .005 g
F
.516 .46095 m .516 .46096 L .51599 .46102 L F
.51599 .46102 m .51599 .46102 L .51158 .48707 L F
.516 .46095 m .51599 .46102 L .51599 .46102 L F
.51047 .49402 m .51562 .46326 L .51044 .4941 L F
.51562 .46326 m .51611 .46031 L .51562 .46326 L F
.51562 .46326 m .51562 .46326 L .51044 .4941 L F
.51178 .47569 m .50975 .47769 L .50914 .47925 L .023 g
F
.50928 .51003 m .51522 .46454 L .51523 .46497 L .006 g
F
.50928 .51003 m .51462 .46829 L .51475 .46815 L F
.51158 .48707 m .51081 .49181 L p .51191 .48531 L .005 g
F P
0 g
s
.51081 .49181 m .51044 .4941 L p .51085 .49171 L .005 g
F P
0 g
s
.51044 .4941 m .51008 .49635 L p .51047 .49402 L .005 g
F P
0 g
s
.50928 .51003 m .51008 .49635 L .51462 .46829 L p .006 g
F P
0 g
s
.51114 .49598 m .51537 .46393 L .51076 .49874 L .006 g
F
.51537 .46393 m .51537 .46394 L .51076 .49874 L F
.51522 .46454 m .51558 .46234 L p .51523 .46497 L F P
0 g
s
.51552 .46279 m .51552 .46279 L .51537 .46394 L .006 g
F
.51537 .46394 m .51537 .46394 L .51076 .49874 L F
.51552 .46279 m .51537 .46394 L .51537 .46394 L F
.51167 .48062 m .51164 .48068 L .51163 .48072 L .005 g
F
.51167 .48061 m .51164 .48068 L .51167 .48062 L F
.51209 .47864 m .51163 .48072 L p .51214 .47851 L F P
0 g
s
.51163 .48072 m .51118 .48277 L p .51167 .48061 L .005 g
F P
0 g
s
.51008 .49635 m .51118 .48277 L .51429 .46863 L p .005 g
F P
0 g
s
.51369 .47197 m .51118 .47386 L .51046 .47576 L .068 g
F
.51558 .46234 m .5152 .46438 L .51518 .46453 L p .006 g
F P
0 g
s
.51518 .46453 m .51112 .49579 L p .51552 .46279 L .006 g
F P
0 g
s
.51552 .46279 m .51558 .46234 L .51518 .46453 L .006 g
F
.51454 .4678 m .51435 .46684 L .51411 .4677 L .187 g
F
.51144 .49358 m .51552 .46279 L .51112 .49579 L .006 g
F
.51552 .46279 m .51558 .46234 L .51552 .46279 L F
.51552 .46279 m .51552 .46279 L .51112 .49579 L F
.51449 .46807 m .51351 .46756 L .51317 .4685 L .187 g
F
.51154 .49793 m .51518 .46457 L .5115 .4983 L .007 g
F
.51518 .46457 m .51518 .46457 L .5115 .4983 L F
.51519 .46444 m .51519 .46444 L .5115 .4983 L F
.51125 .50067 m .51498 .46648 L p .51222 .49637 L .01 g
F P
0 g
s
.51178 .49573 m .5152 .46438 L .5116 .4974 L .007 g
F
.5101 .50378 m .51114 .49598 L .51006 .50401 L .006 g
F
.51114 .49598 m .51076 .49874 L .51006 .50401 L F
.51112 .49579 m .51006 .50401 L p .51144 .49358 L F P
0 g
s
.51006 .50401 m .50928 .51003 L p .5101 .50378 L .006 g
F P
0 g
s
.50883 .52279 m .50928 .51003 L .5152 .46438 L p .007 g
F P
0 g
s
.51482 .46796 m .51454 .4678 L .51449 .46807 L .187 g
F
.50878 .53366 m .51498 .46648 L .51498 .46651 L .01 g
F
.51609 .46989 m .51673 .46943 L p .51607 .47028 L .022 g
F P
0 g
s
.51553 .47011 m .51509 .47607 L p .51512 .47007 L .016 g
F P
0 g
s
.54561 .62771 m .54722 .61794 L .50913 .54178 L p .678 g
F P
0 g
s
.50913 .54178 m .50986 .54648 L .52374 .57803 L p .678 g
F P
0 g
s
.52374 .57803 m .54561 .62771 L p .52038 .56827 L .678 g
F P
0 g
s
.52038 .56827 m .50913 .54178 L .52374 .57803 L .678 g
F
.54326 .63036 m .54561 .62771 L .50986 .54648 L p .591 g
F P
0 g
s
.50986 .54648 m .51091 .54736 L .5253 .58428 L p .591 g
F P
0 g
s
.5253 .58428 m .54326 .63036 L p .52542 .58556 L .591 g
F P
0 g
s
.52542 .58556 m .50986 .54648 L .5253 .58428 L .591 g
F
.54031 .62553 m .54326 .63036 L .53054 .59772 L p .475 g
F P
0 g
s
.53698 .61355 m .54031 .62553 L .52503 .58138 L p .038 g
F P
0 g
s
.52503 .58138 m .51221 .54434 L p .52488 .57974 L .038 g
F P
0 g
s
.53698 .61355 m .52503 .58138 L .52488 .57974 L .038 g
F
.53054 .59772 m .51091 .54736 L p .53819 .61988 L .475 g
F P
0 g
s
.54031 .62553 m .53054 .59772 L .53819 .61988 L .475 g
F
.51509 .47607 m .50986 .54648 L p .51458 .47787 L .016 g
F P
0 g
s
.51458 .47787 m .51512 .47007 L .51509 .47607 L .016 g
F
.51483 .4696 m .50878 .53366 L p .51475 .46899 L .01 g
F P
0 g
s
.51008 .49635 m .51429 .46863 L .51437 .46854 L .005 g
F
.514 .4754 m .5116 .4974 L p .51399 .47544 L .007 g
F P
0 g
s
.51367 .53768 m .51221 .54434 L .5167 .46989 L p .026 g
F P
0 g
s
.51367 .53768 m .5167 .46989 L .51671 .46988 L .026 g
F
.51448 .47098 m .514 .4754 L p .51451 .47075 L .007 g
F P
0 g
s
.51399 .47544 m .5152 .46438 L .514 .4754 L .007 g
F
.50913 .54178 m .50878 .53366 L .51363 .48237 L p .012 g
F P
0 g
s
.51363 .48237 m .51483 .4696 L p .514 .48045 L .012 g
F P
0 g
s
.514 .48045 m .51483 .4696 L .51485 .46969 L .012 g
F
.51609 .46989 m .51091 .54736 L p .51554 .47012 L .019 g
F P
0 g
s
.51556 .46978 m .51609 .46989 L p .51554 .47012 L .019 g
F P
0 g
s
.51091 .54736 m .50986 .54648 L .51509 .47607 L p .019 g
F P
0 g
s
.51509 .47607 m .51556 .46978 L p .51522 .47551 L .019 g
F P
0 g
s
.51091 .54736 m .51509 .47607 L .51522 .47551 L .019 g
F
.51511 .47003 m .51437 .47891 L p .51489 .46974 L .012 g
F P
0 g
s
.514 .48045 m .51499 .46791 L .51437 .47891 L .012 g
F
.50986 .54648 m .50913 .54178 L .51437 .47891 L p .016 g
F P
0 g
s
.51576 .47087 m .51545 .47132 L .51576 .47088 L .005 g
F
.51437 .47891 m .51511 .47003 L p .51458 .47787 L .016 g
F P
0 g
s
.51458 .47787 m .51511 .47003 L .51512 .47007 L .016 g
F
.51545 .47132 m .50797 .48241 L .51545 .47134 L .005 g
F
.51545 .47132 m .51545 .47134 L .51576 .47088 L F
.51511 .47003 m .51518 .4691 L p .51512 .47007 L .016 g
F P
0 g
s
.50986 .54648 m .51437 .47891 L .51458 .47787 L .016 g
F
.51221 .54434 m .51091 .54736 L .51609 .46989 L p .022 g
F P
0 g
s
.47487 .55213 m .51743 .46844 L p .47767 .53755 L F P
s
.46533 .559 m .47487 .55213 L p .47767 .53755 L F P
s
.51409 .47501 m .50148 .4998 L p .51409 .47499 L .69 g
F P
0 g
s
.50148 .4998 m .48013 .54179 L p .50139 .49998 L .69 g
F P
0 g
s
.48013 .54179 m .47487 .55213 L p .48006 .54191 L .69 g
F P
0 g
s
.50139 .49998 m .48013 .54179 L .48006 .54191 L .69 g
F
.41629 .60696 m .43853 .59163 L .47472 .55229 L p .833 g
F P
0 g
s
.51409 .47499 m .50148 .4998 L .50139 .49998 L .69 g
F
.5169 .46948 m .51409 .47501 L p .51704 .4692 L F P
0 g
s
.51704 .4692 m .51409 .47501 L .51409 .47499 L .69 g
F
.51727 .46875 m .5169 .46948 L p .51768 .46793 L F P
0 g
s
.51768 .46793 m .5169 .46948 L .51704 .4692 L .69 g
F
.51743 .46844 m .51609 .47054 L .51613 .47069 L 0 g
F
.5167 .46989 m .51221 .54434 L p .51602 .47125 L .022 g
F P
0 g
s
.47487 .55213 m .48471 .54143 L .51729 .46892 L p .69 g
F P
0 g
s
.51814 .46702 m .51727 .46875 L .51768 .46793 L .69 g
F
.51743 .46844 m .51814 .46702 L .51811 .46767 L p .029 g
F P
0 g
s
.51811 .46767 m .51517 .528 L p .51736 .47034 L .029 g
F P
0 g
s
.51736 .47034 m .51743 .46844 L .51811 .46767 L .029 g
F
.53968 .48207 m .51954 .46017 L .51826 .46336 L .869 g
F
.51941 .46345 m .51882 .46531 L .51868 .46567 L p .882 g
F P
0 g
s
.51729 .46892 m .51814 .46702 L p .51635 .47055 L .69 g
F P
0 g
s
.51635 .47055 m .47487 .55213 L .51729 .46892 L .69 g
F
.51729 .46892 m .51684 .46992 L p .51691 .46968 L .819 g
F P
0 g
s
.51799 .46736 m .51729 .46892 L p .51838 .46629 L .819 g
F P
0 g
s
.51838 .46629 m .51729 .46892 L .51645 .47059 L .819 g
F
.51882 .46531 m .51799 .46736 L .51838 .46629 L F
.51661 .51616 m .51517 .528 L .51811 .46767 L p .031 g
F P
0 g
s
.51811 .46767 m .51814 .46702 L p .51812 .4676 L .031 g
F P
0 g
s
.51661 .51616 m .51811 .46767 L .51812 .4676 L .031 g
F
.51611 .47146 m .51607 .47163 L .51569 .47248 L .819 g
F
.51569 .47248 m .48991 .52986 L p .51572 .47232 L F P
0 g
s
.48991 .52986 m .48471 .54143 L p .48987 .52993 L .819 g
F P
0 g
s
.51572 .47232 m .48991 .52986 L .48987 .52993 L .819 g
F
.39492 .61571 m .41629 .60696 L .47487 .55213 L p 0 g
F P
s
.47487 .55213 m .46533 .559 L .41591 .5988 L p F P
s
.41591 .5988 m .39492 .61571 L p .427 .59019 L F P
s
.47487 .55213 m .41591 .5988 L .427 .59019 L F
.48471 .54143 m .47487 .55213 L .41629 .60696 L p .833 g
F P
0 g
s
.47603 .54974 m .4628 .56242 L .47472 .55229 L .833 g
F
.4628 .56242 m .41629 .60696 L .46922 .55743 L F
.4628 .56242 m .46922 .55743 L .47472 .55229 L F
.47472 .55229 m .48471 .54143 L p .47603 .54974 L F P
0 g
s
.51611 .47146 m .51569 .47248 L .51572 .47232 L .819 g
F
.51618 .47139 m .51607 .47163 L p .51622 .4712 L F P
0 g
s
.51622 .4712 m .51607 .47163 L .51611 .47146 L .819 g
F
.51684 .46992 m .51618 .47139 L p .51691 .46968 L F P
0 g
s
.51691 .46968 m .51618 .47139 L .51622 .4712 L .819 g
F
.51645 .47059 m .51691 .46968 L .51545 .47294 L F
.51548 .47275 m .51545 .47294 L .48471 .54143 L F
.51645 .47059 m .51545 .47294 L .51548 .47275 L F
.51814 .46702 m .51882 .46531 L .51661 .51616 L p .031 g
F P
0 g
s
.4872 .53587 m .49414 .52792 L .51882 .46531 L p .819 g
F P
0 g
s
.49414 .52792 m .48471 .54143 L .43853 .59163 L p .927 g
F P
0 g
s
.43853 .59163 m .45997 .57124 L .48386 .54095 L p .927 g
F P
0 g
s
.48386 .54095 m .49414 .52792 L p .48569 .53761 L .927 g
F P
0 g
s
.48569 .53761 m .43853 .59163 L .48386 .54095 L .927 g
F
.51868 .46567 m .51831 .46659 L p .51901 .46448 L .882 g
F P
0 g
s
.51831 .46659 m .49414 .52792 L p .51843 .46596 L .882 g
F P
0 g
s
.51901 .46448 m .51831 .46659 L .51843 .46596 L .882 g
F
.51941 .46345 m .51868 .46567 L .51901 .46448 L F
.51609 .46989 m .51509 .47143 L p .51648 .46972 L .005 g
F P
0 g
s
.51673 .46943 m .51609 .46989 L p .51648 .46972 L .005 g
F P
0 g
s
.51622 .47018 m .51576 .47087 L .51623 .47018 L .005 g
F
.51576 .47087 m .51576 .47088 L .51623 .47018 L F
.51499 .46791 m .51518 .4691 L .51511 .47003 L p .012 g
F P
0 g
s
.51489 .46974 m .51499 .46791 L .51511 .47003 L .012 g
F
.51437 .47891 m .50913 .54178 L p .514 .48045 L F P
0 g
s
.51524 .47171 m .50797 .48241 L .50751 .48356 L .005 g
F
.51623 .47018 m .51523 .47167 L .51524 .47171 L F
.51523 .47167 m .50797 .48241 L .51524 .47171 L F
.51483 .4696 m .51499 .46791 L p .51485 .46969 L .012 g
F P
0 g
s
.50913 .54178 m .51363 .48237 L .514 .48045 L .012 g
F
.51673 .46943 m .51622 .47018 L .51623 .47018 L .005 g
F
.51556 .46978 m .51553 .47011 L p .51516 .4694 L .016 g
F P
0 g
s
.51512 .47007 m .51516 .4694 L .51553 .47011 L .016 g
F
.51518 .4691 m .51556 .46978 L p .51516 .4694 L F P
0 g
s
.51556 .46978 m .51533 .4701 L p .51576 .46998 L .023 g
F P
0 g
s
.51609 .46989 m .51556 .46978 L p .51576 .46998 L .023 g
F P
0 g
s
.51509 .47143 m .50797 .48241 L p .51513 .47179 L .005 g
F P
0 g
s
.51673 .46943 m .51509 .47143 L .51513 .47179 L .005 g
F
.51499 .46791 m .51483 .4696 L p .51498 .46651 L .01 g
F P
0 g
s
.51475 .46899 m .51498 .46651 L .51483 .4696 L .01 g
F
.515 .46629 m .51499 .46791 L p .51498 .46651 L F P
0 g
s
.5152 .46438 m .515 .46629 L .51498 .46648 L p .007 g
F P
0 g
s
.51498 .46648 m .51448 .47098 L p .51512 .46508 L .007 g
F P
0 g
s
.51512 .46508 m .5152 .46438 L .51498 .46648 L .007 g
F
.51464 .46965 m .51346 .47 L .51312 .47118 L .117 g
F
.51501 .46928 m .51467 .46944 L .51464 .46965 L F
.51467 .46944 m .51346 .47 L .51464 .46965 L F
.51518 .4691 m .51501 .4692 L .51501 .46928 L F
.51451 .47075 m .5152 .46438 L .51448 .47098 L .007 g
F
.51501 .4692 m .51346 .47 L .51501 .46928 L .117 g
F
.51499 .46791 m .51483 .46787 L .51482 .46796 L .187 g
F
.51483 .46787 m .51454 .4678 L .51482 .46796 L F
.51454 .4678 m .51351 .46756 L .51449 .46807 L F
.51483 .46787 m .51462 .46741 L .51454 .4678 L F
.51462 .46741 m .51435 .46684 L .51454 .4678 L F
.51518 .46457 m .51519 .46444 L .51518 .46457 L .007 g
F
.51499 .46791 m .51485 .46767 L .51483 .46787 L .187 g
F
.51485 .46767 m .51435 .46684 L .51483 .46787 L F
.51556 .46978 m .51381 .47141 L .51369 .47197 L .068 g
F
.51381 .47141 m .51118 .47386 L .51369 .47197 L F
.51556 .46978 m .51518 .4691 L .514 .47052 L p F P
0 g
s
.514 .47052 m .51118 .47386 L p .51381 .47141 L .068 g
F P
0 g
s
.51556 .46978 m .514 .47052 L .51381 .47141 L .068 g
F
.51537 .46393 m .51552 .46279 L .51537 .46394 L .006 g
F
.51552 .46279 m .51558 .46234 L .51552 .46279 L F
.51498 .46648 m .515 .46629 L p .51498 .46651 L .01 g
F P
0 g
s
.51499 .46791 m .515 .46629 L .5148 .46646 L p .187 g
F P
0 g
s
.5148 .46646 m .51435 .46684 L p .51463 .46731 L .187 g
F P
0 g
s
.51499 .46791 m .5148 .46646 L .51463 .46731 L .187 g
F
.51534 .4709 m .51204 .47487 L .51178 .47569 L .023 g
F
.51609 .46989 m .51534 .47081 L .51534 .4709 L F
.51534 .47081 m .51204 .47487 L .51534 .4709 L F
.51204 .47487 m .50975 .47769 L .51178 .47569 L F
.51435 .46684 m .51351 .46756 L p .51411 .4677 L .187 g
F P
0 g
s
.51346 .47 m .51242 .47054 L .51312 .47118 L .117 g
F
.51429 .46863 m .51452 .46754 L p .51437 .46854 L .005 g
F P
0 g
s
.51499 .46791 m .51396 .46824 L .51376 .46894 L .187 g
F
.51396 .46824 m .51317 .4685 L .51376 .46894 L F
.51599 .46101 m .516 .46095 L .51599 .46102 L .005 g
F
.516 .46095 m .51611 .46031 L .516 .46096 L F
.51464 .46965 m .51312 .47118 L .51463 .46976 L .117 g
F
.51312 .47118 m .51162 .47268 L .5131 .47125 L F
.51312 .47118 m .5131 .47125 L .51463 .46976 L F
.51499 .46791 m .51367 .46926 L p .51501 .4692 L F P
0 g
s
.51367 .46926 m .51242 .47054 L p .51346 .47 L .117 g
F P
0 g
s
.51501 .4692 m .51367 .46926 L .51346 .47 L .117 g
F
.51367 .46926 m .51499 .46791 L p .51376 .46894 L .187 g
F P
0 g
s
.51376 .46894 m .51267 .46985 L .51367 .46926 L .187 g
F
.51554 .46262 m .51552 .46268 L p .516 .46096 L .005 g
F P
0 g
s
.51599 .46102 m .516 .46096 L .51552 .46268 L .005 g
F
.516 .46096 m .51611 .46031 L .51554 .46262 L F
.51118 .47386 m .51296 .47175 L p .51162 .47268 L .117 g
F P
0 g
s
.51296 .47175 m .51463 .46976 L p .5131 .47125 L .117 g
F P
0 g
s
.5131 .47125 m .51162 .47268 L .51296 .47175 L .117 g
F
.51533 .4701 m .51219 .47436 L p .51534 .47081 L .023 g
F P
0 g
s
.51219 .47436 m .50975 .47769 L p .51204 .47487 L .023 g
F P
0 g
s
.51534 .47081 m .51219 .47436 L .51204 .47487 L .023 g
F
.51609 .46989 m .51533 .4701 L .51534 .47081 L F
.51519 .46444 m .5152 .46438 L .51519 .46444 L .007 g
F
.51463 .46976 m .51518 .4691 L p .51464 .46965 L .117 g
F P
0 g
s
.51162 .47268 m .44287 .54162 L .51118 .47386 L .117 g
F
.51657 .45925 m .51675 .45846 L .51657 .45925 L .005 g
F
.50938 .5178 m .51154 .49793 L .50936 .51799 L .007 g
F
.51154 .49793 m .5115 .4983 L .50936 .51799 L F
.51267 .46985 m .4432 .52789 L .51242 .47054 L .187 g
F
.51606 .46056 m .51601 .4608 L p .51664 .45897 L .005 g
F P
0 g
s
.51656 .4593 m .51664 .45897 L .51601 .4608 L .005 g
F
.51664 .45897 m .51675 .45846 L .51606 .46056 L F
.51365 .46719 m .51351 .46555 L .51236 .46805 L .234 g
F
.515 .46629 m .51413 .46586 L .51365 .46719 L F
.51413 .46586 m .51351 .46555 L .51365 .46719 L F
.51657 .45925 m .51657 .45925 L .51646 .45974 L .005 g
F
.51657 .45925 m .51646 .45974 L .51646 .45974 L F
.51664 .45896 m .51657 .45925 L .51657 .45925 L F
.51198 .46888 m .51351 .46756 L p .51236 .46805 L .234 g
F P
0 g
s
.51351 .46756 m .515 .46629 L p .51365 .46719 L .234 g
F P
0 g
s
.51365 .46719 m .51236 .46805 L .51351 .46756 L .234 g
F
.51317 .4685 m .51198 .46888 L .51267 .46985 L .187 g
F
.51656 .45929 m .51664 .45896 L .51656 .45929 L .005 g
F
.51664 .45896 m .51675 .45846 L .51664 .45896 L F
.51432 .46518 m .51538 .46226 L p .51461 .46604 L F P
0 g
s
.515 .46629 m .5152 .46438 L .51449 .46487 L p .234 g
F P
0 g
s
.51449 .46487 m .51351 .46555 L p .51413 .46586 L .234 g
F P
0 g
s
.515 .46629 m .51449 .46487 L .51413 .46586 L .234 g
F
.50797 .48241 m .45679 .5614 L p .50751 .48356 L .005 g
F P
0 g
s
.50975 .47769 m .4499 .5591 L p .50914 .47925 L .023 g
F P
0 g
s
.51631 .46038 m .51631 .46039 L .51608 .46138 L .005 g
F
.51631 .46038 m .51608 .46138 L .51608 .46138 L F
.51646 .45974 m .51631 .46039 L .51631 .46038 L F
.51118 .47386 m .44515 .55232 L p .51046 .47576 L .068 g
F P
0 g
s
.51421 .47381 m .50552 .48895 L p .51418 .47408 L F P
s
.50552 .48895 m .46533 .559 L p .50524 .48963 L F P
s
.51418 .47408 m .50552 .48895 L .50524 .48963 L F
.50878 .53366 m .50883 .52279 L .51125 .50067 L p .01 g
F P
0 g
s
.50878 .53366 m .51125 .50067 L .51222 .49637 L .01 g
F
.51518 .4691 m .51499 .46791 L p .51501 .4692 L .117 g
F P
0 g
s
.51609 .47054 m .51421 .47381 L p .51613 .47069 L F P
s
.51613 .47069 m .51421 .47381 L .51418 .47408 L F
.51673 .46943 m .51609 .47054 L p .51714 .46888 L F P
s
.5116 .4974 m .50936 .51799 L p .51178 .49573 L .007 g
F P
0 g
s
.51646 .45974 m .51656 .45929 L .51646 .45974 L .005 g
F
.51242 .47054 m .44287 .54162 L p .51162 .47268 L .117 g
F P
0 g
s
.51351 .46555 m .44609 .51221 L p .51236 .46805 L .234 g
F P
0 g
s
.51733 .457 m .51745 .45691 L p .51716 .45768 L .005 g
F P
0 g
s
.51727 .45739 m .51734 .45721 L .51727 .45739 L .005 g
F
.51734 .45721 m .51745 .45691 L .51734 .45721 L F
.5152 .46438 m .51558 .46234 L .5148 .46274 L p .263 g
F P
0 g
s
.5148 .46274 m .4513 .49578 L p .51372 .46511 L .263 g
F P
0 g
s
.5152 .46438 m .5148 .46274 L .51372 .46511 L .263 g
F
.51656 .45929 m .51675 .45846 L .51656 .4593 L .005 g
F
.51351 .46756 m .51198 .46888 L p .51317 .4685 L .187 g
F P
0 g
s
.51198 .46888 m .4432 .52789 L p .51081 .4714 L .187 g
F P
0 g
s
.51236 .46805 m .44609 .51221 L .51198 .46888 L .234 g
F
.51662 .45884 m .51659 .45891 L p .51722 .45754 L .005 g
F P
0 g
s
.51719 .45762 m .51722 .45754 L .51659 .45891 L .005 g
F
.51722 .45754 m .51726 .45742 L .51662 .45884 L F
.4513 .49578 m .44609 .51221 L .51351 .46555 L p .263 g
F P
0 g
s
.51351 .46555 m .5152 .46438 L p .51372 .46511 L .263 g
F P
0 g
s
.51372 .46511 m .4513 .49578 L .51351 .46555 L .263 g
F
.51714 .45714 m .51733 .457 L p .517 .4579 L .005 g
F P
0 g
s
.45843 .47979 m .4513 .49578 L .5148 .46274 L p .277 g
F P
0 g
s
.5148 .46274 m .51558 .46234 L p .51489 .46255 L .277 g
F P
0 g
s
.51489 .46255 m .45843 .47979 L .5148 .46274 L .277 g
F
.51745 .45691 m .51675 .45846 L .51665 .45874 L p .005 g
F P
0 g
s
.51726 .45742 m .51745 .45691 L .51665 .45874 L .005 g
F
.53968 .48207 m .53646 .46197 L .51954 .46017 L p .869 g
F P
0 g
s
.50936 .51799 m .50883 .52279 L p .50938 .5178 L .007 g
F P
0 g
s
.53315 .44644 m .53646 .46197 L .57667 .45859 L p .846 g
F P
0 g
s
.57667 .45859 m .56497 .43966 L .53315 .44644 L p .846 g
F P
0 g
s
.56497 .43966 m .57667 .45859 L .61432 .44894 L p .774 g
F P
0 g
s
.61432 .44894 m .59469 .4311 L .56497 .43966 L p .774 g
F P
0 g
s
.59469 .4311 m .61432 .44894 L .62957 .43767 L p .609 g
F P
0 g
s
.65219 .45624 m .62957 .43767 L .61432 .44894 L p .589 g
F P
0 g
s
.54796 .60192 m .54722 .61794 L .61553 .65576 L p .618 g
F P
0 g
s
.6101 .66938 m .61553 .65576 L .54722 .61794 L p .558 g
F P
0 g
s
.61553 .65576 m .6101 .66938 L .67102 .66537 L p .255 g
F P
0 g
s
.67102 .66537 m .68012 .65004 L .61553 .65576 L p .255 g
F P
0 g
s
.67287 .47942 m .69024 .50577 L .65861 .47379 L p .34 g
F P
0 g
s
.69024 .50577 m .70311 .53357 L .6691 .49143 L p .473 g
F P
0 g
s
.6691 .49143 m .65861 .47379 L .69024 .50577 L p .473 g
F P
0 g
s
.68012 .65004 m .67102 .66537 L .6959 .62043 L p .781 g
F P
0 g
s
.6959 .62043 m .70693 .60612 L .68012 .65004 L p .781 g
F P
0 g
s
.65219 .45624 m .67287 .47942 L .6443 .45758 L p .246 g
F P
0 g
s
.65861 .47379 m .6443 .45758 L .67287 .47942 L p .34 g
F P
0 g
s
.70311 .53357 m .71053 .56088 L .67496 .50919 L p .626 g
F P
0 g
s
.67496 .50919 m .6691 .49143 L .70311 .53357 L p .626 g
F P
0 g
s
.2859 .43186 m .2582 .4566 L .29163 .48791 L p .785 g
F P
0 g
s
.29163 .48791 m .31543 .46032 L .2859 .43186 L p .785 g
F P
0 g
s
.27385 .51789 m .29163 .48791 L .2582 .4566 L p .787 g
F P
0 g
s
.31543 .46032 m .29163 .48791 L .36605 .50415 L p .624 g
F P
0 g
s
.29163 .48791 m .27385 .51789 L .35428 .5302 L p .615 g
F P
0 g
s
.35428 .5302 m .36605 .50415 L .29163 .48791 L p .615 g
F P
0 g
s
.31856 .41208 m .2859 .43186 L .31543 .46032 L p .774 g
F P
0 g
s
.31543 .46032 m .3436 .43691 L .31856 .41208 L p .774 g
F P
0 g
s
.3436 .43691 m .31543 .46032 L .38194 .47939 L p .624 g
F P
0 g
s
.36605 .50415 m .38194 .47939 L .31543 .46032 L p .624 g
F P
0 g
s
.38194 .47939 m .36605 .50415 L .4513 .49578 L p .491 g
F P
0 g
s
.36605 .50415 m .35428 .5302 L .44609 .51221 L p .479 g
F P
0 g
s
.44609 .51221 m .4513 .49578 L .36605 .50415 L p .479 g
F P
0 g
s
.52996 .4362 m .53315 .44644 L .56497 .43966 L p .889 g
F P
0 g
s
.56497 .43966 m .55369 .42739 L .52996 .4362 L p .889 g
F P
0 g
s
.55369 .42739 m .56497 .43966 L .59469 .4311 L p .824 g
F P
0 g
s
.62957 .43767 m .65219 .45624 L .62723 .44388 L p .198 g
F P
0 g
s
.6443 .45758 m .62723 .44388 L .65219 .45624 L p .246 g
F P
0 g
s
.38194 .47939 m .40087 .45756 L .3436 .43691 L p .624 g
F P
0 g
s
.40087 .45756 m .38194 .47939 L .45843 .47979 L p .493 g
F P
0 g
s
.4513 .49578 m .45843 .47979 L .38194 .47939 L p .491 g
F P
0 g
s
.34752 .55572 m .35428 .5302 L .27385 .51789 L p .594 g
F P
0 g
s
.35428 .5302 m .34752 .55572 L .4432 .52789 L p .454 g
F P
0 g
s
.4432 .52789 m .44609 .51221 L .35428 .5302 L p .454 g
F P
0 g
s
.59469 .4311 m .5758 .42021 L .55369 .42739 L p .824 g
F P
0 g
s
.5758 .42021 m .59469 .4311 L .60649 .42467 L p .564 g
F P
0 g
s
.62957 .43767 m .60649 .42467 L .59469 .4311 L p .609 g
F P
0 g
s
.71053 .56088 m .71188 .58571 L .6757 .52572 L p .75 g
F P
0 g
s
.6757 .52572 m .67496 .50919 L .71053 .56088 L p .75 g
F P
0 g
s
.2582 .4566 m .23743 .48469 L .27385 .51789 L p .787 g
F P
0 g
s
.26342 .54819 m .27385 .51789 L .23743 .48469 L p .778 g
F P
0 g
s
.27385 .51789 m .26342 .54819 L .34752 .55572 L p .594 g
F P
0 g
s
.35396 .39838 m .31856 .41208 L .3436 .43691 L p .756 g
F P
0 g
s
.3436 .43691 m .37427 .41906 L .35396 .39838 L p .756 g
F P
0 g
s
.37427 .41906 m .3436 .43691 L .40087 .45756 L p .615 g
F P
0 g
s
.54722 .61794 m .54561 .62771 L .6101 .66938 L p .558 g
F P
0 g
s
.60194 .67414 m .6101 .66938 L .54561 .62771 L p .439 g
F P
0 g
s
.6101 .66938 m .60194 .67414 L .6573 .67178 L p .081 g
F P
0 g
s
.6573 .67178 m .67102 .66537 L .6101 .66938 L p .081 g
F P
0 g
s
.40087 .45756 m .42156 .44001 L .37427 .41906 L p .615 g
F P
0 g
s
.42156 .44001 m .40087 .45756 L .46697 .46535 L p .484 g
F P
0 g
s
.45843 .47979 m .46697 .46535 L .40087 .45756 L p .493 g
F P
0 g
s
.53646 .46197 m .53315 .44644 L .5154 .45141 L p .878 g
F P
0 g
s
.34631 .57883 m .34752 .55572 L .26342 .54819 L p .558 g
F P
0 g
s
.34752 .55572 m .34631 .57883 L .44287 .54162 L p .413 g
F P
0 g
s
.44287 .54162 m .4432 .52789 L .34752 .55572 L p .413 g
F P
0 g
s
.23743 .48469 m .22522 .51417 L .26342 .54819 L p .778 g
F P
0 g
s
.26123 .57658 m .26342 .54819 L .22522 .51417 L p .743 g
F P
0 g
s
.26342 .54819 m .26123 .57658 L .34631 .57883 L p .558 g
F P
0 g
s
.60649 .42467 m .62957 .43767 L .6086 .43357 L p .192 g
F P
0 g
s
.62723 .44388 m .6086 .43357 L .62957 .43767 L p .198 g
F P
0 g
s
.3898 .39133 m .35396 .39838 L .37427 .41906 L p .726 g
F P
0 g
s
.37427 .41906 m .40546 .40767 L .3898 .39133 L p .726 g
F P
0 g
s
.40546 .40767 m .37427 .41906 L .42156 .44001 L p .595 g
F P
0 g
s
.71188 .58571 m .70693 .60612 L .67118 .53971 L p .81 g
F P
0 g
s
.67118 .53971 m .6757 .52572 L .71188 .58571 L p .81 g
F P
0 g
s
.67102 .66537 m .6573 .67178 L .67945 .62736 L p .934 g
F P
0 g
s
.67945 .62736 m .6959 .62043 L .67102 .66537 L p .934 g
F P
0 g
s
.42156 .44001 m .44266 .42768 L .40546 .40767 L p .595 g
F P
0 g
s
.44266 .42768 m .42156 .44001 L .47633 .45339 L p .462 g
F P
0 g
s
.46697 .46535 m .47633 .45339 L .42156 .44001 L p .484 g
F P
0 g
s
.35086 .59773 m .34631 .57883 L .26123 .57658 L p .492 g
F P
0 g
s
.34631 .57883 m .35086 .59773 L .44515 .55232 L p .348 g
F P
0 g
s
.44515 .55232 m .44287 .54162 L .34631 .57883 L p .348 g
F P
0 g
s
.5271 .43139 m .52996 .4362 L .55369 .42739 L p .928 g
F P
0 g
s
.55369 .42739 m .54352 .42199 L .5271 .43139 L p .928 g
F P
0 g
s
.54352 .42199 m .55369 .42739 L .5758 .42021 L p .859 g
F P
0 g
s
.5758 .42021 m .55879 .41632 L .54352 .42199 L p .859 g
F P
0 g
s
.55879 .41632 m .5758 .42021 L .58438 .41772 L p .013 g
F P
0 g
s
.60649 .42467 m .58438 .41772 L .5758 .42021 L p .564 g
F P
0 g
s
.34116 .39711 m .31169 .40989 L .2859 .43186 L p .688 g
F P
0 g
s
.31169 .40989 m .28669 .42746 L .28041 .43388 L p F P
s
.2859 .43186 m .31856 .41208 L p .30535 .41963 L .688 g
F P
0 g
s
.2859 .43186 m .31169 .40989 L p .27354 .4432 L F P
s
.2582 .4566 m .27575 .44093 L p .27354 .4432 L F P
s
.27575 .44093 m .2859 .43186 L p .27354 .4432 L F P
s
.28041 .43388 m .2582 .4566 L p .29141 .4276 L F P
s
.29141 .4276 m .31169 .40989 L .28041 .43388 L F
.2582 .4566 m .28669 .42746 L p .2462 .47449 L F P
s
.23743 .48469 m .24319 .4769 L p .2462 .47449 L F P
s
.24319 .4769 m .2582 .4566 L p .2462 .47449 L F P
s
.25728 .472 m .22522 .51417 L p .25656 .46624 L .008 g
F P
0 g
s
.28669 .42746 m .26805 .44866 L .26683 .4501 L p F P
s
.26683 .4501 m .23743 .48469 L p .27005 .44679 L F P
s
.27005 .44679 m .28669 .42746 L .26683 .4501 L F
.26805 .44866 m .26191 .46198 L p .25656 .46624 L .008 g
F P
0 g
s
.26191 .46198 m .25728 .472 L p .25656 .46624 L .008 g
F P
0 g
s
.31856 .41208 m .34116 .39711 L p .30535 .41963 L .688 g
F P
0 g
s
.31856 .41208 m .31856 .41208 L p .30535 .41963 L .688 g
F P
0 g
s
.51679 .44608 m .5154 .45141 L .53315 .44644 L p .874 g
F P
0 g
s
.53315 .44644 m .52996 .4362 L .51679 .44608 L p .874 g
F P
0 g
s
.37301 .38979 m .34116 .39711 L .31856 .41208 L p .643 g
F P
0 g
s
.31856 .41208 m .35396 .39838 L .36749 .39227 L p .643 g
F P
0 g
s
.31856 .41208 m .36749 .39227 L .36365 .39362 L .643 g
F
.42385 .39089 m .3898 .39133 L .40546 .40767 L p .678 g
F P
0 g
s
.40546 .40767 m .43524 .40302 L .42385 .39089 L p .678 g
F P
0 g
s
.43524 .40302 m .40546 .40767 L .44266 .42768 L p .56 g
F P
0 g
s
.43076 .40506 m .41118 .4069 L .3501 .39836 L p .331 g
F P
0 g
s
.3501 .39836 m .34116 .39711 L p .35114 .398 L .331 g
F P
0 g
s
.40513 .3881 m .37301 .38979 L .35396 .39838 L p .53 g
F P
0 g
s
.35396 .39838 m .3898 .39133 L .39671 .38987 L p .53 g
F P
0 g
s
.39671 .38987 m .40513 .3881 L p .38812 .39151 L .53 g
F P
0 g
s
.35396 .39838 m .39671 .38987 L .38812 .39151 L .53 g
F
.35114 .398 m .35328 .39819 L .3501 .39836 L .331 g
F
.35328 .39819 m .43076 .40506 L .35169 .39849 L F
.35328 .39819 m .35169 .39849 L .3501 .39836 L F
.36749 .39227 m .37301 .38979 L p .36365 .39362 L .643 g
F P
0 g
s
.22522 .51417 m .22265 .54285 L .26123 .57658 L p .743 g
F P
0 g
s
.26765 .60085 m .26123 .57658 L .22265 .54285 L p .629 g
F P
0 g
s
.26123 .57658 m .26765 .60085 L .35086 .59773 L p .492 g
F P
0 g
s
.54561 .62771 m .54326 .63036 L .60194 .67414 L p .439 g
F P
0 g
s
.58438 .41772 m .60649 .42467 L .58967 .42713 L p .216 g
F P
0 g
s
.6086 .43357 m .58967 .42713 L .60649 .42467 L p .192 g
F P
0 g
s
.59158 .66942 m .60194 .67414 L .54326 .63036 L p .215 g
F P
0 g
s
.60194 .67414 m .59158 .66942 L .63985 .66855 L p .621 g
F P
0 g
s
.63985 .66855 m .6573 .67178 L .60194 .67414 L p .621 g
F P
0 g
s
.70693 .60612 m .6959 .62043 L .66168 .55001 L p .82 g
F P
0 g
s
.66168 .55001 m .67118 .53971 L .70693 .60612 L p .82 g
F P
0 g
s
.44266 .42768 m .46284 .42101 L .43524 .40302 L p .56 g
F P
0 g
s
.46284 .42101 m .44266 .42768 L .48586 .44459 L p .423 g
F P
0 g
s
.47633 .45339 m .48586 .44459 L .44266 .42768 L p .462 g
F P
0 g
s
.36098 .61085 m .35086 .59773 L .26765 .60085 L p .368 g
F P
0 g
s
.35086 .59773 m .36098 .61085 L .4499 .5591 L p .243 g
F P
0 g
s
.4499 .5591 m .44515 .55232 L .35086 .59773 L p .243 g
F P
0 g
s
.22522 .51417 m .23743 .48469 L .26805 .44866 L p .008 g
F P
0 g
s
.6573 .67178 m .63985 .66855 L .65874 .62626 L p .877 g
F P
0 g
s
.65874 .62626 m .67945 .62736 L .6573 .67178 L p .877 g
F P
0 g
s
.46697 .46535 m .45843 .47979 L .51611 .46031 L p .276 g
F P
0 g
s
.51558 .46234 m .51611 .46031 L .45843 .47979 L p .277 g
F P
0 g
s
.45417 .39639 m .42385 .39089 L .43524 .40302 L p .596 g
F P
0 g
s
.43524 .40302 m .46187 .40472 L .45417 .39639 L p .596 g
F P
0 g
s
.47923 .40655 m .45417 .39639 L .46187 .40472 L p .453 g
F P
0 g
s
.46187 .40472 m .48397 .41173 L .47923 .40655 L p .453 g
F P
0 g
s
.45417 .39639 m .47493 .40481 L p .46549 .40187 L .156 g
F P
0 g
s
.46236 .39977 m .45417 .39639 L p .46549 .40187 L .156 g
F P
0 g
s
.49812 .41956 m .47923 .40655 L .48397 .41173 L p .255 g
F P
0 g
s
.47923 .40655 m .48444 .41102 L p .46549 .40187 L .156 g
F P
0 g
s
.47493 .40481 m .47923 .40655 L p .46549 .40187 L .156 g
F P
0 g
s
.48444 .41102 m .46236 .39977 L p .46549 .40187 L .156 g
F P
0 g
s
.42385 .39089 m .45417 .39639 L .45433 .39646 L p .092 g
F P
0 g
s
.45433 .39646 m .46236 .39977 L p .45039 .39701 L .092 g
F P
0 g
s
.43903 .39439 m .45385 .3964 L .45039 .39701 L .092 g
F
.45385 .3964 m .45433 .39646 L .45039 .39701 L F
.46187 .40472 m .43524 .40302 L .46284 .42101 L p .502 g
F P
0 g
s
.46284 .42101 m .48089 .41985 L .46187 .40472 L p .502 g
F P
0 g
s
.48397 .41173 m .46187 .40472 L .48089 .41985 L p .411 g
F P
0 g
s
.48089 .41985 m .49586 .42348 L .48397 .41173 L p .411 g
F P
0 g
s
.48547 .41833 m .46908 .41131 L .44594 .39781 L p .403 g
F P
0 g
s
.42385 .39089 m .45433 .39646 L .43903 .39439 L .092 g
F
.46236 .39977 m .43549 .39171 L .43223 .39148 L p F P
0 g
s
.43223 .39148 m .42385 .39089 L p .44871 .39663 L .092 g
F P
0 g
s
.44871 .39663 m .46236 .39977 L .43223 .39148 L .092 g
F
.44594 .39781 m .43549 .39171 L p .44669 .39767 L .403 g
F P
0 g
s
.44669 .39767 m .48547 .41833 L .44594 .39781 L .403 g
F
.43549 .39171 m .40513 .3881 L .4002 .38914 L p .02 g
F P
0 g
s
.46908 .41131 m .45049 .40666 L .41554 .39236 L p .371 g
F P
0 g
s
.42385 .39089 m .42818 .39119 L p .3898 .39133 L .02 g
F P
0 g
s
.3898 .39133 m .42818 .39119 L .42368 .39161 L .02 g
F
.4002 .38914 m .3898 .39133 L p .41813 .39156 L F P
0 g
s
.42818 .39119 m .43549 .39171 L p .42368 .39161 L .02 g
F P
0 g
s
.41813 .39156 m .43549 .39171 L .4002 .38914 L .02 g
F
.41554 .39236 m .40513 .3881 L p .4166 .39226 L .371 g
F P
0 g
s
.4166 .39226 m .46908 .41131 L .41554 .39236 L .371 g
F
.22265 .54285 m .23019 .56847 L .26765 .60085 L p .629 g
F P
0 g
s
.28245 .619 m .26765 .60085 L .23019 .56847 L p .3 g
F P
0 g
s
.26765 .60085 m .28245 .619 L .36098 .61085 L p .368 g
F P
0 g
s
.52996 .4362 m .5271 .43139 L .51803 .44363 L p .848 g
F P
0 g
s
.51906 .44438 m .52355 .441 L p .52028 .44256 L .848 g
F P
0 g
s
.52333 .44061 m .51803 .44363 L .52355 .441 L .848 g
F
.52513 .43921 m .51803 .44363 L .52643 .43885 L F
.52418 .43748 m .52168 .44219 L p .52525 .43517 L 0 g
F P
s
.52471 .43158 m .5271 .43139 L .53339 .42779 L p .929 g
F P
0 g
s
.53339 .42779 m .54352 .42199 L p .53177 .42798 L .929 g
F P
0 g
s
.53177 .42798 m .52471 .43158 L .53339 .42779 L .929 g
F
.52028 .44256 m .51803 .44363 L .51906 .44438 L .848 g
F
.52168 .44219 m .52031 .4493 L p .52178 .44155 L 0 g
F P
s
.52238 .43775 m .52445 .43496 L p .52226 .43854 L .792 g
F P
0 g
s
.52354 .43687 m .51904 .44378 L .52445 .43496 L .792 g
F
.51904 .44378 m .51803 .44363 L .52238 .43775 L p F P
0 g
s
.52226 .43854 m .51904 .44378 L .52238 .43775 L .792 g
F
.52445 .43496 m .5271 .43139 L p .52354 .43687 L F P
0 g
s
.5271 .43139 m .52471 .43158 L .52058 .44047 L p .792 g
F P
0 g
s
.5271 .43139 m .52058 .44047 L .52133 .44026 L .792 g
F
.52058 .44047 m .51904 .44378 L p .52133 .44026 L F P
0 g
s
.5198 .44595 m .51904 .44378 L .52144 .43861 L p .707 g
F P
0 g
s
.52144 .43861 m .52471 .43158 L p .52298 .43665 L .707 g
F P
0 g
s
.52298 .43665 m .5198 .44595 L .52144 .43861 L .707 g
F
.52289 .4357 m .52208 .43841 L p .52293 .43618 L F P
0 g
s
.52471 .43158 m .52289 .4357 L p .52293 .43618 L .707 g
F P
0 g
s
.52471 .43158 m .52208 .43841 L .52226 .43874 L .707 g
F
.54352 .42199 m .53501 .42288 L .52471 .43158 L p .929 g
F P
0 g
s
.52289 .4357 m .52471 .43158 L .52799 .4288 L p .831 g
F P
0 g
s
.52799 .4288 m .53154 .42581 L p .52691 .43144 L .831 g
F P
0 g
s
.5327 .42532 m .52691 .43144 L .53154 .42581 L .831 g
F
.53154 .42581 m .53501 .42288 L p .5327 .42532 L F P
0 g
s
.53501 .42288 m .52899 .4283 L p .53 .42818 L .831 g
F P
0 g
s
.52899 .4283 m .52853 .42871 L p .53 .42818 L .831 g
F P
0 g
s
.53376 .42599 m .52853 .42871 L p .53664 .42367 L F P
s
.54352 .42199 m .55879 .41632 L p .53755 .42218 L .792 g
F P
0 g
s
.53501 .42288 m .53629 .42274 L p .53755 .42218 L .792 g
F P
0 g
s
.53629 .42274 m .54352 .42199 L p .53755 .42218 L .792 g
F P
0 g
s
.55879 .41632 m .54457 .41874 L .53501 .42288 L p .792 g
F P
0 g
s
.54293 .41976 m .53817 .42303 L .53664 .42367 L F
.54457 .41874 m .54329 .4196 L p .54293 .41976 L F P
s
.54329 .4196 m .53817 .42303 L p .54293 .41976 L F P
s
.53817 .42303 m .53376 .42599 L p .53664 .42367 L F P
s
.52691 .43144 m .52289 .4357 L .52799 .4288 L .831 g
F
.52598 .43373 m .52557 .43469 L .52525 .43517 L 0 g
F
.52853 .42871 m .52289 .4357 L p .52775 .43056 L .831 g
F P
0 g
s
.52853 .42871 m .52567 .43448 L p .52598 .43373 L F P
s
.52567 .43448 m .52557 .43469 L p .52598 .43373 L F P
s
.53 .42818 m .52946 .42838 L .52775 .43056 L .831 g
F
.52946 .42838 m .52853 .42871 L .52775 .43056 L F
.52289 .4357 m .52169 .44209 L p .52178 .44155 L 0 g
F P
s
.52169 .44209 m .52168 .44219 L p .52178 .44155 L F P
s
.52557 .43469 m .52418 .43748 L p .52525 .43517 L F P
s
.51803 .44363 m .51679 .44608 L .51906 .44438 L p .848 g
F P
0 g
s
.52355 .441 m .52643 .43885 L p .52333 .44061 L .848 g
F P
0 g
s
.52208 .43841 m .5198 .44595 L p .52226 .43874 L .707 g
F P
0 g
s
.52643 .43885 m .52996 .4362 L p .52513 .43921 L .848 g
F P
0 g
s
.58438 .41772 m .56458 .41667 L .55879 .41632 L p .013 g
F P
0 g
s
.56458 .41667 m .58438 .41772 L .57166 .42471 L p .26 g
F P
0 g
s
.58967 .42713 m .57166 .42471 L .58438 .41772 L p .216 g
F P
0 g
s
.54326 .63036 m .54031 .62553 L .59158 .66942 L p .215 g
F P
0 g
s
.22265 .54285 m .22522 .51417 L .25728 .472 L p .048 g
F P
0 g
s
.51739 .45695 m .51393 .45958 L p .51802 .45587 L .006 g
F P
0 g
s
.51802 .45587 m .51816 .45574 L .51739 .45695 L .006 g
F
.51393 .45958 m .5154 .45141 L .51816 .45574 L p F P
0 g
s
.51745 .45691 m .47633 .45339 L p .51654 .45843 L .262 g
F P
0 g
s
.51675 .45846 m .51745 .45691 L p .51654 .45843 L .262 g
F P
0 g
s
.51745 .45691 m .51739 .45695 L p .51799 .45602 L .006 g
F P
0 g
s
.51816 .45574 m .51745 .45691 L p .51799 .45602 L .006 g
F P
0 g
s
.47633 .45339 m .46697 .46535 L .5165 .4585 L p .262 g
F P
0 g
s
.5165 .4585 m .51675 .45846 L p .51654 .45843 L .262 g
F P
0 g
s
.51654 .45843 m .47633 .45339 L .5165 .4585 L .262 g
F
.51611 .46031 m .51675 .45846 L .46697 .46535 L p .276 g
F P
0 g
s
.54457 .41874 m .55879 .41632 L .56458 .41667 L p .088 g
F P
0 g
s
.6959 .62043 m .67945 .62736 L .64785 .55575 L p .804 g
F P
0 g
s
.64785 .55575 m .66168 .55001 L .6959 .62043 L p .804 g
F P
0 g
s
.48089 .41985 m .46284 .42101 L .49495 .43928 L p .363 g
F P
0 g
s
.48586 .44459 m .49495 .43928 L .46284 .42101 L p .423 g
F P
0 g
s
.62723 .44388 m .6443 .45758 L .56191 .45199 L p .527 g
F P
0 g
s
.6443 .45758 m .65861 .47379 L .5667 .45625 L p .564 g
F P
0 g
s
.57015 .46107 m .5667 .45625 L .56767 .45643 L p .6 g
F P
0 g
s
.56767 .45643 m .65861 .47379 L p .57015 .46107 L .6 g
F P
0 g
s
.57198 .4661 m .57015 .46107 L .57115 .46138 L p .633 g
F P
0 g
s
.57115 .46138 m .6691 .49143 L p .57198 .4661 L .633 g
F P
0 g
s
.57205 .47093 m .57198 .4661 L .57302 .46653 L p .659 g
F P
0 g
s
.57302 .46653 m .67496 .50919 L p .57205 .47093 L .659 g
F P
0 g
s
.57034 .47517 m .57205 .47093 L .57313 .4715 L p .676 g
F P
0 g
s
.57313 .4715 m .6757 .52572 L p .57034 .47517 L .676 g
F P
0 g
s
.56698 .47848 m .57034 .47517 L .57148 .4759 L p .684 g
F P
0 g
s
.57148 .4759 m .67118 .53971 L p .56698 .47848 L .684 g
F P
0 g
s
.56223 .4806 m .56698 .47848 L .56818 .47939 L p .686 g
F P
0 g
s
.56818 .47939 m .66168 .55001 L p .56223 .4806 L .686 g
F P
0 g
s
.55647 .48138 m .56223 .4806 L .5635 .48171 L p .681 g
F P
0 g
s
.5635 .48171 m .64785 .55575 L p .55647 .48138 L .681 g
F P
0 g
s
.55014 .48079 m .55647 .48138 L .55779 .48271 L p .672 g
F P
0 g
s
.55779 .48271 m .63073 .55648 L p .55014 .48079 L .672 g
F P
0 g
s
.5437 .47895 m .55014 .48079 L .5515 .48237 L p .66 g
F P
0 g
s
.5515 .48237 m .61162 .55221 L p .5437 .47895 L .66 g
F P
0 g
s
.53759 .47608 m .5437 .47895 L .59196 .54343 L p .644 g
F P
0 g
s
.5437 .47895 m .53759 .47608 L .5383 .47659 L p .365 g
F P
0 g
s
.61162 .55221 m .59196 .54343 L .5437 .47895 L p .66 g
F P
0 g
s
.5383 .47659 m .55042 .48512 L p .5437 .47895 L .365 g
F P
0 g
s
.55014 .48079 m .5437 .47895 L .54444 .47936 L p .365 g
F P
0 g
s
.63073 .55648 m .61162 .55221 L .55014 .48079 L p .672 g
F P
0 g
s
.54444 .47936 m .56117 .48856 L p .55014 .48079 L .365 g
F P
0 g
s
.55647 .48138 m .55014 .48079 L .5509 .48111 L p .377 g
F P
0 g
s
.64785 .55575 m .63073 .55648 L .55647 .48138 L p .681 g
F P
0 g
s
.5509 .48111 m .57253 .49021 L p .55647 .48138 L .377 g
F P
0 g
s
.56223 .4806 m .55647 .48138 L .55724 .48162 L p .396 g
F P
0 g
s
.67118 .53971 m .66168 .55001 L .56698 .47848 L p .684 g
F P
0 g
s
.66168 .55001 m .64785 .55575 L .56223 .4806 L p .686 g
F P
0 g
s
.55724 .48162 m .58368 .4898 L p .56223 .4806 L .396 g
F P
0 g
s
.56698 .47848 m .56223 .4806 L .563 .48076 L p .42 g
F P
0 g
s
.6757 .52572 m .67118 .53971 L .57034 .47517 L p .676 g
F P
0 g
s
.563 .48076 m .59381 .48724 L p .56698 .47848 L .42 g
F P
0 g
s
.57034 .47517 m .56698 .47848 L .56775 .47858 L p .446 g
F P
0 g
s
.67496 .50919 m .6757 .52572 L .57205 .47093 L p .659 g
F P
0 g
s
.56775 .47858 m .60213 .48267 L p .57034 .47517 L .446 g
F P
0 g
s
.57205 .47093 m .57034 .47517 L .57111 .4752 L p .472 g
F P
0 g
s
.6691 .49143 m .67496 .50919 L .57198 .4661 L p .633 g
F P
0 g
s
.57111 .4752 m .60798 .47642 L p .57205 .47093 L .472 g
F P
0 g
s
.57198 .4661 m .57205 .47093 L .57283 .47089 L p .499 g
F P
0 g
s
.65861 .47379 m .6691 .49143 L .57015 .46107 L p .6 g
F P
0 g
s
.57283 .47089 m .61091 .469 L p .57198 .4661 L .499 g
F P
0 g
s
.57015 .46107 m .57198 .4661 L .57277 .466 L p .524 g
F P
0 g
s
.57277 .466 m .6107 .461 L p .57015 .46107 L .524 g
F P
0 g
s
.5667 .45625 m .57015 .46107 L .57095 .4609 L p .548 g
F P
0 g
s
.56287 .45206 m .6443 .45758 L p .5667 .45625 L .564 g
F P
0 g
s
.57095 .4609 m .60739 .45308 L p .5667 .45625 L .548 g
F P
0 g
s
.5667 .45625 m .56191 .45199 L .56287 .45206 L p .564 g
F P
0 g
s
.37601 .61704 m .36098 .61085 L .28245 .619 L p .138 g
F P
0 g
s
.36098 .61085 m .37601 .61704 L .45679 .5614 L p .107 g
F P
0 g
s
.45679 .5614 m .4499 .5591 L .36098 .61085 L p .107 g
F P
0 g
s
.6086 .43357 m .62723 .44388 L .55616 .44862 L p .496 g
F P
0 g
s
.56191 .45199 m .5667 .45625 L .56753 .456 L p .571 g
F P
0 g
s
.5571 .44856 m .55998 .44837 L p .56191 .45199 L .527 g
F P
0 g
s
.55998 .44837 m .62723 .44388 L p .56338 .45181 L .527 g
F P
0 g
s
.56191 .45199 m .55998 .44837 L .56338 .45181 L .527 g
F
.60739 .45308 m .60127 .44586 L .56753 .456 L p .548 g
F P
0 g
s
.56753 .456 m .5667 .45625 L p .5677 .45617 L .548 g
F P
0 g
s
.60739 .45308 m .56753 .456 L .5677 .45617 L .548 g
F
.56753 .456 m .60127 .44586 L p .56191 .45199 L .571 g
F P
0 g
s
.56191 .45199 m .55616 .44862 L .5571 .44856 L p .527 g
F P
0 g
s
.57974 .65545 m .59158 .66942 L .54031 .62553 L p .246 g
F P
0 g
s
.59158 .66942 m .57974 .65545 L .61992 .65579 L p .729 g
F P
0 g
s
.61992 .65579 m .63985 .66855 L .59158 .66942 L p .729 g
F P
0 g
s
.48586 .44459 m .47633 .45339 L .51745 .45691 L p .233 g
F P
0 g
s
.58967 .42713 m .6086 .43357 L .54988 .44633 L p .472 g
F P
0 g
s
.55616 .44862 m .54988 .44633 L .6086 .43357 L p .496 g
F P
0 g
s
.44515 .55232 m .4499 .5591 L .51556 .46978 L p .068 g
F P
0 g
s
.25728 .472 m .25537 .49568 L .22265 .54285 L p .048 g
F P
0 g
s
.23019 .56847 m .22265 .54285 L .25537 .49568 L p .208 g
F P
0 g
s
.56458 .41667 m .54812 .42071 L .54457 .41874 L p .088 g
F P
0 g
s
.54812 .42071 m .56458 .41667 L .55564 .42601 L p .317 g
F P
0 g
s
.57166 .42471 m .55564 .42601 L .56458 .41667 L p .26 g
F P
0 g
s
.23019 .56847 m .24749 .58889 L .28245 .619 L p .3 g
F P
0 g
s
.28245 .619 m .3047 .62943 L .37601 .61704 L p .138 g
F P
0 g
s
.3047 .62943 m .28245 .619 L .24749 .58889 L p .322 g
F P
0 g
s
.63985 .66855 m .61992 .65579 L .63529 .61719 L p .808 g
F P
0 g
s
.63529 .61719 m .65874 .62626 L .63985 .66855 L p .808 g
F P
0 g
s
.49495 .43928 m .48586 .44459 L .51816 .45574 L p .185 g
F P
0 g
s
.51745 .45691 m .51816 .45574 L .48586 .44459 L p .233 g
F P
0 g
s
.34116 .39711 m .37301 .38979 L .43076 .40506 L p .331 g
F P
0 g
s
.31169 .40989 m .34116 .39711 L .41118 .4069 L p .328 g
F P
0 g
s
.51091 .54736 m .51221 .54434 L .54031 .62553 L p .475 g
F P
0 g
s
.54031 .62553 m .53698 .61355 L .57974 .65545 L p .246 g
F P
0 g
s
.45049 .40666 m .43076 .40506 L .37301 .38979 L p .346 g
F P
0 g
s
.37301 .38979 m .40513 .3881 L .45049 .40666 L p .346 g
F P
0 g
s
.49586 .42348 m .48089 .41985 L .50305 .43744 L p .273 g
F P
0 g
s
.49495 .43928 m .50305 .43744 L .48089 .41985 L p .363 g
F P
0 g
s
.41118 .4069 m .39311 .41227 L .31169 .40989 L p .328 g
F P
0 g
s
.28669 .42746 m .31169 .40989 L .39311 .41227 L p .337 g
F P
0 g
s
.67945 .62736 m .65874 .62626 L .63073 .55648 L p .778 g
F P
0 g
s
.63073 .55648 m .64785 .55575 L .67945 .62736 L p .778 g
F P
0 g
s
.57166 .42471 m .58967 .42713 L .54351 .44523 L p .458 g
F P
0 g
s
.54988 .44633 m .54351 .44523 L .58967 .42713 L p .472 g
F P
0 g
s
.52853 .42871 m .53501 .42288 L .54457 .41874 L p F P
s
.53376 .42599 m .54457 .41874 L .54812 .42071 L p .137 g
F P
0 g
s
.39492 .61571 m .37601 .61704 L .3047 .62943 L p .015 g
F P
0 g
s
.37601 .61704 m .39492 .61571 L .46533 .559 L p .011 g
F P
0 g
s
.46533 .559 m .45679 .5614 L .37601 .61704 L p .011 g
F P
0 g
s
.40513 .3881 m .43549 .39171 L .46908 .41131 L p .371 g
F P
0 g
s
.4499 .5591 m .45679 .5614 L .51609 .46989 L p .023 g
F P
0 g
s
.39311 .41227 m .37788 .4209 L .28669 .42746 L p .337 g
F P
0 g
s
.26805 .44866 m .28669 .42746 L .37788 .4209 L p .357 g
F P
0 g
s
.51884 .455 m .51816 .45574 L .5154 .45141 L p .008 g
F P
0 g
s
.51679 .44608 m .51803 .44363 L .51929 .45357 L p .01 g
F P
0 g
s
.51991 .45477 m .51943 .4547 L .51941 .45453 L p .012 g
F P
0 g
s
.52051 .45565 m .52027 .45513 L .52026 .45485 L p .017 g
F P
0 g
s
.52026 .45485 m .5198 .44595 L p .52051 .45565 L .017 g
F P
0 g
s
.52064 .45618 m .52058 .45595 L p .52031 .4493 L .018 g
F P
0 g
s
.52058 .45595 m .52051 .45565 L .5205 .45537 L p .018 g
F P
0 g
s
.5205 .45537 m .52031 .4493 L p .52058 .45595 L .018 g
F P
0 g
s
.51904 .44378 m .51968 .44558 L p .52025 .45474 L .015 g
F P
0 g
s
.51968 .44558 m .5198 .44595 L .51998 .44947 L p .015 g
F P
0 g
s
.51998 .44947 m .52025 .45474 L p .51968 .44558 L .015 g
F P
0 g
s
.51904 .44378 m .52025 .45474 L .52025 .45475 L .015 g
F
.51904 .44378 m .52026 .45485 L .52025 .45488 L F
.52025 .45474 m .52026 .45485 L p .52025 .45475 L F P
0 g
s
.51935 .4541 m .51803 .44363 L p .51988 .45458 L .012 g
F P
0 g
s
.52027 .45513 m .51991 .45477 L .51904 .44378 L p .015 g
F P
0 g
s
.52033 .45499 m .52006 .45404 L p .52021 .45488 L .043 g
F P
0 g
s
.51803 .44363 m .51904 .44378 L .51987 .45429 L p .012 g
F P
0 g
s
.51987 .45429 m .51989 .45454 L p .51986 .45434 L .012 g
F P
0 g
s
.51803 .44363 m .51987 .45429 L .51986 .45434 L .012 g
F
.51803 .44363 m .51989 .45454 L .51988 .45458 L F
.51807 .44692 m .51982 .45152 L p .51887 .448 L .048 g
F P
0 g
s
.51982 .45152 m .51989 .45172 L .51983 .45149 L p .048 g
F P
0 g
s
.51983 .45149 m .51887 .448 L p .51982 .45152 L .048 g
F P
0 g
s
.5198 .44595 m .52031 .4493 L .52051 .45557 L p .017 g
F P
0 g
s
.52051 .45557 m .52051 .45565 L p .52051 .45558 L .017 g
F P
0 g
s
.5198 .44595 m .52051 .45557 L .52051 .45558 L .017 g
F
.51989 .45172 m .52057 .45521 L p .52046 .45434 L .043 g
F P
0 g
s
.52057 .45521 m .52059 .45535 L .52057 .45517 L p .043 g
F P
0 g
s
.52057 .45517 m .52046 .45434 L p .52057 .45521 L .043 g
F P
0 g
s
.51989 .45172 m .51807 .44692 L .52051 .45565 L p .034 g
F P
0 g
s
.52046 .45434 m .51887 .448 L .51989 .45172 L p .043 g
F P
0 g
s
.52019 .45314 m .52043 .45419 L .52035 .45375 L .045 g
F
.51887 .448 m .52018 .4532 L p .52019 .45314 L F P
0 g
s
.52018 .4532 m .52043 .45419 L p .52019 .45314 L .045 g
F P
0 g
s
.52006 .45404 m .51807 .44692 L p .52002 .45417 L .043 g
F P
0 g
s
.52021 .45488 m .52006 .45404 L .52002 .45417 L .043 g
F
.51989 .45454 m .51991 .45477 L p .51988 .45458 L .012 g
F P
0 g
s
.51911 .45086 m .51804 .44709 L p .52033 .4547 L .524 g
F P
0 g
s
.51474 .44217 m .51807 .44692 L .51472 .43942 L p .061 g
F P
0 g
s
.51679 .44608 m .51929 .45357 L .5192 .45388 L .01 g
F
.51679 .44608 m .51941 .45453 L .51939 .45459 L F
.5154 .45141 m .51679 .44608 L .51875 .45465 L p .008 g
F P
0 g
s
.51875 .45465 m .51881 .45491 L p .51872 .45481 L .008 g
F P
0 g
s
.5154 .45141 m .51875 .45465 L .51872 .45481 L .008 g
F
.51963 .45362 m .51918 .45258 L p .51956 .45391 L .052 g
F P
0 g
s
.51943 .4547 m .51884 .455 L .51679 .44608 L p .01 g
F P
0 g
s
.51918 .45258 m .51474 .44217 L p .51913 .45286 L .052 g
F P
0 g
s
.51956 .45391 m .51918 .45258 L .51913 .45286 L .052 g
F
.51807 .44692 m .51474 .44217 L .52027 .45513 L p .043 g
F P
0 g
s
.50068 .42236 m .51184 .43447 L .51071 .43324 L p .055 g
F P
0 g
s
.51184 .43447 m .50068 .42236 L .50717 .43058 L p .121 g
F P
0 g
s
.50717 .43058 m .51472 .43942 L .51184 .43447 L p .121 g
F P
0 g
s
.51472 .43942 m .50717 .43058 L .51474 .44217 L p .061 g
F P
0 g
s
.51804 .44709 m .51201 .43645 L .51071 .43324 L p .516 g
F P
0 g
s
.51472 .43942 m .51887 .448 L .51798 .4457 L p .054 g
F P
0 g
s
.51472 .43942 m .51807 .44692 L p .51885 .44798 L .048 g
F P
0 g
s
.51887 .448 m .51883 .44792 L p .51885 .44798 L .048 g
F P
0 g
s
.51883 .44792 m .51472 .43942 L p .51885 .44798 L .048 g
F P
0 g
s
.52022 .45341 m .51989 .45172 L .52012 .45307 L p .031 g
F P
0 g
s
.52012 .45307 m .52064 .45618 L p .52022 .45341 L .031 g
F P
0 g
s
.52059 .45535 m .52022 .45341 L p .52064 .45618 L .031 g
F P
0 g
s
.52031 .45371 m .51764 .44532 L .51793 .44563 L p .608 g
F P
0 g
s
.51793 .44563 m .51798 .4457 L p .51807 .446 L .608 g
F P
0 g
s
.51807 .446 m .52031 .45371 L .51793 .44563 L .608 g
F
.51071 .43324 m .51764 .44532 L .51804 .44709 L p .516 g
F P
0 g
s
.51201 .43645 m .51804 .44709 L .51911 .45086 L p .521 g
F P
0 g
s
.51798 .4457 m .52035 .45375 L .52031 .45371 L p .608 g
F P
0 g
s
.52035 .45375 m .51798 .4457 L .51887 .448 L p .045 g
F P
0 g
s
.52036 .45423 m .51804 .44709 L .51764 .44532 L p .538 g
F P
0 g
s
.52037 .45459 m .51989 .45172 L p .52035 .45464 L .034 g
F P
0 g
s
.51764 .44532 m .52031 .45371 L .52036 .45423 L p .538 g
F P
0 g
s
.52051 .45565 m .52037 .45459 L .52035 .45464 L .034 g
F
.51804 .44709 m .52036 .45423 L .52049 .45524 L p .524 g
F P
0 g
s
.51958 .45178 m .51881 .44675 L p .51714 .4458 L .577 g
F P
0 g
s
.51704 .44573 m .51761 .4471 L p .51714 .4458 L .577 g
F P
0 g
s
.51761 .4471 m .51958 .45178 L p .51714 .4458 L .577 g
F P
0 g
s
.52033 .4547 m .52042 .455 L .52029 .45459 L .524 g
F
.52033 .4547 m .52029 .45459 L .51911 .45086 L F
.52059 .45591 m .52037 .45459 L p .52051 .45565 L .034 g
F P
0 g
s
.52049 .45524 m .52042 .45501 L p .52042 .455 L .524 g
F P
0 g
s
.52042 .45501 m .51911 .45086 L p .52042 .455 L .524 g
F P
0 g
s
.52043 .45419 m .52046 .45434 L .52043 .45417 L p .045 g
F P
0 g
s
.52043 .45417 m .52035 .45375 L p .52043 .45419 L .045 g
F P
0 g
s
.52026 .45485 m .52027 .45513 L p .52025 .45488 L .015 g
F P
0 g
s
.52035 .45375 m .5207 .45676 L .52031 .45371 L closepath p .702 g
F P
0 g
s
.52046 .45434 m .5207 .45676 L .52035 .45375 L closepath p .669 g
F P
0 g
s
.52055 .45552 m .52046 .4543 L p .52032 .45391 L .578 g
F P
0 g
s
.52031 .45371 m .5207 .45676 L .52036 .45423 L closepath p .677 g
F P
0 g
s
.52049 .45524 m .52036 .45423 L .52063 .4563 L p .013 g
F P
0 g
s
.51958 .45178 m .52051 .45535 L p .52032 .45391 L .578 g
F P
0 g
s
.52051 .45535 m .52055 .45552 L p .52032 .45391 L .578 g
F P
0 g
s
.52063 .4563 m .52066 .45647 L p .52063 .45624 L .013 g
F P
0 g
s
.52051 .45565 m .52064 .45618 L .52059 .45591 L p .034 g
F P
0 g
s
.52063 .4563 m .52063 .45624 L .52049 .45524 L .013 g
F
.52066 .45647 m .5207 .45676 L .52066 .45649 L p F P
0 g
s
.52066 .45649 m .52049 .45524 L p .52066 .45647 L .013 g
F P
0 g
s
.52027 .45513 m .52051 .45565 L .52033 .45499 L p .043 g
F P
0 g
s
.52027 .45513 m .52033 .45499 L .52021 .45488 L .043 g
F
.51941 .45453 m .51935 .4541 L p .51991 .45477 L .012 g
F P
0 g
s
.51988 .45458 m .51991 .45477 L .51935 .4541 L .012 g
F
.51991 .45477 m .52027 .45513 L .52017 .45489 L p .052 g
F P
0 g
s
.52017 .45489 m .51963 .45362 L p .51991 .45477 L .052 g
F P
0 g
s
.51929 .45357 m .51941 .45453 L p .5192 .45388 L .01 g
F P
0 g
s
.51941 .45453 m .51943 .4547 L p .51939 .45459 L .01 g
F P
0 g
s
.51991 .45477 m .51963 .45362 L .51956 .45391 L .052 g
F
.51943 .4547 m .51991 .45477 L .51908 .45345 L p .06 g
F P
0 g
s
.51908 .45345 m .50972 .43866 L p .51899 .45397 L .06 g
F P
0 g
s
.51943 .4547 m .51908 .45345 L .51899 .45397 L .06 g
F
.51881 .45491 m .51884 .455 L p .5188 .45497 L .008 g
F P
0 g
s
.5154 .45141 m .51881 .45491 L .5188 .45497 L .008 g
F
.50305 .43744 m .49495 .43928 L .51884 .455 L p .118 g
F P
0 g
s
.51816 .45574 m .51884 .455 L .49495 .43928 L p .185 g
F P
0 g
s
.43549 .39171 m .46236 .39977 L .48547 .41833 L p .403 g
F P
0 g
s
.25537 .49568 m .26265 .51779 L .23019 .56847 L p .208 g
F P
0 g
s
.24749 .58889 m .23019 .56847 L .26265 .51779 L p .365 g
F P
0 g
s
.55564 .42601 m .57166 .42471 L .53748 .44529 L p .454 g
F P
0 g
s
.54351 .44523 m .53748 .44529 L .57166 .42471 L p .458 g
F P
0 g
s
.56732 .6334 m .57974 .65545 L .53698 .61355 L p .406 g
F P
0 g
s
.57974 .65545 m .56732 .6334 L .59902 .6346 L p .705 g
F P
0 g
s
.59902 .6346 m .61992 .65579 L .57974 .65545 L p .705 g
F P
0 g
s
.48397 .41173 m .50068 .42236 L .49812 .41956 L p .255 g
F P
0 g
s
.50068 .42236 m .48397 .41173 L .49586 .42348 L p .272 g
F P
0 g
s
.54812 .42071 m .53566 .42846 L .53376 .42599 L p .137 g
F P
0 g
s
.53566 .42846 m .54812 .42071 L .54244 .43035 L p .373 g
F P
0 g
s
.55564 .42601 m .54244 .43035 L .54812 .42071 L p .317 g
F P
0 g
s
.37788 .4209 m .36669 .43224 L .26805 .44866 L p .357 g
F P
0 g
s
.25728 .472 m .26805 .44866 L .36669 .43224 L p .383 g
F P
0 g
s
.24749 .58889 m .27345 .60236 L .3047 .62943 L p .322 g
F P
0 g
s
.3047 .62943 m .33285 .63118 L .39492 .61571 L p .015 g
F P
0 g
s
.33285 .63118 m .3047 .62943 L .27345 .60236 L p .601 g
F P
0 g
s
.45679 .5614 m .46533 .559 L .51673 .46943 L p .005 g
F P
0 g
s
.49586 .42348 m .50717 .43058 L .50068 .42236 L p .272 g
F P
0 g
s
.50101 .42384 m .48444 .41102 L .47923 .40655 L p .299 g
F P
0 g
s
.47923 .40655 m .49812 .41956 L .50101 .42384 L p .299 g
F P
0 g
s
.50717 .43058 m .49586 .42348 L .50972 .43866 L p .151 g
F P
0 g
s
.50305 .43744 m .50972 .43866 L .49586 .42348 L p .273 g
F P
0 g
s
.51221 .54434 m .51367 .53768 L .53698 .61355 L p .038 g
F P
0 g
s
.53351 .59544 m .53698 .61355 L .51367 .53768 L p .055 g
F P
0 g
s
.53698 .61355 m .53351 .59544 L .56732 .6334 L p .406 g
F P
0 g
s
.61992 .65579 m .59902 .6346 L .61088 .60101 L p .751 g
F P
0 g
s
.61088 .60101 m .63529 .61719 L .61992 .65579 L p .751 g
F P
0 g
s
.49887 .42683 m .48547 .41833 L .46236 .39977 L p .439 g
F P
0 g
s
.46236 .39977 m .48444 .41102 L .49887 .42683 L p .439 g
F P
0 g
s
.65874 .62626 m .63529 .61719 L .61162 .55221 L p .748 g
F P
0 g
s
.61162 .55221 m .63073 .55648 L .65874 .62626 L p .748 g
F P
0 g
s
.50972 .43866 m .50305 .43744 L .51943 .4547 L p .06 g
F P
0 g
s
.51884 .455 m .51943 .4547 L .50305 .43744 L p .118 g
F P
0 g
s
.54244 .43035 m .55564 .42601 L .53215 .44636 L p .459 g
F P
0 g
s
.53748 .44529 m .53215 .44636 L .55564 .42601 L p .454 g
F P
0 g
s
.41629 .60696 m .39492 .61571 L .33285 .63118 L p .747 g
F P
0 g
s
.52418 .43748 m .52853 .42871 L .53376 .42599 L p .035 g
F P
0 g
s
.53376 .42599 m .5265 .43594 L .52418 .43748 L p .035 g
F P
0 g
s
.5265 .43594 m .53376 .42599 L .53566 .42846 L p .271 g
F P
0 g
s
.52168 .44219 m .52289 .4357 L .52853 .42871 L p F P
s
.36669 .43224 m .36046 .44546 L .25728 .472 L p .383 g
F P
0 g
s
.25537 .49568 m .25728 .472 L .36046 .44546 L p .414 g
F P
0 g
s
.55003 .43418 m .56057 .43277 L .61736 .38682 L p .566 g
F P
0 g
s
.61736 .38682 m .5918 .39347 L .55003 .43418 L p .566 g
F P
0 g
s
.64448 .38432 m .61736 .38682 L .56057 .43277 L p .559 g
F P
0 g
s
.5918 .39347 m .61736 .38682 L .65147 .34136 L p .616 g
F P
0 g
s
.61736 .38682 m .64448 .38432 L .68864 .3327 L p .597 g
F P
0 g
s
.68864 .3327 m .65147 .34136 L .61736 .38682 L p .597 g
F P
0 g
s
.56057 .43277 m .57171 .4332 L .64448 .38432 L p .559 g
F P
0 g
s
.67147 .38653 m .64448 .38432 L .57171 .4332 L p .549 g
F P
0 g
s
.64448 .38432 m .67147 .38653 L .72593 .3306 L p .573 g
F P
0 g
s
.72593 .3306 m .68864 .3327 L .64448 .38432 L p .573 g
F P
0 g
s
.51071 .43324 m .49812 .41956 L .50068 .42236 L p .055 g
F P
0 g
s
.26265 .51779 m .27874 .53651 L .24749 .58889 L p .365 g
F P
0 g
s
.27345 .60236 m .24749 .58889 L .27874 .53651 L p .475 g
F P
0 g
s
.50972 .43866 m .51474 .44217 L .50717 .43058 L p .151 g
F P
0 g
s
.50888 .43577 m .49887 .42683 L .48444 .41102 L p .473 g
F P
0 g
s
.48444 .41102 m .50101 .42384 L .50888 .43577 L p .473 g
F P
0 g
s
.54073 .43711 m .55003 .43418 L .5918 .39347 L p .571 g
F P
0 g
s
.5918 .39347 m .56928 .40334 L .54073 .43711 L p .571 g
F P
0 g
s
.56928 .40334 m .5918 .39347 L .6167 .35545 L p .63 g
F P
0 g
s
.65147 .34136 m .6167 .35545 L .5918 .39347 L p .616 g
F P
0 g
s
.6167 .35545 m .5862 .37338 L .56928 .40334 L p .63 g
F P
0 g
s
.5862 .37338 m .6167 .35545 L .618 .33142 L p .724 g
F P
0 g
s
.6167 .35545 m .65147 .34136 L .6536 .31067 L p .705 g
F P
0 g
s
.6536 .31067 m .618 .33142 L .6167 .35545 L p .705 g
F P
0 g
s
.53566 .42846 m .52733 .43804 L .5265 .43594 L p .271 g
F P
0 g
s
.52733 .43804 m .53566 .42846 L .53251 .43672 L p .425 g
F P
0 g
s
.54244 .43035 m .53251 .43672 L .53566 .42846 L p .373 g
F P
0 g
s
.57171 .4332 m .58273 .43559 L .67147 .38653 L p .549 g
F P
0 g
s
.69647 .39359 m .67147 .38653 L .58273 .43559 L p .539 g
F P
0 g
s
.67147 .38653 m .69647 .39359 L .76084 .33568 L p .543 g
F P
0 g
s
.76084 .33568 m .72593 .3306 L .67147 .38653 L p .543 g
F P
0 g
s
.51201 .43645 m .50101 .42384 L .49812 .41956 L p .436 g
F P
0 g
s
.49812 .41956 m .51071 .43324 L .51201 .43645 L p .436 g
F P
0 g
s
.65147 .34136 m .68864 .3327 L .69191 .29545 L p .679 g
F P
0 g
s
.69191 .29545 m .6536 .31067 L .65147 .34136 L p .679 g
F P
0 g
s
.52031 .4493 m .5198 .44595 L .52289 .4357 L p F P
s
.55084 .4152 m .56928 .40334 L .5862 .37338 L p .638 g
F P
0 g
s
.5862 .37338 m .56131 .39328 L .55084 .4152 L p .638 g
F P
0 g
s
.56131 .39328 m .5862 .37338 L .58694 .35567 L p .736 g
F P
0 g
s
.618 .33142 m .58694 .35567 L .5862 .37338 L p .724 g
F P
0 g
s
.53251 .43672 m .54244 .43035 L .52779 .4482 L p .471 g
F P
0 g
s
.53215 .44636 m .52779 .4482 L .54244 .43035 L p .459 g
F P
0 g
s
.55525 .60527 m .56732 .6334 L .53351 .59544 L p .485 g
F P
0 g
s
.56732 .6334 m .55525 .60527 L .57871 .60699 L p .669 g
F P
0 g
s
.57871 .60699 m .59902 .6346 L .56732 .6334 L p .669 g
F P
0 g
s
.51474 .44217 m .50972 .43866 L .51991 .45477 L p .052 g
F P
0 g
s
.53311 .44108 m .54073 .43711 L .56928 .40334 L p .574 g
F P
0 g
s
.56928 .40334 m .55084 .4152 L .53311 .44108 L p .574 g
F P
0 g
s
.27345 .60236 m .30615 .60772 L .33285 .63118 L p .601 g
F P
0 g
s
.33285 .63118 m .36479 .62411 L .41629 .60696 L p .747 g
F P
0 g
s
.36479 .62411 m .33285 .63118 L .30615 .60772 L p .711 g
F P
0 g
s
.50951 .44882 m .43076 .40506 L p .51065 .44871 L .529 g
F P
0 g
s
.43076 .40506 m .45049 .40666 L .51275 .44985 L p .529 g
F P
0 g
s
.41118 .4069 m .43076 .40506 L .51051 .44937 L p .528 g
F P
0 g
s
.51275 .44985 m .51174 .44916 L p .51414 .45033 L .53 g
F P
0 g
s
.51174 .44916 m .45049 .40666 L p .51281 .44928 L .53 g
F P
0 g
s
.51281 .44928 m .51486 .45068 L .51174 .44916 L .53 g
F
.51065 .44871 m .51275 .44985 L .50951 .44882 L .529 g
F
.45049 .40666 m .46908 .41131 L .51486 .45068 L p .53 g
F P
0 g
s
.51486 .45068 m .51459 .45045 L p .51563 .45109 L .532 g
F P
0 g
s
.51459 .45045 m .51289 .44899 L p .51596 .4511 L .532 g
F P
0 g
s
.51289 .44899 m .46908 .41131 L p .51378 .44926 L .532 g
F P
0 g
s
.51378 .44926 m .51596 .4511 L .51289 .44899 L .532 g
F
.46908 .41131 m .48547 .41833 L .51672 .45175 L p F P
0 g
s
.47084 .40297 m .4567 .3977 L .50885 .44778 L p .626 g
F P
0 g
s
.50885 .44778 m .51009 .44897 L p .50915 .44787 L .626 g
F P
0 g
s
.50915 .44787 m .47084 .40297 L .50885 .44778 L .626 g
F
.51596 .4511 m .51672 .45175 L .51459 .45045 L .532 g
F
.51648 .4515 m .51589 .45087 L p .51756 .45224 L .534 g
F P
0 g
s
.51495 .44986 m .48547 .41833 L p .51565 .45022 L .534 g
F P
0 g
s
.48547 .41833 m .49887 .42683 L .51824 .45295 L p .534 g
F P
0 g
s
.48413 .41044 m .47084 .40297 L .51089 .44777 L p .615 g
F P
0 g
s
.51089 .44777 m .51275 .44985 L p .51144 .44806 L .615 g
F P
0 g
s
.51144 .44806 m .48413 .41044 L .51089 .44777 L .615 g
F
.51589 .45087 m .51495 .44986 L p .51665 .45127 L .534 g
F P
0 g
s
.51565 .45022 m .51665 .45127 L .51495 .44986 L .534 g
F
.51665 .45127 m .51756 .45224 L .51589 .45087 L F
.51824 .45295 m .51675 .45095 L p .51905 .45375 L .536 g
F P
0 g
s
.51675 .45095 m .49887 .42683 L p .51726 .45134 L .536 g
F P
0 g
s
.51726 .45134 m .51937 .45414 L .51675 .45095 L .536 g
F
.51549 .44411 m .50888 .43577 L .50101 .42384 L p .502 g
F P
0 g
s
.50101 .42384 m .51201 .43645 L .51549 .44411 L p .502 g
F P
0 g
s
.49887 .42683 m .50888 .43577 L .51937 .45414 L p .536 g
F P
0 g
s
.52011 .45521 m .51937 .45414 L .51921 .45386 L p .538 g
F P
0 g
s
.51888 .45329 m .50888 .43577 L p .51923 .45369 L .538 g
F P
0 g
s
.50888 .43577 m .51549 .44411 L .52011 .45521 L p .538 g
F P
0 g
s
.51238 .43783 m .5053 .42875 L .51771 .45196 L p .575 g
F P
0 g
s
.51771 .45196 m .51824 .45295 L p .51797 .45225 L .575 g
F P
0 g
s
.51797 .45225 m .51238 .43783 L .51771 .45196 L .575 g
F
.51921 .45386 m .51888 .45329 L p .51967 .45445 L .538 g
F P
0 g
s
.51923 .45369 m .51967 .45445 L .51888 .45329 L .538 g
F
.51967 .45445 m .52011 .45521 L .51921 .45386 L F
.51824 .45295 m .51771 .45196 L p .51642 .45114 L .589 g
F P
0 g
s
.51771 .45196 m .5053 .42875 L p .51531 .44891 L .589 g
F P
0 g
s
.51642 .45114 m .51771 .45196 L .51531 .44891 L .589 g
F
.51672 .45175 m .51648 .4515 L p .51734 .45221 L .534 g
F P
0 g
s
.51824 .45295 m .51672 .45175 L p .51734 .45221 L .534 g
F P
0 g
s
.51756 .45224 m .51824 .45295 L .51648 .4515 L .534 g
F
.51672 .45175 m .51824 .45295 L p .51642 .45114 L .589 g
F P
0 g
s
.51937 .45414 m .51824 .45295 L p .51905 .45375 L .536 g
F P
0 g
s
.4958 .41932 m .48413 .41044 L .51268 .44783 L p .602 g
F P
0 g
s
.51268 .44783 m .5144 .45007 L p .51335 .44834 L .602 g
F P
0 g
s
.51335 .44834 m .4958 .41932 L .51268 .44783 L .602 g
F
.5144 .45007 m .51486 .45068 L p .51453 .45015 L F P
0 g
s
.51453 .45015 m .4958 .41932 L .5144 .45007 L .602 g
F
.51672 .45175 m .4958 .41932 L p .51453 .45015 L F P
0 g
s
.51672 .45175 m .51486 .45068 L p .51563 .45109 L .532 g
F P
0 g
s
.51486 .45068 m .51672 .45175 L p .51453 .45015 L .602 g
F P
0 g
s
.51486 .45068 m .5144 .45007 L p .51241 .44939 L .615 g
F P
0 g
s
.5144 .45007 m .48413 .41044 L p .51144 .44806 L .615 g
F P
0 g
s
.51241 .44939 m .5144 .45007 L .51144 .44806 L .615 g
F
.51486 .45068 m .51275 .44985 L p .51414 .45033 L .53 g
F P
0 g
s
.51275 .44985 m .51486 .45068 L p .51241 .44939 L .615 g
F P
0 g
s
.51275 .44985 m .5123 .44936 L p .51017 .44898 L .626 g
F P
0 g
s
.5123 .44936 m .47084 .40297 L p .50925 .4479 L .626 g
F P
0 g
s
.51017 .44898 m .5123 .44936 L .50925 .4479 L .626 g
F
.51051 .44937 m .50951 .44882 L p .51197 .44961 L .529 g
F P
0 g
s
.51275 .44985 m .51051 .44937 L p .51197 .44961 L .529 g
F P
0 g
s
.5083 .44931 m .41118 .4069 L p .51018 .44923 L .528 g
F P
0 g
s
.51051 .44937 m .5083 .44931 L p .51018 .44923 L .528 g
F P
0 g
s
.39311 .41227 m .41118 .4069 L .5083 .44931 L p .527 g
F P
0 g
s
.50628 .44968 m .39311 .41227 L p .50797 .4492 L .527 g
F P
0 g
s
.5083 .44931 m .50628 .44968 L p .50797 .4492 L .527 g
F P
0 g
s
.37788 .4209 m .39311 .41227 L .50628 .44968 L p .527 g
F P
0 g
s
.5046 .45048 m .37788 .4209 L p .50595 .4496 L .527 g
F P
0 g
s
.50628 .44968 m .5046 .45048 L p .50595 .4496 L .527 g
F P
0 g
s
.36669 .43224 m .37788 .4209 L .5046 .45048 L p .528 g
F P
0 g
s
.5034 .45165 m .36669 .43224 L p .50427 .45043 L .528 g
F P
0 g
s
.5046 .45048 m .5034 .45165 L p .50427 .45043 L .528 g
F P
0 g
s
.36046 .44546 m .36669 .43224 L .5034 .45165 L p .529 g
F P
0 g
s
.50276 .45311 m .36046 .44546 L p .50306 .45164 L .529 g
F P
0 g
s
.5034 .45165 m .50276 .45311 L p .50306 .45164 L .529 g
F P
0 g
s
.51046 .46093 m .50995 .4612 L p .50902 .46071 L .545 g
F P
0 g
s
.50995 .4612 m .42786 .50509 L p .50677 .46112 L .545 g
F P
0 g
s
.50677 .46112 m .50825 .4603 L .50995 .4612 L .545 g
F
.51271 .46116 m .44824 .50449 L p .51005 .46122 L .547 g
F P
0 g
s
.51046 .46093 m .51271 .46116 L p .51005 .46122 L .547 g
F P
0 g
s
.51483 .461 m .46754 .50061 L p .51229 .46152 L .55 g
F P
0 g
s
.51271 .46116 m .51483 .461 L p .51229 .46152 L .55 g
F P
0 g
s
.5167 .46051 m .48455 .49415 L p .51442 .46144 L .553 g
F P
0 g
s
.51483 .461 m .5167 .46051 L p .51442 .46144 L .553 g
F P
0 g
s
.51822 .45979 m .4984 .48604 L p .51632 .46104 L .555 g
F P
0 g
s
.5167 .46051 m .51822 .45979 L p .51632 .46104 L .555 g
F P
0 g
s
.51936 .45895 m .50868 .47737 L p .51787 .46043 L .557 g
F P
0 g
s
.51822 .45979 m .51936 .45895 L p .51787 .46043 L .557 g
F P
0 g
s
.51936 .45895 m .52011 .45812 L .51542 .46922 L p .56 g
F P
0 g
s
.51542 .46922 m .50868 .47737 L .51936 .45895 L p .56 g
F P
0 g
s
.50868 .47737 m .4984 .48604 L .51757 .46065 L p .557 g
F P
0 g
s
.51757 .46065 m .51822 .45979 L p .51787 .46043 L .557 g
F P
0 g
s
.50868 .47737 m .51757 .46065 L .51787 .46043 L .557 g
F
.51936 .45895 m .51892 .45967 L p .51996 .45835 L .353 g
F P
0 g
s
.51892 .45967 m .51225 .47072 L p .51931 .45939 L .353 g
F P
0 g
s
.51996 .45835 m .51892 .45967 L .51931 .45939 L .353 g
F
.52011 .45812 m .51936 .45895 L p .51996 .45835 L F P
0 g
s
.4984 .48604 m .48455 .49415 L .51611 .46113 L p .555 g
F P
0 g
s
.51611 .46113 m .5167 .46051 L p .51632 .46104 L .555 g
F P
0 g
s
.4984 .48604 m .51611 .46113 L .51632 .46104 L .555 g
F
.51822 .45979 m .51763 .4605 L p .51917 .45917 L .362 g
F P
0 g
s
.51763 .4605 m .50502 .47559 L p .51824 .46025 L .362 g
F P
0 g
s
.51917 .45917 m .51763 .4605 L .51824 .46025 L .362 g
F
.51936 .45895 m .51822 .45979 L p .51917 .45917 L F P
0 g
s
.48455 .49415 m .46754 .50061 L .51427 .46147 L p .553 g
F P
0 g
s
.51427 .46147 m .51483 .461 L p .51442 .46144 L .553 g
F P
0 g
s
.48455 .49415 m .51427 .46147 L .51442 .46144 L .553 g
F
.5167 .46051 m .51596 .46116 L p .518 .45998 L .371 g
F P
0 g
s
.51596 .46116 m .49526 .47933 L p .5168 .461 L .371 g
F P
0 g
s
.518 .45998 m .51596 .46116 L .5168 .461 L .371 g
F
.51822 .45979 m .5167 .46051 L p .518 .45998 L F P
0 g
s
.46754 .50061 m .44824 .50449 L .51217 .46152 L p .55 g
F P
0 g
s
.51217 .46152 m .51271 .46116 L p .51229 .46152 L .55 g
F P
0 g
s
.46754 .50061 m .51217 .46152 L .51229 .46152 L .55 g
F
.51483 .461 m .51399 .46153 L p .51645 .46067 L .387 g
F P
0 g
s
.51399 .46153 m .48324 .48118 L p .51503 .46154 L .387 g
F P
0 g
s
.51645 .46067 m .51399 .46153 L .51503 .46154 L .387 g
F
.5167 .46051 m .51483 .461 L p .51645 .46067 L F P
0 g
s
.51271 .46116 m .51246 .46127 L p .51362 .46114 L .41 g
F P
0 g
s
.51483 .461 m .51271 .46116 L p .51362 .46114 L .41 g
F P
0 g
s
.44824 .50449 m .42786 .50509 L .50995 .4612 L p .547 g
F P
0 g
s
.50995 .4612 m .51046 .46093 L p .51005 .46122 L .547 g
F P
0 g
s
.44824 .50449 m .50995 .4612 L .51005 .46122 L .547 g
F
.51246 .46127 m .51181 .46156 L p .51411 .46131 L .41 g
F P
0 g
s
.42786 .50509 m .40785 .50204 L .50643 .46105 L p .545 g
F P
0 g
s
.50643 .46105 m .50825 .4603 L p .50677 .46112 L .545 g
F P
0 g
s
.42786 .50509 m .50643 .46105 L .50677 .46112 L .545 g
F
.51181 .46156 m .51091 .46196 L p .513 .46179 L .41 g
F P
0 g
s
.51091 .46196 m .46955 .48054 L p .51205 .4622 L .41 g
F P
0 g
s
.513 .46179 m .51091 .46196 L .51205 .4622 L .41 g
F
.51411 .46131 m .51181 .46156 L .513 .46179 L F
.50623 .45928 m .50825 .4603 L .50777 .4605 L p .542 g
F P
0 g
s
.50777 .4605 m .40785 .50204 L p .50483 .45989 L .542 g
F P
0 g
s
.50483 .45989 m .50623 .45928 L .50777 .4605 L .542 g
F
.51046 .46093 m .51021 .461 L p .51243 .46123 L .44 g
F P
0 g
s
.40785 .50204 m .38974 .49539 L .50459 .45979 L p .542 g
F P
0 g
s
.50459 .45979 m .50623 .45928 L p .50483 .45989 L .542 g
F P
0 g
s
.40785 .50204 m .50459 .45979 L .50483 .45989 L .542 g
F
.51021 .461 m .50874 .46143 L p .51195 .46137 L .44 g
F P
0 g
s
.50874 .46143 m .45502 .47708 L p .50996 .46191 L .44 g
F P
0 g
s
.51195 .46137 m .50874 .46143 L .50996 .46191 L .44 g
F
.51243 .46123 m .51021 .461 L .51195 .46137 L F
.51271 .46116 m .51046 .46093 L p .51243 .46123 L F P
0 g
s
.50825 .4603 m .51046 .46093 L p .50902 .46071 L .545 g
F P
0 g
s
.51483 .461 m .51246 .46127 L .51411 .46131 L .41 g
F
.51046 .46093 m .50825 .4603 L .50749 .46041 L p .475 g
F P
0 g
s
.50307 .45826 m .37496 .48554 L p .50054 .45703 L .536 g
F P
0 g
s
.50623 .45928 m .38974 .49539 L p .50417 .45807 L .539 g
F P
0 g
s
.50456 .45795 m .50623 .45928 L p .50417 .45807 L .539 g
F P
0 g
s
.50297 .45644 m .50202 .45655 L p .50161 .45488 L .533 g
F P
0 g
s
.50202 .45655 m .3647 .47324 L p .50019 .45507 L .533 g
F P
0 g
s
.50019 .45507 m .50161 .45488 L .50202 .45655 L .533 g
F
.50623 .45928 m .50543 .45931 L p .50796 .4603 L .513 g
F P
0 g
s
.3647 .47324 m .35975 .4595 L .49992 .45482 L p .533 g
F P
0 g
s
.49992 .45482 m .50275 .45473 L p .50019 .45507 L .533 g
F P
0 g
s
.3647 .47324 m .49992 .45482 L .50019 .45507 L .533 g
F
.50543 .45931 m .50472 .45933 L p .5067 .46033 L .513 g
F P
0 g
s
.37496 .48554 m .3647 .47324 L .50297 .45644 L p .536 g
F P
0 g
s
.50297 .45644 m .50337 .45639 L p .503 .45647 L .536 g
F P
0 g
s
.37496 .48554 m .50297 .45644 L .503 .45647 L .536 g
F
.38974 .49539 m .37496 .48554 L .50307 .45826 L p .539 g
F P
0 g
s
.50307 .45826 m .50456 .45795 L p .50325 .45837 L .539 g
F P
0 g
s
.38974 .49539 m .50307 .45826 L .50325 .45837 L .539 g
F
.50749 .46041 m .50673 .46053 L p .50898 .46114 L .475 g
F P
0 g
s
.50673 .46053 m .44068 .47079 L p .50795 .46129 L .475 g
F P
0 g
s
.50898 .46114 m .50673 .46053 L .50795 .46129 L .475 g
F
.50428 .45792 m .50407 .45791 L p .50541 .45921 L .549 g
F P
0 g
s
.50456 .45795 m .50307 .45826 L p .503 .45647 L .536 g
F P
0 g
s
.50054 .45703 m .503 .45647 L .50307 .45826 L .536 g
F
.35975 .4595 m .36046 .44546 L .50025 .45297 L p .531 g
F P
0 g
s
.50025 .45297 m .50163 .45305 L p .50035 .45311 L .531 g
F P
0 g
s
.35975 .4595 m .50025 .45297 L .50035 .45311 L .531 g
F
.50407 .45791 m .50372 .45788 L p .50502 .45917 L .549 g
F P
0 g
s
.35975 .4595 m .50163 .45305 L .50167 .45312 L .531 g
F
.35975 .4595 m .50241 .45309 L .50242 .45312 L F
.50276 .45311 m .50275 .45473 L .35975 .4595 L p F P
0 g
s
.50372 .45788 m .4169 .45116 L p .50461 .45913 L .549 g
F P
0 g
s
.50502 .45917 m .50372 .45788 L .50461 .45913 L .549 g
F
.50541 .45921 m .50407 .45791 L .50502 .45917 L F
.50472 .45933 m .42763 .46195 L p .5058 .46035 L .513 g
F P
0 g
s
.5067 .46033 m .50472 .45933 L .5058 .46035 L .513 g
F
.50796 .4603 m .50543 .45931 L .5067 .46033 L F
.50825 .4603 m .50623 .45928 L p .50796 .4603 L F P
0 g
s
.51046 .46093 m .50749 .46041 L .50898 .46114 L .475 g
F
.50337 .45639 m .50456 .45795 L p .503 .45647 L .536 g
F P
0 g
s
.50456 .45795 m .50428 .45792 L p .50525 .4586 L .549 g
F P
0 g
s
.50623 .45928 m .50456 .45795 L p .50525 .4586 L .549 g
F P
0 g
s
.50623 .45928 m .50428 .45792 L .50541 .45921 L .549 g
F
.50337 .45639 m .50297 .45644 L p .5024 .45478 L .533 g
F P
0 g
s
.50161 .45488 m .5024 .45478 L .50297 .45644 L .533 g
F
.50275 .45473 m .50337 .45639 L p .5024 .45478 L F P
0 g
s
.50337 .45639 m .50308 .45634 L p .50382 .45715 L .582 g
F P
0 g
s
.50456 .45795 m .50337 .45639 L p .50382 .45715 L .582 g
F P
0 g
s
.50308 .45634 m .40939 .43926 L p .50372 .45778 L .582 g
F P
0 g
s
.50456 .45795 m .50308 .45634 L .50372 .45778 L .582 g
F
.50163 .45305 m .50241 .45309 L p .50167 .45312 L .531 g
F P
0 g
s
.50241 .45309 m .50276 .45311 L p .50242 .45312 L .531 g
F P
0 g
s
.50275 .45473 m .40569 .42722 L p .50308 .4563 L .608 g
F P
0 g
s
.50337 .45639 m .50275 .45473 L p .50308 .4563 L .608 g
F P
0 g
s
.50276 .45311 m .4061 .41603 L p .50246 .45461 L .627 g
F P
0 g
s
.50275 .45473 m .50276 .45311 L p .50246 .45461 L .627 g
F P
0 g
s
.5034 .45165 m .41054 .40662 L p .50246 .45296 L .639 g
F P
0 g
s
.50276 .45311 m .5034 .45165 L p .50246 .45296 L .639 g
F P
0 g
s
.5046 .45048 m .41859 .39973 L p .50309 .45146 L .645 g
F P
0 g
s
.5034 .45165 m .5046 .45048 L p .50309 .45146 L .645 g
F P
0 g
s
.50628 .44968 m .42958 .39585 L p .50428 .45025 L .646 g
F P
0 g
s
.5046 .45048 m .50628 .44968 L p .50428 .45025 L .646 g
F P
0 g
s
.5083 .44931 m .44261 .3952 L p .50595 .4494 L .642 g
F P
0 g
s
.50628 .44968 m .5083 .44931 L p .50595 .4494 L .642 g
F P
0 g
s
.51051 .44937 m .4567 .3977 L p .50797 .44897 L .636 g
F P
0 g
s
.5083 .44931 m .51051 .44937 L p .50797 .44897 L .636 g
F P
0 g
s
.51051 .44937 m .51275 .44985 L p .51017 .44898 L .626 g
F P
0 g
s
.51009 .44897 m .51051 .44937 L p