function [f] = dXdT123(t,x)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% FUNCTION dXdT: computes the derivatives for an ODE model governing the
%   dynamic changes in the intramitochondrial Na and Ca with the addition 
%   of Na or Ca to the extramitochondrial buffer medium.
%
% INPUTS:
%   t - time (not used)
%   c - vector of state variables (Na_x, Na_e and H_x)
%
% OUTPUTS:
%   f - vector of time derivatives of concentrations
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

global J C exchange p
format long

%% Thermodynamic parameters
F = 0.0964846;     % Faraday's constant (10^6 Coul * mol^-1 = kJ * mV^-1 * mol^-1)
R = 8314e-6;       % Universal gas constant (kJ * mol^-1 * K^-1)
T = 303;           % Temperature (K)
FRT = F/(R*T);     % mV^-1

%% Anatomical parameters (volume, surface area, etc...)
% Mitocondrial preparation of Cox and Matlib (1993) contained 1.4 mg mito
% protein per 2 ml of buffer solution. Thus, W_e = 1e-3/(0.7*Fact).
Fact = 3.6697e-6;         % 1 mg protein = 3.6697e-6 L mito
W_e = 1e-3/(0.7*Fact);    % External buffer water volume (ml water / ml mito)
W_m = 0.143/0.20;         % mitochondrial water space (ml water / ml mito)
W_x = 0.9*W_m;            % Matrix water space (ml water / ml mito)
%W_i = 0.1*W_m;           % IM water space (ml water / ml mito)

%% Buffer concentrations
% The idea is to use a pH gradient of about 0.1 unit across the IMM. This
% seems to give better fit of the model to the data on the kinetics of 
% initial NCE flux rates and dynamics of mitochondrial Ca2+.
pH_e = 7.2;               % external pH
H_e  = 10^(-pH_e);        % external [H+], M
pH_x = 7.3;              % matrix pH
H_x  = 10^(-pH_x);        % matrix [H+], M
dPsi = 190;               % membrane potential (constant)

%% Concentration state variables:
Na_x   = x(1);
Ca_fx  = x(2);     % free Ca in matrix
Na_e   = x(3);
Ca_te  = x(4);     % Total buffer Ca

%% EGTA/Ca equilibrium binding in external space using coopertivity of nH.
x_EGTACa = 1e6;                 % EGTA-Ca binding activity (mol s^{-1} M^{-2} (l mito)^{-1})
EGTA_te = 50e-6;                % Total EGTA concentration in the matrix (M)
K_EGTACa = 10^(-6.7);           % EGTA-Ca binding dissociation constant (M)
nH = 1.0;                       % Hill coefficient for EGTA-Ca binding
if (Ca_te < EGTA_te)
    %Ca_fe = (Ca_te/(EGTA_te-Ca_te))^(1/nH)*K_EGTACa;   % This initial guess may lead to imaginary estimates of Ca_fe
    Ca_fe = Ca_te/2;
    for j = 1:10
        Ca_fe_old = Ca_fe;
        Ca_fe = ((Ca_te-Ca_fe_old)/(EGTA_te-(Ca_te-Ca_fe_old)))^(1/nH)*K_EGTACa;
    end
else
    eps = 5e-6;                   % regularization parameter
    Ca_te_1 = EGTA_te - eps;
    Ca_fe_1 = Ca_te_1/2;
    for j = 1:10
        Ca_fe_1_old = Ca_fe_1;
        Ca_fe_1 = ((Ca_te_1-Ca_fe_1_old)/(EGTA_te-(Ca_te_1-Ca_fe_1_old)))^(1/nH)*K_EGTACa;
    end
    Ca_fe_2 = Ca_te - EGTA_te;
    Ca_fe = max(Ca_fe_1,Ca_fe_2);
end
Ca_be = (EGTA_te*Ca_fe^nH)/(K_EGTACa^nH + Ca_fe^nH);
EGTA_be = Ca_be;
EGTA_fe = EGTA_te - EGTA_be;
J_EGTACa = x_EGTACa*(EGTA_fe*Ca_fe^nH - K_EGTACa^nH*Ca_be);

%% Indexing of concentrations
C = zeros(8,1);
C(1) = Na_x;
C(2) = Ca_fx;
C(3) = Na_e;
C(4) = Ca_fe;
C(5) = Ca_be;
C(6) = Ca_te;
C(7) = EGTA_be;
C(8) = EGTA_fe;

%% Parameters for Ca_Uni, NaCa_Exch, and NaH_Exch
Z_Ca = 2;                        % Valency of Ca (uniteless)
f_CU1 = 0.001;                   % Factor for x_CU
f_CU2 = 1.0;                     % Factor of Km_Ca_CU
x_CU = f_CU1*1.5e-2*1e-9/Fact;   % Ca uniporter activity coefficient (mol/s/L mito)
Km_Ca_CU = f_CU2*90e-6;          % MM binding constant for Ca in Ca_Uni (M)

if (exchange == 2)
   f_NCE1 = p(1);                             % Factor for x_NCE
   f_NCE2 = p(2);                             % Factor of Km_Na_NCE
   f_NCE3 = p(3);                             % Factor of Km_Ca_NCE
   x_NCE = f_NCE1*0.176*1e-6/(2*Fact);        % mol/L mito/s (0.176 umol/mg/s, estimated to fit Na influx)
   Km_Na_NCE = f_NCE2*8.2e-3;                 % M (Aleks and Vickie, 9.5e-3)
   Km_Ca_NCE = f_NCE3*2.1e-6;                 % M (Aleks and Vickie, 7e-7)
elseif (exchange == 3)
   f_NCE1 = p(1);                             % Factor for x_NCE
   f_NCE2 = p(2);                             % Factor of Km_Na_NCE
   f_NCE3 = p(3);                             % Factor of Km_Ca_NCE
   x_NCE = f_NCE1*0.176*0.023e-6/(3*Fact);    % mol/L mito/s (0.023*0.176 umol/mg/s, estimated to fit Na influx)
   Km_Na_NCE = f_NCE2*8.2e-3;                 % M (Aleks and Vickie, 9.5e-3)
   Km_Ca_NCE = f_NCE3*2.1e-6;                 % M (Aleks and Vickie, 7e-7)
end

x_NHE = 12e-9/Fact;              % mol/L mito/s (12 nmol/mg/s) (estimated to fit the data)
Km_Na_NHE = 22e-3;               % M
%beta_H  = 235;                   % matrix buffering coefficient (pH M^{-1})

f_betaCa = 1.0;                  % Factor for beta_Ca
beta_Ca = f_betaCa*10;           % matrix buffering coefficient, M^{-1}

%% Fluxes through Ca_Unu, NaCa_Exch, and NaH_Exch
nH_CU = 2.7;
Ge = Ca_fe^2 / (Km_Ca_CU^2 + Ca_fx^2 + Ca_fe^2);
Gx = Ca_fx^2 / (Km_Ca_CU^2 + Ca_fx^2 + Ca_fe^2);
dPhi = Z_Ca*dPsi*FRT;    % Composite term (nondimensional)
Fact = ((dPhi/nH_CU)/sinh(dPhi/nH_CU))^nH_CU;
J_CU = x_CU*Fact*(Ge*exp(dPhi) - Gx*exp(-dPhi));

if (exchange == 2) % 2:1 NCE exchanger
   J_NCE = x_NCE * ((Na_e^2 * Ca_fx)/(Km_Na_NCE^2*Km_Ca_NCE) - (Na_x^2 * Ca_fe)/...
           (Km_Na_NCE^2*Km_Ca_NCE)) / (1 + Na_e^2/Km_Na_NCE^2 + Ca_fx/Km_Ca_NCE + ...
           (Na_e^2 * Ca_fx)/(Km_Na_NCE^2*Km_Ca_NCE) + Na_x^2/Km_Na_NCE^2 + ...
           Ca_fe/Km_Ca_NCE + (Na_x^2 * Ca_fe)/(Km_Na_NCE^2*Km_Ca_NCE));
elseif (exchange == 3) % 3:1 NCE exchanger
   n_Ca = 0.5;
   J_NCE = x_NCE * (exp(n_Ca*FRT*dPsi) * (Na_e^3 * Ca_fx)/(Km_Na_NCE^3*Km_Ca_NCE) - ...
           exp(-(1-n_Ca)*FRT*dPsi) * (Na_x^3 * Ca_fe)/(Km_Na_NCE^3*Km_Ca_NCE)) / ...
           (1 + Na_e^3/Km_Na_NCE^3 + Ca_fx/Km_Ca_NCE + (Na_e^3 * Ca_fx)/...
           (Km_Na_NCE^3*Km_Ca_NCE) + Na_x^3/Km_Na_NCE^3 + Ca_fe/Km_Ca_NCE + ...
           (Na_x^3 * Ca_fe)/(Km_Na_NCE^3*Km_Ca_NCE));
end

J_NHE =  x_NHE * (H_x/(H_x+1e-7))^1 * (H_x/1e-7)^1 * ...
    (H_e*Na_x - H_x*Na_e) / (Km_Na_NHE*(H_e+H_x) + H_e*Na_x + H_x*Na_e);

%% Indexing of Fluxes
J = zeros(4,1);
J(1) = J_CU;
J(2) = J_NCE;
J(3) = J_NHE;
J(4) = J_EGTACa;

%% Derivatives: RHS of the differential equations of the Ca handling model (4 state variables).
f = zeros(4,1);
if (exchange == 2) % 2:1 NaCa exchange
    f(1) = (2*J_NCE - J_NHE)/W_x;  % Na_x
    f(2) = (J_CU - J_NCE)/(W_x*beta_Ca);  % Ca_fx
    f(3) = -(2*J_NCE - J_NHE)/W_e;  % Na_e
    f(4) = -(J_CU - J_NCE)/W_e;   % Ca_fe
elseif (exchange == 3) % 3:1 NaCa exchange
    f(1) = (3*J_NCE - J_NHE)/W_x;  % Na_x
    f(2) = (J_CU - J_NCE)/(W_x*beta_Ca);  % Ca_fx
    f(3) = -(3*J_NCE - J_NHE)/W_e;  % Na_e
    f(4) = -(J_CU - J_NCE)/W_e;   % Ca_fe
end
