#! /usr/bin/env python
## gp_regression_cmdstan.py

import cmdstanpy
import numpy as np
import matplotlib.pyplot as plt

def reg_fn(t): return(10+5*np.sin(t)+t**2/5.0)

# Simulate data
gen = np.random.default_rng(seed=0)
n = 40
m = 50
T = 10
x = np.linspace(start=0, stop=T, num=n)
y = [gen.normal(loc=reg_fn(x_i)) for x_i in x]
grid = np.linspace(start=0, stop=T, num=m)
sm_data = {'n':n, 'x':x, 'y':y, 'a':1, 'b':0.5, 'm':m, 'grid':grid}

# Initialise stan object
sm = cmdstanpy.CmdStanModel(stan_file='gp_regression.stan')


# Select the number of MCMC chains and iterations, then sample
chains, samples, burn = 1, 10000, 1000
fit=sm.sample(data=sm_data, chains=chains, iter_sampling=samples, iter_warmup=burn, save_warmup=False, seed=1)

# Plot regression function and posterior for rho
fig,axs=plt.subplots(1,2,figsize=(10,4),constrained_layout=True)
fig.canvas.manager.set_window_title('GP regression posterior')
f = np.mean(fit.stan_variable('fn_vals'), axis=0)
true_f = [reg_fn(x_i) for x_i in grid]
r = fit.stan_variable('rho')[:,]
axs[0].plot(grid,f)
axs[0].plot(grid,true_f, color='c', lw=2, linestyle='--')
axs[0].scatter(x,y, color='black')
axs[0].set_title('Posterior mean regression function')
axs[0].set_xlabel(r'$x$')
h = axs[1].hist(r,200, density=True);
axs[1].set_title('Approximate posterior density of '+r'$\rho$')
axs[1].set_xlabel(r'$\rho$')
plt.show()
