// Interface for the implementation of chopthin for MATLAB
// Copyright (C)2015 Axel Gandy

// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

// MEX extension file for MATLAB implementing the chopthin algorithm
// 
// Calling Syntax:
//  [w,ind]= chopthin(w,eta,N) 
// The arguments are: the vector of weights (w), eta, N
// Output: 2 vectors 
// Examples:
// mex chopthin.cpp
// [w,ind] = chopthin(rand(1,5),5.8,3)
// [w,ind] = chopthin(rand(1,1e4),5.8,1e4);      

//  Problems/ things to watch out for
// - currently only the standard random number generator is used in a crude waqy.
// - Limited checking of the input and the output.

#include "mex.h"
#include <vector>
#include <math.h> //ceil

inline int intrand(int n) { return floor(((double)rand())/((double)RAND_MAX)*(double)(n)); }
#define myrunif() ((double)rand()/((double)RAND_MAX))
#define chopthin_error(x) mexErrMsgTxt((x))
#include "chopthin_internal.h"

void mexFunction( int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{

  if (nrhs != 3) {
    mexErrMsgIdAndTxt("MATLAB:chopthin:nargin", 
            "chopthin requires three input arguments.");
  } else if (nlhs != 2) {
    mexErrMsgIdAndTxt("MATLAB:chopthin:nargout",
            "chopthin requires two output argument.");
  }


  double* w= mxGetPr(prhs[0]);
  int n = mxGetN(prhs[0]);
  if (n==0)  mexErrMsgTxt("First argument w has no elements.");

  double eta=mxGetScalar(prhs[1]);
  int N=mxGetScalar(prhs[2]);
  if (N<=0)  mexErrMsgTxt("N  (the third argument) must be >= 1");
  plhs[0]=mxCreateDoubleMatrix(1,N,mxREAL);
  double *wres=mxGetPr(plhs[0]);
  plhs[1]=mxCreateDoubleMatrix(1,N,mxREAL);// could be improved by creating an integer array for the indices
  double* Indres=mxGetPr(plhs[1]);
  std::vector<double> wvec(w,w+n);
  std::vector<int> indvec(N);
  std::vector<double> wresvec(N);
  chopthin_internal<true>(wvec,N,wresvec,indvec,eta);
  for (int i=0; i<N; i++){
    Indres[i]=indvec[i];
    wres[i]=wresvec[i];
  }
}
